// SecureHash.cpp: implementation of the CSecureHash class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "TestHash.h"
#include "SecureHash.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSecureHash::CSecureHash()
{

}

CSecureHash::~CSecureHash()
{

}

/*
 * SHA-256 context setup
 */
void CSecureHash::Starts ( int nHashType )
{
    m_Context.m_lTotal[0] = 0;
    m_Context.m_lTotal[1] = 0;

    if( nHashType == HASHTYPE_SHA_256 )
    {
        /* SHA-256 */
        m_Context.m_lState[0] = 0x6A09E667;
        m_Context.m_lState[1] = 0xBB67AE85;
        m_Context.m_lState[2] = 0x3C6EF372;
        m_Context.m_lState[3] = 0xA54FF53A;
        m_Context.m_lState[4] = 0x510E527F;
        m_Context.m_lState[5] = 0x9B05688C;
        m_Context.m_lState[6] = 0x1F83D9AB;
        m_Context.m_lState[7] = 0x5BE0CD19;
    }
    else
    {
        /* SHA-224 */
        m_Context.m_lState[0] = 0xC1059ED8;
        m_Context.m_lState[1] = 0x367CD507;
        m_Context.m_lState[2] = 0x3070DD17;
        m_Context.m_lState[3] = 0xF70E5939;
        m_Context.m_lState[4] = 0xFFC00B31;
        m_Context.m_lState[5] = 0x68581511;
        m_Context.m_lState[6] = 0x64F98FA7;
        m_Context.m_lState[7] = 0xBEFA4FA4;
    }

    m_Context.m_nHashType = nHashType;
}

void CSecureHash::Process ( unsigned char pData[64] )
{
    unsigned long temp1, temp2, W[64] = { 0 };
    unsigned long A, B, C, D, E, F, G, H;

    GetUlongBigEndian ( W[ 0], pData,  0 );
    GetUlongBigEndian ( W[ 1], pData,  4 );
    GetUlongBigEndian ( W[ 2], pData,  8 );
    GetUlongBigEndian ( W[ 3], pData, 12 );
    GetUlongBigEndian ( W[ 4], pData, 16 );
    GetUlongBigEndian ( W[ 5], pData, 20 );
    GetUlongBigEndian ( W[ 6], pData, 24 );
    GetUlongBigEndian ( W[ 7], pData, 28 );
    GetUlongBigEndian ( W[ 8], pData, 32 );
    GetUlongBigEndian ( W[ 9], pData, 36 );
    GetUlongBigEndian ( W[10], pData, 40 );
    GetUlongBigEndian ( W[11], pData, 44 );
    GetUlongBigEndian ( W[12], pData, 48 );
    GetUlongBigEndian ( W[13], pData, 52 );
    GetUlongBigEndian ( W[14], pData, 56 );
    GetUlongBigEndian ( W[15], pData, 60 );

#define  SHR(x,n) ((x & 0xFFFFFFFF) >> n)
#define ROTR(x,n) (SHR(x,n) | (x << (32 - n)))

#define S0(x) (ROTR(x, 7) ^ ROTR(x,18) ^  SHR(x, 3))
#define S1(x) (ROTR(x,17) ^ ROTR(x,19) ^  SHR(x,10))

#define S2(x) (ROTR(x, 2) ^ ROTR(x,13) ^ ROTR(x,22))
#define S3(x) (ROTR(x, 6) ^ ROTR(x,11) ^ ROTR(x,25))

#define F0(x,y,z) ((x & y) | (z & (x | y)))
#define F1(x,y,z) (z ^ (x & (y ^ z)))

#define R(t)                                    \
(                                               \
    W[t] = S1(W[t -  2]) + W[t -  7] +          \
           S0(W[t - 15]) + W[t - 16]            \
)

#define P(a,b,c,d,e,f,g,h,x,K)                  \
{                                               \
    temp1 = h + S3(e) + F1(e,f,g) + K + x;      \
    temp2 = S2(a) + F0(a,b,c);                  \
    d += temp1; h = temp1 + temp2;              \
}

    A = m_Context.m_lState[0];
    B = m_Context.m_lState[1];
    C = m_Context.m_lState[2];
    D = m_Context.m_lState[3];
    E = m_Context.m_lState[4];
    F = m_Context.m_lState[5];
    G = m_Context.m_lState[6];
    H = m_Context.m_lState[7];

    P( A, B, C, D, E, F, G, H, W[ 0], 0x428A2F98 );
    P( H, A, B, C, D, E, F, G, W[ 1], 0x71374491 );
    P( G, H, A, B, C, D, E, F, W[ 2], 0xB5C0FBCF );
    P( F, G, H, A, B, C, D, E, W[ 3], 0xE9B5DBA5 );
    P( E, F, G, H, A, B, C, D, W[ 4], 0x3956C25B );
    P( D, E, F, G, H, A, B, C, W[ 5], 0x59F111F1 );
    P( C, D, E, F, G, H, A, B, W[ 6], 0x923F82A4 );
    P( B, C, D, E, F, G, H, A, W[ 7], 0xAB1C5ED5 );
    P( A, B, C, D, E, F, G, H, W[ 8], 0xD807AA98 );
    P( H, A, B, C, D, E, F, G, W[ 9], 0x12835B01 );
    P( G, H, A, B, C, D, E, F, W[10], 0x243185BE );
    P( F, G, H, A, B, C, D, E, W[11], 0x550C7DC3 );
    P( E, F, G, H, A, B, C, D, W[12], 0x72BE5D74 );
    P( D, E, F, G, H, A, B, C, W[13], 0x80DEB1FE );
    P( C, D, E, F, G, H, A, B, W[14], 0x9BDC06A7 );
    P( B, C, D, E, F, G, H, A, W[15], 0xC19BF174 );
    P( A, B, C, D, E, F, G, H, R(16), 0xE49B69C1 );
    P( H, A, B, C, D, E, F, G, R(17), 0xEFBE4786 );
    P( G, H, A, B, C, D, E, F, R(18), 0x0FC19DC6 );
    P( F, G, H, A, B, C, D, E, R(19), 0x240CA1CC );
    P( E, F, G, H, A, B, C, D, R(20), 0x2DE92C6F );
    P( D, E, F, G, H, A, B, C, R(21), 0x4A7484AA );
    P( C, D, E, F, G, H, A, B, R(22), 0x5CB0A9DC );
    P( B, C, D, E, F, G, H, A, R(23), 0x76F988DA );
    P( A, B, C, D, E, F, G, H, R(24), 0x983E5152 );
    P( H, A, B, C, D, E, F, G, R(25), 0xA831C66D );
    P( G, H, A, B, C, D, E, F, R(26), 0xB00327C8 );
    P( F, G, H, A, B, C, D, E, R(27), 0xBF597FC7 );
    P( E, F, G, H, A, B, C, D, R(28), 0xC6E00BF3 );
    P( D, E, F, G, H, A, B, C, R(29), 0xD5A79147 );
    P( C, D, E, F, G, H, A, B, R(30), 0x06CA6351 );
    P( B, C, D, E, F, G, H, A, R(31), 0x14292967 );
    P( A, B, C, D, E, F, G, H, R(32), 0x27B70A85 );
    P( H, A, B, C, D, E, F, G, R(33), 0x2E1B2138 );
    P( G, H, A, B, C, D, E, F, R(34), 0x4D2C6DFC );
    P( F, G, H, A, B, C, D, E, R(35), 0x53380D13 );
    P( E, F, G, H, A, B, C, D, R(36), 0x650A7354 );
    P( D, E, F, G, H, A, B, C, R(37), 0x766A0ABB );
    P( C, D, E, F, G, H, A, B, R(38), 0x81C2C92E );
    P( B, C, D, E, F, G, H, A, R(39), 0x92722C85 );
    P( A, B, C, D, E, F, G, H, R(40), 0xA2BFE8A1 );
    P( H, A, B, C, D, E, F, G, R(41), 0xA81A664B );
    P( G, H, A, B, C, D, E, F, R(42), 0xC24B8B70 );
    P( F, G, H, A, B, C, D, E, R(43), 0xC76C51A3 );
    P( E, F, G, H, A, B, C, D, R(44), 0xD192E819 );
    P( D, E, F, G, H, A, B, C, R(45), 0xD6990624 );
    P( C, D, E, F, G, H, A, B, R(46), 0xF40E3585 );
    P( B, C, D, E, F, G, H, A, R(47), 0x106AA070 );
    P( A, B, C, D, E, F, G, H, R(48), 0x19A4C116 );
    P( H, A, B, C, D, E, F, G, R(49), 0x1E376C08 );
    P( G, H, A, B, C, D, E, F, R(50), 0x2748774C );
    P( F, G, H, A, B, C, D, E, R(51), 0x34B0BCB5 );
    P( E, F, G, H, A, B, C, D, R(52), 0x391C0CB3 );
    P( D, E, F, G, H, A, B, C, R(53), 0x4ED8AA4A );
    P( C, D, E, F, G, H, A, B, R(54), 0x5B9CCA4F );
    P( B, C, D, E, F, G, H, A, R(55), 0x682E6FF3 );
    P( A, B, C, D, E, F, G, H, R(56), 0x748F82EE );
    P( H, A, B, C, D, E, F, G, R(57), 0x78A5636F );
    P( G, H, A, B, C, D, E, F, R(58), 0x84C87814 );
    P( F, G, H, A, B, C, D, E, R(59), 0x8CC70208 );
    P( E, F, G, H, A, B, C, D, R(60), 0x90BEFFFA );
    P( D, E, F, G, H, A, B, C, R(61), 0xA4506CEB );
    P( C, D, E, F, G, H, A, B, R(62), 0xBEF9A3F7 );
    P( B, C, D, E, F, G, H, A, R(63), 0xC67178F2 );

    m_Context.m_lState[0] += A;
    m_Context.m_lState[1] += B;
    m_Context.m_lState[2] += C;
    m_Context.m_lState[3] += D;
    m_Context.m_lState[4] += E;
    m_Context.m_lState[5] += F;
    m_Context.m_lState[6] += G;
    m_Context.m_lState[7] += H;
}

/*
 * SHA-256 process buffer
 */
void CSecureHash::Update ( unsigned char* pInput , int nLength )
{
    int fill;
    unsigned long left;

    if( nLength <= 0 )
        return;

    left = m_Context.m_lTotal[0] & 0x3F;
    fill = 64 - left;

    m_Context.m_lTotal[0] += nLength;
    m_Context.m_lTotal[0] &= 0xFFFFFFFF;

    if( m_Context.m_lTotal[0] < (unsigned long) nLength )
	{
        m_Context.m_lTotal[1]++;
	}

    if( left && nLength >= fill )
    {
        memcpy ( (void *) (m_Context.m_pBuffer + left),
                (void *) pInput, fill );
        Process ( m_Context.m_pBuffer );
        pInput += fill;
        nLength  -= fill;
        left = 0;
    }

    while( nLength >= 64 )
    {
        Process ( pInput );
        pInput += 64;
        nLength  -= 64;
    }

    if( nLength > 0 )
    {
        memcpy( (void *) (m_Context.m_pBuffer + left) , (void *) pInput, nLength );
    }
}

static const unsigned char sha2_padding[64] =
{
 0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

/*
 * SHA-256 final digest
 */
void CSecureHash::Finish ( unsigned char pszOutput[32] )
{
    unsigned long last, padn;
    unsigned long high, low;
    unsigned char msglen[8];

    high = ( m_Context.m_lTotal[0] >> 29 )
         | ( m_Context.m_lTotal[1] <<  3 );
    low  = ( m_Context.m_lTotal[0] <<  3 );

    PutUlongBigEndian ( high , msglen , 0 );
    PutUlongBigEndian ( low ,  msglen , 4 );

    last = m_Context.m_lTotal[0] & 0x3F;
    padn = ( last < 56 ) ? ( 56 - last ) : ( 120 - last );

    Update ( (unsigned char *)sha2_padding , padn );
    Update ( msglen , 8 );

    PutUlongBigEndian ( m_Context.m_lState[0] , pszOutput ,  0 );
    PutUlongBigEndian ( m_Context.m_lState[1] , pszOutput ,  4 );
    PutUlongBigEndian ( m_Context.m_lState[2] , pszOutput ,  8 );
    PutUlongBigEndian ( m_Context.m_lState[3] , pszOutput , 12 );
    PutUlongBigEndian ( m_Context.m_lState[4] , pszOutput , 16 );
    PutUlongBigEndian ( m_Context.m_lState[5] , pszOutput , 20 );
    PutUlongBigEndian ( m_Context.m_lState[6] , pszOutput , 24 );

    if ( m_Context.m_nHashType == HASHTYPE_SHA_256 )
	{
        PutUlongBigEndian ( m_Context.m_lState[7] , pszOutput , 28 );
	}
}

/*
 * pszOutput = SHA-256( pInput buffer )
 */
void CSecureHash::OneShot ( unsigned char *pInput , int nLength , unsigned char pszOutput[32] , int nHashType )
{
    Starts ( nHashType );
    Update ( pInput, nLength );
    Finish ( pszOutput );

    memset( &m_Context , 0 , sizeof( CSha2Context ) );
}

/*
 * pszOutput = SHA-256( file contents )
 */
int CSecureHash::OneShotFromFile ( char* pszPath , unsigned char pszOutput[32] , int nHashType )
{
    FILE* f;
    size_t n;
    unsigned char buf[1024];

    if ( ( f = fopen ( pszPath, "rb" ) ) == NULL )
	{
        return( 1 );
	}

    Starts ( nHashType );

    while ( ( n = fread( buf, 1, sizeof( buf ), f ) ) > 0 )
	{
        Update ( buf , (int) n );
	}

    Finish ( pszOutput );

    memset ( &m_Context , 0 , sizeof ( m_Context ) );

    if( ferror ( f ) != 0 )
    {
        fclose ( f );
        return( 2 );
    }

    fclose ( f );
    return( 0 );
}

/*
 * SHA-256 HMAC context setup
 */
void CSecureHash::HmacStarts ( unsigned char* pKey , int nKeyLength , int nHashType )
{
    int i;
    unsigned char sum[32];

    if ( nKeyLength > 64 )
    {
        OneShot ( pKey , nKeyLength , sum , nHashType );
        nKeyLength = ( nHashType ) ? 28 : 32;
        pKey = sum;
    }

    memset ( m_Context.m_byInPad , 0x36, 64 );
    memset ( m_Context.m_byOutPad , 0x5C, 64 );

    for( i = 0; i < nKeyLength; i++ )
    {
        m_Context.m_byInPad[i] = (unsigned char)( m_Context.m_byInPad[i] ^ pKey[i] );
        m_Context.m_byOutPad[i] = (unsigned char)( m_Context.m_byOutPad[i] ^ pKey[i] );
    }

    Starts ( nHashType );
    Update ( m_Context.m_byInPad , 64 );

    memset ( sum , 0 , sizeof ( sum ) );
}

/*
 * SHA-256 HMAC process buffer
 */
void CSecureHash::HmacUpdate ( unsigned char *pInput , int nLength )
{
    Update ( pInput , nLength );
}

/*
 * SHA-256 HMAC final digest
 */
void CSecureHash::HmacFinish ( unsigned char pszOutput[32] )
{
    int nHashType, hlen;
    unsigned char tmpbuf[32];

    nHashType = m_Context.m_nHashType;
    hlen = ( nHashType == HASHTYPE_SHA_256 ) ? 32 : 28;

    Finish ( tmpbuf );
    Starts ( nHashType );
    Update ( m_Context.m_byOutPad, 64 );
    Update ( tmpbuf, hlen );
    Finish ( pszOutput );

    memset( tmpbuf, 0, sizeof( tmpbuf ) );
}

/*
 * pszOutput = HMAC-SHA-256( hmac pKey, pInput buffer )
 */
void CSecureHash::HmacOneShot ( unsigned char* pKey , int nKeyLength ,
								unsigned char* pInput , int nLength ,
								unsigned char pszOutput[32] , int nHashType )
{
    HmacStarts ( pKey , nKeyLength , nHashType );
    HmacUpdate ( pInput , nLength );
    HmacFinish ( pszOutput );

    memset ( &m_Context , 0 , sizeof ( CSha2Context ) );
}

#if defined(POLARSSL_SELF_TEST)
/*
 * FIPS-180-2 test vectors
 */
static unsigned char sha2_test_buf[3][57] = 
{
    { "abc" },
    { "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq" },
    { "" }
};

static const int sha2_test_buflen[3] =
{
    3, 56, 1000
};

static const unsigned char sha2_test_sum[6][32] =
{
    /*
     * SHA-224 test vectors
     */
    { 0x23, 0x09, 0x7D, 0x22, 0x34, 0x05, 0xD8, 0x22,
      0x86, 0x42, 0xA4, 0x77, 0xBD, 0xA2, 0x55, 0xB3,
      0x2A, 0xAD, 0xBC, 0xE4, 0xBD, 0xA0, 0xB3, 0xF7,
      0xE3, 0x6C, 0x9D, 0xA7 },
    { 0x75, 0x38, 0x8B, 0x16, 0x51, 0x27, 0x76, 0xCC,
      0x5D, 0xBA, 0x5D, 0xA1, 0xFD, 0x89, 0x01, 0x50,
      0xB0, 0xC6, 0x45, 0x5C, 0xB4, 0xF5, 0x8B, 0x19,
      0x52, 0x52, 0x25, 0x25 },
    { 0x20, 0x79, 0x46, 0x55, 0x98, 0x0C, 0x91, 0xD8,
      0xBB, 0xB4, 0xC1, 0xEA, 0x97, 0x61, 0x8A, 0x4B,
      0xF0, 0x3F, 0x42, 0x58, 0x19, 0x48, 0xB2, 0xEE,
      0x4E, 0xE7, 0xAD, 0x67 },

    /*
     * SHA-256 test vectors
     */
    { 0xBA, 0x78, 0x16, 0xBF, 0x8F, 0x01, 0xCF, 0xEA,
      0x41, 0x41, 0x40, 0xDE, 0x5D, 0xAE, 0x22, 0x23,
      0xB0, 0x03, 0x61, 0xA3, 0x96, 0x17, 0x7A, 0x9C,
      0xB4, 0x10, 0xFF, 0x61, 0xF2, 0x00, 0x15, 0xAD },
    { 0x24, 0x8D, 0x6A, 0x61, 0xD2, 0x06, 0x38, 0xB8,
      0xE5, 0xC0, 0x26, 0x93, 0x0C, 0x3E, 0x60, 0x39,
      0xA3, 0x3C, 0xE4, 0x59, 0x64, 0xFF, 0x21, 0x67,
      0xF6, 0xEC, 0xED, 0xD4, 0x19, 0xDB, 0x06, 0xC1 },
    { 0xCD, 0xC7, 0x6E, 0x5C, 0x99, 0x14, 0xFB, 0x92,
      0x81, 0xA1, 0xC7, 0xE2, 0x84, 0xD7, 0x3E, 0x67,
      0xF1, 0x80, 0x9A, 0x48, 0xA4, 0x97, 0x20, 0x0E,
      0x04, 0x6D, 0x39, 0xCC, 0xC7, 0x11, 0x2C, 0xD0 }
};

/*
 * RFC 4231 test vectors
 */
static unsigned char sha2_hmac_test_key[7][26] =
{
    { "\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B\x0B"
      "\x0B\x0B\x0B\x0B" },
    { "Jefe" },
    { "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
      "\xAA\xAA\xAA\xAA" },
    { "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10"
      "\x11\x12\x13\x14\x15\x16\x17\x18\x19" },
    { "\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C\x0C"
      "\x0C\x0C\x0C\x0C" },
    { "" }, /* 0xAA 131 times */
    { "" }
};

static const int sha2_hmac_test_nKeyLength[7] =
{
    20, 4, 20, 25, 20, 131, 131
};

static unsigned char sha2_hmac_test_buf[7][153] =
{
    { "Hi There" },
    { "what do ya want for nothing?" },
    { "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
      "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
      "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
      "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
      "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD" },
    { "\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD"
      "\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD"
      "\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD"
      "\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD"
      "\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD\xCD" },
    { "Test With Truncation" },
    { "Test Using Larger Than Block-Size Key - Hash Key First" },
    { "This is a test using a larger than block-size key "
      "and a larger than block-size data. The key needs to "
      "be hashed before being used by the HMAC algorithm." }
};

static const int sha2_hmac_test_buflen[7] =
{
    8, 28, 50, 50, 20, 54, 152
};

static const unsigned char sha2_hmac_test_sum[14][32] =
{
    /*
     * HMAC-SHA-224 test vectors
     */
    { 0x89, 0x6F, 0xB1, 0x12, 0x8A, 0xBB, 0xDF, 0x19,
      0x68, 0x32, 0x10, 0x7C, 0xD4, 0x9D, 0xF3, 0x3F,
      0x47, 0xB4, 0xB1, 0x16, 0x99, 0x12, 0xBA, 0x4F,
      0x53, 0x68, 0x4B, 0x22 },
    { 0xA3, 0x0E, 0x01, 0x09, 0x8B, 0xC6, 0xDB, 0xBF,
      0x45, 0x69, 0x0F, 0x3A, 0x7E, 0x9E, 0x6D, 0x0F,
      0x8B, 0xBE, 0xA2, 0xA3, 0x9E, 0x61, 0x48, 0x00,
      0x8F, 0xD0, 0x5E, 0x44 },
    { 0x7F, 0xB3, 0xCB, 0x35, 0x88, 0xC6, 0xC1, 0xF6,
      0xFF, 0xA9, 0x69, 0x4D, 0x7D, 0x6A, 0xD2, 0x64,
      0x93, 0x65, 0xB0, 0xC1, 0xF6, 0x5D, 0x69, 0xD1,
      0xEC, 0x83, 0x33, 0xEA },
    { 0x6C, 0x11, 0x50, 0x68, 0x74, 0x01, 0x3C, 0xAC,
      0x6A, 0x2A, 0xBC, 0x1B, 0xB3, 0x82, 0x62, 0x7C,
      0xEC, 0x6A, 0x90, 0xD8, 0x6E, 0xFC, 0x01, 0x2D,
      0xE7, 0xAF, 0xEC, 0x5A },
    { 0x0E, 0x2A, 0xEA, 0x68, 0xA9, 0x0C, 0x8D, 0x37,
      0xC9, 0x88, 0xBC, 0xDB, 0x9F, 0xCA, 0x6F, 0xA8 },
    { 0x95, 0xE9, 0xA0, 0xDB, 0x96, 0x20, 0x95, 0xAD,
      0xAE, 0xBE, 0x9B, 0x2D, 0x6F, 0x0D, 0xBC, 0xE2,
      0xD4, 0x99, 0xF1, 0x12, 0xF2, 0xD2, 0xB7, 0x27,
      0x3F, 0xA6, 0x87, 0x0E },
    { 0x3A, 0x85, 0x41, 0x66, 0xAC, 0x5D, 0x9F, 0x02,
      0x3F, 0x54, 0xD5, 0x17, 0xD0, 0xB3, 0x9D, 0xBD,
      0x94, 0x67, 0x70, 0xDB, 0x9C, 0x2B, 0x95, 0xC9,
      0xF6, 0xF5, 0x65, 0xD1 },

    /*
     * HMAC-SHA-256 test vectors
     */
    { 0xB0, 0x34, 0x4C, 0x61, 0xD8, 0xDB, 0x38, 0x53,
      0x5C, 0xA8, 0xAF, 0xCE, 0xAF, 0x0B, 0xF1, 0x2B,
      0x88, 0x1D, 0xC2, 0x00, 0xC9, 0x83, 0x3D, 0xA7,
      0x26, 0xE9, 0x37, 0x6C, 0x2E, 0x32, 0xCF, 0xF7 },
    { 0x5B, 0xDC, 0xC1, 0x46, 0xBF, 0x60, 0x75, 0x4E,
      0x6A, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xC7,
      0x5A, 0x00, 0x3F, 0x08, 0x9D, 0x27, 0x39, 0x83,
      0x9D, 0xEC, 0x58, 0xB9, 0x64, 0xEC, 0x38, 0x43 },
    { 0x77, 0x3E, 0xA9, 0x1E, 0x36, 0x80, 0x0E, 0x46,
      0x85, 0x4D, 0xB8, 0xEB, 0xD0, 0x91, 0x81, 0xA7,
      0x29, 0x59, 0x09, 0x8B, 0x3E, 0xF8, 0xC1, 0x22,
      0xD9, 0x63, 0x55, 0x14, 0xCE, 0xD5, 0x65, 0xFE },
    { 0x82, 0x55, 0x8A, 0x38, 0x9A, 0x44, 0x3C, 0x0E,
      0xA4, 0xCC, 0x81, 0x98, 0x99, 0xF2, 0x08, 0x3A,
      0x85, 0xF0, 0xFA, 0xA3, 0xE5, 0x78, 0xF8, 0x07,
      0x7A, 0x2E, 0x3F, 0xF4, 0x67, 0x29, 0x66, 0x5B },
    { 0xA3, 0xB6, 0x16, 0x74, 0x73, 0x10, 0x0E, 0xE0,
      0x6E, 0x0C, 0x79, 0x6C, 0x29, 0x55, 0x55, 0x2B },
    { 0x60, 0xE4, 0x31, 0x59, 0x1E, 0xE0, 0xB6, 0x7F,
      0x0D, 0x8A, 0x26, 0xAA, 0xCB, 0xF5, 0xB7, 0x7F,
      0x8E, 0x0B, 0xC6, 0x21, 0x37, 0x28, 0xC5, 0x14,
      0x05, 0x46, 0x04, 0x0F, 0x0E, 0xE3, 0x7F, 0x54 },
    { 0x9B, 0x09, 0xFF, 0xA7, 0x1B, 0x94, 0x2F, 0xCB,
      0x27, 0x63, 0x5F, 0xBC, 0xD5, 0xB0, 0xE9, 0x44,
      0xBF, 0xDC, 0x63, 0x64, 0x4F, 0x07, 0x13, 0x93,
      0x8A, 0x7F, 0x51, 0x53, 0x5C, 0x3A, 0x35, 0xE2 }
};

/*
 * Checkup routine
 */
int CSecureHash::SelfTest ( int verbose )
{
    int i, j, k, buflen;
    unsigned char buf[1024];
    unsigned char sha2sum[32];
//    CSha2Context pSha2Context;

    for( i = 0; i < 6; i++ )
    {
        j = i % 3;
        k = i < 3;

        if( verbose != 0 )
            TRACE( "  SHA-%d test #%d: ", 256 - k * 32, j + 1 );

        Starts ( k );

        if( j == 2 )
        {
            memset( buf, 'a', buflen = 1000 );

            for( j = 0; j < 1000; j++ )
			{
                Update ( buf, buflen );
			}
        }
        else
		{
            Update ( sha2_test_buf[j] , sha2_test_buflen[j] );
		}

        Finish ( sha2sum );

        if( memcmp ( sha2sum, sha2_test_sum[i], 32 - k * 4 ) != 0 )
        {
            if( verbose != 0 )
                TRACE( "failed\n" );

            return( 1 );
        }

        if( verbose != 0 )
            TRACE( "passed\n" );
    }

    if( verbose != 0 )
        TRACE( "\n" );

    for( i = 0; i < 14; i++ )
    {
        j = i % 7;
        k = i < 7;

        if( verbose != 0 )
            TRACE( "  HMAC-SHA-%d test #%d: ", 256 - k * 32, j + 1 );

        if( j == 5 || j == 6 )
        {
            memset( buf, '\xAA', buflen = 131 );
            HmacStarts ( buf, buflen, k );
        }
        else
            HmacStarts( sha2_hmac_test_key[j],
                                    sha2_hmac_test_nKeyLength[j], k );

        HmacUpdate( sha2_hmac_test_buf[j],
                                sha2_hmac_test_buflen[j] );

        HmacFinish( sha2sum );

        buflen = ( j == 4 ) ? 16 : 32 - k * 4;

        if( memcmp( sha2sum, sha2_hmac_test_sum[i], buflen ) != 0 )
        {
            if( verbose != 0 )
                TRACE( "failed\n" );

            return( 1 );
        }

        if( verbose != 0 )
            TRACE( "passed\n" );
    }

    if( verbose != 0 )
        TRACE( "\n" );

    return( 0 );
}
#endif