/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<NoLockList.c>	2005-03-13,13:41

*/

#include "NoLockList.h"


/*******************************************************************************/
/* ChainUnit member
*/
/* ChainUnit setting:
	setting data pointer,
	connect with previous and next chain.
*/
HS_RESULT new_ChainUnit( ChainUnit *pObj, void *pPrev, void *pData, void *pNext )
{
	if( pObj == HS_NULL ) return HS_ERR_NULL_PARAM;

	pObj->prev = pPrev;
	pObj->data = pData;
	pObj->next = pNext;

	if( pPrev != HS_NULL ) ((ChainUnit*)pPrev)->next = pObj;
	if( pNext != HS_NULL ) ((ChainUnit*)pNext)->prev = pObj;

	return HS_OK;
}


/* ChainUnit destroy:
	HSFree #data# variable,
	disconnect previous and next chain.
*/
HS_RESULT delete_ChainUnit( ChainUnit *pObj, HS_RESULT (*pDeleteFunction)(void*) )
{
	ChainUnit *tPrev = HS_NULL;
	ChainUnit *tNext = HS_NULL;

	if( pObj == HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->data != HS_NULL )
	{
		if( pDeleteFunction != HS_NULL ) pDeleteFunction( pObj->data );
		HSFree( pObj->data );
	}

	tPrev = (ChainUnit *)(pObj->prev);
	tNext = (ChainUnit *)(pObj->next);

	if( tPrev != HS_NULL ) tPrev->next = pObj->next;
	if( tNext != HS_NULL ) tNext->prev = pObj->prev;
	return HS_OK;
}





/*******************************************************************************/
/* NoLockList member
*/
/* NoLockList initialize,
   set delete function of #data# variable.
*/
HS_RESULT new_NoLockList( NoLockList *pObj, HS_RESULT (*pDeleteFunction)(void*) )
{
	if( pObj == HS_NULL ) return HS_ERR_NULL_PARAM;

	pObj->units = HS_NULL;
	pObj->size = 0;
	pObj->DeleteFunction = pDeleteFunction;

	return HS_OK;
}


/* NoLockList clearing
*/
HS_RESULT delete_NoLockList( NoLockList *pObj )
{
	return NoLockList_Clear( pObj );
}


/* NoLockList clearing:
	delete all #data# variables and chains,
	NoLockList be the state like doing #new_NoLockList()#.
*/
HS_RESULT NoLockList_Clear( NoLockList *pObj )
{
	HS_UINT i;
	ChainUnit *deleter = HS_NULL;
	ChainUnit *rounder = HS_NULL;

	if( pObj == HS_NULL ) return HS_ERR_NULL_PARAM;

	rounder = pObj->units;
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323_CONFLICT;
		/*if( rounder->data == HS_NULL ) return HS_ERR_H323_CONFLICT;*/

		delete_ChainUnit( rounder, pObj->DeleteFunction );

		deleter = rounder;
		rounder = rounder->next;
		HSFree( deleter );
	}

	if( rounder != HS_NULL ) return HS_ERR_H323_CONFLICT;

	pObj->units = HS_NULL;
	pObj->size = 0;
	return HS_OK;
}


/* Data attach on the list:
	making ChainUnit(connecting with list) and #pData# put it on.
	new ChainUnit attached on tail of list.
*/
HS_RESULT NoLockList_AttachData( NoLockList *pObj, void *pData )
{
	HS_UINT i;
	HS_RESULT tRet;
	ChainUnit *tChain = HS_NULL;
	ChainUnit *rounder = HS_NULL;

	if( pObj == HS_NULL || pData == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( (tChain=(ChainUnit*)HSMalloc(sizeof(ChainUnit))) == HS_NULL ) return HS_ERR_H323_MALLOC;

	if( pObj->size == 0 )
	{
		if( pObj->units != HS_NULL )
		{
			HSFree(tChain);
			return HS_ERR_H323_CONFLICT;
		}
		if( (tRet=new_ChainUnit(tChain,HS_NULL,pData,HS_NULL)) != HS_OK )
		{
			HSFree(tChain);
			return tRet;
		}
		
		pObj->units = tChain;
		pObj->size = 1;
		return HS_OK;
	}

	rounder = pObj->units;
	for( i=0; i< (pObj->size-1); i++ )
	{
		if(rounder == HS_NULL)
		{
			HSFree(tChain);
			return HS_ERR_H323_CONFLICT;
		}
		rounder = rounder->next;
	}

	if( rounder == HS_NULL )
	{
		HSFree(tChain);
		return HS_ERR_H323_CONFLICT;
	}

	if( (tRet=new_ChainUnit(tChain,rounder,pData,HS_NULL)) != HS_OK )
	{
		HSFree(tChain);
		return tRet;
	}

	pObj->size++;
	return HS_OK;
}


/* get ChainUnit pointer by pIndex,
   NO FREE and NO DETACH from list.
*/
ChainUnit *NoLockList_GetChain( NoLockList *pObj, HS_UINT pIndex )
{
	HS_UINT i;
	ChainUnit *rounder = HS_NULL;

	if( pObj == HS_NULL ) return HS_NULL;
	if( pObj->size == 0 || pIndex > (pObj->size - 1) ) return HS_NULL;

	rounder = pObj->units;
	for( i=0; i<pIndex; i++ )
	{
		if( rounder == HS_NULL ) return HS_NULL;
		rounder = (ChainUnit*)(rounder->next);
	}

	return rounder;
}


/* disconnect chain and HSFree #data# in ChainUnit and HSFree ChainUnit
*/
HS_RESULT NoLockList_DeleteChain( NoLockList *pObj, ChainUnit *pChain )
{
	HS_RESULT tRet;
	ChainUnit *tChain = HS_NULL;

	if( pObj == HS_NULL || pChain == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pObj->size == 0 ) return HS_ERR_H323_CONFLICT;

	if( pChain == pObj->units )
	{
		if( pObj->size == 1 )
		{
			if( (tRet=delete_ChainUnit(pChain, pObj->DeleteFunction)) != HS_OK ) return tRet;
			HSFree(pChain);
			pObj->units = HS_NULL;
			pObj->size = 0;
		}
		else
		{
			if( pChain->next == HS_NULL ) return HS_ERR_H323_CONFLICT;
			tChain = pChain->next;

			if( (tRet=delete_ChainUnit(pChain, pObj->DeleteFunction)) != HS_OK ) return tRet;
			HSFree(pChain);
			pObj->units = tChain;
			pObj->size--;
		}

		return HS_OK;
	}

	if( (tRet=delete_ChainUnit(pChain, pObj->DeleteFunction)) != HS_OK ) return tRet;
	HSFree(pChain);
	pObj->size--;
	return HS_OK;
}


/* detach(disconnect from list) ChainUnit by pIndex,
   HSFree ChainUnit but return #data# of ChainUnit
*/
void *NoLockList_DetachData(NoLockList *pObj, HS_UINT pIndex)
{
	void *result = HS_NULL;
	HS_RESULT tRet = HS_OK;
	ChainUnit *tChain = HS_NULL;

	if( (tChain=NoLockList_GetChain(pObj, pIndex)) == HS_NULL ) return HS_NULL;

	result = tChain->data;
	tChain->data = HS_NULL;

	if( pIndex == 0 )
		pObj->units = tChain->next;
	
	delete_ChainUnit(tChain,HS_NULL);
	pObj->size--;

	if( pObj->size == 0 )
		pObj->units = HS_NULL;

	return result;
}


/* get #data# pointer by pIndex,
   NO FREE and NO DETACH from list.
*/
void *NoLockList_GetData( NoLockList *pObj, HS_UINT pIndex )
{
	ChainUnit *tChain = HS_NULL;

	if( (tChain=NoLockList_GetChain(pObj,pIndex)) == HS_NULL ) return HS_NULL;
	return tChain->data;
}

