/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<ppHandle.c>	2006-03-14,21:01

*/

#include "ppHandle.h"

#include "ppCbH323.h"
#include "ppCbSip.h"





/*
 *
 * global variables
 *
 */

static HS_QID gCoreQid = HS_INVALID_QID;
static PPOption gOption;





/*
 *
 * PPCore member functions
 *
 */
HS_RESULT new_H323Core(void *pObject)
{
	H323Core *pObj = (H323Core*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mStack = HS_INVALID_STACK_HANDLE;
	pObj->mIsPrime = FALSE;
	pObj->mIsRegisted = FALSE;
	pObj->mCall = HS_INVALID_HANDLE;
	return HS_OK;
}


HS_RESULT delete_H323Core(void *pObject)
{
	H323Core *pObj = (H323Core*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mStack != HS_INVALID_STACK_HANDLE )
	{
		if( pObj->mCall != HS_INVALID_HANDLE )
			HapiCommandRemoveCall(pObj->mStack,pObj->mCall,e_CallCloseReason_Undefined);
		pObj->mCall = HS_INVALID_HANDLE;

		HapiStopStack(pObj->mStack);
		pObj->mStack = HS_INVALID_STACK_HANDLE;
	}

	return HS_OK;
}



HS_RESULT new_SipCore(void *pObject)
{
	SipCore *pObj = (SipCore*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mStack = HS_INVALID_STACK_HANDLE;
	pObj->mUaPrime = HS_INVALID_HANDLE;
	pObj->mUaSecond = HS_INVALID_HANDLE;
	pObj->mIsPrime = FALSE;
	pObj->mIsRegistedPrime = FALSE;
	pObj->mIsRegistedSecond = FALSE;
	pObj->mCall = HS_INVALID_HANDLE;
	return HS_OK;
}


HS_RESULT delete_SipCore(void *pObject)
{
	SipCore *pObj = (SipCore*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mStack != HS_INVALID_STACK_HANDLE )
	{
		if( pObj->mCall != HS_INVALID_HANDLE )
			SapiCommandRemoveCall(pObj->mStack,pObj->mCall,e_SipResponse_Ok);
		pObj->mCall = HS_INVALID_HANDLE;

		SapiStopStack(pObj->mStack);
		pObj->mStack = HS_INVALID_STACK_HANDLE;
	}

	return HS_OK;
}



HS_RESULT new_PPCore(void *pObject,HWND pHwnd,HS_UCHAR *pLocalIp,PPOption pOption)
{
	PPCore *pObj = (PPCore*)pObject;

	if( pObj==NULL || pLocalIp==NULL ) return HS_ERR_NULL_PARAM;

	HS_THREAD_INIT(pObj,"ppcore");
	pObj->mHwnd = pHwnd;
	pObj->mState = e_PPState_Idle;
	new_PPOption(&(pObj->mOption));
	memcpy(&(pObj->mOption),&pOption,sizeof(PPOption));

	pObj->mRtp = HS_INVALID_QID;
	new_H323Core(&(pObj->mH323));
	new_SipCore(&(pObj->mSip));

	memcpy(pObj->mLocalIp,pLocalIp,4);
	memset(pObj->mDial,0,128);
	pObj->mDialPos = 0;
	return HS_OK;
}


PPCore *newm_PPCore(HWND pHwnd,HS_UCHAR *pLocalIp,PPOption pOption)
{
	PPCore *tResult = NULL;

	if( (tResult=(PPCore*)HSMalloc(sizeof(PPCore)))==NULL ) return NULL;
	if( new_PPCore(tResult,pHwnd,pLocalIp,pOption) != HS_OK )
	{
		HSFree(tResult);
		return NULL;
	}

	return tResult;
}


HS_RESULT delete_PPCore(void *pObject)
{
	PPCore *pObj = (PPCore*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	pObj->mHwnd = NULL;
	pObj->mState = e_PPState_Idle;
	delete_PPOption(&(pObj->mOption));

	if( pObj->mRtp != HS_INVALID_QID )
	{
		PPMedia_Stop(pObj->mRtp);
		pObj->mRtp = HS_INVALID_QID;
	}
	delete_H323Core(&(pObj->mH323));
	delete_SipCore(&(pObj->mSip));
	return HS_OK;
}


HS_RESULT deletem_PPCore(void *pObject)
{
	PPCore *pObj = (PPCore*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	delete_PPCore(pObj);
	HSFree(pObj);
	return HS_OK;
}





/*
 *
 * utilities
 *
 */

HS_RESULT ppHandle_GetOption(PPOption *pOption)
{
	if( pOption==NULL ) return HS_ERR_NULL_PARAM;

	memcpy(pOption,&gOption,sizeof(PPOption));
	return HS_OK;
}


void ppHandle_SetOption(PPOption pOption)
{
	gOption = pOption;
}


HS_QID ppHandle_GetCoreQid()
{
	return gCoreQid;
}


void ppHandle_SetCoreQid(HS_QID pQid)
{
	gCoreQid = pQid;
}





/*
 *
 * H.323 stack wrapping functions
 *
 */
HS_STACK_HANDLE PPStartH323Stack(PPOption *pOption,HS_UCHAR *pLocalIp)
{
	IEndpoint *tEndpoint = NULL;
	HS_STACK_HANDLE tStackHandle = HS_INVALID_STACK_HANDLE;

	if( pOption==NULL || pLocalIp==NULL ) return HS_INVALID_STACK_HANDLE;

	if( (tEndpoint=newm_IEndpoint())==NULL )
		return HS_INVALID_STACK_HANDLE;

	new_IEndpoint(tEndpoint);
	/* set type
	*/
#if 0
	tEndpoint->type = e_EndpointType_gateway;
#else
	tEndpoint->type = e_EndpointType_terminal;
#endif
	/* set addresses
	*/
	HapiSetRasAddress(tEndpoint,pLocalIp,1729);
	HapiSetCallSignalingAddress(tEndpoint,pLocalIp,1720);
	/* set aliases
	*/
	if( ! is_blank(pOption->mH323.mH323Id) )
		HapiAddEndpointAliasH323Id(tEndpoint,pOption->mH323.mH323Id);
	if( ! is_blank(pOption->mH323.mPhoneNumber) )
		HapiAddEndpointAliasDial(tEndpoint,pOption->mH323.mPhoneNumber);
#if 0
	HapiAddEndpointAliasUrlId	(tEndpoint, "www.voiper.co.kr");
	HapiAddEndpointAliasEmailId	(tEndpoint, "voiper@voiper.co.kr");
#endif

	/* set product id
	*/
	strcpy(tEndpoint->productId,"cryptack_h323_endpoint_by_voiper");
	strcpy(tEndpoint->versionId,"0.001");

	tEndpoint->gatekeeper.ttl			= pOption->mH323.mTtl;
	tEndpoint->gatekeeper.ttlForce		= FALSE;
	tEndpoint->gatekeeper.attemptTime	= 5;
	tEndpoint->gatekeeper.isGRQ			= pOption->mH323.mUseGRQ;
	tEndpoint->gatekeeper.isUseLightRRQ = pOption->mH323.mUseLightWeightRRQ;

	/* set callback functions
	*/
	tEndpoint->CallbackException			= AppOnException;
	tEndpoint->CallbackAsnDecodingError		= AppOnAsnDecodingError;
	tEndpoint->CallbackCallIncoming			= AppOnCallIncoming;
	tEndpoint->CallbackCallRemoved			= AppOnCallRemoved;

	tEndpoint->CallbackRegistrationTimeout	= AppOnRegistrationTimeout;
	tEndpoint->CallbackRasMessageTimeout	= AppOnRasMessageTimeout;

	tEndpoint->CallbackReceiveRasRawData	= AppOnReceiveRasRawData;
	tEndpoint->CallbackReceiveRasMessage	= AppOnReceiveRasMessage;
	tEndpoint->CallbackReceiveQ931RawData	= AppOnReceiveQ931RawData;
	tEndpoint->CallbackReceiveQ931Message	= AppOnReceiveQ931Message;
	tEndpoint->CallbackReceiveH245RawData	= AppOnReceiveH245RawData;
	tEndpoint->CallbackReceiveH245Message	= AppOnReceiveH245Message;
	tEndpoint->CallbackSendRasMessage		= AppOnSendRasMessage;
	tEndpoint->CallbackSendRasRawData		= AppOnSendRasRawData;
	tEndpoint->CallbackSendQ931Message		= AppOnSendQ931Message;
	tEndpoint->CallbackSendQ931RawData		= AppOnSendQ931RawData;
	tEndpoint->CallbackSendH245Message		= AppOnSendH245Message;
	tEndpoint->CallbackSendH245RawData		= AppOnSendH245RawData;

	/* ! NOTE:	DO NOT put the function pointer,
				if you don't check faststart at application.
	tEndpoint->CallbackCheckFastStart		= AppOnCheckFastStart;
	*/
	/* ! NOTE:	DO NOT put the function pointer,
				if you don't check TCS at application.
	tEndpoint->CallbackCheckTerminalCapabilitySet = AppOnCheckTerminalCapabilitySet;
	*/
	tEndpoint->CallbackOpenReceiveMedia		= AppOnOpenReceiveMedia;
	tEndpoint->CallbackOpenSendMedia		= AppOnOpenSendMedia;
	tEndpoint->CallbackCloseReceiveMedia	= AppOnCloseReceiveMedia;
	tEndpoint->CallbackCloseSendMedia		= AppOnCloseSendMedia;

	if( (tStackHandle=HapiStartStack(tEndpoint))==HS_INVALID_STACK_HANDLE )
	{
		/* DO NOT handle IEndpoint after calling HapiStartStack
		 *
		delete_IEndpoint(tEndpoint);
		HSFree(tEndpoint);
		*/
		return HS_INVALID_STACK_HANDLE;
	}
	
	return tStackHandle;
}


HS_CALL_HANDLE PPStartH323Call(PPOption *pOption,HS_STACK_HANDLE pHandle,HS_UCHAR *pLocalIp,char *pDest)
{
	ICall *tCall = NULL;

	if( pOption==NULL || pHandle==HS_INVALID_STACK_HANDLE || pLocalIp==NULL || pDest==HS_NULL )
		return HS_INVALID_CALL_HANDLE;
	if( (tCall=HapiCallOpen())==NULL )
		return HS_INVALID_CALL_HANDLE;

	/* set dest address
	*/
	if( IsIp(pDest)==TRUE )
	{
		HS_UINT lDestIp;
		HS_UCHAR tDestIp[4];

		lDestIp = htonl(inet_addr(pDest));
		Uint2Uchar(tDestIp, &lDestIp);

		Tsap2ASNH225TransportAddress(&(tCall->dcsaTsap),tDestIp,1720);
		strcpy(tCall->calledParty,"0123456789");
	}
	else if( IsE164(pDest) )
	{
		HapiAddCalledAliasDial(tCall,pDest);
		strcpy(tCall->calledParty,pDest);
	}
	else
	{
		HapiAddCalledAliasH323Id(tCall,pDest);
		strcpy(tCall->calledParty,pDest);
	}

#if 0
	/* set source address
	*/
	HapiAddCallingAliasH323Id(tCall, "my_h323id");
	HapiAddCallingAliasDial(tCall, "023337777" );
#endif

	/* set source call tsap
	*/
	Tsap2ASNH225TransportAddress(&(tCall->scsaTsap),pLocalIp,1720);

	/* set options
	*/
	tCall->isTunneling = pOption->mH323.mTunneling;
	tCall->isFastStart = pOption->mH323.mFastStart;
	tCall->canMapAlias = pOption->mH323.mCanMapAlias;

	if( pOption->mH323.mEarlyH245==TRUE )
		tCall->h245ListenPoint = e_H245ListenPoint_Setup;
	else
		tCall->h245ListenPoint = e_H245ListenPoint_Connect;

	/* set capabilities
	*/
	if( PP_INCLUDE_G711A(pOption->mGeneral.mCodec) )
		HapiAddG711Alaw64k(tCall,HS_H323_AUDIO_GROUP,10);
	if( PP_INCLUDE_G711U(pOption->mGeneral.mCodec) )
		HapiAddG711Ulaw64k(tCall,HS_H323_AUDIO_GROUP,10);
	if( PP_INCLUDE_G729(pOption->mGeneral.mCodec) )
	{
		HapiAddG729(tCall,HS_H323_AUDIO_GROUP,10);
		HapiAddG729AnnexA(tCall,HS_H323_AUDIO_GROUP,10);
		HapiAddG729AnnexAwAnnexB(tCall,HS_H323_AUDIO_GROUP,10);
	}

	/* set calling party number
	*/
	if( ! is_blank(pOption->mH323.mCallingPartyNumber) )
		strcpy(tCall->callingParty,pOption->mH323.mCallingPartyNumber);

#if 0
	/* set called party number
	*/
	strcpy(tCall->calledParty,"0123456789");
#endif

	if( HapiCommandMakeCall(pHandle,tCall) != HS_OK ) return HS_INVALID_CALL_HANDLE;
	return tCall->handle;
}


HS_RESULT PPH323ChangeEndpointAlias(HS_STACK_HANDLE pHandle,PPOption *pOption)
{
	HS_RESULT tRet = HS_OK;
	NoLockList *tAliases = HS_NULL;

	if( pHandle==HS_INVALID_STACK_HANDLE || pOption==NULL ) return HS_ERR_NULL_PARAM;
	if( (tAliases=HapiMakeAliasList())==NULL ) return HS_ERR_H323_MALLOC;

	if( ! is_blank(pOption->mH323.mH323Id) )
	{
		if( (tRet=HapiAddAliasH323Id(tAliases,pOption->mH323.mH323Id)) != HS_OK )
		{
			delete_NoLockList(tAliases);
			HSFree(tAliases);
			return tRet;
		}
	}
	if( ! is_blank(pOption->mH323.mPhoneNumber) )
	{
		if( (tRet=HapiAddAliasDial(tAliases,pOption->mH323.mPhoneNumber)) != HS_OK )
		{
			delete_NoLockList(tAliases);
			HSFree(tAliases);
			return tRet;
		}
	}

	return HapiCommandChangeEndpointAliases(pHandle,tAliases);
}


HS_RESULT PPH323DtmfSignalByH245(HS_STACK_HANDLE pStackHandle,HS_CALL_HANDLE pCallHandle,char pDtmf,HS_USHORT pDur)
{
	ControlMsg *tUserInput = NULL;

	if( (tUserInput=UtilH323MakeDtmfH245UserInputIndication(pDtmf,pDur))==NULL )
		return HS_ERR;
	return HapiCommandH245UserInputIndication(pStackHandle,pCallHandle,tUserInput);
}





/*
 *
 * sip stack wrapping functions
 *
 */

HS_STACK_HANDLE PPStartSipStack(PPOption *pOption,HS_UCHAR *pLocalIp)
{
	char tIpString[32];
	IStack *tResult = NULL;

	if( pOption==NULL || pLocalIp==NULL ) return HS_INVALID_STACK_HANDLE;

	sprintf(tIpString,"%u.%u.%u.%u",pLocalIp[0],pLocalIp[1],pLocalIp[2],pLocalIp[3]);
	if( (tResult=(IStack*)SapiMakeStackHandle(tIpString,"cryptack_sip_ua_by_voiper"))==HS_INVALID_STACK_HANDLE )
		return HS_INVALID_STACK_HANDLE;

	/* set callback functions
	*/
	tResult->CallbackBnfDecodingError = AppCallbackBnfDecodingError;
	tResult->CallbackCallIncoming = AppCallbackCallIncoming;
	tResult->CallbackCallEnded = AppCallbackCallEnded;
	tResult->CallbackCallRemoved = AppCallbackCallRemoved;
	tResult->CallbackCallToRemove = AppCallbackCallToRemove;
	tResult->CallbackOutbandDtmf = AppCallbackOutbandDtmf;
	tResult->CallbackInstantMessage = AppCallbackInstantMessage;
	tResult->CallbackCheckSdp = AppCallbackCheckSdp;
	tResult->CallbackCloseMedia = AppCallbackCloseMedia;
	tResult->CallbackOpenMedia = AppCallbackOpenMedia;
	tResult->CallbackReceiveMessage = AppCallbackReceiveMessage;
	tResult->CallbackReceiveMessageToResponse = AppCallbackReceiveMessageToResponse;
	tResult->CallbackReceiveRawData = AppCallbackReceiveRawData;
	tResult->CallbackRegisted = AppCallbackRegisted;
	tResult->CallbackRegisterFail = AppCallbackRegisterFail;
	tResult->CallbackSendMessage = AppCallbackSendMessage;
	tResult->CallbackSendRawData = AppCallbackSendRawData;
	tResult->CallbackStackInformation = AppCallbackStackInformation;
	tResult->CallbackUaRemoved = AppCallbackUaRemoved;

	if( SapiStartStack(tResult,pOption->mSip.mLocalUdpPort,pOption->mSip.mLocalTcpPort) != HS_OK )
	{
		/* DO NOT handle IStack after calling SapiStartStack
		 *
		SapiRemoveStackHandle(tResult);
		*/
		return HS_INVALID_STACK_HANDLE;
	}

	return tResult;
}


static HS_UA_HANDLE PPAddSipUaEx(
	HS_STACK_HANDLE pHandle,
	char *pProxy,char *pOutboundProxy,HS_USHORT pProxyPort,HS_UINT pExpires,
	char *pDisplayName,char *pUserName,HS_UCHAR *pLocalIp,HS_USHORT pLocalPort,
	char *pAuthUser,char *pPassword
)
{
	IUa *tUa = NULL;
	char tIpString[32];
	HS_UA_HANDLE tResult = HS_INVALID_HANDLE;

	if( pHandle==HS_INVALID_STACK_HANDLE || pProxy==NULL || pUserName==NULL || pLocalIp==NULL || pExpires==0 )
		return HS_INVALID_HANDLE;
	if( (tUa=SapiMakeUa(pProxy,pOutboundProxy,pProxyPort,pExpires))==NULL )
		return HS_INVALID_HANDLE;

	sprintf(tIpString,"%u.%u.%u.%u",pLocalIp[0],pLocalIp[1],pLocalIp[2],pLocalIp[3]);
	if( SapiAddContactToUa(pHandle,tUa,pDisplayName,pUserName,tIpString,pLocalPort,HS_UINT_MAX)!=HS_OK )
	{
		SapiRemoveUa(tUa);
		return HS_INVALID_HANDLE;
	}
	if( pPassword != NULL )
	{
		if( SapiSetPasswordToUa(tUa,pPassword)!=HS_OK )
		{
			SapiRemoveUa(tUa);
			return HS_INVALID_HANDLE;
		}
	}
	if( pAuthUser != NULL )
	{
		if( SapiSetAuthUserToUa(tUa,pAuthUser) != HS_OK )
		{
			SapiRemoveUa(tUa);
			return HS_INVALID_HANDLE;
		}
	}
	if( (tResult=SapiCommandAddUa(pHandle,tUa))==HS_INVALID_HANDLE )
	{
		SapiRemoveUa(tUa);
		return HS_INVALID_HANDLE;
	}

	return tResult;
}


HS_UA_HANDLE PPAddSipUa(PPOption *pOption,HS_STACK_HANDLE pHandle,HS_UCHAR *pLocalIp,BOOL pIsPrime)
{
	if( pHandle==HS_INVALID_STACK_HANDLE || pLocalIp==NULL || pOption==NULL )
		return HS_INVALID_HANDLE;
	if( is_blank(pOption->mSip.mUserName) )
		return HS_INVALID_HANDLE;

	/* make ua object
	*/
	if( pIsPrime==TRUE )
	{
		if( is_blank(pOption->mSip.mPrimeProxy) )
			return HS_INVALID_HANDLE;

		return PPAddSipUaEx(pHandle,
			pOption->mSip.mPrimeProxy,
			is_blank(pOption->mSip.mOutboundProxy)? NULL:pOption->mSip.mOutboundProxy,
			pOption->mSip.mPrimePort,pOption->mSip.mTtl,
			is_blank(pOption->mSip.mDisplayName)? NULL:pOption->mSip.mDisplayName,
			pOption->mSip.mUserName,pLocalIp,pOption->mSip.mLocalUdpPort,
			is_blank(pOption->mSip.mAuthId)? NULL:pOption->mSip.mAuthId,
			is_blank(pOption->mSip.mPassword)? NULL:pOption->mSip.mPassword
		);
	}

	if( is_blank(pOption->mSip.mSecondProxy) )
		return HS_INVALID_HANDLE;

	return PPAddSipUaEx(pHandle,
		pOption->mSip.mSecondProxy,
		is_blank(pOption->mSip.mOutboundProxy)? NULL:pOption->mSip.mOutboundProxy,
		pOption->mSip.mSecondPort,pOption->mSip.mTtl,
		is_blank(pOption->mSip.mDisplayName)? NULL:pOption->mSip.mDisplayName,
		pOption->mSip.mUserName,pLocalIp,pOption->mSip.mLocalUdpPort,
		is_blank(pOption->mSip.mAuthId)? NULL:pOption->mSip.mAuthId,
		is_blank(pOption->mSip.mPassword)? NULL:pOption->mSip.mPassword
	);
}


HS_CALL_HANDLE PPStartSipCall(PPOption *pOption,HS_STACK_HANDLE pHandle,HS_UA_HANDLE pUa,char *pDest)
{
	IDialogue *tDialogue = NULL;
	HS_CALL_HANDLE tResult = HS_INVALID_HANDLE;

	if( pOption==NULL || pHandle==HS_INVALID_STACK_HANDLE || pUa==HS_INVALID_HANDLE || pDest==NULL )
		return HS_INVALID_HANDLE;
	if( (tDialogue=SapiMakeDialogue(NULL,pDest,NULL,HS_INVALID_TSAP_PORT,FALSE))==NULL )
		return HS_INVALID_HANDLE;

	if( PP_INCLUDE_G711A(pOption->mGeneral.mCodec) )
		SapiAddMediaCapabilityToDialogue(pHandle,tDialogue,"audio",e_RtpPayloadType_Pcma);
	if( PP_INCLUDE_G711U(pOption->mGeneral.mCodec) )
		SapiAddMediaCapabilityToDialogue(pHandle,tDialogue,"audio",e_RtpPayloadType_Pcmu);
	if( PP_INCLUDE_G729(pOption->mGeneral.mCodec)  )
		SapiAddMediaCapabilityToDialogue(pHandle,tDialogue,"audio",e_RtpPayloadType_G729);
	if( PP_INCLUDE_MSGSM(pOption->mGeneral.mCodec) )
		SapiAddMediaCapabilityToDialogue(pHandle,tDialogue,"audio",e_RtpPayloadType_Gsm);
	if( PP_INCLUDE_ILBC(pOption->mGeneral.mCodec)  )
		SapiAddMediaCapabilityToDialogue(pHandle,tDialogue,"audio",e_RtpPayloadType_iLbc);

	if( (tResult=SapiCommandMakeCall(pHandle,pUa,tDialogue))==HS_INVALID_HANDLE )
	{
		SapiRemoveDialogue(tDialogue);
		return HS_INVALID_HANDLE;
	}
	return tResult;
}


HS_CALL_HANDLE PPStartSipCallDirect(
	PPOption *pOption,
	HS_STACK_HANDLE pHandle,
	char *pSource,
	HS_UCHAR *pSourceIp,
	HS_USHORT pSourcePort,
	char *pDest,
	char *pDestIp,
	HS_USHORT pDestPort
)
{
	HS_UA_HANDLE tUaHandle = HS_INVALID_HANDLE;
	HS_CALL_HANDLE tCallHandle = HS_INVALID_HANDLE;

	if( pOption==NULL || pHandle==HS_INVALID_STACK_HANDLE ||
		pSource==NULL || pSourceIp==NULL ||
		pDest==NULL || pDestIp==NULL )
		return HS_INVALID_HANDLE;

	/* add instant ua to stack
	*/
	tUaHandle = PPAddSipUaEx(pHandle,
		pDestIp,NULL,pDestPort,60,NULL,
		pSource,pSourceIp,pSourcePort,NULL,NULL
	);
	if( tUaHandle==HS_INVALID_HANDLE ) return HS_INVALID_HANDLE;

	/* start call
	*/
	tCallHandle = PPStartSipCall(pOption,pHandle,tUaHandle,pDest);
	/* when direct call is over,
	   temp UA have to delete on stack.
	*/
	SapiCommandRemoveUa(pHandle,tUaHandle);
	return tCallHandle;
}


HS_RESULT PPSipChangeListenPort(HS_STACK_HANDLE pHandle,BOOL pIsUdp,HS_USHORT pPort)
{
	if( pHandle==HS_INVALID_STACK_HANDLE )
		return HS_ERR_NULL_PARAM;

	return SapiCommandChangeListenPort(pHandle,pIsUdp,pPort);
}


HS_RESULT PPSipSendInstantMessage(HS_STACK_HANDLE pHandle,HS_CALL_HANDLE pCall,char *pIM)
{
	if( pHandle==HS_INVALID_STACK_HANDLE || pCall==HS_INVALID_HANDLE )
		return HS_ERR_NULL_PARAM;

	return SapiCommandSendMESSAGE(pHandle,pCall,NULL,pIM);
}


HS_RESULT PPSipSendINFO(HS_STACK_HANDLE pHandle,HS_CALL_HANDLE pCall,char pDtmf)
{
	char tBody[32];

	if( pHandle==HS_INVALID_STACK_HANDLE || pCall==HS_INVALID_HANDLE )
		return HS_ERR_NULL_PARAM;

	sprintf(tBody,"signal=%c\r\n",pDtmf);
	return SapiCommandSendINFO(pHandle,pCall,NULL,tBody);
}





/*
 *
 * handlers
 *
 */

HS_RESULT PPCore_Initialize(void *pObject)
{
	PPCore *pObj = (PPCore*)pObject;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( (pObj->mH323.mStack=PPStartH323Stack(&(pObj->mOption),pObj->mLocalIp))
		== HS_INVALID_STACK_HANDLE ) return HS_ERR;
	if( (pObj->mSip.mStack=PPStartSipStack(&(pObj->mOption),pObj->mLocalIp))
		== HS_INVALID_STACK_HANDLE ) return HS_ERR;

	pObj->mSip.mUaPrime	= PPAddSipUa(&(pObj->mOption),pObj->mSip.mStack,pObj->mLocalIp,TRUE);
	pObj->mSip.mUaSecond= PPAddSipUa(&(pObj->mOption),pObj->mSip.mStack,pObj->mLocalIp,FALSE);
	pObj->mRtp			= PPMedia_Start(newm_PPMedia());

	ppHandle_SetOption(pObj->mOption);
	PPCore_HandleRegistCheck(pObj);
	return HS_OK;
}


HS_RESULT PPCore_HandleOption(PPCore *pObj,PPOption* pOption)
{
	BOOL tIsChange = FALSE;

	if( pOption==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj==NULL )
	{
		HSFree(pOption);
		return HS_ERR_NULL_PARAM;
	}

	/* check H.323 registration
	*/
	tIsChange = FALSE;
	if( pObj->mH323.mIsRegisted==TRUE )
	{
		if( pObj->mH323.mIsPrime==TRUE )
		{
			if( strcmp(pOption->mH323.mPrimeGk,pObj->mOption.mH323.mPrimeGk) )
				tIsChange = TRUE;
		}
		else
		{
			if( strcmp(pOption->mH323.mSecondGk,pObj->mOption.mH323.mSecondGk) )
				tIsChange = TRUE;
		}
	}

	if( tIsChange==TRUE )
	{
		pObj->mH323.mIsRegisted = FALSE;
		HapiCommandChangeGatekeeper(pObj->mH323.mStack,"");
		if( pObj->mHwnd != NULL )
			SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_FAIL,0,1);
	}

	if( strcmp(pOption->mH323.mH323Id,pObj->mOption.mH323.mH323Id) ||
		strcmp(pOption->mH323.mPhoneNumber,pObj->mOption.mH323.mPhoneNumber) )
		PPH323ChangeEndpointAlias(pObj->mH323.mStack,pOption);

	/* check SIP registration
	*/
	if( memcmp(&(pOption->mSip),&(pObj->mOption.mSip),sizeof(PPOptionSip)) )
	{
		pObj->mSip.mIsPrime = FALSE;
		pObj->mSip.mIsRegistedPrime = FALSE;
		pObj->mSip.mIsRegistedSecond = FALSE;
		if( pObj->mHwnd != NULL )
			SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_FAIL,0,0);

		if( pObj->mSip.mUaPrime	!= HS_INVALID_HANDLE )
		{
			SapiCommandRemoveUa(pObj->mSip.mStack,pObj->mSip.mUaPrime);
			pObj->mSip.mUaPrime	= HS_INVALID_HANDLE;
		}
		if( pObj->mSip.mUaSecond != HS_INVALID_HANDLE )
		{
			SapiCommandRemoveUa(pObj->mSip.mStack,pObj->mSip.mUaSecond);
			pObj->mSip.mUaSecond = HS_INVALID_HANDLE;
		}

		if( pObj->mSip.mUaPrime==HS_INVALID_HANDLE )
			pObj->mSip.mUaPrime	= PPAddSipUa(pOption,pObj->mSip.mStack,pObj->mLocalIp,TRUE);
		if( pObj->mSip.mUaSecond==HS_INVALID_HANDLE )
			pObj->mSip.mUaSecond = PPAddSipUa(pOption,pObj->mSip.mStack,pObj->mLocalIp,FALSE);

		if( (pOption->mSip.mLocalUdpPort != pObj->mOption.mSip.mLocalUdpPort) )
			SapiCommandChangeListenPort(pObj->mSip.mStack,TRUE,pOption->mSip.mLocalUdpPort);
		if( (pOption->mSip.mLocalTcpPort != pObj->mOption.mSip.mLocalTcpPort) )
			SapiCommandChangeListenPort(pObj->mSip.mStack,FALSE,pOption->mSip.mLocalTcpPort);
	}


	memcpy(&(pObj->mOption),pOption,sizeof(PPOption));
	ppHandle_SetOption(pObj->mOption);
	HSFree(pOption);

	PPCore_HandleRegistCheck(pObj);
	return HS_OK;
}


HS_RESULT PPCore_HandleRegistCheck(PPCore *pObj)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	/* check H.323 registration
	*/
	if( pObj->mH323.mStack != HS_INVALID_STACK_HANDLE )
	{
		if( pObj->mH323.mIsRegisted==FALSE )
		{
			if( pObj->mH323.mIsPrime==FALSE )
			{
				pObj->mH323.mIsPrime = TRUE;
				if( ! is_blank(pObj->mOption.mH323.mPrimeGk) )
					HapiCommandChangeGatekeeper(pObj->mH323.mStack,pObj->mOption.mH323.mPrimeGk);
			}
			else
			{
				pObj->mH323.mIsPrime = FALSE;
				if( ! is_blank(pObj->mOption.mH323.mSecondGk) )
					HapiCommandChangeGatekeeper(pObj->mH323.mStack,pObj->mOption.mH323.mSecondGk);
			}
		}
	}

	/* check Sip registration
	*/
	if( pObj->mSip.mStack != HS_INVALID_STACK_HANDLE )
	{
		if( pObj->mSip.mIsRegistedPrime==FALSE && pObj->mSip.mIsRegistedSecond==FALSE )
		{
			if( pObj->mSip.mIsPrime==FALSE )
			{
				pObj->mSip.mIsPrime = TRUE;
				if( pObj->mSip.mUaPrime != HS_INVALID_HANDLE )
					SapiCommandRegistUa(pObj->mSip.mStack,pObj->mSip.mUaPrime);
			}
			else
			{
				pObj->mSip.mIsPrime = FALSE;
				if( pObj->mSip.mUaSecond != HS_INVALID_HANDLE )
					SapiCommandRegistUa(pObj->mSip.mStack,pObj->mSip.mUaSecond);
			}
		}
	}

	return HS_OK;
}





/* event from user interface to PP
*/
HS_RESULT PPCore_HandleHookOff(PPCore *pObj)
{
	HS_RESULT tRet = HS_OK;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Idle:
			tRet = PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Dial);
			if( tRet==HS_OK ) pObj->mState = e_PPState_DialTone;
			return tRet;
		case e_PPState_Powerring:
			if( pObj->mH323.mCall != HS_INVALID_HANDLE )
				return HapiCommandAcceptCall(pObj->mH323.mStack,pObj->mH323.mCall);
			else if( pObj->mSip.mCall != HS_INVALID_HANDLE )
				return SapiCommandAcceptCall(pObj->mSip.mStack,pObj->mSip.mCall);
			else
				return HS_ERR;
		default:
			return HS_ERR;
	}
	
	return HS_OK;
}


HS_RESULT PPCore_HandleHookOn(PPCore *pObj)
{
	HS_RESULT tRet = HS_OK;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_DialTone:
			tRet = PPMedia_StopTone(pObj->mRtp);
			if( tRet==HS_OK ) pObj->mState = e_PPState_Idle;
			return tRet;
		case e_PPState_Dialing:
			pObj->mDialPos = 0;
			pObj->mState = e_PPState_Idle;
			return HS_OK;
		case e_PPState_RingbackTone:
			tRet = PPMedia_StopTone(pObj->mRtp);
		case e_PPState_Proceed:
		case e_PPState_Powerring:
		case e_PPState_Connect:
			if( pObj->mH323.mCall != HS_INVALID_HANDLE )
			{
				tRet = HapiCommandRemoveCall(pObj->mH323.mStack,pObj->mH323.mCall,e_CallCloseReason_Undefined);
				pObj->mH323.mCall = HS_INVALID_HANDLE;
			}
			else if( pObj->mSip.mCall != HS_INVALID_HANDLE )
			{
				if( pObj->mState==e_PPState_Powerring )
					tRet = SapiCommandRemoveCall(pObj->mSip.mStack,pObj->mSip.mCall,e_SipResponse_RequestTerminated);
				else
					tRet = SapiCommandRemoveCall(pObj->mSip.mStack,pObj->mSip.mCall,e_SipResponseMax);
				pObj->mSip.mCall = HS_INVALID_HANDLE;
			}
			else
				tRet = HS_ERR;
			pObj->mState = e_PPState_Idle;
			return tRet;
		case e_PPState_FailTone:
			tRet = PPMedia_StopTone(pObj->mRtp);
			pObj->mState = e_PPState_Idle;
			return tRet;
		default:
			return HS_ERR;
	}

	return HS_OK;
}


static HS_RESULT PPCore_SendDtmf(PPCore *pObj,char pDtmf)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mOption.mGeneral.mDtmfBand)
	{
		case e_PPOptionDtmfBand_Inband:
			return PPMedia_AddDtmfInband(pObj->mRtp,pDtmf);
		case e_PPOptionDtmfBand_Rfc2833:
			return PPMedia_AddDtmfRfc2833(pObj->mRtp,pDtmf);
		case e_PPOptionDtmfBand_Outband:
			if( pObj->mH323.mCall != HS_INVALID_HANDLE )
				return PPH323DtmfSignalByH245(pObj->mH323.mStack,pObj->mH323.mCall,pDtmf,1);
			else if( pObj->mSip.mCall != HS_INVALID_HANDLE )
				return PPSipSendINFO(pObj->mSip.mStack,pObj->mSip.mCall,pDtmf);
			else
				return HS_ERR;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}
HS_RESULT PPCore_HandleDtmf(PPCore *pObj,BOOL pIsSound,char pDtmf)
{
	DtmfSignal tDtmfSignal;
	HS_RESULT tRet = HS_OK;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pIsSound==TRUE )
	{
		switch(pDtmf)
		{
			case '0':	tDtmfSignal = e_DtmfSignal_0;	break;
			case '1':	tDtmfSignal = e_DtmfSignal_1;	break;
			case '2':	tDtmfSignal = e_DtmfSignal_2;	break;
			case '3':	tDtmfSignal = e_DtmfSignal_3;	break;
			case '4':	tDtmfSignal = e_DtmfSignal_4;	break;
			case '5':	tDtmfSignal = e_DtmfSignal_5;	break;
			case '6':	tDtmfSignal = e_DtmfSignal_6;	break;
			case '7':	tDtmfSignal = e_DtmfSignal_7;	break;
			case '8':	tDtmfSignal = e_DtmfSignal_8;	break;
			case '9':	tDtmfSignal = e_DtmfSignal_9;	break;
			case '*':	tDtmfSignal = e_DtmfSignal_Star;	break;
			case '#':	tDtmfSignal = e_DtmfSignal_Shap;	break;
			default:
				return HS_ERR;
		}
	}

	switch(pObj->mState)
	{
		case e_PPState_DialTone:
		case e_PPState_FailTone:
			pObj->mDialPos = 0;
			tRet = PPMedia_StopTone(pObj->mRtp);
			pObj->mState = e_PPState_Dialing;
		case e_PPState_Dialing:
			pObj->mDial[ pObj->mDialPos++ ] = pDtmf;
			if( pIsSound==TRUE )
			{
				PPMedia_StartTone(pObj->mRtp,tDtmfSignal);
				PPMedia_StopTone(pObj->mRtp);
			}
			return tRet;

		case e_PPState_Connect:
			return PPCore_SendDtmf(pObj,pDtmf);
	}

	return HS_OK;
}


HS_RESULT PPCore_HandleDtmfCancel(PPCore *pObj)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_FailTone:
		case e_PPState_Dialing:
			if( pObj->mDialPos < 1 )
			{
				pObj->mDialPos = 0;
				return HS_ERR;
			}
			if( pObj->mState==e_PPState_FailTone )
			{
				PPMedia_StopTone(pObj->mRtp);
				pObj->mState = e_PPState_Dialing;
			}

			pObj->mDialPos--;
			if( pObj->mDialPos==0 )
			{
				PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Dial);
				pObj->mState = e_PPState_DialTone;
			}
			return HS_OK;
		default:
			return HS_ERR;
	}

	return HS_OK;
}


HS_RESULT PPCore_HandleDtmfClear(PPCore *pObj)
{
	HS_RESULT tRet = HS_OK;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_FailTone:
		case e_PPState_Dialing:
			pObj->mDialPos = 0;
			if( pObj->mState==e_PPState_FailTone )
				PPMedia_StopTone(pObj->mRtp);
			PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Dial);
			pObj->mState = e_PPState_DialTone;
			return HS_OK;
		default:
			return HS_ERR;
	}

	return HS_OK;
}


HS_RESULT PPCore_HandleMakeCall(PPCore *pObj,HS_UINT pProtocol,char* pDest/*(if it's null,try by DTMF signal buffer)*/)
{
	HS_CALL_HANDLE tRet = HS_OK;

	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	if( pObj->mH323.mCall != HS_INVALID_HANDLE || pObj->mH323.mCall != HS_INVALID_HANDLE )
		return HS_ERR;
	if( pDest==NULL && pObj->mDialPos==0 )
		return HS_ERR;
	if( pDest==NULL )
		pDest = pObj->mDial;
	pObj->mDial[ pObj->mDialPos ] = '\0';

	switch(pObj->mState)
	{
		case e_PPState_Idle:
		case e_PPState_Dialing:
			if( pProtocol==0 )
			{
				if( IsIp(pDest) )
					tRet = PPStartSipCallDirect(&(pObj->mOption),pObj->mSip.mStack,
						pObj->mOption.mSip.mUserName,pObj->mLocalIp,5060,
						"unknown",pDest,5060
					);
				else if( ! is_blank(pObj->mOption.mSip.mTargetGw) )
					tRet = PPStartSipCallDirect(&(pObj->mOption),pObj->mSip.mStack,
						pObj->mOption.mSip.mUserName,pObj->mLocalIp,5060,
						pDest,pObj->mOption.mSip.mTargetGw,5060
					);
				else if( pObj->mSip.mUaPrime != HS_INVALID_HANDLE )
					tRet = PPStartSipCall(&(pObj->mOption),pObj->mSip.mStack,pObj->mSip.mUaPrime,pDest);
				else if( pObj->mSip.mUaSecond != HS_INVALID_HANDLE )
					tRet = PPStartSipCall(&(pObj->mOption),pObj->mSip.mStack,pObj->mSip.mUaSecond,pDest);
				else
				{
					if( IsIp(pDest) || pObj->mH323.mIsRegisted==TRUE )
						tRet = PPStartH323Call(&(pObj->mOption),pObj->mH323.mStack,pObj->mLocalIp,pDest);
					else
						return HS_ERR;

					if( tRet != HS_INVALID_HANDLE )
					{
						pObj->mH323.mCall = tRet;
						pObj->mState = e_PPState_Proceed;
						return HS_OK;
					}
					return HS_ERR;
				}

				if( tRet != HS_INVALID_HANDLE )
				{
					pObj->mSip.mCall = tRet;
					pObj->mState = e_PPState_Proceed;
					return HS_OK;
				}
				return HS_ERR;
			}
			else
			{
				if( IsIp(pDest) || pObj->mH323.mIsRegisted==TRUE )
				{
					tRet = PPStartH323Call(&(pObj->mOption),pObj->mH323.mStack,pObj->mLocalIp,pDest);

					if( tRet != HS_INVALID_HANDLE )
					{
						pObj->mH323.mCall = tRet;
						pObj->mState = e_PPState_Proceed;
						return HS_OK;
					}
					return HS_ERR;
				}
				else
				{
					if( IsIp(pDest) )
						tRet = PPStartSipCallDirect(&(pObj->mOption),pObj->mSip.mStack,
							pObj->mOption.mSip.mUserName,pObj->mLocalIp,5060,
							"unknown",pDest,5060
						);
					else if( ! is_blank(pObj->mOption.mSip.mTargetGw) )
						tRet = PPStartSipCallDirect(&(pObj->mOption),pObj->mSip.mStack,
							pObj->mOption.mSip.mUserName,pObj->mLocalIp,5060,
							pDest,pObj->mOption.mSip.mTargetGw,5060
						);
					else if( pObj->mSip.mUaPrime != HS_INVALID_HANDLE )
						tRet = PPStartSipCall(&(pObj->mOption),pObj->mSip.mStack,pObj->mSip.mUaPrime,pDest);
					else if( pObj->mSip.mUaSecond != HS_INVALID_HANDLE )
						tRet = PPStartSipCall(&(pObj->mOption),pObj->mSip.mStack,pObj->mSip.mUaSecond,pDest);
					else
						return HS_ERR;

					if( tRet != HS_INVALID_HANDLE )
					{
						pObj->mSip.mCall = tRet;
						pObj->mState = e_PPState_Proceed;
						return HS_OK;
					}
					return HS_ERR;
				}
			}
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleRtpHold(PPCore *pObj,PPDirection pDirection)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Connect:
			return PPMedia_HoldResume(pObj->mRtp,pDirection,e_PPHold_Hold);
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleRtpResume(PPCore *pObj,PPDirection pDirection)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Connect:
			return PPMedia_HoldResume(pObj->mRtp,pDirection,e_PPHold_Resume);
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleSndInstantMessage(PPCore *pObj,char* pMsg)
{
	if( pObj==NULL || pMsg==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Connect:
			return PPSipSendInstantMessage(pObj->mSip.mStack,pObj->mSip.mCall,pMsg);
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


/* event from callback to PP
*/
HS_RESULT PPCore_HandleCid(PPCore *pObj,HS_UINT pProtocol,char *pCid)
{
	if( pCid==NULL ) return HS_ERR_NULL_PARAM;
	if( pObj==NULL )
	{
		HSFree(pCid);
		return HS_ERR_NULL_PARAM;
	}

	switch(pObj->mState)
	{
		case e_PPState_Idle:
		case e_PPState_Powerring:
			break;
		case e_PPState_DialTone:
		case e_PPState_Dialing:
		case e_PPState_Proceed:
		case e_PPState_RingbackTone:
		case e_PPState_FailTone:
		case e_PPState_Connect:
		case e_PPStateMax:
			HSFree(pCid);
			return HS_ERR;
	}

	if( pObj->mHwnd==NULL )
	{
		HSFree(pCid);
		return HS_ERR;
	}

	SendMessage(pObj->mHwnd,HS_QM_APP_CID,0,(LPARAM)pCid);
	return HS_OK;
}


HS_RESULT PPCore_HandleIncomingCall(PPCore *pObj,HS_UINT pProtocol,HS_CALL_HANDLE pHandle)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pHandle==HS_INVALID_HANDLE ) return HS_ERR_INVALID_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_DialTone:
		case e_PPState_FailTone:
			PPMedia_StopTone(pObj->mRtp);
		case e_PPState_Idle:
		case e_PPState_Dialing:
			pObj->mDialPos = 0;
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_INCOMING,0,0);
			pObj->mState = e_PPState_Powerring;

			if( pProtocol==1 )	pObj->mH323.mCall = pHandle;
			else				pObj->mSip.mCall = pHandle;
			return HS_OK;
		default:
			if( pProtocol==0 )
				return SapiCommandRemoveCall(pObj->mSip.mStack,pHandle,e_SipResponse_BusyHere);
			return HapiCommandRemoveCall(pObj->mH323.mStack,pHandle,e_CallCloseReason_Undefined);
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleRingback(PPCore *pObj,HS_UINT pProtocol,HS_CALL_HANDLE pHandle)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pHandle==HS_INVALID_HANDLE ) return HS_ERR_INVALID_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Proceed:
			if( pHandle==pObj->mSip.mCall || pHandle==pObj->mH323.mCall )
			{
				PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Ringback);
				pObj->mState = e_PPState_RingbackTone;
				return HS_OK;
			}
			return HS_ERR;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleSession(PPCore *pObj,QmMediaInfo* pMedia)
{
	if( pObj==NULL || pMedia==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_RingbackTone:
			PPMedia_StopTone(pObj->mRtp);
		case e_PPState_Proceed:
		case e_PPState_Powerring:
		case e_PPState_Connect:
			if( pMedia->mAddrIn.sin_family==AF_MAX )
			{
				PPMedia_ReverseMedia(pObj->mRtp,pMedia);

				if( pObj->mHwnd != NULL )
					SendMessage(pObj->mHwnd,HS_QM_APP_CODEC,0,(LPARAM)(pMedia->mType));
			}
			else
				PPMedia_ForwardMedia(pObj->mRtp,pMedia);
			pObj->mState = e_PPState_Connect;
			return HS_OK;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleUnsession(PPCore *pObj,PPDirection pDirection)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Idle:
		case e_PPState_Connect:
			return PPMedia_StopMedia(pObj->mRtp);
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleConnected(PPCore *pObj,HS_UINT pProtocol,HS_CALL_HANDLE pHandle)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pHandle==HS_INVALID_HANDLE ) return HS_ERR_INVALID_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_RingbackTone:
			PPMedia_StopTone(pObj->mRtp);
		case e_PPState_Powerring:
		case e_PPState_Connect:
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_CONNECTED,0,0);
			pObj->mState = e_PPState_Connect;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleOutDtmf(PPCore *pObj,QmSipContext* pContext)
{
	switch(pObj->mState)
	{
		case e_PPState_Connect:
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_DTMF,0,(LPARAM)pContext);
			return HS_OK;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleRcvInstantMessage(PPCore *pObj,QmSipContext* pContext)
{
	if( pObj==NULL || pContext==NULL ) return HS_ERR_NULL_PARAM;

	switch(pObj->mState)
	{
		case e_PPState_Connect:
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_MSG,0,(LPARAM)pContext);
			return HS_OK;
		default:
			return HS_ERR;
	}

	return HS_ERR;
}


HS_RESULT PPCore_HandleCallEnd(PPCore *pObj,HS_UINT pProtocol,QmCallEnd* pEnd)
{
	BOOL tIsBusy = FALSE;

	if( pObj==NULL || pEnd==NULL ) return HS_ERR_NULL_PARAM;

	if( pProtocol==0/*sip*/ )
	{
		if( pEnd->mHandle==HS_INVALID_CALL_HANDLE ||
			(pEnd->mHandle != pObj->mSip.mCall)
		)
		{
			deletem_QmCallEnd(pEnd);
			return HS_ERR;
		}
	}
	else/*h323*/
	{
		if( pEnd->mHandle==HS_INVALID_CALL_HANDLE ||
			(pEnd->mHandle != pObj->mH323.mCall)
		)
		{
			deletem_QmCallEnd(pEnd);
			return HS_ERR;
		}
	}

	switch(pObj->mState)
	{
		case e_PPState_RingbackTone:
			PPMedia_StopTone(pObj->mRtp);
		case e_PPState_Proceed:
			if( (pProtocol==0 && pEnd->mReason==e_SipResponse_BusyHere) ||
				(pProtocol==1 && pEnd->mReason==e_Q931CauseType_UserBusy)
			) tIsBusy = TRUE;
		case e_PPState_Connect:
			if( tIsBusy==TRUE )
				PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Busy);
			else
				PPMedia_StartTone(pObj->mRtp,e_DtmfSignal_Reorder);
		case e_PPState_Powerring:
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_CALL_END,0,0);
			if( pObj->mState==e_PPState_Powerring )
				pObj->mState = e_PPState_Idle;
			else
				pObj->mState = e_PPState_FailTone;

			if( pProtocol==0 && pEnd->mHandle==pObj->mSip.mCall )
				pObj->mSip.mCall = HS_INVALID_HANDLE;
			if( pProtocol==1 && pEnd->mHandle==pObj->mH323.mCall )
				pObj->mH323.mCall = HS_INVALID_HANDLE;

			deletem_QmCallEnd(pEnd);
			return HS_OK;

		default:
			deletem_QmCallEnd(pEnd);
			return HS_ERR;
	}

	deletem_QmCallEnd(pEnd);
	return HS_ERR;
}


HS_RESULT PPCore_HandleRegistSuccess(PPCore *pObj,HS_UINT pProtocol,HS_UINT pHandle)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pHandle==HS_INVALID_HANDLE ) return HS_ERR_NULL_PARAM;

	if( pProtocol==0 )
	{
		if( pHandle==pObj->mSip.mUaPrime )
		{
			pObj->mSip.mIsRegistedPrime = TRUE;
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_SUCC,0,0);
		}
		else if (pHandle==pObj->mSip.mUaSecond )
		{
			pObj->mSip.mIsRegistedSecond = TRUE;
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_SUCC,0,0);
		}
		else
			return HS_ERR;
	}
	else
	{
		pObj->mH323.mIsRegisted = TRUE;
		if( pObj->mHwnd != NULL )
			SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_SUCC,0,1);
	}

	return HS_OK;
}


HS_RESULT PPCore_HandleRegistFail(PPCore *pObj,HS_UINT pProtocol,HS_UINT pHandle)
{
	if( pObj==NULL ) return HS_ERR_NULL_PARAM;
	if( pHandle==HS_INVALID_HANDLE ) return HS_ERR_NULL_PARAM;

	if( pProtocol==0 )
	{
		if( pHandle==pObj->mSip.mUaPrime )
		{
			pObj->mSip.mIsRegistedPrime = FALSE;
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_FAIL,0,0);
		}
		else if (pHandle==pObj->mSip.mUaSecond )
		{
			pObj->mSip.mIsRegistedSecond = FALSE;
			if( pObj->mHwnd != NULL )
				SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_FAIL,0,0);
		}
		else
			return HS_ERR;
	}
	else
	{
		pObj->mH323.mIsRegisted = FALSE;
		if( pObj->mHwnd != NULL )
			SendMessage(pObj->mHwnd,HS_QM_APP_REGIST_FAIL,0,1);
	}

	return HS_OK;
}





