/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<SipMessageHandler.c>	2005-10-25,21:22

*/

#include "SipMessageHandler.h"

#if defined(_LINUX)
#include <netdb.h>
extern int h_errno;
#endif





/* Utilities
*/
HS_RESULT String2SockAddr(struct sockaddr_in *pTo, char *pFrom, HS_USHORT pPort)
{
	unsigned long laddr;

	if( pTo == NULL || pFrom == NULL ) return HS_ERR_NULL_PARAM;
	/*multicast address*/
	if( !strcmp(pFrom,"@") )
	{
		pTo->sin_family = AF_INET;
		pTo->sin_addr.s_addr = inet_addr("224.0.1.41");
		pTo->sin_port = htons(pPort);
		return HS_OK;
	}

	if( (laddr=inet_addr(pFrom)) == INADDR_NONE )
	{
#if defined(_VXWORKS)
		laddr = hostGetByName(pFrom);
#elif defined(_WIN32) || defined(_WIN32_WCE) || defined(_LINUX)
		struct hostent *hp = gethostbyname(pFrom);
		if( hp == NULL )
			return HS_ERR;

		laddr = ((struct in_addr*)hp->h_addr_list[0])->s_addr;
#endif
	}

	pTo->sin_family = AF_INET;
    pTo->sin_addr.s_addr = laddr;
	pTo->sin_port = htons(pPort);
	return HS_OK;
}


QmEnetMessage *MakeQmEnetMessage(ITransaction *pTransaction,char *pPacket,HS_UINT pLen)
{
	HS_USHORT tPort = 5060;
	HS_TR_HANDLE tTrHandle;
	struct sockaddr_in tAddrIn;
	SipContactUnit *tTrViaContactUnit = NULL;

	if( pTransaction==NULL || pPacket==NULL ) return NULL;

	tAddrIn.sin_family = AF_MAX;
	if( pTransaction->mIsTcp==TRUE )
		tTrHandle = pTransaction->mHandle;
	else
		tTrHandle = HS_INVALID_HANDLE;

	if( pTransaction->mIsClient==TRUE )
	{
		if( pTransaction->mDialogue->mUa->mOutBoundProxy[0] != '\0' )
			String2SockAddr(&tAddrIn,
				pTransaction->mDialogue->mUa->mOutBoundProxy,
				pTransaction->mDialogue->mUa->mProxyPort
			);
		else
			String2SockAddr(&tAddrIn,
				pTransaction->mDialogue->mUa->mProxy,
				pTransaction->mDialogue->mUa->mProxyPort
			);
	}
	else
	{
		if( (tTrViaContactUnit=ITransaction_GetPrimeContactUnit(pTransaction))==NULL )
			return NULL;

		if( tTrViaContactUnit->mUri.mPort != NULL )
			tPort = (HS_USHORT)atoi(tTrViaContactUnit->mUri.mPort);
		else
			tPort = 5060;

		if( String2SockAddr(&tAddrIn,tTrViaContactUnit->mUri.mAddress,tPort) != HS_OK )
			return NULL;
	}

	return newm_QmEnetMessage(pPacket,pLen,tTrHandle,tAddrIn);
}


BOOL IsNeedTransaction(SipMessage *pMessage)
{
	if( pMessage==NULL ) return FALSE;

	if( pMessage->mHeadLine.mResponseValue > e_SipResponse_Request ) return TRUE;
	if( !strcmp(pMessage->mHeadLine.mMethod,"ACK") ||
		!strcmp(pMessage->mHeadLine.mMethod,"PRACK")
	) return TRUE;

	return FALSE;
}


IDialogue *MakeNewDialogue(IStack *pStack, SipMessage *pMessage, BOOL pIsTcp)
{
	IUa *tUa = NULL;
	IDialogue *tResult = NULL;
	SipContact *tContact = NULL;
	SipContactUnit *tContactUnit = NULL;
	SipContactUnit *tRContactUnit = NULL;
	SipContactUnit *tToContactUnit = NULL;
	SipContactUnit *tFromContactUnit = NULL;

	if( pStack==NULL || pMessage==NULL ) return NULL;
	if( pMessage->mTo==NULL || pMessage->mFrom==NULL ) return NULL;
	if( (tToContactUnit=SipTo_GetPrimeContactUnit(pMessage->mTo))==NULL ||
		(tFromContactUnit=SipFrom_GetPrimeContactUnit(pMessage->mFrom))==NULL
	) return NULL;

	if( (tUa=IStack_FindUaByUserName(pStack,tToContactUnit->mUri.mUserName))==NULL )
	{
		/* maybe direct call
		*/
		if( (tRContactUnit=SipContacts_GetPrimeContactUnit(&(pMessage->mContacts)))==NULL )
			return NULL;
		if( tRContactUnit->mUri.mAddress==NULL ) return NULL;
		if( (tUa=(IUa*)HSMalloc(sizeof(IUa)))==NULL ) return NULL;

		new_IUa(tUa,pStack,110/*dont care value*/);
		tUa->mHandle = IStack_MakeUaHandle(pStack);

		strcpy(tUa->mProxy,tRContactUnit->mUri.mAddress);
		if( tRContactUnit->mUri.mPort != NULL )
			tUa->mProxyPort = (HS_USHORT)atoi(tRContactUnit->mUri.mPort);
		else
		{
			if( pIsTcp )
				tUa->mProxyPort = pStack->mTcpPort;
			else
				tUa->mProxyPort = pStack->mUdpPort;
		}

		if( (tContactUnit=newm_SipContactUnit(FALSE))==NULL )
		{
			delete_IUa(tUa);
			HSFree(tUa);
			return NULL;
		}
		if( (tContact=newm_SipContact())==NULL )
		{
			delete_IUa(tUa);
			HSFree(tUa);
			delete_SipContactUnit(tContactUnit);
			HSFree(tContactUnit);
			return NULL;
		}

		if( pIsTcp==TRUE )
			tContactUnit->mUri.mPort = SipUintAlloc((HS_UINT)(pStack->mTcpPort));
		else
			tContactUnit->mUri.mPort = SipUintAlloc((HS_UINT)(pStack->mUdpPort));
		tContactUnit->mUri.mProtocol = SipStringAlloc(pStack->mProtocol);
		tContactUnit->mUri.mUserName = SipStringAlloc(tToContactUnit->mUri.mUserName);
		tContactUnit->mUri.mAddress = SipStringAlloc(pStack->mLocalIp);

		NoLockList_AttachData(&(tContact->mContactUnits),tContactUnit);
		NoLockList_AttachData(&(tUa->mContacts),tContact);
		tUa->mState = e_UaState_KillMe;
	}

	if( (tResult=(IDialogue*)HSMalloc(sizeof(IDialogue)))==NULL ) return NULL;
	new_IDialogue(tResult,tUa);

	strcpy(tResult->mCallId,pMessage->mCallId->mInfo);
	tResult->mDestAddress = SipStringAlloc(tFromContactUnit->mUri.mAddress);
	tResult->mDestDisplayName = SipStringAlloc(tFromContactUnit->mDisplayName);
	if( tFromContactUnit->mUri.mPort!=NULL )
		tResult->mDestPort = (HS_USHORT)atoi(tFromContactUnit->mUri.mPort);
	else
	{
		if( pIsTcp )
			tResult->mDestPort = pStack->mTcpPort;
		else
			tResult->mDestPort = pStack->mUdpPort;
	}
	tResult->mDestUser = SipStringAlloc(tFromContactUnit->mUri.mUserName);
	tResult->mHandle = IStack_MakeDialogueHandle(pStack);
	tResult->mIsTcp = pIsTcp;
	IUa_MakeTag(tUa,tResult->mLocalTag);
	tResult->mState = e_DialogueState_ReadyConnect;
	if( pStack->CallbackCallIncoming != NULL )
		pStack->CallbackCallIncoming(pStack,tResult);

	return tResult;
}


ITransaction *MakeNewServerTransaction(IStack *pStack, IDialogue *pDialogue, SipMessage *pMessage, HS_TR_HANDLE pHandle)
{
	IUa *tUa = NULL;
	ITransaction *tResult = NULL;

	if( pStack==NULL || pDialogue==NULL || pMessage==NULL ) return NULL;
	if( pMessage->mCSeq==NULL ) return NULL;
	if( pMessage->mCSeq->mNumber==NULL ) return NULL;
	if( (tUa=pDialogue->mUa) == NULL ) return NULL;

	if( (tResult=(ITransaction*)HSMalloc(sizeof(ITransaction)))==NULL ) return NULL;
	new_ITransaction(
		tResult,
		pDialogue,
		!strcmp(pMessage->mHeadLine.mMethod,"INVITE"),
		FALSE, (pHandle!=HS_INVALID_HANDLE),
		atoi(pMessage->mCSeq->mNumber)
	);

	/* some transactions(ex:OPTIONS,MESSAGE) have different peer tag with dialogue
	*/
	if( !strcmp(pMessage->mHeadLine.mMethod,"OPTIONS") ||
		!strcmp(pMessage->mHeadLine.mMethod,"MESSAGE")
	)
	{
		tResult->mIsNewPeerTag = TRUE;
		IUa_MakeTag(tUa,tResult->mLocalTag);
	}

	tResult->mBranchId = SipStringAlloc( SipMessage_GetBranchId(pMessage) );
	if( pHandle != HS_INVALID_HANDLE )
		tResult->mHandle = pHandle;
	else
		tResult->mHandle = IStack_MakeTransactionHandle(pStack);
	tResult->mMethod = SipStringAlloc(pMessage->mHeadLine.mMethod);
	tResult->mState = e_TransactionStateMax;

	return tResult;
}


ITransaction *MakeNewClientTransaction(IStack *pStack, IDialogue *pDialogue, char *pMethod)
{
	ChainUnit *tChain = NULL;
	IUa *tUa = NULL;
	ITransaction *tResult = NULL;
	ITransaction *tINVITEtr = NULL;

	if( pStack==NULL || pDialogue==NULL || pMethod==NULL ) return NULL;
	if( (tUa=pDialogue->mUa) == NULL ) return NULL;

	/* 'CANCEL' transaction's IDs(branch,CSeq) are must same with INVITE transaction
	*/
	if( !strcmp(pMethod,"CANCEL") )
	{
		if( (tINVITEtr=IDialogue_FindActiveTransactionByMethod(pDialogue,"INVITE"))==NULL )
			return NULL;
	}

	if( (tResult=(ITransaction*)HSMalloc(sizeof(ITransaction)))==NULL ) return NULL;
	new_ITransaction(
		tResult,
		pDialogue,
		!strcmp(pMethod,"INVITE"),
		TRUE, pDialogue->mIsTcp,
		pDialogue->mCseqSeed++
	);

	/* some transactions(ex:OPTIONS,MESSAGE) have different peer tag with dialogue
	*/
	if( !strcmp(pMethod,"OPTIONS") ||
		!strcmp(pMethod,"MESSAGE")
	)
	{
		tResult->mIsNewPeerTag = TRUE;
		IUa_MakeTag(tUa,tResult->mLocalTag);
	}

	/* 'CANCEL' transaction's IDs(branch,CSeq) are must same with INVITE transaction
	*/
	if( !strcmp(pMethod,"CANCEL") )
	{
		pDialogue->mCseqSeed--;
		tResult->mCseq = tINVITEtr->mCseq;
 		tResult->mBranchId = SipStringAlloc(tINVITEtr->mBranchId);
	}
	else
	{
		tResult->mBranchId = IDialouge_MakeBranchId(pDialogue);
	}

	tResult->mHandle = IStack_MakeTransactionHandle(pStack);
	tResult->mMethod = SipStringAlloc(pMethod);
	tResult->mState = e_TransactionStateMax;
	ITransaction_MakeLocalVia(tResult);
	return tResult;
}


HS_RESULT SendSipMessage(IStack *st,IUa *ua,IDialogue *dg,ITransaction *tr,SipMessage *pMessage,char *pPacket,HS_UINT *pPos,int pLen)
{
	HS_RESULT tRet = HS_OK;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL ) return HS_ERR_NULL_PARAM;

	/* check message send callback
	*/
	if( st->CallbackSendMessage != NULL )
	{
		if( st->CallbackSendMessage(ua->mHandle,dg->mHandle,pMessage)==FALSE )
			return HS_ERR_SIP_USER;
	}
	/* encoding
	*/
	HS_TRY( SipMessage_EncodeEx(pMessage,pPacket,pPos,pLen) );
	/* make network message
	*/
	if( (tEnetMessage=MakeQmEnetMessage(tr,pPacket,*pPos))==NULL )
		return HS_ERR_MALLOC;
	/* check raw data send callback
	*/
	if( st->CallbackSendRawData != NULL )
	{
		if( st->CallbackSendRawData(ua->mHandle,dg->mHandle,tEnetMessage->mRaw,tEnetMessage->mLen)==FALSE )
		{
			deletem_QmEnetMessage(tEnetMessage);
			return HS_ERR_MALLOC;
		}
	}
	_HSThreadSendMessage(st->mNetQ,HS_QM_ENET_MESSAGE,0,(LPARAM)tEnetMessage);

	return HS_OK;
}


HS_RESULT StartNewTransaction(IStack *st, IUa *ua, IDialogue *dg, SipMethod pMethod, char *pBodyType, char *pBody)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	SipMessage *tMessage = NULL;
	ITransaction *tTransaction = NULL;

	if( st==NULL || ua==NULL || dg==NULL ) return HS_ERR_NULL_PARAM;

	switch(pMethod)
	{
		case e_SipMethod_Register:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"REGISTER"))==NULL )
				return HS_ERR_MALLOC;
			if( (tMessage=MakeRequestRegister(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			if( ua->mState==e_UaState_KillMe )
			{
				SipContact *tUnregContact = NULL;

				NoLockList_Clear(&(tMessage->mContacts));
				if( (tUnregContact=newm_SipContact())==NULL )
				{
					IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
					return HS_ERR_SIP_MAKE_MESSAGE;
				}
				tUnregContact->mIsStar = TRUE;
				NoLockList_AttachData(&(tMessage->mContacts),tUnregContact);
				if( tMessage->mExpires->mInfo != NULL )
					HSFree(tMessage->mExpires->mInfo);
				tMessage->mExpires->mInfo = SipStringAlloc("0");
			}
			break;
		case e_SipMethod_Invite:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"INVITE"))==NULL )
				return HS_ERR_MALLOC;
			if( (tMessage=MakeRequestInvite(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			break;
		case e_SipMethod_Cancel:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"CANCEL"))==NULL )
				return HS_ERR_MALLOC;
			/* CANCEL message DO NOT include 'remote tag'.
			*/
			memset(dg->mRemoteTag,0,MAX_NAME_SIZE);
			if( (tMessage=MakeRequestCancel(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			break;
		case e_SipMethod_Bye:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"BYE"))==NULL )
				return HS_ERR_MALLOC;
			if( (tMessage=MakeRequestBye(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			break;
		case e_SipMethod_Info:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"INFO"))==NULL )
				return HS_ERR_MALLOC;
			if( (tMessage=MakeRequestInfo(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			break;
		case e_SipMethod_Message:
			if( (tTransaction=MakeNewClientTransaction(st,dg,"MESSAGE"))==NULL )
				return HS_ERR_MALLOC;
			if( (tMessage=MakeRequestMESSAGE(st,ua,dg,tTransaction,pBodyType,pBody))==NULL )
			{
				IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
				return HS_ERR_SIP_MAKE_MESSAGE;
			}
			break;

		case e_SipMethod_Refer:
		case e_SipMethod_Notify:
		case e_SipMethod_Options:
		case e_SipMethod_Update:
		case e_SipMethod_Subscribe:
		case e_SipMethod_Ack:
		case e_SipMethod_Prack:
		default:
			return HS_ERR_SIP_NOT_SUPPORT;
	}

	tPos = 0;
	if( (tRet=SendSipMessage(st,ua,dg,tTransaction,tMessage,tPacket,&tPos,2000)) != HS_OK )
	{
		IDialogue_DeleteTransaction(dg,tTransaction->mHandle);
		return tRet;
	}

	ITransaction_SetLinger(tTransaction,tPacket,tPos);
	if( pMethod != e_SipMethod_Invite )
	{
		tTransaction->mState = e_TransactionState_Trying;
		tTransaction->mTimer = HS_SIP_TIMER_E;
		HSSetTimer(tTransaction->mTimer,st->mSipQ,HS_TID_SIP_TIMER_E,tTransaction->mHandle,e_TimerType_Once);
		HSSetTimer(HS_SIP_TIMER_F,st->mSipQ,HS_TID_SIP_TIMER_F,tTransaction->mHandle,e_TimerType_Once);
	}
	else
	{
		tTransaction->mState = e_TransactionState_Calling;
		tTransaction->mTimer = HS_SIP_TIMER_A;
		HSSetTimer(tTransaction->mTimer,st->mSipQ,HS_TID_SIP_TIMER_A,tTransaction->mHandle,e_TimerType_Once);
		HSSetTimer(HS_SIP_TIMER_B,st->mSipQ,HS_TID_SIP_TIMER_B,tTransaction->mHandle,e_TimerType_Once);
	}

	delete_SipMessage(tMessage);
	HSFree(tMessage);
	return HS_OK;
}


HS_RESULT TrySipRegistration(IUa *pUa)
{
	IDialogue *tDialogue = NULL;
	ITransaction *tTransaction = NULL;

	if( pUa==NULL ) return HS_ERR_NULL_PARAM;
	if( (tDialogue=(IDialogue*)(pUa->mRegistrar))==NULL ) return HS_ERR_NULL_PARAM;

	return StartNewTransaction(pUa->mStack,pUa,tDialogue,e_SipMethod_Register,NULL,NULL);
}


HS_RESULT CheckReceivedSdp(IStack *st, IDialogue *dg, SipMessage *pMessage)
{
	BOOL tIsNewPort = TRUE;
	HS_RESULT tRet = HS_OK;
	SdpMessage *tSdp = NULL;

	if( st==NULL || dg==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;
	if( pMessage->mBody == NULL ) return HS_ERR_INVALID_PARAM;

	/* check new SDP or not
	*/
	if( pMessage->mHeadLine.mResponseValue==e_SipResponse_Request )
		tIsNewPort = TRUE;
	else
		tIsNewPort = FALSE;

	/* check which sdp or not
	*/
	if( pMessage->mBody == NULL || pMessage->mContentType == NULL ) return HS_OK;
	if( SipFindNextTokenOnlyEx(pMessage->mContentType->mInfo,0,strlen(pMessage->mContentType->mInfo),"sdp") == HS_INVALID_POS )
		return HS_OK;
	dg->mSdpState = SdpState_ChangeReceive(dg->mSdpState);

	/* decode sdp
	*/
	if( (tSdp=(SdpMessage*)HSMalloc(sizeof(SdpMessage)))==NULL )
		return HS_ERR_MALLOC;
	new_SdpMessage(tSdp);
	if( SdpMessage_Decode(tSdp,pMessage->mBody,pMessage->mBodyLen) != HS_OK )
	{
		delete_SdpMessage(tSdp);
		HSFree(tSdp);
		return HS_ERR_SIP_NOT_SUPPORT;
	}

	/* matching media type
	*/
	if( st->CallbackCheckSdp == NULL )
	{
		if( IDialogue_FindMatchMedia(dg,tSdp,tIsNewPort)!=HS_OK )
			tRet = HS_ERR_SIP_NOT_SUPPORT;
	}
	else
	{
		if( st->CallbackCheckSdp(dg,tSdp) != HS_OK )
		{
			if( IDialogue_FindMatchMedia(dg,tSdp,tIsNewPort)!=HS_OK )
				tRet = HS_ERR_SIP_NOT_SUPPORT;
		}
	}

	if( dg->mSdpState == e_SdpState_Idle )
		IDialogue_CallbackOpenMedia(dg,st);

	delete_SdpMessage(tSdp);
	HSFree(tSdp);
	return tRet;
}


HS_RESULT MakeSdpAnswer(IDialogue *dg, char *pPacket, int pLen)
{
	HS_RESULT tRet = HS_OK;
	SdpMessage *tSdp = NULL;

	pPacket[0] = '\0';
	if( dg==NULL || pPacket==NULL ) return HS_ERR_NULL_PARAM;
	if( (tSdp=IDialogue_MakeSdpAnswer(dg))==NULL ) return HS_ERR_SIP_NOT_SUPPORT;
	tRet = SdpMessage_Encode(tSdp,pPacket,pLen);

	delete_SdpMessage(tSdp);
	HSFree(tSdp);
	return tRet;
}





/*
 *
 * handling functions
 *
 */
HS_RESULT HandleSipMessage(IStack *pStack, QmEnetMessage *pMessage)
{
	HS_RESULT tRet = HS_OK;
	SipMessage *tMessage = NULL;

	IDialogue *tDialogue = NULL;
	ITransaction *tTransaction = NULL;

	if( pMessage==NULL || pStack==NULL )
		return HS_ERR_NULL_PARAM;
	if( pStack->CallbackReceiveRawData != NULL )
	{
		if( pStack->CallbackReceiveRawData(pMessage->mRaw,pMessage->mLen) == FALSE )
			return HS_OK_SIP;
	}

	/* decode message
	*/
	if( (tMessage=(SipMessage*)HSMalloc(sizeof(SipMessage)))==NULL )
		return HS_ERR_MALLOC;
	new_SipMessage(tMessage);
	if( (tRet=SipMessage_Decode(tMessage,pMessage->mRaw,pMessage->mLen)) != HS_OK )
	{
		if( pStack->CallbackBnfDecodingError != NULL )
			pStack->CallbackBnfDecodingError(pMessage->mRaw,pMessage->mLen);
		HS_SIP_MSG_CLEAR(tRet);
	}
	/* primary header field check
	*/
	if( tMessage->mCallId==NULL || tMessage->mCSeq==NULL )
	{
		HS_SIP_MSG_CLEAR(HS_ERR_SIP_NOT_SUPPORT);
	}

	/* find dialogue
	*/
	tDialogue = IStack_FindDialogueById(pStack,tMessage->mCallId->mInfo);

	/* response messages or ACK,PRACK request
	   it must have transaction.
	*/
	if( IsNeedTransaction(tMessage) )
	{
		if( tDialogue==NULL )
		{
			HS_SIP_MSG_CLEAR(HS_ERR_SIP_NO_EXIST);
		}
		if( (tTransaction=IDialogue_FindTransactionById(tDialogue,SipMessage_GetBranchId(tMessage),tMessage->mCSeq->mMethod,tMessage->mCSeq->mNumber))==NULL )
		{
			HS_SIP_MSG_CLEAR(HS_ERR_SIP_NO_EXIST);
		}
	}

	/* request message except ACK,PRACK
	*/
	else
	{
		BOOL tCallMadeHere = FALSE;

		/* make IDialogue about INVITE request
		*/
		if( tDialogue==NULL )
		{
			if( (tDialogue=MakeNewDialogue(pStack,tMessage,(pMessage->mHandle!=HS_INVALID_HANDLE)))==NULL )
			{
				HS_SIP_MSG_CLEAR(HS_ERR_SIP_MAKE_OBJECT);
			}
			if( strcmp(tMessage->mHeadLine.mMethod,"INVITE") )
			{
				tDialogue->mState = e_DialogueState_KillMe;
			}
			tCallMadeHere = TRUE;
		}

		/* make ITransaction
		*/
		if( (tTransaction=MakeNewServerTransaction(pStack,tDialogue,tMessage,pMessage->mHandle))==NULL )
		{
			if( tCallMadeHere==TRUE )
				IUa_DeleteDialogue(tDialogue->mUa,tDialogue->mHandle);
			HS_SIP_MSG_CLEAR(HS_ERR_SIP_MAKE_OBJECT);
		}
	}

	/* let's go to handler
	*/
	SipVias_Copy(&(tTransaction->mVias),&(tMessage->mVias));
	RegulatingCopiedVia(&(tTransaction->mVias));
	SipRecordRoutes2SipRoutes(&(tTransaction->mRoutes),&(tMessage->mRecordRoutes));

	if( SipMessage_IsRequest(tMessage)==TRUE )
		tRet = HandleSipRequest(tTransaction,tMessage);
	else
		tRet = HandleSipResponse(tTransaction,tMessage);

	HS_SIP_MSG_CLEAR(tRet);
}


HS_RESULT HandleSipRequest(ITransaction *pTransaction, SipMessage *pMessage)
{
	IStack *tStack = NULL;
	IUa *tUa = NULL;
	IDialogue *tDialogue = NULL;
	SipContactUnit *tFromContactUnit = NULL;

	if( pTransaction==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;
	if( pTransaction->mIsClient == TRUE ) return HS_ERR_SIP;
	if( pMessage->mHeadLine.mMethod==NULL ) return HS_ERR_NULL_PARAM;
	if( pTransaction->mDialogue==NULL ) return HS_ERR_SIP;
	tDialogue = pTransaction->mDialogue;
	if( tDialogue->mUa==NULL ) return HS_ERR_SIP;
	tUa = tDialogue->mUa;
	if( tUa->mStack==NULL ) return HS_ERR_SIP;
	tStack = tUa->mStack;

	if( pMessage->mFrom != NULL )
	{
		if( (tFromContactUnit=SipFrom_GetPrimeContactUnit(pMessage->mFrom)) != NULL )
		{
			if( tFromContactUnit->mTag != NULL )
			{
				/* some transactions(ex:OPTIONS,MESSAGE) have different peer tag with dialogue
				*/
				if( pTransaction->mIsNewPeerTag == TRUE )
					strcpy(pTransaction->mRemoteTag,tFromContactUnit->mTag);
				else
					strcpy(tDialogue->mRemoteTag,tFromContactUnit->mTag);
			}
		}
	}

	if( strcmp(pMessage->mHeadLine.mMethod,"ACK") &&
		strcmp(pMessage->mHeadLine.mMethod,"PRACK") )
		IDialogue_ChangeTarget(tDialogue,tUa,pMessage);

	if( !strcmp(pMessage->mHeadLine.mMethod,"INVITE") )
		return HandleINVITE(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"CANCEL") )
		return HandleCANCEL(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"BYE") )
		return HandleBYE(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"OPTIONS") )
		return HandleOPTIONS(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"INFO") )
		return HandleINFO(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"UPDATE") )
		return HandleUPDATE(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"SUBSCRIBE") )
		return HandleSUBSCRIBE(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"NOTIFY") )
		return HandleNOTIFY(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"MESSAGE") )
		return HandleMESSAGE(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"REFER") )
		return HandleREFER(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"ACK") )
		return HandleACK(tStack,tUa,tDialogue,pTransaction,pMessage);
	else if( !strcmp(pMessage->mHeadLine.mMethod,"PRACK") )
		return HS_OK;	/*nothing to do with PRACK message*/

	return HS_ERR_SIP_NOT_SUPPORT;
}


HS_RESULT HandleINVITE(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	SipResponse	tResponse = e_SipResponseMax;
	SipMessage *tMessage = NULL;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	/* change state on incoming
	*/
	if( tr->mState==e_TransactionStateMax )
		tr->mState = e_TransactionState_Proceeding;
	else if( (tr->mState==e_TransactionState_Proceeding || tr->mState==e_TransactionState_Completed ) &&
		tr->mLingerMessage != NULL
	)
	{
		if( (tEnetMessage=MakeQmEnetMessage(tr,tr->mLingerMessage,tr->mLingerLen))==NULL )
			return HS_ERR_MALLOC;
		_HSThreadSendMessage(st->mNetQ,HS_QM_ENET_MESSAGE,0,(LPARAM)tEnetMessage);
		return HS_OK;
	}

	/* get response value
	*/
	if( st->CallbackReceiveMessageToResponse != NULL )
		tResponse = st->CallbackReceiveMessageToResponse(ua->mHandle,dg->mHandle,pMessage);
	if( tResponse==e_SipResponseMax )
		tResponse = e_SipResponse_Ringing;

	/* SDP process
	*/
	if( CheckReceivedSdp(st,dg,pMessage) != HS_OK )
		tResponse = e_SipResponse_UnsupportedMediaType;
	/* make SDP answer
	*/
	if( tResponse==e_SipResponse_SessionProgress || tResponse==e_SipResponse_Ok )
	{
		if( MakeSdpAnswer(dg,tPacket,2000) != HS_OK )
			tResponse = e_SipResponse_UnsupportedMediaType;
	}

	/* make response
	*/
	if( tResponse==e_SipResponse_SessionProgress || tResponse==e_SipResponse_Ok )
	{
		if( (tMessage=MakeResponseXxx(st,ua,dg,tr,"application/sdp",tPacket,tResponse))==NULL )
			return HS_ERR_SIP_MAKE_MESSAGE;
	}
	else
	{
		if( (tMessage=MakeResponseXxx(st,ua,dg,tr,NULL,NULL,tResponse))==NULL )
			return HS_ERR_SIP_MAKE_MESSAGE;
	}
	/* send response
	*/
	tPos = 0;
	if( (tRet=SendSipMessage(st,ua,dg,tr,tMessage,tPacket,&tPos,2000)) != HS_OK )
	{
		HS_SIP_MSG_CLEAR(tRet);
	}
	if( tResponse==e_SipResponse_SessionProgress || tResponse==e_SipResponse_Ok )
	{
		dg->mSdpState = SdpState_ChangeSend(dg->mSdpState);

		if( dg->mSdpState == e_SdpState_Idle )
			IDialogue_CallbackOpenMedia(dg,st);
	}

	/* change state on outgoing
	*/
	if( tResponse < e_SipResponse_Ok )
	{
		dg->mState = e_DialogueState_ReadyConnect;
		tr->mState = e_TransactionState_Proceeding;
		ITransaction_SetLinger(tr,tPacket,tPos);
	}
	else if( tResponse < e_SipResponse_MultipleChoice )
	{
		dg->mState = e_DialogueState_Connect;
		tr->mState = e_TransactionState_Terminated;
		HSSetTimer(HS_SIP_TIMER_K,st->mSipQ,HS_TID_SIP_TIMER_K,tr->mHandle,e_TimerType_Once);
	}
	else
	{
		dg->mState = e_DialogueState_KillMe;
		tr->mState = e_TransactionState_Completed;
		ITransaction_SetLinger(tr,tPacket,tPos);
		tr->mTimer = HS_SIP_TIMER_G;
		HSSetTimer(tr->mTimer,st->mSipQ,HS_TID_SIP_TIMER_G,tr->mHandle,e_TimerType_Once);
		HSSetTimer(HS_SIP_TIMER_H,st->mSipQ,HS_TID_SIP_TIMER_H,tr->mHandle,e_TimerType_Once);
	}
	HS_SIP_MSG_CLEAR(HS_OK);
}


HS_RESULT HandleGeneralRequest(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	SipResponse	tResponse = e_SipResponseMax;
	SipMessage *tMessage = NULL;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	/* change state on incoming
	*/
	if( tr->mState==e_TransactionStateMax )
		tr->mState = e_TransactionState_Trying;
	else if( (tr->mState==e_TransactionState_Proceeding || tr->mState==e_TransactionState_Completed) && 
		tr->mLingerMessage != NULL
	)
	{
		if( (tEnetMessage=MakeQmEnetMessage(tr,tr->mLingerMessage,tr->mLingerLen))==NULL )
			return HS_ERR_MALLOC;
		_HSThreadSendMessage(st->mNetQ,HS_QM_ENET_MESSAGE,0,(LPARAM)tEnetMessage);
		return HS_OK;
	}

	/* get response value
	*/
	if( st->CallbackReceiveMessageToResponse != NULL )
		tResponse = st->CallbackReceiveMessageToResponse(ua->mHandle,dg->mHandle,pMessage);
	if( tResponse==e_SipResponseMax )
		tResponse = e_SipResponse_Ok;

	/* make response
	*/
	if( (tMessage=MakeResponseXxx(st,ua,dg,tr,NULL,NULL,tResponse))==NULL )
		return HS_ERR_SIP_MAKE_MESSAGE;
	/* send message
	*/
	tPos = 0;
	if( (tRet=SendSipMessage(st,ua,dg,tr,tMessage,tPacket,&tPos,2000)) != HS_OK )
	{
		HS_SIP_MSG_CLEAR(tRet);
	}

	/* change state on outgoing
	*/
	if( tResponse < e_SipResponse_Ok )
	{
		tr->mState = e_TransactionState_Proceeding;
		ITransaction_SetLinger(tr,tPacket,tPos);
	}
	else
	{
		tr->mState = e_TransactionState_Completed;
		HSSetTimer(HS_SIP_TIMER_J,st->mSipQ,HS_TID_SIP_TIMER_J,tr->mHandle,e_TimerType_Once);
	}
	HS_SIP_MSG_CLEAR(HS_OK);
}


HS_RESULT HandleDialogueDestroyRequest(
	IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage,
	const char *pMethod
)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	SipResponse	tResponse = e_SipResponseMax;

	SipMessage *tMessage = NULL;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL || pMethod==NULL ) return HS_ERR_NULL_PARAM;

	/* just callback.
	   NOTE: CANCEL,BYE message's response don't refer to return value of callback functions.
	         It JUST depend on Stack's status.
	*/
	if( !strcmp(pMethod,"CANCEL") )
	{
		if( dg->mState == e_DialogueState_ReadyConnect )
		{
			/* 487 RequestTerminated by CANCEL
			*/
			_HSThreadSendMessage(st->mSipQ,HS_QM_REMOVE_CALL,(WPARAM)e_SipResponse_RequestTerminated,(LPARAM)(dg->mHandle));
			tResponse = e_SipResponse_Ok;
		}
		else
			tResponse = e_SipResponse_BadRequest;
	}
	else if( !strcmp(pMethod,"BYE") )
	{
		if( dg->mState == e_DialogueState_Connect )
			tResponse = e_SipResponse_Ok;
		else
			tResponse = e_SipResponse_BadRequest;
	}
	else
		return HS_ERR_SIP;

	/* change state on incoming
	*/
	if( tr->mState==e_TransactionStateMax )
		tr->mState = e_TransactionState_Trying;
	else if( (tr->mState==e_TransactionState_Proceeding || tr->mState==e_TransactionState_Completed) && 
		tr->mLingerMessage != NULL
	)
	{
		if( (tEnetMessage=MakeQmEnetMessage(tr,tr->mLingerMessage,tr->mLingerLen))==NULL )
			return HS_ERR_MALLOC;
		_HSThreadSendMessage(st->mNetQ,HS_QM_ENET_MESSAGE,0,(LPARAM)tEnetMessage);
		return HS_OK;
	}

	/* just callback.
	   NOTE: CANCEL,BYE message's response don't refer to return value of callback functions.
	         It JUST depend on Stack's status.
	*/
	if( st->CallbackReceiveMessageToResponse != NULL )
		st->CallbackReceiveMessageToResponse(ua->mHandle,dg->mHandle,pMessage);

	/* make response
	*/
	if( (tMessage=MakeResponseXxx(st,ua,dg,tr,NULL,NULL,tResponse))==NULL )
		return HS_ERR_SIP_MAKE_MESSAGE;
	/* send message
	*/
	tPos = 0;
	if( (tRet=SendSipMessage(st,ua,dg,tr,tMessage,tPacket,&tPos,2000)) != HS_OK )
	{
		HS_SIP_MSG_CLEAR(tRet);
	}

	/* change state on outgoing
	*/
	if( tResponse == e_SipResponse_Ok )
	{
		if( !strcmp(pMethod,"BYE") )
			IDialogue_StateToKillMe(dg,st,ua,e_RemoveReason_Bye);

		/* close media channel
		*/
		IDialogue_CallbackCloseMedia(dg,st);
		if( dg->mAudio != NULL )
		{
			delete_MediaInfo(dg->mAudio);
			HSFree(dg->mAudio);
			dg->mAudio = NULL;
		}
		if( dg->mVideo != NULL )
		{
			delete_MediaInfo(dg->mVideo);
			HSFree(dg->mVideo);
			dg->mVideo = NULL;
		}
	}

	/* XStack send JUST '200 OK' or '400 Bad Request' about CANCEL message
	*/
	tr->mState = e_TransactionState_Completed;
	HSSetTimer(HS_SIP_TIMER_J,st->mSipQ,HS_TID_SIP_TIMER_J,tr->mHandle,e_TimerType_Once);
	HS_SIP_MSG_CLEAR(HS_OK);
}


HS_RESULT HandleCANCEL(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleDialogueDestroyRequest(st,ua,dg,tr,pMessage,"CANCEL");
}


HS_RESULT HandleBYE(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleDialogueDestroyRequest(st,ua,dg,tr,pMessage,"BYE");
}


HS_RESULT HandleOPTIONS(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleINFO(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	IDialogue_CallbackOutbandDtmf(dg,st,ua,pMessage);
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleUPDATE(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleSUBSCRIBE(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleNOTIFY(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleMESSAGE(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	IDialogue_CallbackInstantMessage(dg,st,ua,pMessage);
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleREFER(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleGeneralRequest(st,ua,dg,tr,pMessage);
}


HS_RESULT HandleACK(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	if( st->CallbackReceiveMessage != NULL )
	{
		if( st->CallbackReceiveMessage(ua->mHandle,dg->mHandle,pMessage) == FALSE )
			return HS_ERR_SIP_USER;
	}

	if( tr->mIsInvite == FALSE ) return HS_ERR_SIP;
	if( tr->mState==e_TransactionState_Completed )
	{
		/* XStack dont implement confirm state in the INVITE server model of RFC 3261
		tr->mState = e_TransactionState_Confirmed;
		HSSetTimer(HS_SIP_TIMER_I,st->mSipQ,HS_TID_SIP_TIMER_I,tr->mHandle,e_TimerType_Once);
		*/
		IDialogue_DeleteTransaction(dg,tr->mHandle);
		IUa_DeleteDialogue(ua,dg->mHandle);

		if( ua->mState==e_UaState_KillMe && ua->mDialogues.size==0 )
			IStack_DeleteUa(st,ua->mHandle);
	}
	else if( tr->mState==e_TransactionState_Terminated )
	{
		HSKillTimer(HS_TID_SIP_TIMER_K,tr->mHandle);
		IDialogue_DeleteTransaction(dg,tr->mHandle);
		/* IDialogue state is may be Connected
		   just change sdp state to idle
		*/
		dg->mSdpState = e_SdpState_Idle;
	}

	return HS_OK;
}





/* handle response
*/
HS_RESULT HandleSipResponse(ITransaction *pTransaction, SipMessage *pMessage)
{
	IStack *tStack = NULL;
	IUa *tUa = NULL;
	IDialogue *tDialogue = NULL;
	SipContactUnit *tToContactUnit = NULL;

	if( pTransaction==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;
	if( pTransaction->mIsClient == FALSE ) return HS_ERR_SIP;
	if( pTransaction->mDialogue==NULL ) return HS_ERR_SIP;
	tDialogue = pTransaction->mDialogue;
	if( tDialogue->mUa==NULL ) return HS_ERR_SIP;
	tUa = tDialogue->mUa;
	if( tUa->mStack==NULL ) return HS_ERR_SIP;
	tStack = tUa->mStack;

	if( tStack->CallbackReceiveMessage != NULL )
	{
		if( tStack->CallbackReceiveMessage(tUa->mHandle,tDialogue->mHandle,pMessage) == FALSE )
			return HS_ERR_SIP_USER;
	}

	if( pMessage->mTo != NULL )
	{
		if( (tToContactUnit=SipTo_GetPrimeContactUnit(pMessage->mTo)) != NULL )
		{
			if( tToContactUnit->mTag != NULL )
			{
				/* some transactions(ex:OPTIONS,MESSAGE) have different peer tag with dialogue
				*/
				if( pTransaction->mIsNewPeerTag==TRUE )
					strcpy(pTransaction->mRemoteTag,tToContactUnit->mTag);
				else
					strcpy(tDialogue->mRemoteTag,tToContactUnit->mTag);
			}
		}
	}

	if( pMessage->mHeadLine.mResponseValue < 100 )
		return HS_ERR_SIP_NOT_SUPPORT;
	if( pMessage->mHeadLine.mResponseValue < 200 )
		return Handle1xx(tStack,tUa,tDialogue,pTransaction,pMessage);
	if( pMessage->mHeadLine.mResponseValue < 300 )
		return Handle2xx(tStack,tUa,tDialogue,pTransaction,pMessage);
	if( pMessage->mHeadLine.mResponseValue < 400 )
		return Handle3xx(tStack,tUa,tDialogue,pTransaction,pMessage);
	if( pMessage->mHeadLine.mResponseValue < 500 )
		return Handle4xx(tStack,tUa,tDialogue,pTransaction,pMessage);
	if( pMessage->mHeadLine.mResponseValue < 600 )
		return Handle5xx(tStack,tUa,tDialogue,pTransaction,pMessage);
	if( pMessage->mHeadLine.mResponseValue > 699 )
		return HS_ERR_SIP_NOT_SUPPORT;

	return Handle6xx(tStack,tUa,tDialogue,pTransaction,pMessage);
}


HS_RESULT Handle1xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	if( pMessage->mHeadLine.mResponseValue == e_SipResponse_SessionProgress )
		CheckReceivedSdp(st,dg,pMessage);

	if( tr->mState==e_TransactionState_Proceeding ) return HS_OK;
	if( tr->mState==e_TransactionState_Trying )
	{
		HSKillTimer(HS_TID_SIP_TIMER_E,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_F,tr->mHandle);
		tr->mState = e_TransactionState_Proceeding;
		tr->mTimer = HS_SIP_TIMER_E;
		HSSetTimer(tr->mTimer,st->mSipQ,HS_TID_SIP_TIMER_E,tr->mHandle,e_TimerType_Once);
		HSSetTimer(HS_SIP_TIMER_F,st->mSipQ,HS_TID_SIP_TIMER_F,tr->mHandle,e_TimerType_Once);
		return HS_OK;
	}
	if( tr->mState==e_TransactionState_Calling )
	{
		HSKillTimer(HS_TID_SIP_TIMER_A,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_B,tr->mHandle);
		tr->mState = e_TransactionState_Proceeding;
		return HS_OK;
	}

	return HS_ERR_SIP;
}


HS_RESULT Handle2xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	HS_RESULT tSdpRet = HS_ERR;
	SipMessage *tMessage = NULL;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	/* non-INVITE transaction
	*/
	if( tr->mIsInvite == FALSE )
	{
		HSKillTimer(HS_TID_SIP_TIMER_E,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_F,tr->mHandle);

		/* check REGISTER transaction
		*/
		if( !strcmp(tr->mMethod,"REGISTER") )
		{
			HS_UINT tExpire = 0;
			if( ua->mState!=e_UaState_KillMe )
			{
				if( (tExpire=SipMessage_GetExpires(pMessage)) != 0 )
				{
					if( ua->mIsRegisted==FALSE )
					{
						if( st->CallbackRegisted != NULL )
							st->CallbackRegisted(ua->mHandle);
					}
					ua->mIsRegisted = TRUE;
					dg->mAuthCount = HS_SIP_MAX_AUTH_COUNT;
					dg->mExpires = SipUintAlloc(tExpire);

					if( tExpire > HS_SIP_TTL_MARGIN )
						tExpire -= HS_SIP_TTL_MARGIN;
					HSSetTimer( (tExpire*1000), st->mSipQ, HS_TID_SIP_REGISTER, ua->mHandle, e_TimerType_Once );
				}
				else
				{
					if( st->CallbackRegisterFail != NULL )
						st->CallbackRegisterFail(ua->mHandle);
				}
			}
		}
		/* check BYE or CANCEL transaction
		*/
		else if( !strcmp(tr->mMethod,"BYE") || !strcmp(tr->mMethod,"CANCEL") )
			dg->mState = e_DialogueState_KillMe;

		IDialogue_DeleteTransaction(dg,tr->mHandle);

		if( dg->mState==e_DialogueStateMax && ua->mState==e_UaState_KillMe && ua->mDialogues.size==0 )
			IStack_DeleteUa(st,ua->mHandle);
		else if( dg->mState==e_DialogueState_KillMe && dg->mTransactions.size==0 )
		{
			IUa_DeleteDialogue(ua,dg->mHandle);
			if( ua->mState==e_UaState_KillMe && ua->mDialogues.size==0 )
				IStack_DeleteUa(st,ua->mHandle);
		}

		return HS_OK;
	}

	/*
	 * INVITE trasaction
	 */
	IDialogue_ChangeTarget(dg,ua,pMessage);

	if( tr->mState == e_TransactionState_Calling )
	{
		HSKillTimer(HS_TID_SIP_TIMER_A,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_B,tr->mHandle);
	}

	if( IDialogue_FindTransactionByMethod(dg,"CANCEL") != NULL )
	{
		/* ! BAD CASE:
			caller sent CANCEL request to cancel the call, but
			callee sent '200 OK' about INVITE request.
			XStack will do BYE transaction in this case.
		*/
		dg->mState = e_DialogueState_KillMe;
		HS_TRY( StartNewTransaction(st,ua,dg,e_SipMethod_Bye,NULL,NULL) );
	}
	else
	{
		dg->mState = e_DialogueState_Connect;
		CheckReceivedSdp(st,dg,pMessage);
	}

	/*
	 *send ACK
	 */
#if 0
/*
 when 183 and 200 are all have SDP message,
 some UAs or proxies need couple of SDP message.
 so, ACK message must have SDP message.
 but some is not.
 NOW. we are selecting after case.
*/
	if( dg->mSdpState == e_SdpState_Received )
		tSdpRet = MakeSdpAnswer(dg,tPacket,2000);
#else
	tSdpRet = HS_ERR;
#endif

	/* IDialogue state is may be Connected
	   just change sdp state to idle
	*/
	dg->mSdpState = e_SdpState_Idle;

	if( tSdpRet==HS_OK )
		tMessage = MakeRequestAck(st,ua,dg,tr,"application/sdp",tPacket);
	else
		tMessage = MakeRequestAck(st,ua,dg,tr,NULL,NULL);

	if( tMessage!=NULL )
	{
		tPos = 0;
		tRet = SendSipMessage(st,ua,dg,tr,tMessage,tPacket,&tPos,2000);
	}
	else
		tRet = HS_ERR_SIP;

	IDialogue_DeleteTransaction(dg,tr->mHandle);
	if( tMessage != NULL )
	{
		HS_SIP_MSG_CLEAR(tRet);
	}
	else
		return tRet;
}


HS_RESULT HandleOver300(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	HS_UINT tPos = 0;
	char tPacket[2048];
	HS_RESULT tRet = HS_OK;
	BOOL tAuthRetry = FALSE;
	HS_UINT tMinExpires = 0;
	SipMessage *tMessage = NULL;
	QmEnetMessage *tEnetMessage = NULL;

	if( st==NULL || ua==NULL || dg==NULL || tr==NULL || pMessage==NULL ) return HS_ERR_NULL_PARAM;

	/* non-INVITE transaction
	*/
	if( tr->mIsInvite == FALSE )
	{
		HSKillTimer(HS_TID_SIP_TIMER_E,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_F,tr->mHandle);

		/* check REGISTER transaction
		*/
		if( !strcmp(tr->mMethod,"REGISTER") )
		{
			tAuthRetry = FALSE;

			switch(pMessage->mHeadLine.mResponseValue)
			{
				case e_SipResponse_Unauthorized:
				case e_SipResponse_ProxyAuthenticationRequired:
					dg->mAuthCount--;

					if( dg->mAuthCount==0 )
						dg->mAuthCount = HS_SIP_MAX_AUTH_COUNT;
					else
					{
						if( IDialogue_MakeAuthorization(dg,pMessage) == HS_OK )
						{
							if( TrySipRegistration(ua) == HS_OK )
								tAuthRetry = TRUE;
						}
					}
					break;

				case e_SipResponse_IntervalTooBrief:
					dg->mAuthCount--;

					if( dg->mAuthCount==0 )
						dg->mAuthCount = HS_SIP_MAX_AUTH_COUNT;
					else
					{
						if( (tMinExpires=SipMessage_GetMinExpires(pMessage))==0 ) break;
						if( IUa_ChangeExpires(ua,tMinExpires+5) != HS_OK ) break;
						tAuthRetry = TRUE;
					}
					break;

				default:
					break;
			}

			if( tAuthRetry==FALSE )
			{
				if( st->CallbackRegisterFail != NULL )
					st->CallbackRegisterFail(ua->mHandle);
			}

			IDialogue_DeleteTransaction(dg,tr->mHandle);
			return HS_OK;
		}

		/* check BYE or CANCEL transaction
		*/
		else if( !strcmp(tr->mMethod,"BYE") || !strcmp(tr->mMethod,"CANCEL") )
			dg->mState = e_DialogueState_KillMe;
		IDialogue_DeleteTransaction(dg,tr->mHandle);

		if( dg->mState==e_DialogueStateMax && ua->mState==e_UaState_KillMe && ua->mDialogues.size==0 )
			IStack_DeleteUa(st,ua->mHandle);
		if( dg->mState==e_DialogueState_KillMe && dg->mTransactions.size==0 )
		{
			IUa_DeleteDialogue(ua,dg->mHandle);
			if( ua->mState==e_UaState_KillMe && ua->mDialogues.size==0 )
				IStack_DeleteUa(st,ua->mHandle);
		}

		return HS_OK;
	}

	/*
	 * INVITE trasaction
	 */
	if( tr->mState==e_TransactionState_Completed )
	{
		if( tr->mLingerMessage!=NULL )
		{
			if( (tEnetMessage=MakeQmEnetMessage(tr,tr->mLingerMessage,tr->mLingerLen))==NULL )
				return HS_ERR_MALLOC;
			_HSThreadSendMessage(st->mNetQ,HS_QM_ENET_MESSAGE,0,(LPARAM)tEnetMessage);
			return HS_OK;
		}
	}
	
	if( tr->mState == e_TransactionState_Calling )
	{
		HSKillTimer(HS_TID_SIP_TIMER_A,tr->mHandle);
		HSKillTimer(HS_TID_SIP_TIMER_B,tr->mHandle);
	}

	/* send ACK
	*/
	if( (tMessage=MakeRequestAck(st,ua,dg,tr,NULL,NULL)) != NULL )
	{
		tPos = 0;
		tRet = SendSipMessage(st,ua,dg,tr,tMessage,tPacket,&tPos,2000);
	}
	else
		tRet = HS_ERR_SIP;

	if( tRet==HS_OK )
	{
		ITransaction_SetLinger(tr,tPacket,tPos);
		HSSetTimer(HS_SIP_TIMER_D,st->mSipQ,HS_TID_SIP_TIMER_D,tr->mHandle,e_TimerType_Once);
		tr->mState = e_TransactionState_Completed;
	}

	/* dialogue re-initialize
	*/
	dg->mSdpState = e_SdpState_Idle;
	/*IUa_MakeTag(ua,dg->mLocalTag);*/
	memset(dg->mRemoteTag,0,MAX_NAME_SIZE);

	/* 401 or 407 process
	*/
	tAuthRetry = FALSE;
	if( pMessage->mHeadLine.mResponseValue == e_SipResponse_Unauthorized ||
		pMessage->mHeadLine.mResponseValue == e_SipResponse_ProxyAuthenticationRequired
	)
	{
		dg->mAuthCount--;

		if( dg->mAuthCount==0 )
			dg->mAuthCount = HS_SIP_MAX_AUTH_COUNT;
		else
		{
			if( IDialogue_MakeAuthorization(dg,pMessage) == HS_OK )
			{
				if( SdpMessage_Encode(&(dg->mSdp),tPacket,2000)==HS_OK )
				{
					if( StartNewTransaction(st,ua,dg,e_SipMethod_Invite,"application/sdp",tPacket) == HS_OK )
					{
						tAuthRetry = TRUE;
						dg->mSdpState = SdpState_ChangeSend(dg->mSdpState);
					}
				}
			}
		}
	}

	if( tAuthRetry==FALSE )
	{
		if( st->CallbackCallEnded != NULL )
			st->CallbackCallEnded(ua->mHandle,dg->mHandle,pMessage->mHeadLine.mResponseValue);
		dg->mState = e_DialogueState_KillMe;
	}

	return tRet;
}

HS_RESULT Handle3xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleOver300(st,ua,dg,tr,pMessage);
}


HS_RESULT Handle4xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleOver300(st,ua,dg,tr,pMessage);
}


HS_RESULT Handle5xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleOver300(st,ua,dg,tr,pMessage);
}


HS_RESULT Handle6xx(IStack *st, IUa *ua, IDialogue *dg, ITransaction *tr, SipMessage *pMessage)
{
	return HandleOver300(st,ua,dg,tr,pMessage);
}


