/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<MessageUtility.c>	2005-04-17,10:47

*/

#include "MessageUtility.h"

/*warning protection*/
#include "RasMessageMaker.h"
#include "CallMessageMaker.h"
#include "CtrlMessageMaker.h"



/*****************************************************************************/
/* RAS Message
*/
HS_RESULT SendRasMessage(StackInfo *pStack, RasWaitSet *pWaitSet)
{
	HS_RESULT tRet;

	if( pStack == HS_NULL || pWaitSet == HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pWaitSet->retryCount > 0 && pWaitSet->raw != HS_NULL && pWaitSet->len > 0 )
	{
		pWaitSet->retryCount--;
		tRet = SendToRas( &(pWaitSet->toAddr), pWaitSet->raw, pWaitSet->len );

		/* if sn of #pWaitSet# is HS_INVALID_RAS_SEQUENCE_NUMBER,
		   message is not a request and don't need timer
		*/
		if( tRet == HS_OK && pWaitSet->sequenceNumber != HS_INVALID_RAS_SEQUENCE_NUMBER )
			HSSetTimer(
				HS_RAS_RETRY_TIMER, pStack->h323DemonQ, HS_TID_RAS_TRANSACTION,
				(HS_UINT)(pWaitSet->sequenceNumber), e_TimerType_Once
			);
	}

	return HS_OK;
}


/* send RAS message object to ethernet:
	check user callback functions, asn encoding,
	attach #RasWaitSet#(if request) and send rawdata on ethernet.
	NOTE : if the #pSN# parameter is 0(invalid requestSeqNum) mean that 'message is not a request'.
*/
HS_RESULT AttemptRasMessage(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, HS_CALL_HANDLE pHandle, HS_USHORT pSN, ASNH225RasMessage *pMsg)
{
	HS_RESULT tRet = HS_OK;
	AsnStream tStrm;
	RasWaitSet *tWaitSet = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL || pMsg==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pEndpoint->gatekeeper.registrar == HS_NULL ) return HS_ERR_H323;

	/* RAS message callback check
	*/
	if( pEndpoint->CallbackSendRasMessage != HS_NULL )
	{
		if( pEndpoint->CallbackSendRasMessage(pStack,pHandle,pMsg) == FALSE )
			return HS_OK;
	}

	/* encode message
	*/
	new_AsnStream(&tStrm, 1024, TRUE);
	if( (tRet=RasMsg_Encode(pMsg, &tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}
	AsnStream_Aligning(&tStrm);

	/* ethernet rawdata callback check
	*/
	if( pEndpoint->CallbackSendRasRawData != HS_NULL )
	{
		if( pEndpoint->CallbackSendRasRawData(pStack,pHandle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	if( (tWaitSet=(RasWaitSet*)HSMalloc(sizeof(RasWaitSet))) == HS_NULL )
	{
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MALLOC;
	}
	if( (tRet=new_RasWaitSet(tWaitSet, pHandle, pSN, tStrm.datas, tStrm.byteOffset)) != HS_OK )
	{
		HSFree(tWaitSet);
		delete_AsnStream(&tStrm);
		return tRet;
	}
	delete_AsnStream(&tStrm);

	if( (tRet=String2SockAddrEx(&(tWaitSet->toAddr), pEndpoint->gatekeeper.registrar->target))
		!= HS_OK )
	{
		delete_RasWaitSet(tWaitSet);
		HSFree(tWaitSet);
		return tRet;
	}

	/* when the RAS message is a request,
	   make #RasWaitSet# object and attach it on #pWaits#.
	   #pSN# == HS_INVALID_RAS_SEQUENCE_NUMBER mean 'not a request message'. 
	*/
	if( pSN != HS_INVALID_RAS_SEQUENCE_NUMBER )
	{
		if( (tRet=NoLockList_AttachData(pWaits,tWaitSet)) != HS_OK )
		{
			delete_RasWaitSet(tWaitSet);
			HSFree(tWaitSet);
			return tRet;
		}
	}

	/* message sending
	*/
	tRet = SendRasMessage(pStack,tWaitSet);

	/* RAS request message: no attach to the #pWaitSet#
	*/
	if( pSN == HS_INVALID_RAS_SEQUENCE_NUMBER )
	{
		delete_RasWaitSet(tWaitSet);
		HSFree(tWaitSet);
	}

	return tRet;
}



/*****************************************************************************/
/* Q931 Message
*/
/* checking UUIEsRequested PDU or not
*/
BOOL IsIRRwithPDU(ASNH225H323_UU_PDU_h323_message_bodyChoice pChoice, ASNH225UUIEsRequested *pRequested)
{
	switch(pChoice)
	{
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_setup:			return pRequested->m_setup.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_callProceeding:	return pRequested->m_callProceeding.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_connect:			return pRequested->m_connect.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_alerting:			return pRequested->m_alerting.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_information:		return pRequested->m_information.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_releaseComplete:	return pRequested->m_releaseComplete.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_facility:			return pRequested->m_facility.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_progress:			return pRequested->m_progress.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_empty:			return pRequested->m_empty.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_status:			return pRequested->m_status.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_statusInquiry:	return pRequested->m_statusInquiry.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_setupAcknowledge:	return pRequested->m_setupAcknowledge.value;
		case e_ASNH225H323_UU_PDU_h323_message_bodyChoice_notify:			return pRequested->m_notify.value;
	}

	return HS_NO;
}
/* start IRR timer by irrFrequency of ACF
*/
HS_RESULT StartIRRTimer(StackInfo *pStack, ICall *pCall)
{
	if( pStack==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State!=e_Q931State_Connect || pCall->irrFrequency==0 ) return HS_OK;

	return HSSetTimer(
		(pCall->irrFrequency)*1000,
		pStack->h323DemonQ,
		HS_TID_IRR,
		pCall->handle,
		e_TimerType_Forever
	);
}
/* stop IRR timer by irrFrequency of ACF
*/
HS_RESULT StopIRRTimer(ICall *pCall)
{
	if( pCall == HS_NULL ) return HS_ERR_NULL_PARAM;
	return HSKillTimer(HS_TID_IRR, pCall->handle);
}
/* make ethernet raw data by Q.931 message and try sending
*/
HS_RESULT AttemptQ931Message(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, Q931Message *pMsg, ICall *pCall)
{
	int tLen = 0;
	HS_UCHAR *tBuffer = HS_NULL;
	HS_USHORT tShort;

	HS_RESULT tRet;
	AsnStream tStrm;
	QMTcpMessage *tTcpMessage = HS_NULL;

	if( pMsg==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	/*message callback check*/
	if( pEndpoint->CallbackSendQ931Message != HS_NULL )
	{
		if( pEndpoint->CallbackSendQ931Message(pStack,pCall->handle,pMsg) == FALSE )
			return HS_OK;
	}

	/*encode message*/
	new_AsnStream(&tStrm, 2048, TRUE);
	if( (tRet=Q931Message_Encode(pMsg, &tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}
	AsnStream_Aligning(&tStrm);

	/*rawdata callback check*/
	if( pEndpoint->CallbackSendQ931RawData != HS_NULL )
	{
		if( pEndpoint->CallbackSendQ931RawData(pStack,pCall->handle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	tLen = tStrm.byteOffset + 4;	/*for TPKT*/
	if( (tBuffer=(HS_UCHAR*)HSMalloc(tLen)) == HS_NULL )
	{
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MALLOC;
	}

	tBuffer[0] = 0x03;	tBuffer[1] = 0x00;
	tShort = (HS_USHORT)(tLen);
	Ushort2Uchar(tBuffer+2, &tShort);
	memcpy(tBuffer+4, tStrm.datas, tStrm.byteOffset);

	if( (tTcpMessage=new_QMTcpMessage(tBuffer, tLen, pCall->handle)) == HS_NULL )
	{
		HSFree(tBuffer);
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MALLOC;
	}

	_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_Q931_MESSAGE, 0, (LPARAM)tTcpMessage);
	delete_AsnStream(&tStrm);

	/* uuiesRequested IRR message send
	*/
	if( pMsg->uuie != HS_NULL )
	{
		if( IsIRRwithPDU(pMsg->uuie->m_h323_uu_pdu.m_h323_message_body.choice, &(pCall->uuiesRequested)) )
		{
			HS_USHORT tSN;
			RasMsg *tIrr = MakeIRRwithPDU(pEndpoint, pCall, &(pMsg->uuie->m_h323_uu_pdu), HS_YES, &tSN);

			if( tIrr != HS_NULL )
			{
				AttemptRasMessage(pStack, pEndpoint, pWaits, pCall->handle, HS_INVALID_RAS_SEQUENCE_NUMBER, tIrr);
				delete_RasMsg(tIrr);
				HSFree(tIrr);
			}
		}
	}

	return HS_OK;
}



/*****************************************************************************/
/* H245 Message
*/
HS_RESULT AttemptH245Message(StackInfo *pStack, IEndpoint *pEndpoint, ControlMsg *pMsg, ICall *pCall)
{
	int tLen = 0;
	HS_UCHAR *tBuffer = HS_NULL;
	HS_USHORT tShort;

	HS_RESULT tRet;
	AsnStream tStrm;
	QMTcpMessage *tTcpMessage = HS_NULL;

	if( pMsg == HS_NULL || pStack == HS_NULL || pEndpoint == HS_NULL || pCall == HS_NULL )
		return HS_ERR_NULL_PARAM;

	/*message callback check*/
	if( pEndpoint->CallbackSendH245Message != HS_NULL )
	{
		if( pEndpoint->CallbackSendH245Message(pStack,pCall->handle,pMsg) == FALSE )
			return HS_OK;
	}

	/*encode message*/
	new_AsnStream(&tStrm, 2048, TRUE);
	if( (tRet=ControlMsg_Encode(pMsg, &tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}
	AsnStream_Aligning(&tStrm);

	/*rawdata callback check*/
	if( pEndpoint->CallbackSendH245RawData != HS_NULL )
	{
		if( pEndpoint->CallbackSendH245RawData(pStack,pCall->handle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	tLen = tStrm.byteOffset + 4;	/*for TPKT*/
	if( (tBuffer=(HS_UCHAR*)HSMalloc(tLen)) == HS_NULL )
	{
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MALLOC;
	}

	tBuffer[0] = 0x03;	tBuffer[1] = 0x00;
	tShort = (HS_USHORT)(tLen);
	Ushort2Uchar(tBuffer+2, &tShort);
	memcpy(tBuffer+4, tStrm.datas, tStrm.byteOffset);

	if( (tTcpMessage=new_QMTcpMessage(tBuffer, tLen, pCall->handle)) == HS_NULL )
	{
		HSFree(tBuffer);
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MALLOC;
	}

	_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_H245_MESSAGE, 0, (LPARAM)tTcpMessage);
	delete_AsnStream(&tStrm);
	return tRet;
}



/*****************************************************************************/
/* FastStart Proceedure
*/
static HS_RESULT GetMediaAddressFromOLC(
	ICall *pCall, ASNH245DataTypeChoice pType,
	ASNH245H2250LogicalChannelParameters *p2250
)
{
	if( pCall==HS_NULL || p2250==HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pType )
	{
		case e_ASNH245DataTypeChoice_audioData:
			ASNH245TransportAddress2SockAddrIn( &(pCall->aRtpIn), &(p2250->m_mediaChannel) );
			ASNH245TransportAddress2SockAddrIn( &(pCall->aRtcpIn), &(p2250->m_mediaControlChannel) );
			return HS_OK;
		case e_ASNH245DataTypeChoice_videoData:
			ASNH245TransportAddress2SockAddrIn( &(pCall->vRtpIn), &(p2250->m_mediaChannel) );
			ASNH245TransportAddress2SockAddrIn( &(pCall->vRtcpIn), &(p2250->m_mediaControlChannel) );
			return HS_OK;
		case e_ASNH245DataTypeChoice_data:
			ASNH245TransportAddress2SockAddrIn( &(pCall->dRtpIn), &(p2250->m_mediaChannel) );
			ASNH245TransportAddress2SockAddrIn( &(pCall->dRtcpIn), &(p2250->m_mediaControlChannel) );
			return HS_OK;
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT AddForwardFastStart(void *pCall, ASNH245OpenLogicalChannel *pOlc)
{
	ICall *tCall = (ICall*)pCall;
	if( tCall==HS_NULL || pOlc==HS_NULL ) return HS_ERR_NULL_PARAM;

	if(tCall->isAnswerCall == HS_NO)
	{
		GetMediaAddressFromOLC(
			tCall, pOlc->m_forwardLogicalChannelParameters.m_dataType.choice,
			pOlc->m_forwardLogicalChannelParameters.m_multiplexParameters.alter
		);
	}

	NoLockList_AttachData( &(tCall->fsList), pOlc );
	return HS_OK;
}


HS_RESULT AddReverseFastStart(void *pCall, ASNH245OpenLogicalChannel *pOlc)
{
	ICall *tCall = (ICall*)pCall;
	if( tCall==HS_NULL || pOlc==HS_NULL ) return HS_ERR_NULL_PARAM;

	if(tCall->isAnswerCall == HS_YES)
	{
		GetMediaAddressFromOLC(
			tCall, pOlc->m_reverseLogicalChannelParameters.m_dataType.choice,
			pOlc->m_reverseLogicalChannelParameters.m_multiplexParameters.alter
		);
	}

	NoLockList_AttachData( &(tCall->fsList), pOlc );
	return HS_OK;
}


static HS_RESULT CheckFastStartUnit(AsnOctetString *pOctetString, ICall *pCall, MediaChannelState *pState )
{
	HS_UINT i;
	ChainUnit *tChain = HS_NULL;
	ICapability *tCapa = HS_NULL;
	NoLockList *tCapas = HS_NULL;

	AsnStream tStrm;
	ASNH245OpenLogicalChannel *olc = HS_NULL;

	/*validation check*/
	if( pOctetString==HS_NULL || pCall==HS_NULL || pState==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pOctetString->length==0 || pOctetString->value==HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ Null FastStart Unit." );
#endif
		return HS_ERR_NULL_PARAM;
	}
	tCapas = &(pCall->capabilities);

	/*decode faststart unit*/
	olc = (ASNH245OpenLogicalChannel*)HSMalloc( sizeof(ASNH245OpenLogicalChannel) );
	if( olc == HS_NULL ) return HS_ERR_H323_MALLOC;

	new_ASNH245OpenLogicalChannel(olc);
	ASNH245OpenLogicalChannel_MakeMold(olc);
	new_AsnStream(&tStrm, pOctetString->length, TRUE);
	memcpy(tStrm.datas, pOctetString->value, pOctetString->length);

	if( ASNH245OpenLogicalChannel_Decode(olc,&tStrm) != HS_OK )
	{
		delete_ASNH245OpenLogicalChannel(olc);
		HSFree(olc);
		delete_AsnStream(&tStrm);

		/*
			if it is a calling endpoint, faststart unit is a response value.
			so, call must stopped.
			otherwise it is a called endpoint, faststart unit is a request value.
			and be able to skip one or more..
		*/
		if( pCall->isAnswerCall ) return HS_OK;
		else					  return HS_ERR_H323_MAKE_MESSAGE;
	}

	/*check reverse logical channel*/
	if( ASNH245OpenLogicalChannel_IsIncludeOptionField(olc, e_ASNH245OpenLogicalChannelOptionMap_reverseLogicalChannelParameters ) &&
		olc->m_reverseLogicalChannelParameters.m_dataType.choice != e_ASNH245DataTypeChoice_nullData
	)
	{
		/*channel selected already*/
		if( (olc->m_reverseLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_audioData &&
			 pState->rAudio == HS_YES ) ||
			(olc->m_reverseLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_videoData &&
			 pState->rVideo == HS_YES ) ||
			(olc->m_reverseLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_data &&
			 pState->rData == HS_YES )
		)
		{
			delete_ASNH245OpenLogicalChannel(olc);
			HSFree(olc);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}

		/*select channel*/
		for( i=0; i<tCapas->size; i++ )
		{
			tChain = NoLockList_GetChain(tCapas, i);
			if( tChain == HS_NULL ) continue;

			tCapa = (ICapability*)(tChain->data);
			if( tCapa == HS_NULL ) continue;

			if( CompareCapabilityWithDataType( &(tCapa->capability), &(olc->m_reverseLogicalChannelParameters.m_dataType) ) == HS_YES )
			{
				switch( olc->m_reverseLogicalChannelParameters.m_dataType.choice )
				{
					case e_ASNH245DataTypeChoice_videoData:	pState->rVideo = HS_YES;	break;
					case e_ASNH245DataTypeChoice_audioData:	pState->rAudio = HS_YES;	break;
					case e_ASNH245DataTypeChoice_data:		pState->rData = HS_YES;		break;
					default:
						/*abnormal case*/
						delete_ASNH245OpenLogicalChannel(olc);
						HSFree(olc);
						delete_AsnStream(&tStrm);
						return HS_ERR_H323;
				}

				delete_AsnStream(&tStrm);
				return AddReverseFastStart(pCall, olc);
			}
		}
	}

	/*check forward logical channel*/
	if( olc->m_forwardLogicalChannelParameters.m_dataType.choice != e_ASNH245DataTypeChoice_nullData )
	{
		/*channel selected already*/
		if( (olc->m_forwardLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_audioData &&
			 pState->sAudio == HS_YES ) ||
			(olc->m_forwardLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_videoData &&
			 pState->sVideo == HS_YES ) ||
			(olc->m_forwardLogicalChannelParameters.m_dataType.choice == e_ASNH245DataTypeChoice_data &&
			 pState->sData == HS_YES )
		)
		{
			delete_ASNH245OpenLogicalChannel(olc);
			HSFree(olc);
			delete_AsnStream(&tStrm);
			return HS_OK;
		}

		/*select channel*/
		for( i=0; i<tCapas->size; i++ )
		{
			tChain = NoLockList_GetChain(tCapas, i);
			if( tChain == HS_NULL ) continue;

			tCapa = (ICapability*)(tChain->data);
			if( tCapa == HS_NULL ) continue;

			if( CompareCapabilityWithDataType( &(tCapa->capability), &(olc->m_forwardLogicalChannelParameters.m_dataType) ) == HS_YES )
			{
				switch( olc->m_forwardLogicalChannelParameters.m_dataType.choice )
				{
					case e_ASNH245DataTypeChoice_videoData:	pState->sVideo = HS_YES;	break;
					case e_ASNH245DataTypeChoice_audioData:	pState->sAudio = HS_YES;	break;
					case e_ASNH245DataTypeChoice_data:		pState->sData = HS_YES;		break;
					default:
						/*abnormal case*/
						delete_ASNH245OpenLogicalChannel(olc);
						HSFree(olc);
						delete_AsnStream(&tStrm);
						return HS_ERR_H323;
				}

				delete_AsnStream(&tStrm);
				return AddForwardFastStart(pCall, olc);
			}
		}
	}

	/*
		if it is a calling endpoint, faststart unit is a response value.
		so, call must stopped.
		otherwise it is a called endpoint, faststart unit is a request value.
		and be able to skip one or more..
	*/
	delete_AsnStream(&tStrm);
	if( pCall->isAnswerCall ) return HS_OK;
	else					  return HS_ERR_H323_NOTFOUND;
}


HS_RESULT CheckFastStart(AsnSequenceOf *pSequenceOf, ICall *pCall)
{
	HS_UINT i;
	HS_RESULT tRet = HS_OK;
	MediaChannelState mState;

	if( pSequenceOf==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pSequenceOf->size==0 || pCall->capabilities.size==0 ) return HS_ERR_H323_NOUNIT;
	/*Recommand: H.323(09/99) 8.1.7 Fast Connect Procedure*/
	if( pCall->q931State == e_Q931State_Connect ) return HS_ERR_H323_CONFLICT;

	new_MediaChannelState(&mState);
	for( i=0; i<pSequenceOf->size; i++ )
	{
		tRet = CheckFastStartUnit( AsnSequenceOf_GetUnit(pSequenceOf, i), pCall, &mState );
		if( tRet != HS_OK )
		{
			delete_MediaChannelState(&mState);
			return tRet;
		}
	}

	delete_MediaChannelState(&mState);
	if(pCall->fsList.size == 0) return HS_ERR_H323_NOTFOUND;
	return HS_OK;
}


HS_RESULT OpenMediaWithFastStart( StackInfo *pStack, IEndpoint *pEndpoint, ICall *pCall )
{
	HS_UINT i;
	ChainUnit *rounder = HS_NULL;
	ASNH245OpenLogicalChannel *olc = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	rounder = pCall->fsList.units;
	for( i=0; i<pCall->fsList.size; i++ )
	{
		if( rounder==HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data==HS_NULL ) return HS_ERR_H323_CONFLICT;

		olc = (ASNH245OpenLogicalChannel*)(rounder->data);

		/*reverse channel : caller->receive, called->send*/
		if( ASNH245OpenLogicalChannel_IsIncludeOptionField( olc,
				e_ASNH245OpenLogicalChannelOptionMap_reverseLogicalChannelParameters
			)
		)
		{
			switch( olc->m_reverseLogicalChannelParameters.m_dataType.choice )
			{
				case e_ASNH245DataTypeChoice_audioData:
					if( pCall->isAnswerCall == HS_NO )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort), (HS_USHORT)(pCall->rtpListenPort+1),
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->aRtpIn, pCall->aRtcpIn,
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					break;
				case e_ASNH245DataTypeChoice_videoData:
					if( pCall->isAnswerCall == HS_NO )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort+2), (HS_USHORT)(pCall->rtpListenPort+3),
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->vRtpIn, pCall->vRtcpIn,
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					break;
				case e_ASNH245DataTypeChoice_data:
					if( pCall->isAnswerCall == HS_NO )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort+4), (HS_USHORT)(pCall->rtpListenPort+5),
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->dRtpIn, pCall->dRtcpIn,
								&(olc->m_reverseLogicalChannelParameters.m_dataType)
							);
					}
					break;

				default:
					break;
			}/*switch*/
		}/*if reverse*/

		/*forward channel : caller->send, called->receive*/
		else
		{
			switch( olc->m_forwardLogicalChannelParameters.m_dataType.choice )
			{
				case e_ASNH245DataTypeChoice_audioData:
					if( pCall->isAnswerCall == HS_YES )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort), (HS_USHORT)(pCall->rtpListenPort+1),
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->aRtpIn, pCall->aRtcpIn,
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					break;
				case e_ASNH245DataTypeChoice_videoData:
					if( pCall->isAnswerCall == HS_YES )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort+2), (HS_USHORT)(pCall->rtpListenPort+3),
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->vRtpIn, pCall->vRtcpIn,
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					break;
				case e_ASNH245DataTypeChoice_data:
					if( pCall->isAnswerCall == HS_YES )
					{
						if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
							pEndpoint->CallbackOpenReceiveMedia(
								pCall->handle, (HS_USHORT)(pCall->rtpListenPort+4), (HS_USHORT)(pCall->rtpListenPort+5),
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					else
					{
						if( pEndpoint->CallbackOpenSendMedia != HS_NULL );
							pEndpoint->CallbackOpenSendMedia(
								pCall->handle, pCall->dRtpIn, pCall->dRtcpIn,
								&(olc->m_forwardLogicalChannelParameters.m_dataType)
							);
					}
					break;

				default:
					break;
			}/*switch*/
		}/*if forward*/

		rounder = rounder->next;
	}/*for faststart list*/

	pCall->fsActive = HS_YES;
	return HS_OK;
}


HS_RESULT CloseMediaWithFastStart( StackInfo *pStack, IEndpoint *pEndpoint, ICall *pCall )
{
	HS_UINT i;
	ChainUnit *rounder = HS_NULL;
	ASNH245OpenLogicalChannel *olc = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->fsActive == HS_NO ) return HS_OK;

	rounder = pCall->fsList.units;
	for( i=0; i<pCall->fsList.size; i++ )
	{
		if( rounder==HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data==HS_NULL ) return HS_ERR_H323_CONFLICT;

		olc = (ASNH245OpenLogicalChannel*)(rounder->data);

		/*reverse channel : caller->receive, called->send*/
		if( ASNH245OpenLogicalChannel_IsIncludeOptionField( olc,
				e_ASNH245OpenLogicalChannelOptionMap_reverseLogicalChannelParameters
			)
		)
		{
			if( pCall->isAnswerCall == HS_NO )
			{
				if( pEndpoint->CallbackCloseReceiveMedia != HS_NULL )
					pEndpoint->CallbackCloseReceiveMedia(
						pCall->handle, &(olc->m_reverseLogicalChannelParameters.m_dataType)
					);
			}
			else
			{
				if( pEndpoint->CallbackCloseSendMedia != HS_NULL )
					pEndpoint->CallbackCloseSendMedia(
						pCall->handle,
						&(olc->m_reverseLogicalChannelParameters.m_dataType)
					);
			}
		}/*if reverse*/

		/*forward channel : caller->send, called->receive*/
		else
		{
			if( pCall->isAnswerCall == HS_YES )
			{
				if( pEndpoint->CallbackCloseReceiveMedia != HS_NULL )
					pEndpoint->CallbackCloseReceiveMedia(
						pCall->handle, &(olc->m_forwardLogicalChannelParameters.m_dataType)
					);
			}
			else
			{
				if( pEndpoint->CallbackCloseSendMedia != HS_NULL )
					pEndpoint->CallbackCloseSendMedia(
						pCall->handle,
						&(olc->m_forwardLogicalChannelParameters.m_dataType)
					);
			}
		}/*if forward*/

		rounder = rounder->next;
	}/*for faststart list*/

	return HS_OK;
}



/****************************************************************/
/* Media Negociation by H.245 Message
*/
HS_RESULT AddOpenLogicalChannel(IEndpoint *pEndpoint, ICall *pCall, ICapability *pCapa)
{
	HS_RESULT tRet = HS_OK;
	ControlMsg *tOlcMsg = HS_NULL;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pCapa==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( (tOlcMsg=MakeOpenLogicalChannel(pCapa, pEndpoint, pCall)) == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	if( (tRet=NoLockList_AttachData(&(pCall->h245Charge), tOlcMsg)) != HS_OK )
	{
		delete_ControlMsg(tOlcMsg);
		HSFree(tOlcMsg);
		return tRet;
	}

	return HS_OK;
}


HS_RESULT AddOpenLogicalChannelEx(IEndpoint *pEndpoint, ICall *pCall, HS_UINT pIndex)
{
	ChainUnit *tChain = HS_NULL;

	if( pEndpoint==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( (tChain=NoLockList_GetChain(&(pCall->capabilities),pIndex)) == HS_NULL )
		return HS_ERR_H323_NOUNIT;

	return AddOpenLogicalChannel(pEndpoint, pCall, tChain->data);
}


static HS_RESULT CheckTerminalCapabilitySetUnit(
	StackInfo *pStack, IEndpoint *pEndpoint, ICall *pCall,
	ASNH245CapabilityTableEntry *cte, MediaChannelState *pState
)
{
	HS_UINT i;
	HS_RESULT tRet = HS_OK;
	ChainUnit *rounder = HS_NULL;
	ICapability *tCapa = HS_NULL;
	NoLockList *tCapas = HS_NULL;
	OpenLogicalChannelSet *tOlcSet = HS_NULL;

	if( cte==HS_NULL || pStack==HS_NULL || pCall==HS_NULL || pState==HS_NULL )
		return HS_ERR_NULL_PARAM;
	if( pCall->capabilities.size == 0 ) return HS_ERR_H323_NOUNIT;

	if(cte->m_capability.choice > e_ASNH245CapabilityChoice_nonStandard)
	{
		if( cte->m_capability.choice < e_ASNH245CapabilityChoice_receiveAudioCapability &&
			pState->sVideo == HS_YES ) return HS_ERR_H323_NOTFOUND;
		else if(cte->m_capability.choice < e_ASNH245CapabilityChoice_receiveDataApplicationCapability &&
			pState->sAudio == HS_YES ) return HS_ERR_H323_NOTFOUND;
		else if(cte->m_capability.choice < e_ASNH245CapabilityChoice_h233EncryptionTransmitCapability &&
			pState->sData == HS_YES ) return HS_ERR_H323_NOTFOUND;
	}
	tCapas = &(pCall->capabilities);

	rounder = tCapas->units;
	for( i=0; i<tCapas->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data == HS_NULL ) return HS_ERR_H323_CONFLICT;

		tCapa = (ICapability*)(rounder->data);
		if( CompareCapability(&(cte->m_capability), &(tCapa->capability)) )
		{
			if( (tRet=AddOpenLogicalChannel(pEndpoint, pCall, tCapa)) != HS_OK ) return tRet;

			if( tCapa->capability.choice > e_ASNH245CapabilityChoice_nonStandard )
			{
				if( tCapa->capability.choice < e_ASNH245CapabilityChoice_receiveAudioCapability )
					pState->sVideo = HS_YES;
				else if( tCapa->capability.choice < e_ASNH245CapabilityChoice_receiveDataApplicationCapability )
					pState->sAudio = HS_YES;
				else if( tCapa->capability.choice < e_ASNH245CapabilityChoice_h233EncryptionTransmitCapability )
					pState->sData = HS_YES;
			}
		}

		rounder = rounder->next;
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT CheckTerminalCapabilitySet(StackInfo *pStack, IEndpoint *pEndpoint, ICall *pCall, ASNH245TerminalCapabilitySet *pTcs)
{
	HS_UINT i;
	MediaChannelState mState;

	if( pTcs==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	new_MediaChannelState(&mState);
	for( i=0; i<pTcs->m_capabilityTable.size; i++ )
	{
		CheckTerminalCapabilitySetUnit(
			pStack, pEndpoint, pCall,
			AsnSequenceOf_GetUnit(&(pTcs->m_capabilityTable), i), &mState
		);
	}

	if( pCall->h245Charge.size == 0 ) return HS_ERR_H323_NOUNIT;
	return HS_OK;
}


HS_RESULT CloseMediaUnit(OpenLogicalChannelSet *pOlcSet, IEndpoint *pEndpoint, ICall *pCall, BOOL isSending)
{
	ASNH245OpenLogicalChannel *tOlc = HS_NULL;

	if( pOlcSet==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	tOlc = (ASNH245OpenLogicalChannel*)
		(((ASNH245RequestMessage*)(pOlcSet->olcMsg->alter))->alter);

	if( isSending )
	{
		if( pEndpoint->CallbackCloseSendMedia != HS_NULL )
			pEndpoint->CallbackCloseSendMedia(
				pCall->handle,
				&(tOlc->m_forwardLogicalChannelParameters.m_dataType)
			);
	}
	else
	{
		if( pEndpoint->CallbackCloseReceiveMedia != HS_NULL )
			pEndpoint->CallbackCloseReceiveMedia(
				pCall->handle,
				&(tOlc->m_forwardLogicalChannelParameters.m_dataType)
			);
	}

	return HS_OK;
}


HS_RESULT CloseMedia(StackInfo *pStack, IEndpoint *pEndpoint, ICall *pCall)
{
	HS_UINT i;
	ChainUnit *rounder = HS_NULL;
	OpenLogicalChannelSet *tOlcSet = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	/*close sending media channel*/
	rounder = pCall->sendedOlcSetList.units;
	for( i=0; i<pCall->sendedOlcSetList.size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data == HS_NULL ) return HS_ERR_H323_CONFLICT;

		tOlcSet = (OpenLogicalChannelSet*)(rounder->data);
		if( tOlcSet->isOpen == HS_YES && tOlcSet->olcMsg != HS_NULL )
			CloseMediaUnit(tOlcSet, pEndpoint, pCall, HS_YES);

		rounder = rounder->next;
	}

	/*close receiving media channel*/
	rounder = pCall->receivedOlcSetList.units;
	for( i=0; i<pCall->receivedOlcSetList.size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data == HS_NULL ) return HS_ERR_H323_CONFLICT;

		tOlcSet = (OpenLogicalChannelSet*)(rounder->data);
		if( tOlcSet->isOpen == HS_YES && tOlcSet->olcMsg != HS_NULL )
			CloseMediaUnit(tOlcSet, pEndpoint, pCall, HS_NO);

		rounder = rounder->next;
	}

	return HS_OK;
}



/****************************************************************/
/* H245, H245Tunneling Procedure
*/
static HS_RESULT SendH245Handler(IEndpoint *pEndpoint, ICall *pCall, ControlMsg *pMsg);

HS_RESULT DischargeH245Message(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet;
	HS_UINT i, tListSize;
	ControlMsg *tOlcMsg = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* h245 tunneling mode
	*/
	if( pCall->isTunneling == HS_YES )
	{
		/* if call connection is established,
		   send Q.931 empty packet.
		*/
		if( pCall->q931State > e_Q931State_ReadySetup && pCall->h245Charge.size != 0 )
		{
			Q931Message *tQ931Msg = MakeQ931Empty(pStack, pEndpoint, pWaits, pCall);
			if(tQ931Msg != HS_NULL)
				AttemptQ931Message(pStack, pEndpoint, pWaits, tQ931Msg, pCall);

			delete_Q931Message(tQ931Msg);
			HSFree(tQ931Msg);
		}
		return HS_OK;
	}

	if( pCall->h245State.session == HS_NO ) return HS_OK;
	if( pCall->h245Charge.size == 0 ) return HS_OK;

	/* Get ControlMsg One by One and processing(send message)
	*/
	tListSize = pCall->h245Charge.size;
	for( i=0; i<tListSize; i++)
	{
		tOlcMsg = (ControlMsg*)NoLockList_DetachData( &(pCall->h245Charge), 0 );
		if( tOlcMsg == HS_NULL ) break;

		if( (tRet=AttemptH245Message(pStack, pEndpoint, tOlcMsg, pCall)) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on DischargeH245Message() -> tRet:%u", tRet );
#endif
		}
		else
		{
			if( SendH245Handler(pEndpoint, pCall, tOlcMsg) == HS_OK_SAVE_IN_OLC )
				NoLockList_AttachOpenLogicalChannelSet(&(pCall->sendedOlcSetList), tOlcMsg);
			else
				delete_ControlMsg(tOlcMsg);
		}
	}

	return NoLockList_Clear( &(pCall->h245Charge) );
}


static HS_RESULT TunnelingH245MessageUnit(IEndpoint *pEndpoint, ICall *pCall, AsnOctetString *pOctetString, ControlMsg *pMsg)
{
	HS_RESULT tRet;
	AsnStream tStrm;

	if( pOctetString==HS_NULL || pMsg==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* message callback check
	*/
	if( pEndpoint->CallbackSendH245Message != HS_NULL )
	{
		if( pEndpoint->CallbackSendH245Message(HS_INVALID_STACK_HANDLE,pCall->handle,pMsg) == FALSE )
			return HS_OK;
	}

	/* encode message
	*/
	new_AsnStream(&tStrm, 2048, TRUE);
	if( (tRet=ControlMsg_Encode(pMsg, &tStrm)) != HS_OK )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on TunnelingH245MessageUnit() -> encoding, tRet:%u", tRet );
#endif
		delete_AsnStream(&tStrm);
		return HS_ERR_H323_MAKE_MESSAGE;
	}

	/* rawdata callback check
	*/
	if( pEndpoint->CallbackSendH245RawData != HS_NULL )
	{
		if( pEndpoint->CallbackSendH245RawData(HS_INVALID_STACK_HANDLE,pCall->handle,&tStrm) == FALSE )
		{
			delete_AsnStream(&tStrm);
			return HS_OK;
		}
	}

	AsnStream_Aligning(&tStrm);
	AsnOctetString_SetValue(pOctetString, tStrm.byteOffset, tStrm.datas);
	delete_AsnStream(&tStrm);
	return HS_OK;
}


HS_RESULT TunnelingH245Message(IEndpoint *pEndpoint, ICall *pCall, ASNH225H323_UserInformation *pMsg)
{
	HS_RESULT tRet;
	HS_UINT i, tListSize;
	ControlMsg *tOlcMsg = HS_NULL;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pMsg==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->h245Charge.size == 0 ) return HS_OK;

	/* make h245Control space on Q.931 message
	*/
	ASNH225H323_UU_PDU_IncludeOptionField(&(pMsg->m_h323_uu_pdu), e_ASNH225H323_UU_PDUOptionMap_h245Control);
	AsnSequenceOf_SetSize( &(pMsg->m_h323_uu_pdu.m_h245Control), pCall->h245Charge.size );

	/* Get ControlMsg One by One and processing(put it on h245Contorl)
	*/
	tListSize = pCall->h245Charge.size;
	for( i=0; i<tListSize; i++ )
	{
		tOlcMsg = (ControlMsg*)NoLockList_DetachData( &(pCall->h245Charge), 0 );
		if( tOlcMsg == HS_NULL ) break;

		tRet=TunnelingH245MessageUnit( 
				pEndpoint,
				pCall,
				AsnSequenceOf_GetUnit(&(pMsg->m_h323_uu_pdu.m_h245Control),i),
				tOlcMsg
		);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on TunnelingH245Message() -> tRet:%u", tRet );
#endif
			return NoLockList_Clear( &(pCall->h245Charge) );
		}

		if( SendH245Handler(pEndpoint, pCall, tOlcMsg) == HS_OK_SAVE_IN_OLC )
			NoLockList_AttachOpenLogicalChannelSet(&(pCall->sendedOlcSetList), tOlcMsg);
		else
			delete_ControlMsg(tOlcMsg);
	}

	return NoLockList_Clear( &(pCall->h245Charge) );
}






/*****************************************************************************/
/* H245 Sending Event of Internal
*/
static HS_RESULT SendH245OpenLogicalChannelAckHandler(
	ASNH245OpenLogicalChannelAck *pOlcAck, IEndpoint *pEndpoint, ICall *pCall
)
{
	HS_RESULT tRet = HS_OK;
	ChainUnit *tChain = HS_NULL;
	OpenLogicalChannelSet *tOlcSet = HS_NULL;
	HS_UINT tMediaPort = HS_INVALID_PORT;
	HS_UINT tMediaControlPort = HS_INVALID_PORT;
	ASNH245OpenLogicalChannel *tOlc = HS_NULL;

	if( pOlcAck==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	pCall->h245State.sendOLCack = HS_YES;

	/*finding OpenLogicalChannelSet*/
	tChain = NoLockList_FindOpenLogicalChannelSetByChannelNumber(
		&(pCall->receivedOlcSetList),
		pOlcAck->m_forwardLogicalChannelNumber.inheritance.value
	);

	if( tChain == HS_NULL ) return HS_ERR_H323_NOUNIT;
	if( (tOlcSet=(OpenLogicalChannelSet*)(tChain->data)) == HS_NULL ) return HS_ERR_H323_CONFLICT;
	if( tOlcSet->olcMsg == HS_NULL ) return HS_ERR_H323_CONFLICT;

	/*get media tsap address*/
	tRet = GetTsapPortFromOpenLogicalChannelAck(pOlcAck, &tMediaPort, &tMediaControlPort);
	if( tRet != HS_OK ) return tRet;
	if( tMediaPort == HS_INVALID_PORT ) return HS_ERR_H323;

	/*media starting event*/
	tOlc = (ASNH245OpenLogicalChannel*)(((ASNH245RequestMessage*)(tOlcSet->olcMsg->alter))->alter);
	if( pEndpoint->CallbackOpenReceiveMedia != HS_NULL )
		pEndpoint->CallbackOpenReceiveMedia(
			pCall->handle,
			(HS_USHORT)tMediaPort,
			(HS_USHORT)tMediaControlPort,
			&(tOlc->m_forwardLogicalChannelParameters.m_dataType)
		);

	tOlcSet->isOpen = HS_YES;
	return HS_OK;
}


static HS_RESULT SendH245RequestHandler(ASNH245RequestMessage *pRequest, IEndpoint *pEndpoint, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pRequest == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch(pRequest->choice)
	{
		case e_ASNH245RequestMessageChoice_nonStandard:			break;
		case e_ASNH245RequestMessageChoice_masterSlaveDetermination:
			pCall->h245State.sendMSD = HS_YES;
			break;
		case e_ASNH245RequestMessageChoice_terminalCapabilitySet:
			pCall->h245State.sendTCS = HS_YES;
			break;
		case e_ASNH245RequestMessageChoice_openLogicalChannel:
			pCall->h245State.sendOLC = HS_YES;
			tRet = HS_OK_SAVE_IN_OLC;
			break;
		case e_ASNH245RequestMessageChoice_closeLogicalChannel:
		case e_ASNH245RequestMessageChoice_requestChannelClose:
		case e_ASNH245RequestMessageChoice_multiplexEntrySend:
		case e_ASNH245RequestMessageChoice_requestMultiplexEntry:
		case e_ASNH245RequestMessageChoice_requestMode:
		case e_ASNH245RequestMessageChoice_roundTripDelayRequest:
		case e_ASNH245RequestMessageChoice_maintenanceLoopRequest:
		case e_ASNH245RequestMessageChoice_communicationModeRequest:
		case e_ASNH245RequestMessageChoice_conferenceRequest:
		case e_ASNH245RequestMessageChoice_multilinkRequest:
		case e_ASNH245RequestMessageChoice_logicalChannelRateRequest:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}
static HS_RESULT SendH245ResponseHandler(ASNH245ResponseMessage *pResponse, IEndpoint *pEndpoint, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pResponse == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch(pResponse->choice)
	{
		case e_ASNH245ResponseMessageChoice_nonStandard:			break;
		case e_ASNH245ResponseMessageChoice_masterSlaveDeterminationAck:
			pCall->h245State.sendMSDack = HS_YES;
			break;
		case e_ASNH245ResponseMessageChoice_masterSlaveDeterminationReject:	break;
		case e_ASNH245ResponseMessageChoice_terminalCapabilitySetAck:
			pCall->h245State.sendTCSack = HS_YES;
			break;
		case e_ASNH245ResponseMessageChoice_terminalCapabilitySetReject:	break;
		case e_ASNH245ResponseMessageChoice_openLogicalChannelAck:
			return SendH245OpenLogicalChannelAckHandler(
				pResponse->alter, pEndpoint, pCall
			);
		case e_ASNH245ResponseMessageChoice_openLogicalChannelReject:
		case e_ASNH245ResponseMessageChoice_closeLogicalChannelAck:
		case e_ASNH245ResponseMessageChoice_requestChannelCloseAck:
		case e_ASNH245ResponseMessageChoice_requestChannelCloseReject:
		case e_ASNH245ResponseMessageChoice_multiplexEntrySendAck:
		case e_ASNH245ResponseMessageChoice_multiplexEntrySendReject:
		case e_ASNH245ResponseMessageChoice_requestMultiplexEntryAck:
		case e_ASNH245ResponseMessageChoice_requestMultiplexEntryReject:
		case e_ASNH245ResponseMessageChoice_requestModeAck:
		case e_ASNH245ResponseMessageChoice_requestModeReject:
		case e_ASNH245ResponseMessageChoice_roundTripDelayResponse:
		case e_ASNH245ResponseMessageChoice_maintenanceLoopAck:
		case e_ASNH245ResponseMessageChoice_maintenanceLoopReject:
		case e_ASNH245ResponseMessageChoice_communicationModeResponse:
		case e_ASNH245ResponseMessageChoice_conferenceResponse:
		case e_ASNH245ResponseMessageChoice_multilinkResponse:
		case e_ASNH245ResponseMessageChoice_logicalChannelRateAcknowledge:
		case e_ASNH245ResponseMessageChoice_logicalChannelRateReject:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}
static HS_RESULT SendH245CommandHandler(ASNH245CommandMessage *pCommand, IEndpoint *pEndpoint, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pCommand == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pCommand->choice )
	{
		case e_ASNH245CommandMessageChoice_nonStandard:
		case e_ASNH245CommandMessageChoice_maintenanceLoopOffCommand:
		case e_ASNH245CommandMessageChoice_sendTerminalCapabilitySet:
		case e_ASNH245CommandMessageChoice_encryptionCommand:
		case e_ASNH245CommandMessageChoice_flowControlCommand:
		case e_ASNH245CommandMessageChoice_endSessionCommand:
		case e_ASNH245CommandMessageChoice_miscellaneousCommand:
		case e_ASNH245CommandMessageChoice_communicationModeCommand:
		case e_ASNH245CommandMessageChoice_conferenceCommand:
		case e_ASNH245CommandMessageChoice_h223MultiplexReconfiguration:
		case e_ASNH245CommandMessageChoice_newATMVCCommand:
		case e_ASNH245CommandMessageChoice_mobileMultilinkReconfigurationCommand:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}
static HS_RESULT SendH245IndicationHandler(ASNH245IndicationMessage *pIndication, IEndpoint *pEndpoint, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pEndpoint==HS_NULL || pCall==HS_NULL || pIndication == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pIndication->choice )
	{
		case e_ASNH245IndicationMessageChoice_nonStandard:
		case e_ASNH245IndicationMessageChoice_functionNotUnderstood:
		case e_ASNH245IndicationMessageChoice_masterSlaveDeterminationRelease:
		case e_ASNH245IndicationMessageChoice_terminalCapabilitySetRelease:
		case e_ASNH245IndicationMessageChoice_openLogicalChannelConfirm:
		case e_ASNH245IndicationMessageChoice_requestChannelCloseRelease:
		case e_ASNH245IndicationMessageChoice_multiplexEntrySendRelease:
		case e_ASNH245IndicationMessageChoice_requestMultiplexEntryRelease:
		case e_ASNH245IndicationMessageChoice_requestModeRelease:
		case e_ASNH245IndicationMessageChoice_miscellaneousIndication:
		case e_ASNH245IndicationMessageChoice_jitterIndication:
		case e_ASNH245IndicationMessageChoice_h223SkewIndication:
		case e_ASNH245IndicationMessageChoice_newATMVCIndication:
		case e_ASNH245IndicationMessageChoice_userInput:
		case e_ASNH245IndicationMessageChoice_h2250MaximumSkewIndication:
		case e_ASNH245IndicationMessageChoice_mcLocationIndication:
		case e_ASNH245IndicationMessageChoice_conferenceIndication:
		case e_ASNH245IndicationMessageChoice_vendorIdentification:
		case e_ASNH245IndicationMessageChoice_functionNotSupported:
		case e_ASNH245IndicationMessageChoice_multilinkIndication:
		case e_ASNH245IndicationMessageChoice_logicalChannelRateRelease:
		case e_ASNH245IndicationMessageChoice_flowControlIndication:
		case e_ASNH245IndicationMessageChoice_mobileMultilinkReconfigurationIndication:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}


static HS_RESULT SendH245Handler(IEndpoint *pEndpoint, ICall *pCall, ControlMsg *pMsg)
{
	if( pEndpoint==HS_NULL || pCall==HS_NULL || pMsg==HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pMsg->choice )
	{
		case e_ASNH245MultimediaSystemControlMessageChoice_request:
			return SendH245RequestHandler(pMsg->alter, pEndpoint, pCall);
		case e_ASNH245MultimediaSystemControlMessageChoice_response:
			return SendH245ResponseHandler(pMsg->alter, pEndpoint, pCall);
		case e_ASNH245MultimediaSystemControlMessageChoice_command:
			return SendH245CommandHandler(pMsg->alter, pEndpoint, pCall);
		case e_ASNH245MultimediaSystemControlMessageChoice_indication:
			return SendH245IndicationHandler(pMsg->alter, pEndpoint, pCall);
	}

	return HS_ERR_H323_NOTFOUND;
}
