/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<MessageHandler.c>	2005-03-20,11:18

*/

#include "MessageHandler.h"



/**************************************************************/
/* Registration
*/
HS_RESULT TryRegistration(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, BOOL isFoundGk )
{
	HS_USHORT tSN;
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;

	if(pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL) return HS_ERR_NULL_PARAM;
	if(pEndpoint->gatekeeper.registrar == HS_NULL ) return HS_ERR_H323;

	if( pEndpoint->gatekeeper.registrar->registed == FALSE &&
		pEndpoint->gatekeeper.isGRQ &&
		isFoundGk == FALSE
	)
		msg = MakeGRQ(pEndpoint,&tSN);
	else
		msg = MakeRRQ(pEndpoint,&tSN);

	if( msg == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;
	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, HS_INVALID_CALL_HANDLE, tSN, msg);
	
	if( tRet == HS_OK )
		pEndpoint->gatekeeper.registrar->sn = tSN;

	delete_RasMsg(msg);
	HSFree(msg);
	return tRet;
}


HS_RESULT TryUnregistration(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	HS_USHORT tSN;
	HS_RESULT tRet = HS_OK;
	RasMsg *urq = HS_NULL;

	if(pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL) return HS_ERR_NULL_PARAM;
	if( pEndpoint->gatekeeper.registrar == HS_NULL ) return HS_OK;
	if( pEndpoint->gatekeeper.registrar->registed == FALSE ) return HS_OK;
	if( (urq=MakeURQ(pEndpoint,&tSN)) == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, HS_INVALID_CALL_HANDLE, tSN, urq);

	delete_RasMsg(urq);
	HSFree(urq);
	return tRet;
}


HS_RESULT CallProcessing(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall )
{
	HS_RESULT tRet;
	Q931Message *tMsg = HS_NULL;
	HS_UCHAR tH245Ip[4];
	HS_USHORT tH245Port;
	TcpConnect *tTcpListen = NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	/*h245 listen*/
	if( pCall->isTunneling==HS_NO &&
		(pCall->h245ListenPoint == e_H245ListenPoint_CallProceed ||
		pCall->h245ListenPoint == e_H245ListenPoint_Alert)
	)
	{
		tRet = ASNH225TransportAddress2Tsap(tH245Ip, &tH245Port, &(pEndpoint->csaTsap));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on Call Proceeding." );
#endif
				return tRet;
		}
		else
		{
			tTcpListen = newm_TcpConnectEx(tH245Ip,(HS_USHORT)(pCall->h245ListenPort),0,pCall->handle);
			if( tTcpListen==NULL )
			{
#ifdef HS_DEBUG_H323
				HSPrint( "\n $ H245 Listen Fail on on Call Proceeding." );
#endif
				return HS_ERR_H323_MAKE_QMESSAGE;
			}
			else
			{
				_HSThreadSendMessage(pStack->enetDemonQ,HS_QMESSAGE_TRY_LISTEN,0,(LPARAM)tTcpListen);
				Sleep(10);	/*time to try listen*/
			}
		}
	}

	/*send Q931 callproceeding*/
	if( (tMsg=MakeQ931CallProceed(pStack,pEndpoint,pWaits,pCall)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on CallProcessing() -> Q931 CallProceed Making Error, skip" );
#endif
		if( pCall->h245ListenPoint == e_H245ListenPoint_CallProceed )
			pCall->h245ListenPoint  = e_H245ListenPoint_Alert;
		if( pCall->fastStartPoint == e_FastStartResponsePoint_CallProceed )
			pCall->fastStartPoint  = e_FastStartResponsePoint_CallProceed;
	}
	else
	{
		if( AttemptQ931Message(pStack, pEndpoint, pWaits, tMsg, pCall) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on CallProcessing() -> Q931 CallProceed Sending Error, skip" );
#endif
			if( pCall->h245ListenPoint == e_H245ListenPoint_CallProceed )
				pCall->h245ListenPoint  = e_H245ListenPoint_Alert;
			if( pCall->fastStartPoint == e_FastStartResponsePoint_CallProceed )
				pCall->fastStartPoint  = e_FastStartResponsePoint_Alert;
		}
		else
		{
			if( pCall->fastStartPoint == e_FastStartResponsePoint_CallProceed && tMsg->uuie != HS_NULL )
			{
				if( ASNH225CallProceeding_UUIE_IsIncludeOptionField(
						tMsg->uuie->m_h323_uu_pdu.m_h323_message_body.alter,
						e_ASNH225CallProceeding_UUIEOptionMap_fastStart
					) && pCall->fsActive == HS_NO
				) OpenMediaWithFastStart(pStack, pEndpoint, pCall);
			}
		}

		delete_Q931Message(tMsg);
		HSFree(tMsg);
		tMsg = HS_NULL;
	}

	/*send Q931 alert*/
	if( (tMsg=MakeQ931Alert(pStack,pEndpoint,pWaits,pCall)) == HS_NULL )
	{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on CallProcessing() -> Q931 Alert Making Error, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall, e_CallCloseReason_Undefined,
			e_ASNH225DisengageReasonChoice_normalDrop, e_Q931CauseType_NormalCallClearing
		);
	}

	if( AttemptQ931Message(pStack, pEndpoint, pWaits, tMsg, pCall) != HS_OK )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on CallProcessing() -> Q931 Alert Sending Error, Call Closing.." );
#endif
		delete_Q931Message(tMsg);
		HSFree(tMsg);

		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall, e_CallCloseReason_Undefined,
			e_ASNH225DisengageReasonChoice_normalDrop, e_Q931CauseType_NormalCallClearing
		);
	}

	if( pCall->fastStartPoint == e_FastStartResponsePoint_Alert && tMsg->uuie != HS_NULL )
	{
		if( ASNH225Alerting_UUIE_IsIncludeOptionField(
				tMsg->uuie->m_h323_uu_pdu.m_h323_message_body.alter,
				e_ASNH225Alerting_UUIEOptionMap_fastStart
			) && pCall->fsActive == HS_NO
		) OpenMediaWithFastStart(pStack, pEndpoint, pCall);
	}

	delete_Q931Message(tMsg);
	HSFree(tMsg);

	pCall->q931State = e_Q931State_ReadyConnect;
	return HS_OK;
}



/******************************************************************************************/
/* RAS message handlers 
*/
/* RAS response message handlers
*/
HS_RESULT GCFHandler( ASNH225GatekeeperConfirm *gcf, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( gcf==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(gcf->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n GCF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(gcf->m_requestSeqNum.inheritance.value));

	if( pEndpoint->gatekeeper.registrar == HS_NULL ) return HS_ERR_H323_NOTFOUND;
	pEndpoint->gatekeeper.registrar->sn = HS_INVALID_RAS_SEQUENCE_NUMBER;

	if( ASNH225GatekeeperConfirm_IsIncludeOptionField(gcf,e_ASNH225GatekeeperConfirmOptionMap_gatekeeperIdentifier) )
		AsnDoubleString_SetValue(
			&(pEndpoint->gatekeeper.gatekeeperIdentifier),
			gcf->m_gatekeeperIdentifier.inheritance.value,
			gcf->m_gatekeeperIdentifier.inheritance.length
		);

	return TryRegistration( pStack, pEndpoint, pWaits, TRUE );
}


HS_RESULT GRJHandler( ASNH225GatekeeperReject *grj, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( grj==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(grj->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n GRJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(grj->m_requestSeqNum.inheritance.value));

	AsnDoubleString_SetValue( &(pEndpoint->gatekeeper.endpointIdentifier), HS_NULL, 0 );

	if( pEndpoint->gatekeeper.registrar != HS_NULL )
	{
		delete_Registrar( pEndpoint->gatekeeper.registrar );
		HSFree( pEndpoint->gatekeeper.registrar );
		pEndpoint->gatekeeper.registrar = HS_NULL;
	}

	HSSetTimer(
		pEndpoint->gatekeeper.attemptTime*1000,
		pStack->h323DemonQ, HS_TID_REGISTER, 0, e_TimerType_Once
	);

	return HS_OK;
}


HS_RESULT RCFHandler( ASNH225RegistrationConfirm *rcf, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( rcf==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(rcf->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n RCF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(rcf->m_requestSeqNum.inheritance.value));

	if( pEndpoint->gatekeeper.registrar == HS_NULL ) return HS_ERR_H323_NOTFOUND;
	pEndpoint->gatekeeper.registrar->registed = TRUE;
	pEndpoint->gatekeeper.registrar->sn = HS_INVALID_RAS_SEQUENCE_NUMBER;

	if( ASNH225RegistrationConfirm_IsIncludeOptionField(rcf,e_ASNH225RegistrationConfirmOptionMap_gatekeeperIdentifier) )
		AsnDoubleString_SetValue(
			&(pEndpoint->gatekeeper.gatekeeperIdentifier),
			rcf->m_gatekeeperIdentifier.inheritance.value,
			rcf->m_gatekeeperIdentifier.inheritance.length
		);

	AsnDoubleString_SetValue(
		&(pEndpoint->gatekeeper.endpointIdentifier),
		rcf->m_endpointIdentifier.inheritance.value,
		rcf->m_endpointIdentifier.inheritance.length
	);

	if( pEndpoint->gatekeeper.ttlForce == HS_NO )
	{
		if( ASNH225RegistrationConfirm_IsIncludeOptionField(rcf,e_ASNH225RegistrationConfirmOptionMap_timeToLive) )
			pEndpoint->gatekeeper.ttl = rcf->m_timeToLive.inheritance.value;
	}

	HSSetTimer(
		(pEndpoint->gatekeeper.ttl - HS_TTL_EXTRA_TIME)*1000,
		pStack->h323DemonQ,
		HS_TID_REGISTER, 0,
		e_TimerType_Once
	);

	return HS_OK;
}


HS_RESULT RRJHandler( ASNH225RegistrationReject *rrj, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( rrj==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(rrj->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n RRJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(rrj->m_requestSeqNum.inheritance.value));

	AsnDoubleString_SetValue( &(pEndpoint->gatekeeper.endpointIdentifier), HS_NULL, 0 );

	if( pEndpoint->gatekeeper.registrar != HS_NULL )
	{
		delete_Registrar( pEndpoint->gatekeeper.registrar );
		HSFree( pEndpoint->gatekeeper.registrar );
		pEndpoint->gatekeeper.registrar = HS_NULL;
	}

	HSSetTimer(
		pEndpoint->gatekeeper.attemptTime*1000,
		pStack->h323DemonQ, HS_TID_REGISTER, 0, e_TimerType_Once
	);

	return HS_OK;
}


HS_RESULT UCFHandler( ASNH225UnregistrationConfirm *ucf, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( ucf==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(ucf->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n UCF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(ucf->m_requestSeqNum.inheritance.value));

	return AsnDoubleString_SetValue( &(pEndpoint->gatekeeper.endpointIdentifier), HS_NULL, 0 );
}


HS_RESULT URJHandler( ASNH225UnregistrationReject *urj, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( urj==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(urj->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n URJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}

	return HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(urj->m_requestSeqNum.inheritance.value));
}


HS_RESULT ACFHandler( ASNH225AdmissionConfirm *acf, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits )
{
	HS_RESULT tRet = HS_OK;
	ICall *tCall = HS_NULL;
	RasWaitSet *tWaitSet = HS_NULL;
	ChainUnit *tWaitChain = HS_NULL;
	ChainUnit *tCallChain = HS_NULL;

	HS_UCHAR tDestIp[4];
	HS_USHORT tDestPort;
	TcpConnect *tTcpCon = NULL;


	if( acf==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	tWaitChain = NoLockList_FindRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(acf->m_requestSeqNum.inheritance.value) );
	if( tWaitChain == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n ACF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(acf->m_requestSeqNum.inheritance.value));

	/*get ras wait set*/
	if( (tWaitSet=(RasWaitSet*)(tWaitChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pWaits, tWaitChain);
		return HS_ERR_H323;
	}

	/* finding call
	*/
	if( (tCallChain=NoLockList_FindCallByHandle(pCalls, tWaitSet->handle)) == HS_NULL )
	{
		NoLockList_DeleteChain(pWaits, tWaitChain);
		return HS_ERR_H323_NOTFOUND;
	}
	NoLockList_DeleteChain(pWaits, tWaitChain);

	if( (tCall=(ICall*)(tCallChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pCalls, tCallChain);
		return HS_ERR_H323;
	}
	tCall->isAdmission = HS_YES;

	/* get destination TSAP address
	*/
	tRet = ASNH225TransportAddress2Tsap( tDestIp, &tDestPort, &(acf->m_destCallSignalAddress) );
	if( tRet != HS_OK )
	{
		CallClosing(
			pStack, pEndpoint, pCalls, pWaits, tCall, e_CallCloseReason_UnknownDestination,
			e_ASNH225DisengageReasonChoice_normalDrop, e_Q931CauseType_NormalCallClearing
		);
		return HS_ERR_H323;
	}

	/* get irrFrequency, callModel and uuiesRequested
	*/
	if( ASNH225AdmissionConfirm_IsIncludeOptionField(acf, e_ASNH225AdmissionConfirmOptionMap_irrFrequency) )
		tCall->irrFrequency = acf->m_irrFrequency.value;
	tCall->callModel = acf->m_callModel.choice;
	tCall->uuiesRequested.m_setup.value				= acf->m_uuiesRequested.m_setup.value;
	tCall->uuiesRequested.m_callProceeding.value	= acf->m_uuiesRequested.m_callProceeding.value;
	tCall->uuiesRequested.m_connect.value			= acf->m_uuiesRequested.m_connect.value;
	tCall->uuiesRequested.m_alerting.value			= acf->m_uuiesRequested.m_alerting.value;
	tCall->uuiesRequested.m_information.value		= acf->m_uuiesRequested.m_information.value;
	tCall->uuiesRequested.m_releaseComplete.value	= acf->m_uuiesRequested.m_releaseComplete.value;
	tCall->uuiesRequested.m_facility.value			= acf->m_uuiesRequested.m_facility.value;
	tCall->uuiesRequested.m_progress.value			= acf->m_uuiesRequested.m_progress.value;
	tCall->uuiesRequested.m_empty.value				= acf->m_uuiesRequested.m_empty.value;
	tCall->uuiesRequested.m_status.value			= acf->m_uuiesRequested.m_status.value;
	tCall->uuiesRequested.m_statusInquiry.value		= acf->m_uuiesRequested.m_statusInquiry.value;
	tCall->uuiesRequested.m_setupAcknowledge.value	= acf->m_uuiesRequested.m_setupAcknowledge.value;
	tCall->uuiesRequested.m_notify.value			= acf->m_uuiesRequested.m_notify.value;

	/* calling party procedure
	*/
	if( tCall->isAnswerCall == HS_NO )
	{
		/* mapping destination aliases by canMapAlias field
		*/
		if( tCall->canMapAlias == HS_YES )
		{
			if( ASNH225AdmissionConfirm_IsIncludeOptionField(acf, e_ASNH225AdmissionConfirmOptionMap_destinationInfo) )
			{
				NoLockList_Clear(&(tCall->destAliases));
				AsnSequenceOf2AliasAddresses(&(tCall->destAliases), &(acf->m_destinationInfo));
			}
			if( ASNH225AdmissionConfirm_IsIncludeOptionField(acf, e_ASNH225AdmissionConfirmOptionMap_destExtraCallInfo) )
			{
				NoLockList_Clear(&(tCall->destExtraCallInfo));
				AsnSequenceOf2AliasAddresses(&(tCall->destExtraCallInfo), &(acf->m_destExtraCallInfo));
			}
			if( ASNH225AdmissionConfirm_IsIncludeOptionField(acf, e_ASNH225AdmissionConfirmOptionMap_remoteExtensionAddress) )
			{
				NoLockList_Clear(&(tCall->remoteExtensionAddress));
				AsnSequenceOf2AliasAddresses(&(tCall->remoteExtensionAddress), &(acf->m_remoteExtensionAddress));
			}
		}
		
		/* queue message making
		*/
		tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,0,tCall->handle);
		if( tTcpCon == NULL )
		{
			CallClosing(
				pStack, pEndpoint, pCalls, pWaits, tCall, e_CallCloseReason_UnknownDestination,
				e_ASNH225DisengageReasonChoice_normalDrop, e_Q931CauseType_NormalCallClearing
			);
			return HS_ERR_H323;
		}

		/*send message*/
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_MAKE_SOCKET_SET, 0, (LPARAM)(tCall->handle) );
		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);
	}
	/*called party procedure*/
	else return CallProcessing(pStack, pEndpoint, pCalls, pWaits, tCall);

	return HS_OK;
}


HS_RESULT ARJHandler( ASNH225AdmissionReject *arj, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits )
{
	RasWaitSet *tWaitSet = HS_NULL;
	ChainUnit *tWaitChain = HS_NULL;
	ICall *tCall = HS_NULL;
	ChainUnit *tCallChain = HS_NULL;

	if( arj==HS_NULL || pEndpoint==HS_NULL || pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	tWaitChain = NoLockList_FindRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(arj->m_requestSeqNum.inheritance.value) );
	if( tWaitChain == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n ARJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}
	HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(arj->m_requestSeqNum.inheritance.value));

	if( (tWaitSet=(RasWaitSet*)(tWaitChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pWaits, tWaitChain);
		return HS_ERR_H323;
	}

	/*finding call*/
	if( (tCallChain=NoLockList_FindCallByHandle(pCalls, tWaitSet->handle)) == HS_NULL )
	{
		NoLockList_DeleteChain(pWaits, tWaitChain);
		return HS_OK;
	}
	if( (tCall=(ICall*)(tCallChain->data)) == HS_NULL )
	{
		NoLockList_DeleteChain(pCalls, tCallChain);
		NoLockList_DeleteChain(pWaits, tWaitChain);
		return HS_OK;
	}
	NoLockList_DeleteChain(pWaits, tWaitChain);

	return CallClosing(
		pStack, pEndpoint, pCalls, pWaits, tCall, e_CallCloseReason_Admission,
		e_ASNH225DisengageReasonChoice_normalDrop, e_Q931CauseType_NormalCallClearing
	);
}


HS_RESULT BCFHandler( ASNH225BandwidthConfirm *bcf, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits )
{
	ChainUnit *tChain = HS_NULL;
	RasWaitSet *tWaitSet = HS_NULL;

	if( bcf==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* finding wait set
	*/
	tChain=NoLockList_FindRasWaitSetBySequenceNumber(pWaits, (HS_USHORT)(bcf->m_requestSeqNum.inheritance.value));
	if( tChain == HS_NULL )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n BCF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}

	/* get new bandwidth
	*/
	if( (tWaitSet=(RasWaitSet*)(tChain->data)) != HS_NULL )
	{
		ICall *tCall = HS_NULL;
		ChainUnit *cChain = HS_NULL;

		cChain = NoLockList_FindCallByHandle(pCalls,tWaitSet->handle);
		if( cChain != HS_NULL )
		{
			if( (tCall=(ICall*)(cChain->data)) != HS_NULL )
				tCall->bandwidth = bcf->m_bandWidth.inheritance.value;
		}
	}
	NoLockList_DeleteChain(pWaits, tChain);

	return HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(bcf->m_requestSeqNum.inheritance.value));
}


HS_RESULT BRJHandler( ASNH225BandwidthReject *brj, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( brj==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(brj->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n BRJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}

	return HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(brj->m_requestSeqNum.inheritance.value));	
}


HS_RESULT DCFHandler( ASNH225DisengageConfirm *dcf, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( dcf==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(dcf->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n DCF have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}

	return HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(dcf->m_requestSeqNum.inheritance.value));	
}


HS_RESULT DRJHandler( ASNH225DisengageReject *drj, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	if( drj==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( NoLockList_DeleteRasWaitSetBySequenceNumber( pWaits, (HS_USHORT)(drj->m_requestSeqNum.inheritance.value) )
		== HS_ERR_H323_NOTFOUND )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n DRJ have a invalid sn, ignored." );
#endif
		return HS_ERR_H323_NOTFOUND;
	}

	return HSKillTimer(HS_TID_RAS_TRANSACTION, (HS_UINT)(drj->m_requestSeqNum.inheritance.value));	
}


/* RAS request message handlers
*/
/* force unregistration process
*/
HS_RESULT URQHandler( ASNH225UnregistrationRequest *urq,StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	HS_RESULT tRet = HS_OK;
	RasMsg *ucf = HS_NULL;

	if( urq==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	/*hsremain: check gatekeeperIdentifier, if exist*/

	/* NOTE: send UCF in NO CONDITION (whether registed or not).
	*/
	ucf = MakeUCF( (HS_USHORT)(urq->m_requestSeqNum.inheritance.value) );
	if( ucf == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;
	tRet = AttemptRasMessage(
		pStack, pEndpoint, pWaits,
		HS_INVALID_CALL_HANDLE,
		HS_INVALID_RAS_SEQUENCE_NUMBER,/*sn=0(HS_INVALID_RAS_SEQUENCE_NUMBER): mean not a request*/
		ucf
	);
	delete_RasMsg(ucf);
	HSFree(ucf);

	/* remove registrar object
	*/
	if( pEndpoint->gatekeeper.registrar != HS_NULL )
	{
		/* delete #RasWaitSet# of GRQ or RRQ */
		if( pEndpoint->gatekeeper.registrar->sn != HS_INVALID_RAS_SEQUENCE_NUMBER )
			NoLockList_DeleteRasWaitSetBySequenceNumber(pWaits,pEndpoint->gatekeeper.registrar->sn);

		delete_Registrar( pEndpoint->gatekeeper.registrar );
		HSFree(pEndpoint->gatekeeper.registrar);
		pEndpoint->gatekeeper.registrar = HS_NULL;

		/* To alert 'registration attempt expired' to user,
		   registration TTL timer will be killed by follow.
		*/
		HSSetTimer(
			pEndpoint->gatekeeper.attemptTime*1000,
			pStack->h323DemonQ, HS_TID_REGISTER, 0, e_TimerType_Once
		);
	}
	AsnDoubleString_SetValue( &(pEndpoint->gatekeeper.endpointIdentifier), HS_NULL, 0 );

	return tRet;
}

/* force bandwidth setting process
*/
static HS_RESULT DoBCF(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, ICall *pCall, HS_USHORT pSN, HS_UINT pBandwidth)
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	msg = MakeBCF(pSN, pBandwidth);
	if( msg == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, pCall->handle, HS_INVALID_RAS_SEQUENCE_NUMBER, msg);
	delete_RasMsg(msg);
	HSFree(msg);

	if( tRet == HS_OK )
		pCall->bandwidth = pBandwidth;
	return tRet;
}
static HS_RESULT DoBRJ(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, ICall *pCall, HS_USHORT pSN, HS_UINT pBandwidth, ASNH225BandRejectReasonChoice pReason)
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	HS_CALL_HANDLE tHandle;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pCall != HS_NULL ) tHandle = pCall->handle;
	else tHandle = HS_INVALID_CALL_HANDLE;

	msg = MakeBRJ(pSN, pReason, pBandwidth);
	if( msg == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, tHandle, HS_INVALID_RAS_SEQUENCE_NUMBER, msg);
	delete_RasMsg(msg);
	HSFree(msg);
	return tRet;
}
HS_RESULT BRQHandler( ASNH225BandwidthRequest *brq, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits )
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;

	if( brq==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* not registed
	*/
	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper)) == HS_NO )
		return DoBRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(brq->m_requestSeqNum.inheritance.value), 0,
			e_ASNH225BandRejectReasonChoice_undefinedReason
		);

	/* cant find call
	*/
	if( brq->m_conferenceID.inheritance.length != 16 ) return HS_ERR_H323_CONFLICT;
	if( (tChain=NoLockList_FindCallByConferenceId(pCalls, brq->m_conferenceID.inheritance.value)) == HS_NULL )
		return DoBRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(brq->m_requestSeqNum.inheritance.value), 0,
			e_ASNH225BandRejectReasonChoice_invalidConferenceID
		);
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
		return DoBRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(brq->m_requestSeqNum.inheritance.value), 0,
			e_ASNH225BandRejectReasonChoice_invalidConferenceID
		);
	if( tCall->crv != brq->m_callReferenceValue.inheritance.value )
		return DoBRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(brq->m_requestSeqNum.inheritance.value), 0,
			e_ASNH225BandRejectReasonChoice_invalidConferenceID
		);

	return DoBCF(
		pStack, pEndpoint, pWaits, tCall, 
		(HS_USHORT)(brq->m_requestSeqNum.inheritance.value),
		tCall->bandwidth
	);
}

/* force call drop process
*/
static HS_RESULT DoDCF(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, ICall *pCall, HS_USHORT pSN)
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	msg = MakeDCF(pSN);
	if( msg == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, pCall->handle, HS_INVALID_RAS_SEQUENCE_NUMBER, msg);
	delete_RasMsg(msg);
	HSFree(msg);

	if( tRet == HS_OK )
		pCall->isAdmission = HS_NO;
	return tRet;
}
static HS_RESULT DoDRJ(StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits, ICall *pCall, HS_USHORT pSN, ASNH225DisengageRejectReasonChoice pReason)
{
	HS_RESULT tRet = HS_OK;
	RasMsg *msg = HS_NULL;
	HS_CALL_HANDLE tHandle;

	if( pStack==HS_NULL || pEndpoint==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pCall != HS_NULL ) tHandle = pCall->handle;
	else tHandle = HS_INVALID_CALL_HANDLE;

	msg = MakeDRJ(pSN, pReason);
	if( msg == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;

	tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, tHandle, HS_INVALID_RAS_SEQUENCE_NUMBER, msg);
	delete_RasMsg(msg);
	HSFree(msg);
	return tRet;
}
HS_RESULT DRQHandler( ASNH225DisengageRequest *drq, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits )
{
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;

	if( drq==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* not registed
	*/
	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper)) == HS_NO )
		return DoDRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(drq->m_requestSeqNum.inheritance.value),
			e_ASNH225DisengageRejectReasonChoice_notRegistered
		);

	/* cant find call
	*/
	if( drq->m_conferenceID.inheritance.length != 16 ) return HS_ERR_H323_CONFLICT;
	if( (tChain=NoLockList_FindCallByConferenceId(pCalls, drq->m_conferenceID.inheritance.value)) == HS_NULL )
		return DoDRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(drq->m_requestSeqNum.inheritance.value),
			e_ASNH225DisengageRejectReasonChoice_requestToDropOther
		);
	if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
		return DoDRJ(
			pStack,pEndpoint,pWaits,HS_NULL,
			(HS_USHORT)(drq->m_requestSeqNum.inheritance.value),
			e_ASNH225DisengageRejectReasonChoice_requestToDropOther
		);
	if( tCall->crv != drq->m_callReferenceValue.inheritance.value )
		return DoDRJ(
			pStack,pEndpoint,pWaits,tCall,
			(HS_USHORT)(drq->m_requestSeqNum.inheritance.value),
			e_ASNH225DisengageRejectReasonChoice_requestToDropOther
		);

	DoDCF(pStack, pEndpoint, pWaits, tCall, (HS_USHORT)(drq->m_requestSeqNum.inheritance.value));

	return CallClosing(
		pStack, pEndpoint, pCalls, pWaits, tCall,
		e_CallCloseReason_ForceDrop, e_ASNH225DisengageReasonChoice_normalDrop,
		e_Q931CauseType_NormalCallClearing
	);
}
/* call information request handler
*/
HS_RESULT IRQHandler(ASNH225InfoRequest *irq, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits)
{
	HS_RESULT tRet = HS_OK;
	RasMsg *tIrr = HS_NULL;
	ICall *tCall = HS_NULL;
	ChainUnit *tChain = HS_NULL;

	if( irq==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL ) return HS_ERR_NULL_PARAM;

	/* 'crv == 0' means that must report about all call
	*/
	if( irq->m_callReferenceValue.inheritance.value == 0 )
		tIrr = MakeIRRwithAll(pEndpoint, (HS_USHORT)(irq->m_requestSeqNum.inheritance.value), pCalls);
	/* just one call reporting
	*/
	else
	{
		/* finding call
		*/
		tChain = NoLockList_FindCallByCRV( pCalls, (HS_USHORT)(irq->m_callReferenceValue.inheritance.value) );
		if( tChain == HS_NULL ) return HS_ERR_H323_NOTFOUND;
		if( (tCall=(ICall*)(tChain->data)) == HS_NULL )
		{
			NoLockList_DeleteChain(pCalls, tChain);
			return HS_ERR_H323_CONFLICT;
		}

		tIrr = MakeIRRwithOne(pEndpoint, (HS_USHORT)(irq->m_requestSeqNum.inheritance.value), tCall);
	}

	/* check message
	*/
	if( tIrr == HS_NULL ) return HS_ERR_H323_MAKE_MESSAGE;
	/* try sending
	*/
	if( tCall == HS_NULL )
		tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, HS_INVALID_CALL_HANDLE, HS_INVALID_RAS_SEQUENCE_NUMBER, tIrr);
	else
		tRet = AttemptRasMessage(pStack, pEndpoint, pWaits, tCall->handle, HS_INVALID_RAS_SEQUENCE_NUMBER, tIrr);

	delete_RasMsg(tIrr);
	HSFree(tIrr);
	return tRet;
}
/* RAS unknown message handler
*/
HS_RESULT UnknownRASHandler( RasMsg *pMsg, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pWaits )
{
	/*hstemp{*/
	return HS_OK;
	/*hstemp}*/
}





/******************************************************************************************/
/* Q931 Message Handlers 
*/
HS_RESULT SetupHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;
	ASNH225Setup_UUIE *tSetup = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State != e_Q931State_ReadySetup )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(RECV_SETUP), state(%d)", pCall->q931State );
#endif
		return HS_ERR_H323_CONFLICT;
	}
	if( pQ931->uuie == HS_NULL ) return HS_ERR_NULL_PARAM;

	/*if call is a h245 tunneling, add h245 TerminalCapabilitySet, MasterSlaveDetermination*/
	if( ASNH225H323_UU_PDU_IsIncludeOptionField(&(pQ931->uuie->m_h323_uu_pdu), e_ASNH225H323_UU_PDUOptionMap_h245Tunneling) )
	{
		if( pQ931->uuie->m_h323_uu_pdu.m_h245Tunneling.value )
		{
			pCall->isTunneling = HS_YES;
			NoLockList_AttachData( &(pCall->h245Charge), MakeTerminalCapabilitySet(pEndpoint, pCall) );
			pCall->h245State.msdValue = (HS_UINT)(rand()%HS_MASTER_SLAVE_MAX_VALUE);
			NoLockList_AttachData( &(pCall->h245Charge), MakeMasterSlaveDetermination(pEndpoint, pCall->h245State.msdValue) );
		}
	}

	tSetup = (ASNH225Setup_UUIE*)(pQ931->uuie->m_h323_uu_pdu.m_h323_message_body.alter);
	if( tSetup == HS_NULL ) return HS_ERR_H323_NOUNIT;

	/*Copy CRV, CID*/
	pCall->crv = pQ931->crv;
	memcpy( pCall->confId, tSetup->m_conferenceID.inheritance.value, 16 );
	memcpy( pCall->callId, tSetup->m_callIdentifier.m_guid.inheritance.value, 16 );

	/*get source, destination and destCallSignal address*/
	if( ASNH225Setup_UUIE_IsIncludeOptionField(tSetup, e_ASNH225Setup_UUIEOptionMap_sourceAddress) )
		AsnSequenceOf2AliasAddresses( &(pCall->sourceAliases), &(tSetup->m_sourceAddress) );
	if( ASNH225Setup_UUIE_IsIncludeOptionField(tSetup, e_ASNH225Setup_UUIEOptionMap_destinationAddress) )
		AsnSequenceOf2AliasAddresses( &(pCall->destAliases), &(tSetup->m_destinationAddress) );
	if( ASNH225Setup_UUIE_IsIncludeOptionField(tSetup, e_ASNH225Setup_UUIEOptionMap_destCallSignalAddress) )
		ASNH225TransportAddress_Copy( &(pCall->dcsaTsap), &(tSetup->m_destCallSignalAddress) );

	/*faststart check*/
	if( ASNH225Setup_UUIE_IsIncludeOptionField(tSetup, e_ASNH225Setup_UUIEOptionMap_fastStart) &&
		pCall->fastStartPoint != e_FastStartResponsePoint_Idle )
	{
		pCall->isFastStart = HS_YES;

		if( pEndpoint->CallbackCheckFastStart != HS_NULL )
			tRet = pEndpoint->CallbackCheckFastStart(&(tSetup->m_fastStart), &(pCall->capabilities), pCall);
		else
			tRet = CheckFastStart( &(tSetup->m_fastStart), pCall );

		if( tRet != HS_OK )
			pCall->fastStartPoint = e_FastStartResponsePoint_Idle;
	}
	else pCall->isFastStart = HS_NO;

	/*get admission*/
	if( IGatekeeper_IsRegisted(&(pEndpoint->gatekeeper)) == HS_YES && pCall->routeState != e_RouteState_Direct )
	{
		HS_USHORT tSN;
		RasMsg *arq = MakeARQ(pEndpoint, pCall, &tSN);
		/*gk routed call*/
		pCall->routeState = e_RouteState_Route;

		if( arq == HS_NULL )
			return CallClosing(
				pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Admission, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);

		if( AttemptRasMessage(pStack, pEndpoint, pWaits, pCall->handle, tSN, arq) != HS_OK )
		{
			delete_RasMsg(arq);
			HSFree(arq);

			return CallClosing(
				pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Admission, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		delete_RasMsg(arq);
		HSFree(arq);

		return HS_OK;
	}

	/*point to point call*/
	pCall->routeState = e_RouteState_Direct;
	return CallProcessing(pStack, pEndpoint, pCalls, pWaits, pCall);
}


HS_RESULT CallProceedHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;
	ASNH225CallProceeding_UUIE *tProceed = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State != e_Q931State_Setup )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(RECV_CALLPROCEED), state(%d)", pCall->q931State );
#endif
		return HS_ERR_H323_CONFLICT;
	}
	if( pQ931->uuie == HS_NULL ) return HS_ERR_NULL_PARAM;

	/*check crv*/
	if( pCall->crv != pQ931->crv )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on CallProceedHandler() -> diff CRV, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}

	tProceed = (ASNH225CallProceeding_UUIE*)(pQ931->uuie->m_h323_uu_pdu.m_h323_message_body.alter);
	if( tProceed == HS_NULL ) return HS_ERR_H323_NOUNIT;

	/*check call identifier*/
	if( ASNH225CallProceeding_UUIE_IsIncludeOptionField(tProceed, e_ASNH225CallProceeding_UUIEOptionMap_callIdentifier) )
	{
		if( memcmp(pCall->callId, tProceed->m_callIdentifier.m_guid.inheritance.value, 16) )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on CallProceedHandler() -> diff CallId, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
	}

	/*h245 address check*/
	if( pCall->isTunneling == HS_NO && pCall->h245State.session == HS_NO &&
		ASNH225CallProceeding_UUIE_IsIncludeOptionField(tProceed, e_ASNH225CallProceeding_UUIEOptionMap_h245Address)
	)
	{
		HS_UCHAR tDestIp[4];
		HS_USHORT tDestPort;
		TcpConnect *tTcpCon = NULL;

		if( ASNH225TransportAddress2Tsap(tDestIp, &tDestPort, &(tProceed->m_h245Address)) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on CallProceedHandler() -> MakeTSAP, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,pStack->h323DemonQ,pCall->handle);
		if( tTcpCon==NULL )
		{
			{
	#ifdef HS_DEBUG_H323
				HSPrint( "\n $ ! Error on CallProceedHandler() -> MakeQ, Call Closing.." );
	#endif
				return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
					e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
					e_Q931CauseType_NormalCallClearing
				);
			}
		}

		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);
	}

	/*faststart check*/
	if( ASNH225CallProceeding_UUIE_IsIncludeOptionField(tProceed, e_ASNH225CallProceeding_UUIEOptionMap_fastConnectRefused) )
	{
		pCall->isFastStart = HS_NO;
		if( pCall->remoteTCS != HS_NULL )
		{
			if( pEndpoint->CallbackCheckTerminalCapabilitySet != HS_NULL )
				tRet = pEndpoint->CallbackCheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);
			else
				tRet = CheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);

			if( tRet == HS_OK ) DischargeH245Message(pStack, pEndpoint, pWaits, pCall);
		}
	}
	else if( ASNH225CallProceeding_UUIE_IsIncludeOptionField(tProceed, e_ASNH225CallProceeding_UUIEOptionMap_fastStart) &&
		pCall->fastStartPoint != e_FastStartResponsePoint_Idle && pCall->fsActive == HS_NO )
	{
		pCall->isFastStart = HS_YES;

		if( pEndpoint->CallbackCheckFastStart != HS_NULL )
			tRet = pEndpoint->CallbackCheckFastStart(&(tProceed->m_fastStart), &(pCall->capabilities), pCall);
		else
			tRet = CheckFastStart( &(tProceed->m_fastStart), pCall );

		if( tRet != HS_OK )
		{
			pCall->fastStartPoint = e_FastStartResponsePoint_Idle;
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on CallProceedHandler() -> Invalid FastStart Response, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
		else
			OpenMediaWithFastStart(pStack, pEndpoint, pCall);
	}

	pCall->q931State = e_Q931State_CallProceed;
	return HS_OK;
}


HS_RESULT ConnectHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;
	ASNH225Connect_UUIE *tConnect = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State != e_Q931State_Setup &&
		pCall->q931State != e_Q931State_CallProceed &&
		pCall->q931State != e_Q931State_Alert )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(RECV_CONNECT), state(%d)", pCall->q931State );
#endif
		return HS_ERR_H323_CONFLICT;
	}
	if( pQ931->uuie == HS_NULL ) return HS_ERR_NULL_PARAM;

	/*check crv*/
	if( pCall->crv != pQ931->crv )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on ConnectHandler() -> diff CRV, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}

	tConnect = (ASNH225Connect_UUIE*)(pQ931->uuie->m_h323_uu_pdu.m_h323_message_body.alter);
	if( tConnect == HS_NULL ) return HS_ERR_H323_NOUNIT;

	/*check conference identifier*/
	if( memcmp(pCall->confId, tConnect->m_conferenceID.inheritance.value, 16) )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on ConnectHandler() -> diff ConfId, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}

	/*check call identifier*/
	if( ASNH225Connect_UUIE_IsIncludeOptionField(tConnect, e_ASNH225Connect_UUIEOptionMap_callIdentifier) )
	{
		if( memcmp(pCall->callId, tConnect->m_callIdentifier.m_guid.inheritance.value, 16) )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on ConnectHandler() -> diff CallId, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
	}

	/*h245 address check*/
	if( pCall->isTunneling == HS_NO && pCall->h245State.session == HS_NO &&
		ASNH225Connect_UUIE_IsIncludeOptionField(tConnect, e_ASNH225Connect_UUIEOptionMap_h245Address)
	)
	{
		HS_UCHAR tDestIp[4];
		HS_USHORT tDestPort;
		TcpConnect *tTcpCon = NULL;

		if( ASNH225TransportAddress2Tsap(tDestIp, &tDestPort, &(tConnect->m_h245Address)) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on ConnectHandler() -> MakeTSAP, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,pStack->h323DemonQ,pCall->handle);
		if( tTcpCon==NULL )
		{
			{
	#ifdef HS_DEBUG_H323
				HSPrint( "\n $ ! Error on ConnectHandler() -> MakeQ, Call Closing.." );
	#endif
				return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
					e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
					e_Q931CauseType_NormalCallClearing
				);
			}
		}

		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);
	}

	/*faststart check*/
	if( ASNH225Connect_UUIE_IsIncludeOptionField(tConnect, e_ASNH225Connect_UUIEOptionMap_fastConnectRefused) )
	{
		pCall->isFastStart = HS_NO;
		if( pCall->remoteTCS != HS_NULL )
		{
			if( pEndpoint->CallbackCheckTerminalCapabilitySet != HS_NULL )
				tRet = pEndpoint->CallbackCheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);
			else
				tRet = CheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);

			if( tRet == HS_OK ) DischargeH245Message(pStack, pEndpoint, pWaits, pCall);
		}
	}
	else if( ASNH225Connect_UUIE_IsIncludeOptionField(tConnect, e_ASNH225Connect_UUIEOptionMap_fastStart) &&
		pCall->fastStartPoint != e_FastStartResponsePoint_Idle && pCall->fsActive == HS_NO )
	{
		pCall->isFastStart = HS_YES;

		if( pEndpoint->CallbackCheckFastStart != HS_NULL )
			tRet = pEndpoint->CallbackCheckFastStart(&(tConnect->m_fastStart), &(pCall->capabilities), pCall);
		else
			tRet = CheckFastStart( &(tConnect->m_fastStart), pCall );

		if( tRet != HS_OK )
		{
			pCall->fastStartPoint = e_FastStartResponsePoint_Idle;
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on ConnectHandler() -> Invalid FastStart Response, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
		else
			OpenMediaWithFastStart(pStack, pEndpoint, pCall);
	}

	pCall->q931State = e_Q931State_Connect;
	/* start IRR message timer, if irrFrequency exist
		NOTE: this function must called in #e_Q931State_Connect# state
	*/
	StartIRRTimer(pStack, pCall);
	
	return HS_OK;
}


HS_RESULT AlertHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;
	ASNH225Alerting_UUIE *tAlert = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State != e_Q931State_Setup &&
		pCall->q931State != e_Q931State_CallProceed )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(RECV_ALERT), state(%d)", pCall->q931State );
#endif
		return HS_ERR_H323_CONFLICT;
	}
	if( pQ931->uuie == HS_NULL ) return HS_ERR_NULL_PARAM;

	/*check crv*/
	if( pCall->crv != pQ931->crv )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on AlertHandler() -> diff CRV, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}

	tAlert = (ASNH225Alerting_UUIE*)(pQ931->uuie->m_h323_uu_pdu.m_h323_message_body.alter);
	if( tAlert == HS_NULL ) return HS_ERR_H323_NOUNIT;

	/*check call identifier*/
	if( ASNH225Alerting_UUIE_IsIncludeOptionField(tAlert, e_ASNH225Alerting_UUIEOptionMap_callIdentifier) )
	{
		if( memcmp(pCall->callId, tAlert->m_callIdentifier.m_guid.inheritance.value, 16) )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on AlertHandler() -> diff CallId, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
	}

	/*h245 address check*/
	if( pCall->isTunneling == HS_NO && pCall->h245State.session == HS_NO &&
		ASNH225Alerting_UUIE_IsIncludeOptionField(tAlert, e_ASNH225Alerting_UUIEOptionMap_h245Address)
	)
	{
		HS_UCHAR tDestIp[4];
		HS_USHORT tDestPort;
		TcpConnect *tTcpCon = NULL;

		if( ASNH225TransportAddress2Tsap(tDestIp, &tDestPort, &(tAlert->m_h245Address)) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on AlertHandler() -> MakeTSAP, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,pStack->h323DemonQ,pCall->handle);
		if( tTcpCon == HS_NULL )
		{
			{
	#ifdef HS_DEBUG_H323
				HSPrint( "\n $ ! Error on AlertHandler() -> MakeQ, Call Closing.." );
	#endif
				return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
					e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
					e_Q931CauseType_NormalCallClearing
				);
			}
		}

		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);
	}

	/*faststart check*/
	if( ASNH225Alerting_UUIE_IsIncludeOptionField(tAlert, e_ASNH225Alerting_UUIEOptionMap_fastConnectRefused) )
	{
		pCall->isFastStart = HS_NO;
		if( pCall->remoteTCS != HS_NULL )
		{
			if( pEndpoint->CallbackCheckTerminalCapabilitySet != HS_NULL )
				tRet = pEndpoint->CallbackCheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);
			else
				tRet = CheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);

			if( tRet == HS_OK ) DischargeH245Message(pStack, pEndpoint, pWaits, pCall);
		}
	}
	else if( ASNH225Alerting_UUIE_IsIncludeOptionField(tAlert, e_ASNH225Alerting_UUIEOptionMap_fastStart) &&
		pCall->fastStartPoint != e_FastStartResponsePoint_Idle && pCall->fsActive == HS_NO )
	{
		pCall->isFastStart = HS_YES;

		if( pEndpoint->CallbackCheckFastStart != HS_NULL )
			tRet = pEndpoint->CallbackCheckFastStart(&(tAlert->m_fastStart), &(pCall->capabilities), pCall);
		else
			tRet = CheckFastStart( &(tAlert->m_fastStart), pCall );

		if( tRet != HS_OK )
		{
			pCall->fastStartPoint = e_FastStartResponsePoint_Idle;
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on AlertHandler() -> Invalid FastStart Response, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
		else
			OpenMediaWithFastStart(pStack, pEndpoint, pCall);
	}

	pCall->q931State = e_Q931State_Alert;
	return HS_OK;
}


HS_RESULT InformationHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}


HS_RESULT ReleaseCompleteHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	Q931CauseType tCause;
	Q931Element *tElement = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	switch(pCall->q931State)
	{
		case e_Q931State_Idle:
		case e_Q931State_ReadySetup:
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! CONFLICT(Q931_RELEASE)" );
#endif
			return HS_ERR_H323;

		case e_Q931State_Setup:
		case e_Q931State_CallProceed:
		case e_Q931State_Alert:
		case e_Q931State_ReadyConnect:
		case e_Q931State_Connect:
			tCause = e_Q931CauseType_NormalCallClearing;
			if( (tElement=Q931Message_GetElement(pQ931, e_Q931ElementType_Cause)) != HS_NULL )
				tCause = Q931Element_GetCauseValue(tElement);

			return CallClosing(
				pStack, pEndpoint, pCalls, pWaits, pCall, e_CallCloseReason_Normal,
				e_ASNH225DisengageReasonChoice_normalDrop, tCause
			);
	}
	
#ifdef HS_DEBUG_H323
	HSPrint( "\n $ ! OutOfState(Q931_RELEASE), Call Closing.." );
#endif
	CallClosing(
		pStack, pEndpoint, pCalls, pWaits, pCall, e_CallCloseReason_Undefined,
		e_ASNH225DisengageReasonChoice_normalDrop,
		e_Q931CauseType_NormalCallClearing
	);
	return HS_ERR_H323_NOUNIT;
}


HS_RESULT FacilityHandler(Q931Message *pQ931, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;
	ASNH225Facility_UUIE *tFacility = HS_NULL;

	if( pQ931==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pCall->q931State < e_Q931State_Setup )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! CONFLICT(RECV_FACILITY), state(%d)", pCall->q931State );
#endif
		return HS_ERR_H323_CONFLICT;
	}
	if( pQ931->uuie == HS_NULL ) return HS_ERR_NULL_PARAM;

	/*check crv*/
	if( pCall->crv != pQ931->crv )
	{
#ifdef HS_DEBUG_H323
		HSPrint( "\n $ ! Error on FacilityHandler() -> diff CRV, Call Closing.." );
#endif
		return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
			e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
			e_Q931CauseType_NormalCallClearing
		);
	}

	tFacility = (ASNH225Facility_UUIE*)(pQ931->uuie->m_h323_uu_pdu.m_h323_message_body.alter);
	if( tFacility == HS_NULL ) return HS_ERR_H323_NOUNIT;

	/*check conference identifier*/
	if( ASNH225Facility_UUIE_IsIncludeOptionField(tFacility, e_ASNH225Facility_UUIEOptionMap_conferenceID) )
	{
		if( memcmp(pCall->confId, tFacility->m_conferenceID.inheritance.value, 16) )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on FacilityHandler() -> diff ConfId, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
	}

	/*check call identifier*/
	if( ASNH225Facility_UUIE_IsIncludeOptionField(tFacility, e_ASNH225Facility_UUIEOptionMap_callIdentifier) )
	{
		if( memcmp(pCall->callId, tFacility->m_callIdentifier.m_guid.inheritance.value, 16) )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on FacilityHandler() -> diff CallId, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
	}

	/*h245 address check*/
	if( pCall->isTunneling == HS_NO && pCall->h245State.session == HS_NO &&
		ASNH225Facility_UUIE_IsIncludeOptionField(tFacility, e_ASNH225Facility_UUIEOptionMap_h245Address)
	)
	{
		HS_UCHAR tDestIp[4];
		HS_USHORT tDestPort;
		TcpConnect *tTcpCon = NULL;

		if( ASNH225TransportAddress2Tsap(tDestIp, &tDestPort, &(tFacility->m_h245Address)) != HS_OK )
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on FacilityHandler() -> MakeTSAP, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}

		tTcpCon = newm_TcpConnectEx(tDestIp,tDestPort,pStack->h323DemonQ,pCall->handle);
		if( tTcpCon==NULL )
		{
			{
	#ifdef HS_DEBUG_H323
				HSPrint( "\n $ ! Error on FacilityHandler() -> MakeQ, Call Closing.." );
	#endif
				return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
					e_CallCloseReason_H245Session, e_ASNH225DisengageReasonChoice_normalDrop,
					e_Q931CauseType_NormalCallClearing
				);
			}
		}

		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_TRY_CONNECT, 0, (LPARAM)tTcpCon);
	}

	/*faststart check*/
	if( ASNH225Facility_UUIE_IsIncludeOptionField(tFacility, e_ASNH225Facility_UUIEOptionMap_fastConnectRefused) )
	{
		pCall->isFastStart = HS_NO;
		if( pCall->remoteTCS != HS_NULL )
		{
			if( pEndpoint->CallbackCheckTerminalCapabilitySet != HS_NULL )
				tRet = pEndpoint->CallbackCheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);
			else
				tRet = CheckTerminalCapabilitySet(
					pStack, pEndpoint, pCall,
					((ASNH245RequestMessage*)(pCall->remoteTCS->alter))->alter
				);

			if( tRet == HS_OK ) DischargeH245Message(pStack, pEndpoint, pWaits, pCall);
		}
	}
	else if( pCall->q931State < e_Q931State_ReadyConnect &&
		ASNH225Facility_UUIE_IsIncludeOptionField(tFacility, e_ASNH225Facility_UUIEOptionMap_fastStart) &&
		pCall->fastStartPoint != e_FastStartResponsePoint_Idle && pCall->fsActive == HS_NO )
	{
		pCall->isFastStart = HS_YES;

		if( pEndpoint->CallbackCheckFastStart != HS_NULL )
			tRet = pEndpoint->CallbackCheckFastStart(&(tFacility->m_fastStart), &(pCall->capabilities), pCall);
		else
			tRet = CheckFastStart( &(tFacility->m_fastStart), pCall );

		if( tRet != HS_OK )
		{
			pCall->fastStartPoint = e_FastStartResponsePoint_Idle;
#ifdef HS_DEBUG_H323
			HSPrint( "\n $ ! Error on FacilityHandler() -> Invalid FastStart Response, Call Closing.." );
#endif
			return CallClosing(pStack, pEndpoint, pCalls, pWaits, pCall,
				e_CallCloseReason_Undefined, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
		}
		else
			OpenMediaWithFastStart(pStack, pEndpoint, pCall);
	}

	return HS_OK;
}


HS_RESULT ProgressHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}
HS_RESULT EmptyHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}
HS_RESULT StatusHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}
HS_RESULT StatusInquiryHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}
HS_RESULT SetupAcknowledgeHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}
HS_RESULT NotifyHandler(Q931Message *pQ931)
{
	/*hsfyet*/
	return HS_OK;
}



/******************************************************************************************/
/* H245 Message Handlers 
*/
static HS_RESULT H245MasterSlaveDeterminationHandler(
	ASNH245MasterSlaveDetermination *pMsd, StackInfo *pStack,
	IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall
)
{
	HS_RESULT tRet = HS_OK;
	ASNH245MasterSlaveDeterminationAck_decisionChoice tDecision;

	if( pMsd==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	if(pCall->h245State.recvMSD == HS_YES) return HS_OK;
	pCall->h245State.recvMSD = HS_YES;

	if( pMsd->m_terminalType.value > (HS_UINT)(pEndpoint->type) )
		tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_master;/*local is a slave*/
	else if( pMsd->m_terminalType.value < (HS_UINT)(pEndpoint->type) )
		tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_slave;/*local is a master*/
	else
	{
		HS_UINT moduloDiff = (pMsd->m_statusDeterminationNumber.value - pCall->h245State.msdValue)&0xffffff;
		if (moduloDiff == 0 || moduloDiff == 0x800000)
		{
			/*hstemp{*/
			if( pCall->isAnswerCall )
				tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_master;/*local is a slave*/
			else
				tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_slave;/*local is a master*/
			/*hstemp}*/
		}
		else if (moduloDiff < 0x800000)
			tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_slave;/*local is a master*/
		else
			tDecision = e_ASNH245MasterSlaveDeterminationAck_decisionChoice_master;/*local is a slave*/
	}

	tRet = NoLockList_AttachData( &(pCall->h245Charge), MakeMasterSlaveDeterminationAck(tDecision) );
	if( tRet != HS_OK ) return tRet;

	if( tDecision == e_ASNH245MasterSlaveDeterminationAck_decisionChoice_slave )
		pCall->h245State.mss = e_MasterSlaveState_Master;
	else
		pCall->h245State.mss = e_MasterSlaveState_Slave;

	return HS_OK;
}


static HS_RESULT H245TerminalCapabilitySetHandler(
	ASNH245TerminalCapabilitySet *pTcs, StackInfo *pStack,
	IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall
)
{
	HS_RESULT tRet = HS_OK;

	if( pTcs==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	pCall->h245State.recvTCS = HS_YES;

	tRet = NoLockList_AttachData( &(pCall->h245Charge), MakeTerminalCapabilitySetAck(pTcs) );
	if( tRet != HS_OK ) return tRet;

	if( (pCall->q931State<e_Q931State_Connect && pCall->isFastStart==HS_YES) ||
		(pCall->q931State>e_Q931State_ReadyConnect && pCall->fsActive==HS_YES)
	) return HS_OK_SAVE_IN_TCS;

	if( pEndpoint->CallbackCheckTerminalCapabilitySet != HS_NULL )
		pEndpoint->CallbackCheckTerminalCapabilitySet(pStack, pEndpoint, pCall, pTcs);
	else
		CheckTerminalCapabilitySet(pStack, pEndpoint, pCall, pTcs);

	return HS_OK_SAVE_IN_TCS;
}


static HS_RESULT H245OpenLogicalChannelHandler(
	ASNH245OpenLogicalChannel *pOlc, StackInfo *pStack,
	IEndpoint *pEndpoint, ICall *pCall
)
{
	HS_UINT i;
	BOOL found = HS_NO;
	HS_RESULT tRet = HS_OK;
	ChainUnit *rounder = HS_NULL;
	ICapability *tCapa = HS_NULL;
	ICapability *tFoundCapa = HS_NULL;
	NoLockList *tCapas = HS_NULL;

	if( pOlc==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	pCall->h245State.recvOLC = HS_YES;
	tCapas = &(pCall->capabilities);

	if( tCapas->size > 0 )
	{
		rounder = tCapas->units;
		for(i=0; i<tCapas->size; i++ )
		{
			if( rounder == HS_NULL ) break;
			if( rounder->data == HS_NULL ) break;

			tCapa = (ICapability*)(rounder->data);
			found =CompareCapabilityWithDataType(
					&(tCapa->capability), 
					&(pOlc->m_forwardLogicalChannelParameters.m_dataType)
			);

			if( found == HS_YES )
			{
				tFoundCapa = tCapa;
				break;
			}

			rounder = rounder->next;
		}
	}

	/*unknown capability : reject*/
	if( found == HS_NO || tFoundCapa == HS_NULL )
	{
		tRet = NoLockList_AttachData( &(pCall->h245Charge),
			MakeOpenLogicalChannelReject(pOlc, e_ASNH245OpenLogicalChannelReject_causeChoice_unknownDataType)
		);

		if( tRet != HS_OK ) return tRet;
		return HS_OK;
	}

	/*send ack*/
	tRet = NoLockList_AttachData( &(pCall->h245Charge),
		MakeOpenLogicalChannelAck(pOlc, pEndpoint, pCall, tFoundCapa)
	);
	if( tRet != HS_OK ) return tRet;

	return HS_OK_SAVE_IN_OLC;
}


HS_RESULT H245RequestHandler(ASNH245RequestMessage *pRequest, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pRequest == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch(pRequest->choice)
	{
		case e_ASNH245RequestMessageChoice_nonStandard:
		case e_ASNH245RequestMessageChoice_masterSlaveDetermination:
			return H245MasterSlaveDeterminationHandler(
				pRequest->alter, pStack, pEndpoint, pCalls, pWaits, pCall
			);
		case e_ASNH245RequestMessageChoice_terminalCapabilitySet:
			return H245TerminalCapabilitySetHandler(
				pRequest->alter, pStack, pEndpoint, pCalls, pWaits, pCall
			);
		case e_ASNH245RequestMessageChoice_openLogicalChannel:
			return H245OpenLogicalChannelHandler(
				pRequest->alter, pStack, pEndpoint, pCall
			);
		case e_ASNH245RequestMessageChoice_closeLogicalChannel:
		case e_ASNH245RequestMessageChoice_requestChannelClose:
		case e_ASNH245RequestMessageChoice_multiplexEntrySend:
		case e_ASNH245RequestMessageChoice_requestMultiplexEntry:
		case e_ASNH245RequestMessageChoice_requestMode:
		case e_ASNH245RequestMessageChoice_roundTripDelayRequest:
		case e_ASNH245RequestMessageChoice_maintenanceLoopRequest:
		case e_ASNH245RequestMessageChoice_communicationModeRequest:
		case e_ASNH245RequestMessageChoice_conferenceRequest:
		case e_ASNH245RequestMessageChoice_multilinkRequest:
		case e_ASNH245RequestMessageChoice_logicalChannelRateRequest:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}






static HS_RESULT H245MasterSlaveDeterminationAckHandler(
	ASNH245MasterSlaveDeterminationAck *pMsdAck, StackInfo *pStack,
	IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall
)
{
	if( pMsdAck==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pCall->h245State.recvMSDack == HS_YES ) return HS_OK;
	pCall->h245State.recvMSDack = HS_YES;

	if( pCall->h245State.mss != e_MasterSlaveState_Idle )
	{
		/*master-slave conflict*/
		if( (pCall->h245State.mss == e_MasterSlaveState_Slave &&
			 pMsdAck->m_decision.choice ==e_ASNH245MasterSlaveDeterminationAck_decisionChoice_master ) ||
			(pCall->h245State.mss == e_MasterSlaveState_Master &&
			 pMsdAck->m_decision.choice ==e_ASNH245MasterSlaveDeterminationAck_decisionChoice_slave )
		)
		{
#ifdef HS_DEBUG_H323
			HSPrint( "\n\n $ ! ERROR: MASTER-SLAVE CONFLICT, SKIP..\n" );
#endif
#if 0/*hstemp*/
			return CallClosing(pStack,pEndpoint,pCalls,pWaits,pCall,
				e_CallCloseReason_MasterSlaveConflict, e_ASNH225DisengageReasonChoice_normalDrop,
				e_Q931CauseType_NormalCallClearing
			);
#endif
		}
	}

	return HS_OK;
}


static HS_RESULT H245OpenLogicalChannelAckHandler(
	ASNH245OpenLogicalChannelAck *pOlcAck, IEndpoint *pEndpoint, ICall *pCall
)
{
	HS_RESULT tRet = HS_OK;
	ChainUnit *tChain = HS_NULL;
	OpenLogicalChannelSet *tOlcSet = HS_NULL;
	struct sockaddr_in tMediaIn;
	struct sockaddr_in tMediaControlIn;
	ASNH245OpenLogicalChannel *tOlc = HS_NULL;

	if( pOlcAck==HS_NULL || pEndpoint==HS_NULL || pCall==HS_NULL )
		return HS_ERR_NULL_PARAM;

	pCall->h245State.recvOLCack = HS_YES;

	/*finding OpenLogicalChannelSet*/
	tChain = NoLockList_FindOpenLogicalChannelSetByChannelNumber(
		&(pCall->sendedOlcSetList),
		pOlcAck->m_forwardLogicalChannelNumber.inheritance.value
	);

	if( tChain == HS_NULL ) return HS_ERR_H323_NOUNIT;
	if( (tOlcSet=(OpenLogicalChannelSet*)(tChain->data)) == HS_NULL ) return HS_ERR_H323_CONFLICT;
	if( tOlcSet->olcMsg == HS_NULL ) return HS_ERR_H323_CONFLICT;

	/*get media tsap address*/
	tMediaIn.sin_family = AF_MAX;
	tMediaControlIn.sin_family = AF_MAX;
	tRet = GetSockAddrInFromOpenLogicalChannelAck(pOlcAck, &tMediaIn, &tMediaControlIn);
	if( tRet != HS_OK ) return tRet;
	if( tMediaIn.sin_family == AF_MAX ) return HS_ERR_H323;

	/*media starting event*/
	tOlc = (ASNH245OpenLogicalChannel*)(((ASNH245RequestMessage*)(tOlcSet->olcMsg->alter))->alter);
	if( pEndpoint->CallbackOpenSendMedia != HS_NULL )
		pEndpoint->CallbackOpenSendMedia(
			pCall->handle,
			tMediaIn,
			tMediaControlIn,
			&(tOlc->m_forwardLogicalChannelParameters.m_dataType)
		);

	tOlcSet->isOpen = HS_YES;
	return HS_OK;
}


HS_RESULT H245ResponseHandler(ASNH245ResponseMessage *pResponse, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pResponse == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch(pResponse->choice)
	{
		case e_ASNH245ResponseMessageChoice_nonStandard:	break;
		case e_ASNH245ResponseMessageChoice_masterSlaveDeterminationAck:
			return H245MasterSlaveDeterminationAckHandler(
				pResponse->alter, pStack, pEndpoint, pCalls, pWaits, pCall
			);
		case e_ASNH245ResponseMessageChoice_masterSlaveDeterminationReject:	break;
		case e_ASNH245ResponseMessageChoice_terminalCapabilitySetAck:
			pCall->h245State.recvTCSack = HS_YES;
			break;
		case e_ASNH245ResponseMessageChoice_terminalCapabilitySetReject:	break;
		case e_ASNH245ResponseMessageChoice_openLogicalChannelAck:
			return H245OpenLogicalChannelAckHandler(
				pResponse->alter, pEndpoint, pCall
			);
		case e_ASNH245ResponseMessageChoice_openLogicalChannelReject:
		case e_ASNH245ResponseMessageChoice_closeLogicalChannelAck:
		case e_ASNH245ResponseMessageChoice_requestChannelCloseAck:
		case e_ASNH245ResponseMessageChoice_requestChannelCloseReject:
		case e_ASNH245ResponseMessageChoice_multiplexEntrySendAck:
		case e_ASNH245ResponseMessageChoice_multiplexEntrySendReject:
		case e_ASNH245ResponseMessageChoice_requestMultiplexEntryAck:
		case e_ASNH245ResponseMessageChoice_requestMultiplexEntryReject:
		case e_ASNH245ResponseMessageChoice_requestModeAck:
		case e_ASNH245ResponseMessageChoice_requestModeReject:
		case e_ASNH245ResponseMessageChoice_roundTripDelayResponse:
		case e_ASNH245ResponseMessageChoice_maintenanceLoopAck:
		case e_ASNH245ResponseMessageChoice_maintenanceLoopReject:
		case e_ASNH245ResponseMessageChoice_communicationModeResponse:
		case e_ASNH245ResponseMessageChoice_conferenceResponse:
		case e_ASNH245ResponseMessageChoice_multilinkResponse:
		case e_ASNH245ResponseMessageChoice_logicalChannelRateAcknowledge:
		case e_ASNH245ResponseMessageChoice_logicalChannelRateReject:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}






static HS_RESULT H245EndSessionCommandHandler(
	ASNH245EndSessionCommand *pEsc, StackInfo *pStack, IEndpoint *pEndpoint,
	NoLockList *pCalls, NoLockList *pWaits, ICall *pCall
)
{
	HS_RESULT tRet = HS_OK;

	if( pEsc==HS_NULL || pStack==HS_NULL || pEndpoint==HS_NULL ||
		pCalls==HS_NULL || pWaits==HS_NULL || pCall==HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pCall->isTunneling == HS_NO && pCall->h245State.session == HS_YES )
	{
		tRet = NoLockList_AttachData( &(pCall->h245Charge), MakeEndSessionCommandForDisconnect() );
		if( tRet != HS_OK ) return tRet;

		tRet = DischargeH245Message(pStack, pEndpoint, pWaits, pCall);
		if( tRet != HS_OK ) return tRet;

		_HSThreadSendMessage(pStack->enetDemonQ, HS_QMESSAGE_H245_CLOSE, 0, (LPARAM)(pCall->handle) );
		pCall->h245State.session = HS_NO;
	}

	return HS_OK;
}


HS_RESULT H245CommandHandler(ASNH245CommandMessage *pCommand, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pCommand == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pCommand->choice )
	{
		case e_ASNH245CommandMessageChoice_nonStandard:
		case e_ASNH245CommandMessageChoice_maintenanceLoopOffCommand:
		case e_ASNH245CommandMessageChoice_sendTerminalCapabilitySet:
		case e_ASNH245CommandMessageChoice_encryptionCommand:
		case e_ASNH245CommandMessageChoice_flowControlCommand:	break;
		case e_ASNH245CommandMessageChoice_endSessionCommand:
			return H245EndSessionCommandHandler(
				pCommand->alter, pStack, pEndpoint, pCalls, pWaits, pCall
			);
		case e_ASNH245CommandMessageChoice_miscellaneousCommand:
		case e_ASNH245CommandMessageChoice_communicationModeCommand:
		case e_ASNH245CommandMessageChoice_conferenceCommand:
		case e_ASNH245CommandMessageChoice_h223MultiplexReconfiguration:
		case e_ASNH245CommandMessageChoice_newATMVCCommand:
		case e_ASNH245CommandMessageChoice_mobileMultilinkReconfigurationCommand:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}


HS_RESULT H245IndicationHandler(ASNH245IndicationMessage *pIndication, StackInfo *pStack, IEndpoint *pEndpoint, NoLockList *pCalls, NoLockList *pWaits, ICall *pCall)
{
	HS_RESULT tRet = HS_OK;

	if( pIndication == HS_NULL ) return HS_ERR_NULL_PARAM;

	switch( pIndication->choice )
	{
		case e_ASNH245IndicationMessageChoice_nonStandard:
		case e_ASNH245IndicationMessageChoice_functionNotUnderstood:
		case e_ASNH245IndicationMessageChoice_masterSlaveDeterminationRelease:
		case e_ASNH245IndicationMessageChoice_terminalCapabilitySetRelease:
		case e_ASNH245IndicationMessageChoice_openLogicalChannelConfirm:
		case e_ASNH245IndicationMessageChoice_requestChannelCloseRelease:
		case e_ASNH245IndicationMessageChoice_multiplexEntrySendRelease:
		case e_ASNH245IndicationMessageChoice_requestMultiplexEntryRelease:
		case e_ASNH245IndicationMessageChoice_requestModeRelease:
		case e_ASNH245IndicationMessageChoice_miscellaneousIndication:
		case e_ASNH245IndicationMessageChoice_jitterIndication:
		case e_ASNH245IndicationMessageChoice_h223SkewIndication:
		case e_ASNH245IndicationMessageChoice_newATMVCIndication:
		case e_ASNH245IndicationMessageChoice_userInput:
		case e_ASNH245IndicationMessageChoice_h2250MaximumSkewIndication:
		case e_ASNH245IndicationMessageChoice_mcLocationIndication:
		case e_ASNH245IndicationMessageChoice_conferenceIndication:
		case e_ASNH245IndicationMessageChoice_vendorIdentification:
		case e_ASNH245IndicationMessageChoice_functionNotSupported:
		case e_ASNH245IndicationMessageChoice_multilinkIndication:
		case e_ASNH245IndicationMessageChoice_logicalChannelRateRelease:
		case e_ASNH245IndicationMessageChoice_flowControlIndication:
		case e_ASNH245IndicationMessageChoice_mobileMultilinkReconfigurationIndication:
			break;
		default:
			tRet = HS_ERR_H323_NOTFOUND;
			break;
	}

	return tRet;
}
