/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/

/*

	<H323Utility.c>		2005-03-13,11:26

*/

#include "H323Utility.h"



/**********************************************************/
/* Copy Functions
*/
HS_RESULT ASNH225AliasAddress_Copy(ASNH225AliasAddress *pTo, ASNH225AliasAddress *pFrom)
{
	AsnSingleString *ssTo = HS_NULL;
	AsnSingleString *ssFrom = HS_NULL;
	AsnDoubleString *dsTo = HS_NULL;
	AsnDoubleString *dsFrom = HS_NULL;
	ASNH225TransportAddress *tAddrTo = HS_NULL;
	ASNH225TransportAddress *tAddrFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH225AliasAddressChoice_dialedDigits:
		case e_ASNH225AliasAddressChoice_url_ID:
		case e_ASNH225AliasAddressChoice_email_ID:
			ssTo = ASNH225AliasAddress_SetChoice(pTo,pFrom->choice);
			ssFrom = (AsnSingleString*)(pFrom->alter);
			if( ssTo == HS_NULL ) return HS_ERR_H323;
			return AsnSingleString_SetValue(ssTo,ssFrom->value,ssFrom->length);

		case e_ASNH225AliasAddressChoice_h323_ID:
			dsFrom = (AsnDoubleString*)(pFrom->alter);
			dsTo = ASNH225AliasAddress_SetChoice(pTo,pFrom->choice);
			if( dsTo == HS_NULL ) return HS_ERR_H323;
			return AsnDoubleString_SetValue(dsTo,dsFrom->value,dsFrom->length);

		case e_ASNH225AliasAddressChoice_transportID:
			tAddrFrom = (ASNH225TransportAddress*)(pFrom->alter);
			tAddrTo = ASNH225AliasAddress_SetChoice(pTo,pFrom->choice);
			if( tAddrTo == HS_NULL ) return HS_ERR_H323;
			return ASNH225TransportAddress_Copy(tAddrTo,tAddrFrom);

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH225AliasAddressChoice_partyNumber
		case e_ASNH225AliasAddressChoice_mobileUIM
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT ASNH225AliasAddresses_Copy(NoLockList *pTo, NoLockList *pFrom)
{
	HS_UINT i;
	HS_RESULT tRet;
	ChainUnit *rounder= HS_NULL;
	ASNH225AliasAddress *tAliasTo = HS_NULL;
	ASNH225AliasAddress *tAliasFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->size == 0 ) return HS_OK;

	rounder = pFrom->units;
	for( i=0; i<pFrom->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323_CONFLICT;
		if( rounder->data == HS_NULL ) return HS_ERR_H323_CONFLICT;

		tAliasFrom = (ASNH225AliasAddress*)(rounder->data);

		tAliasTo = (ASNH225AliasAddress*)HSMalloc( sizeof(ASNH225AliasAddress) );
		if( tAliasTo == HS_NULL ) return HS_ERR_H323_MALLOC;

		new_ASNH225AliasAddress(tAliasTo);
		ASNH225AliasAddress_MakeMold(tAliasTo);

		if( (tRet=ASNH225AliasAddress_Copy(tAliasTo, tAliasFrom)) != HS_OK )
			return tRet;

		NoLockList_AttachData(pTo, tAliasTo);

		rounder = rounder->next;
	}

	return HS_OK;
}


HS_RESULT AliasAddresses2AsnSequenceOf(AsnSequenceOf *pTo, NoLockList *pList )
{
	HS_UINT i;
	HS_RESULT tRet;
	ASNH225AliasAddress *tAlias = HS_NULL;
	ChainUnit *rounder = HS_NULL;

	if( pTo == HS_NULL || pList == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pList->size == 0 )
		return AsnSequenceOf_SetSize(pTo,0);

	HS_TRY( AsnSequenceOf_SetSize(pTo,pList->size) )
	rounder = pList->units;
	for( i=0; i<pList->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323;
		if( rounder->data == HS_NULL ) return HS_ERR_H323;

		HS_TRY(
			ASNH225AliasAddress_Copy( AsnSequenceOf_GetUnit(pTo,i), rounder->data )
		)

		rounder = rounder->next;
	}

	return HS_OK;
}


HS_RESULT AsnSequenceOf2AliasAddresses(NoLockList *pList, AsnSequenceOf *pFrom)
{
	HS_UINT i;
	ASNH225AliasAddress *tAlias = HS_NULL;

	if( pList==HS_NULL || pFrom==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->size==0 ) return HS_ERR_NULL_PARAM;

	for( i=0; i<pFrom->size; i++ )
	{
		if( (tAlias=(ASNH225AliasAddress*)HSMalloc(sizeof(ASNH225AliasAddress))) == HS_NULL ) continue;
		new_ASNH225AliasAddress(tAlias);
		ASNH225AliasAddress_MakeMold(tAlias);
		if( ASNH225AliasAddress_Copy( tAlias, AsnSequenceOf_GetUnit(pFrom, i) ) != HS_OK )
		{
			delete_ASNH225AliasAddress(tAlias);
			HSFree(tAlias);
			continue;
		}

		NoLockList_AttachData(pList, tAlias);
	}

	return HS_OK;
}


static HS_RESULT ASNH225TransportAddress_CopyEx(ASNH225TransportAddress *pTo, ASNH225TransportAddress *pFrom, HS_USHORT pPort, BOOL pChangePort)
{
	ASNH225TransportAddress_ipAddress *ipAddrTo = HS_NULL;
	ASNH225TransportAddress_ipAddress *ipAddrFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH225TransportAddressChoice_ipAddress:
			ipAddrFrom = (ASNH225TransportAddress_ipAddress*)(pFrom->alter);
			ipAddrTo = (ASNH225TransportAddress_ipAddress*)ASNH225TransportAddress_SetChoice(pTo,pFrom->choice);
			if(ipAddrTo==HS_NULL) return HS_ERR_H323;

			AsnOctetString_SetValue(&(ipAddrTo->m_ip),ipAddrFrom->m_ip.length,ipAddrFrom->m_ip.value);

			if( pChangePort == HS_YES )
				AsnInteger_SetValue(&(ipAddrTo->m_port),pPort);
			else
				AsnInteger_SetValue(&(ipAddrTo->m_port),ipAddrFrom->m_port.value);

			return HS_OK;
#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH225TransportAddressChoice_ipSourceRoute:
		case e_ASNH225TransportAddressChoice_ipxAddress:
		case e_ASNH225TransportAddressChoice_ip6Address:
		case e_ASNH225TransportAddressChoice_netBios:
		case e_ASNH225TransportAddressChoice_nsap:
		case e_ASNH225TransportAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT ASNH225TransportAddress_Copy(ASNH225TransportAddress *pTo, ASNH225TransportAddress *pFrom)
{
	return ASNH225TransportAddress_CopyEx(pTo, pFrom, 0, HS_NO);
}


HS_RESULT ASNH225TransportAddress_CopyWithPort(ASNH225TransportAddress *pTo, ASNH225TransportAddress *pFrom, HS_USHORT pPort)
{
	return ASNH225TransportAddress_CopyEx(pTo, pFrom, pPort, HS_YES);
}


static HS_RESULT ASNH245TransportAddress_CopyEx(ASNH245TransportAddress *pTo, ASNH245TransportAddress *pFrom, HS_USHORT pPort, BOOL pChangePort)
{
	ASNH245UnicastAddress *uAddrTo = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrTo = HS_NULL;
	ASNH245UnicastAddress *uAddrFrom = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH245TransportAddressChoice_unicastAddress:
		{
			uAddrFrom = (ASNH245UnicastAddress*)(pFrom->alter);
			uAddrTo = (ASNH245UnicastAddress*)ASNH245TransportAddress_SetChoice(pTo,e_ASNH245TransportAddressChoice_unicastAddress);
			if( uAddrTo == HS_NULL ) return HS_ERR_H323;

			switch( uAddrFrom->choice )
			{
				case e_ASNH245UnicastAddressChoice_iPAddress:
					ipAddrFrom = (ASNH245UnicastAddress_iPAddress*)(uAddrFrom->alter);
					if( ipAddrFrom == HS_NULL ) return HS_ERR_H323;
					ipAddrTo = (ASNH245UnicastAddress_iPAddress*)ASNH245UnicastAddress_SetChoice(uAddrTo,e_ASNH245UnicastAddressChoice_iPAddress);
					if( ipAddrTo == HS_NULL ) return HS_ERR_H323;

					AsnOctetString_SetValue(&(ipAddrTo->m_network),ipAddrFrom->m_network.length,ipAddrFrom->m_network.value);

					if( pChangePort == HS_YES )
						AsnInteger_SetValue(&(ipAddrTo->m_tsapIdentifier),pPort);
					else
						AsnInteger_SetValue(&(ipAddrTo->m_tsapIdentifier),ipAddrFrom->m_tsapIdentifier.value);

					return HS_OK;

#if 0/*NOT_SUPPORT_YET*/
				case e_ASNH245UnicastAddressChoice_iPXAddress:
				case e_ASNH245UnicastAddressChoice_iP6Address:
				case e_ASNH245UnicastAddressChoice_netBios:
				case e_ASNH245UnicastAddressChoice_iPSourceRouteAddress:
				case e_ASNH245UnicastAddressChoice_nsap:
				case e_ASNH245UnicastAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
			}
			return HS_ERR_H323_NOTFOUND;
		}

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH245TransportAddressChoice_multicastAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT ASNH245TransportAddress_Copy(ASNH245TransportAddress *pTo, ASNH245TransportAddress *pFrom)
{
	return ASNH245TransportAddress_CopyEx(pTo, pFrom, 0, HS_NO);
}


HS_RESULT ASNH245TransportAddress_CopyWithPort(ASNH245TransportAddress *pTo, ASNH245TransportAddress *pFrom, HS_USHORT pPort)
{
	return ASNH245TransportAddress_CopyEx(pTo, pFrom, pPort, HS_YES);
}


HS_RESULT TransportAddresses2AsnSequenceOf(AsnSequenceOf *pTo, NoLockList *pList )
{
	HS_UINT i;
	HS_RESULT tRet;
	ASNH225TransportAddress *tAddr = HS_NULL;
	ChainUnit *rounder = HS_NULL;

	if( pTo == HS_NULL || pList == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pList->size == 0 )
		return AsnSequenceOf_SetSize(pTo,0);

	HS_TRY( AsnSequenceOf_SetSize(pTo,pList->size) )
	rounder = pList->units;
	for( i=0; i<pList->size; i++ )
	{
		if( rounder == HS_NULL ) return HS_ERR_H323;
		if( rounder->data == HS_NULL ) return HS_ERR_H323;

		HS_TRY(
			ASNH225TransportAddress_Copy( AsnSequenceOf_GetUnit(pTo,i), rounder->data )
		)

		rounder = rounder->next;
	}

	return HS_OK;
}


HS_RESULT ASNH225TransportAddress2ASNH245TransportAddress(ASNH245TransportAddress *pTo, ASNH225TransportAddress *pFrom)
{
	ASNH245UnicastAddress *uAddrTo = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrTo = HS_NULL;
	ASNH225TransportAddress_ipAddress *ipAddrFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH225TransportAddressChoice_ipAddress:
			ipAddrFrom = (ASNH225TransportAddress_ipAddress*)(pFrom->alter);
			uAddrTo = (ASNH245UnicastAddress*)ASNH245TransportAddress_SetChoice(pTo,e_ASNH245TransportAddressChoice_unicastAddress);
			if(uAddrTo==HS_NULL) return HS_ERR_H323;
			ipAddrTo = (ASNH245UnicastAddress_iPAddress*)ASNH245UnicastAddress_SetChoice(uAddrTo,e_ASNH245UnicastAddressChoice_iPAddress);
			if(ipAddrTo==HS_NULL) return HS_ERR_H323;

			AsnOctetString_SetValue(&(ipAddrTo->m_network),ipAddrFrom->m_ip.length,ipAddrFrom->m_ip.value);
			AsnInteger_SetValue(&(ipAddrTo->m_tsapIdentifier),ipAddrFrom->m_port.value);
			return HS_OK;

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH225TransportAddressChoice_ipSourceRoute:
		case e_ASNH225TransportAddressChoice_ipxAddress:
		case e_ASNH225TransportAddressChoice_ip6Address:
		case e_ASNH225TransportAddressChoice_netBios:
		case e_ASNH225TransportAddressChoice_nsap:
		case e_ASNH225TransportAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT ASNH225TransportAddress2ASNH245TransportAddress_WithPort(ASNH245TransportAddress *pTo, ASNH225TransportAddress *pFrom, HS_USHORT pPort)
{
	ASNH245UnicastAddress *uAddrTo = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrTo = HS_NULL;
	ASNH225TransportAddress_ipAddress *ipAddrFrom = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH225TransportAddressChoice_ipAddress:
			ipAddrFrom = (ASNH225TransportAddress_ipAddress*)(pFrom->alter);
			uAddrTo = (ASNH245UnicastAddress*)ASNH245TransportAddress_SetChoice(pTo,e_ASNH245TransportAddressChoice_unicastAddress);
			if(uAddrTo==HS_NULL) return HS_ERR_H323;
			ipAddrTo = (ASNH245UnicastAddress_iPAddress*)ASNH245UnicastAddress_SetChoice(uAddrTo,e_ASNH245UnicastAddressChoice_iPAddress);
			if(ipAddrTo==HS_NULL) return HS_ERR_H323;

			AsnOctetString_SetValue(&(ipAddrTo->m_network),ipAddrFrom->m_ip.length,ipAddrFrom->m_ip.value);
			AsnInteger_SetValue(&(ipAddrTo->m_tsapIdentifier),pPort);
			return HS_OK;

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH225TransportAddressChoice_ipSourceRoute:
		case e_ASNH225TransportAddressChoice_ipxAddress:
		case e_ASNH225TransportAddressChoice_ip6Address:
		case e_ASNH225TransportAddressChoice_netBios:
		case e_ASNH225TransportAddressChoice_nsap:
		case e_ASNH225TransportAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


HS_RESULT ASNH245TransportAddress2ASNH225TransportAddress(ASNH225TransportAddress *pTo, ASNH245TransportAddress *pFrom)
{
	ASNH245UnicastAddress *uAddrFrom = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrFrom = HS_NULL;
	ASNH225TransportAddress_ipAddress *ipAddrTo = HS_NULL;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH245TransportAddressChoice_unicastAddress:
		{
			uAddrFrom = (ASNH245UnicastAddress*)(pFrom->alter);
			switch( uAddrFrom->choice )
			{
				case e_ASNH245UnicastAddressChoice_iPAddress:
					ipAddrFrom = (ASNH245UnicastAddress_iPAddress*)(uAddrFrom->alter);
					if( ipAddrFrom == HS_NULL ) return HS_ERR_H323;
					ipAddrTo = (ASNH225TransportAddress_ipAddress*)ASNH225TransportAddress_SetChoice(pTo,e_ASNH225TransportAddressChoice_ipAddress);
					if( ipAddrTo == HS_NULL ) return HS_ERR_H323;

					AsnOctetString_SetValue(&(ipAddrTo->m_ip),ipAddrFrom->m_network.length,ipAddrFrom->m_network.value);
					AsnInteger_SetValue(&(ipAddrTo->m_port),ipAddrFrom->m_tsapIdentifier.value);
					return HS_OK;

#if 0/*NOT_SUPPORT_YET*/
				case e_ASNH245UnicastAddressChoice_iPXAddress:
				case e_ASNH245UnicastAddressChoice_iP6Address:
				case e_ASNH245UnicastAddressChoice_netBios:
				case e_ASNH245UnicastAddressChoice_iPSourceRouteAddress:
				case e_ASNH245UnicastAddressChoice_nsap:
				case e_ASNH245UnicastAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
			}
			return HS_ERR_H323_NOTFOUND;
		}

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH245TransportAddressChoice_multicastAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


/* Tsap2ASNH225TransportAddress */
HS_RESULT Tsap2ASNH225TransportAddress(ASNH225TransportAddress *pTo, HS_UCHAR *pIp, HS_USHORT pPort)
{
	ASNH225TransportAddress_ipAddress *ipAddr = HS_NULL;

	if(pTo == HS_NULL || pIp == HS_NULL ) return HS_ERR_NULL_PARAM;

	ipAddr = (ASNH225TransportAddress_ipAddress*)
		ASNH225TransportAddress_SetChoice(pTo,e_ASNH225TransportAddressChoice_ipAddress);

	if(ipAddr==HS_NULL) return HS_ERR_H323;

	AsnOctetString_SetValue( &(ipAddr->m_ip), 4, pIp);
	AsnInteger_SetValue( &(ipAddr->m_port), pPort );

	return HS_OK;

}


/* ASNH225TransportAddress2Tsap */
HS_RESULT ASNH225TransportAddress2Tsap(HS_UCHAR *pIp, HS_USHORT *pPort, ASNH225TransportAddress *pFrom)
{
	ASNH225TransportAddress_ipAddress *ipAddr = HS_NULL;

	if( pIp == HS_NULL || pPort == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;

	if( pFrom->choice != e_ASNH225TransportAddressChoice_ipAddress ) return HS_ERR_H323;
	ipAddr = (ASNH225TransportAddress_ipAddress*)(pFrom->alter);
	if( ipAddr == HS_NULL ) return HS_ERR_H323;

	memcpy(pIp, ipAddr->m_ip.value, 4);
	*pPort = ipAddr->m_port.value;

	return HS_OK;
}


/* ASNH245TransportAddress2Tsap */
HS_RESULT ASNH245TransportAddress2SockAddrIn(struct sockaddr_in *pTo, ASNH245TransportAddress *pFrom)
{
	HS_UINT laddr;
	ASNH245UnicastAddress *uAddrFrom = HS_NULL;
	ASNH245UnicastAddress_iPAddress *ipAddrFrom = HS_NULL;

	if( pFrom==HS_NULL || pTo==HS_NULL ) return HS_ERR_NULL_PARAM;
	if( pFrom->alter == HS_NULL ) return HS_ERR_H323_INVALID_VALUE;

	switch( pFrom->choice )
	{
		case e_ASNH245TransportAddressChoice_unicastAddress:
		{
			uAddrFrom = (ASNH245UnicastAddress*)(pFrom->alter);
			switch( uAddrFrom->choice )
			{
				case e_ASNH245UnicastAddressChoice_iPAddress:
					ipAddrFrom = (ASNH245UnicastAddress_iPAddress*)(uAddrFrom->alter);
					if( ipAddrFrom == HS_NULL ) return HS_ERR_H323;
					if( ipAddrFrom->m_network.length != 4 ) return HS_ERR_H323;

					pTo->sin_family = AF_INET;
					Uchar2Uint(&laddr, ipAddrFrom->m_network.value);
					pTo->sin_addr.s_addr = htonl( laddr );
					pTo->sin_port = htons( (HS_USHORT)(ipAddrFrom->m_tsapIdentifier.value) );
					return HS_OK;

#if 0/*NOT_SUPPORT_YET*/
				case e_ASNH245UnicastAddressChoice_iPXAddress:
				case e_ASNH245UnicastAddressChoice_iP6Address:
				case e_ASNH245UnicastAddressChoice_netBios:
				case e_ASNH245UnicastAddressChoice_iPSourceRouteAddress:
				case e_ASNH245UnicastAddressChoice_nsap:
				case e_ASNH245UnicastAddressChoice_nonStandardAddress:
#endif/*NOT_SUPPORT_YET*/
			}
			return HS_ERR_H323_NOTFOUND;
		}

#if 0/*NOT_SUPPORT_YET*/
		case e_ASNH245TransportAddressChoice_multicastAddress:
#endif/*NOT_SUPPORT_YET*/
	}

	return HS_ERR_H323_NOTFOUND;
}


/* String2SockAddrEx */
HS_RESULT String2SockAddrEx( struct sockaddr_in *pTo, char *pFrom )
{
	int i, len;
	unsigned long laddr;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	/*multicast address*/
	if( !strcmp(pFrom,"@") )
	{
		pTo->sin_family = AF_INET;
		pTo->sin_addr.s_addr = inet_addr("224.0.1.41");
		pTo->sin_port = htons(1718);
		return HS_OK;
	}

	len = strlen(pFrom);
	for( i=0; i<len; i++ )
	{
		if( pFrom[i] == '@' ) break;
	}

	if( i==len ) return HS_ERR_H323;
	pFrom[i] = 0;

	if( Name2IpAddr(&laddr,pFrom+i+1) != HS_OK )
	{
		pFrom[i] = '@';
		return HS_ERR_H323;
	}

	pTo->sin_family = AF_INET;
    pTo->sin_addr.s_addr = laddr;
	pTo->sin_port = htons( (HS_USHORT)atoi(pFrom) );

	pFrom[i] = '@';
	return HS_OK;
}


HS_RESULT ASNH225GatekeeperIdentifier_Copy( ASNH225GatekeeperIdentifier *pTo, ASNH225GatekeeperIdentifier *pFrom )
{
	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	return AsnDoubleString_SetValue(&(pTo->inheritance),pFrom->inheritance.value,pFrom->inheritance.length);
}


HS_RESULT ASNH225EndpointIdentifier_Copy( ASNH225EndpointIdentifier *pTo, ASNH225EndpointIdentifier *pFrom )
{
	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;
	return AsnDoubleString_SetValue(&(pTo->inheritance),pFrom->inheritance.value,pFrom->inheritance.length);
}


HS_RESULT ASNH225H323_UU_PDU_Copy( ASNH225H323_UU_PDU *pTo, ASNH225H323_UU_PDU *pFrom )
{
	HS_RESULT tRet;
	AsnStream tStrm;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;

	new_AsnStream(&tStrm, 256, TRUE);
	if( (tRet=ASNH225H323_UU_PDU_Encode(pFrom,&tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}

	AsnStream_Reset(&tStrm);
	tRet = ASNH225H323_UU_PDU_Decode(pTo, &tStrm);
	delete_AsnStream(&tStrm);
	return tRet;
}



/**********************************************************/
/* media copy functions
*/
HS_RESULT ASNH245Capability_Copy( ASNH245Capability *pTo, ASNH245Capability *pFrom )
{
	HS_RESULT tRet;
	AsnStream tStrm;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;

	new_AsnStream(&tStrm, 256, TRUE);
	if( (tRet=ASNH245Capability_Encode(pFrom,&tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}

	AsnStream_Reset(&tStrm);
	tRet = ASNH245Capability_Decode(pTo, &tStrm);
	delete_AsnStream(&tStrm);
	return tRet;
}

HS_RESULT ASNH245VideoCapability_Copy( ASNH245VideoCapability *pTo, ASNH245VideoCapability *pFrom )
{
	HS_RESULT tRet;
	AsnStream tStrm;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;

	new_AsnStream(&tStrm, 256, TRUE);
	if( (tRet=ASNH245VideoCapability_Encode(pFrom,&tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}

	AsnStream_Reset(&tStrm);
	tRet = ASNH245VideoCapability_Decode(pTo, &tStrm);
	delete_AsnStream(&tStrm);
	return tRet;
}
HS_RESULT ASNH245AudioCapability_Copy( ASNH245AudioCapability *pTo, ASNH245AudioCapability *pFrom )
{
	HS_RESULT tRet;
	AsnStream tStrm;

	if( pTo == HS_NULL || pFrom == HS_NULL ) return HS_ERR_NULL_PARAM;

	new_AsnStream(&tStrm, 256, TRUE);
	if( (tRet=ASNH245AudioCapability_Encode(pFrom,&tStrm)) != HS_OK )
	{
		delete_AsnStream(&tStrm);
		return tRet;
	}

	AsnStream_Reset(&tStrm);
	tRet = ASNH245AudioCapability_Decode(pTo, &tStrm);
	delete_AsnStream(&tStrm);
	return HS_OK;
}




/**********************************************************/
/* Check Function
*/
BOOL IsE164( char *pE164 )
{
	int i, len;
	if( pE164 == HS_NULL ) return HS_NO;

	len = strlen(pE164);
	if( len > HS_MAX_DIGIT-1 ) return HS_NO;
	for( i=0; i<len; i++ )
	{
		if( pE164[i] > 0x39 ) return HS_NO;
		if( pE164[i] < 0x30 && pE164[i] != '#' && pE164[i] != '*' && pE164[i] != ',' ) return HS_NO;
	}

	return HS_YES;
}


BOOL IsIp(char *pIp)
{
	char tIp[16];
	char *tPointer = HS_NULL;
	int i, len, dot=0;
	int tValue;

	if( pIp == HS_NULL ) return HS_NO;
	if( (len=strlen(pIp)) > 15 ) return HS_NO;

	tPointer = tIp;
	strcpy(tIp, pIp);

	for( i=0; i<len; i++ )
	{
		if(tIp[i] == '.')
		{
			dot++;
			if(dot>3) return HS_NO;
			tIp[i] = '\0';

			tValue = atoi(tPointer);
			if( tValue < 0 || tValue > 255 ) return HS_NO;
			tPointer = tIp+1;
		}
	}

	if(dot!=3) return HS_NO;

	tValue = atoi(tPointer);
	if( tValue < 0 || tValue > 255 ) return HS_NO;
	return HS_YES;
}


BOOL IsASNH245TransportAddress( ASNH245TransportAddress *pAddr )
{
	if( pAddr == HS_NULL ) return HS_NO;
	if( pAddr->alter == HS_NULL ) return HS_NO;
	return HS_YES;
}





/**********************************************************************/
/* Get Functions
*/
HS_RESULT GetTsapPortFromOpenLogicalChannelAck(ASNH245OpenLogicalChannelAck *pOlcAck, HS_UINT *pMediaPort, HS_UINT *pMediaControlPort)
{
	HS_RESULT tRet = HS_OK;
	struct sockaddr_in tMediaIn;
	struct sockaddr_in tMediaControlIn;

	if( pOlcAck==HS_NULL || pMediaPort==HS_NULL || pMediaControlPort==HS_NULL )
		return HS_ERR_NULL_PARAM;

	*pMediaPort = HS_INVALID_PORT;
	*pMediaControlPort = HS_INVALID_PORT;
	tMediaIn.sin_family = AF_MAX;
	tMediaControlIn.sin_family = AF_MAX;

	tRet = GetSockAddrInFromOpenLogicalChannelAck(pOlcAck, &tMediaIn, &tMediaControlIn);
	if( tRet != HS_OK ) return tRet;

	if( tMediaIn.sin_family != AF_MAX )
		*pMediaPort = (HS_UINT)( htons(tMediaIn.sin_port) );
	if( tMediaControlIn.sin_family != AF_MAX )
		*pMediaControlPort = (HS_UINT)( htons(tMediaControlIn.sin_port) );

	return HS_OK;
}


HS_RESULT GetSockAddrInFromOpenLogicalChannelAck(
	ASNH245OpenLogicalChannelAck *pOlcAck,
	struct sockaddr_in *pMediaIn,
	struct sockaddr_in *pMediaControlIn
)
{
	HS_RESULT tRet = HS_OK;
	ASNH245H2250LogicalChannelAckParameters *h2250Ack = HS_NULL;

	if( pOlcAck==HS_NULL || pMediaIn==HS_NULL || pMediaControlIn==HS_NULL )
		return HS_ERR_NULL_PARAM;

	pMediaIn->sin_family = AF_MAX;
	pMediaControlIn->sin_family = AF_MAX;

	if( !ASNH245OpenLogicalChannelAck_IsIncludeOptionField(pOlcAck, e_ASNH245OpenLogicalChannelAckOptionMap_forwardMultiplexAckParameters) )
		return HS_ERR_H323;
	if( pOlcAck->m_forwardMultiplexAckParameters.choice != e_ASNH245OpenLogicalChannelAck_forwardMultiplexAckParametersChoice_h2250LogicalChannelAckParameters )
		return HS_ERR_H323;
	if( (h2250Ack=(ASNH245H2250LogicalChannelAckParameters*)(pOlcAck->m_forwardMultiplexAckParameters.alter))
		== HS_NULL ) return HS_ERR_H323;

	if( ASNH245H2250LogicalChannelAckParameters_IsIncludeOptionField(h2250Ack, e_ASNH245H2250LogicalChannelAckParametersOptionMap_mediaChannel) )
	{
		if( ASNH245TransportAddress2SockAddrIn(pMediaIn, &(h2250Ack->m_mediaChannel)) != HS_OK )
			pMediaIn->sin_family = AF_MAX;
	}

	if( ASNH245H2250LogicalChannelAckParameters_IsIncludeOptionField(h2250Ack, e_ASNH245H2250LogicalChannelAckParametersOptionMap_mediaControlChannel) )
	{
		if( ASNH245TransportAddress2SockAddrIn(pMediaControlIn, &(h2250Ack->m_mediaControlChannel)) != HS_OK )
			pMediaControlIn->sin_family = AF_MAX;
	}

	return HS_OK;
}
