/********************************************************************************

  Copyright (c) 2006, Hyoung-Sun Kim.
  All Rights Reserved.

  You can contact us with
  web site <http://www.voiper.co.kr>
  e-mail <voiper@voiper.co.kr>

  This software is distributed under the terms of the BSD license

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the <organization> nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*********************************************************************************/


/*

	<asnPrime.cxx>	2004-07-03,14:20

*/

#include "asnPrime.h"

extern unsigned char gRMASK[9];
extern unsigned char gLMASK[9];
extern unsigned char gBMASK[8];



/*********************************************************************************************************/
/* Functional Functions
*/
HS_UINT Log2Size( HS_UINT pLen, BOOL zeroPossible )
{
	/* distance is impossible to be zero */
	/* data overflow : 0xffffffff - 0 + 1 = 0 */
	if( !zeroPossible && pLen == 0 ) return 32;

	if( pLen < 0x00000003 ) return 1;
	if( pLen < 0x00000006 ) return 2;
	if( pLen < 0x00000009 ) return 3;
	if( pLen < 0x00000011 ) return 4;
	if( pLen < 0x00000021 ) return 5;
	if( pLen < 0x00000041 ) return 6;
	if( pLen < 0x00000081 ) return 7;
	if( pLen < 0x00000101 ) return 8;
	if( pLen < 0x00000201 ) return 9;
	if( pLen < 0x00000401 ) return 10;
	if( pLen < 0x00000801 ) return 11;
	if( pLen < 0x00001001 ) return 12;
	if( pLen < 0x00002001 ) return 13;
	if( pLen < 0x00004001 ) return 14;
	if( pLen < 0x00008001 ) return 15;
	if( pLen < 0x00010001 ) return 16;
	if( pLen < 0x00020001 ) return 17;
	if( pLen < 0x00040001 ) return 18;
	if( pLen < 0x00080001 ) return 19;
	if( pLen < 0x00100001 ) return 20;
	if( pLen < 0x00200001 ) return 21;
	if( pLen < 0x00300001 ) return 22;
	if( pLen < 0x00800001 ) return 23;
	if( pLen < 0x01000001 ) return 24;
	if( pLen < 0x02000001 ) return 25;
	if( pLen < 0x04000001 ) return 26;
	if( pLen < 0x08000001 ) return 27;
	if( pLen < 0x10000001 ) return 28;
	if( pLen < 0x20000001 ) return 29;
	if( pLen < 0x40000001 ) return 30;
	if( pLen < 0x80000001 ) return 31;
	return 32;
}



HS_UINT Log256Size( HS_UINT pLen, BOOL zeroPossible )
{
	/* distance is impossible to be zero */
	/* data overflow : 0xffffffff - 0 + 1 = 0 */
	if( !zeroPossible && pLen == 0 ) return 32;

	if( pLen < 0x00000101 ) return 1;
	if( pLen < 0x00010001 ) return 2;
	if( pLen < 0x01000001 ) return 3;
	return 4;
}



HS_UINT Log2SizeInterval( HS_UINT pMins, HS_UINT pMaxs )
{
	HS_UINT tSize;

	if( pMins > pMaxs )	tSize = pMins-pMaxs;
	else				tSize = pMaxs-pMins;

	return Log2Size( tSize, FALSE );
}



HS_UINT Log256SizeInterval( HS_UINT pMins, HS_UINT pMaxs )
{
	HS_UINT tSize;

	if( pMins > pMaxs )	tSize = pMins-pMaxs;
	else				tSize = pMaxs-pMins;

	return Log256Size( tSize, FALSE );
}



HS_RESULT SetLengthOctet( unsigned char *pLength, AsnStream *pStrm )
{
	if( pLength == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SetLengthOctet(%x,%x) -> null parameter", pLength, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pStrm->bitOffset != 0 )
		*pLength = pStrm->datas + pStrm->byteOffset - pLength;
	else
		*pLength = pStrm->datas + pStrm->byteOffset - pLength -1;

	return HS_OK;
}



HS_RESULT UintFixedOctetEncoding( AsnStream *pStrm, HS_UINT byteSize, HS_UINT pValue )
{
	unsigned char aValue[4];

	if( pStrm == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintFixedOctetEncoding(NULL,%u,%u) -> null parameter", byteSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( byteSize == 0 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintFixedOctetEncoding(pStrm,%u,%u) -> asn range over", byteSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_OUTOF_RANGE;
	}
	
	if( byteSize > 4 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintFixedOctetEncoding(pStrm,%u,%u) -> not support yet", byteSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NOT_SUPPORT_YET;
	}

	Uint2Uchar( aValue, &pValue );

#ifdef HS_CPLUSPLUS
	return pStrm->BytesEncoding(aValue+4-byteSize, byteSize);
#else
	return AsnStream_BytesEncoding(pStrm, aValue+4-byteSize, byteSize);
#endif
}



HS_RESULT UintEncoding( AsnStream *pStrm, HS_UINT pValue )
{
	return UintFixedOctetEncoding( pStrm, Log256Size(pValue, TRUE), pValue );
}



HS_RESULT UintDecoding( AsnStream *pStrm, HS_UINT byteSize, HS_UINT *pValue )
{
	HS_RESULT		tRet;
	unsigned char	aValue[4];

	if( pStrm == NULL || pValue == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintDecoding(%x,%u,%x) -> null parameter", pStrm, byteSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( byteSize == 0 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintDecoding(pStrm,%u,pValue) -> asn range over", byteSize );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_OUTOF_RANGE;
	}
	
	if( byteSize > 4 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintDecoding(pStrm,%u,pValue) -> not support yet", byteSize );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NOT_SUPPORT_YET;
	}	

	memset( aValue, 0, 4 );

#ifdef HS_CPLUSPLUS
	tRet = pStrm->BytesDecoding(aValue+4-byteSize, byteSize);
#else
	tRet = AsnStream_BytesDecoding(pStrm, aValue+4-byteSize, byteSize);
#endif

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UintDecoding(pStrm,%u,pValue) -> %d line(%d)", byteSize, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	return Uchar2Uint( pValue, aValue );
}



HS_RESULT TwosComplementEncoding( AsnStream *pStrm, int pValue, HS_UINT *pLength )
{
	HS_RESULT		tRet;
	unsigned char	aValue[4];	/* array value */


	if( pStrm == HS_NULL || pLength == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on TwosComplementEncoding(%x,%d,%x) -> null parameter", pStrm, pValue, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning( pStrm );
#endif

	Uint2Uchar( aValue, (unsigned int*)&pValue );


	if( pValue < -8388608 )
	{
		*pLength = 4;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue, 4 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue, 4 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else if( pValue < -32768 )
	{
		*pLength = 3;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue+1, 3 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue+1, 3 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else if( pValue < -128 )
	{
		*pLength = 2;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue+2, 2 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue+2, 2 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else if( pValue < 128 )
	{
		*pLength = 1;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue+3, 1 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue+3, 1 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else if( pValue < 32768 )
	{
		*pLength = 2;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue+2, 2 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue+2, 2 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	/* Need 3 Bytes */
	else if( pValue < 8388608 )
	{
		*pLength = 3;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue+1, 3 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue+1, 3 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
		*pLength = 4;

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesEncoding( aValue, 4 );
#else
		tRet = AsnStream_BytesEncoding( pStrm, aValue, 4 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on TwosComplementEncoding(pStrm,%d,pLength) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	return HS_OK;
}



HS_RESULT TwosComplementDecoding( AsnStream *pStrm, int *pValue, HS_UINT pLength )
{
	HS_RESULT		tRet;
	unsigned char	aValue[4];
	BOOL			isNegative;
	HS_UINT			i, crosser;
	HS_UINT			tValue;


	if( pStrm == HS_NULL || pValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on TwosComplementDecoding(%x,%x,%u) -> null parameter", pStrm, pValue, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pLength > 4 || pLength == 0 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on TwosComplementDecoding(pStrm,pValue,%u) -> asn range over, line(%d)", pLength, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_OUTOF_RANGE;
	}


#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
	tRet = pStrm->BytesDecoding( aValue, pLength );
#else
	AsnStream_Aligning( pStrm );
	tRet = AsnStream_BytesDecoding( pStrm, aValue, pLength );
#endif

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on TwosComplementDecoding(pStrm,pValue,%u) -> %d, line(%d)", pLength, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}


	if( aValue[0] & 0x80 )
	{
		isNegative = TRUE;
		aValue[0] &= 0x7f;
	}

	tValue = 0;
	crosser = 1;
	for( i=0; i<pLength; i++ )
	{
		tValue += aValue[pLength-i-1] * crosser;
		crosser *= 256;
	}

	if( isNegative )
	{
		HS_UINT subValue;
		switch( pLength )
		{
			case 1: subValue = 128;			break;
			case 2: subValue = 32768;		break;
			case 3: subValue = 8388608;		break;
			case 4: subValue = 2147483648;	break;
		}

		*pValue = (int)(tValue - subValue);
	}
	else
		*pValue = tValue;

	return HS_OK;
}






/*********************************************************************************************************/
/* PER Length En/Decoding Rule
*/
HS_RESULT LengthEncoding( AsnStream *pStrm, HS_UINT pMin, HS_UINT pMax, HS_UINT pLength )
{
	HS_RESULT		tRet;
	HS_UINT			encodingValue;

	unsigned short	usValue;	/* unsigned short value */
	unsigned char	aValue[16];	/* array value */


	if( pStrm == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on LengthEncoding(NULL,%u,%u,%u) -> null parameter", pMin, pMax, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pMax < 65536 )
		return ConstrainedNumberEncoding( pStrm, pMin, pMax, pLength );


	if( pLength < pMin ) encodingValue = 0;
	else				 encodingValue = pLength - pMin;

	if( encodingValue < 128 )
	{
		aValue[0] = (unsigned char)encodingValue;

#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		tRet = pStrm->BytesEncoding(aValue, 1);
#else
		AsnStream_Aligning(pStrm);
		tRet = AsnStream_BytesEncoding(pStrm, aValue, 1);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on LengthEncoding(pStrm,%u,%u,%u) -> %d, line(%d)", pMin, pMax, pLength, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else if( encodingValue < 16384 )
	{
		usValue = (unsigned short)encodingValue;

		Ushort2Uchar( aValue, &usValue );
		aValue[0] |= 0x80;		/* '01' bits append */

#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		tRet = pStrm->BytesEncoding(aValue, 2);
#else
		AsnStream_Aligning(pStrm);
		tRet = AsnStream_BytesEncoding(pStrm, aValue, 2);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on LengthEncoding(pStrm,%u,%u,%u) -> %d, line(%d)", pMin, pMax, pLength, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on LengthEncoding(pStrm,%u,%u,%u) -> not support yet, line(%d)", pMin, pMax, pLength, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NOT_SUPPORT_YET;
	}

	return HS_OK;
}



HS_RESULT LengthDecoding( AsnStream *pStrm, HS_UINT pMin, HS_UINT pMax, HS_UINT *pLength )
{
	HS_RESULT		tRet;
	unsigned char	aValue[4];


	if( pStrm == NULL || pLength == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on LengthDecoding(%x,%u,%u,%x) -> null parameter", pStrm, pMin, pMax, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pMax < 65536 )
		return ConstrainedNumberDecoding( pStrm, pMin, pMax, pLength );


#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
	tRet = pStrm->BytesDecoding(aValue, 1);
#else
	AsnStream_Aligning(pStrm);
	tRet = AsnStream_BytesDecoding(pStrm, aValue, 1);
#endif

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on LengthDecoding(pStrm,%u,%u,pLength) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	/* 127 < length < 16384 */
	if( aValue[0] & 0x80 )
	{
		if( aValue[0] & 0x40 )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on LengthDecoding(pStrm,%u,%u,pLength) -> not support yet, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return HS_ERR_NOT_SUPPORT_YET;
		}

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BytesDecoding(aValue+1, 1);
#else
		tRet = AsnStream_BytesDecoding(pStrm, aValue+1, 1);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on LengthDecoding(pStrm,%u,%u,pLength) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		aValue[0] &= 0x7f;
		*pLength = (HS_UINT)(aValue[0]*256 + aValue[1]);
	}
	/* length < 128 */
	else
		*pLength = (HS_UINT)aValue[0];

	return HS_OK;
}






/*********************************************************************************************************/
/* PER Integer En/Decoding Rule
*/
HS_RESULT ConstrainedNumberEncoding( AsnStream *pStrm, int pMin, HS_UINT pMax, HS_UINT pValue )
{
	HS_RESULT		tRet;

	unsigned char	ucValue;	/* unsigned char value  */
	unsigned short	usValue;	/* unsigned short value */
	unsigned char	aValue[16];	/* array value */

	HS_UINT			distance;
	HS_UINT			encodingValue;


	if( pStrm == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on ConstrainedNumberEncoding(NULL,%d,%u,%u) -> null parameter", pMin, pMax, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	distance = pMax - pMin + 1;
	if( distance == 0 ) distance = HS_ASN_MAX_CONSTRAINT;

	if( pMin < 0 )	encodingValue = (int)pValue - pMin;
	else			encodingValue = pValue - pMin;


	if( distance == 1 ) return HS_OK;

	if( distance < 256 )
	{
		if( encodingValue > distance )	ucValue = distance;
		else							ucValue = (unsigned char)encodingValue;

#ifdef HS_CPLUSPLUS
		return pStrm->BitsEncoding( &ucValue, Log2Size(distance,FALSE), e_alignRight );
#else
		return AsnStream_BitsEncoding( pStrm, &ucValue, Log2Size(distance,FALSE), e_alignRight );
#endif
	}

	else if( distance == 256 )
	{
		if( encodingValue > distance )	ucValue = distance;
		else							ucValue = (unsigned char)encodingValue;

#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		return pStrm->BytesEncoding( &ucValue, 1 );
#else
		AsnStream_Aligning( pStrm );
		return AsnStream_BytesEncoding( pStrm, &ucValue, 1 );
#endif
	}

	else if( distance < 65537 )
	{
		if( encodingValue > distance )	usValue = distance;
		else							usValue = (unsigned short)encodingValue;

		Ushort2Uchar( aValue, &usValue );

#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		return pStrm->BytesEncoding( aValue, 2 );
#else
		AsnStream_Aligning( pStrm );
		return AsnStream_BytesEncoding( pStrm, aValue, 2 );
#endif
	}

	else
	{
		/* distance > 65536 */
		if( (tRet=LengthEncoding( pStrm, 1, Log256Size(distance,FALSE), Log256Size(encodingValue,TRUE))) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberEncoding(pStrm,%d,%u,%u) -> %d, line(%d)", pMin, pMax, pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif

		return UintEncoding(pStrm, encodingValue);
	}

	/* Warning Protection */
	return HS_OK;
}



HS_RESULT SemiConstrainedNumberEncoding( AsnStream *pStrm, int pMin, HS_UINT pValue )
{
	HS_RESULT	tRet;
	HS_UINT		byteSize;
	HS_UINT		encodingValue;


	if( pStrm == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SemiConstrainedNumberEncoding(NULL,%d,%u) -> null parameter", pMin, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pMin < 0 )	encodingValue = (int)pValue - pMin;
	else			encodingValue = pValue - pMin;

	byteSize = Log256Size( encodingValue, TRUE );


	if( (tRet=LengthEncoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, byteSize)) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SemiConstrainedNumberEncoding(pStrm,%d,%u) -> %d, line(%d)", pMin, pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning( pStrm );
#endif

	return UintEncoding(pStrm, encodingValue);
}



HS_RESULT UnconstrainedNumberEncoding( AsnStream *pStrm, int pValue )
{
	HS_RESULT	tRet;
	HS_UINT		tLength;
	HS_UCHAR	*tLengthPointer;

	if( pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UnconstrainedNumberEncoding(NULL,%d) -> null parameter", pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	
#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning(pStrm);
#endif

	tLengthPointer = pStrm->datas + pStrm->byteOffset;
	pStrm->byteOffset++;

	tRet = TwosComplementEncoding(pStrm, pValue, &tLength);
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UnconstrainedNumberEncoding(pStrm,%u) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	*tLengthPointer = tLength;
	return HS_OK;
}



HS_RESULT SmallNumberEncoding( AsnStream *pStrm, HS_UINT pValue )
{
	HS_RESULT		tRet;
	unsigned char	aValue[16];

	if( pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SmallNumberEncoding(NULL,%u) -> null parameter", pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if(pValue < 64)
	{
		aValue[0] = (unsigned char)pValue;

#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding(FALSE);
		tRet = pStrm->BitsEncoding(aValue, 6, e_alignRight);
#else
		AsnStream_BitEncoding(pStrm, FALSE);
		tRet = AsnStream_BitsEncoding(pStrm, aValue, 6, e_alignRight);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on SmallNumberEncoding(pStrm,%u) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding(TRUE);
#else
		AsnStream_BitEncoding(pStrm, TRUE);
#endif

		if( (tRet = SemiConstrainedNumberEncoding(pStrm, 0, pValue)) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on SmallNumberEncoding(pStrm,%u) -> %d, line(%d)", pValue, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		}
	}

	return HS_OK;
}



HS_RESULT ConstrainedNumberDecoding( AsnStream *pStrm, int pMin, HS_UINT pMax, HS_UINT *pValue )
{
	HS_RESULT		tRet;

	unsigned short	usValue;
	unsigned char	aValue[4];

	HS_UINT			tLength;
	HS_UINT			distance;


	if( pStrm == NULL || pValue == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on ConstrainedNumberDecoding(%x,%d,%u,%x) -> null parameter", pStrm, pMin, pMax, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	distance = pMax - pMin + 1;
	if( distance == 0 ) distance = HS_ASN_MAX_CONSTRAINT;

	if( distance == 1 )
	{
		*pValue = pMax;
		return HS_OK;
	}

	if( distance < 256 )
	{
#ifdef HS_CPLUSPLUS
		tRet = pStrm->BitsDecoding(aValue, Log2Size(distance,FALSE), e_alignRight);
#else
		tRet = AsnStream_BitsDecoding(pStrm, aValue, Log2Size(distance,FALSE), e_alignRight);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberDecoding(pStrm,%d,%u,pValue) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		*pValue = pMin + aValue[0];
	}

	else if( distance == 256 )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		tRet = pStrm->BytesDecoding( aValue, 1 );
#else
		AsnStream_Aligning( pStrm );
		tRet = AsnStream_BytesDecoding( pStrm, aValue, 1 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberDecoding(pStrm,%d,%u,pValue) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		*pValue = pMin + aValue[0];
	}

	else if( distance < 65537 )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
		tRet = pStrm->BytesDecoding( aValue, 2 );
#else
		AsnStream_Aligning( pStrm );
		tRet = AsnStream_BytesDecoding( pStrm, aValue, 2 );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberDecoding(pStrm,%d,%u,pValue) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		Uchar2Ushort( &usValue, aValue );
		*pValue = pMin + usValue;
	}

	else
	{
		/* distance > 65536 */
		if( (tRet = LengthDecoding(pStrm, 1, Log256Size(distance,FALSE), &tLength )) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberDecoding(pStrm,%d,%u,pValue) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}


#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif

		if( (tRet=UintDecoding( pStrm, tLength, pValue )) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on ConstrainedNumberDecoding(pStrm,%d,%u,pValue) -> %d, line(%d)", pMin, pMax, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		*pValue = pMin + *pValue;
	}

	return HS_OK;
}



HS_RESULT SemiConstrainedNumberDecoding( AsnStream *pStrm, int pMin, HS_UINT *pValue )
{
	HS_RESULT	tRet;
	HS_UINT		tLength;

	if( pStrm == NULL || pValue == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SemiConstrainedNumberDecoding(%x,%d,%x) -> null parameter", pStrm, pMin, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( (tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &tLength)) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SemiConstrainedNumberDecoding(pStrm,%d,pValue) -> %d, line(%d)", pMin, tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning( pStrm );
#endif

	return UintDecoding(pStrm, tLength, pValue);
}



HS_RESULT UnconstrainedNumberDecoding( AsnStream *pStrm, int *pValue )
{
	HS_RESULT	tRet;
	HS_UINT		tLength;

	if( pStrm == HS_NULL || pValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UnconstrainedNumberDecoding(%x,%x) -> null parameter", pStrm, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( (tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &tLength)) != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on UnconstrainedNumberDecoding(pStrm,pValue) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	return TwosComplementDecoding(pStrm, pValue, tLength);
}



HS_RESULT SmallNumberDecoding( AsnStream *pStrm, HS_UINT *pValue )
{
	HS_RESULT		tRet;

	BOOL			sizeCheck;
	unsigned char	ucValue;

	if( pStrm == NULL || pValue == NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on SmallNumberDecoding(%x,%x) -> null parameter", pStrm, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

#ifdef HS_CPLUSPLUS
	pStrm->BitDecoding( &sizeCheck );
#else
	AsnStream_BitDecoding( pStrm, &sizeCheck );
#endif

	/* 63 < length */
	if( sizeCheck )
		return SemiConstrainedNumberDecoding( pStrm, 0, pValue );
	else
	{
#ifdef HS_CPLUSPLUS
		tRet = pStrm->BitsDecoding(&ucValue, 6, e_alignRight);
#else
		tRet = AsnStream_BitsDecoding(pStrm, &ucValue, 6, e_alignRight);
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on SmallNumberDecoding(pStrm,pValue) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
		
		*pValue = (HS_UINT)ucValue;
	}

	return HS_OK;
}






/*********************************************************************************************************/
/* AsnObject Members
*/
#ifdef HS_CPLUSPLUS
AsnObject::AsnObject( AsnTag pTag )
{
	tag = pTag;
	length = 0;
}

AsnObject::~AsnObject()
{
}

#else
HS_RESULT new_AsnObject(
	void *pObject,
	void *pElement,
	int pSize,
	HS_RESULT (*pMakeMoldFunction)(void*),
	HS_RESULT (*pEncodeFunction)(void*,AsnStream*),
	HS_RESULT (*pDecodeFunction)(void*,AsnStream*),
#ifdef HS_DEBUG_ASN_PRINT
	HS_RESULT (*pPrintFunction)(void*,HS_UINT,char*),
#endif
	HS_RESULT (*pDeleteFunction)(void*)
)
{
	AsnObject *pObj = (AsnObject *)pObject;

	if( pObj == HS_NULL || pElement == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
#ifdef HS_DEBUG_ASN_PRINT
		HSPrint( "\n Error on AsnObject::AsnObject(%x,%x,%d,%x,%x,%x,%x) -> null parameter",
			pObj, pElement, pSize, pEncodeFunction, pDecodeFunction, pPrintFunction, pDeleteFunction
		);
#else
		HSPrint( "\n Error on AsnObject::AsnObject(%x,%x,%d,%x,%x,%x) -> null parameter",
			pObj, pElement, pSize, pEncodeFunction, pDecodeFunction, pDeleteFunction
		);
#endif
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->obj = pElement;
	pObj->size = pSize;
	pObj->MakeMoldFunction = pMakeMoldFunction;
	pObj->EncodeFunction = pEncodeFunction;
	pObj->DecodeFunction = pDecodeFunction;
#ifdef HS_DEBUG_ASN_PRINT
	pObj->PrintFunction = pPrintFunction;
#endif
	pObj->DeleteFunction = pDeleteFunction;

	return HS_OK;
}


HS_RESULT delete_AsnObject( void *pObject )
{
	AsnObject *pObj = (AsnObject *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObject::~AsnObject(NULL) -> null parameter");
#endif
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->obj != HS_NULL )
	{
		if( pObj->DeleteFunction != HS_NULL )
			pObj->DeleteFunction( pObj->obj );

		HSFree( pObj->obj );
	}

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* En/Decoding Function Extension Area Values
*/
#ifdef HS_CPLUSPLUS
HS_RESULT WellKnownExtensionEncoding(
	AsnObject *pEncodeObject,
	AsnStream *pStrm,
	AsnStream *pTempStrm
)
#else
HS_RESULT WellKnownExtensionEncoding(
	void *pEncodeObject,
	HS_RESULT(*pEncodeFunction)(void*,AsnStream*),
	AsnStream *pStrm,
	AsnStream *pTempStrm
)
#endif
{
	HS_RESULT	tRet;
	HS_UINT		octetSize;

#ifdef HS_CPLUSPLUS
	if( pEncodeObject == HS_NULL || pStrm == HS_NULL || pTempStrm == HS_NULL )
#else
	if( pEncodeObject == HS_NULL || pEncodeFunction == HS_NULL || pStrm == HS_NULL || pTempStrm == HS_NULL )
#endif
	{

#ifdef HS_DEBUG_ASN
#ifdef HS_CPLUSPLUS
		HSPrint( "\n Error on WellKnownExtensionEncoding(%x,%x,%x) -> null parameter", pEncodeObject, pStrm, pTempStrm );
#else
		HSPrint( "\n Error on WellKnownExtensionEncoding(%x,%x,%x,%x) -> null parameter", pEncodeObject, pEncodeFunction, pStrm, pTempStrm );
#endif/*HS_CPLUSPLUS*/
#endif/*HS_DEBUG_ASN*/

		return HS_ERR_NULL_PARAM;
	}


#ifdef HS_CPLUSPLUS
	pTempStrm->Clear();
	tRet = pEncodeObject->Encode(pTempStrm);
#else
	AsnStream_Clear(pTempStrm);
	tRet = pEncodeFunction(pEncodeObject, pTempStrm);
#endif

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionEncoding(Object,Function,pStrm,pTempStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	octetSize = pTempStrm->byteOffset + ( (pTempStrm->bitOffset)? 1:0 );

	tRet = LengthEncoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, octetSize);
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionEncoding(Object,Function,pStrm,pTempStrm) -> %d, line(%d), octetSize(%d)", tRet, __LINE__, octetSize );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning(pStrm);
#endif

	tRet = CheckBytesOverflow(pStrm,octetSize);
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionEncoding(Object,Function,pStrm,pTempStrm) -> %d, line(%d), octetSize(%d)", tRet, __LINE__, octetSize );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

	memcpy( pStrm->datas + pStrm->byteOffset, pTempStrm->datas, octetSize );
	pStrm->byteOffset += octetSize;
	return HS_OK;
}


/******************************/
/* AsnSingleString ExtensionDecode */
#ifdef HS_CPLUSPLUS
HS_RESULT WellKnownExtensionDecoding(
	AsnObject *pDecodeObject,
	AsnStream *pStrm
)
#else
HS_RESULT WellKnownExtensionDecoding(
	void *pDecodeObject,
	HS_RESULT(*pDecodeFunction)(void*,AsnStream*),
	AsnStream *pStrm
)
#endif
{
	HS_RESULT	tRet;
	HS_UINT		octetSize;
	HS_UINT		startOffset;

#ifdef HS_CPLUSPLUS
	if( pDecodeObject == HS_NULL || pStrm == HS_NULL )
#else
	if( pDecodeObject == HS_NULL || pDecodeFunction == HS_NULL || pStrm == HS_NULL )
#endif
	{

#ifdef HS_DEBUG_ASN
#ifdef HS_CPLUSPLUS
		HSPrint( "\n Error on WellKnownExtensionDecoding(%x,%x) -> null parameter", pDecodeObject, pStrm );
#else
		HSPrint( "\n Error on WellKnownExtensionDecoding(%x,%x,%x) -> null parameter", pDecodeObject, pDecodeFunction, pStrm );
#endif/*HS_CPLUSPLUS*/
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &octetSize);
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionDecoding(Object,Function,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
	startOffset = pStrm->byteOffset;
	tRet = pDecodeObject->Decode(pStrm);
#else
	AsnStream_Aligning(pStrm);
	startOffset = pStrm->byteOffset;
	tRet = pDecodeFunction(pDecodeObject,pStrm);
#endif
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionDecoding(Object,Function,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning(pStrm);
#endif
	if( octetSize != (pStrm->byteOffset - startOffset) )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on WellKnownExtensionDecoding(Object,Function,pStrm) -> asn invalid length, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		pStrm->byteOffset = startOffset + octetSize;
	}

	return HS_OK;
}







/*********************************************************************************************************/
/* AsnBoolean Members
*/
/**************************/
/* AsnBoolean Constructor */
#ifdef HS_CPLUSPLUS
AsnBoolean::AsnBoolean()
: AsnObject( e_asnTagBoolean )
#else
HS_RESULT new_AsnBoolean( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	length = 1;
#else
	AsnBoolean *pObj = (AsnBoolean *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBoolean::AsnBoolean() -> null parameter" );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	pObj->tag = e_asnTagBoolean;
	pObj->length = 1;
	
	return HS_OK;
#endif
}


/*************************/
/* AsnBoolean Destructor */
#ifdef HS_CPLUSPLUS
AsnBoolean::~AsnBoolean()
#else
HS_RESULT delete_AsnBoolean( void *pObject )
#endif
{

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif

}


/***********************/
/* AsnBoolean MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnBoolean_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/*********************/
/* AsnBoolean Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBoolean::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnBoolean_Encode( void *pObject, AsnStream *pStrm )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnBoolean *pObj = this;
#else
	AsnBoolean *pObj = (AsnBoolean *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBoolean::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

#ifdef HS_CPLUSPLUS
	pStrm->BitEncoding( pObj->value );
#else
	AsnStream_BitEncoding( pStrm, pObj->value );
#endif


	return HS_OK;
}


/*********************/
/* AsnBoolean Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBoolean::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnBoolean_Decode( void *pObject, AsnStream *pStrm )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnBoolean *pObj = this;
#else
	AsnBoolean *pObj = (AsnBoolean *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBoolean::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

#ifdef HS_CPLUSPLUS
	pStrm->BitDecoding( &(pObj->value) );
#else
	AsnStream_BitDecoding( pStrm, &(pObj->value) );
#endif

	return HS_OK;
}


/***********************/
/* AsnBoolean SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBoolean::SetValue( BOOL pValue )
#else
HS_RESULT AsnBoolean_SetValue( void *pObject, BOOL pValue )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnBoolean *pObj = this;
#else
	AsnBoolean *pObj = (AsnBoolean *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBoolean::SetValue(%s) -> null parameter", pValue? "TRUE":"FALSE" );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	pObj->value = pValue;

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/********************/
/* AsnBoolean Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBoolean::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnBoolean_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnBoolean *pObj = this;
#else
	AsnBoolean *pObj = (AsnBoolean *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBoolean::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<BOOLEAN> %s", pTypeName, pObj->value? "TRUE":"FALSE" );
	else
		HSPrint( "<BOOLEAN> %s", pObj->value? "TRUE":"FALSE" );

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnNull Members
*/
/***********************/
/* AsnNull Constructor */
#ifdef HS_CPLUSPLUS
AsnNull::AsnNull()
: AsnObject( e_asnTagNull )
#else
HS_RESULT new_AsnNull( void *pObject )
#endif
{
#ifndef HS_CPLUSPLUS
	AsnNull *pObj = (AsnNull *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnNull::AsnNull(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	pObj->tag = e_asnTagNull;
	pObj->length = 0;
	
	return HS_OK;
#endif
}


/**********************/
/* AsnNull Destructor */
#ifdef HS_CPLUSPLUS
AsnNull::~AsnNull()
#else
HS_RESULT delete_AsnNull( void *pObject )
#endif
{

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif

}


/********************/
/* AsnNull MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnNull_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/******************/
/* AsnNull Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnNull::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnNull_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	/* Nothing To Do About Null on PER Encoding */
	return HS_OK;
}


/******************/
/* AsnNull Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnNull::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnNull_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	/* Nothing To Do About Null on PER Decoding */
	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/********************/
/* AsnNull Print    */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnNull::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnNull_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<NULL>", pTypeName );
	else
		HSPrint( "<NULL>" );

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnInteger Members
*/
/**************************/
/* AsnInteger Constructor */
#ifdef HS_CPLUSPLUS
AsnInteger::AsnInteger(ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
: AsnObject( e_asnTagInteger )
#else
HS_RESULT new_AsnInteger(void *pObject, ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnInteger *pObj = this;
#else
	AsnInteger *pObj = (AsnInteger *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnInteger::AsnInteger(NULL,%d,%d,%u,%d,%u) -> null parameter",
			(int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = e_asnTagInteger;
	pObj->length = 0;
#endif

	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	if( pType == e_constrainedExt )
	{
		pObj->extMinValue = pExtMinValue;
		pObj->extMaxValue = pExtMaxValue;
	}

	pObj->extended = FALSE;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/*************************/
/* AsnInteger Destructor */
#ifdef HS_CPLUSPLUS
AsnInteger::~AsnInteger()
#else
HS_RESULT delete_AsnInteger( void *pObject )
#endif
{

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif

}


/***********************/
/* AsnInteger MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnInteger_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/*********************/
/* AsnInteger Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnInteger::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnInteger_Encode( void *pObject, AsnStream *pStrm )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnInteger *pObj = this;
#else
	AsnInteger *pObj = (AsnInteger *)pObject;
#endif


	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnInteger::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->constraint.distance == 1 ) return HS_OK;


	if( pObj->constraint.type == e_constrainedExt )
	{
		if( pObj->extended )
		{
#ifdef HS_CPLUSPLUS
			pStrm->BitEncoding(TRUE);
#else
			AsnStream_BitEncoding(pStrm,TRUE);
#endif

			return UnconstrainedNumberEncoding(pStrm, pObj->value);
		}
		else
		{
#ifdef HS_CPLUSPLUS
			pStrm->BitEncoding(FALSE);
#else
			AsnStream_BitEncoding(pStrm,FALSE);
#endif

			return ConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->value);
		}
	}

	else if( pObj->constraint.type == e_unconstrained )
		return UnconstrainedNumberEncoding(pStrm, pObj->value);
	
	else if( pObj->constraint.type == e_semiConstrained )
		return SemiConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->value);

	else /* pObj->constraint.type == e_constrained */
		return ConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->value);

	return HS_OK;
}


/*********************/
/* AsnInteger Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnInteger::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnInteger_Decode( void *pObject, AsnStream *pStrm )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnInteger *pObj = this;
#else
	AsnInteger *pObj = (AsnInteger *)pObject;
#endif


	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnInteger::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->constraint.distance == 1 )
	{
		pObj->value = pObj->constraint.maxValue;
		return HS_OK;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding( &(pObj->extended) );
#else
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );
#endif

		if( pObj->extended )
			return UnconstrainedNumberDecoding( pStrm, (int*)(&(pObj->value)) );
		else
			return ConstrainedNumberDecoding( pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->value) );
	}

	else if( pObj->constraint.type == e_unconstrained )
		return UnconstrainedNumberDecoding( pStrm, (int*)(&(pObj->value)) );
	
	else if( pObj->constraint.type == e_semiConstrained )
		return SemiConstrainedNumberDecoding( pStrm, pObj->constraint.minValue, &(pObj->value) );

	else
		return ConstrainedNumberDecoding( pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->value) );

	return HS_OK;
}


/***********************/
/* AsnInteger SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnInteger::SetValue( HS_UINT pValue )
#else
HS_RESULT AsnInteger_SetValue( void *pObject, HS_UINT pValue )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnInteger *pObj = this;
#else
	AsnInteger *pObj = (AsnInteger *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnInteger::SetValue(NULL,%u) -> null parameter", pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pValue > pObj->constraint.maxValue )
		pObj->extended = TRUE;
	else
		pObj->extended = FALSE;

	pObj->value = pValue;

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/********************/
/* AsnInteger Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnInteger::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnInteger_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnInteger *pObj = this;
#else
	AsnInteger *pObj = (AsnInteger *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnInteger::Print(%x,%u,%s) -> null parameter", pObj, pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
	{
		if( pObj->constraint.minValue < 0 || pObj->constraint.type == e_unconstrained )
			HSPrint( "%s<INTEGER> %d", pTypeName, (int)pObj->value );
		else
			HSPrint( "%s<INTEGER> %u", pTypeName, pObj->value );
	}
	else
	{
		if( pObj->constraint.minValue < 0 || pObj->constraint.type == e_unconstrained )
			HSPrint( "<INTEGER> %d", (int)pObj->value );
		else
			HSPrint( "<INTEGER> %u", pObj->value );
	}

	return HS_OK;
}
#endif






#if 0 /* Sample Code */
/*********************************************************************************************************/
/* AsnEnumerated Members
*/
/*****************************/
/* AsnEnumerated Constructor */
#ifdef HS_CPLUSPLUS
AsnEnumerated::AsnEnumerated( AsnEnumerated_AsnEnumerated pExtensionPoint )
: AsnObject( e_asnTagEnumerated )
#else
HS_RESULT new_AsnEnumerated( void *pObject, AsnEnumerated_AsnEnumerated pExtensionPoint )
#endif
{
	int i=0;

#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::AsnEnumerated(NULL,%d) -> null parameter", (int)pExtensionPoint );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = e_asnTagEnumerated;
	pObj->length = 0;
#endif

	pObj->value = e_enum0;
	pObj->extensionPoint = pExtensionPoint;

	pObj->enumSize = 0;
	for( i=0; i<HS_MAX_ENUMERATED; i++ )
		pObj->enumValue[i] = i;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/****************************/
/* AsnEnumerated Destructor */
#ifdef HS_CPLUSPLUS
AsnEnumerated::~AsnEnumerated()
#else
HS_RESULT delete_AsnEnumerated( void *pObject )
#endif
{
#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/**************************/
/* AsnEnumerated MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnEnumerated_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/******************************/
/* AsnEnumerated AddEnumValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::AddEnumValue( int pEnumValue )
#else
HS_RESULT AsnEnumerated_AddEnumValue( void *pObject, int pEnumValue )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::AddEnumValue(NULL,%d) -> null parameter", pEnumValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->enumSize < HS_MAX_ENUMERATED )
		pObj->enumValue[ pObj->enumSize ++ ] = pEnumValue;
	else
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::AddEnumValue(pObj,%d) -> full", pEnumValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_ENUMFULL;
	}

	return HS_OK;
}


/******************************/
/* AsnEnumerated GetEnumValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::GetEnumValue( HS_UINT pPoint, int *pEnumValue )
#else
HS_RESULT AsnEnumerated_GetEnumValue( void *pObject, HS_UINT pPoint, int *pEnumValue )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif
	
	if( pObj == HS_NULL || pEnumValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::GetEnumValue(%x,%u,%x) -> null parameter", pObj, pPoint, pEnumValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pPoint > (pObj->enumSize-1) )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::GetEnumValue(pObj,%u,pEnumValue) -> range over", pPoint );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_OUTOF_RANGE;
	}
	
	*pEnumValue = pObj->enumValue[ pPoint ];
	return HS_OK;
}


/**************************/
/* AsnEnumerated SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::SetValue( AsnEnumerated_AsnEnumerated pEnum )
#else
HS_RESULT AsnEnumerated_SetValue( void *pObject, AsnEnumerated_AsnEnumerated pEnum )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::SetValue(NULL,%d) -> null parameter", (int)pEnum );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	pObj->value = pEnum;
	return HS_OK;
}


/************************/
/* AsnEnumerated Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnEnumerated_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	BOOL			tExtended = FALSE;

#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_ASN_DEBG*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->extensionPoint < e_enum9 )
	{
		if( pObj->extensionPoint < pObj->value ) tExtended = TRUE;
		else									 tExtended = FALSE;

#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding(tExtended);
#else
		AsnStream_BitEncoding(pStrm,tExtended);
#endif
	}

	if( tExtended )
		return SmallNumberEncoding( pStrm, (HS_UINT)(pObj->value) );
	else
		return ConstrainedNumberEncoding( pStrm, 0, (HS_UINT)e_enum9, (HS_UINT)(pObj->value) );

	return HS_OK;
}


/************************/
/* AsnEnumerated Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnEnumerated_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
	HS_UINT			tValue;
	BOOL			tExtended = FALSE;

#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_ASN_DEBG*/
		return HS_ERR_NULL_PARAM;
	}


	if( pObj->extensionPoint < e_enum9 )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding(&tExtended);
#else
		AsnStream_BitDecoding(pStrm,&tExtended);
#endif
	}

	if( tExtended )
	{
		if( (tRet = SmallNumberDecoding(pStrm,&tValue)) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnEnumerated::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_ASN_DEBG*/
			return tRet;
		}

		pObj->value = (AsnEnumerated_AsnEnumerated)( pObj->extensionPoint + tValue + 1 );
	}
	else
	{
		if( (tRet = ConstrainedNumberDecoding(pStrm,0,e_enum9,&tValue)) != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnEnumerated::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_ASN_DEBG*/
			return tRet;
		}

		pObj->value = (AsnEnumerated_AsnEnumerated)tValue;
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/***********************/
/* AsnEnumerated Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnEnumerated::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnEnumerated_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnEnumerated *pObj = this;
#else
	AsnEnumerated *pObj = (AsnEnumerated *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnEnumerated::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<ENUMERATED> %d", pTypeName, (int)pObj->value );
	else
		HSPrint( "<ENUMERATED> %d", (int)pObj->value );

	return HS_OK;
}
#endif/*HS_DEBUG_ASN_PRINT*/
#endif/*0:Sample Code*/






/*********************************************************************************************************/
/* AsnReal Members
*/
#if 0 /*NOT_SUPPORT_YET*/
#endif/*NOT_SUPPORT_YET*/






/*********************************************************************************************************/
/* AsnBitString Members
*/
/****************************/
/* AsnBitString Constructor */
#ifdef HS_CPLUSPLUS
AsnBitString::AsnBitString(ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
: AsnObject( e_asnTagBitString )
#else
HS_RESULT new_AsnBitString(void *pObject, ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::AsnBitString(NULL,%d,%d,%u,%d,%u) -> null parameter",
				 (int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = e_asnTagBitString;
	pObj->length = 0;
#endif

	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	if( pType == e_constrainedExt )
	{
		pObj->extMinValue = pExtMinValue;
		pObj->extMaxValue = pExtMaxValue;
	}

	pObj->extended = FALSE;

	pObj->value = HS_NULL;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/***************************/
/* AsnBitString Destructor */
#ifdef HS_CPLUSPLUS
AsnBitString::~AsnBitString()
#else
HS_RESULT delete_AsnBitString(void *pObject)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::~AsnBitString(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	if( pObj->value != HS_NULL ) HSFree( pObj->value );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/*************************/
/* AsnBitString MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnBitString_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/*************************/
/* AsnBitString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBitString::SetValue(HS_UINT pBitSize, unsigned char *pValue)
#else
HS_RESULT AsnBitString_SetValue(void *pObject, HS_UINT pBitSize, unsigned char *pValue)
#endif
{

	HS_UINT tByteSize;

#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::SetValue(NULL,%u,%x) -> null parameter", pBitSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pBitSize == 0 )
	{
		pObj->length = 0;

		if(pObj->value != HS_NULL)
		{
			HSFree(pObj->value);
			pObj->value = HS_NULL;
		}

		return HS_OK;
	}

	if( pValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::SetValue(%x,%u,NULL) -> null parameter", pObj, pBitSize );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	
	if( pObj->value != HS_NULL ) HSFree(pObj->value);

	tByteSize = (HS_UINT)((pBitSize+7)/8);
	pObj->value = (unsigned char*)HSMalloc( tByteSize );
	memcpy( pObj->value, pValue, tByteSize );
	pObj->length = pBitSize;

	if( pBitSize > pObj->constraint.maxValue ) pObj->extended = TRUE;
	else pObj->extended = FALSE;

	return HS_OK;
}


/***********************/
/* AsnBitString Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBitString::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnBitString_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->length != 0 && pObj->value == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> asn novalue, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_CONFLICT;
	}
	if( pObj->length > 65536 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> not support yet, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NOT_SUPPORT_YET;
	}


	/* extension preamble */
	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding( pObj->extended );
#else
		AsnStream_BitEncoding( pStrm, pObj->extended );
#endif
	}


	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberEncoding(pStrm, 0, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
		if( pObj->constraint.type == e_unconstrained )
		{
			tRet = LengthEncoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.type == e_semiConstrained )
		{
			tRet = SemiConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> %d, line(%d)"
						 "\n AsnStream : size[%u], byteOffset[%u], bitOffset[%u]",
						 tRet, __LINE__, pStrm->size, pStrm->byteOffset, pStrm->bitOffset
				);
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.distance > 1 )
		{
			tRet = ConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
	}

	/* data encoding */
	if( pObj->length > 16 )
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif


#ifdef HS_CPLUSPLUS
	return pStrm->BitsEncoding(pObj->value, pObj->length, e_alignLeft);
#else
	return AsnStream_BitsEncoding(pStrm, pObj->value, pObj->length, e_alignLeft);
#endif
}


/***********************/
/* AsnBitString Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBitString::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnBitString_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
	HS_UINT			tByteSize;

#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding( &(pObj->extended) );
#else
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );
#endif
	}


	/* length decoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberDecoding( pStrm, 0, &(pObj->length) );
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnBitString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
		if( pObj->constraint.type == e_unconstrained )
		{
			tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.type == e_semiConstrained )
		{
			tRet = SemiConstrainedNumberDecoding(pStrm, pObj->constraint.minValue, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.distance > 1 )
		{
			tRet = ConstrainedNumberDecoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnBitString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else pObj->length = pObj->constraint.minValue;
	}


	/* DataDecoding */
	tByteSize = (HS_UINT)((pObj->length+7)/8);

	if( pObj->value != HS_NULL ) HSFree(pObj->value);
	pObj->value = (unsigned char*)HSMalloc( tByteSize );
	memset( pObj->value, 0, tByteSize );

	if( pObj->length > 16 )
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif

#ifdef HS_CPLUSPLUS
	return pStrm->BitsDecoding(pObj->value, pObj->length, e_alignLeft);
#else
	return AsnStream_BitsDecoding(pStrm, pObj->value, pObj->length, e_alignLeft);
#endif
}


#ifdef HS_DEBUG_ASN_PRINT
/**********************/
/* AsnBitString Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnBitString::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnBitString_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i, j;
	HS_UCHAR tByte;
	
#ifdef HS_CPLUSPLUS
	AsnBitString *pObj = this;
#else
	AsnBitString *pObj = (AsnBitString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnBitString::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<BIT STRING>", pTypeName );
	else
		HSPrint( "<BIT STRING>" );

	for( j=0; j<pObj->length; j++ )
	{
		if( j%64 == 0 )
		{
			HSPrint("\n");
			for( i=0; i<pDepth; i++ ) HSPrint( "  " );
		}
		if( j%8 == 0 ) HSPrint( " " );
		if( j%32 == 0 ) HSPrint( "  " );

		tByte = 1<<(7-(j%8));
		if( tByte & pObj->value[j/8] ) HSPrint( "1" );
		else HSPrint( "0" );
	}

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnOctetString Members
*/
/******************************/
/* AsnOctetString Constructor */
#ifdef HS_CPLUSPLUS
AsnOctetString::AsnOctetString(ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
: AsnObject( e_asnTagOctetString )
#else
HS_RESULT new_AsnOctetString(void *pObject, ConstraintType pType, int pMinValue, HS_UINT pMaxValue, int pExtMinValue, HS_UINT pExtMaxValue)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::AsnOctetString(NULL,%d,%d,%u,%d,%u) -> null parameter",
				 (int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = e_asnTagOctetString;
	pObj->length = 0;
#endif

	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	if( pType == e_constrainedExt )
	{
		pObj->extMinValue = pExtMinValue;
		pObj->extMaxValue = pExtMaxValue;
	}

	pObj->extended = FALSE;

	pObj->value = HS_NULL;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/*****************************/
/* AsnOctetString Destructor */
#ifdef HS_CPLUSPLUS
AsnOctetString::~AsnOctetString()
#else
HS_RESULT delete_AsnOctetString(void *pObject)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::~AsnOctetString(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	if( pObj->value != HS_NULL ) HSFree( pObj->value );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/***************************/
/* AsnOctetString MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnOctetString_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/***************************/
/* AsnOctetString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnOctetString::SetValue(HS_UINT pByteSize, unsigned char *pValue)
#else
HS_RESULT AsnOctetString_SetValue(void *pObject, HS_UINT pByteSize, unsigned char *pValue)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::SetValue(NULL,%u,%x) -> null parameter", pByteSize, pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pByteSize == 0 )
	{
		pObj->length = 0;

		if(pObj->value != HS_NULL)
		{
			HSFree(pObj->value);
			pObj->value = HS_NULL;
		}

		return HS_OK;
	}

	if( pValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::SetValue(%x,%u,NULL) -> null parameter", pObj, pByteSize );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->value != HS_NULL ) HSFree(pObj->value);

	pObj->value = (unsigned char*)HSMalloc( pByteSize );
	memcpy( pObj->value, pValue, pByteSize );
	pObj->length = pByteSize;

	if( pByteSize > pObj->constraint.maxValue ) pObj->extended = TRUE;
	else pObj->extended = FALSE;

	return HS_OK;
}


/*************************/
/* AsnOctetString Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnOctetString::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnOctetString_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->length != 0 && pObj->value == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> asn novalue, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_CONFLICT;
	}
	if( pObj->length > 65536 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> not support yet, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NOT_SUPPORT_YET;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding( pObj->extended );
#else
		AsnStream_BitEncoding( pStrm, pObj->extended );
#endif
	}


	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberEncoding(pStrm, 0, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
		if( pObj->constraint.type == e_unconstrained )
		{
			tRet = LengthEncoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.type == e_semiConstrained )
		{
			tRet = SemiConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.distance > 1 )
		{
			tRet = ConstrainedNumberEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->length);
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
	}


	/* data encoding */
	if( pObj->length > 2 )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

#ifdef HS_CPLUSPLUS
	return pStrm->BytesEncoding(pObj->value, pObj->length);
#else
	return AsnStream_BytesEncoding(pStrm, pObj->value, pObj->length);
#endif
}


/*************************/
/* AsnOctetString Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnOctetString::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnOctetString_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;

#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding( &(pObj->extended) );
#else
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );
#endif
	}

	/* length decoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberDecoding(pStrm, 0, &(pObj->length));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnOctetString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	else
	{
		if( pObj->constraint.type == e_unconstrained )
		{
			tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.type == e_semiConstrained )
		{
			tRet = SemiConstrainedNumberDecoding(pStrm, pObj->constraint.minValue, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Decode(pObj,pStrm) -> %d, line(%d)"
						 "\n AsnStream : size[%u], byteOffset[%u], bitOffset[%u]",
						 tRet, __LINE__, pStrm->size, pStrm->byteOffset, pStrm->bitOffset
				);
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else if( pObj->constraint.distance > 1 )
		{
			tRet = ConstrainedNumberDecoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->length));
			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnOctetString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
		else pObj->length = pObj->constraint.minValue;
	}


	/* DataDecoding */
	if( pObj->value != HS_NULL ) HSFree(pObj->value);
	pObj->value = (unsigned char*)HSMalloc( pObj->length );

	if( pObj->length > 2 )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

#ifdef HS_CPLUSPLUS
	return pStrm->BytesDecoding(pObj->value, pObj->length);
#else
	return AsnStream_BytesDecoding(pStrm, pObj->value, pObj->length);
#endif
}


#ifdef HS_DEBUG_ASN_PRINT
/************************/
/* AsnOctetString Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnOctetString::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnOctetString_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i, j;
	
#ifdef HS_CPLUSPLUS
	AsnOctetString *pObj = this;
#else
	AsnOctetString *pObj = (AsnOctetString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnOctetString::Print(%x,%u,%s) -> null parameter", pObj, pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<OCTET STRING>", pTypeName );
	else
		HSPrint( "<OCTET STRING>" );

	/* Octet Value */
	for( j=0; j<pObj->length; j++ )
	{
		if( j%16 == 0 )
		{
			HSPrint("\n");
			for( i=0; i<pDepth; i++ ) HSPrint( "  " );
		}
		if( j%8 == 0 ) HSPrint( "   " );

		HSPrint( "%02X ", pObj->value[j] );
	}
	/* String Value */
	if( pObj->length < 128 )
	{
		HSPrint("\n");
		for( i=0; i<pDepth; i++ ) HSPrint( "  " );
		HSPrint( "   (" );

		for( i=0; i<pObj->length; i++ )
		{
			if(pObj->value[i]<32 || pObj->value[i]>126) HSPrint( "." );
			else HSPrint( "%c", pObj->value[i] );
		}
		HSPrint( ")" );
	}

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnObjectIdentifier Members
*/
/***********************************/
/* AsnObjectIdentifier Constructor */
#ifdef HS_CPLUSPLUS
AsnObjectIdentifier::AsnObjectIdentifier()
: AsnObject( e_asnTagObjectIdentifier )
#else
HS_RESULT new_AsnObjectIdentifier( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::AsnObjectIdentifier(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = e_asnTagObjectIdentifier;
	pObj->length = 0;
#endif

	pObj->value = HS_NULL;

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/**********************************/
/* AsnObjectIdentifier Destructor */
#ifdef HS_CPLUSPLUS
AsnObjectIdentifier::~AsnObjectIdentifier()
#else
HS_RESULT delete_AsnObjectIdentifier( void *pObject )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::~AsnObjectIdentifier(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	if( pObj->value != HS_NULL ) HSFree( pObj->value );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif

}


/********************************/
/* AsnObjectIdentifier MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnObjectIdentifier_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/********************************/
/* AsnObjectIdentifier SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnObjectIdentifier::SetValue( unsigned char *pValue, unsigned pLength )
#else
HS_RESULT AsnObjectIdentifier_SetValue( void *pObject, unsigned char *pValue, unsigned pLength )
#endif
{
#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL || pValue == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::SetValue(%x,%x,%u) -> null parameter", pObj, pValue, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pObj->value != HS_NULL) HSFree( pObj->value );
	pObj->value = (unsigned char*)HSMalloc( pLength );

	memcpy( pObj->value, pValue, pLength );
	pObj->length = pLength;

	return HS_OK;
}


/******************************/
/* AsnObjectIdentifier Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnObjectIdentifier::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnObjectIdentifier_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;

#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->value == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Encode(pObj,pStrm) -> no value, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_ASN_CONFLICT;
	}


	/* Length Encoding */
	tRet = LengthEncoding( pStrm, 0, HS_ASN_MAX_CONSTRAINT, pObj->length );
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning( pStrm );
#endif

	/* Data Encoding */
#ifdef HS_CPLUSPLUS
	return pStrm->BytesEncoding(pObj->value, pObj->length);
#else
	return AsnStream_BytesEncoding(pStrm, pObj->value, pObj->length);
#endif
}


/******************************/
/* AsnObjectIdentifier Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnObjectIdentifier::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnObjectIdentifier_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;

#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	/* Length Decoding */
	tRet = LengthDecoding(pStrm, 0, HS_ASN_MAX_CONSTRAINT, &(pObj->length));
	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
		return tRet;
	}

#ifdef HS_CPLUSPLUS
	pStrm->Aligning();
#else
	AsnStream_Aligning( pStrm );
#endif

	/* Data Decoding */
	if( pObj->value != HS_NULL ) HSFree( pObj->value );
	pObj->value = (unsigned char*)HSMalloc( pObj->length );

#ifdef HS_CPLUSPLUS
	return pStrm->BytesDecoding(pObj->value, pObj->length);
#else
	return AsnStream_BytesDecoding(pStrm, pObj->value, pObj->length);
#endif
}


#ifdef HS_DEBUG_ASN_PRINT
/*****************************/
/* AsnObjectIdentifier Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnObjectIdentifier::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnObjectIdentifier_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnObjectIdentifier *pObj = this;
#else
	AsnObjectIdentifier *pObj = (AsnObjectIdentifier *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnObjectIdentifier::Print(%x,%u,%s) -> null parameter", pObj, pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
		HSPrint( "%s<OBJECT IDENTIFIER> ", pTypeName );
	else
		HSPrint( "<OBJECT IDENTIFIER> " );

	for( i=0; i<pObj->length; i++ ) HSPrint( "%02X ", pObj->value[i] );

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnSingleString members
*/

/*
	#GetStringTypeEncodingBits# function calculating the bit size that encoding of one charactor.
	Return value is TRUE if need re-indexing, otherwise FALSE.

	Notice : this function must be called after set FROM constraint.
*/
BOOL GetStringTypeEncodingBits( HS_UINT pN, HS_UINT pVmax, HS_UINT *pAlignBitSize, HS_UINT *pUnalignBitSize )
{
	HS_UINT b, c, tBitMax;

	b = Log2Size( pN, TRUE );
	c = Log2Size( b,  TRUE );
	
	*pAlignBitSize = 1<<c;
	*pUnalignBitSize = b;
	
	tBitMax = 1 << (*pAlignBitSize) ;
	return (pVmax>tBitMax);
}

/********************************/
/* AsnSingleString Constructor */
#ifdef HS_CPLUSPLUS
AsnSingleString::AsnSingleString(
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
) : AsnObject( pTag )
#else
HS_RESULT new_AsnSingleString(
	void *pObject,
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::AsnSingleString(NULL,%d,%d,%d,%u,%d,%u) -> null parameter",
				 (int)pTag, (int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = pTag;
	pObj->length = 0;
#endif

	pObj->value = HS_NULL;
	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	pObj->extended = FALSE;
	pObj->extMinValue = pExtMinValue;
	pObj->extMaxValue = pExtMaxValue;

	if( pTag == e_asnTagNumericString )
	{
		pObj->fromSize = 11;
		pObj->encodingBitsOfAlign = 4;
		pObj->encodingBitsOfUnalign = 4;
		pObj->reIndexing = TRUE;
		memcpy( pObj->fromConstraint, " 0123456789", 11 );
	}
	else if( pTag == e_asnTagPrintableString )
	{
		pObj->fromSize = 74;
		pObj->encodingBitsOfAlign = 8;
		pObj->encodingBitsOfUnalign = 7;
		pObj->reIndexing = FALSE;
		memcpy( pObj->fromConstraint, " '()+,-./0123456789:=?ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", 74 );
	}
	else if( pTag == e_asnTagVisibleString )
	{
		pObj->fromSize = 97;
		pObj->encodingBitsOfAlign = 8;
		pObj->encodingBitsOfUnalign = 7;
		pObj->reIndexing = FALSE;
		memcpy( pObj->fromConstraint, " !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~", 97 );
	}
	else if( pTag == e_asnTagIA5String )
	{
		unsigned char i;

		pObj->fromSize = 128;
		pObj->encodingBitsOfAlign = 8;
		pObj->encodingBitsOfUnalign = 7;
		pObj->reIndexing = FALSE;
		for( i=0; i<pObj->fromSize; i++ ) pObj->fromConstraint[i] = i;
	}
	else if( pTag == e_asnTagGeneralString )
	{
		unsigned char i;

		pObj->fromSize = 256;
		pObj->encodingBitsOfAlign = 8;
		pObj->encodingBitsOfUnalign = 8;
		pObj->reIndexing = FALSE;
		for( i=0; i<HS_UCHAR_MAX; i++ ) pObj->fromConstraint[i] = i;
	}

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/*******************************/
/* AsnSingleString Destructor */
#ifdef HS_CPLUSPLUS
AsnSingleString::~AsnSingleString()
#else
HS_RESULT delete_AsnSingleString(void *pObject)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::~AsnSingleString(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	if( pObj->value != HS_NULL ) HSFree( pObj->value );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/****************************/
/* AsnSingleString MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnSingleString_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/****************************/
/* AsnSingleString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnSingleString::SetValue( unsigned char *pValue, HS_UINT pLength )
#else
HS_RESULT AsnSingleString_SetValue( void *pObject, unsigned char *pValue, HS_UINT pLength )
#endif
{

#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::SetValue(NULL,%x,%u) -> null parameter", pValue, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pValue != HS_NULL && pLength > 0 )
	{
		if( pObj->value != HS_NULL ) HSFree( pObj->value );

		pObj->length = pLength;
		pObj->value = (unsigned char*)HSMalloc( pObj->length );
		memcpy( pObj->value, pValue, pObj->length );

		pObj->extended = FALSE;
		if( pObj->constraint.type == e_constrainedExt )
		{
			if( pObj->constraint.maxValue < pObj->length ) pObj->extended = TRUE;
		}

		return HS_OK;
	}

	if( pObj->value != HS_NULL )
	{
		HSFree( pObj->value );
		pObj->value = HS_NULL;
	}

	pObj->length = 0;
	return HS_OK;
}


/*****************************/
/* AsnSingleString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnSingleString::SetValueInVisible( char *pValue )
#else
HS_RESULT AsnSingleString_SetValueInVisible( void *pObject, char *pValue )
#endif
{

#ifdef HS_CPLUSPLUS
	return SetValue( (unsigned char*)pValue, strlen(pValue) );
#else
	return AsnSingleString_SetValue( pObject, (unsigned char*)pValue, strlen(pValue) );
#endif
}


/***************************/
/* AsnSingleString Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnSingleString::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnSingleString_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
	unsigned char	j;
	HS_UINT			i, encodingBits;

#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->value == HS_NULL )
	{
		if( pObj->constraint.minValue > 0 )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> asn no value, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return HS_ERR_ASN_CONFLICT;
		}
		else pObj->length = 0;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding( pObj->extended );
#else
		AsnStream_BitEncoding( pStrm, pObj->extended );
#endif
	}

	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberEncoding(pStrm, 0, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}
	else if( pObj->constraint.distance > 1 || pObj->constraint.maxValue > 65535 )
	{
		tRet = LengthEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	if( pObj->length == 0 ) return HS_OK;


	/* data encoding */
	if( pStrm->isAlign ) encodingBits = pObj->encodingBitsOfAlign;
	else				 encodingBits = pObj->encodingBitsOfUnalign;

	if(  ( encodingBits*(pObj->constraint.maxValue) )  >  16  )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

	if( pObj->reIndexing )
	{
		for( i=0; i<pObj->length; i++ )
		{
			for( j=0; j<pObj->fromSize; j++ )
			{
				if( pObj->fromConstraint[j] == pObj->value[i] )
				{
#ifdef HS_CPLUSPLUS
					tRet = pStrm->BitsEncoding( &j, encodingBits, e_alignRight );
#else
					tRet = AsnStream_BitsEncoding( pStrm, &j, encodingBits, e_alignRight );
#endif

					if( tRet != HS_OK )
					{
#ifdef HS_DEBUG_ASN
						HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
						return tRet;
					}

					break;
				}
			}

			/* Cant Found fromConstraint */
			if( j == pObj->fromSize )
			{
				j = 0;
#ifdef HS_CPLUSPLUS
				tRet = pStrm->BitsEncoding( &j, encodingBits, e_alignRight );
#else
				tRet = AsnStream_BitsEncoding( pStrm, &j, encodingBits, e_alignRight );
#endif

				if( tRet != HS_OK )
				{
#ifdef HS_DEBUG_ASN
					HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
					return tRet;
				}
			}

		}
	}
	else
	{
		for( i=0; i<pObj->length; i++ )
		{
#ifdef HS_CPLUSPLUS
			tRet = pStrm->BitsEncoding( pObj->value+i, encodingBits, e_alignRight );
#else
			tRet = AsnStream_BitsEncoding( pStrm, pObj->value+i, encodingBits, e_alignRight );
#endif

			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnSingleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
	}

	return HS_OK;
}


/***************************/
/* AsnSingleString Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnSingleString::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnSingleString_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT			tRet;
	HS_UINT				i, encodingBits;

#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding( &(pObj->extended) );
#else
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );
#endif
	}

	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberDecoding(pStrm, 0, &(pObj->length));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}
	else if( pObj->constraint.distance > 1 || pObj->constraint.maxValue > 65535 )
	{
		tRet = LengthDecoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->length));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	if( pObj->constraint.distance < 2 ) pObj->length = pObj->constraint.minValue;

	/* value field reset */
	if( pObj->value != HS_NULL ) HSFree( pObj->value );
	if( pObj->length == 0 ) 
	{
		pObj->value = HS_NULL;
		return HS_OK;
	}
	pObj->value = (unsigned char*)HSMalloc( pObj->length );
	memset(pObj->value, 0, pObj->length );

	/* data encoding */
	if( pStrm->isAlign ) encodingBits = pObj->encodingBitsOfAlign;
	else				 encodingBits = pObj->encodingBitsOfUnalign;

	if(  ( encodingBits*(pObj->constraint.maxValue) )  >  16  )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

	for( i=0; i<pObj->length; i++ )
	{
#ifdef HS_CPLUSPLUS
		tRet = pStrm->BitsDecoding( pObj->value+i, encodingBits, e_alignRight );
#else
		tRet = AsnStream_BitsDecoding( pStrm, pObj->value+i, encodingBits, e_alignRight );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSingleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		if( pObj->reIndexing )
		{
			if( pObj->value[i] > pObj->fromSize - 1 )
				pObj->value[i] = pObj->fromConstraint[0];
			else
				pObj->value[i] = pObj->fromConstraint[ pObj->value[i] ];
		}
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/*************************/
/* AsnSingleString Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnSingleString::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnSingleString_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnSingleString *pObj = this;
#else
	AsnSingleString *pObj = (AsnSingleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSingleString::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
	{
		switch( pObj->tag )
		{
			case e_asnTagNumericString:		HSPrint( "%s<NumericString> "	, pTypeName );		break;
			case e_asnTagPrintableString:	HSPrint( "%s<PrintableString> "	, pTypeName );		break;
			case e_asnTagVisibleString:		HSPrint( "%s<VisibleString> "	, pTypeName );		break;
			case e_asnTagIA5String:			HSPrint( "%s<IA5String> "		, pTypeName );		break;
			case e_asnTagGeneralString:		HSPrint( "%s<GeneralString> "	, pTypeName );		break;
			default:						HSPrint( "%s<SingleString> "		, pTypeName );		break;
		}
	}
	else
	{
		switch( pObj->tag )
		{
			case e_asnTagNumericString:		HSPrint( "<NumericString> " );		break;
			case e_asnTagPrintableString:	HSPrint( "<PrintableString> " );		break;
			case e_asnTagVisibleString:		HSPrint( "<VisibleString> " );		break;
			case e_asnTagIA5String:			HSPrint( "<IA5String> " );			break;
			case e_asnTagGeneralString:		HSPrint( "<GeneralString> " );		break;
			default:						HSPrint( "<SingleString> " );		break;
		}
	}

	for( i=0; i<pObj->length; i++ )
	{
		if( pObj->value[i] < 32 || pObj->value[i] > 126 ) HSPrint( "\\%d", pObj->value[i] );
		else HSPrint( "%c", pObj->value[i] );
	}

	return HS_OK;
}
#endif






/*********************************************************************************************************/
/* AsnDoubleString members
*/
/********************************/
/* AsnDoubleString Constructor */
#ifdef HS_CPLUSPLUS
AsnDoubleString::AsnDoubleString(
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
) : AsnObject( pTag )
#else
HS_RESULT new_AsnDoubleString(
	void *pObject,
	AsnTag pTag,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::AsnDoubleString(NULL,%d,%d,%d,%u,%d,%u) -> null parameter",
				 (int)pTag, (int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

#ifndef HS_CPLUSPLUS
	pObj->tag = pTag;
	pObj->length = 0;
#endif
	
	pObj->value = HS_NULL;
	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	pObj->extended = FALSE;
	pObj->extMinValue = pExtMinValue;
	pObj->extMaxValue = pExtMaxValue;

	if( pTag == e_asnTagBMPString )
	{
		unsigned short i;

		pObj->fromSize = 65536;
		pObj->encodingBitsOfAlign = 16;
		pObj->encodingBitsOfUnalign = 16;
		pObj->reIndexing = FALSE;
		for( i=0; i<HS_USHORT_MAX; i++ ) pObj->fromConstraint[i] = i;
	}

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/*******************************/
/* AsnDoubleString Destructor */
#ifdef HS_CPLUSPLUS
AsnDoubleString::~AsnDoubleString()
#else
HS_RESULT delete_AsnDoubleString(void *pObject)
#endif
{

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::~AsnDoubleString(NULL) -> null parameter" );
#endif/*HS_DEBUG_ASN*/

#ifdef HS_CPLUSPLUS
		return;
#else
		return HS_ERR_NULL_PARAM;
#endif
	}

	if( pObj->value != HS_NULL ) HSFree( pObj->value );

#ifndef HS_CPLUSPLUS
	return HS_OK;
#endif
}


/****************************/
/* AsnDoubleString MakeMold */
#ifndef HS_CPLUSPLUS
HS_RESULT AsnDoubleString_MakeMold( void *pObject )
{
	return HS_OK;
}
#endif


/****************************/
/* AsnDoubleString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnDoubleString::SetValue( unsigned short *pValue, HS_UINT pLength )
#else
HS_RESULT AsnDoubleString_SetValue( void *pObject, unsigned short *pValue, HS_UINT pLength )
#endif
{
	HS_UINT	i;

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::SetValue(NULL,%x,%u) -> null parameter", pValue, pLength );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pValue != HS_NULL && pLength > 0 )
	{
		if( pObj->value != HS_NULL ) HSFree( pObj->value );

		pObj->length = pLength;
		pObj->value = (unsigned short*)HSMalloc( pLength*2 );
		for( i=0; i<pLength; i++) 
			pObj->value[i] = pValue[i];

		pObj->extended = FALSE;
		if( pObj->constraint.type == e_constrainedExt )
		{
			if( pObj->constraint.maxValue < pObj->length ) pObj->extended = TRUE;
		}

		return HS_OK;
	}

	if( pObj->value != HS_NULL )
	{
		HSFree( pObj->value );
		pObj->value = HS_NULL;
	}

	pObj->length = 0;
	return HS_OK;
}


/*****************************/
/* AsnDoubleString SetValue */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnDoubleString::SetValueInVisible( char *pValue )
#else
HS_RESULT AsnDoubleString_SetValueInVisible( void *pObject, char *pValue )
#endif
{
	HS_UINT			i;
	HS_UINT			tLength;

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::SetValueInVisible(NULL,%x) -> null parameter", pValue );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}

	if( pValue != HS_NULL )
	{
		if( (tLength=strlen(pValue)) > 0 )
		{
			if( pObj->value != HS_NULL ) HSFree( pObj->value );

			pObj->length = tLength;
			pObj->value = (unsigned short *)HSMalloc( tLength*2 );
			for( i=0; i<tLength; i++) 
				pObj->value[i] = (unsigned short)pValue[i];

			pObj->extended = FALSE;
			if( pObj->constraint.type == e_constrainedExt )
			{
				if( pObj->constraint.maxValue < pObj->length ) pObj->extended = TRUE;
			}

			return HS_OK;
		}
	}

	if( pObj->value != HS_NULL )
	{
		HSFree( pObj->value );
		pObj->value = HS_NULL;
	}

	pObj->length = 0;
	return HS_OK;
}


/***************************/
/* AsnDoubleString Encode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnDoubleString::Encode( AsnStream *pStrm )
#else
HS_RESULT AsnDoubleString_Encode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT		tRet;
	unsigned short	j;
	HS_UINT			i, encodingBits;
	unsigned char	encodingShort[2];

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->value == HS_NULL )
	{
		if( pObj->constraint.minValue > 0 )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> asn no value, line(%d)", __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return HS_ERR_ASN_CONFLICT;
		}
		else pObj->length = 0;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitEncoding( pObj->extended );
#else
		AsnStream_BitEncoding( pStrm, pObj->extended );
#endif
	}

	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberEncoding(pStrm, 0, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}
	else if( pObj->constraint.distance > 1 || pObj->constraint.maxValue > 65535 )
	{
		tRet = LengthEncoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, pObj->length);
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	if( pObj->length == 0 ) return HS_OK;


	/* data encoding */
	if( pStrm->isAlign ) encodingBits = pObj->encodingBitsOfAlign;
	else				 encodingBits = pObj->encodingBitsOfUnalign;

	if(  ( encodingBits*(pObj->constraint.maxValue) )  >  16  )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

	if( pObj->reIndexing )
	{
		for( i=0; i<pObj->length; i++ )
		{
			for( j=0; j<pObj->fromSize; j++ )
			{
				if( pObj->fromConstraint[j] == pObj->value[i] )
				{
					Ushort2Uchar( encodingShort, &j );

#ifdef HS_CPLUSPLUS
					tRet = pStrm->BitsEncoding( encodingShort, encodingBits, e_alignRight );
#else
					tRet = AsnStream_BitsEncoding( pStrm, encodingShort, encodingBits, e_alignRight );
#endif

					if( tRet != HS_OK )
					{
#ifdef HS_DEBUG_ASN
						HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
						return tRet;
					}
				}
			}

			/* Cant found fromConstraint */
			if( j == pObj->fromSize )
			{
				j = 0;
				Ushort2Uchar( encodingShort, &j );

#ifdef HS_CPLUSPLUS
				tRet = pStrm->BitsEncoding( encodingShort, encodingBits, e_alignRight );
#else
				tRet = AsnStream_BitsEncoding( pStrm, encodingShort, encodingBits, e_alignRight );
#endif

				if( tRet != HS_OK )
				{
#ifdef HS_DEBUG_ASN
					HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
					return tRet;
				}
			}

		}
	}
	else
	{
		for( i=0; i<pObj->length; i++ )
		{
			Ushort2Uchar( encodingShort, &(pObj->value[i]) );

#ifdef HS_CPLUSPLUS
			tRet = pStrm->BitsEncoding( encodingShort, encodingBits, e_alignRight );
#else
			tRet = AsnStream_BitsEncoding( pStrm, encodingShort, encodingBits, e_alignRight );
#endif

			if( tRet != HS_OK )
			{
#ifdef HS_DEBUG_ASN
				HSPrint( "\n Error on AsnDoubleString::Encode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
				return tRet;
			}
		}
	}

	return HS_OK;
}


/***************************/
/* AsnDoubleString Decode */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnDoubleString::Decode( AsnStream *pStrm )
#else
HS_RESULT AsnDoubleString_Decode( void *pObject, AsnStream *pStrm )
#endif
{
	HS_RESULT			tRet;
	HS_UINT				i, encodingBits;
	unsigned char		decodingShort[2];
	unsigned short		tShort;

#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif/*HS_DEBUG_ASN*/
		return HS_ERR_NULL_PARAM;
	}


	if( pObj->constraint.type == e_constrainedExt )
	{
#ifdef HS_CPLUSPLUS
		pStrm->BitDecoding( &(pObj->extended) );
#else
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );
#endif
	}

	/* length encoding */
	if( pObj->extended )
	{
		tRet = SemiConstrainedNumberDecoding(pStrm, 0, &(pObj->length));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}
	else if( pObj->constraint.distance > 1 || pObj->constraint.maxValue > 65535 )
	{
		tRet = LengthDecoding(pStrm, pObj->constraint.minValue, pObj->constraint.maxValue, &(pObj->length));
		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}
	}

	if( pObj->constraint.distance < 2 ) pObj->length = pObj->constraint.minValue;

	/* value field reset */
	if( pObj->value != HS_NULL ) HSFree( pObj->value );
	if( pObj->length == 0 ) 
	{
		pObj->value = HS_NULL;
		return HS_OK;
	}
	pObj->value = (unsigned short*)HSMalloc( (pObj->length)*2 );
	memset(pObj->value, 0, (pObj->length*2) );

	/* data encoding */
	if( pStrm->isAlign ) encodingBits = pObj->encodingBitsOfAlign;
	else				 encodingBits = pObj->encodingBitsOfUnalign;

	if(  ( encodingBits*(pObj->constraint.maxValue) )  >  16  )
	{
#ifdef HS_CPLUSPLUS
		pStrm->Aligning();
#else
		AsnStream_Aligning( pStrm );
#endif
	}

	for( i=0; i<pObj->length; i++ )
	{
		memset( decodingShort, 0 , 2 );

#ifdef HS_CPLUSPLUS
		tRet = pStrm->BitsDecoding( decodingShort, encodingBits, e_alignRight );
#else
		tRet = AsnStream_BitsDecoding( pStrm, decodingShort, encodingBits, e_alignRight );
#endif

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnDoubleString::Decode(pObj,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif/*HS_DEBUG_ASN*/
			return tRet;
		}

		Uchar2Ushort( &tShort, decodingShort );

		if( pObj->reIndexing )
		{
			if( tShort > pObj->fromSize - 1 )
				pObj->value[i] = pObj->fromConstraint[0];
			else
				pObj->value[i] = pObj->fromConstraint[ tShort ];
		}
		else
			pObj->value[i] = tShort;
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/*************************/
/* AsnDoubleString Print */
#ifdef HS_CPLUSPLUS
HS_RESULT AsnDoubleString::Print( HS_UINT pDepth, char *pTypeName )
#else
HS_RESULT AsnDoubleString_Print( void *pObject, HS_UINT pDepth, char *pTypeName )
#endif
{
	HS_UINT i;
	
#ifdef HS_CPLUSPLUS
	AsnDoubleString *pObj = this;
#else
	AsnDoubleString *pObj = (AsnDoubleString *)pObject;
#endif

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnDoubleString::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );

	if( pTypeName != HS_NULL )
	{
		switch( pObj->tag )
		{
			case e_asnTagBMPString:		HSPrint( "%s<BMPString> "	, pTypeName );		break;
			default:					HSPrint( "%s<DoubleString> "	, pTypeName );		break;
		}
	}
	else
	{
		switch( pObj->tag )
		{
			case e_asnTagBMPString:		HSPrint( "<BMPString> " );			break;
			default:					HSPrint( "<DoubleString> " );		break;
		}
	}

	for( i=0; i<pObj->length; i++ )
	{
		if( pObj->value[i] < 32 || pObj->value[i] > 126 ) HSPrint( "\\%d", pObj->value[i] );
		else HSPrint( "%c", (char)(pObj->value[i]) );
	}

	return HS_OK;
}
#endif






#ifndef HS_CPLUSPLUS
/***********************************************************************************************************/
/* SequenceOf Class
*/
/*****************************/
/* AsnSequenceOf Constructor */
HS_RESULT new_AsnSequenceOf(
	void *pObject,
	ConstraintType pType,
	int pMinValue,
	HS_UINT pMaxValue,
	int pExtMinValue,
	HS_UINT pExtMaxValue
)
{
	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::AsnSequenceOf(NULL,%d,%d,%u,%d,%u) -> null parameter",
			(int)pType, pMinValue, pMaxValue, pExtMinValue, pExtMaxValue
		);
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->mold = HS_NULL;

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;

	pObj->constraint.type = pType;
	pObj->constraint.minValue = pMinValue;
	pObj->constraint.maxValue = pMaxValue;
	pObj->constraint.distance = pMaxValue - pMinValue + 1;
	if( pObj->constraint.distance == 0 ) pObj->constraint.distance = HS_ASN_MAX_CONSTRAINT;

	pObj->extMinValue = pExtMinValue;
	pObj->extMaxValue = pExtMaxValue;
	pObj->extended = FALSE;

	return HS_OK;
}


/****************************/
/* AsnSequenceOf Destructor */
HS_RESULT delete_AsnSequenceOf(void *pObject)
{
	HS_RESULT tRet;
	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::~AsnSequenceOf(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}

	tRet = AsnSequenceOf_Clear(pObj);

	if( pObj->mold != HS_NULL )
	{
		delete_AsnObject( pObj->mold );
		HSFree( pObj->mold );
		pObj->mold = HS_NULL;
	}

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::~AsnSequenceOf(Object) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}

	return HS_OK;
}


/*************************/
/* AsnSequenceOf SetMold */
HS_RESULT AsnSequenceOf_SetMold(void *pObject, AsnObject *pMold)
{
	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL || pMold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetMold(%x,%s) -> null parameter", pObj, pMold );
#endif
		return HS_ERR_NULL_PARAM;
	}

	pObj->mold = pMold;
	return HS_OK;
}


/***********************/
/* AsnSequenceOf Clear */
HS_RESULT AsnSequenceOf_Clear(void *pObject)
{
	HS_UINT	i;
	UnitOf *rounder = HS_NULL;
	UnitOf *deleter = HS_NULL;

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(NULL) -> null parameter" );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	
	if( pObj->size == 0 )
	{
		if( pObj->units.unit != HS_NULL || pObj->units.next != HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		else
			return HS_OK;
	}

	rounder = &(pObj->units);

	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		if( rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_ERR_ASN_CONFLICT;
		}

		pObj->mold->DeleteFunction( rounder->unit );
		HSFree(rounder->unit);

		deleter = rounder;
		rounder = (UnitOf *)rounder->next;

		if( i ) HSFree( deleter );
	}

	if( rounder != HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Clear(Object) -> conflict size, line(%d)", __LINE__ );
#endif
		return HS_ERR_ASN_CONFLICT;
	}

	pObj->units.unit = HS_NULL;
	pObj->units.next = HS_NULL;
	pObj->size = 0;
	return HS_OK;
}


/*************************/
/* AsnSequenceOf GetSize */
HS_UINT AsnSequenceOf_GetSize(void *pObject )
{
	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetSize(NULL) -> null parameter" );
#endif
		return 0;
	}

	return pObj->size;
}


/*************************/
/* AsnSequenceOf SetSize */
HS_RESULT AsnSequenceOf_SetSize(void *pObject, HS_UINT pSize)
{

	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetSize(NULL,%u) -> null parameter", pSize );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::SetSize(Object,%u) -> no mold on 'sequence of'", pSize );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	pObj->extended = FALSE;
	if( pObj->constraint.type == e_semiConstrained )
	{
		if( (int)pSize < pObj->constraint.minValue )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> size is too small, minValue(%d)", pSize, pObj->constraint.minValue );
#endif
			return HS_ERR_ASN_OUTOF_RANGE;
		}
	}
	else if( pObj->constraint.type == e_constrained )
	{
		if( (int)pSize < pObj->constraint.minValue || pSize > pObj->constraint.maxValue )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> asn size conflict, minValue(%d), maxValue(%u)", pSize, pObj->constraint.minValue, pObj->constraint.maxValue );
#endif
			return HS_ERR_ASN_OUTOF_RANGE;
		}
	}
	else if( pObj->constraint.type == e_constrainedExt )
	{
		if( (int)pSize < pObj->constraint.minValue || pSize > pObj->constraint.maxValue ) pObj->extended = TRUE;
	}

	
	tRet = AsnSequenceOf_Clear(pObj);

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::SetSize(Object,%u) -> %d, line(%d)", pSize, __LINE__ );
#endif
		return tRet;
	}

	pObj->size = pSize;
	if( pSize == 0 ) return HS_OK;


	rounder = &(pObj->units);
	rounder->unit = (void*)HSMalloc( pObj->mold->size );
	memcpy( rounder->unit, pObj->mold->obj, pObj->mold->size );
	if( pObj->mold->MakeMoldFunction != HS_NULL ) pObj->mold->MakeMoldFunction( rounder->unit );
	rounder->next = HS_NULL;

	if( pSize == 1 ) return HS_OK;


	for( i=1; i<pSize; i++ )
	{
		UnitOf *maker = (UnitOf*)HSMalloc( sizeof(UnitOf) );
		maker->unit = (void*)HSMalloc( pObj->mold->size );
		memcpy( maker->unit, pObj->mold->obj, pObj->mold->size );
		if( pObj->mold->MakeMoldFunction != HS_NULL ) pObj->mold->MakeMoldFunction( maker->unit );
		maker->next = HS_NULL;

		rounder->next = maker;
		rounder = (UnitOf*)rounder->next;
	}

	return HS_OK;
}


/*************************/
/* AsnSequenceOf GetUnit */
void *AsnSequenceOf_GetUnit(void *pObject, HS_UINT pIndex)
{
	HS_UINT i;
	UnitOf *rounder = HS_NULL;

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetUnit(NULL,%u) -> null parameter", pIndex );
#endif
		return HS_NULL;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::GetUnit(Object,%u) -> no mold on 'sequence of'", pIndex );
#endif
		return HS_NULL;
	}
	if( pIndex > pObj->size - 1 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::GetUnit(Object,%u) -> index is too big, size(%u)", pIndex, pObj->size );
#endif
		return HS_NULL;
	}

	rounder = &(pObj->units);
	for( i=0; i<pIndex; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::GetUnit(Object) -> conflict size, line(%d)", __LINE__ );
#endif
			return HS_NULL;
		}

		rounder = (UnitOf*)rounder->next;
	}

	if( rounder == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::GetUnit(Object) -> conflict size, line(%d)", __LINE__ );
#endif
		return HS_NULL;
	}

	return rounder->unit;
}


/************************/
/* AsnSequenceOf Encode */
HS_RESULT AsnSequenceOf_Encode(void *pObject, AsnStream *pStrm)
{
	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}
	if( pObj->size == 0 && pObj->constraint.minValue > 0 )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> size is too small, size(0), minValue(%d), constraintType(%d)",
			pObj->constraint.minValue, (int)pObj->constraint.type
		);
#endif
		return HS_ERR_ASN_OUTOF_RANGE;
	}

	if( pObj->constraint.type == e_constrainedExt )
		AsnStream_BitEncoding(pStrm,pObj->extended);

	if( pObj->extended )
		tRet = LengthEncoding(pStrm,0,HS_ASN_MAX_CONSTRAINT,pObj->size);
	else
		tRet = LengthEncoding(pStrm,pObj->constraint.minValue,pObj->constraint.maxValue,pObj->size);

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}

	AsnStream_Aligning(pStrm);

	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if( rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_ERR_CONFLICT;
		}
		if( rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Encode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_ERR_CONFLICT;
		}

		tRet = pObj->mold->EncodeFunction(rounder->unit,pStrm);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error on AsnSequenceOf::Encode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
			return tRet;
		}

		/*Mis-Standard:ASN{
		AsnStream_Aligning(pStrm);
		/*}*/

		rounder = (UnitOf*)rounder->next;
	}

	return HS_OK;
}


/************************/
/* AsnSequenceOf Decode */
HS_RESULT AsnSequenceOf_Decode(void *pObject, AsnStream *pStrm)
{
	HS_RESULT	tRet;
	HS_UINT		tSize, i;
	UnitOf		*rounder = HS_NULL;

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL || pStrm == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(%x,%x) -> null parameter", pObj, pStrm );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> no mold on 'sequence of'" );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	if( pObj->constraint.type == e_constrainedExt )
		AsnStream_BitDecoding( pStrm, &(pObj->extended) );

	if( pObj->extended )
		tRet = LengthDecoding(pStrm,0,HS_ASN_MAX_CONSTRAINT, &(tSize) );
	else
		tRet = LengthDecoding(pStrm,pObj->constraint.minValue,pObj->constraint.maxValue, &(tSize) );

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}

	AsnStream_Aligning(pStrm);

	tRet = AsnSequenceOf_SetSize(pObj,tSize);

	if( tRet != HS_OK )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
		return tRet;
	}


	rounder = &(pObj->units);
	for( i=0; i<tSize; i++ )
	{
		if(rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_ERR_CONFLICT;
		}
		if(rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> asn size conflict, line(%d)", __LINE__ );
#endif
			return HS_ERR_CONFLICT;
		}

		tRet = pObj->mold->DecodeFunction(rounder->unit, pStrm);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Decode(Object,pStrm) -> %d, line(%d)", tRet, __LINE__ );
#endif
			return HS_ERR_CONFLICT;
		}

		/*Mis-Standard:ASN{
		AsnStream_Aligning(pStrm);
		/*}*/

		rounder = (UnitOf *)rounder->next;
	}

	return HS_OK;
}


#ifdef HS_DEBUG_ASN_PRINT
/***********************/
/* AsnSequenceOf Print */
HS_RESULT AsnSequenceOf_Print(void *pObject, HS_UINT pDepth, char *pTypeName)
{
	HS_RESULT	tRet;
	HS_UINT		i;
	UnitOf		*rounder = HS_NULL;
	char		indexString[16];

	AsnSequenceOf *pObj = (AsnSequenceOf *)pObject;

	if( pObj == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error on AsnSequenceOf::Print(NULL,%u,%s) -> null parameter", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_NULL_PARAM;
	}
	if( pObj->mold == HS_NULL )
	{
#ifdef HS_DEBUG_ASN
		HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> no mold on 'sequence of'", pDepth, (pTypeName==HS_NULL)? "":pTypeName );
#endif
		return HS_ERR_ASN_NO_MOLD;
	}

	HSPrint("\n");
	for( i=0; i<pDepth; i++ ) HSPrint( "  " );


	if( pTypeName != HS_NULL )
		HSPrint( "%s<SEQUENCE OF> = size[%u]", pTypeName, pObj->size );
	else
		HSPrint( "<SEQUENCE OF> = size[%u]", pObj->size );


	rounder = &(pObj->units);
	for( i=0; i<pObj->size; i++ )
	{
		if(rounder == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> asn size conflict, line(%d)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, __LINE__
			);
#endif
			return HS_ERR_ASN_CONFLICT;
		}
		if(rounder->unit == HS_NULL )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> asn size conflict, line(%d)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, __LINE__
			);
#endif
			return HS_ERR_ASN_CONFLICT;
		}

		sprintf( indexString, "[%u] ", i );
		tRet = pObj->mold->PrintFunction(rounder->unit,pDepth+1,indexString);

		if( tRet != HS_OK )
		{
#ifdef HS_DEBUG_ASN
			HSPrint( "\n Error AsnSequenceOf::Print(Object,%u,%s) -> %d, line(%d), index(%u)",
				pDepth, (pTypeName==HS_NULL)? "":pTypeName, tRet, __LINE__, i
			);
#endif
			return HS_ERR_CONFLICT;
		}

		rounder = (UnitOf *)rounder->next;
	}

	return HS_OK;
}
#endif/*HS_DEBUG_ASN_PRINT*/
#endif/*!HS_CPLUSPLUS*/




void PrintNoUseEncode( const char *myName )
{
#ifdef HS_DEBUG_ASN
	HSPrint( "\n Error on %s::Encode(pObj,pStrm) -> not supported yet", myName );
#endif
}
void PrintNoUseDecode( const char *myName )
{
#ifdef HS_DEBUG_ASN
	HSPrint( "\n Error on %s::Decode(pObj,pStrm) -> not supported yet", myName );
#endif
}


