<?php
	/**
	 * Class to hold various commonly used functions
	 *
	 * $Id: Misc.php,v 1.137.2.2 2007/07/09 15:23:56 soranzo Exp $
	 */
	 
	class Misc {
		// Tracking string to include in HREFs 
		var $href;
		// Tracking string to include in forms
		var $form;
		
		/* Constructor */
		function Misc() {
		}

		/** 
		 * Checks if dumps are properly set up
		 * @param $all (optional) True to check pg_dumpall, false to just check pg_dump
		 * @return True, dumps are set up, false otherwise
		 */
		function isDumpEnabled($all = false) {
			$info = $this->getServerInfo();
			return !empty($info[$all ? 'pg_dumpall_path' : 'pg_dump_path']);
		}

		/**
		 * Sets the href tracking variable
		 */
		function setHREF() {
			$this->href = $this->getHREF();
		}
		
		/**
		 * Get a href query string, excluding objects below the given object type (inclusive)
		 */
		function getHREF($exclude_from = null) {
			$href = '';
			if (isset($_REQUEST['server']) && $exclude_from != 'server') {
				$href .= 'server=' . urlencode($_REQUEST['server']);
				if (isset($_REQUEST['database']) && $exclude_from != 'database') {
					$href .= '&amp;database=' . urlencode($_REQUEST['database']);
					if (isset($_REQUEST['schema']) && $exclude_from != 'schema') {
						$href .= '&amp;schema=' . urlencode($_REQUEST['schema']);
					}
				}
			}
			return $href;
		}

		/**
		 * Sets the form tracking variable
		 */
		function setForm() {
			$this->form = '';
			if (isset($_REQUEST['server'])) {
				$this->form .= "<input type=\"hidden\" name=\"server\" value=\"" . htmlspecialchars($_REQUEST['server']) . "\" />\n";
				if (isset($_REQUEST['database'])) {
					$this->form .= "<input type=\"hidden\" name=\"database\" value=\"" . htmlspecialchars($_REQUEST['database']) . "\" />\n";
					if (isset($_REQUEST['schema'])) {
						$this->form .= "<input type=\"hidden\" name=\"schema\" value=\"" . htmlspecialchars($_REQUEST['schema']) . "\" />\n";
					}
				}
			}
		}

		function array_implode($array, $deli = ' ')
		{
			if (is_array ($array)) {
				$output = '';
				foreach ($array as $k => $v) {
					$output .= $this->array_implode($v, $deli). " ";
				}
				return $output;
			}
			else
				return $array;
		}
		
		/**
		 * Render a value into HTML using formatting rules specified
		 * by a type name and parameters.
		 *
		 * @param $str The string to change
		 *
		 * @param $type Field type (optional), this may be an internal PostgreSQL type, or:
		 *			yesno    - same as bool, but renders as 'Yes' or 'No'.
		 *			pre      - render in a <pre> block.
		 *			nbsp     - replace all spaces with &nbsp;'s
		 *			verbatim - render exactly as supplied, no escaping what-so-ever.
		 *			callback - render using a callback function supplied in the 'function' param.
		 *
		 * @param $params Type parameters (optional), known parameters:
		 *			null     - string to display if $str is null, or set to TRUE to use a default 'NULL' string,
		 *			           otherwise nothing is rendered.
		 *			clip     - if true, clip the value to a fixed length, and append an ellipsis...
		 *			cliplen  - the maximum length when clip is enabled (defaults to $conf['max_chars'])
		 *			ellipsis - the string to append to a clipped value (defaults to $lang['strellipsis'])
		 *			tag      - an HTML element name to surround the value.
		 *			class    - a class attribute to apply to any surrounding HTML element.
		 *			align    - an align attribute ('left','right','center' etc.)
		 *			true     - (type='bool') the representation of true.
		 *			false    - (type='bool') the representation of false.
		 *			function - (type='callback') a function name, accepts args ($str, $params) and returns a rendering.
		 *			lineno   - prefix each line with a line number.
		 *			map      - an associative array.
		 *
		 * @return The HTML rendered value
		 */
		function printVal($str, $type = null, $params = array()) {
			global $lang, $conf, $data;
			
			// Shortcircuit for a NULL value
			if (is_null($str))
				return isset($params['null'])
						? ($params['null'] === true ? '<i>NULL</i>' : $params['null'])
						: '';
			
			if (isset($params['map']) && isset($params['map'][$str])) $str = $params['map'][$str];
			
			// Clip the value if the 'clip' parameter is true.
			if (isset($params['clip']) && $params['clip'] === true) {
				$maxlen = isset($params['cliplen']) && is_integer($params['cliplen']) ? $params['cliplen'] : $conf['max_chars'];
				$ellipsis = isset($params['ellipsis']) ? $params['ellipsis'] : $lang['strellipsis'];
				if (strlen($str) > $maxlen) {
					$str = substr($str, 0, $maxlen-1) . $ellipsis;
				}
			}

			$out = '';
			
			if (!$type && is_array($str)) $type = "array";
			switch ($type) {
				case 'int2':
				case 'int4':
				case 'int8':
				case 'float4':
				case 'float8':
				case 'money':
				case 'numeric':
				case 'integer':
				case 'smallint':
				case 'monetary':
				case 'float':
				case 'double':
				case 'oid':
				case 'xid':
				case 'cid':
				case 'tid':
					$align = 'right';
					$out = nl2br(htmlspecialchars($str));
					break;
				case 'array':
				case 'sequence':
				case 'set':
				case 'multiset':
					$out = $this->array_implode($str);
					break;
				case 'yn':
					$out = ($str == 'y') ? "" : "";
					break;
				case 'yesno':
					if (!isset($params['true'])) $params['true'] = $lang['stryes'];
					if (!isset($params['false'])) $params['false'] = $lang['strno'];
					// No break - fall through to boolean case.
				case 'bool':
				case 'boolean':
					if (is_bool($str)) $str = $str ? 't' : 'f';
					switch ($str) {
						case 't':
							$out = (isset($params['true']) ? $params['true'] : $lang['strtrue']);
							$align = 'center';
							break;
						case 'f':
							$out = (isset($params['false']) ? $params['false'] : $lang['strfalse']);
							$align = 'center';
							break;
						default:
							$out = htmlspecialchars($str);
					}
					break;
				case 'bytea':
					$out = $data->escapeBytea($str);
					break;
				case 'pre':
					$tag = 'pre';
					$out = htmlspecialchars($str);
					break;
				case 'prenoescape':
					$tag = 'pre';
					$out = $str;
					break;
				case 'nbsp':
					$out = nl2br(str_replace(' ', '&nbsp;', htmlspecialchars($str)));
					break;
				case 'verbatim':
					$out = $str;
					break;
				case 'callback':
					$out = $params['function']($str, $params);
					break;
				case 'prettysize':
					$mult = 1;
					$limit = 10 * 1024;

					if ($str < $limit * $mult)
						$out = $str.' '.$lang['strbytes'];
					else
					{
						$mult *= 1024;
						if ($str < $limit * $mult)
							$out = floor(($str + $mult / 2) / $mult).' '.$lang['strkb'];
						else
						{ 							
							$mult *= 1024;
							if ($str < $limit * $mult)
								$out = floor(($str + $mult / 2) / $mult).' '.$lang['strmb'];
							else
							{ 							
								$mult *= 1024;
								if ($str < $limit * $mult)
									$out = floor(($str + $mult / 2) / $mult).' '.$lang['strgb'];
								else
								{ 							
									$mult *= 1024;
									if ($str < $limit * $mult)
										$out = floor(($str + $mult / 2) / $mult).' '.$lang['strtb'];
								}
							}
						}
					}
					break;
				default:
					// If the string contains at least one instance of >1 space in a row, a tab
					// character, a space at the start of a line, or a space at the start of
					// the whole string then render within a pre-formatted element (<pre>).
					if (preg_match('/(^ |  |\t|\n )/m', $str)) {
						$tag = 'pre';
						$class = 'data';
						$out = htmlspecialchars($str);
					} else {
						$out = nl2br(htmlspecialchars($str));
					}
			}
			
			if (isset($params['class'])) $class = $params['class'];
			if (isset($params['align'])) $align = $params['align'];
			
			if (!isset($tag) && (isset($class) || isset($align))) $tag = 'div';
			
			if (isset($tag)) {
				$alignattr = isset($align) ? " align=\"{$align}\"" : '';
				$classattr = isset($class) ? " class=\"{$class}\"" : '';				
				$out = "<{$tag}{$alignattr}{$classattr}>{$out}</{$tag}>";
			}

			// Add line numbers if 'lineno' param is true
			if (isset($params['lineno']) && $params['lineno'] === true) {
				$lines = explode("\n", $str);
				$num = count($lines);
				if ($num > 0) {
					$temp = "<table>\n<tr><td class=\"{$class}\" style=\"vertical-align: top; padding-right: 10px;\"><pre class=\"{$class}\">";
					for ($i = 1; $i <= $num; $i++) {
						$temp .= $i . "\n";
					}
					$temp .= "</pre></td><td class=\"{$class}\" style=\"vertical-align: top;\">{$out}</td></tr></table>\n";
					$out = $temp;
				}
				unset($lines);
			}

			return $out;
		}

		/**
		 * A function to recursively strip slashes.  Used to
		 * enforce magic_quotes_gpc being off.
		 * @param &var The variable to strip
		 */
		function stripVar(&$var) {
			if (is_array($var)) {
				foreach($var as $k => $v) {
					$this->stripVar($var[$k]);
				}		
			}
			else
				$var = stripslashes($var);	
		}
		
		/**
		 * Print out the page heading and help link
		 * @param $title Title, already escaped
		 * @param $help (optional) The identifier for the help link
		 */
		function printTitle($title, $help = null) {
			global $data, $lang;
			
			echo "<h2>";
			$this->printHelp($title, null);
			echo "</h2>\n";
		}
		
		/**
		 * Print out a message
		 * @param $msg The message to print
		 */
		function printMsg($msg) {
			if ($msg != '') echo "<p class=\"message\">{$msg}</p>\n";
		}

		/**
		 * Prints the page header.  If global variable $_no_output is
		 * set then no header is drawn.
		 * @param $title The title of the page
		 * @param $script script tag
		 */
		function printHeader($title = '', $script = null) {
			global $appName, $lang, $_no_output, $conf;

			if (!isset($_no_output)) {
				header("Content-Type: text/html; charset=" . $lang['appcharset']);
				// Send XHTML headers, or regular HTML headers
				if (isset($conf['use_xhtml']) && $conf['use_xhtml']) {
					echo "<?xml version=\"1.0\" encoding=\"", htmlspecialchars($lang['appcharset']), "\"?>\n";
					echo "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-Transitional.dtd\">\n";
					echo "<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"{$lang['applocale']}\" lang=\"{$lang['applocale']}\"";
					if (strcasecmp($lang['applangdir'], 'ltr') != 0) echo " dir=\"", htmlspecialchars($lang['applangdir']), "\"";
					echo ">\n";
				} else {
					echo "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">\n";
					echo "<html lang=\"{$lang['applocale']}\"";
					if (strcasecmp($lang['applangdir'], 'ltr') != 0) echo " dir=\"", htmlspecialchars($lang['applangdir']), "\"";
					echo ">\n";
				}
				echo "<head>\n";
				echo "<title>", htmlspecialchars($appName);
				if ($title != '') echo " - {$title}";
				echo "</title>\n";
				echo "<meta http-equiv=\"Content-Type\" content=\"text/html; charset={$lang['appcharset']}\" />\n";
				
				// Theme
				echo "<link rel=\"stylesheet\" href=\"themes/{$conf['theme']}/global.css\" type=\"text/css\" />\n";
				if ($script) echo "{$script}\n";
				echo "</head>\n";
			}
		}

		/**
		 * Prints the page footer
		 * @param $doBody True to output body tag, false otherwise
		 */
		function printFooter($doBody = true) {
			global $_reload_browser, $_reload_drop_database;

			if ($doBody) {
				if (isset($_reload_browser)) $this->printReload(false);
				elseif (isset($_reload_drop_database)) $this->printReload(true);
				echo "</body>\n";
			}
			echo "</html>\n";
		}

		/**
		 * Prints the page body.
		 * @param $doBody True to output body tag, false otherwise
		 * @param $bodyClass - name of body class
		 */
		function printBody($bodyClass = '', $doBody = true ) {
			global $_no_output;			

			if (!isset($_no_output)) {
				if ($doBody) {
					$bodyClass = htmlspecialchars($bodyClass);
					echo "<body", ($bodyClass == '' ? '' : " class=\"{$bodyClass}\"");
					echo ">\n";
				}
			}
		}

		/**
		 * Outputs JavaScript code that will reload the browser
		 * @param $database True if dropping a database, false otherwise
		 */
		function printReload($database) {
			echo "<script type=\"text/javascript\">\n";
			if ($database)
				echo "\tparent.frames.browser.location.href=\"browser.php\";\n";
			else
				echo "\tparent.frames.browser.location.reload();\n";
			echo "</script>\n";
		}

		/**
		 * Display navigation tabs
		 * @param $tabs An associative array of tabs definitions, see printNav() for an example.
		 * @param $activetab The name of the tab to be highlighted.
		 */
		function printTabs($tabs, $activetab) {
			global $misc, $conf, $data, $lang;
			
			if (is_string($tabs)) {
				$_SESSION['webdbLastTab'][$tabs] = $activetab;
				$tabs = $this->getNavTabs($tabs);
			}
			
			echo "<table class=\"tabs\"><tr>\n";
			#echo "<div class=\"tabs\">\n";
			
			# FIXME: don't count hidden tags
			$width = round(100 / count($tabs)).'%';
			
			foreach ($tabs as $tab_id => $tab) {
				$active = ($tab_id == $activetab) ? ' active' : '';
				
				if (!isset($tab['hide']) || $tab['hide'] !== true) {
					if ($tab['title'] == 'SQL') {
						$sqllink = "target=\"sqledit\" onclick=\"window.open('".$this->printActionUrl($tab, $_REQUEST).
								   "','sqledit','toolbar=no,width=800,height=600,resizable=yes,scrollbars=yes').focus(); return false;\"";
					}	
					else $sqllink = "";
					$tablink = "<a" . $this->printActionUrl($tab, $_REQUEST, 'href') . $sqllink. ">";
					
					if (isset($tab['icon']) && $icon = $this->icon($tab['icon']))
						$tablink .= "<span class=\"icon\"><img src=\"{$icon}\" alt=\"{$tab['title']}\" /></span>";
					
					$tablink .= "<span class=\"label\">{$tab['title']}</span></a>";
					
					echo "<td width=\"{$width}\" class=\"tab{$active}\">";
					#echo "<span class=\"tab{$active}\" style=\"white-space:nowrap;\">";
					
					echo $tablink;
					
					echo "</td>\n";
					#echo "</span>\n";
				}
			}
			
			echo "</tr></table>\n";
			#echo "</div>\n";
		}

		/**
		 * Retrieve the tab info for a specific tab bar.
		 * @param $section The name of the tab bar.
		 */
		function getNavTabs($section) {
			global $data, $lang, $conf, $db, $_server_info;

			switch ($section) {
				case 'root':
					return array (
						'intro' => array (
							'title' => $lang['strintroduction'],
							'url'   => "intro.php",
							'icon'  => 'Introduction',
						),
						'servers' => array (
							'title' => $lang['strservers'],
							'url'   => "servers.php",
							'icon'  => 'Servers',
						),
					);

				case 'server':
					$server_info = $this->getServerInfo();
					$tmp = array (
						'databases' => array (
							'title' => $lang['strdatabases'],
							'url'   => 'all_db.php',
							'urlvars' => array('subject' => 'server'),
							'icon'  => 'Databases',
						));
					if ($_server_info['cassuper']!="none")
					  $tmp = array_merge( $tmp, array(
						'broker' => array (
							'title' => $lang['strbroker'],
							'url'   => 'all_broker.php',
							'urlvars' => array('subject' => 'server'),
							'icon'  => 'Broker',
						)));
					$tmp = array_merge( $tmp, array(
						'account' => array (
							'title' => $lang['straccount'],
							'url'   => 'adminusers.php', // admin̸  . ƴϸ йȣ .
							'urlvars' => array('subject' => 'server', 'action' => 'account'),
							'icon'  => 'User',
						)
					));
					//@@@TODO 1.2 broker log & dbmt user log
					return $tmp;
					break;

				case 'database':
					$tabs = array (
						'database' => array (
							'title' => $lang['strdatabase'],
							'url'   => 'database.php',
							'urlvars' => array('subject' => 'database'),
							'tree'  => false,
							'icon'  => 'Database',
						)
					);
					if ($_server_info['dbstat'][$db]=="start") {
						$tabs = array_merge($tabs, array(
						'tables' => array (
							'title' => $lang['strtables'],
							'url'   => 'tables.php',
							'urlvars' => array('subject' => 'database'),
							'icon'  => 'Tables',
						),
						'users' => array (
							'title' => $lang['strusers'],
							'url'   => 'users.php',
							'urlvars' => array('subject' => 'database'),
							'tree'  => false,
							'icon'  => 'User',
						)));
						if ($_server_info['broker']=="ON" &&
						    isset($_server_info['sqlport'])) {
							$tabs = array_merge($tabs, array(
								'sql' => array (
									'title' => $lang['strsql'],
									'url'   => 'sqledit.php',
									'urlvars' => array('subject' => 'database', 'action' => 'sql'),
									'tree'  => false,
									'icon'  => 'SqlEditor'
								)
							));
						}
					}
					return $tabs;

				case 'broker':
					return array (
						'appserver' => array (
							'title' => $lang['strappserver'],
							'url'   => 'broker.php',
							'urlvars' => array('subject' => 'broker', 'broker' => field('broker')),
							'icon'  => 'Broker',
						)
						//@@@TODO Logs(access/error/script)
					);

				case 'table':
					return array (
						'columns' => array (
							'title' => $lang['strcolumns'],
							'url'   => 'tblproperties.php',
							'urlvars' => array('subject' => 'table', 'table' => field('table')),
							'icon'  => 'Columns',
						),
						'indexes' => array (
							'title' => $lang['strindexes'],
							'url'   => 'indexes.php',
							'urlvars' => array('subject' => 'table', 'table' => field('table')),
							'icon'  => 'Indexes',
						)
					);
				
				case 'popup':
					return array (
						'sql' => array (
							'title' => $lang['strsql'],
							'url'   => 'sqledit.php',
							'urlvars' => array('subject' => 'schema', 'action' => 'sql'),
							'icon'  => 'SqlEditor',
						),
					);
					
				default:
					return array();
			}
		}

		/**
		 * Get the URL for the last active tab of a particular tab bar.
		 */
		function getLastTabURL($section) {
			global $data;

			$tabs = $this->getNavTabs($section);
		
			if (isset($_SESSION['webdbLastTab'][$section]) && isset($tabs[$_SESSION['webdbLastTab'][$section]]))
				$tab = $tabs[$_SESSION['webdbLastTab'][$section]];
			else
				$tab = reset($tabs);
			
			return isset($tab['url']) ? $tab['url'] : null;
		}

		function printTopbar() {
			global $lang, $conf, $appName, $appVersion, $appLangFiles;
			
			$server_info = $this->getServerInfo();
			
			echo "<div class=\"topbar\"><table width=\"100%\"><tr><td>";
			
			if ($server_info && isset($server_info['platform']) && isset($server_info['username'])) {
				echo sprintf($lang['strtopbar'],
					'<span class="platform">'.htmlspecialchars($server_info['platform']).'</span>',
					'<span class="host">'.htmlspecialchars((empty($server_info['host'])) ? 'localhost':$server_info['host']).'</span>',
					'<span class="port">'.htmlspecialchars($server_info['port']).'</span>',
					'<span class="username">'.htmlspecialchars($server_info['username']).'</span>',
					'<span class="date">'.date($lang['strtimefmt']).'</span>');
			} else {
				echo "<span class=\"appname\">$appName</span> <span class=\"version\">$appVersion</span>";
			}
			
			echo "</td>";

			if (isset($_REQUEST['server'])) {
				echo "<td align=\"right\">";
 				echo "<a class=\"toplink\" href=\"servers.php?action=logout&amp;logoutServer=".htmlspecialchars($server_info['host']).":".htmlspecialchars($server_info['port'])."\">{$lang['strlogout']}</a>";
				echo "</td>";
			}
/*
			echo "<td align=\"right\" width=\"1%\">";
			
			echo "<form method=\"get\"><select name=\"language\" onchange=\"this.form.submit()\">\n";
			$language = isset($_SESSION['webdbLanguage']) ? $_SESSION['webdbLanguage'] : 'english';
			foreach ($appLangFiles as $k => $v) {
				echo "<option value=\"{$k}\"",
					($k == $language) ? ' selected="selected"' : '',
					">{$v}</option>\n";
			}
			echo "</select>\n";
			echo "<noscript><input type=\"submit\" value=\"Set Language\"></noscript>\n";
			foreach ($_GET as $key => $val) {
				if ($key == 'language') continue;
				echo "<input type=\"hidden\" name=\"$key\" value=\"", htmlspecialchars($val), "\" />\n";
			}
			echo "</form>\n";
			
			echo "</td>";
*/
			echo "</tr></table></div>\n";
		}
		
		/**
		 * Display a bread crumb trail.
		 */
		function printTrail($trail = array()) {
			global $lang;
			
			$this->printTopbar();
			
			if (is_string($trail)) {
				$trail = $this->getTrail($trail);
			}
			
			echo "<div class=\"trail\"><table><tr>";
			
			foreach ($trail as $crumb) {
				echo "<td class=\"crumb\">";
				$crumblink = "<a";
				
				if (isset($crumb['url']))
					$crumblink .= ' href="' . $this->printVal($crumb['url'], 'nbsp') . '"';
				
				if (isset($crumb['title']))
					$crumblink .= " title=\"{$crumb['title']}\"";
				
				$crumblink .= ">";
				
				if (isset($crumb['title']))
					$iconalt = $crumb['title'];
				else
					$iconalt = 'Database Root';

				if (isset($crumb['icon']) && $icon = $this->icon($crumb['icon']))
					$crumblink .= "<span class=\"icon\"><img src=\"{$icon}\" alt=\"{$iconalt}\" /></span>";
				 
				$crumblink .= "<span class=\"label\">" . htmlspecialchars($crumb['text']) . "</span></a>";
				
				echo $crumblink;
				
				echo "{$lang['strseparator']}";
				echo "</td>";
			}
			
			echo "</tr></table></div>\n";
		}

		/**
		 * Create a bread crumb trail of the object hierarchy.
		 * @param $object The type of object at the end of the trail.
		 */
		function getTrail($subject = null) {
			global $lang, $conf, $data, $appName;

			$trail = array();
			$vars = '';
			$done = false;
			
			$trail['root'] = array(
				'text'  => $appName,
				'url'   => 'redirect.php?subject=root',
				'icon'  => 'Introduction'
			);
			
			if ($subject == 'root') $done = true;
			
			if (!$done) {
				$vars = 'server='.urlencode($_REQUEST['server']).'&';
				$server_info = $this->getServerInfo();
				$trail['server'] = array(
					'title' => $lang['strserver'],
					'text'  => $server_info['desc'],
					'url'   => "redirect.php?subject=server&{$vars}",
					'icon'  => 'Server'
				);
			}
			if ($subject == 'server') $done = true;
			
			if (isset($_REQUEST['database']) && !$done) {
				$vars .= 'database='.urlencode($_REQUEST['database']).'&';
				$trail['database'] = array(
					'title' => $lang['strdatabase'],
					'text'  => $_REQUEST['database'],
					'url'   => "redirect.php?subject=database&{$vars}",
					'icon'  => 'Database'
				);
			}
			if ($subject == 'database') $done = true;
			
			if (isset($_REQUEST['table']) && !$done) {
				$vars .= "subject=table&table=".urlencode($_REQUEST['table']);
				$trail['table'] = array(
					'title' => $lang['strtable'],
					'text'  => $_REQUEST['table'],
					'url'   => "redirect.php?{$vars}",
					'icon'  => 'Table'
				);
			}
			if ($subject == 'table') $done = true;

			if (isset($_REQUEST['broker']) && !$done) {
				$vars .= "subject=broker&broker=".urlencode($_REQUEST['broker']);
				$trail['broker'] = array(
					'title' => $lang['strbroker'],
					'text'  => $_REQUEST['broker'],
					'url'   => "redirect.php?{$vars}",
					'icon'  => 'Broker'
				);
			}
			if ($subject == 'table') $done = true;
			
			return $trail;
		}

		/**
		 * Do multi-page navigation.  Displays the prev, next and page options.
		 * @param $page the page currently viewed
		 * @param $pages the maximum number of pages
		 * @param $url the url to refer to with the page number inserted
		 * @param $max_width the number of pages to make available at any one time (default = 20)
		 */
		function printPages($page, $pages, $url, $max_width = 20) {
			global $lang;

			$window = 10;

			if ($page < 0 || $page > $pages) return;
			if ($pages < 0) return;
			if ($max_width <= 0) return;

			if ($pages > 1) {
				echo "<center><p>\n";
				if ($page != 1) {
					$temp = str_replace('%s', 1, $url);
					echo "<a class=\"pagenav\" href=\"{$temp}\">{$lang['strfirst']}</a>\n";
					$temp = str_replace('%s', $page - 1, $url);
					echo "<a class=\"pagenav\" href=\"{$temp}\">{$lang['strprev']}</a>\n";
				}
				
				if ($page <= $window) { 
					$min_page = 1; 
					$max_page = min(2 * $window, $pages); 
				}
				elseif ($page > $window && $pages >= $page + $window) { 
					$min_page = ($page - $window) + 1; 
					$max_page = $page + $window; 
				}
				else { 
					$min_page = ($page - (2 * $window - ($pages - $page))) + 1; 
					$max_page = $pages; 
				}
				
				// Make sure min_page is always at least 1
				// and max_page is never greater than $pages
				$min_page = max($min_page, 1);
				$max_page = min($max_page, $pages);
				
				for ($i = $min_page; $i <= $max_page; $i++) {
					$temp = str_replace('%s', $i, $url);
					if ($i != $page) echo "<a class=\"pagenav\" href=\"{$temp}\">$i</a>\n";
					else echo "$i\n";
				}
				if ($page != $pages) {
					$temp = str_replace('%s', $page + 1, $url);
					echo "<a class=\"pagenav\" href=\"{$temp}\">{$lang['strnext']}</a>\n";
					$temp = str_replace('%s', $pages, $url);
					echo "<a class=\"pagenav\" href=\"{$temp}\">{$lang['strlast']}</a>\n";
				}
				echo "</p></center>\n";
			}
		}		

		/**
		 * Displays link to the context help.
		 * @param $str   - the string that the context help is related to (already escaped) 
		 * @param $help  - help section identifier
		 */
		function printHelp($str, $help) {
			global $lang, $data;
			
			echo $str;
			if ($help) {
				echo "<a class=\"help\" href=\"";
				echo htmlspecialchars("help.php?help=".urlencode($help)."&server=".urlencode($_REQUEST['server']));
				echo "\" title=\"{$lang['strhelp']}\" target=\"phppgadminhelp\">{$lang['strhelpicon']}</a>";
			}
		}
	
		/** 
		 * Outputs JavaScript to set default focus
		 * @param $object eg. forms[0].username
		 */
		function setFocus($object) {
			echo "<script type=\"text/javascript\">\n";
			echo "<!--\n";
			echo "   document.{$object}.focus();\n";
			echo "-->\n";
			echo "</script>\n";
		}
		
		/**
		 * Outputs JavaScript to set the name of the browser window.
		 * @param $name the window name
		 * @param $addServer if true (default) then the server id is
		 *        attached to the name.
		 */
		function setWindowName($name, $addServer = true) {
			echo "<script type=\"text/javascript\">\n<!--\n";
			echo "   window.name = '{$name}", ($addServer ? ':'.htmlspecialchars($_REQUEST['server']) : ''), "';\n";
			echo "-->\n</script>\n";
		}

		/**
		 * Converts a PHP.INI size variable to bytes.  Taken from publically available
		 * function by Chris DeRose, here: http://www.php.net/manual/en/configuration.directives.php#ini.file-uploads
		 * @param $strIniSize The PHP.INI variable
		 * @return size in bytes, false on failure
		 */
		function inisizeToBytes($strIniSize) {
			// This function will take the string value of an ini 'size' parameter,
			// and return a double (64-bit float) representing the number of bytes
			// that the parameter represents. Or false if $strIniSize is unparseable.
			$a_IniParts = array();

			if (!is_string($strIniSize))
				return false;

			if (!preg_match ('/^(\d+)([bkm]*)$/i', $strIniSize,$a_IniParts))
				return false;

			$nSize = (double) $a_IniParts[1];
			$strUnit = strtolower($a_IniParts[2]);

			switch($strUnit) {
				case 'm':
					return ($nSize * (double) 1048576);
				case 'k':
					return ($nSize * (double) 1024);
				case 'b':
				default:
					return $nSize;
			}
		}

		/**
		 * Display a URL given an action associative array.
		 * @param $action An associative array of the follow properties:
		 *			'url'  => The first part of the URL (before the ?)
		 *			'urlvars' => Associative array of (URL variable => field name)
		 *						these are appended to the URL
		 *			'urlfn' => Function to apply to URL before display
		 * @param $fields Field data from which 'urlfield' and 'vars' are obtained.
		 * @param $attr If supplied then the URL will be quoted and prefixed with
		 *				'$attr='.
		 */
		function printActionUrl(&$action, &$fields, $attr = null) {
			$url = value($action['url'], $fields);
			
			if ($url === false) return '';
			
			if (!empty($action['urlvars'])) {
				$urlvars = value($action['urlvars'], $fields);
			} else {
				$urlvars = array();
			}
			
			if (isset($urlvars['subject'])) {
				$subject = value($urlvars['subject'], $fields);
				if (isset($_REQUEST['server']) && $subject != 'root') {
					$urlvars['server'] = $_REQUEST['server'];
					if (isset($_REQUEST['database']) && $subject != 'server') {
						$urlvars['database'] = $_REQUEST['database'];
						if (isset($_REQUEST['schema']) && $subject != 'database') {
							$urlvars['schema'] = $_REQUEST['schema'];
						}
					}
				}
			}
			
			$sep = '?';
			foreach ($urlvars as $var => $varfield) {
				$url .= $sep . value_url($var, $fields) . '=' . value_url($varfield, $fields);
				$sep = '&';
			}
			
			$url = htmlentities($url);
			
			if ($attr !== null && $url != '')
				return ' '.$attr.'="'.$url.'"';
			else
				return $url;
		}

		function getRequestVars($subject = '') {
			$v = array();
			if (!empty($subject))
				$v['subject'] = $subject;
			if (isset($_REQUEST['server']) && $subject != 'root') {
				$v['server'] = $_REQUEST['server'];
				if (isset($_REQUEST['database']) && $subject != 'server') {
					$v['database'] = $_REQUEST['database'];
					if (isset($_REQUEST['schema']) && $subject != 'database') {
						$v['schema'] = $_REQUEST['schema'];
					}
				}
			}
			return $v;
		}
		
		function printUrlVars(&$vars, &$fields) {
			foreach ($vars as $var => $varfield) {
				echo "{$var}=", urlencode($fields[$varfield]), "&amp;";
			}
		}
		
		/**
		 * Display a table of data.
		 * @param $tabledata A set of data to be formatted, as returned by $data->getDatabases() etc.
		 * @param $columns   An associative array of columns to be displayed:
		 *			$columns = array(
		 *				column_id => array(
		 *					'title' => Column heading,
		 *					'field' => Field name for $tabledata->f[...],
		 *				), ...
		 *			);
		 * @param $actions   Actions that can be performed on each object:
		 *			$actions = array(
		 *				action_id => array(
		 *					'title' => Action heading,
		 *					'url'   => Static part of URL,
		 *					'vars'  => Associative array of (URL variable => field name),
		 *				), ...
		 *			);
		 * @param $nodata    (optional) Message to display if data set is empty.
		 * @param $pre_fn    (optional) Name of a function to call for each row,
		 *					 it will be passed two params: $rowdata and $actions,
		 *					 it may be used to derive new fields or modify actions.
		 *					 It can return an array of actions specific to the row,
		 *					 or if nothing is returned then the standard actions are used.
		 *					 (see functions.php and constraints.php for examples)
		 */
		function printTable(&$tabledata, &$columns, &$actions, $nodata = null, $pre_fn = null, $chk_auth = FALSE) {
			global $data, $conf, $misc;

			if ($tabledata->recordCount() > 0) {
				// Apply the 'properties' action to the first column
				// and remove it from the action list.
				// (Remove this section to keep the 'Properties' button instead of links)
				if (isset($actions['properties'])) {
					reset($columns);
					$first_column = key($columns);
					$columns[$first_column]['url'] = $actions['properties']['url'];
					$columns[$first_column]['vars'] = $actions['properties']['vars'];
					unset($actions['properties']);
				}
				
				echo "<table cellpadding=3>\n";
				echo "<tr>\n";
				// Display column headings
				foreach ($columns as $column_id => $column) {
					switch ($column_id) {
						case 'actions':
							if (sizeof($actions) > 0) echo "<th class=\"data\" colspan=\"", count($actions), "\">{$column['title']}</th>\n";
							break;
						default:
							echo "<th class=\"data\">";
							echo $column['title'];
							echo "</th>\n";
							break;
					}
				}
				echo "</tr>\n";
				
				// Display table rows
				$i = 0;
				while (!$tabledata->EOF) {
					$id = ($i % 2) + 1;
					
					unset($alt_actions);
					if (!is_null($pre_fn)) $alt_actions = $pre_fn($tabledata, $actions);
					if (!isset($alt_actions)) $alt_actions =& $actions;
					
					echo "<tr>\n";
					$a_act = 0;
					if ($chk_auth) {
						if (isset($columns['authority'])) {
							$col = $columns['authority'];
							if (!isset($tabledata->f[$col['field']]) ||
							    strlen($tabledata->f[$col['field']]) <= 0) {
								$a_act = 1;
							}
						}
					}
					
					foreach ($columns as $column_id => $column) {
					
						// Apply default values for missing parameters
						if (isset($column['url']) && !isset($column['vars'])) $column['vars'] = array();
						
						switch ($column_id) {
							case 'actions':
								foreach ($alt_actions as $action) {
									if ((isset($action['disable']) && $action['disable'] === true) || $a_act) {
										echo "<td class=\"data{$id}\"></td>";
									} else {
										echo "<td class=\"opbutton{$id}\">";
										echo "<a href=\"{$action['url']}";
										$misc->printUrlVars($action['vars'], $tabledata->f);
										echo "\">{$action['title']}</a></td>";
									}
								}
								break;
							default;
								echo "<td class=\"data{$id}\">";
								if (isset($tabledata->f[$column['field']])) {
									if (isset($column['url'])) {
										echo "<a href=\"{$column['url']}";
										$misc->printUrlVars($column['vars'], $tabledata->f);
										echo "\">";
									}
								
									$type = isset($column['type']) ? $column['type'] : null;
									$params = isset($column['params']) ? $column['params'] : array();
									echo $misc->printVal($tabledata->f[$column['field']], $type, $params);
								}
								
								if (isset($column['url'])) echo "</a>";

								echo "</td>\n";
								break;
						}
					}
					echo "</tr>\n";
					
					$tabledata->moveNext();
					$i++;
				}
				
				echo "</table>\n";
				
				return true;
			} else {
				if (!is_null($nodata)) {
					echo "<p>{$nodata}</p>\n";
				}
				return false;
			}
		}
		
		/** Produce XML data for the browser tree
		 * @param $treedata A set of records to populate the tree.
		 * @param $attrs Attributes for tree items
		 *        'text' - the text for the tree node
		 *        'icon' - an icon for node
		 *        'openIcon' - an alternative icon when the node is expanded
		 *        'toolTip' - tool tip text for the node
		 *        'action' - URL to visit when single clicking the node
		 *        'iconAction' - URL to visit when single clicking the icon node
		 *        'branch' - URL for child nodes (tree XML)
		 *        'expand' - the action to return XML for the subtree
		 *        'nodata' - message to display when node has no children
		 *        'nohead' - suppress headers and opening <tree> tag
		 *        'nofoot' - suppress closing </tree> tag
		 */
		function printTreeXML(&$treedata, &$attrs, $chk_auth = FALSE) {
			global $conf, $lang;
			
			if (!isset($attrs['nohead']) || $attrs['nohead'] === false) {
				header("Content-Type: text/xml; charset=EUC_KR");
				header("Cache-Control: no-cache");
				
				echo "<?xml version=\"1.0\" encoding=\"", htmlspecialchars($lang['appxmlencoding']), "\"?>\n";
				
				echo "<tree>\n";
			}
			
			if ($treedata->recordCount() > 0) {
				$a_cnt=0;
				while (!$treedata->EOF) {
					$rec =& $treedata->f;

					$a_act = 0;
					if ($chk_auth) {
						if (!isset($rec['authority']) ||
							strlen($rec['authority']) <= 0 ||
							$rec['status'] != 'start')
							$a_act = 1;
					}
					
					if (!$a_act) {
							echo "<tree";
							echo value_xml_attr('text', $attrs['text'], $rec);
							echo value_xml_attr('action', $attrs['action'], $rec);
							echo value_xml_attr('src', $attrs['branch'], $rec);
							
							$icon = $this->icon(value($attrs['icon'], $rec));
							echo value_xml_attr('icon', $icon, $rec);
							echo value_xml_attr('iconaction', $attrs['iconAction'], $rec);
							
							if (!empty($attrs['openicon'])) {
								$icon = $this->icon(value($attrs['openIcon'], $rec));
							}
							echo value_xml_attr('openicon', $icon, $rec);
							
							echo value_xml_attr('tooltip', $attrs['toolTip'], $rec);
							
							echo "/>\n";
							$a_cnt ++;
					}
					
					$treedata->moveNext();
				}
				if ($a_cnt <= 0) {
					$msg = isset($attrs['nodata']) ? $attrs['nodata'] : $lang['strnoobjects'];
					echo "<tree text=\"{$msg}\" onaction=\"this.parentNode.reload()\" icon=\"",$this->icon('ObjectNotFound'), "\" openicon=\"", $this->icon('ObjectNotFound'), "\"/>\n";
				}
			} else {
				$msg = isset($attrs['nodata']) ? $attrs['nodata'] : $lang['strnoobjects'];
				echo "<tree text=\"{$msg}\" onaction=\"this.parentNode.reload()\" icon=\"",$this->icon('ObjectNotFound'), "\" openicon=\"", $this->icon('ObjectNotFound'), "\"/>\n";
			}
			
			if (!isset($attrs['nofoot']) || $attrs['nofoot'] === false) {
				echo "</tree>\n";
			}
		}
		
		function adjustTabsForTree(&$tabs) {
			include_once('./classes/ArrayRecordSet.php');
			
			foreach ($tabs as $i => $tab) {
				if ((isset($tab['hide']) && $tab['hide'] === true) || (isset($tab['tree']) && $tab['tree'] === false)) {
					unset($tabs[$i]);
				}
			}
			return new ArrayRecordSet($tabs);
		}
		
		function icon($icon) {
			global $conf;
			$path = "images/themes/{$conf['theme']}/{$icon}";
			if (file_exists($path.'.png')) return $path.'.png';
			if (file_exists($path.'.gif')) return $path.'.gif';
			$path = "images/themes/default/{$icon}";
			if (file_exists($path.'.png')) return $path.'.png';
			if (file_exists($path.'.gif')) return $path.'.gif';
			return '';
		}
		
		/**
		 * Function to escape command line parameters
		 * @param $str The string to escape
		 * @return The escaped string
		 */
		function escapeShellArg($str) {
			global $data, $lang;
			
			if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
				// Due to annoying PHP bugs, shell arguments cannot be escaped
				// (command simply fails), so we cannot allow complex objects
				// to be dumped.
				if (ereg('^[_.[:alnum:]]+$', $str))
					return $str;
				else {
					echo $lang['strcannotdumponwindows'];
					exit;
				}				
			}
			else	
				return escapeshellarg($str);
		}

		/**
		 * Function to escape command line programs
		 * @param $str The string to escape
		 * @return The escaped string
		 */
		function escapeShellCmd($str) {
			global $data;
			
			if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
				$data->fieldClean($str);
				return '"' . $str . '"';
			}
			else	
				return escapeshellcmd($str);
		}
		
		/**
		 * Get list of servers
		 * @param $recordset return as RecordSet suitable for printTable if true,
		 *                   otherwise just return an array.
		 */
		function getServers($recordset = false) {
			global $conf;
			
			$srvs = isset($_SESSION['webdbLogin']) && is_array($_SESSION['webdbLogin']) ? $_SESSION['webdbLogin'] : array();

			foreach($conf['servers'] as $idx => $info) {
				$server_id = $info['host'].':'.$info['port'];
				
				if (!isset($srvs[$server_id])) {
					$srvs[$server_id] = $info;
				}
				$srvs[$server_id]['id'] = $server_id;
			}
			
			function _cmp_desc($a, $b) {
				return strcmp($a['desc'], $b['desc']);
			}
			uasort($srvs, '_cmp_desc');
			
			if ($recordset) {
				include_once('./classes/ArrayRecordSet.php');
				return new ArrayRecordSet($srvs);
			}
			return $srvs;
		}
		
		/**
		 * Validate and retrieve information on a server
		 * If the parameter isn't supplied then the currently
		 * connected server is returned.
		 * @param $server_id A server identifier (host:port)
		 * @return An associative array of server properties
		 */
		function getServerInfo($server_id = null) {
			global $conf, $_reload_browser, $lang;

			if ($server_id === null && isset($_REQUEST['server']))
				$server_id = $_REQUEST['server'];
		//var_dump($_SESSION['webdbLogin'][$server_id]);//@@@
			// Check for the server in the logged-in list
			if (isset($_SESSION['webdbLogin'][$server_id]))
				return $_SESSION['webdbLogin'][$server_id];
			
			// Otherwise, look for it in the conf file
			foreach($conf['servers'] as $idx => $info) {
				if ($server_id == $info['host'].':'.$info['port']) {
					// Automatically use shared credentials if available
					if (!isset($info['username']) && isset($_SESSION['sharedUsername'])) {
						$info['username'] = $_SESSION['sharedUsername'];
						$info['password'] = $_SESSION['sharedPassword'];
						$_reload_browser = true;
						$this->setServerInfo(null, $info, $server_id);
					}
					
					return $info;
				}
			}
		        
                        if ($server_id === null){
                            return null;
                        } else {
                            // Unable to find a matching server, are we being hacked?
			    echo $lang['strinvalidserverparam'];
                            exit;
                        }; 
		}
		
		/**
		 * Set server information.
		 * @param $key parameter name to set, or null to replace all
		 *             params with the assoc-array in $value.
		 * @param $value the new value, or null to unset the parameter
		 * @param $server_id the server identifier, or null for current
		 *                   server.
		 */
		function setServerInfo($key, $value, $server_id = null)
		{
			if ($server_id === null && isset($_REQUEST['server']))
				$server_id = $_REQUEST['server'];
			
			if ($key === null) {
				if ($value === null)
					unset($_SESSION['webdbLogin'][$server_id]);
				else
					$_SESSION['webdbLogin'][$server_id] = $value;
			} else {
				if ($value === null)
					unset($_SESSION['webdbLogin'][$server_id][$key]);
				else
					$_SESSION['webdbLogin'][$server_id][$key] = $value;
			}
		}

		function getTokens($conn, $debug = FALSE)
		{
			$ret="";
			$retry_cnt = 0;
			
			while (1) {
			  //var_dump(stream_get_meta_data($conn));
			  unset($retone);
			  $retone = socket_read($conn, 1500);
				if ($retone === FALSE) {
				  if ($debug) $ret .= " #1"; 
				  break;
				}
				if (strlen($retone) <= 0) {
				  $retry_cnt ++;
				  if ($retry_cnt>5) {
				    if ($debug) $ret .= " #3"; 
				    break;
				  }
				  sleep(1);
				  continue;
				}
				$ret .= $retone;
			  //var_dump(stream_get_meta_data($conn));
				if (strstr($retone,"\n\n")) {
				  if ($debug) $ret .= " #2"; 
				  break;
				}
			}
			if ($debug) echo "(((".$ret.")))";
			$lines = explode("\n", $ret);
			$pp = array();
			$cur = array();
			$inary = 0;
			foreach($lines as $idx => $v) {
				$pos=strpos($v, ":");
				if ($pos===FALSE) continue;
				$key=substr($v, 0, $pos);
				$val=substr($v, $pos+1);
				if ($key=="open") {
					array_push($pp, $cur);
					$cur=array();
					$inary++;
				}
				else if ($key=="close") {
					$old = array_pop($pp);
					$old[$val][]=$cur;
					$cur=$old;
					$inary--;
				}
				else {
					if ($inary>0) {
						if (isset($cur[$key])) {
							if (is_array($cur[$key])) {
								$cur[$key][]=$val;
							}
							else {
								$tmp=$cur[$key];
								unset($cur[$key]);
								$cur[$key][]=$tmp;
								$cur[$key][]=$val;
							}
						}
						else
							$cur[$key]=$val;
					}
					else
						$cur[$key]=$val;
				}
			}
			if ($debug) var_dump($cur);
			return $cur;
		}

		function CloseCubridManager($server_id)
		{
			if (isset($_SESSION['webdbLogin'][$server_id]['sessfile'])) {
					$psessfile = $_SESSION['webdbLogin'][$server_id]['sessfile'];
					if(file_exists($psessfile))
						unlink($psessfile);
			}
  		$server_info = $this->getServerInfo($server_id);
  		$this->setServerInfo(null, null, $server_id);
		}

		function ManagerCommand($server_id, $tasktype, $msg, $debug=FALSE)
		{
			$timeout = "60";
			list($host, $port) = explode(":", $server_id);
			$port ++;

			if (!($mConnect = socket_create(AF_INET, SOCK_STREAM, 0))) {
			  $toks=array();
			  $toks['status']="fail";
			  $toks['note']="$host, $port socket fail".socket_strerror(socket_last_error());
			  return $toks;
			}

			if (!socket_connect($mConnect, $host, $port)) {
			  $toks=array();
			  $toks['status']="fail";
			  $toks['note']="$host, $port connect fail".socket_strerror(socket_last_error());
			  return $toks;
			}

			$hosttoken = $_SESSION['webdbLogin'][$server_id]['token'];
      if ($debug) {
        echo " request:$tasktype, $hosttoken, $msg)))";
      }

      $sendmsg = "task:$tasktype\ntoken:$hosttoken\n$msg\n\n";
			if (socket_send($mConnect, $sendmsg, strlen($sendmsg), 0) < 0) {
			  $toks=array();
			  $toks['status']="fail";
			  $toks['note']="$host, $port send fail";
			  return $toks;
			}

			$toks=$this->getTokens($mConnect, $debug, TRUE);

			socket_close($mConnect);

			if (!isset($toks['status']) || !isset($toks['task']) || !isset($toks['note'])) {
			  $tok=array();
			  $tok['status']="fail";
			  $tok['note']="$host, $port message format error";
			  return $tok;
			}
			return $toks;
		}

		function ManagerLogin($server_id, $user_id, $user_pass)
		{
			global $mgrmsg;

			$timeout = "60";
			list($host, $port) = explode(":", $server_id);
			$user_ip = $_SERVER['REMOTE_ADDR'];
			$psessfile = "./".$user_id.$host.$port.$user_ip;
			while (file_exists($psessfile)) {
				unlink($psessfile);
				sleep(1);
			}
			sleep(1);
			$thisuri = $_SERVER['REQUEST_URI'];
			$endpos = strrpos($thisuri, "/");
			$thisuri = ($endpos === FALSE) ? "/" : substr($thisuri, 0, $endpos+1);

			$sendmsg = "GET $thisuri"."psesscntl.php?server_id=".urlencode($server_id)."&user_id=$user_id&user_pass=$user_pass&user_ip=$user_ip HTTP/1.1\r\n";
			$sendmsg .= "Host: ".$_SERVER['SERVER_ADDR']."\r\n";
			$sendmsg .= "Connection: Close\r\n\r\n";
			$bConnect = fsockopen($_SERVER['SERVER_ADDR'], $_SERVER['SERVER_PORT'], $errno, $errstr, $timeout);
			if ($bConnect === FALSE) {
				$mgrmsg = "Socket open failed!";
			  return FALSE;
			}
			if (fwrite($bConnect, $sendmsg, strlen($sendmsg)) === FALSE) {
				$mgrmsg = "Message Send failed!";
				return FALSE;
			}

			$_server_info = $this->getServerInfo($server_id);
			$_SESSION['webdbLogin'][$server_id] = $_server_info;

			$waitcnt = 0;
			while (1) {
				if(file_exists($psessfile)){
					$sesscntl=file($psessfile);
					$status=substr($sesscntl[0], 0, 1);
					$status_msg=substr($sesscntl[0], 1);
					if ($status == "X") {
						$mgrmsg = $status_msg."($waitcnt)";
						unlink($psessfile);
						return FALSE;
					}
					$_SESSION['webdbLogin'][$server_id]['token'] = $status_msg;
					break;
				}
				$waitcnt ++;
				if ($waitcnt > 20) {
					$mgrmsg = "Can not connect to $server_id (timeout)";
					return FALSE;
				}
				sleep(1);
			}

			$_SESSION['webdbLogin'][$server_id]['id'] = $server_id;
			$_SESSION['webdbLogin'][$server_id]['username'] = $user_id;
			$_SESSION['webdbLogin'][$server_id]['password'] = $user_pass;
			$_SESSION['webdbLogin'][$server_id]['sessfile'] = $psessfile;

			return $this->ManagerRefresh($server_id);
		}

		function ManagerRefresh($server_id, $func = "all")
		{
		  global $mgrmsg;
		 
		  $mgrmsg = ""; 
		  if ($func == "all") {
			$tk1 = $this->ManagerCommand($server_id, "getenv" , "");
			if ($tk1['status']!="success") {
				$mgrmsg = $tk1['note'];
				$this->CloseCubridManager($server_id);
				return FALSE;
			}
			
			$_SESSION['webdbLogin'][$server_id]['UNISQLX'] = $tk1['UNISQLX'];
			$_SESSION['webdbLogin'][$server_id]['UNISQLX_DATABASES'] = $tk1['UNISQLX_DATABASES'];
			$_SESSION['webdbLogin'][$server_id]['UNISQLVER'] = $tk1['UNISQLVER'];
			$_SESSION['webdbLogin'][$server_id]['UNICASVER'] = $tk1['UNICASVER'];
			$_SESSION['webdbLogin'][$server_id]['osinfo'] = $tk1['osinfo'];
			$verrel = explode(" ", $tk1['UNISQLVER']);
			$_SESSION['webdbLogin'][$server_id]['platform'] = "$verrel[0] $verrel[1] $verrel[2] $verrel[3]";
			unset($tk1);

			$tk1 = $this->ManagerCommand($server_id, "checkaccessright" , "");
			if ($tk1['status']!="success") {
				$mgrmsg = $tk1['note'];
				$this->CloseCubridManager($server_id);
				return FALSE;
			}
			$_SESSION['webdbLogin'][$server_id]['casauth'] = $tk1['userauth'][0]['casauth'];
		  }

		  if ($func == "all" || $func == "dbmtuserinfo") {
			$tk2 = $this->ManagerCommand($server_id, "getdbmtuserinfo" , "");
			if ($tk2['status']!="success") {
				$mgrmsg = $tk2['note'];
				$this->CloseCubridManager($server_id);
				return FALSE;
			}
			$_SESSION['webdbLogin'][$server_id]['dblist'] = $tk2['dblist'][0]['dbname'];
			$_SESSION['webdbLogin'][$server_id]['userlist'] = $tk2['userlist'][0]['user'];
			$_SESSION['webdbLogin'][$server_id]['dbsuper'] = "none";
			$_SESSION['webdbLogin'][$server_id]['cassuper'] = "none";
			$users = $tk2['userlist'][0]['user'];
			for ($i=0; $i < count($users); $i++) {
				if ($users[$i]['id'] == $_SESSION['webdbLogin'][$server_id]['username']) {
					$_SESSION['webdbLogin'][$server_id]['dbsuper'] = $users[$i]['dbcreate'];
					$_SESSION['webdbLogin'][$server_id]['cassuper'] = $users[$i]['casauth'];
					break;
				}
			}
		  }

		  if ($func == "all" || $func == "unicasinfo") {
			$tk3 = $this->ManagerCommand($server_id, "getunicasinfo" , "");
			if ($tk3['status']!="success") {
				$mgrmsg = $tk3['note'];
				$this->CloseCubridManager($server_id);
				return FALSE;
			}
			$_SESSION['webdbLogin'][$server_id]['brokers'] = $tk3['brokersinfo'][0]['broker'];
			$_SESSION['webdbLogin'][$server_id]['broker'] = $tk3['unicasstatus'];
			if (isset($_SESSION['webdbLogin'][$server_id]['sqlport'])) {
				unset($_SESSION['webdbLogin'][$server_id]['sqlport']);
				unset($_SESSION['webdbLogin'][$server_id]['sqlbroker']);
			}
			$brk = $tk3['brokersinfo'][0]['broker'];
			$cnt = count($brk);
			for ($i = 0; $i < $cnt; $i++) {
				if (isset($brk[$i]['state']) &&
				    $brk[$i]['state'] == "ON") {
						$_SESSION['webdbLogin'][$server_id]['sqlport'] = $brk[$i]['port'];
						$_SESSION['webdbLogin'][$server_id]['sqlbroker'] = $brk[$i]['name'];
				}
				if ($brk[$i]['name'] == "query_editor") break;
			}
		  }

		  if ($func == "all") {
	    	$tk = $this->ManagerCommand($server_id, "startinfo" , "");
			if ($tk['status']!="success") {
				$mgrmsg = $tk['note'];
				$this->CloseCubridManager($server_id);
				return FALSE;
			}
			
			$recname = $this->array_adjust($tk['dblist'][0]['dbname']);
			$_SESSION['webdbLogin'][$server_id]['dblist'] = $recname;
			$recdir = $this->array_adjust($tk['dblist'][0]['dbdir']);

			for ($i = 0; $i < count($recname); $i++) {
				$_SESSION['webdbLogin'][$server_id]['dbdir'][$recname[$i]] = $recdir[$i];
				$_SESSION['webdbLogin'][$server_id]['dbstat'][$recname[$i]] = 
					(isset($tk['activelist'][0]['dbname']))
				   	? (($this->search_var($tk['activelist'][0]['dbname'], $recname[$i])) 
					? 'start' : 'stop') : 'stop';
			}
		  }
		  return TRUE;
		}

		function search_var($base, $srch) {
			if (!$base) return FALSE;
			if (is_array($base)) {
				$cnt = count($base);
				for ($i = 0; $i < $cnt; $i ++) {
					if ($base[$i] == $srch) return TRUE;
				}
				return FALSE;
			}
			else {
				if ($base != $srch) return FALSE;
			}
			return TRUE;
		}

		function array_adjust($var) {
			if (!$var) return array();
			if (is_array($var)) return $var;
			$ret = array();
			$ret[]=$var;
			return $ret;
		}

		function getDatabases() {
			global $misc;

			$server_info = $misc->getServerInfo();

			//var_dump($server_info);
			$rst = array();
			for ($i=0; $i < count($server_info['dblist']); $i++) {
				$rec = array();
				$rec['dbname'] = $server_info['dblist'][$i];
				$rec['dbdir'] = $server_info['dbdir'][$rec['dbname']];
				$rec['status'] = $server_info['dbstat'][$rec['dbname']];
				$rec['authority'] = (isset($server_info['login'][$rec['dbname']]))
									? $server_info['login'][$rec['dbname']] : "";
				$rst[]=$rec;
			}

			include_once('./classes/ArrayRecordSet.php');
			return new ArrayRecordSet($rst);
		}

		function getSchema($dbname, $refresh = false) {
			global $_server_info;

			if (isset($_SESSION['webdbLogin'][$_server_info['id']]['schema'][$dbname]) && !$refresh)
				   return $_SESSION['webdbLogin'][$_server_info['id']]['schema'][$dbname];
	    	$tk = $this->ManagerCommand($_server_info['id'], "classinfo" , "dbname:$dbname\ndbstatus:on");
			if ($tk['status']!="success") {
				echo $tk['note'];
				return FALSE;
			}
			$_SESSION['webdbLogin'][$_server_info['id']]['schema'][$dbname] = $tk['userclass'][0];

			return $tk['userclass'][0];
		}

		function getTables($refresh = false) {
			global $_server_info, $db;

			$tbls = $this->getSchema($db, $refresh);
			if ($tbls === FALSE) return FALSE;
			if (!isset($tbls['class'])) $tbls['class'] = array();
			include_once('./classes/ArrayRecordSet.php');
			return new ArrayRecordSet($tbls['class']);
		}

		function getClass($tbl)
		{
			global $_server_info, $db;

			if (isset($_SESSION['webdbLogin'][$_server_info['id']]['table'][$db][$tbl]))
				   return $_SESSION['webdbLogin'][$_server_info['id']]['table'][$db][$tbl];
	    	$tk = $this->ManagerCommand($_server_info['id'], "class" , "dbname:$db\nclassname:$tbl");
			if ($tk['status']!="success") {
				echo $tk['note'];
				return FALSE;
			}
			$_SESSION['webdbLogin'][$_server_info['id']]['table'][$db][$tbl] = $tk['classinfo'][0];

			return $tk['classinfo'][0];
		}

		function getBroker($brk, $refresh = false)
		{
			global $_server_info;

			if (isset($_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk]) && !$refresh)
				   return $_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk];
	    	$tk = $this->ManagerCommand($_server_info['id'], "getaslimit" , "bname:$brk");
			if ($tk['status']!="success") {
				echo $tk['note'];
				return FALSE;
			}
	    	$tk1 = $this->ManagerCommand($_server_info['id'], "getbrokerstatus" , "bname:$brk");
			if ($tk1['status']!="success") {
				echo $tk1['note'];
				return FALSE;
			}
			$_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk] = $tk1['asinfo'];
			$_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk]['minas'] = $tk['minas'];
			$_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk]['maxas'] = $tk['maxas'];
			$_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk]['asnum'] = $tk['asnum'];

			return $_SESSION['webdbLogin'][$_server_info['id']]['brokerstat'][$brk];
		}

		function getUsers($refresh = false)
		{
			global $_server_info, $db;

			if (isset($_SESSION['webdbLogin'][$_server_info['id']]['dbusers'][$db]) && !$refresh)
				   return $_SESSION['webdbLogin'][$_server_info['id']]['dbusers'][$db];
	    	$tk = $this->ManagerCommand($_server_info['id'], "userinfo" , "dbname:$db" );
			if ($tk['status']!="success") {
				echo $tk['note'];
				return FALSE;
			}
			$_SESSION['webdbLogin'][$_server_info['id']]['dbusers'][$db] = $tk['user'];

			return $tk['user'];
		}

		function getDbUsers($refresh = false)
		{
			$user = $this->getUsers($refresh);
			if ($user === FALSE) return FALSE;

			include_once('./classes/ArrayRecordSet.php');
			return new ArrayRecordSet($user);
		}

		function getUserMessage($type, $id)
		{
			$user = $this->getUsers();
			if ($user === FALSE) return FALSE;
			$id = strtoupper($id);

			for ($i = 0; $i < count($user); $i++) {
				if ($user[$i]['name'] == $id) break;
			}
			if ($i >= count($user)) return FALSE;
			$user = $user[$i];

			$ret = "";
			switch ($type) {
				case 'group':
					$ret .= "open:groups\n";
					if (isset($user['groups'][0]['group'])) {
						$user = $this->array_adjust($user['groups'][0]['group']);
						for ($i = 0; $i < count($user); $i++) {
							$ret .= "group:".$user[$i]."\n";
						}
					}
					$ret .= "close:groups\n";
					return $ret;

				case 'auth':
					$ret .= "open:authorization\n";
					$user = $user['authorization'][0];
					foreach ($user as $tblname => $authnum) {
						$ret .= "$tblname:$authnum\n";
					}
					$ret .= "close:authorization\n";
					return $ret;
			}
		}

		function getTableIndexes($table)
		{
			global $_server_info;

			$tbl = $this->getClass($table);
			if ($tbl === FALSE) return FALSE;
			if (isset($tbl['constraint'])) {
					$cnt=count($tbl['constraint']);
					for ($i = 0; $i < $cnt; $i++) {
							if ($tbl['constraint'][$i]['type']=='NOT NULL')
								unset($tbl['constraint'][$i]);
					}
			}
			else
					$tbl['constraint'] = array();

			include_once('./classes/ArrayRecordSet.php');
			return new ArrayRecordSet($tbl['constraint']);
		}

		function getTableAttributes($table)
		{
			global $_server_info;

			$tbl = $this->getClass($table);
			if ($tbl === FALSE) return FALSE;

			$attr = array();
			if (isset($tbl['classattribute'])) 
				$attr = array_merge($attr, $tbl['classattribute']);
			if (isset($tbl['attribute'])) 
				$attr = array_merge($attr, $tbl['attribute']);
			$cnt=count($attr);
			for ($i = 0; $i < $cnt; $i++) {
					if ($attr[$i]['inherit']==$table)
						$attr[$i]['inherit']="";
			}

			include_once('./classes/ArrayRecordSet.php');
			return new ArrayRecordSet($attr);
		}

		function DbLogin($server_info, $dbname, $user_id, $user_pass)
		{
			global $mgrmsg;

			$msg = "targetid:".$server_info['username']."\n";
			$msg .= "dbname:$dbname\n";
			$msg .= "dbuser:".strtoupper($user_id)."\n";
			$msg .= "dbpasswd:$user_pass\n";
				
	    	$tk = $this->ManagerCommand($server_info['id'], "dbmtuserlogin" , $msg);
			if ($tk['status']!="success") {
				$mgrmsg = $tk['note'];
				return FALSE;
			}
			$_SESSION['webdbLogin'][$server_info['id']]['login'][$dbname] = strtoupper($user_id); 
			$_SESSION['webdbLogin'][$server_info['id']]['login_pass'][$dbname] = $user_pass; 
			$_SESSION['webdbLogin'][$server_info['id']]['isdba'][$dbname] = 
				($tk['authority'] == 'isdba') ? 'y' : 'n';

	    	$tk = $this->ManagerCommand($server_info['id'], "dbspaceinfo" , "dbname:$dbname");
			if ($tk['status']!="success") {
				$mgrmsg = $tk['note'];
				return FALSE;
			}
			$cnt = count($tk['spaceinfo']);
			if ($tk['spaceinfo'][$cnt - 1]['spacename'] == "Total")
				unset($tk['spaceinfo'][$cnt - 1]);
			$_SESSION['webdbLogin'][$server_info['id']]['spaceinfo'][$dbname] = $tk['spaceinfo'];
			$_SESSION['webdbLogin'][$server_info['id']]['pagesize'][$dbname] = $tk['pagesize'];
			return TRUE;
		}

		function SQLLogin(&$err)
		{
			global $_server_info, $db;

			list($host, $port) = explode(":", $_server_info['id']);
			$cubrid_con = @cubrid_connect($host, $_server_info['sqlport'], $db, $_server_info['login'][$db], $_server_info['login_pass'][$db]);

			if (!$cubrid_con) {
				$err = $lang['strconnectfail']." $host ".$_server_info['sqlport']." ".$_server_info['login'][$db]." ".cubrid_error_msg();
				return FALSE;
			}
			return $cubrid_con;
		}
	}
?>
