#ifndef VDSP_H
#define VDSP_H


#include <vecLib/vDSP_translate.h>


#include <APPLE/vecLib/vDSP.h>

#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*DSPSplitComplex

The split complex data type is defined in the header file vDSP.h. Its definition is:
struct DSPSplitComplex {
    float *                         realp;
    float *                         imagp;
};
typedef struct DSPSplitComplex          DSPSplitComplex;


Split complex vectors are stored as two vectors, a vector containing all the real values and a vector containing all the imaginary values.*/
struct DSPSplitComplex {
    float *                         realp;
    float *                         imagp;
};
typedef struct DSPSplitComplex          DSPSplitComplex;

/*DSPComplex

The complex data type is defined in the header file vDSP.h. Its definition is:
struct DSPComplex {
    float                           real;
    float                           imag;
};
typedef struct DSPComplex               DSPComplex;


Complex data are stored as ordered pairs of floating-point numbers. Because they are stored as ordered pairs, complex vectors require address strides that are multiples of two.*/
struct DSPComplex {
    float                           real;
    float                           imag;
};
typedef struct DSPComplex               DSPComplex;

//func

/*vDSP_destroy_fftsetup

Frees an existing Fourier Transforms data structure.

void
vDSP_destroy_fftsetup (FFTSetup setup);
Discussion

vDSP_destroy_fftsetup frees an existing weights array. Any memory allocated for the array is released. Parameter setup identifies the weights array, and must point to a data structure previously created by vDSP_create_fftsetup. After the vDSP_destroy_fftsetup function returns, the structure is no longer valid and cannot be passed to any subsequent frequency-domain functions. 

When forecasting memory requirements for building a weights array, keep in mind that this function generate complex numbers. See also Chapter 2, “Using Fourier Transforms.”
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_destroy_fftsetup (FFTSetup setup);

/*vDSP_create_fftsetup

Builds a data structure that contains precalculated data for use by Fourier Transform functions.

FFTSetup
vDSP_create_fftsetup (UInt32 log2n,
   FFTRadix radix);
Discussion

A single weights array can serve multiple FFT functions as long as the array is large enough for each function called and as long as the weights array is preserved. Functions that use the weights array automatically adjust their strides through the array when the array is larger than required. Using a shared weights array for similarly sized FFTs conserves memory. However, if the size disparity is great, using a large weights array for an FFT that processes a small number of elements can degrade performance. 

Parameter log2n is a base 2 exponent that represents the number of divisions of the complex unit circle and thus specifies the largest power of two that can be processed by a subsequent frequency-domain function. Parameter log2n must equal or exceed the largest power of 2 that any subsequent function processes using the weights array. 

Parameter radix specifies radix options. Radix 2, radix 3, and radix 5 functions are supported.

vDSP_create_fftsetup builds a weights array and returns an address (type FFTSetup ) that points to a data structure that contains the weights array to any functions that subsequently use the array. The address is subsequently passed as an argument to FFT functions.

If zero is returned, the vDSP_create_fftsetup function failed to find enough storage to allocate the weights array. 

When forecasting memory requirements for building a weights array, keep in mind that this function generates complex numbers.

See also Chapter 2, “Using Fourier Transforms.”

Availability
Available in Mac OS X v10.4 and later.*/
FFTSetup
vDSP_create_fftsetup (UInt32 log2n,
   FFTRadix radix);

/*vDSP_fft_zop

Computes an out-of-place complex discrete Fourier transform of the input vector, either from the time domain to the frequency domain (forward) or from the frequency domain to the time domain (inverse).

void
vDSP_fft_zop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   DSPSplitComplex * result,
   SInt32 resultStride,
   UInt32 log2n,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the value supplied as parameter log2n of the transform function.
signalStride

Specifies an address stride through input vector signal. Parameter resultStride specifies an address stride through output vector result. Thus, to process every element, specify a signalStride of 1; to process every other element, specify 2. The values of signalStride and resultStride should be 1 for best performance.

For vDSP_fft_zopt and vDSP_fft_zoptD, parameter tempBuffer is a temporary vector used for storing interim results. The size of temporary memory for each part (real and imaginary) is the lower value of 4*n or 16k. Or you can simply pass the buffer of size 2^(log2n) for each part (real and imaginary). If possible, tempBuffer.realp and tempBuffer.imagp should be 32-byte aligned for best performance.
log2n

The base 2 exponent of the number of elements to process. For example, to process 1024 elements, specify 10 for parameter log2n. See also the FFT Limitations sections in the Target chapters of the Developer’s Guide. The value of log2n must be between 2 and 20, inclusive.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

This performs the operation




Criteria to invoke vectorized code:
The input vectors signal.realp and signal.imagp, the output vectors result.realp and result.imagp, and the buffer vectors tempBuffer.realp and tempBuffer.imagp must be aligned on 16-byte boundaries. 
The values of signalStride and resultStride must be 1. 
The value of log2n must be between 2 and 20, inclusive. 

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft_zop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   DSPSplitComplex * result,
   SInt32 resultStride,
   UInt32 log2n,
   FFTDirection direction);

/*vDSP_vsmul

Multiplies vector signal1 by scalar signal2 and leaves the result in vector result.

void
vDSP_vsmul (const float signal1[],
   SInt32 signal1stride,
   const float * signal2,
   float result[],
   SInt32 resultStride,
   UInt32 size);
Discussion

This performs the operation




Criteria to invoke vectorized code:
The vectors signal1 and result must be relatively aligned.
The value of size must be equal to or greater than 8.
The values of signal1stride and resultStride must be 1.

If any of these criteria is not satisfied, the function invokes scalar code.
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_vsmul (const float signal1[],
   SInt32 signal1stride,
   const float * signal2,
   float result[],
   SInt32 resultStride,
   UInt32 size);

/*vDSP_fft2d_zrop

Computes an out-of-place real discrete Fourier transform, either from the spatial domain to the frequency domain (forward) or from the frequency domain to the spatial domain (inverse). 

void
vDSP_fft2d_zrop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalRowStride,
   SInt32 signalColStride,
   DSPSplitComplex * result,
   SInt32 resultRowStride,
   SInt32 resultColStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the value supplied as parameter log2n or log2m, whichever is larger, of the transform function.
signalRowStride

Specifies a stride across each row of matrix signal. Specifying 1 for signalRowStride processes every element across each row, specifying 2 processes every other element across each row, and so forth.
signalColStride

If not 0, represents the distance between each row of the input /output matrix. If parameter signalColStride is 1024, for instance, element 512 equates to element (1,0) of matrix a, element 1024 equates to element (2,0), and so forth.
colStride

Specifies a column stride for the matrix, and should generally be allowed to default unless the matrix is a submatrix. Parameter colStride can be defaulted by specifying 0. The default stride equals the row stride multiplied by the column count. Thus, if signalRowStride is 1 and signalColStride is 0, every element of matrix signal is processed. If signalRowStride is 2 and signalColStride is 0, every other element of each row is processed.
resultRowStride

Specifies a row stride for output matrix c in the same way that signalRowStride specifies strides for input the matrix.
resultColStride

Specifies a column stride for output matrix c in the same way that signalColStride specify strides for input the matrix.
log2nInCol

The base 2 exponent of the number of columns to process for each row. log2nInCol must be between 3 and 10, inclusive.
log2nInRow

The base 2 exponent of the number of rows to process. For example, to process 64 rows of 128 columns, specify 7 for log2nInCol and 6 for log2nInRow. log2nInRow must be between 3 and 10, inclusive.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

Forward transforms read real input and write packed complex output. Inverse transforms read packed complex input and write real output. As a result of packing the frequency-domain data, spatial-domain data and its equivalent frequency-domain data have the same storage requirements; see “Data Packing for Real FFTs”. 




Real data is stored in split complex form, with odd reals stored on the imaginary side of the split complex form and even reals in stored on the real side.

Criteria to invoke vectorized code:
The input vectors signal.realp and signal.imagp, the output vectors result.realp and result.imagp, and the temporary vectors tempBuffer.realp and tempBuffer.imagp must be aligned on 16-byte boundaries. 
The values of signalRowStride and resultRowStride must be 1. 
The values of signalColStride and resultColStride must be multiples of 4. 
The value of log2nInRow and log2nInCol must be between 3 and 10, inclusive. 

If any of these criteria is not satisfied, the function invokes scalar code. 

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft2d_zrop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalRowStride,
   SInt32 signalColStride,
   DSPSplitComplex * result,
   SInt32 resultRowStride,
   SInt32 resultColStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);

/*vDSP_ztoc

Copies the contents of a split complex vector A to an interleaved complex vector C.

void
vDSP_ztoc (const DSPSplitComplex * A,
   SInt32 I,
   float *C,
   SInt32 K,
   SInt32 N);
Discussion

This peforms the operation




K is an address stride through C. I is an address stride through A. 

For best performance, C->realp, C->imagp, A->realp, and A->imagp should be 16-byte aligned.

Criteria to invoke vectorized code:
The value of N must be greater than 3. 
The value of K must be 2. 
The value of I must be 1. 
Vectors A->realp and A->imagp must be relatively aligned.
Vector C must 8 bytes aligned if A->realp and A->imagp are 4-byte aligned or C must be 16 bytes aligned if A->realp and A->imagp are at least 8-byte aligned. 

If any of these criteria is not satisfied, the function invokes scalar code.

See also “vDSP_ctoz” and “vDSP_ctozD”.
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_ztoc (const DSPSplitComplex * A,
   SInt32 I,
   float *C,
   SInt32 K,
   SInt32 N);

/*vDSP_fft_zrop

Computes an out-of-place real discrete Fourier transform, either from the time domain to the frequency domain (forward) or from the frequency domain to the time domain (inverse). 

void
vDSP_fft_zrop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   DSPSplitComplex * result,
   SInt32 resultStride,
   UInt32 log2n,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the value supplied as parameter log2n of the transform function.
signalStride

Specifies an address stride through input vector signal. Thus, to process every element, specify a stride of 1; to process every other element, specify 2. The value of signalStride should be 1 for best performance.
resultStride

Specifies an address stride through output vector result. Thus, to process every element, specify a stride of 1; to process every other element, specify 2. The value of resultStride should be 1 for best performance.
log2n

The base 2 exponent of the number of elements to process. For example, to process 1024 elements, specify 10 for parameter log2n. The value of log2n must be between 3 and 20, inclusive. See also the FFT Limitations sections in the Target chapters of the Developer’s Guide.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

This performs the operation




Forward transforms read real input and write packed complex output. Inverse transforms read packed complex input and write real output. As a result of packing the frequency-domain data, time-domain data and its equivalent frequency-domain data have the same storage requirements; see “Data Packing for Real FFTs”. Real data is stored in split complex form, with odd reals stored on the imaginary side of the split complex form and even reals in stored on the real side. 

Criteria to invoke vectorized code:
The input vectors signal.realp and signal.imagp and the output vectors result.realp and result.imagp must be aligned on 16-byte boundaries. 
The values of signalStride and resultStride must be 1. 
The value of log2n must be between 3 and 20, inclusive. 

If any of these criteria is not satisfied, the function invokes scalar code. 

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft_zrop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   DSPSplitComplex * result,
   SInt32 resultStride,
   UInt32 log2n,
   FFTDirection direction);

/*vDSP_conv

Performs either correlation or convolution on two vectors. 

void
vDSP_conv (const float signal[],
   SInt32signalStride,
   const floatfilter[],
   SInt32 filterStride,
   float result[],
   SInt32 resultStride,
   SInt32 lenResult,
   SInt32 lenFilter);
Discussion





If filterStride is positive, vDSP_conv performs correlation. If filterStride is negative, it performs convolution and *filter must point to the last vector element. The function can run in place, but result cannot be in place with filter. 

The value of lenFilter must be less than or equal to 2044. The length of vector signal must satisfy two criteria: it must be 
equal to or greater than 12
equal to or greater than the sum of N–1 plus the nearest multiple of 4 that is equal to or greater than the value of lenFilter. 

Criteria to invoke vectorized code:
The vectors signal and result must be relatively aligned.
The value of lenFilter must be between 4 and 256, inclusive.
The value of lenResult must be greater than 36.
The values of signalStride and resultStride must be 1.
The value of filterStride must be either 1 or –1.

If any of these criteria is not satisfied, the function invokes scalar code.
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_conv (const float signal[],
   SInt32 signalStride,
   const floatfilter[],
   SInt32 filterStride,
   float result[],
   SInt32 resultStride,
   SInt32 lenResult,
   SInt32 lenFilter);

/*vDSP_fft2d_zop

Computes an out-of-place complex discrete Fourier transform of the matrix represented by signal, either from the spatial domain to the frequency domain (forward) or from the frequency domain to the spatial domain (inverse).

void
vDSP_fft2d_zop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalRowStride,
   SInt32 signalColStride,
   DSPSplitComplex * result,
   SInt32 resultRowStride,
   SInt32 resultColStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the values supplied as parameters log2nInCol and log2nInRow of the transform function.
signalRowStride

Specifies a stride across each row of matrix a. Specifying 1 for signalRowStride processes every element across each row, specifying 2 processes every other element across each row, and so forth.
signalColStride

If not 0, this parameter represents the distance between each row of the input /output matrix. If parameter signalColStride is 1024, for instance, element 512 equates to element (1,0) of matrix a, element 1024 equates to element (2,0), and so forth.
colStride

Specifies a column stride for the input /output matrix, and should generally be allowed to default unless the input /output matrix is a submatrix. Parameter colStride can be defaulted by specifying 0. The default stride equals the row stride multiplied by the column count. Thus, if signalRowStride is 1 and signalColStride is 0, every element of matrix a is processed. If signalRowStride is 2 and signalColStride is 0, every other element of each row is processed.
resultRowStride

Specifies a row stride for output matrix result in the same way that signalRowStride specifies a stride for input the input /output matrix.
resultColStride

Specifies a column stride for output matrix result in the same way that signalColStride specifies a stride for input the input /output matrix.
log2nInCol

The base 2 exponent of the number of columns to process for each row. log2nInCol must be between 2 and 10, inclusive.
log2nInRow

The base 2 exponent of the number of rows to process. For example, to process 64 rows of 128 columns, specify 7 for log2nInCol and 6 for log2nInRow. log2nInRow must be between 2 and 10, inclusive.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

This performs the operation




Criteria to invoke vectorized code:
The input vectors signal.realp and signal.imagp, the output vectors result.realp and result.imagp, and the temporary vectors tempBuffer.realp and tempBuffer.imagp must be aligned on 16-byte boundaries. 
The values of signalRowStride and resultRowStride must be 1. 
The values of signalColStride and resultColStride must be multiples of 4. 
The values of log2nInRow and log2nInCol must be between 2 and 10, inclusive. 

If any of these criteria is not satisfied, the function invokes scalar code. 

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft2d_zop (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalRowStride,
   SInt32 signalColStride,
   DSPSplitComplex * result,
   SInt32 resultRowStride,
   SInt32 resultColStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);

/*vDSP_ctoz

Copies the contents of an interleaved complex vector C to a split complex vector Z.

void
vDSP_ctoz (const DSPComplex C[],
   SInt32 cStride,
   DSPSplitComplex * Z,
   SInt32 zStride,
   SInt32 size);
Discussion

Performs the operation




cStride is an address stride through cStride. zStride is an address stride through Z. The value of cStride must be a multiple of 2. 

For best performance, C.realp, C.imagp, Z.realp, and Z.imagp should be 16-byte aligned.

Criteria to invoke vectorized code:
The value of size must be greater than 3. 
The value of cStride must be 2. 
The value of zStride must be 1. 
Vectors Z.realp and Z.imagp must be relatively aligned.
Vector C must 8 bytes aligned if Z.realp and Z.imagp are 4 bytes aligned or C must be 16 bytes aligned if Z.realp and Z.imagp are at least 8 bytes aligned.

If any of these criteria is not satisfied, the function invokes scalar code.

See also functions“vDSP_ztoc” and “vDSP_ztocD”.
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_ctoz (const DSPComplex C[],
   SInt32 cStride,
   DSPSplitComplex * Z,
   SInt32 zStride,
   SInt32 size);

/*vDSP_fft2d_zrip

Computes an in-place real discrete Fourier transform, either from the spatial domain to the frequency domain (forward) or from the frequency domain to the spatial domain (inverse). 

void
vDSP_fft2d_zrip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 rowStride,
   SInt32 colStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);
   
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the values supplied as parameters log2nInCol and log2nInRow of the transform function.
rowStride

Specifies a stride across each row of the input matrix signal. Specifying 1 for rowStride processes every element across each row, specifying 2 processes every other element across each row, and so forth.
colStride

Specifies a column stride for the matrix, and should generally be allowed to default unless the matrix is a submatrix. Parameter colStride can be defaulted by specifying 0. The default column stride equals the row stride multiplied by the column count. Thus, if rowStride is 1 and colStride is 0, every element of the input /output matrix is processed. If rowStride is 2 and colStride is 0, every other element of each row is processed.

If not 0, colStride represents the distance between each row of the matrix. If colStride is 1024, for instance, complex element 512 of the matrix equates to element (1,0), element 1024 equates to element (2,0), and so forth.
log2nInCol

The base 2 exponent of the number of columns to process for each row. log2nInCol must be between 2 and 10, inclusive.
log2nInRow

The base 2 exponent of the number of rows to process. For example, to process 64 rows of 128 columns, specify 7 for log2nInCol and 6 for log2nInRow. log2nInRow must be between 2 and 10, inclusive.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

Forward transforms read real input and write packed complex output. Inverse transforms read packed complex input and write real output. As a result of packing the frequency-domain data, spatial-domain data and its equivalent frequency-domain data have the same storage requirements; see “Data Packing for Real FFTs”. 




Real data is stored in split complex form, with odd reals stored on the imaginary side of the split complex form and even reals in stored on the real side.int nr, nc, tempSize;
nr = 1<<log2InRow;
nc = 1<<log2InCol;
if ( ( (log2InCol-1) < 3 ) || ( log2InRow > 9)
{
    tempSize = 9 * nr;
}
else
{
    tempSize = 17 * nr
}
tempBuffer.realp = ( float* ) malloc (tempSize * sizeOf ( float ) );
tempBuffer.imagp = ( float* ) malloc (tempSize * sizeOf ( float ) );


Criteria to invoke vectorized code:
The input/output vectors signal.realp and signal.imagp and the temporary vectors tempBuffer.realp and tempBuffer.imagp must be aligned on 16-byte boundaries. 
The value of rowStride must be 1. 
The value of colStride must be a multiple of 4. 
The value of log2nInRow and log2nInCol must be between 3 and 10, inclusive. 

If any of these criteria is not satisfied, the function invokes scalar code. 

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft2d_zrip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 rowStride,
   SInt32 colStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);

/*vDSP_fft2d_zip

Computes an in-place complex discrete Fourier transform of matrix represented by signal, either from the spatial domain to the frequency domain (forward) or from the frequency domain to the spatial domain (inverse). 

void
vDSP_fft2d_zip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 rowStride,
   SInt32 colStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the values supplied as parameters log2nInCol and log2nInRow of the transform function.
rowStride

Specifies a stride across each row of the matrix signal. Specifying 1 for rowStride processes every element across each row, specifying 2 processes every other element across each row, and so forth.
colStride

Specifies a column stride for the matrix, and should generally be allowed to default unless the matrix is a submatrix. Parameter colStride can be defaulted by specifying 0. The default column stride equals the row stride multiplied by the column count. Thus, if rowStride is 1 and colStride is 0, every element of the input /output matrix is processed. If rowStride is 2 and colStride is 0, every other element of each row is processed.

If not 0, parameter colStride represents the distance between each row of the matrix. If parameter colStride is 1024, for instance, complex element 512 of the matrix equates to element (1,0), element 1024 equates to element (2,0), and so forth.
log2nInCol

The base 2 exponent of the number of columns to process for each row. log2nInCol must be between 2 and 10, inclusive.
log2nInRow

The base 2 exponent of the number of rows to process. For example, to process 64 rows of 128 columns, specify 7 for parameter log2nInCol and 6 for parameter log2nInRow. log2nInRow must be between 2 and 10, inclusive. See also the FFT Limitations sections in the Target chapters of the Developer’s Guide.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

This performs the operation




Criteria to invoke vectorized code:

No altivec support for double precision. The function always invokes scalar code.

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft2d_zip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 rowStride,
   SInt32 colStride,
   UInt32 log2nInCol,
   UInt32 log2nInRow,
   FFTDirection direction);

/*vDSP_fft_zip

Computes an in-place complex discrete Fourier transform of the input/output vector signal, either from the time domain to the frequency domain (forward) or from the frequency domain to the time domain (inverse).

void
vDSP_fft_zip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   UInt32 log2n,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to the FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD.The value supplied as parameter log2n of the earlier call to the setup function must equal or exceed the value supplied as parameter log2n of the transform function. 
signalStride

Specifies an address stride through the input/output vector signal. To process every element of the vector, specify 1 for parameter signalStride; to process every other element, specify 2. The value of signalStride should be 1 for best performance.
log2n

The base 2 exponent of the number of elements to process. For example, to process 1024 elements, specify 10 for parameter log2n. The value of log2n must be between 2 and 20, inclusive.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

This performs the operation




Criteria to invoke vectorized code:
Input/output vectors tempBuffer.realp and tempBuffer.imagp and temporary vectors tempBuffer.realp and tempBuffer.imagp must be aligned on 16-byte boundaries. 
The value of signalStride must be 1. 

If any of these criteria is not satisfied, the function invokes scalar code.

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft_zip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   UInt32 log2n,
   FFTDirection direction);

/*vDSP_fft_zrip

Computes an in-place real discrete Fourier transform, either from the time domain to the frequency domain (forward) or from the frequency domain to the time domain (inverse). 

void
vDSP_fft_zrip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   UInt32 log2n,
   FFTDirection direction);
Parameters
setup

Points to a structure initialized by a prior call to FFT weights array function vDSP_create_fftsetup or vDSP_create_fftsetupD. The value supplied as parameter log2n of the setup function must equal or exceed the value supplied as parameter log2n of the transform function.
signalStride

Specifies an address stride through the input/output vector . To process every element of the vector, specify 1 for parameter signalStride; to process every other element, specify 2. The value of signalStride should be 1 for best performance.
log2n

The base 2 exponent of the number of elements to process. For example, to process 1024 elements, specify 10 for parameter log2n. The value of log2n must be between 3 and 20, inclusive. See also the FFT Limitations sections in the Target chapters of the Developer’s Guide.
direction

A forward/inverse directional flag, and must specify one of the following:
FFT_FORWARD

Forward transform
FFT_INVERSE

Inverse transform

Results are undefined for other values of direction.
Discussion

Forward transforms read real input and write packed complex output. Inverse transforms read packed complex input and write real output. As a result of packing the frequency-domain data, time-domain data and its equivalent frequency-domain data have the same storage requirements.




Real data is stored in split complex form, with odd reals stored on the imaginary side of the split complex form and even reals in stored on the real side. See the section “Data Packing for Real FFTs”. 

Criteria to invoke vectorized code:
The input/output vectors signal.realp and signal.imagp must be aligned on 16-byte boundaries. 
The value of signalStride must be 1. 
The value of log2n must be between 3 and 20, inclusive. 

If any of these criteria is not satisfied, the function invokes scalar code.

See also functions “vDSP_create_fftsetup”, “vDSP_create_fftsetupD”, “vDSP_destroy_fftsetup”, “vDSP_destroy_fftsetupD”, and Chapter 2, “Using Fourier Transforms.” 
Availability
Available in Mac OS X v10.4 and later.*/
void
vDSP_fft_zrip (FFTSetup setup,
   DSPSplitComplex * signal,
   SInt32 signalStride,
   UInt32 log2n,
   FFTDirection direction);


//#include <HYS/vecLib/vDSP_E.h>

#ifdef __cplusplus
}
#endif


#endif 
