#ifndef CFRUNLOOP_H
#define CFRUNLOOP_H

#ifndef CFBASE_H
#include <CoreFoundation/CFBase.h>
#endif

#ifndef CFARRAY_H
#include <CoreFoundation/CFArray.h>
#endif

#ifndef CFDATE_H
#include <CoreFoundation/CFDate.h>
#endif

#ifndef CFSTRING_H
#include <CoreFoundation/CFString.h>
#endif


#include <mach/port.h>


#if defined(__cplusplus)
extern "C" {
#endif

/*CFRunLoopTimerRef

A reference to a run loop timer object.

typedef struct __CFRunLoopTimer *CFRunLoopTimerRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFRunLoopTimer *CFRunLoopTimerRef;

/*Default Run Loop Mode

Default run loop mode.

const CFStringRef kCFRunLoopDefaultMode;
Constants
kCFRunLoopDefaultMode

Run loop mode that should be used when a thread is in its default, or idle, state, waiting for an event. This mode is used when the run loop is started with CFRunLoopRun.

Available in Mac OS X v10.0 and later.*/

extern const CFStringRef kCFRunLoopDefaultMode;

/*Common Mode Flag

A run loop pseudo-mode that manages objects monitored in the “common” modes.

const CFStringRef kCFRunLoopCommonModes;
Constants
kCFRunLoopCommonModes

Objects added to a run loop using this value as the mode are monitored by all run loop modes that have been declared as a member of the set of “common” modes with CFRunLoopAddCommonMode.

Available in Mac OS X v10.0 and later.
Discussion

Run loops never run in this mode. This pseudo-mode is used only as a special set of sources, timers, and observers that is shared by other modes. See “Managing Objects in a Run Loop” for more details.*/
extern const CFStringRef kCFRunLoopCommonModes;

/*CFRunLoopRef

A reference to a run loop object.

typedef struct __CFRunLoop *CFRunLoopRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFRunLoop *CFRunLoopRef;


/*CFRunLoopSourceRef

A reference to a run loop source object.

typedef struct __CFRunLoopSource *CFRunLoopSourceRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFRunLoopSource *CFRunLoopSourceRef;

/*CFRunLoopEqualCallBack

Callback invoked to test two CFRunLoopSource objects for equality.

typedef Boolean (*CFRunLoopEqualCallBack) (
   const void *info1,
   const void *info2
);

If you name your function MyCallBack, you would declare it like this:

Boolean MyCallBack (
   const void *info1,
   const void *info2
);
Parameters
info1

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the first run loop source to test.
info2

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the second run loop source to test.
Return Value

true if info1 and info2 should be considered equal; otherwise false. 
Discussion

You specify this callback in the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure when creating the run loop source.*/
typedef Boolean (*CFRunLoopEqualCallBack) (
   const void *info1,
   const void *info2
);

/*CFRunLoopHashCallBack

Callback invoked to compute a hash code for the info pointer of a CFRunLoopSource object.

typedef CFHashCode (*CFRunLoopHashCallBack) (
   const void *info
);

If you name your function MyCallBack, you would declare it like this:

CFHashCode MyCallBack (
   const void *info
);
Parameters
info

The info member of the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure that was used when creating the run loop source.
Return Value

A hash code value for info. 
Discussion

If a hash callback is not provided for a source, the info pointer is used.

You specify this callback in the CFRunLoopSourceContext or CFRunLoopSourceContext1 structure when creating the run loop source.*/

typedef CFHashCode (*CFRunLoopHashCallBack) (
   const void *info
);

/*CFRunLoopScheduleCallBack

Callback invoked when a version 0 CFRunLoopSource object is added to a run loop mode.

typedef void (*CFRunLoopScheduleCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
rl

The run loop in which the source is being scheduled. 
mode

The run loop mode in which the source is being scheduled. 
Discussion

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopScheduleCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

/*CFRunLoopCancelCallBack

Callback invoked when a version 0 CFRunLoopSource object is removed from a run loop mode.

typedef void (*CFRunLoopCancelCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
rl

The run loop from which the run loop source is being removed. 
mode

The run loop mode from which the run loop source is being removed. 
Discussion

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopCancelCallBack) (
   void *info,
   CFRunLoopRef rl,
   CFStringRef mode
);

/*CFRunLoopPerformCallBack

Callback invoked when a message is received on a version 0 CFRunLoopSource object.

typedef void (*CFRunLoopPerformCallBack) (
   void *info
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   void *info
);
Parameters
info

The info member of the CFRunLoopSourceContext structure that was used when creating the run loop source.
Discussion

You only need to provide this callback if you create your own version 0 run loop source. CFSocket run loop sources already implement this callback to forward the received message to the CFSocket’s own callback function, which you do need to implement.

You specify this callback in the CFRunLoopSourceContext structure when creating the run loop source.*/
typedef void (*CFRunLoopPerformCallBack) (
   void *info
);


/*CFRunLoopSourceContext

A structure that contains program-defined data and callbacks with which you can configure a version 0 CFRunLoopSource’s behavior.

struct CFRunLoopSourceContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
   CFRunLoopEqualCallBack equal;
   CFRunLoopHashCallBack hash;
   CFRunLoopScheduleCallBack schedule;
   CFRunLoopCancelCallBack cancel;
   CFRunLoopPerformCallBack perform;
};
typedef struct CFRunLoopSourceContext CFRunLoopSourceContext;
Fields
version

Version number of the structure. Must be 0. 
info

An arbitrary pointer to program-defined data, which can be associated with the CFRunLoopSource at creation time. This pointer is passed to all the callbacks defined in the context.
retain

A retain callback for your program-defined info pointer. Can be NULL.
release

A release callback for your program-defined info pointer. Can be NULL. 
copyDescription

A copy description callback for your program-defined info pointer. Can be NULL. 
equal

An equality test callback for your program-defined info pointer. Can be NULL.
hash

A hash calculation callback for your program-defined info pointer. Can be NULL.
schedule

A scheduling callback for the run loop source. This callback is called when the source is added to a run loop mode. Can be NULL. 
cancel

A cancel callback for the run loop source. This callback is called when the source is removed from a run loop mode. Can be NULL. 
perform

A perform callback for the run loop source. This callback is called when the source has fired.*/
struct CFRunLoopSourceContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
   CFRunLoopEqualCallBack equal;
   CFRunLoopHashCallBack hash;
   CFRunLoopScheduleCallBack schedule;
   CFRunLoopCancelCallBack cancel;
   CFRunLoopPerformCallBack perform;
};
typedef struct CFRunLoopSourceContext CFRunLoopSourceContext;

/*CFRunLoopTimerContext

A structure that contains program-defined data and callbacks with which you can configure a CFRunLoopTimer’s behavior.

struct CFRunLoopTimerContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
};
typedef struct CFRunLoopTimerContext CFRunLoopTimerContext;
Fields
version

Version number of the structure. Must be 0.
info

An arbitrary pointer to program-defined data, which can be associated with the run loop timer at creation time. This pointer is passed to all the callbacks defined in the context.
retain

A retain callback for your program-defined info pointer. Can be NULL. 
release

A release callback for your program-defined info pointer. Can be NULL. 
copyDescription

A copy description callback for your program-defined info pointer. Can be NULL.*/
struct CFRunLoopTimerContext {
   CFIndex version;
   void *info;
   CFAllocatorRetainCallBack retain;
   CFAllocatorReleaseCallBack release;
   CFAllocatorCopyDescriptionCallBack copyDescription;
};
typedef struct CFRunLoopTimerContext CFRunLoopTimerContext;

/*CFRunLoopTimerCallBack

Callback invoked when a CFRunLoopTimer object fires.

typedef void (*CFRunLoopTimerCallBack) (
   CFRunLoopTimerRef timer,
   void *info
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   CFRunLoopTimerRef timer,
   void *info
);
Parameters
timer

The run loop timer that is firing. 
info

The info member of the CFRunLoopTimerContext structure that was used when creating the run loop timer.
Discussion

If timer repeats, the run loop automatically schedules the next firing time after calling this function, unless you manually update the firing time within this callback by calling CFRunLoopTimerSetNextFireDate. If timer does not repeat, the run loop invalidates timer.

You specify this callback when you create the timer with CFRunLoopTimerCreate.*/
typedef void (*CFRunLoopTimerCallBack) (
   CFRunLoopTimerRef timer,
   void *info
);


//function

/*CFRunLoopSourceCreate

Creates a CFRunLoopSource object.

CFRunLoopSourceRef CFRunLoopSourceCreate (
   CFAllocatorRef allocator,
   CFIndex order,
   CFRunLoopSourceContext *context
);
Parameters
allocator

The allocator to use to allocate memory for the new object. Pass NULL or kCFAllocatorDefault to use the current default allocator.
order

A priority index indicating the order in which run loop sources are processed. When multiple run loop sources are firing in a single pass through the run loop, the sources are processed in increasing order of this parameter. If the run loop is set to process only one source per loop, only the highest priority source, the one with the lowest order value, is processed. This value is ignored for version 1 sources. Pass 0 unless there is a reason to do otherwise.
context

A structure holding contextual information for the run loop source. The function copies the information out of the structure, so the memory pointed to by context does not need to persist beyond the function call.
Return Value

The new CFRunLoopSource object. You are responsible for releasing this object.
Discussion

The run loop source is not automatically added to a run loop. Ownership follows the Create Rule.
Availability
Available in Mac OS X v10.0 and later.*/
CFRunLoopSourceRef CFRunLoopSourceCreate (
   CFAllocatorRef allocator,
   CFIndex order,
   CFRunLoopSourceContext *context
);

/*CFRunLoopAddSource

Adds a CFRunLoopSource object to a run loop mode.

void CFRunLoopAddSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);
Parameters
rl

The run loop to modify. 
source

The run loop source to add. 
mode

The run loop mode to which to add source. Use the constant kCFRunLoopCommonModes to add source to the set of objects monitored by all the common modes. 
Discussion

If source is a version 0 source, this function calls the schedule callback function specified in the context structure for source. See CFRunLoopSourceContext for more details.

A run loop source can be registered in multiple run loops and run loop modes at the same time. When the source is signaled, whichever run loop that happens to detect the signal first will fire the source.

If rl already contains source in mode, this function does nothing.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopAddSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);

/*CFRunLoopGetCurrent

Returns the CFRunLoop object for the current thread.

CFRunLoopRef CFRunLoopGetCurrent ();
Return Value

Current thread’s run loop. Ownership follows the Get Rule.
Discussion

Each thread has exactly one run loop associated with it.
Availability
Available in Mac OS X v10.0 and later.*/
CFRunLoopRef CFRunLoopGetCurrent ();

/*CFRunLoopRun

Runs the current thread’s CFRunLoop object in its default mode indefinitely.

void CFRunLoopRun ();
Discussion

The current thread’s run loop runs in the default mode (see "Default Run Loop Mode") until the run loop is stopped with CFRunLoopStop or all the sources and timers are removed from the default run loop mode.

Run loops can be run recursively. You can call CFRunLoopRun from within any run loop callout and create nested run loop activations on the current thread’s call stack.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopRun ();

/*CFRunLoopStop

Forces a CFRunLoop object to stop running.

void CFRunLoopStop (
   CFRunLoopRef rl
);
Parameters
rl

The run loop to stop.
Discussion

This function forces rl to stop running and return control to the function that called CFRunLoopRun or CFRunLoopRunInMode for the current run loop activation. If the run loop is nested with a callout from one activation starting another activation running, only the innermost activation is exited.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopStop (
   CFRunLoopRef rl
);

/*CFRunLoopSourceInvalidate

Invalidates a CFRunLoopSource object, stopping it from ever firing again.

void CFRunLoopSourceInvalidate (
   CFRunLoopSourceRef source
);
Parameters
source

The run loop source to invalidate.
Discussion

Once invalidated, source will never fire and call its perform callback function again. This function automatically removes source from all the run loop modes in which it was registered. If source is a version 0 source, this function calls its cancel callback function as it is removed from each run loop mode. The memory for source is not deallocated unless the run loop held the only reference to source.
Availability
Available in Mac OS X v10.0 and later.*/

void CFRunLoopSourceInvalidate (
   CFRunLoopSourceRef source
);

/*CFRunLoopGetTypeID

Returns the type identifier for the CFRunLoop opaque type.

CFTypeID CFRunLoopGetTypeID ();
Return Value

The type identifier for the CFRunLoop opaque type.
Availability
Available in Mac OS X v10.0 and later.*/

CFTypeID CFRunLoopGetTypeID ();

/*CFRunLoopRunInMode

Runs the current thread’s CFRunLoop object in a particular mode.

SInt32 CFRunLoopRunInMode (
   CFStringRef mode,
   CFTimeInterval seconds,
   Boolean returnAfterSourceHandled
);
Parameters
mode

The run loop mode to run. mode can be any arbitrary CFString. You do not need to explicitly create a run loop mode, although a run loop mode needs to contain at least one source or timer to run.
seconds

The length of time to run the run loop. If 0, only one pass is made through the run loop before returning; if multiple sources or timers are ready to fire immediately, only one (possibly two if one is a version 0 source) will be fired, regardless of the value of returnAfterSourceHandled.
returnAfterSourceHandled

A flag indicating whether the run loop should exit after processing one source. If false, the run loop continues processing events until seconds has passed.
Return Value

A value indicating the reason the run loop exited. Possible values are described below.
Discussion

Run loops can be run recursively. You can call CFRunLoopRunInMode from within any run loop callout and create nested run loop activations on the current thread’s call stack. You are not restricted in which modes you can run from within a callout. You can create another run loop activation running in any available run loop mode, including any modes already running higher in the call stack.

The run loop exits with the following return values under the indicated conditions:
kCFRunLoopRunFinished. The run loop mode mode has no sources or timers.
kCFRunLoopRunStopped. The run loop was stopped with CFRunLoopStop.
kCFRunLoopRunTimedOut. The time interval seconds passed.
kCFRunLoopRunHandledSource. A source was processed. This exit condition only applies when returnAfterSourceHandled is true.
Availability
Available in Mac OS X v10.0 and later.*/
SInt32 CFRunLoopRunInMode (
   CFStringRef mode,
   CFTimeInterval seconds,
   Boolean returnAfterSourceHandled
);

/*CFRunLoopRemoveSource

Removes a CFRunLoopSource object from a run loop mode.

void CFRunLoopRemoveSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);
Parameters
rl

The run loop to modify. 
source

The run loop source to remove. 
mode

The run loop mode of rl from which to remove source. Use the constant kCFRunLoopCommonModes to remove source from the set of objects monitored by all the common modes. 
Discussion

If source is a version 0 source, this function calls the cancel callback function specified in the context structure for source. See CFRunLoopSourceContext and CFRunLoopSourceContext1for more details.

If rl does not contain source in mode, this function does nothing.
Availability
Available in Mac OS X v10.0 and later.*/

void CFRunLoopRemoveSource (
   CFRunLoopRef rl,
   CFRunLoopSourceRef source,
   CFStringRef mode
);

/*CFRunLoopTimerInvalidate

Invalidates a CFRunLoopTimer object, stopping it from ever firing again.

void CFRunLoopTimerInvalidate (
   CFRunLoopTimerRef timer
);
Parameters
timer

The run loop timer to invalidate. 
Discussion

Once invalidated, timer will never fire and call its callback function again. This function automatically removes timer from all run loop modes in which it had been added. The memory is not deallocated unless the run loop held the only reference to timer.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopTimerInvalidate (
   CFRunLoopTimerRef timer
);

/*CFRunLoopAddTimer

Adds a CFRunLoopTimer object to a run loop mode.

void CFRunLoopAddTimer (
   CFRunLoopRef rl,
   CFRunLoopTimerRef timer,
   CFStringRef mode
);
Parameters
rl

The run loop to modify. 
timer

The run loop timer to add. 
mode

The run loop mode of rl to which to add timer. Use the constant kCFRunLoopCommonModes to add timer to the set of objects monitored by all the common modes. 
Discussion

A run loop timer can be registered in only one run loop at a time, although it can be added to multiple run loop modes within that run loop.

If rl already contains timer in mode, this function does nothing.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopAddTimer (
   CFRunLoopRef rl,
   CFRunLoopTimerRef timer,
   CFStringRef mode
);

/*CFRunLoopTimerCreate

Creates a new CFRunLoopTimer object.

CFRunLoopTimerRef CFRunLoopTimerCreate (
   CFAllocatorRef allocator,
   CFAbsoluteTime fireDate,
   CFTimeInterval interval,
   CFOptionFlags flags,
   CFIndex order,
   CFRunLoopTimerCallBack callout,
   CFRunLoopTimerContext *context
);
Parameters
allocator

The allocator to use to allocate memory for the new object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
fireDate

The time at which the timer should first fire. The fine precision (sub-millisecond at most) of the fire date may be adjusted slightly by the timer if there are implementation reasons to do.
interval

The firing interval of the timer. If 0 or negative, the timer fires once and then is automatically invalidated. The fine precision (sub-millisecond at most) of the interval may be adjusted slightly by the timer if implementation reasons to do so exist.
flags

Currently ignored. Pass 0 for future compatibility.
order

A priority index indicating the order in which run loop timers are processed. Run loop timers currently ignore this parameter. Pass 0.
callout

The callback function that is called when the timer fires. 
context

A structure holding contextual information for the run loop timer. The function copies the information out of the structure, so the memory pointed to by context does not need to persist beyond the function call. Can be NULL if the callback function does not need the context’s info pointer to keep track of state. 
Return Value

The new CFRunLoopTimer object. Ownership follows the Create Rule.
Discussion

A timer needs to be added to a run loop mode before it will fire. To add the timer to a run loop, use CFRunLoopAddTimer. A timer can be registered to only one run loop at a time, although it can be in multiple modes within that run loop.
Availability
Available in Mac OS X v10.0 and later.*/
CFRunLoopTimerRef CFRunLoopTimerCreate (
   CFAllocatorRef allocator,
   CFAbsoluteTime fireDate,
   CFTimeInterval interval,
   CFOptionFlags flags,
   CFIndex order,
   CFRunLoopTimerCallBack callout,
   CFRunLoopTimerContext *context
);

/*CFRunLoopTimerSetNextFireDate

Sets the next firing date for a CFRunLoopTimer object .

void CFRunLoopTimerSetNextFireDate (
   CFRunLoopTimerRef timer,
   CFAbsoluteTime fireDate
);
Parameters
timer

The run loop timer to modify. 
fireDate

The new firing time for timer. 
Discussion

Resetting a timer’s next firing time is a relatively expensive operation and should not be done if it can be avoided; letting timers autorepeat is more efficient. In some cases, however, manually-adjusted, repeating timers are useful. For example, if you have an action that will be performed multiple times in the future, but at irregular time intervals, it would be very expensive to create, add to run loop modes, and then destroy a timer for each firing event. Instead, you can create a repeating timer with an initial firing time in the distant future (or the initial firing time) and a very large repeat interval—on the order of decades or more—and add it to all the necessary run loop modes. Then, when you know when the timer should fire next, you reset the firing time with CFRunLoopTimerSetNextFireDate, perhaps from the timer’s own callback function. This technique effectively produces a reusable, asynchronous timer.
Availability
Available in Mac OS X v10.0 and later.*/
void CFRunLoopTimerSetNextFireDate (
   CFRunLoopTimerRef timer,
   CFAbsoluteTime fireDate
);

/*CFRunLoopTimerGetInterval

Returns the firing interval of a repeating CFRunLoopTimer object.

CFTimeInterval CFRunLoopTimerGetInterval (
   CFRunLoopTimerRef timer
);
Parameters
timer

The run loop timer to examine. 
Return Value

The firing interval of timer. Returns 0 if timer does not repeat. 
Availability
Available in Mac OS X v10.0 and later.*/
CFTimeInterval CFRunLoopTimerGetInterval (
   CFRunLoopTimerRef timer
);


#ifdef __cplusplus
}
#endif

#endif