#ifndef CGPATH_H
#define CGPATH_H

#include <CoreGraphics/CGBase.h>
#include <CoreGraphics/CGAffineTransform.h>
#include <CoreFoundation/CFBase.h>

/*CGPathRef

Defines an opaque type that represents an immutable graphics path.

   
typedef const struct CGPath *CGPathRef;
Discussion

For a description of graphics paths, see CGMutablePathRef.

In Mac OS X version 10.2 and later, CGPathRef is derived from CFTypeRef and inherits the properties that all Core Foundation types have in common. For more information, see CFType Reference.
Availability
Available in Mac OS X v10.2 and later.*/
typedef const struct CGPath *CGPathRef;

/*Path Drawing Modes

Specify ways of rendering a path.

enum CGPathDrawingMode {
   kCGPathFill,
   kCGPathEOFill,
   kCGPathStroke,
   kCGPathFillStroke,
   kCGPathEOFillStroke
};
typedef enum CGPathDrawingMode CGPathDrawingMode;
Constants
kCGPathFill

Render the area contained within the path using the non-zero winding number rule. 

Available in Mac OS X v10.0 and later.
kCGPathEOFill

Render the area within the path using the even-odd rule. 

Available in Mac OS X v10.0 and later.
kCGPathStroke

Render a line along the path. 

Available in Mac OS X v10.0 and later.
kCGPathFillStroke

First fill and then stroke the path, using the nonzero winding number rule.

Available in Mac OS X v10.0 and later.
kCGPathEOFillStroke

First fill and then stroke the path, using the even-odd rule. 

Available in Mac OS X v10.0 and later.
Discussion

To draw a shape, you set the current path to that shape and then paint the shape by stroking, filling, or both stroking and filling. Stroking is the process of painting a line along the current path. Filling is the process of painting the area contained within the path.

When you fill the current path, Quartz fills each subpath independently. Any subpath that has not been explicitly closed is closed implicitly by the fill routines. There are two different methods of filling paths: the nonzero winding number rule and the even-odd rule.

The default fill rule is called the nonzero winding number rule. To understand how this rule works, draw a line in any direction, starting from a given point and moving to infinity. Starting with a count of 0, add one to the count every time a path segment crosses the line from left to right, and subtract one every time a path segment crosses the line from right to left. If the result is 0, the point is outside the path and is not filled.

Alternately, using the even-odd rule, you draw a line in any direction, starting from a given point and moving to infinity. Count the number of path segments that the line crosses. If the result is odd, the point is inside the path and therefore filled. If the result is even, the point is outside the path and is not filled.

You can pass a path drawing mode constant to the function CGContextDrawPath to specify how Quartz should paint a graphics context’s current path.*/
enum CGPathDrawingMode {
   kCGPathFill,
   kCGPathEOFill,
   kCGPathStroke,
   kCGPathFillStroke,
   kCGPathEOFillStroke
};
typedef enum CGPathDrawingMode CGPathDrawingMode;

/*CGMutablePathRef

Defines an opaque type that represents a mutable graphics path.

   
typedef struct CGPath *CGMutablePathRef;
Discussion

A graphics path is a description of a 2D geometric scene using sequences of lines and Bézier curves. CGPathRef defines an opaque type that represents an immutable graphics path. CGMutablePathRef defines an opaque type that represents a mutable graphics path.To draw using a Quartz path, you need to add the path to a graphics context—see CGContextAddPath.

Each figure in a scene may be described by a subpath. A subpath has an ordered set of path elements, which represent single steps in the construction of a subpath (see “CGPathElement”). A subpath also maintains state information, including a starting point and a current point. When drawing a path, Quartz traverses each subpath using its path elements and its state. 

For example, a subpath for a rectangle might contain the path elements shown in Table 1.
Table 1  Path elements in a rectangular subpath
Path element	
Description

MoveToPoint (bottom left)	
Starts a new subpath; initializes the starting point and the current point.

AddLineToPoint (bottom right)	
Adds a line segment from the current point to the new point; updates the current point.

AddLineToPoint (top right)	
Adds a line segment from the current point to the new point; updates the current point.

AddLineToPoint (top left)	
Adds a line segment from the current point to the new point; updates the current point.

CloseSubpath	
Draws a line segment from the current point to the starting point.


The lines and curves in a subpath are always connected, but they do not necessarily form a closed figure. Furthermore, subpaths do not need to be connected to each other. For example, you could use a graphics path to draw the outlines of a sequence of text characters. 

In Mac OS X version 10.2 and later, CGMutablePathRef is derived from CFTypeRef and inherits the properties that all Core Foundation types have in common. For more information, see CFType Reference.
Availability
Available in Mac OS X v10.2 and later.*/
typedef struct CGPath *CGMutablePathRef;

/*Path Element Types

Specify the type of element found in a path.

enum CGPathElementType {
   kCGPathElementMoveToPoint,
   kCGPathElementAddLineToPoint,
   kCGPathElementAddQuadCurveToPoint,
   kCGPathElementAddCurveToPoint,
   kCGPathElementCloseSubpath
};
typedef enum CGPathElementType CGPathElementType;
Constants
kCGPathElementMoveToPoint

The path element that starts a new subpath. See the function CGPathMoveToPoint.

Available in Mac OS X v10.2 and later.
kCGPathElementAddLineToPoint

The path element that adds a line from the current point to the specified point. See the function CGPathAddLineToPoint.

Available in Mac OS X v10.2 and later.
kCGPathElementAddQuadCurveToPoint

The path element that adds a quadratic curve from the current point to the specified point. See the function CGPathAddQuadCurveToPoint.

Available in Mac OS X v10.2 and later.
kCGPathElementAddCurveToPoint

The path element that adds a cubic curve from the current point to the specified point. See the function CGPathAddCurveToPoint.

Available in Mac OS X v10.2 and later.
kCGPathElementCloseSubpath

The path element that closes and completes a subpath. See the function CGPathCloseSubpath.

Available in Mac OS X v10.2 and later.
Discussion

For more information about paths, see CGPathRef.*/

enum CGPathElementType {
   kCGPathElementMoveToPoint,
   kCGPathElementAddLineToPoint,
   kCGPathElementAddQuadCurveToPoint,
   kCGPathElementAddCurveToPoint,
   kCGPathElementCloseSubpath
};
typedef enum CGPathElementType CGPathElementType;


/*CGPathElement

Defines the data structure used by Quartz to provide information about a path element.

struct CGPathElement {
   CGPathElementType type;
   CGPoint *points;
};
typedef struct CGPathElement CGPathElement;
Discussion

A path element consists of an element type (or operation), together with an array of one or more points that serve as arguments. 
Availability
Available in Mac OS X v10.2 and later.*/
struct CGPathElement {
   CGPathElementType type;
   CGPoint *points;
};
typedef struct CGPathElement CGPathElement;

/*CGPathApplierFunction

Defines a callback function that can view an element in a graphics path.

typedef void (*CGPathApplierFunction) (
   void *info,
   const CGPathElement *element
);
   

If you name your function MyCGPathApplierFunc, you would declare it like this:

void MyCGPathApplierFunc (
   void *info,
   const CGPathElement *element
);
Discussion

See also CGPathApply.*/

typedef void (*CGPathApplierFunction) (
   void *info,
   const CGPathElement *element
);


//function

/*CGPathApply

For each element in a Quartz path, calls a custom applier function.

void CGPathApply (
   CGPathRef path,
   void *info,
   CGPathApplierFunction function
);
Parameters
path

The path to which the function will be applied.
info

A pointer to the user data that Quartz will pass to the function being applied, or NULL.
function

A pointer to the function to apply. See CGPathApplierFunction for more information.
Discussion

For each element in the specified path, Quartz calls the applier function, which can examine (but not modify) the element.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/
void CGPathApply (
   CGPathRef path,
   void *info,
   CGPathApplierFunction function
);

/*CGPathRelease

Decrements the retain count of a Quartz path.

void CGPathRelease (
   CGPathRef path
);
Parameters
path

The Quartz path to release.
Discussion

This function is equivalent to CFRelease, except that it does not cause an error if the path parameter is NULL.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/
void CGPathRelease (
   CGPathRef path
);

/*CGPathAddLineToPoint

Appends a line segment to a mutable Quartz path.

void CGPathAddLineToPoint (
   CGMutablePathRef path,
   const CGAffineTransform *m,
   float x,
   float y
);
Parameters
path

The mutable path to change. The path must not be empty.
m

A pointer to an affine transformation matrix, or NULL if no transformation is needed. If specified, Quartz applies the transformation to the line before it is added to the path.
x

The x-coordinate of the end point of the line.
y

The y-coordinate of the end point of the line.
Discussion

Before returning, this function updates the current point to the specified location (x,y).
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/
void CGPathAddLineToPoint (
   CGMutablePathRef path,
   const CGAffineTransform *m,
   float x,
   float y
);

/*CGPathMoveToPoint

Starts a new subpath at a specified location in a mutable Quartz path.

void CGPathMoveToPoint (
   CGMutablePathRef path,
   const CGAffineTransform *m,
   float x,
   float y
);
Parameters
path

The mutable path to change.
m

A pointer to an affine transformation matrix, or NULL if no transformation is needed. If specified, Quartz applies the transformation to the point before changing the path.
x

The x-coordinate of the new location.
y

The y-coordinate of the new location.
Discussion

This function initializes the starting point and the current point to the specified location (x,y) after an optional transformation.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/

void CGPathMoveToPoint (
   CGMutablePathRef path,
   const CGAffineTransform *m,
   float x,
   float y
);

/*CGPathCreateMutable

Creates a mutable Quartz graphics path.

CGMutablePathRef CGPathCreateMutable (void);
Return Value

A new mutable path. You are responsible for releasing this object.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/
CGMutablePathRef CGPathCreateMutable (void);

/*CGPathCloseSubpath

Closes and completes a subpath in a mutable Quartz path.

void CGPathCloseSubpath (
   CGMutablePathRef path
);
Parameters
path

The path to change.
Discussion

Appends a line from the current point in a path to the starting point of the current subpath and ends the subpath. On return, the current point is now the previous starting point.
Availability
Available in Mac OS X version 10.2 and later.
Declared In
CGPath.h*/
void CGPathCloseSubpath (
   CGMutablePathRef path
);


#endif