#include <PrintCore/PrintCore.h>

/*PMCreateSession

Creates and initializes a printing session object and creates a context for printing operations.

OSStatus PMCreateSession (
   PMPrintSession *printSession
);
Parameters
printSession

A pointer to your PMPrintSession variable. On return, the variable refers to a new printing session object. You should call the function PMRelease to release the printing session object when you no longer need it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function allocates memory for a new printing session object in your application’s memory space and sets its reference count to 1. The new printing session object is initialized with information that the printing system uses for a print job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMCreateSession (
   PMPrintSession *printSession
){};

/*PMPrinterPrintWithFile

Submits a print job to a specified printer using a file that contains print data.

OSStatus PMPrinterPrintWithFile (
   PMPrinter printer,
   PMPrintSettings settings,
   PMPageFormat format,
   CFStringRef mimeType,
   CFURLRef fileURL
);
Parameters
printer

The destination printer.
settings

The print settings for the print job.
format

The physical page size and orientation with which the document should be printed. This parameter can be NULL.
mimeType

The MIME type of the data to be printed. If this parameter is NULL, the specified file will be typed automatically.
fileURL

The URL of the file that supplies the print data.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function can fail if the specified printer cannot handle the file's MIME type. Use the function PMPrinterGetMimeTypes to check whether a MIME type is supported. 
Availability
Available in Mac OS X v10.3 and later.
Declared In
PMCore.h*/

OSStatus PMPrinterPrintWithFile (
   PMPrinter printer,
   PMPrintSettings settings,
   PMPageFormat format,
   CFStringRef mimeType,
   CFURLRef fileURL
){};


/*PMRelease

Releases a printing object by decrementing its reference count.

OSStatus PMRelease (
   PMObject object
);
Parameters
object

The printing object you want to release.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Your application should use the PMRelease function to release any printing objects it creates or retains. When an object’s reference count reaches 0, the object is deallocated.

For example, to terminate a printing session created with the function PMCreateSession, pass the associated PMPrintSession object to PMRelease. To release printing objects created with the functions PMCreatePageFormat and PMCreatePrintSettings, pass the associated PMPageFormat and PMPrintSettings objects to PMRelease.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMRelease (
   PMObject object
){};

/*PMSessionDefaultPrintSettings

Assigns default parameter values to a print settings object for the specified printing session.

OSStatus PMSessionDefaultPrintSettings (
   PMPrintSession printSession,
   PMPrintSettings printSettings
);
Parameters
printSession

The printing session for the specified print settings object.
printSettings

The print settings object to which you want to assign default values.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call the function PMSessionDefaultPrintSettings between the creation and release of a printing session. See the function PMCreateSession. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSessionDefaultPrintSettings (
   PMPrintSession printSession,
   PMPrintSettings printSettings
){};





/*PMCreatePageFormat

Creates a new page format object.

OSStatus PMCreatePageFormat (
   PMPageFormat *pageFormat
);
Parameters
pageFormat

A pointer to your PMPageFormat variable. On return, the variable refers to a new page format object. You should call the function PMRelease to release the page format object when you no longer need it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function allocates memory for a new page format object in your application’s memory space and sets its reference count to 1. The new page format object is empty and unusable until you call PMSessionDefaultPageFormat or PMCopyPageFormat.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMCreatePageFormat (
   PMPageFormat *pageFormat
){};


/*PMPrinterGetMimeTypes

Obtains a list of MIME content types supported by a printer using the specified print settings.

OSStatus PMPrinterGetMimeTypes (
   PMPrinter printer,
   PMPrintSettings settings,
   CFArrayRef *mimeTypes
);
Parameters
printer

The printer whose supported MIME types you want to obtain.
settings

The print settings for the print job. The print settings object contains the job destination, which affects the available MIME types. This parameter may be NULL.
mimeTypes

A pointer to your CFArrayRef variable. On return, the variable refers to a Core Foundation array containing the MIME types supported by the specified printer. Each element in the array is a Core Foundation string. You should not release the array without first retaining it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function retrieves the types of data that can be submitted to a printer with the specified print settings; for example, application/pdf. This function is typically used in conjunction with the function PMPrinterPrintWithFile.
Availability
Available in Mac OS X v10.3 and later.
Declared In
PMCore.h*/
OSStatus PMPrinterGetMimeTypes (
   PMPrinter printer,
   PMPrintSettings settings,
   CFArrayRef *mimeTypes
){};

/*PMSessionValidatePrintSettings

Validates a print settings object within the context of the specified printing session.

OSStatus PMSessionValidatePrintSettings (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   Boolean *result
);
Parameters
printSession

The printing session for the specified print settings object.
printSettings

The print settings object to validate.
result

A pointer to your Boolean variable. On return, true if any parameters changed, or false if no parameters changed.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call this function between the creation and release of a printing session. See the function PMCreateSession.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSessionValidatePrintSettings (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   Boolean *result
){};


/*PMCreatePrintSettings

Creates a new print settings object.

OSStatus PMCreatePrintSettings (
   PMPrintSettings *printSettings
);
Parameters
printSettings

A pointer to your PMPrintSettings variable. On return, the variable refers to a new print settings object. You should call the function PMRelease to release the print settings object when you no longer need it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function allocates memory for a new print settings object in your application’s memory space and sets its reference count to 1. The new print settings object is empty and unusable until you call PMSessionDefaultPrintSettings or PMCopyPrintSettings.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMCreatePrintSettings (
   PMPrintSettings *printSettings
){};


/*PMSessionDefaultPageFormat

Assigns default parameter values to a page format object used in the specified printing session.

OSStatus PMSessionDefaultPageFormat (
   PMPrintSession printSession,
   PMPageFormat pageFormat
);
Parameters
printSession

The printing session for the specified page format object.
pageFormat

The page format object to which you want to assign default values.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call the function PMSessionDefaultPageFormat between the creation and release of the printing session. See the function PMCreateSession.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSessionDefaultPageFormat (
   PMPrintSession printSession,
   PMPageFormat pageFormat
){};


/*PMSessionValidatePageFormat

Updates the values in a page format object and validates them against the current formatting printer.

OSStatus PMSessionValidatePageFormat (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   Boolean *result
);
Parameters
printSession

The printing session for the specified page format object.
pageFormat

The page format object to validate.
result

A pointer to your Boolean variable. On return, true if the function set the page format object to default values; otherwise, false.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call this function between the creation and release of the printing session. See the function PMCreateSession.

The printing system validates the page format object against the current formatting printer. The formatting printer is displayed in the Format for pop-up menu in the Page Setup dialog. The default formatting printer is the generic Any Printer. If the function PMSessionValidatePageFormat finds that the page format object contains values that are not valid for the formatting printer, then the page format object is set to default values and the result parameter is set to true. 

Validating a page format object also causes calculated fields (such as the adjusted paper and page rectangles) to be updated based on the changed settings (such as resolution, scaling, and page orientation). If the page format object contains values that are valid for the formatting printer but need to be updated, the result parameter is set to false.

After you call any function that makes changes to a page format object (such as PMSetResolution), you should call the function PMSessionValidatePageFormat to validate the page format object before using that object. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSessionValidatePageFormat (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   Boolean *result
){};

/*PMSessionGetDestinationType

Obtains the output destination for a print job.

OSStatus PMSessionGetDestinationType (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMDestinationType *destTypeP
);
Parameters
printSession

The printing session that provides a context for the print job. This must be the same printing session used for the Print dialog. The printing session contains the preview setting, which can override the destination type in the print settings.
printSettings

The print settings for the print job whose destination you want to obtain.
destTypeP

A pointer to your PMDestinationType variable. On return, the variable contains the destination type for the specified print job. Possible values include:
kPMDestinationPrinter (output to a printer)
kPMDestinationFile (output to a file)
kPMDestinationFax (output to a fax)
kPMDestinationPreview (output to print preview)
kPMDestinationProcessPDF (output to a PDF workflow option)

See “Destination Types” for a complete list of destination types that can be returned.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. 
Discussion

You must call this function between the creation and release of a printing session. See the function PMCreateSession.

The first three destination types (see above) are stored in the print settings object. The destination type kPMDestinationPreview is stored in the printing session. If the destination type is set as preview, the preview setting overrides the destination set in the print settings object. 
Availability
Available in Mac OS X v10.1 and later.
Declared In
PMCore.h*/
OSStatus PMSessionGetDestinationType (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMDestinationType *destTypeP
){};


/*PMSessionGetCurrentPrinter

Obtains the current printer associated with a printing session.

OSStatus PMSessionGetCurrentPrinter (
   PMPrintSession printSession,
   PMPrinter *currentPrinter
);
Parameters
printSession

The printing session whose printer you want to obtain.
currentPrinter

A pointer to your PMPrinter variable. On return, the variable refers to the printer associated with the specified printing session. You should not release this object without first retaining it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call this function between the creation and release of a printing session. See the function PMCreateSession. The printer object is valid as long as the printing session is valid or the current printer hasn’t changed.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSessionGetCurrentPrinter (
   PMPrintSession printSession,
   PMPrinter *currentPrinter
){};


/*PMRetain

Retains a printing object by incrementing its reference count.

OSStatus PMRetain (
   PMObject object
);
Parameters
object

The printing object you want to retain.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You should retain a printing object when you receive it from elsewhere (that is, you did not create or copy it) and you want it to persist. If you retain a printing object, you are responsible for releasing it. (See PMRelease.) You can use the function PMRetain to increment a printing object’s reference count so that multiple threads or routines can use the object without the risk of another thread or routine deallocating the object.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMRetain (
   PMObject object
){};



/*PMGetFirstPage

Obtains the number of the first page to be printed.

OSStatus PMGetFirstPage (
   PMPrintSettings printSettings,
   UInt32 *first
);
Parameters
printSettings

The print settings object whose first page number you want to obtain.
first

A pointer to your UInt32 variable. On return, the variable contains the page number of the first page to print. The default first page number is 1.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You can use this function to obtain the page number entered by the user in the From field of the Print dialog. If the user selects the All button, the function returns a value of 1. If the user did not enter a value, the function returns the value of the previous call to PMSetFirstPage, if any, or the default value of 1.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMGetFirstPage (
   PMPrintSettings printSettings,
   UInt32 *first
){};

/*PMGetLastPage

Obtains the number of the last page to be printed.

OSStatus PMGetLastPage (
   PMPrintSettings printSettings,
   UInt32 *last
);
Parameters
printSettings

The print settings object whose last page number you want to obtain.
last

A pointer to your UInt32 variable. On return, the variable contains the page number of the last page to print.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You use this function to obtain the page number entered by the user in the To field of the Print dialog. If the user did not enter a value, the function returns the value of the previous call to PMSetLastPage, if any, or a default value. 

You should not look for the constant kPMPrintAllPages. That constant is used only with the PMSetLastPage and PMSetPageRange functions to specify a last page. It is not returned by the PMGetLastPage function.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMGetLastPage (
   PMPrintSettings printSettings,
   UInt32 *last
){};



/*PMSetLastPage

Sets the page number of the last page to be printed.

OSStatus PMSetLastPage (
   PMPrintSettings printSettings,
   UInt32 last,
   Boolean lock
);
Parameters
printSettings

The print settings object whose last page number you want to set.
last

The page number of the last page to print. This value appears in the To field of the Print dialog. Pass the constant kPMPrintAllPages to print the entire document. 
lock

The lock state of the setting. Locking is not supported at this time. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”. 
Discussion

Typically, you call this function after the Print dialog is displayed to indicate the number of the last page number to be printed. In Mac OS X, setting the last page provides information used by the progress dialog that is shown during printing.

If you call the function PMSetPageRange and then call PMSetFirstPage or PMSetLastPage using the same page range you specified for PMSetPageRange, then the Print dialog shows the From button selected. If you use the constant kPMPrintAllPages to set the page range with the function PMSetPageRange, then the Print dialog opens with the All button selected regardless of whether you also call PMSetFirstPage or PMSetLastPage. 

If you call this function after initiating a print job (for example, by calling PMSessionBeginCGDocument), the change is ignored for the current job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSetLastPage (
   PMPrintSettings printSettings,
   UInt32 last,
   Boolean lock
){};

/*PMSessionError

Obtains the result code for any error returned by the printing session. 

OSStatus PMSessionError (
   PMPrintSession printSession
);
Parameters
printSession

The printing session whose last error you want to obtain.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. The result code kPMCancel indicates the user canceled the current print job.
Discussion

You must call this function between the creation and release of a printing session. See the function PMCreateSession.

In Mac OS X, the function PMSessionError returns the last printing session error, not the last error from a printing manager function (PMxxx). Because all Carbon Printing Manager functions return a result code, the PMSessionError function is not required for general error checking, but can be used in your print loop to determine if the user cancels the current print job or if any other errors occur during printing but are not explicitly returned by one of the other calls. 

The result code kPMCancel is returned if the user clicks the Cancel button in the printer driver’s status dialog. If this or any other error is encountered during the print loop, your application should call the appropriate functions (for example, PMSessionEndPage and PMSessionEndDocument) to exit the print loop before your application reports the error.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMSessionError (
   PMPrintSession printSession
){};

/*PMSessionGetGraphicsContext

Obtains the graphics context for the current page in a printing session.

OSStatus PMSessionGetGraphicsContext (
   PMPrintSession printSession,
   CFStringRef graphicsType,
   void **graphicsContext
);
Parameters
printSession

The printing session whose current graphics context you want to obtain.
graphicsType

The desired graphics context type. This parameter is currently ignored.
graphicsContext

On return, a reference to the current graphics context. The graphics context returned is the one last set by a call to the function PMSessionSetDocumentFormatGeneration or the default (QuickDraw) if there was no call to the function. You must typecast the context to an appropriate graphics type, either grafPtr or CGContextRef.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You must call this function between the creation and release of a printing session. See the function PMCreateSession. You must also call the function PMSessionGetGraphicsContext within the scope of the functions PMSessionBeginPage and PMSessionEndPage.

In Mac OS X v10.3 and earlier, you should call this function for each page you draw for a print job. After each call to the function PMSessionBeginPage your application should call PMSessionGetGraphicsContext to obtain the current graphics context. If that context is a QuickDraw context, then set the drawing port to this port by calling the QuickDraw SetPort function. See the discussion of the function PMSessionBeginPage for more information.
Special Considerations

In Mac OS X v10.4 and later, Apple recommends using the function PMSessionGetCGGraphicsContext instead of this function. QuickDraw is deprecated and your application should be using Quartz 2D for its rendering.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSessionGetGraphicsContext (
   PMPrintSession printSession,
   CFStringRef graphicsType,
   void **graphicsContext
){};


/*PMSessionSetDestination

Sets the destination location, format, and type for a print job.

OSStatus PMSessionSetDestination (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMDestinationType destType,
   CFStringRef destFormat,
   CFURLRef destLocation
);
Parameters
printSession

The printing session that provides a context for the print job.
printSettings

The print settings for the print job whose destination you want to set.
destType

The destination type for the print job associated with the specified printing session and print settings. Possible values include:
kPMDestinationPrinter (output to a printer)
kPMDestinationFile (output to a file)
kPMDestinationFax (output to a fax)
kPMDestinationPreview (output to print preview)
kPMDestinationProcessPDF (output to a PDF workflow option)

See “Destination Types” for a complete list of destination types you can specify.
destFormat

The MIME type to be generated for the specified destination type. Pass NULL if you want to use the default format for the specified destination type. To obtain a list of valid formats for a given destination type, use the function PMSessionCopyOutputFormatList.
destLocation

A reference to a Core Foundation URL that specifies a destination location. You can provide this if the destination type supports a destination location. Otherwise, pass NULL. For example, if the destination type is a file (kPMDestinationFile) you can supply a file system URL to specify where the file resides.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You can use the function PMSessionSetDestination when you want to send print output to a file without requiring user interaction. You must call this function between the creation and release of a printing session. See the function PMCreateSession.
Availability
Available in Mac OS X v10.1 and later.
Declared In
PMCore.h*/

OSStatus PMSessionSetDestination (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMDestinationType destType,
   CFStringRef destFormat,
   CFURLRef destLocation
){};

/*PMSetJobNameCFString

Specifies the name of a print job.

OSStatus PMSetJobNameCFString (
   PMPrintSettings printSettings,
   CFStringRef name
);
Parameters
printSettings

The print settings object whose job name you want to set.
name

The new name for the print job.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. The result code kPMInvalidParameter is returned if you pass NULL or an empty string in the name parameter.
Discussion

You should call this function before you open the Print dialog.

If you call this function after initiating a print job (for example, by calling PMSessionBeginCGDocument), the change is ignored for the current job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSetJobNameCFString (
   PMPrintSettings printSettings,
   CFStringRef name
){};



/*PMSetPageRange

Sets the valid range of pages that can be printed.

OSStatus PMSetPageRange (
   PMPrintSettings printSettings,
   UInt32 minPage,
   UInt32 maxPage
);
Parameters
printSettings

The print settings object whose page range you want to set.
minPage

The minimum page number allowed. This value appears as the default in the From field of the Print dialog. 
maxPage

The maximum page number allowed. This value appears as the default in the To field of the Print dialog. Pass the constant kPMPrintAllPages to allow the user to print the entire document. If the first page is set to 1, then passing kPMPrintAllPages as the maximum page number causes the All button to be selected.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

The function PMSetPageRange allows applications to set the minimum and maximum page numbers that can be printed for a document. If the user enters a value outside of this range in the Print dialog, the Carbon Printing Manager displays an alert message and sets the value to the closest allowed. You can use the PMGetFirstPage and PMGetLastPage functions to obtain the values entered by the user in the Print dialog.)

If you call the function PMSetPageRange to set the maximum page to a value other than the constant kPMPrintAllPages, the function PMSetPageRange causes the page range in the Print dialog to be properly restricted to the specified range. If you call the function PMSetPageRange without also calling the functions PMSetFirstPage or PMSetLastPage, then the Print dialog shows the specified page range in the From and To fields but with the All button selected. If you call the function PMSetPageRange and then call PMSetFirstPage or PMSetLastPage using the same page range you specified for PMSetPageRange, then the Print dialog shows the From button selected.

In all cases, if your application sets a range with PMSetPageRange and subsequently calls PMSetFirstPage or PMSetLastPage with values outside of the specified range, the Carbon Printing Manager returns a result code of kPMValueOutOfRange. Conversely, if your application calls PMSetPageRange after calling PMSetFirstPage or PMSetLastPage (or after displaying the Print dialog), the page range specified by PMSetPageRange takes precedence, and the first and last page values are adjusted accordingly.

If you call this function after initiating a print job (for example, by calling PMSessionBeginCGDocument), the change is ignored for the current job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSetPageRange (
   PMPrintSettings printSettings,
   UInt32 minPage,
   UInt32 maxPage
){};



/*PMSessionEndDocumentNoDialog

Ends a print job started by calling the function PMSessionBeginCGDocumentNoDialog or PMSessionBeginDocumentNoDialog.

OSStatus PMSessionEndDocument (
   PMPrintSession printSession
);
Parameters
printSession

The current printing session. On return, the printing session is no longer valid; however, you must still call the function PMRelease to release the object. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function is similar to the function PMSessionEndDocument except that the printing status dialog is suppressed.

This function is used to end a print job, and it should be called within your application’s print loop after the call to the function PMSessionEndPageNoDialog and before releasing the printing session. The same printing session that is created by the function PMCreateSession for the Print dialog should be used for the print loop.

The function PMSessionEndDocumentNoDialog must be called after its corresponding Begin function (PMSessionBeginCGDocumentNoDialog or PMSessionBeginDocumentNoDialog). If the Begin function returns noErr, the function PMSessionEndDocument must be called, even if errors occur within the scope of the Begin and End functions. You should not call PMSessionEndDocumentNoDialog if the Begin function returns an error.
Availability
Available in Mac OS X v10.2 and later.
Declared In
PMCore.h*/
OSStatus PMSessionEndDocumentNoDialog (
   PMPrintSession printSession
){};

/*PMSessionEndPageNoDialog

Indicates the end of drawing the current page for the specified printing session.

OSStatus PMSessionEndPageNoDialog (
   PMPrintSession printSession
);
Parameters
printSession

The printing session that provides a context for the print job.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function is similar to the function PMSessionEndPage except that the printing status dialog is suppressed.

You must call this function between the creation and release of a printing session. See the function PMCreateSession. You must call the functions PMSessionBeginPageNoDialog and PMSessionEndPageNoDialog within the scope of calls to the Begin print job function (PMSessionBeginCGDocumentNoDialog or PMSessionBeginDocumentNoDialog) and the End print job function (PMSessionEndDocumentNoDialog).

If the function PMSessionBeginPageNoDialog returns noErr, you must later call the function PMSessionEndPageNoDialog, even if errors occur within the scope of PMSessionBeginPageNoDialog and PMSessionEndPageNoDialog. You should not call PMSessionEndPageNoDialog if PMSessionBeginPageNoDialog returns an error.
Availability
Available in Mac OS X v10.2 and later.
Declared In
PMCore.h*/
OSStatus PMSessionEndPageNoDialog (
   PMPrintSession printSession
){};



/*PMSessionBeginPageNoDialog

Starts a new page for printing in the specified printing session and suppresses the printing status dialog.

OSStatus PMSessionBeginPage (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   const PMRect *pageFrame
);
Parameters
printSession

The printing session that provides a context for the print job.
pageFormat

The page format you are using for the print job.
pageFrame

You should pass NULL, as this parameter is currently unsupported.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. Returns kPMCancel if the user cancels the print job.
Discussion

This function is similar to the function PMSessionBeginPage except that the function PMSessionBeginPageNoDialog suppresses the printing status dialog. You must call this function between the creation and release of a printing session. See the function PMCreateSession. You must call the functions PMSessionBeginPage and PMSessionEndPage within the scope of calls to the Begin print job function (PMSessionBeginCGDocumentNoDialog or PMSessionBeginDocumentNoDialog) and the End print job function (PMSessionEndDocumentNoDialog).

You should call the function PMSessionError immediately before you call PMSessionBeginPage. If PMSessionError returns an error, then you should not call the function PMSessionBeginPage. Because PMSessionBeginPage also initializes the printing graphics context, your application should not make assumptions about the state of the context (for example, the current font) between successive pages. After each call to PMSessionBeginPage, your application should call PMSessionGetCGGraphicsContext or PMSessionGetGraphicsContext to obtain the current printing context. If that context is a QuickDraw context, set the drawing port to this context by calling the QuickDraw SetPort function.

If the function PMSessionBeginPageNoDialog returns noErr, you must later call the function PMSessionEndPageNoDialog, even if errors occur within the scope of PMSessionBeginPageNoDialog and PMSessionEndPageNoDialog.

The printing system automatically handles printing multiple copies. Your application does not need to perform any tasks other than specifying the number of copies in the printing session.
Availability
Available in Mac OS X v10.2 and later.
Declared In
PMCore.h*/

OSStatus PMSessionBeginPage (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   const PMRect *pageFrame
){};


/*PMSessionBeginDocumentNoDialog

Begins a print job that, by default, draws into a QuickDraw graphics port, and suppresses the printing status dialog.

OSStatus PMSessionBeginDocumentNoDialog (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMPageFormat pageFormat
);
Parameters
printSession

The printing session that provides a context for the new print job.
printSettings

The print settings to use for the new print job.
pageFormat

The page format to use for the new print job.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

The function PMSessionBeginDocumentNoDialog starts a print job and should be called within your application’s print loop. This function is similar to the function PMSessionBeginDocument except that the printing status dialog is suppressed.

You must call this function between the creation and release of a printing session. See the function PMCreateSession. If you present a printing dialog before you call PMSessionBeginDocumentNoDialog, when calling this function you should use the same PMPrintSession object you used to present the dialog.

Before you call PMSessionBeginDocumentNoDialog, you should call PMSessionValidatePrintSettings and PMSessionValidatePageFormat to make sure the specified print settings and page format objects are updated and valid. After you call PMSessionBeginDocumentNoDialog, if you call a function that changes the specified print settings or page format object, the change is ignored for the current print job.

This function must be called before its corresponding End function (PMSessionEndDocumentNoDialog). If the function PMSessionBeginDocumentNoDialog returns noErr, you must call the End function, even if errors occur within the scope of the Begin and End functions.

The printing system automatically handles printing multiple copies. Your application does not need to perform any tasks other than specifying the number of copies in the printing session.
Special Considerations

In Mac OS X v10.4 and later, Apple recommends using the function PMSessionBeginCGDocumentNoDialog instead of this function. QuickDraw is deprecated and your application should be using Quartz 2D for its rendering.
Availability
Available in Mac OS X v10.2 and later.
Declared In
PMCore.h*/

OSStatus PMSessionBeginDocumentNoDialog (
   PMPrintSession printSession,
   PMPrintSettings printSettings,
   PMPageFormat pageFormat
){};




/*PMSessionBeginPageNoDialog

Starts a new page for printing in the specified printing session and suppresses the printing status dialog.

OSStatus PMSessionBeginPage (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   const PMRect *pageFrame
);
Parameters
printSession

The printing session that provides a context for the print job.
pageFormat

The page format you are using for the print job.
pageFrame

You should pass NULL, as this parameter is currently unsupported.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. Returns kPMCancel if the user cancels the print job.
Discussion

This function is similar to the function PMSessionBeginPage except that the function PMSessionBeginPageNoDialog suppresses the printing status dialog. You must call this function between the creation and release of a printing session. See the function PMCreateSession. You must call the functions PMSessionBeginPage and PMSessionEndPage within the scope of calls to the Begin print job function (PMSessionBeginCGDocumentNoDialog or PMSessionBeginDocumentNoDialog) and the End print job function (PMSessionEndDocumentNoDialog).

You should call the function PMSessionError immediately before you call PMSessionBeginPage. If PMSessionError returns an error, then you should not call the function PMSessionBeginPage. Because PMSessionBeginPage also initializes the printing graphics context, your application should not make assumptions about the state of the context (for example, the current font) between successive pages. After each call to PMSessionBeginPage, your application should call PMSessionGetCGGraphicsContext or PMSessionGetGraphicsContext to obtain the current printing context. If that context is a QuickDraw context, set the drawing port to this context by calling the QuickDraw SetPort function.

If the function PMSessionBeginPageNoDialog returns noErr, you must later call the function PMSessionEndPageNoDialog, even if errors occur within the scope of PMSessionBeginPageNoDialog and PMSessionEndPageNoDialog.

The printing system automatically handles printing multiple copies. Your application does not need to perform any tasks other than specifying the number of copies in the printing session.
Availability
Available in Mac OS X v10.2 and later.
Declared In
PMCore.h*/

OSStatus PMSessionBeginPageNoDialog (
   PMPrintSession printSession,
   PMPageFormat pageFormat,
   const PMRect *pageFrame
){};



/*PMGetResolution

Obtains the current application’s drawing resolution.

OSStatus PMGetResolution (
   PMPageFormat pageFormat,
   PMResolution *res
);
Parameters
pageFormat

The page format object whose drawing resolution you want to obtain.
res

A pointer to your PMResolution structure. On return, the structure contains the drawing resolution of the current application.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

This function obtains the drawing resolution of the current application, not the resolution of the current printer. You can call PMPrinterGetPrinterResolution to see what resolutions are available for the current printer.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMGetResolution (
   PMPageFormat pageFormat,
   PMResolution *res
){};


/*PMSetResolution

Sets the application drawing resolution.

OSStatus PMSetResolution (
   PMPageFormat pageFormat,
   const PMResolution *res
);
Parameters
pageFormat

The page format object whose drawing resolution you want to set.
res

A pointer to a structure of type PMResolution that specifies the desired drawing resolution for your application. You should specify the best resolution for your data. The printing system handles the mapping between the resolution you specify and the printer resolution.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

If you call this function after initiating a print job (for example, by calling PMSessionBeginCGDocument), the change is ignored for the current job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSetResolution (
   PMPageFormat pageFormat,
   const PMResolution *res
){};


/*PMGetGrafPtr

Obtains the printing port from an opaque printing context. (Deprecated in Mac OS X v10.4. Use PMSessionGetCGGraphicsContext or PMSessionGetGraphicsContext instead.)

OSStatus PMGetGrafPtr (
   PMPrintContext printContext,
   GrafPtr *grafPort
);
Parameters
printContext

A PMPrintContext object. 
grafPort

On return, a pointer to a grafPort defining the current printing port. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin and creating a printing context.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMGetGrafPtr (
   PMPrintContext printContext,
   GrafPtr *grafPort
){};


/*PMNewPageFormat

Creates a new PMPageFormat object. (Deprecated in Mac OS X v10.4. Use PMCreatePageFormat instead.)

OSStatus PMNewPageFormat (
   PMPageFormat *pageFormat
);
Parameters
pageFormat

On return, an initialized PMPageFormat object.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin. The function PMNewPageFormat allocates memory for a new PMPageFormat object in your application’s memory space. The new page format object is empty until you set its values, or until you call PMDefaultPageFormat or PMValidatePageFormat.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/

OSStatus PMNewPageFormat (
   PMPageFormat *pageFormat
){};
/*PMDefaultPageFormat

Assigns default parameter values to an existing PMPageFormat object, for the current printer. (Deprecated in Mac OS X v10.4. Use PMSessionDefaultPageFormat instead.)

OSStatus PMDefaultPageFormat (
   PMPageFormat pageFormat
);
Parameters
pageFormat

On return, a PMPageFormat object containing default parameter values.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin and creating a page format object.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/

OSStatus PMDefaultPageFormat (
   PMPageFormat pageFormat
){};



/*PMGetAdjustedPageRect

Obtains the imageable area or page rectangle, taking into account orientation, application drawing resolution, and scaling settings.

OSStatus PMGetAdjustedPageRect (
   PMPageFormat pageFormat,
   PMRect *pageRect
);
Parameters
pageFormat

The page format object whose adjusted page rectangle you want to obtain.
pageRect

A pointer to your PMRect structure. On return, the structure contains the current imageable area, in points, taking into account scaling, rotation, and application resolution settings. The page rectangle is the area of the page to which an application can draw. The coordinates for the upper-left corner of the page rectangle are (0,0). See Supporting Printing in Your Carbon Application for more information on page and paper rectangles.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Before using this function, you must call PMSessionValidatePageFormat to ensure that the values for the adjusted page rectangle correctly account for scaling, rotation, and application resolution settings.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMGetAdjustedPageRect (
   PMPageFormat pageFormat,
   PMRect *pageRect
){};


/*PMBegin

Prepares the Carbon Printing Manager for use. (Deprecated in Mac OS X v10.4. Use PMCreateSession instead.)

OSStatus PMBegin ();
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Do not nest calls to PMBegin. 
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMBegin (){};



/*PMEnd

Closes the Carbon Printing Manager and releases its allocated memory. (Deprecated in Mac OS X v10.4. Use PMRelease to release a PMPrintSession object instead.)

OSStatus PMEnd (void);
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMEnd (void){};


/*PMNewPrintSettings

Creates a new PMPrintSettings object. (Deprecated in Mac OS X v10.4. Use PMCreatePrintSettings instead.)

OSStatus PMNewPrintSettings (
   PMPrintSettings *printSettings
);
Parameters
printSettings

On return, an initialized PMPrintSettings object. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin. The function PMNewPrintSettings allocates memory for a new PMPrintSettings object in your application’s memory space. The new print settings object is empty until you set its values, or until you call PMDefaultPrintSettings or PMValidatePrintSettings.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/

OSStatus PMNewPrintSettings (
   PMPrintSettings *printSettings
){};


/*PMValidatePrintSettings

Obtains a valid PMPrintSettings object. (Deprecated in Mac OS X v10.4. Use PMSessionValidatePrintSettings instead.)

OSStatus PMValidatePrintSettings (
   PMPrintSettings printSettings,
   Boolean *result
);
Parameters
printSettings

The PMPrintSettings object to be validated. 
result

On return, a value of true if any parameters were changed, or false if no changes were required. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin and creating a print settings object.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/

OSStatus PMValidatePrintSettings (
   PMPrintSettings printSettings,
   Boolean *result
){}


/*PMDefaultPrintSettings

Assigns default parameter values to a PMPrintSettings object. (Deprecated in Mac OS X v10.4. Use PMSessionDefaultPrintSettings instead.)

OSStatus PMDefaultPrintSettings (
   PMPrintSettings printSettings
);
Parameters
printSettings

A PMPrintSettings object. On return, the object contains default parameter values. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin and creating a print settings object.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/

OSStatus PMDefaultPrintSettings (
   PMPrintSettings printSettings
){};

/*PMUnflattenPrintSettings

Rebuilds a print settings object from a Memory Manager handle that contains flattened print settings data.

OSStatus PMUnflattenPrintSettings (
   Handle flatSettings,
   PMPrintSettings *printSettings
);
Parameters
flatSettings

A handle to a flattened representation of a print settings object.
printSettings

A pointer to your PMPrintSettings variable. On return, the variable refers to a print settings object that contains the data retrieved from the flattened print settings. Your application is responsible for releasing the print settings object when you no longer need it. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”. The result code kPMInvalidParameter is returned if the flattened PMPrintSettings object was created by an incompatible version of the Carbon Printing Manager.
Discussion

The PMUnflattenPrintSettings function creates a new PMPrintSettings object containing the data from the flattened print settings. You should call the function PMSessionValidatePrintSettings, as some values in the print settings object may no longer be valid.

If the function returns the result code kPMInvalidParameter you need to create a new, default print settings object. You should also notify the user that the print settings are not valid. 

There are no scoping requirements as to when you may use this function.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMUnflattenPrintSettings (
   Handle flatSettings,
   PMPrintSettings *printSettings
){};


/*PMError

Obtains the result code from the last Carbon Printing Manager function called by your application. (Deprecated in Mac OS X v10.4. Use PMSessionError instead.)

OSStatus PMError ();
Return Value

A result code. The result code kPMCancel indicates the user canceled the current print job.
Discussion

Valid after calling PMBegin.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMError (){};


/*PMFlattenPageFormat

Flattens a page format object into a Memory Manager handle for storage in a user document.

OSStatus PMFlattenPageFormat (
   PMPageFormat pageFormat,
   Handle *flatFormat
);
Parameters
pageFormat

The page format object to flatten.
flatFormat

A pointer to your Handle variable. On return, the variable refers to a Memory Manager handle that contains the flattened page format object. The handle is allocated by the function. You are responsible for disposing of the handle when you no longer need it.
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMFlattenPageFormat (
   PMPageFormat pageFormat,
   Handle *flatFormat
){};


/*PMUnflattenPageFormat

Rebuilds a page format object from a Memory Manager handle that contains flattened page format data.

OSStatus PMUnflattenPageFormat (
   Handle flatFormat,
   PMPageFormat *pageFormat
);
Parameters
flatFormat

A handle to a previously flattened page format object. Your application is responsible for disposing of the handle.
pageFormat

A pointer to your PMPageFormat variable. On return, the variable refers to a page format object that contains the data retrieved from the flattened page format data. Your application is responsible for releasing the page format object when you no longer need it. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”. The result code kPMInvalidParameter is returned if the flattened PMPageFormat object was created by an incompatible version of the Carbon Printing Manager.
Discussion

The PMUnflattenPageFormat function creates a new PMPageFormat object that contains the data from the flattened page format data. You should call the function PMSessionValidatePageFormat to make sure the page format object contains valid values. 

If the function returns the result code kPMInvalidParameter you need to create a new, default page format object. You should also notify the user that the page format is not valid. 
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMUnflattenPageFormat (
   Handle flatFormat,
   PMPageFormat *pageFormat
){};

/*PMFlattenPrintSettings

Flattens a print settings object into a Memory Manager handle for storage in a user document.

OSStatus PMFlattenPrintSettings (
   PMPrintSettings printSettings,
   Handle *flatSettings
);
Parameters
printSettings

The print settings object to flatten.
flatSettings

A pointer to your Handle variable. On return, the variable refers to a Memory Manager handle that contains a flattened print settings object. The handle is allocated by the function. You are responsible for disposing of the handle when you no longer need it. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

There are no scoping requirements as to when you may use this function. 

Apple recommends that you do not reuse the print settings information if the user prints the document again. The information supplied by the user in the Print dialog should pertain to the document only while the document prints, so there is no need to save the print settings object.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/
OSStatus PMFlattenPrintSettings (
   PMPrintSettings printSettings,
   Handle *flatSettings
){};


/*PMValidatePageFormat

Obtains a valid PMPageFormat object. (Deprecated in Mac OS X v10.4. Use PMSessionValidatePageFormat instead.)

OSStatus PMValidatePageFormat (
   PMPageFormat pageFormat,
   Boolean *result
);
Parameters
pageFormat

A PMPageFormat object to be validated. 
result

Returns true if any parameters were changed, false if no changes were required. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

Valid after calling PMBegin and creating a page format object.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMValidatePageFormat (
   PMPageFormat pageFormat,
   Boolean *result
){};



/*PMSetFirstPage

Sets the default page number of the first page to be printed.

OSStatus PMSetFirstPage (
   PMPrintSettings printSettings,
   UInt32 first,
   Boolean lock
);
Parameters
printSettings

The print settings object whose first page number you want to set.
first

The page number of the first page to print. This value appears in the From field of the Print dialog.
lock

The lock state of the setting. Locking is not supported at this time.
Return Value

A result code. See “Carbon Printing Manager Result Codes”. 
Discussion

Typically, this function isn’t used. In Mac OS X, if you call the function PMSetPageRange and then call PMSetFirstPage or PMSetLastPage using the same page range you specified for PMSetPageRange, then the Print dialog shows the From button selected. If you use the constant kPMPrintAllPages to set the page range with the function PMSetPageRange, then the Print dialog opens with the All button selected regardless of whether you also call PMSetFirstPage or PMSetLastPage.

If you call this function after initiating a print job (for example, by calling PMSessionBeginCGDocument), the change is ignored for the current job.
Availability
Available in Mac OS X v10.0 and later.
Declared In
PMCore.h*/

OSStatus PMSetFirstPage (
   PMPrintSettings printSettings,
   UInt32 first,
   Boolean lock
){};


/*PMSessionSetDocumentFormatGeneration

Requests a specified spool file format and supplies the graphics context type to use for drawing pages within the print loop. (Deprecated in Mac OS X v10.4. If you’re drawing using Quartz 2D instead of QuickDraw, you should use PMSessionBeginCGDocument or PMSessionBeginCGDocumentNoDialog; for submitting PostScript data, you should use PMPrinterPrintWithFile, PMPrinterPrintWithProvider, or PMCGImageCreateWithEPSDataProvider to draw EPS documents.)

OSStatus PMSessionSetDocumentFormatGeneration (
   PMPrintSession printSession,
   CFStringRef docFormat,
   CFArrayRef graphicsContexts,
   CFTypeRef options
);
Parameters
printSession

The printing session whose spool file format and graphics context type you want to specify.
docFormat

A Core Foundation string that specifies the desired spool file format as a MIME type. See “Document Format Strings” for a description of the constants you can use to specify the document format.
graphicsContexts

A reference to a Core Foundation array of graphics contexts to use for drawing pages within the print loop. You can supply a QuickDraw graphics context (kPMGraphicsContextQuickDraw) or a Quartz 2D graphics context (kPMGraphicsContextCoreGraphics). An array of length 1 is the only length that is supported, regardless of graphics context type. See “Graphics Context Types” for a description of the constants you can use to specify a graphics context.
options

Reserved for future use. 
Return Value

A result code. See “Carbon Printing Manager Result Codes”.
Discussion

You only need to call the function PMSessionSetDocumentFormatGeneration if you want to specify a format other than the default format (PDF) or a graphics context other than the default context (QuickDraw). If you want to use the default format for the operating system and to draw with QuickDraw, then you do not need to call this function. If you want to generate PICT + PS to use as one of the supported formats, then call PMSessionSetDocumentFormatGeneration to set the graphics context to QuickDraw and the format to PICT + PS. Note that the PICT + PS format is not available on Intel-based systems.

If you want to use a Quartz 2D graphics context to draw each page, you can call the following code to inform the printing system in all versions of Mac OS X. static OSStatus MyPMSessionBeginCGDocument (
    PMPrintSession printSession,
    PMPrintSettings printSettings,
    PMPageFormat pageFormat)
{
    OSStatus err = noErr;
 
    // Use the simpler call if it is present.
    if(&PMSessionBeginCGDocument != NULL) {
        err = PMSessionBeginCGDocument (printSession, printSettings, pageFormat);
    }
    else {
        CFStringRef s[1] = { kPMGraphicsContextCoreGraphics };
        CFArrayRef graphicsContextsArray = CFArrayCreate (
            kCFAllocatorDefault, (const void**)s, 1, &kCFTypeArrayCallBacks);
        err = PMSessionSetDocumentFormatGeneration (
            printSession, kPMDocumentFormatPDF, graphicsContextsArray, NULL);
        CFRelease (graphicsContextsArray);
        if(!err)
            err = PMSessionBeginDocument (
                printSession, printSettings, pageFormat);
    }
    return err;
}


The previous code informs the printing system that you want a Quartz graphics context, but you get the actual context for your printing port only after you call the function PMSessionBeginPage and then call the following code.static OSStatus MyPMSessionGetCGGraphicsContext (
    PMPrintSession printSession,
    CGContextRef *printingContextP)
{
    OSStatus err = noErr;
 
    // Use the simpler call if it is present.
    if(&PMSessionGetCGGraphicsContext != NULL) {
        err = PMSessionGetCGGraphicsContext (printSession, printingContextP);
    }
    else {
        err = PMSessionGetGraphicsContext (
            printSession, kPMGraphicsContextCoreGraphics,
            (void**)printingContextP);
    }
    return err;
}


The printing context you get is a Quartz context into which you can draw. Note that the default coordinate system for Quartz 2D is not the same as that used for QuickDraw. Quartz 2D defines the coordinates of the lower-left corner of the sheet as (0,0) whereas the origin for the QuickDraw port is the upper-left corner of the imageable area.

You must call this function between the creation and release of a printing session. See the function PMCreateSession. You must call the function PMSessionSetDocumentFormatGeneration before you call PMSessionBeginDocument or PMSessionBeginDocumentNoDialog. Before requesting a spool file format using this function, you should call the function PMSessionGetDocumentFormatGeneration to get the list of supported formats.
Special Considerations

The PICT + PS spool file format is not available on Intel-based systems.
Availability
Available in Mac OS X v10.0 and later.
Deprecated in Mac OS X v10.4.
Declared In
PMCore.h*/
OSStatus PMSessionSetDocumentFormatGeneration (
   PMPrintSession printSession,
   CFStringRef docFormat,
   CFArrayRef graphicsContexts,
   CFTypeRef options
){};
























