#ifndef UNICODECONVERTER_H
#define UNICODECONVERTER_H

#ifndef MACTYPES_H
#include <CarbonCore/MacTypes.h>
#endif

#ifndef TEXTCOMMON_H
#include <CarbonCore/TextCommon.h>
#endif

#ifndef MIXEDMODE_H
#include <CarbonCore/MixedMode.h>
#endif



#include <AvailabilityMacros.h>

#if PRAGMA_ONCE
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*TextToUnicodeInfo

Defines reference to an opaque Unicode converter object.

typedef struct OpaqueTextToUnicodeInfo * TextToUnicodeInfo;
Discussion

A Unicode converter object is a private object containing mapping and state information. Many of the Unicode Converter functions that perform conversions require a Unicode converter object containing information used for the conversion process. There are three types of Unicode converter objects, all serving the same purpose but used for different types of conversions. You use the TextToUnicodeInfo type, described here, for converting from non-Unicode text to Unicode text.

Because your application cannot directly create or modify the contents of the private Unicode converter object, the Unicode Converter provides functions to create and dispose of it. To create a Unicode converter object for converting from non-Unicode text to Unicode text, your application must first call either the function CreateTextToUnicodeInfo or the function CreateTextToUnicodeInfoByEncoding to provide the mapping information required for the conversion. You can then pass this object to the function ConvertFromTextToUnicode or ConvertFromPStringToUnicode to identify the information to be used in performing the actual conversion. After you have finished using the object, you should release the memory allocated for it by calling the function DisposeTextToUnicodeInfo. The TextToUnicodeInfo data type defines the Unicode converter object. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueTextToUnicodeInfo * TextToUnicodeInfo;

/*Conversion Flags

Specify how to perform conversion of text from one encoding to another.

enum {
   kUnicodeUseFallbacksBit = 0,
   kUnicodeKeepInfoBit = 1,
   kUnicodeDirectionalityBits = 2,
   kUnicodeVerticalFormBit = 4,
   kUnicodeLooseMappingsBit = 5,
   kUnicodeStringUnterminatedBit = 6,
   kUnicodeTextRunBit = 7,
   kUnicodeKeepSameEncodingBit = 8,
   kUnicodeForceASCIIRangeBit = 9,
   kUnicodeNoHalfwidthCharsBit = 10,
   kUnicodeTextRunHeuristicsBit = 11,
   kUnicodeMapLineFeedToReturnBit = 12
};*/
enum {
   kUnicodeUseFallbacksBit = 0,
   kUnicodeKeepInfoBit = 1,
   kUnicodeDirectionalityBits = 2,
   kUnicodeVerticalFormBit = 4,
   kUnicodeLooseMappingsBit = 5,
   kUnicodeStringUnterminatedBit = 6,
   kUnicodeTextRunBit = 7,
   kUnicodeKeepSameEncodingBit = 8,
   kUnicodeForceASCIIRangeBit = 9,
   kUnicodeNoHalfwidthCharsBit = 10,
   kUnicodeTextRunHeuristicsBit = 11,
   kUnicodeMapLineFeedToReturnBit = 12
};


/*Unicode Mapping Versions

Specify a Unicode mapping version.

typedef SInt32 UnicodeMapVersion;
enum {
   kUnicodeUseLatestMapping = -1,
   kUnicodeUseHFSPlusMapping = 4
};
Constants
kUnicodeUseLatestMapping

Instead of explicitly specifying the mapping version of the Unicode mapping table to be used for conversion of a text string, you can use this constant to specify that the latest version be used. 

Available in Mac OS X v10.0 and later.
kUnicodeUseHFSPlusMapping

Indicates the mapping version used by HFS Plus to convert filenames between Mac OS encodings and Unicode. Only one constant is defined so far for a specific mapping version. 

Available in Mac OS X v10.0 and later.
Discussion

When performing conversions, you specify the version of the Unicode mapping table to be used for the conversion. You provide the version number in the mapping version field of the structure UnicodeMapping that is passed to a function. A Unicode mapping version is defined by the UnicodeMapVersion data type.*/
typedef SInt32 UnicodeMapVersion;
enum {
   kUnicodeUseLatestMapping = -1,
   kUnicodeUseHFSPlusMapping = 4
};

/*UnicodeToTextInfo

Defines a reference to an opaque Unicode to text converter object.

typedef struct OpaqueUnicodeToTextInfo * UnicodeToTextInfo;
Discussion

Many of the Unicode Converter functions that perform conversions require a Unicode converter object containing information used for the conversion process. There are three types of Unicode converter objects used for different types of conversions. You use the UnicodeToTextInfo type, described here, for converting from Unicode to text.

Because your application cannot directly create or modify the contents of the private Unicode converter object, the Unicode Converter provides functions to create and dispose of it. To create a Unicode converter object for converting from Unicode to text, your application must first call either the function CreateUnicodeToTextInfo or CreateUnicodeToTextInfoByEncoding. 

You can then pass this object to the function ConvertFromUnicodeToText or ConvertFromUnicodeToPString to identify the information used to perform the actual conversion. After you have finished using the object, you should release the memory allocated for it by calling the function DisposeUnicodeToTextInfo.

A Unicode converter object for this purpose is defined by the UnicodeToTextInfo data type. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct OpaqueUnicodeToTextInfo * UnicodeToTextInfo;

/*UnicodeMapping

Contains information for mapping to or from Unicode encoding.

struct UnicodeMapping {
   TextEncoding unicodeEncoding;
   TextEncoding otherEncoding;
   UnicodeMapVersion mappingVersion;
};
typedef struct UnicodeMapping UnicodeMapping;
typedef UnicodeMapping * UnicodeMappingPtr;
Fields
unicodeEncoding

A Unicode text encoding specification of type TextEncoding. 
otherEncoding

A text encoding specification for the text to be converted to or from Unicode.
mappingVersion

The version of the Unicode mapping table to be used. 
Discussion

A Unicode mapping structure contains a complete text encoding specification for a Unicode encoding, a complete non-Unicode text encoding specification giving the encoding for the text to be converted to or from Unicode, and the version of the mapping table to be used for conversion. You use a structure of this type to specify the text encodings to and from which the text string is to be converted. A Unicode mapping structure is defined by the UnicodeMapping data type. 

You can specify a variety of normalization options by setting up the Unicode mapping structure as described in the following.

To specify normal canonical decomposition according to Unicode 3.2 rules, with no exclusions ("Canonical decomposition 3.2"), set up the UnicodeMapping structure as follows:mapping.unicodeEncoding (in) = Unicode 2.x-3.x, kUnicodeNoSubset,  kUnicode16BitFormat
mapping.otherEncoding (out) = Unicode 2.x-3.x, kUnicodeCanonicalDecompVariant,  kUnicode16BitFormat
mapping.mappingVersion = kUnicodeUseLatestMapping


Examples:u00E0 -> u0061 + u0300
u0061 + u0300 -> u0061 + u0300
u03AC ->  u03B1 + u0301 (3.2 rules)
uF900 -> u8C48
u00E0 + u0323 -> u0061 + u0323 + u0300  (correct)


To specify canonical decomposition according to Unicode 3.2 rules, with HFS+ exclusions ("HFS+ decomposition 3.2"), set up the UnicodeMapping structure in one of the following ways. The second method is for compatibility with the old method of using mappingVersion = kUnicodeUseHFSPlusMapping.// Method 1
mapping.unicodeEncoding (in) = Unicode 2.x-3.x, kUnicodeNoSubset,  kUnicode16BitFormat
mapping.otherEncoding (out) = Unicode 2.x-3.x, kUnicodeHFSPlusDecompVariant,  kUnicode16BitFormat
mapping.mappingVersion = kUnicodeUseLatestMapping
// Method 2
mapping.unicodeEncoding (in) = Unicode 2.x-3.x, kUnicode16BitFormat,  kUnicode16BitFormat
mapping.otherEncoding (out) = Unicode 2.x, kUnicodeCanonicalDecompVariant,  kUnicode16BitFormat
mapping.mappingVersion = kUnicodeUseHFSPlusMapping


Examples:u00E0 -> u0061 + u0300
u0061 + u0300 -> u0061 + u0300
u03AC ->  u03B1 + u0301 (3.2 rules)
uF900 -> uF900 (decomposition excluded for HFS+)
u00E0 + u0323 -> u0061 + u0323 + u0300  (correct)


To specify normal canonical composition according to Unicode 3.2 rules, set up the UnicodeMapping structure as follows:mapping.unicodeEncoding (in) = Unicode 2.x-3.x, kUnicodeNoSubset,  kUnicode16BitFormat
mapping.otherEncoding (out) = Unicode 2.x-3.x, kUnicodeCanonicalCompVariant,  kUnicode16BitFormat
mapping.mappingVersion = kUnicodeUseLatestMapping


Examples:u00E0 -> u00E0
u0061 + u0300 -> u00E0
u03AC ->  u03AC
uF900 -> u8C48
u00E0 + u0323 -> u1EA1 u0300 (correct)


To specify canonical composition according to Unicode 3.2 rules, but using the HFS+ decomposition exclusions, set up the UnicodeMapping structure as follows. This is the form to use if you want to obtain a composed form that dervide from the decomposed form used for HFS+ filenames.mapping.unicodeEncoding (in) = Unicode 2.x-3.x, kUnicodeNoSubset,  kUnicode16BitFormat
mapping.otherEncoding (out) = Unicode 2.x-3.x, kUnicodeHFSPlusCompVariant,  kUnicode16BitFormat
mapping.mappingVersion = kUnicodeUseLatestMapping


Examples:u00E0 -> u00E0
u0061 + u0300 -> u00E0
u03AC ->  u03AC
uF900 -> uF900
u00E0 + u0323 -> u1EA1 u0300 (correct)

Availability
Available in Mac OS X v10.0 and later.*/

struct UnicodeMapping {
   TextEncoding unicodeEncoding;
   TextEncoding otherEncoding;
   UnicodeMapVersion mappingVersion;
};
typedef struct UnicodeMapping UnicodeMapping;
typedef UnicodeMapping * UnicodeMappingPtr;


/*ConstUnicodeMappingPtr

Defines a constant Unicode mapping pointer.

typedef const UnicodeMapping *     ConstUnicodeMappingPtr;
Discussion

Many Unicode Converter functions take a pointer to a Unicode mapping structure as a parameter. For functions that do not modify the Unicode mapping contents, the Unicode Converter provides a constant pointer to a Unicode mapping structure defined by the ConstUnicodeMappingPtr data type. 
Availability
Available in Mac OS X v10.0 and later.*/
typedef const UnicodeMapping *     ConstUnicodeMappingPtr;

/*Conversion Masks

Set or text for conversion flags.

enum {
   kUnicodeUseFallbacksMask = 1L << kUnicodeUseFallbacksBit,
   kUnicodeKeepInfoMask = 1L << kUnicodeKeepInfoBit,
   kUnicodeDirectionalityMask = 3L << kUnicodeDirectionalityBits,
   kUnicodeVerticalFormMask = 1L << kUnicodeVerticalFormBit,
   kUnicodeLooseMappingsMask = 1L << kUnicodeLooseMappingsBit,
   kUnicodeStringUnterminatedMask = 1L << kUnicodeStringUnterminatedBit,
   kUnicodeTextRunMask = 1L << kUnicodeTextRunBit,
   kUnicodeKeepSameEncodingMask = 1L << kUnicodeKeepSameEncodingBit,
   kUnicodeForceASCIIRangeMask = 1L << kUnicodeForceASCIIRangeBit,
   kUnicodeNoHalfwidthCharsMask = 1L << kUnicodeNoHalfwidthCharsBit,
   kUnicodeTextRunHeuristicsMask = 1L << kUnicodeTextRunHeuristicsBit,
   kUnicodeMapLineFeedToReturnMask = 1L << kUnicodeMapLineFeedToReturnBit
};
Constants
kUnicodeUseFallbacksMask

A mask for setting the Unicode-use-fallbacks conversion flag. The Unicode Converter uses fallback mappings when it encounters a source text element for which there is no equivalent destination encoding. Fallback mappings are mappings that do not preserve the meaning or identity of the source character but represent a useful approximation of it. See the function SetFallbackUnicodeToText.

Available in Mac OS X v10.0 and later.
kUnicodeKeepInfoMask

A mask for setting the keep-information control flag which governs whether the Unicode Converter keeps the current state stored in the Unicode converter object before converting the text string. 

If you clear this flag, the converter will initialize the Unicode converter object before converting the text string and assume that subsequent calls do not need any context, such as direction state for the current call. 

If you set the flag, the converter uses the current state. This is useful if your application must convert a stream of text in pieces that are not block delimited. You should set this flag for each call in a series of calls on the same text stream.

Available in Mac OS X v10.0 and later.
kUnicodeDirectionalityMask

A mask for setting the directionality control flag

Available in Mac OS X v10.0 and later.
kUnicodeVerticalFormMask

A mask for setting the vertical form control flag. The vertical form control flag tells the Unicode Converter how to map text elements for which there are both abstract and vertical presentation forms in the destination encoding. 

If set, the converter maps these text elements to their vertical forms, if they are available.

Available in Mac OS X v10.0 and later.
kUnicodeLooseMappingsMask

A mask that determines whether the Unicode Converter should use the loose-mapping portion of a mapping table for character mapping if the strict mapping portion of the table does not include a destination encoding equivalent for the source text element. 

If you clear this flag, the converter will use only the strict equivalence portion. 

If set this flag and a conversion for the source text element does not exist in the strict equivalence portion of the mapping table, then the converter uses the loose mapping section. 

Available in Mac OS X v10.0 and later.
kUnicodeStringUnterminatedMask

A mask for setting the string-unterminated control flag. Determines how the Unicode Converter handles text-element boundaries and direction resolution at the end of an input buffer.

If you clear this bit, the converter treats the end of the buffer as the end of text. 

If you set this bit, the converter assumes that the next call you make using the current context will supply another buffer of text that should be treated as a continuation of the current text. For example, if the last character in the input buffer is 'A', ConvertFromUnicodeToText stops conversion at the 'A' and returns kTECIncompleteElementErr, because the next buffer could begin with a combining diacritical mark that should be treated as part of the same text element. If the last character in the input buffer is a control character, ConvertFromUnicodeToText does not return kTECIncompleteElementErr because a control character could not be part of a multiple character text element. 

In attempting to analyze the text direction, when the Unicode Converter reaches the end of the current input buffer and the direction of the current text element is still unresolved, if you clear this flag, the converter treats the end of the buffer as a block separator for direction resolution. If you set this flag, it sets the direction as undetermined

Available in Mac OS X v10.0 and later.
kUnicodeTextRunMask

A mask for setting the text-run control flag which determines how the Unicode Converter converts Unicode text to a non-Unicode encoding when more than one possible destination encoding exists. 

If you clear this flag, the function ConvertFromUnicodeToTextRun or ConvertFromUnicodeToScriptCodeRun attempts to convert the Unicode text to the single encoding from the list of encodings in the Unicode converter object that produces the best result, that is, that provides for the greatest amount of source text conversion.

If you set this flag, ConvertFromUnicodeToTextRun or ConvertFromUnicodeToScriptCodeRun, which are the only functions to which it applies, may generate a destination string that combines text in any of the encodings specified by the Unicode converter object.

Available in Mac OS X v10.0 and later.
kUnicodeKeepSameEncodingMask

A mask for setting the keep-same-encoding control flag. Determines how the Unicode Converter treats the conversion of Unicode text following a text element that could not be converted to the first destination encoding when multiple destination encodings exist. This control flag applies only if the kUnicodeTextRunMask control flag is set. 

If you set this flag, the function ConvertFromUnicodeToTextRun attempts to minimize encoding changes in the conversion of the source text string; that is, once it is forced to make an encoding change, it attempts to use that encoding as the conversion destination for as long as possible. 

If you clear this flag, ConvertFromUnicodeToTextRun attempts to keep most of the converted string in one encoding, switching to other encodings only when necessary. 

Available in Mac OS X v10.0 and later.
kUnicodeForceASCIIRangeMask

A mask for setting the force ASCII range control flag. If an encoding normally treats 1-byte code points 0x00–0x7F as an ISO 646 national variant that is different from ASCII, setting this flag forces 0x00–0x7F to be treated as ASCII. For example, Japanese encodings such as Shift-JIS generally treat 0x00–0x7F as JIS Roman, with 0x5C as YEN SIGN instead of REVERSE SOLIDUS, but when converting a DOS file path you may want to set this flag so that 0x5C is mapped as REVERSE SOLIDUS.

Available in Mac OS X v10.0 and later.
kUnicodeNoHalfwidthCharsMask

Sets the no halfwidth characters control flag.

Available in Mac OS X v10.0 and later.
kUnicodeTextRunHeuristicsMask

Available in Mac OS X v10.0 and later.
kUnicodeMapLineFeedToReturnMask

Specifies mapping o the LF (LineFeed) character used in Unix to represent new lines to the CR (CarriageReturn) used in Mac encodings. This option has an effect only when used with the constant kUnicodeLooseMappingsMask. You can pass both constants as iControlFlags parameters to the functions ConvertFromUnicodeToText, ConvertFromUnicodeToTextRun, and ConvertFromUnicodeToScriptCodeRun.

Available in Mac OS X v10.2 and later.
Discussion

You use these constants to specify how the conversion of text from one encoding to another is performed. You use these masks as the controlFlags parameter in the ConvertFromTextToUnicode, ConvertFromUnicodeToText, ConvertFromUnicodeToScriptCodeRun, ConvertFromUnicodeToTextRun, and TruncateForUnicodeToText functions. A different subset of control masks applies to each of these functions. Using the bitmask constants, you can perform a bitwise OR operation to set the pertinent flags for a particular function’s parameters. For example, when you call a function, you might pass the following controlFlags parameter setting: 

controlflags=kUnicodeUseFallbacksMask | kUnicodeLooseMappingsMask;*/
enum {
   kUnicodeUseFallbacksMask = 1L << kUnicodeUseFallbacksBit,
   kUnicodeKeepInfoMask = 1L << kUnicodeKeepInfoBit,
   kUnicodeDirectionalityMask = 3L << kUnicodeDirectionalityBits,
   kUnicodeVerticalFormMask = 1L << kUnicodeVerticalFormBit,
   kUnicodeLooseMappingsMask = 1L << kUnicodeLooseMappingsBit,
   kUnicodeStringUnterminatedMask = 1L << kUnicodeStringUnterminatedBit,
   kUnicodeTextRunMask = 1L << kUnicodeTextRunBit,
   kUnicodeKeepSameEncodingMask = 1L << kUnicodeKeepSameEncodingBit,
   kUnicodeForceASCIIRangeMask = 1L << kUnicodeForceASCIIRangeBit,
   kUnicodeNoHalfwidthCharsMask = 1L << kUnicodeNoHalfwidthCharsBit,
   kUnicodeTextRunHeuristicsMask = 1L << kUnicodeTextRunHeuristicsBit,
   kUnicodeMapLineFeedToReturnMask = 1L << kUnicodeMapLineFeedToReturnBit
};


//function

/*CreateUnicodeToTextInfo

Creates and returns a Unicode converter object containing information required for converting strings from Unicode to a non-Unicode encoding. 

OSStatus CreateUnicodeToTextInfo (
   ConstUnicodeMappingPtr iUnicodeMapping,
   UnicodeToTextInfo * oUnicodeToTextInfo
);
Parameters
iUnicodeMapping

A pointer to a structure of type UnicodeMapping. Your application provides this structure to identify the mapping to be used for the conversion. The unicodeEncoding field of this structure can specify a Unicode format of kUnicode16BitFormat or kUnicodeUTF8Format. Note that the versions of the Unicode Converter prior to 1.2.1 do not support kUnicodeUTF8Format. 
oUnicodeToTextInfo

On return, a pointer to a Unicode converter object that holds the mapping table information you supply as the iUnicodeMapping parameter and the state information related to the conversion. The information contained in the Unicode converter object is required for the conversion of a Unicode string to a non-Unicode encoding. 
Return Value

A result code. See “TEC Manager Result Codes”.
Discussion

You pass the Unicode converter object returned from CreateUnicodeToTextInfo to the function ConvertFromUnicodeToText or ConvertFromUnicodeToPString to identify the information to be used for the conversion. These two functions modify the contents of the Unicode converter object. 

If an error is returned, the Unicode converter object is invalid. 

Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/

OSStatus CreateUnicodeToTextInfo (
   ConstUnicodeMappingPtr iUnicodeMapping,
   UnicodeToTextInfo * oUnicodeToTextInfo
);

/*ConvertFromUnicodeToText

Converts a Unicode text string to the destination encoding you specify. 

OSStatus ConvertFromUnicodeToText (
   UnicodeToTextInfo iUnicodeToTextInfo,
   ByteCount iUnicodeLen,
   const UniChar iUnicodeStr[],
   OptionBits iControlFlags,
   ItemCount iOffsetCount,
   ByteOffset iOffsetArray[],
   ItemCount * oOffsetCount,
   ByteOffset oOffsetArray[],
   ByteCount iOutputBufLen,
   ByteCount * oInputRead,
   ByteCount * oOutputLen,
   LogicalAddress oOutputStr
);
Parameters
iUnicodeToTextInfo

A Unicode converter object of type UnicodeToTextInfo for converting text from Unicode. You use the function CreateUnicodeToTextInfo or CreateUnicodeToTextInfoByEncoding to obtain a Unicode converter object to specify for this parameter. This function modifies the contents of the iUnicodeToTextInfo parameter. 
iUnicodeLen

The length in bytes of the Unicode string to be converted. 
iUnicodeStr

A pointer to the Unicode string to be converted. If the input text is UTF-8, which is supported for versions 1.2.1 or later of the converter, you must cast the UTF-8 buffer pointer to ConstUniCharArrayPtr before you can pass it as this parameter. 
iControlFlags

Conversion control flags. You can use “Conversion Masks” and “Directionality Masks” to set the iControlFlags parameter. 
iOffsetCount

The number of offsets contained in the array provided by the iOffsetArray parameter. Your application supplies this value. If you don’t want offsets returned to you, specify 0 (zero) for this parameter. 
iOffsetArray

An array of type ByteOffset. On input, you specify the array that gives an ordered list of significant byte offsets pertaining to the Unicode source string to be converted. These offsets may identify font or style changes, for example, in the source string. If you don’t want offsets returned to your application, specify NULL for this parameter and 0 (zero) for iOffsetCount. All offsets must be less than iUnicodeLen. 
oOffsetCount

On return, a pointer to the number of offsets that were mapped in the output stream.
oOffsetArray

An array of type ByteOffset. On return, this array contains the corresponding new offsets for the converted string in the new encoding.
iOutputBufLen

The length in bytes of the output buffer pointed to by the oOutputStr parameter. Your application supplies this buffer to hold the returned converted string. The oOutputLen parameter may return a byte count that is less than this value if the converted byte string is smaller than the buffer size you allocated.
oInputRead

On return, a pointer to a the number of bytes of the Unicode string that were converted. If the function returns a kTECUnmappableElementErr result code, this parameter returns the number of bytes that were converted before the error occurred.
oOutputLen

On return, a pointer to the length in bytes of the converted text stream.
oOutputStr

A value of type LogicalAddress. On input, this value points to a buffer for the converted string. On return, the buffer holds the converted text string. (For guidelines on estimating the size of the buffer needed, see the following discussion. 
Return Value

A result code. See “TEC Manager Result Codes”.
Discussion

This function can also map offsets for style or font information from the source text string to the returned converted string. The converter reads the application-supplied offsets and returns the corresponding new offsets in the converted string. If you do not want font or style information offsets mapped to the resulting string, you should pass NULL for iOffsetArray and 0 (zero) for iOffsetCount. 

Your application must allocate a buffer to hold the resulting converted string and pass a pointer to the buffer in the oOutputStr parameter. To determine the size of the output buffer to allocate, you should consider the size and content of the Unicode source string in relation to the type of encoding to which it will be converted. For example, for many encodings, such as MacRoman and Shift-JIS, the size of the returned string will be between half the size and the same size as the source Unicode string. However, for some encodings that are not Mac OS ones, such as EUC-JP, which has some 3-byte characters for Kanji, the returned string could be larger than the source Unicode string. For MacArabic and MacHebrew, the result will usually be less than half the size of the Unicode string. 
Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/
OSStatus ConvertFromUnicodeToText (
   UnicodeToTextInfo iUnicodeToTextInfo,
   ByteCount iUnicodeLen,
   const UniChar iUnicodeStr[],
   OptionBits iControlFlags,
   ItemCount iOffsetCount,
   ByteOffset iOffsetArray[],
   ItemCount * oOffsetCount,
   ByteOffset oOffsetArray[],
   ByteCount iOutputBufLen,
   ByteCount * oInputRead,
   ByteCount * oOutputLen,
   LogicalAddress oOutputStr
);

/*DisposeUnicodeToTextInfo

Releases the memory allocated for the specified Unicode converter object. 

OSStatus DisposeUnicodeToTextInfo (
   UnicodeToTextInfo * ioUnicodeToTextInfo
);
Parameters
ioUnicodeToTextInfo

A pointer to a Unicode converter object for converting from Unicode to a non-Unicode encoding. You specify a Unicode converter object that your application created using the function CreateUnicodeToTextInfo or CreateUnicodeToTextInfoByEncoding. You must not point to any other type of Unicode converter object. Your application should not attempt to dispose of the same Unicode converter object more than once. 
Return Value

A result code. See “TEC Manager Result Codes”. The function returns noErr if it disposes of the Unicode converter object successfully. If your application specifies an invalid Unicode converter object, such as NULL, the function returns a paramErr result code.
Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/

OSStatus DisposeUnicodeToTextInfo (
   UnicodeToTextInfo * ioUnicodeToTextInfo
);

/*CreateTextToUnicodeInfo

Creates and returns a Unicode converter object containing information required for converting strings from a non-Unicode encoding to Unicode. 

OSStatus CreateTextToUnicodeInfo (
   ConstUnicodeMappingPtr iUnicodeMapping,
   TextToUnicodeInfo * oTextToUnicodeInfo
);
Parameters
iUnicodeMapping

A pointer to a structure of type UnicodeMapping. Your application provides this structure to identify the mapping to use for the conversion. You must supply a value of type TextEncoding in the unicodeEncoding field of this structure. A TextEncoding is a triple composed of an encoding base, an encoding variant, and a format. You can obtain a UnicodeMapping value by calling the function CreateTextEncoding. 
oTextToUnicodeInfo

On return, the Unicode converter object holds mapping table information you supplied as the UnicodeMapping parameter and state information related to the conversion. This information is required for conversion of a text stream in a non-Unicode encoding to Unicode. 
Return Value

A result code. See “TEC Manager Result Codes”.
Discussion

You pass a Unicode converter object returned from the function CreateTextToUnicodeInfo to the function ConvertFromTextToUnicode or ConvertFromPStringToUnicode to identify the information to be used for the conversion. These two functions modify the contents of the object. 

You pass a Unicode converter object returned from CreateTextToUnicodeInfo to the function TruncateForTextToUnicode to identify the information to be used to truncate the string. This function does not modify the contents of the Unicode converter object.

If an error is returned, the Unicode converter object is invalid. 
Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/
OSStatus CreateTextToUnicodeInfo (
   ConstUnicodeMappingPtr iUnicodeMapping,
   TextToUnicodeInfo * oTextToUnicodeInfo
);

/*ConvertFromTextToUnicode

Converts a string from any encoding to Unicode. 

OSStatus ConvertFromTextToUnicode (
   TextToUnicodeInfo iTextToUnicodeInfo,
   ByteCount iSourceLen,
   ConstLogicalAddress iSourceStr,
   OptionBits iControlFlags,
   ItemCount iOffsetCount,
   ByteOffset iOffsetArray[],
   ItemCount * oOffsetCount,
   ByteOffset oOffsetArray[],
   ByteCount iOutputBufLen,
   ByteCount * oSourceRead,
   ByteCount * oUnicodeLen,
   UniChar oUnicodeStr[]
);
Parameters
iTextToUnicodeInfo

A Unicode converter object of type TextToUnicodeInfo containing mapping and state information used for the conversion. The contents of this Unicode converter object are modified by the function. Your application obtains a Unicode converter object using the function CreateTextToUnicodeInfo. 
iSourceLen

The length in bytes of the source string to be converted. 
iSourceStr

The address of the source string to be converted. 
iControlFlags

Conversion control flags. You can use “Conversion Masks” to set the iControlFlags parameter. 
iOffsetCount

The number of offsets in the iOffsetArray parameter. Your application supplies this value. The number of entries in iOffsetArray must be fewer than the number of bytes specified in iSourceLen. If you don’t want offsets returned to you, specify 0 (zero) for this parameter. 
iOffsetArray

An array of type ByteOffset. On input, you specify the array that contains an ordered list of significant byte offsets pertaining to the source string. These offsets may identify font or style changes, for example, in the source string. All array entries must be less than the length in bytes specified by the iSourceLen parameter. If you don’t want offsets returned to your application, specify NULL for this parameter and 0 (zero) for iOffsetCount. 
oOffsetCount

On return, a pointer to the number of offsets that were mapped in the output stream.
oOffsetArray

An array of type ByteOffset. On return, this array contains the corresponding new offsets for the Unicode string produced by the converter.
iOutputBufLen

The length in bytes of the output buffer pointed to by the oUnicodeStr parameter. Your application supplies this buffer to hold the returned converted string. The oUnicodeLen parameter may return a byte count that is less than this value if the converted byte string is smaller than the buffer size you allocated. The relationship between the size of the source string and the Unicode string is complex and depends on the source encoding and the contents of the string.
oSourceRead

On return, a pointer to the number of bytes of the source string that were converted. If the function returns a kTECUnmappableElementErr result code, this parameter returns the number of bytes that were converted before the error occurred.
oUnicodeLen

On return, a pointer to the length in bytes of the converted stream.
oUnicodeStr

A pointer to an array used to hold a Unicode string. On input, this value points to the beginning of the array for the converted string. On return, this buffer holds the converted Unicode string. (For guidelines on estimating the size of the buffer needed, see the discussion. 
Return Value

A result code. See “TEC Manager Result Codes”. The function returns a noErr result code if it has completely converted the input string to Unicode without using fallback characters.
Discussion

You specify the source string’s encoding in the Unicode mapping structure that you pass to the function CreateTextToUnicodeInfo to obtain a Unicode converter object for the conversion. You pass the Unicode converter object returned by CreateTextToUnicodeInfo to ConvertFromTextToUnicode as the iTextToUnicodeInfo parameter. 

In addition to converting a text string in any encoding to Unicode, the ConvertFromTextToUnicode function can map offsets for style or font information from the source text string to the returned converted string. The converter reads the application-supplied offsets, which apply to the source string, and returns the corresponding new offsets in the converted string. If you do not want the offsets at which font or style information occurs mapped to the resulting string, you should pass NULL for iOffsetArray and 0 (zero) for iOffsetCount. 

Your application must allocate a buffer to hold the resulting converted string and pass a pointer to the buffer in the oUnicodeStr parameter. To determine the size of the output buffer to allocate, you should consider the size of the source string, its encoding type, and its content in relation to the resulting Unicode string. 

For example, for 1-byte encodings, such as MacRoman, the Unicode string will be at least double the size (more if it uses noncomposed Unicode) for MacArabic and MacHebrew, the corresponding Unicode string could be up to six times as big. For most 2-byte encodings, for example Shift-JIS, the Unicode string will be less than double the size. For international robustness, your application should allocate a buffer three to four times larger than the source string. If the output Unicode text is actually UTF-8—which could occur beginning with the current release of the Text Encoding Conversion Manager, version 1.2.1—the UTF-8 buffer pointer must be cast to UniCharArrayPtr before it can be passed as the oUnicodeStr parameter. Also, the output buffer length will have a wider range of variation than for UTF-16; for ASCII input, the output will be the same size; for Han input, the output will be twice as big, and so on. 
Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/
OSStatus ConvertFromTextToUnicode (
   TextToUnicodeInfo iTextToUnicodeInfo,
   ByteCount iSourceLen,
   ConstLogicalAddress iSourceStr,
   OptionBits iControlFlags,
   ItemCount iOffsetCount,
   ByteOffset iOffsetArray[],
   ItemCount * oOffsetCount,
   ByteOffset oOffsetArray[],
   ByteCount iOutputBufLen,
   ByteCount * oSourceRead,
   ByteCount * oUnicodeLen,
   UniChar oUnicodeStr[]
);

/*DisposeTextToUnicodeInfo

Releases the memory allocated for the specified Unicode converter object. 

OSStatus DisposeTextToUnicodeInfo (
   TextToUnicodeInfo * ioTextToUnicodeInfo
);
Parameters
ioTextToUnicodeInfo

A pointer to a Unicode converter object of type TextToUnicodeInfo, used for converting text to Unicode. On input, you specify the object to dispose. It must be an object which your application created using the function CreateTextToUnicodeInfo or CreateTextToUnicodeInfoByEncoding. You must not point to any other type of Unicode converter object. Your application should not use this function with the same structure more than once. 
Return Value

A result code. See “TEC Manager Result Codes”. If your application specifies an invalid Unicode converter object, such as NULL, the function returns a paramErr result code.
Availability
Available in CarbonLib 1.0 and later when Unicode Utilities 1.1 or later is present.
Available in Mac OS X 10.0 and later.
Declared In
UnicodeConverter.h*/
OSStatus DisposeTextToUnicodeInfo (
   TextToUnicodeInfo * ioTextToUnicodeInfo
);


#ifdef __cplusplus
}
#endif

#endif