
#ifndef CFSTRING_H
 #define CFSTRING_H

#include <CoreFoundation/CFBase.h>

#ifndef CFARRAY_H
#include <CoreFoundation/CFArray.h>
#endif

#include <CoreFoundation/CFData.h>
#include <CoreFoundation/CFDictionary.h>
#include <CoreFoundation/CFCharacterSet.h>
#include <CoreFoundation/CFLocale.h>
#include <stdarg.h>

#include <ETC/CoreFoundation/CFString.h>
#include <HYS/CoreFoundation/CFString.h>

#if defined(__cplusplus)
extern "C" {
#endif


/*String Comparison Flags

Flags that specify how string comparisons are performed.

typedef enum {
   kCFCompareCaseInsensitive = 1,
   kCFCompareBackwards = 4,
   kCFCompareAnchored = 8,
   kCFCompareNonliteral = 16,
   kCFCompareLocalized = 32,
   kCFCompareNumerically = 64
};
CFStringCompareFlags;
Constants
kCFCompareCaseInsensitive

Specifies that the comparison should ignore differences in case between alphabetical characters.

Available in Mac OS X v10.0 and later.
kCFCompareBackwards

Specifies that the comparison should start at the last elements of the entities being compared (for example, strings or arrays).

Available in Mac OS X v10.0 and later.
kCFCompareAnchored

Performs searching only on characters at the beginning or end of the range.

No match at the beginning or end means nothing is found, even if a matching sequence of characters occurs elsewhere in the string.

Available in Mac OS X v10.0 and later.
kCFCompareNonliteral

Specifies that loose equivalence is acceptable, especially as pertains to diacritical marks. 

For example, “ö” represented as two distinct characters (“o” and “umlaut”) is equivalent to “ö” represented by a single character (“o-umlaut”). Note that this is not the same as diacritic insensitivity.

Available in Mac OS X v10.0 and later.
kCFCompareLocalized

Specifies that the comparison should take into account differences related to locale, such as the thousands separator character.

Available in Mac OS X v10.0 and later.
kCFCompareNumerically

Specifies that represented numeric values should be used as the basis for comparison and not the actual character values.

For example, “version 2” is less than “version 2.5”. Does not work if kCFCompareLocalized is specified on systems before 10.3.

Available in Mac OS X v10.0 and later.
Discussion

These constants are flags intended for use in the comparison-option parameters in comparison functions such as CFStringCompare. If you want to request multiple options, combine them with a bitwise-OR operation. 
Declared In
CFString.h*/

typedef enum {
   kCFCompareCaseInsensitive = 1,
   kCFCompareBackwards = 4,
   kCFCompareAnchored = 8,
   kCFCompareNonliteral = 16,
   kCFCompareLocalized = 32,
   kCFCompareNumerically = 64
};


/*CFStringEncoding

An integer type for constants used to specify supported string encodings in various CFString functions.

typedef UInt32 CFStringEncoding;
Discussion

This type is used to define the constants for the built-in encodings (see Built-in String Encodings for a list) and for platform-dependent encodings (see External String Encodings). If CFString does not recognize or support the string encoding of a particular string, CFString functions will identify the strings encoding as kCFStringEncodingInvalidId.
Availability
Available in Mac OS X v10.0 and later.*/
typedef UInt32 CFStringEncoding;


/*CFAllocatorCopyDescriptionCallBack

A prototype for a function callback that provides a description of the specified data.

typedef CFStringRef (*CFAllocatorCopyDescriptionCallBack) (
   const void *info
);

If you name your function MyCallBack, you would declare it like this:

CFStringRef MyCallBack (
   const void *info
);
Parameters
info

An untyped pointer to program-defined data.
Return Value

A CFString object that describes the allocator. The caller is responsible for releasing this object.
Discussion

A prototype for a function callback that provides a description of the data pointed to by the info field. In implementing this function, return a reference to a CFString object that describes your allocator, particularly some characteristics of your program-defined data.*/
typedef CFStringRef (*CFAllocatorCopyDescriptionCallBack) (
   const void *info
);


/*Built-in String Encodings

CFStringEncoding constants for the supported encodings that are built-in on all platforms on which Mac OS X runs.

enum {
   kCFStringEncodingMacRoman = 0,
   kCFStringEncodingWindowsLatin1 = 0x0500,
   kCFStringEncodingISOLatin1 = 0x0201,
   kCFStringEncodingNextStepLatin = 0x0B01,
   kCFStringEncodingASCII = 0x0600,
   kCFStringEncodingUnicode = 0x0100,
   kCFStringEncodingUTF8 = 0x08000100,
   kCFStringEncodingNonLossyASCII = 0x0BFF,
   
   kCFStringEncodingUTF16 = 0x0100,
   kCFStringEncodingUTF16BE = 0x10000100,
   kCFStringEncodingUTF16LE = 0x14000100,
   kCFStringEncodingUTF32 = 0x0c000100,
   kCFStringEncodingUTF32BE = 0x18000100,
   kCFStringEncodingUTF32LE = 0x1c000100
} CFStringBuiltInEncodings;
Constants
kCFStringEncodingMacRoman

An encoding constant that identifies the Mac Roman encoding.

Available in Mac OS X v10.0 and later.
kCFStringEncodingWindowsLatin1

An encoding constant that identifies the Windows Latin 1 encoding (ANSI codepage 1252).

Available in Mac OS X v10.0 and later.
kCFStringEncodingISOLatin1

An encoding constant that identifies the ISO Latin 1 encoding (ISO 8859-1)

Available in Mac OS X v10.0 and later.
kCFStringEncodingNextStepLatin

An encoding constant that identifies the NextStep/OpenStep encoding.

Available in Mac OS X v10.0 and later.
kCFStringEncodingASCII

An encoding constant that identifies the ASCII encoding (decimal values 0 through 127).

Available in Mac OS X v10.0 and later.
kCFStringEncodingUnicode

An encoding constant that identifies the Unicode encoding. 

Available in Mac OS X v10.0 and later.
kCFStringEncodingUTF8

An encoding constant that identifies the UTF 8 encoding.

Available in Mac OS X v10.0 and later.
kCFStringEncodingNonLossyASCII

An encoding constant that identifies non-lossy ASCII encoding.

Available in Mac OS X v10.0 and later.
kCFStringEncodingUTF16

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF16Format encoding (alias of kCFStringEncodingUnicode).

Available in Mac OS X v10.4 and later.
kCFStringEncodingUTF16BE

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF16BEFormat encoding.

Available in Mac OS X v10.4 and later.
kCFStringEncodingUTF16LE

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF16LEFormat encoding.

Available in Mac OS X v10.4 and later.
kCFStringEncodingUTF32

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF32Format encoding.

Available in Mac OS X v10.4 and later.
kCFStringEncodingUTF32BE

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF32BEFormat encoding.

Available in Mac OS X v10.4 and later.
kCFStringEncodingUTF32LE

An encoding constant that identifies kTextEncodingUnicodeDefault + kUnicodeUTF32LEFormat encoding.

Available in Mac OS X v10.4 and later.
Declared In
CFString.h*/

enum {
   kCFStringEncodingMacRoman = 0,
   kCFStringEncodingWindowsLatin1 = 0x0500,
   kCFStringEncodingISOLatin1 = 0x0201,
   kCFStringEncodingNextStepLatin = 0x0B01,
   kCFStringEncodingASCII = 0x0600,
   kCFStringEncodingUnicode = 0x0100,
   kCFStringEncodingUTF8 = 0x08000100,
   kCFStringEncodingNonLossyASCII = 0x0BFF,
   
   kCFStringEncodingUTF16 = 0x0100,
   kCFStringEncodingUTF16BE = 0x10000100,
   kCFStringEncodingUTF16LE = 0x14000100,
   kCFStringEncodingUTF32 = 0x0c000100,
   kCFStringEncodingUTF32BE = 0x18000100,
   kCFStringEncodingUTF32LE = 0x1c000100
} CFStringBuiltInEncodings;

/*CFStringGetCString

Copies the character contents of a string to a local C string buffer after converting the characters to a given encoding.

Boolean CFStringGetCString (
   CFStringRef theString,
   char *buffer,
   CFIndex bufferSize,
   CFStringEncoding encoding
);
Parameters
theString

The string whose contents you wish to access. 
buffer

The C string buffer into which to copy the string. The buffer must be at least bufferSize bytes in length. On return, the buffer contains the converted characters. If there is an error in conversion, the buffer contains only partial results.
bufferSize

The length of the local buffer in bytes (accounting for the NULL-terminator byte). 
encoding

The string encoding to which the character contents of theString should be converted. 
Return Value

true upon success or false if the conversion fails or the provided buffer is too small. 
Discussion

This function is useful when you need your own copy of a strings character data as a C string. You also typically call it as a backup when a prior call to the CFStringGetCStringPtr function fails. 
Availability
Available in Mac OS X v10.0 and later.*/
Boolean CFStringGetCString (
   CFStringRef theString,
   char *buffer,
   CFIndex bufferSize,
   CFStringEncoding encoding
);

/*CFStringCreateMutable

Creates an empty CFMutableString object.

CFMutableStringRef CFStringCreateMutable (
   CFAllocatorRef alloc,
   CFIndex maxLength
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
maxLength

The maximum number of Unicode characters that can be stored by the returned string. Pass 0 if there should be no character limit. Note that initially the string still has a length of 0; this parameter simply specifies what the maximum size is. CFMutableString might try to optimize its internal storage by paying attention to this value. 
Return Value

A new empty CFMutableString object or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Discussion

This function creates an empty (that is, content-less) CFMutableString object. You can add character data to this object with any of the CFStringAppend... functions, and thereafter you can insert, delete, replace, pad, and trim characters with the appropriate CFString functions. If the maxLength parameter is greater than 0, any attempt to add characters beyond this limit results in a run-time error. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
CFMutableStringRef CFStringCreateMutable (
   CFAllocatorRef alloc,
   CFIndex maxLength
);

/*CFStringReplaceAll

Replaces all characters of a CFMutableString object with other characters.

void CFStringReplaceAll (
   CFMutableStringRef theString,
   CFStringRef replacement
);
Parameters
theString

The string to modify. If this value is not a CFMutableString object, an assertion is raised. 
replacement

The replacement string to put into theString.
Discussion

The character buffer of theString is resized according to the length of the new characters.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringReplaceAll (
   CFMutableStringRef theString,
   CFStringRef replacement
);

/*CFStringGetSystemEncoding

Returns the default encoding used by the operating system when it creates strings.

CFStringEncoding CFStringGetSystemEncoding ();
Return Value

The default string encoding. 
Discussion

This function returns the default text encoding used by the OS when it creates strings. In Mac OS X, this encoding is determined by the user's preferred language setting. The preferred language is the first language listed in the International pane of the System Preferences.

In most situations you will not want to use this function, however, because your primary interest will be your application's default text encoding. The application encoding is required when you create a CFStringRef from strings stored in Resource Manager resources, which typically use one of the Mac encodings such as MacRoman or MacJapanese.

To get your application's default text encoding, call the GetApplicationTextEncoding Carbon function.
Availability
Available in Mac OS X v10.0 and later.*/
CFStringEncoding CFStringGetSystemEncoding ();

/*CFStringGetCStringPtr

Quickly obtains a pointer to a C-string buffer containing the characters of a string in a given encoding.

const char *CFStringGetCStringPtr (
   CFStringRef theString,
   CFStringEncoding encoding
);
Parameters
theString

The string whose contents you wish to access. 
encoding

The string encoding to which the character contents of theString should be converted. 
Return Value

A pointer to a C string or NULL if the internal storage of theString does not allow this to be returned efficiently. 
Discussion

This function either returns the requested pointer immediately, with no memory allocations and no copying, in constant time, or returns NULL. If the latter is the result, call an alternative function such as the CFStringGetCString function to extract the characters.

Whether or not this function returns a valid pointer or NULL depends on many factors, all of which depend on how the string was created and its properties. In addition, the function result might change between different releases and on different platforms. So do not count on receiving a non-NULL result from this function under any circumstances. 
Availability
Available in Mac OS X v10.0 and later.*/
const char *CFStringGetCStringPtr (
   CFStringRef theString,
   CFStringEncoding encoding
);

/*CFStringCompare

Compares one string with another string.

CFComparisonResult CFStringCompare (
   CFStringRef theString1,
   CFStringRef theString2,
   CFOptionFlags compareOptions
);
Parameters
theString1

The first string to use in the comparison. 
theString2

The second string to use in the comparison.
compareOptions

Flags that select different types of comparisons, such as localized comparison, case-insensitive comparison, and non-literal comparison. If you want the default comparison behavior, pass 0. See String Comparison Flags for the available flags.
Return Value

A CFComparisonResult value that indicates whether theString1 is equal to, less than, or greater than theString2. 
Discussion

You can affect how the comparison proceeds by specifying one or more option flags in compareOptions. Not all comparison options are currently implemented. 
Availability
Available in Mac OS X v10.0 and later.*/
CFComparisonResult CFStringCompare (
   CFStringRef theString1,
   CFStringRef theString2,
   CFOptionFlags compareOptions
);

/*CFStringAppend

Appends the characters of a string to those of a CFMutableString object.

void CFStringAppend (
   CFMutableStringRef theString,
   CFStringRef appendedString
);
Parameters
theString

The string to which appendedString is appended. If theString is not a CFMutableString object, an assertion is raised.
appendedString

The string to append. 
Discussion

This function reallocates the backing store of theString to accommodate the new length.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringAppend (
   CFMutableStringRef theString,
   CFStringRef appendedString
);

/*CFStringCreateMutableCopy

Creates a mutable copy of a string.

CFMutableStringRef CFStringCreateMutableCopy (
   CFAllocatorRef alloc,
   CFIndex maxLength,
   CFStringRef theString
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
maxLength

The maximum number of Unicode characters that can be stored by the returned object. Pass 0 if there should be no character limit. Note that initially the returned object still has the same length as the string argument; this parameter simply specifies what the maximum size is. CFString might try to optimize its internal storage by paying attention to this value. 
theString

A string to copy. 
Return Value

A string that has the same contents as theString. Returns NULL if there was a problem copying the object. Ownership follows the Create Rule.
Discussion

The returned mutable string is identical to the original string except for (perhaps) the mutability attribute. You can add character data to the returned string with any of the CFStringAppend... functions, and you can insert, delete, replace, pad, and trim characters with the appropriate CFString functions. If the maxLength parameter is greater than 0, any attempt to add characters beyond this limit results in a run-time error. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/CFMutableStringRef CFStringCreateMutableCopy (
   CFAllocatorRef alloc,
   CFIndex maxLength,
   CFStringRef theString
);

/*CFStringAppend

Appends the characters of a string to those of a CFMutableString object.

void CFStringAppend (
   CFMutableStringRef theString,
   CFStringRef appendedString
);
Parameters
theString

The string to which appendedString is appended. If theString is not a CFMutableString object, an assertion is raised.
appendedString

The string to append. 
Discussion

This function reallocates the backing store of theString to accommodate the new length.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/

void CFStringAppend (
   CFMutableStringRef theString,
   CFStringRef appendedString
);

/*CFStringInsert

Inserts a string at a specified location in the character buffer of a CFMutableString object.

void CFStringInsert (
   CFMutableStringRef str,
   CFIndex index,
   CFStringRef insertedStr
);
Parameters
str

The string to be modified. If this value is not a CFMutableString object, an assertion is raised. 
index

The index of the character in str after which the new characters are to be inserted. If the index is out of bounds, an assertion is raised. 
insertedStr

The string to insert into str. 
Discussion

To accommodate the new characters, this function moves any existing characters to the right of the inserted characters the appropriate number of positions. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringInsert (
   CFMutableStringRef str,
   CFIndex index,
   CFStringRef insertedStr
);

/*CFStringGetBytes

Fetches a range of the characters from a string into a byte buffer after converting the characters to a specified encoding.

CFIndex CFStringGetBytes (
   CFStringRef theString,
   CFRange range,
   CFStringEncoding encoding,
   UInt8 lossByte,
   Boolean isExternalRepresentation,
   UInt8 *buffer,
   CFIndex maxBufLen,
   CFIndex *usedBufLen
);
Parameters
theString

The string upon which to operate. 
range

The range of characters in theString to process. The specified range must not exceed the length of the string.
encoding

The string encoding of the characters to copy to the byte buffer. 
lossByte

A character (for example, '?') that should be substituted for characters that cannot be converted to the specified encoding. Pass 0 if you do not want lossy conversion to occur. 
isExternalRepresentation

true if you want the result to be in an “external representation” format. In such a format, it may contain a byte order marker (BOM) specifying endianness and this function might have to perform byte swapping. 
buffer

The byte buffer into which the converted characters are written. The buffer can be allocated on the heap or stack. Pass NULL if you do not want conversion to take place but instead want to know if conversion will succeed (the function result is greater than 0) and, if so, how many bytes are required (usedBufLen).
maxBufLen

The size of buffer and the maximum number of bytes that can be written to it. 
usedBufLen

On return, the number of converted bytes actually in buffer. You may pass NULL if you are not interested in this information.
Return Value

The number of characters converted. 
Discussion

This function is the basic encoding-conversion function for CFString objects. As with the other functions that get the character contents of CFString objects, it allows conversion to a supported 8-bit encoding. Unlike most of those other functions, it also allows “lossy conversion.” The function permits the specification of a “loss byte” in a parameter; if a character cannot be converted this character is substituted and conversion proceeds. (With the other functions, conversion stops at the first error and the operation fails.)

Because this function takes a range and returns the number of characters converted, it can be called repeatedly with a small fixed size buffer and different ranges of the string to do the conversion incrementally.

This function also handles any necessary manipulation of character data in an “external representation” format. This format makes the data portable and persistent (disk-writable); in Unicode it often includes a BOM (byte order marker) that specifies the endianness of the data.

The CFStringCreateExternalRepresentation function also handles external representations and performs lossy conversions. The complementary function CFStringCreateWithBytes creates a string from the characters in a byte buffer.
Availability
Available in Mac OS X v10.0 and later.*/
CFIndex CFStringGetBytes (
   CFStringRef theString,
   CFRange range,
   CFStringEncoding encoding,
   UInt8 lossByte,
   Boolean isExternalRepresentation,
   UInt8 *buffer,
   CFIndex maxBufLen,
   CFIndex *usedBufLen
);

/*CFStringGetLength

Returns the number (in terms of UTF-16 code pairs) of characters in a string.

CFIndex CFStringGetLength (
   CFStringRef theString
);
Parameters
theString

The string to examine. 
Return Value

The number (in terms of UTF-16 code pairs) of characters stored in theString. 
Availability
Available in Mac OS X v10.0 and later.*/
CFIndex CFStringGetLength (
   CFStringRef theString
);

/*CFStringHasPrefix

Determines if the character data of a string begin with a specified sequence of characters.

Boolean CFStringHasPrefix (
   CFStringRef theString,
   CFStringRef prefix
);
Parameters
theString

The string to search. 
prefix

The prefix to search for.
Return Value

true if theString begins with prefix, false if otherwise. 
Availability
Available in Mac OS X v10.0 and later.*/
Boolean CFStringHasPrefix (
   CFStringRef theString,
   CFStringRef prefix
);

/*CFStringAppendCString

Appends a C string to the character contents of a CFMutableString object.

void CFStringAppendCString (
   CFMutableStringRef theString,
   const char *cStr,
   CFStringEncoding encoding
);
Parameters
theString

The string to which the characters from cStr are appended. If this value is not a CFMutableString object, an assertion is raised. 
cStr

A pointer to a C string buffer.
encoding

The encoding of the characters in cStr. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringAppendCString (
   CFMutableStringRef theString,
   const char *cStr,
   CFStringEncoding encoding
);

/*CFStringFindAndReplace

Replaces all occurrences of a substring within a given range.

CFIndex CFStringFindAndReplace (
   CFMutableStringRef theString,
   CFStringRef stringToFind,
   CFStringRef replacementString,
   CFRange rangeToSearch,
   CFOptionFlags compareOptions
);
Parameters
theString

The string to modify. 
stringToFind

The substring to search for in theString.
replacementString

The replacement string for stringToFind. 
rangeToSearch

The range within which to search in theString. 
compareOptions

Flags that select different types of comparisons, such as localized comparison, case-insensitive comparison, and non-literal comparison. If you want the default comparison behavior, pass 0. See “String Comparison Flags” for the available flags.
Return Value

The number of instances of stringToFind that were replaced. 
Discussion

The possible values of compareOptions are combinations of the kCFCompareCaseInsensitive, kCFCompareBackwards, kCFCompareNonliteral, and kCFCompareAnchored constants. 

The kCFCompareBackwards option can be used to replace a substring starting from the end, which could produce different results. For example, if the parameter theString is “AAAAA”, stringToFind is “AA”, and replacementString is “B”, then the result is normally “BBA”. However, if the kCFCompareBackwards constant is used, the result is “ABB.” 

The kCFCompareAnchored option assures that only anchored but multiple instances are found (the instances must be consecutive at start or end). For example, if the parameter theString is “AAXAA”, stringToFind is “A”, and replacementString is “B”, then the result is normally “BBXBB.” However, if the kCFCompareAnchored constant is used, the result is “BBXAA.”
Availability
Available in Mac OS X v10.2 and later.*/
CFIndex CFStringFindAndReplace (
   CFMutableStringRef theString,
   CFStringRef stringToFind,
   CFStringRef replacementString,
   CFRange rangeToSearch,
   CFOptionFlags compareOptions
);

/*CFStringGetMaximumSizeForEncoding

Returns the maximum number of bytes a string of a specified length (in Unicode characters) will take up if encoded in a specified encoding.

CFIndex CFStringGetMaximumSizeForEncoding (
   CFIndex length,
   CFStringEncoding encoding
);
Parameters
length

The number of Unicode characters to evaluate. 
encoding

The string encoding for the number of characters specified by length. 
Return Value

The maximum number of bytes that could be required to represent length number of Unicode characters with the string encoding encoding. The number of bytes that the encoding actually ends up requiring when converting any particular string could be less than this, but never more.
Availability
Available in Mac OS X v10.0 and later.*/
CFIndex CFStringGetMaximumSizeForEncoding (
   CFIndex length,
   CFStringEncoding encoding
);

/*CFSTR

Creates an immutable string from a constant compile-time string.

CFStringRef CFSTR (
   const char *cStr
);
Parameters
cStr

A C string (that is, text enclosed in double-quotation marks) from which the string is to be created. The characters enclosed by the quotation marks must be ASCII characters, otherwise the behavior is undefined.
Return Value

An immutable string, or NULL if there was a problem creating the object. The returned object is a constant. You may retain and release it, similar to other immutable CFString objects, but are not required to do so—it will remain valid until the program terminates.
Discussion

The CFSTR macro is a convenient way to create CFString representations of constant compile-time strings.

A value returned by CFSTR has the following semantics:
Values returned from CFSTR are not released by CFString—they are guaranteed to be valid until the program terminates. 
Values returned from CFSTR can be retained and released in a balanced fashion, like any other CFString, but you are not required to do so. 

Non-ASCII characters (that is, character codes greater than 127) are not supported. If you use them, the result is undefined. Even if using them works for you in testing, it might not work if the user selects a different language preference. 

Note that when using this macro as an initializer, you must compile using the flag -fconstant-cfstrings (see Options Controlling C Dialect).
Availability
Available in Mac OS X v10.0 and later.*/

CFStringRef CFSTR (
   const char *cStr
);

/*CFShow

Prints a description of a Core Foundation object.

void CFShow (
   CFTypeRef obj
);
Parameters
obj

A Core Foundation object derived from CFType. If obj is not a Core Foundation object, an assertion is raised. 
Discussion

This function is useful as a debugging aid for Core Foundation objects. Because these objects are based on opaque types, it is difficult to examine their contents directly. However, the opaque types implement description function callbacks that return descriptions of their objects. This function invokes these callbacks. 
Special Considerations

You can use CFShow in one of two general ways. If your debugger supports function calls (such as gdb does), call CFShow in the debugger:(gdb) call (void) CFShow(string)
Hello World


You can also incorporate calls to CFShow in a test version of your code to print out "snapshots" of Core Foundation objects to the console.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFString.h*/
void CFShow (
   CFTypeRef obj
);

/*CFStringCreateWithCString

Creates an immutable string from a C string.

CFStringRef CFStringCreateWithCString (
   CFAllocatorRef alloc,
   const char *cStr,
   CFStringEncoding encoding
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator.
cStr

The NULL-terminated C string to be used to create the CFString object.
encoding

The encoding of the characters in the C string. 
Return Value

An immutable string containing cStr (after stripping off the NULL terminating character), or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Discussion

A C string is a string of 8-bit characters terminated with an 8-bit NULL. Unichar and Unichar32 are not considered C strings.
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateWithCString (
   CFAllocatorRef alloc,
   const char *cStr,
   CFStringEncoding encoding
);

/*CFStringCreateArrayBySeparatingStrings

Creates an array of CFString objects from a single CFString object.

CFArrayRef CFStringCreateArrayBySeparatingStrings (
   CFAllocatorRef alloc,
   CFStringRef theString,
   CFStringRef separatorString
);
Parameters
alloc

The allocator to use to allocate memory for the new CFArray object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
theString

The string to be divided into substrings. The substrings should be separated by separatorString. 
separatorString

A string containing the character or characters used to separate the substrings in theString.
Return Value

A new array that contains CFString objects that represent substrings of theString, or NULL if there was a problem creating the object. The order of elements in the array is identical to the order of the substrings in theString. If separatorString does not occur in theString, the result is an array containing theString. If separatorString is equal to theString, then the result is an array containing two empty strings. Ownership follows the Create Rule.
Discussion

This function provides a convenient way to convert units of data captured in a single string to a form (an array) suitable for iterative processing. One or more delimiter characters (or “separator string”) separates the substrings in the source string—these characters are frequently whitespace characters such as tabs and newlines (carriage returns). For example, you might have a file containing a localized list of place names with each name separated by a tab character. You could create a CFString object from this file and call this function on the string to obtain a CFArray object whose elements are these place names.

See also CFStringCreateByCombiningStrings. 
Availability
Available in Mac OS X v10.0 and later.*/

CFArrayRef CFStringCreateArrayBySeparatingStrings (
   CFAllocatorRef alloc,
   CFStringRef theString,
   CFStringRef separatorString
);

/*CFStringCreateWithCharacters

Creates a string from a buffer of Unicode characters.

CFStringRef CFStringCreateWithCharacters (
   CFAllocatorRef alloc,
   const UniChar *chars,
   CFIndex numChars
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
chars

The buffer of Unicode characters to copy into the new string. 
numChars

The number of characters in the buffer pointed to by chars. Only this number of characters will be copied to internal storage. 
Return Value

An immutable string containing chars, or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Discussion

This function creates an immutable string from a client-supplied Unicode buffer. You must supply a count of the characters in the buffer. This function always copies the characters in the provided buffer into internal storage.

To save memory, this function might choose to store the characters internally in a 8-bit backing store. That is, just because a buffer of UniChar characters was used to initialize the object does not mean you will get back a non-NULL result from CFStringGetCharactersPtr. 
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateWithCharacters (
   CFAllocatorRef alloc,
   const UniChar *chars,
   CFIndex numChars
);

/*CFStringCreateWithFormat

Creates an immutable string from a formatted string and a variable number of arguments.

CFStringRef CFStringCreateWithFormat (
   CFAllocatorRef alloc,
   CFDictionaryRef formatOptions,
   CFStringRef format,
   ...
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
formatOptions

A CFDictionary object containing formatting options for the string (such as the thousand-separator character, which is dependent on locale). Currently, these options are an unimplemented feature. 
format

The formatted string with printf-style specifiers. For information on supported specifiers, see String Format Specifiers.
...

Variable list of the values to be inserted in format.
Return Value

An immutable string, or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateWithFormat (
   CFAllocatorRef alloc,
   CFDictionaryRef formatOptions,
   CFStringRef format,
   ...
);


/*CFStringAppendFormat

Appends a formatted string to the character contents of a CFMutableString object.

void CFStringAppendFormat (
   CFMutableStringRef theString,
   CFDictionaryRef formatOptions,
   CFStringRef format,
   ...
);
Parameters
theString

The string to which the formatted characters from format are appended. If this value is not a CFMutableString object, an assertion is raised.
formatOptions

A dictionary containing formatting options for the string (such as the thousand-separator character, which is dependent on locale). Currently, these options are an unimplemented feature. 
format

A formatted string with printf-style specifiers. 
...

Variable list of the values to be inserted in format.
Discussion

A formatted string is one with printf-style format specifiers embedded in the text such as %d (decimal), %f (double), and %@ (Core Foundation object). The subsequent arguments, in order, are substituted for the specifiers in the character data appended to theString. You can also reorder the arguments in the string by using modifiers of the form "n$" with the format specifiers (for example, %2$d).

For more information on supported specifiers, see the relevant section in “Strings”. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/

void CFStringAppendFormat (
   CFMutableStringRef theString,
   CFDictionaryRef formatOptions,
   CFStringRef format,
   ...
);

/*CFStringCreateWithSubstring

Creates an immutable string from a segment (substring) of an existing string.

CFStringRef CFStringCreateWithSubstring (
   CFAllocatorRef alloc,
   CFStringRef str,
   CFRange range
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
str

The string from which to create the new string. 
range

The range of characters in str to copy. The specified range must not exceed the length of the string.
Return Value

An immutable string, or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateWithSubstring (
   CFAllocatorRef alloc,
   CFStringRef str,
   CFRange range
);

/*CFStringGetCharacterAtIndex

Returns the Unicode character at a specified location in a string.

UniChar CFStringGetCharacterAtIndex (
   CFStringRef theString,
   CFIndex idx
);
Parameters
theString

The string from which the Unicode character is obtained. 
idx

The position of the Unicode character in the CFString. 
Return Value

A Unicode character. 
Discussion

This function is typically called in a loop to fetch the Unicode characters of a string in sequence or to fetch a character at a known position (first or last, for example). Using it in a loop can be inefficient, especially with longer strings, so consider the CFStringGetCharacters function or the in-line buffer functions (CFStringInitInlineBuffer and CFStringGetCharacterFromInlineBuffer) as alternatives.
Availability
Available in Mac OS X v10.0 and later.*/
UniChar CFStringGetCharacterAtIndex (
   CFStringRef theString,
   CFIndex idx
);

/*CFStringCreateWithPascalString

Creates an immutable CFString object from a Pascal string.

CFStringRef CFStringCreateWithPascalString (
   CFAllocatorRef alloc,
   ConstStr255Param pStr,
   CFStringEncoding encoding
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
pStr

The Pascal string to be used to create the string. 
encoding

The encoding of the characters in the Pascal string. 
Return Value

An immutable string containing pStr, or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Discussion

This function creates an immutable CFString objects from the character contents of a Pascal string (after stripping off the initial length byte). 
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateWithPascalString (
   CFAllocatorRef alloc,
   ConstStr255Param pStr,
   CFStringEncoding encoding
);

/*CFStringGetTypeID

Returns the type identifier for the CFString opaque type.

CFTypeID CFStringGetTypeID ();
Return Value

The type identifier for the CFString opaque type. 
Discussion

CFMutableString objects have the same type identifier as CFString objects.
Availability
Available in Mac OS X v10.0 and later.*/
CFTypeID CFStringGetTypeID ();

/*CFStringGetCharacters

Copies a range of the Unicode characters from a string to a user-provided buffer.

void CFStringGetCharacters (
   CFStringRef theString,
   CFRange range,
   UniChar *buffer
);
Parameters
theString

The string from which the characters are to be obtained. 
range

The range of characters to copy. The specified range must not exceed the length of the string.
buffer

The UniChar buffer of length range.length that you have allocated on the stack or heap. On return, the buffer contains the requested Unicode characters. 
Discussion

Use this function to obtain some or all of the Unicode characters represented by a CFString object. If this operation involves a large number of characters, the function call can be expensive in terms of memory. Instead you might want to consider using the in-line buffer functions CFStringInitInlineBuffer and CFStringGetCharacterFromInlineBuffer to extract the characters incrementally. 
Availability
Available in Mac OS X v10.0 and later.*/
void CFStringGetCharacters (
   CFStringRef theString,
   CFRange range,
   UniChar *buffer
);

/*CFStringReplace

Replaces part of the character contents of a CFMutableString object with another string.

void CFStringReplace (
   CFMutableStringRef theString,
   CFRange range,
   CFStringRef replacement
);
Parameters
theString

The string to modify. The characters are adjusted left or right (depending on the length of the substituted string) and the character buffer of the object is resized accordingly. If this value is not a CFMutableString object, an assertion is raised. 
range

The range of characters in theString to replace. 
replacement

The replacement string to put into theString. 
Discussion

Although you can use this function to replace all characters in a CFMutableString object (by specifying a range of (0, CFStringGetLength(theString)) ), it is more convenient to use the CFStringReplaceAll function for that purpose. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringReplace (
   CFMutableStringRef theString,
   CFRange range,
   CFStringRef replacement
);

/*CFStringDelete

Deletes a range of characters in a string.

void CFStringDelete (
   CFMutableStringRef theString,
   CFRange range
);
Parameters
theString

A string from which characters are to be deleted.
range

The range of characters in theString to delete. 
Discussion

The characters after the deleted range are adjusted to “fill in” the gap. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFStringDelete (
   CFMutableStringRef theString,
   CFRange range
);

/*CFStringCreateCopy

Creates an immutable copy of a string.

CFStringRef CFStringCreateCopy (
   CFAllocatorRef alloc,
   CFStringRef theString
);
Parameters
alloc

The allocator to use to allocate memory for the new string. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
theString

The string to copy. 
Return Value

An immutable string whose contents are identical to theString. Returns NULL if there was a problem copying the object. Ownership follows the Create Rule.
Discussion

The resulting object has the same Unicode contents as the original object, but it is always immutable. It might also have different storage characteristics, and hence might reply differently to functions such as CFStringGetCStringPtr. Also, if the specified allocator and the allocator of the original object are the same, and the string is already immutable, this function may simply increment the retention count without making a true copy. However, the resulting object is a true immutable copy, except the operation was a lot more efficient.

You should use this function in situations where a string is or could be mutable, and you need to take a snapshot of its current value. For example, you might decide to pass a copy of a string to a function that stores its current value in a list for later use.
Availability
Available in Mac OS X v10.0 and later.*/
CFStringRef CFStringCreateCopy (
   CFAllocatorRef alloc,
   CFStringRef theString
);

/*CFStringFindWithOptions

Searches for a substring within a range of the characters represented by a string and, if the substring is found, returns its range within the object's characters.

Boolean CFStringFindWithOptions (
   CFStringRef theString,
   CFStringRef stringToFind,
   CFRange rangeToSearch,
   CFOptionFlags searchOptions,
   CFRange *result
);
Parameters
theString

The string in which to to search for stringToFind. 
stringToFind

The substring to search for in theString.
rangeToSearch

A range of the characters to search in theString. The specified range must not exceed the length of the string.
searchOptions

The option flags to control the search behavior. The supported options are kCFCompareBackwards, kCFCompareAnchored, kCFCompareCaseInsensitive, and kCFCompareNonliteral (available in Mac OS X v10.0 and later). If other option flags are specified, the behavior is undefined. 
result

On return, if the function result is true, contains the starting location and length of the found substring. You may pass NULL if you only want to know if the substring exists in the larger string.
Return Value

true if the substring was found, false otherwise. 
Discussion

This function allows you to search only part of the characters of a string for a substring. It returns the found range indirectly, in the final result parameter. If you want to know if the entire range of characters represented by a string contains a particular substring, you can use the convenience function CFStringFind. Both of these functions return upon finding the first occurrence of the substring, so if you want to find out about multiple occurrences, call the CFStringCreateArrayWithFindResults function.

Depending on the comparison-option flags specified, the length of the resulting range might be different than the length of the search string. 
Availability
Available in Mac OS X v10.0 and later.*/
Boolean CFStringFindWithOptions (
   CFStringRef theString,
   CFStringRef stringToFind,
   CFRange rangeToSearch,
   CFOptionFlags searchOptions,
   CFRange *result
);

#ifdef __cplusplus
}
#endif

#endif