#ifndef CFURL_H
#define CFURL_H

#include <CoreFoundation/CFBase.h>
#include <CoreFoundation/CFData.h>
#include <CoreFoundation/CFString.h>

#ifndef  FILES_H
#include <CarbonCore/Files.h> //hys
#endif

#if defined(__cplusplus)
extern "C" {
#endif

/*Path Style

Options you can use to determine how CFURL functions parse a file system path name. 

enum CFURLPathStyle {
   kCFURLPOSIXPathStyle = 0,
   kCFURLHFSPathStyle = 1,
   kCFURLWindowsPathStyle = 2
};
typedef enum CFURLPathStyle CFURLPathStyle;
Constants
kCFURLPOSIXPathStyle

Indicates a POSIX style path name. Components are slash delimited. A leading slash indicates an absolute path; a trailing slash is not significant.

Available in Mac OS X v10.0 and later.
kCFURLHFSPathStyle

Indicates a HFS style path name. Components are colon delimited. A leading colon indicates a relative path, otherwise the first path component denotes the volume.

Available in Mac OS X v10.0 and later.
kCFURLWindowsPathStyle

Indicates a Windows style path name.

Available in Mac OS X v10.0 and later.*/
enum CFURLPathStyle {
   kCFURLPOSIXPathStyle = 0,
   kCFURLHFSPathStyle = 1,
   kCFURLWindowsPathStyle = 2
};
typedef enum CFURLPathStyle CFURLPathStyle;

/*CFURLRef

A reference to a CFURL object.

typedef const struct __CFURL *CFURLRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef const struct __CFURL *CFURLRef;


/*CFURLCopyPath

Returns the path portion of a given URL.

CFStringRef CFURLCopyPath (
   CFURLRef anURL
);
Parameters
anURL

The CFURL object to examine. 
Return Value

The path of anURL, or NULL if the URL cannot be decomposed (doesn't conform to RFC 1808). Ownership follows the Create Rule.
Discussion

This function does not resolve the URL against its base and replaces all percent escape sequences. This function's return value includes any leading slash (giving the path the normal POSIX appearance), if present. If this behavior is not appropriate, use CFURLCopyStrictPath whose return value omits any leading slash. You may also want to use the function CFURLCopyFileSystemPath, which returns the URL's path as a file system path for the given path style. If the path is to be passed to file system calls, you may also want to use the function CFURLGetFileSystemRepresentation, which returns a C string.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFStringRef CFURLCopyPath (
   CFURLRef anURL
);

/*CFURLCreateCopyAppendingPathComponent

Creates a copy of a given URL and appends a path component.

CFURLRef CFURLCreateCopyAppendingPathComponent (
   CFAllocatorRef allocator,
   CFURLRef url,
   CFStringRef pathComponent,
   Boolean isDirectory
);
Parameters
allocator

The allocator to use to allocate memory for the new CFURL object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
url

The CFURL object to which to append a path component. 
pathComponent

The path component to append to url. 
isDirectory

A Boolean value that specifies whether the string is treated as a directory path when resolving against relative path components. Pass true if the new component indicates a directory, false otherwise. 
Return Value

A copy of url appended with pathComponent. Ownership follows the Create Rule.
Discussion

The isDirectory argument specifies whether or not the new path component points to a file or a to directory. Note that the URL syntax for a directory and for a file at otherwise the same location are slightly different—directory URLs must end in “/”. If you have the URL http://www.apple.com/foo/ and you append the path component bar, then if isDirectory is YES then the resulting URL is http://www.apple.com/foo/bar/, whereas if isDirectory is NO then the resulting URL is http://www.apple.com/foo/bar. This difference is particularly important if you resolve another URL against this new URL. file.html relative to http://www.apple.com/foo/bar is http://www.apple.com/foo/file.html, whereas file.html relative to http://www.apple.com/foo/bar/ is http://www.apple.com/foo/bar/file.html.
Availability
Available in CarbonLib v1.1 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFURLRef CFURLCreateCopyAppendingPathComponent (
   CFAllocatorRef allocator,
   CFURLRef url,
   CFStringRef pathComponent,
   Boolean isDirectory
);

/*CFURLGetFSRef

Converts a given URL to a file or directory object.

Boolean CFURLGetFSRef (
   CFURLRef url,
   FSRef *fsRef
);
Parameters
url

The CFURL object to convert to a file or directory object.
fsRef

Upon return, contains the file or directory object representing url.
Return Value

true if the conversion was successful, otherwise false.
Special Considerations

The function cannot create an FSRef object if the path specified by url contains an alias. The function can, however, traverse symbolic links.
Availability
Available in CarbonLib v1.1 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
Boolean CFURLGetFSRef (
   CFURLRef url,
   FSRef *fsRef
);

/*CFURLCreateFromFSRef

Creates a URL from a given directory or file.

CFURLRef CFURLCreateFromFSRef (
   CFAllocatorRef allocator,
   const FSRef *fsRef
);
Parameters
allocator

The allocator to use to allocate memory for the new CFURL object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
fsRef

The file or directory representing the URL. 
Return Value

A new CFURL object. Ownership follows the Create Rule.
Availability
Available in CarbonLib v1.1 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFURLRef CFURLCreateFromFSRef (
   CFAllocatorRef allocator,
   const FSRef *fsRef
);

/*CFURLCreateFromFileSystemRepresentation

Creates a new CFURL object for a file system entity using the native representation.

CFURLRef CFURLCreateFromFileSystemRepresentation (
   CFAllocatorRef allocator,
   const UInt8 *buffer,
   CFIndex bufLen,
   Boolean isDirectory
);
Parameters
allocator

The allocator to use to allocate memory for the new CFURL object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
buffer

The character bytes to convert into a CFURL object. This should be the path as you would use in POSIX function calls.
bufLen

The number of bytes in the buffer. 
isDirectory

A Boolean value that specifies whether the string is treated as a directory path when resolving against relative path components—true if the pathname indicates a directory, false otherwise. 
Return Value

A new CFURL object. Ownership follows the Create Rule.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFURLRef CFURLCreateFromFileSystemRepresentation (
   CFAllocatorRef allocator,
   const UInt8 *buffer,
   CFIndex bufLen,
   Boolean isDirectory
);

/*CFURLGetFileSystemRepresentation

Fills a buffer with the file system's native string representation of a given URL's path.

Boolean CFURLGetFileSystemRepresentation (
   CFURLRef url,
   Boolean resolveAgainstBase,
   UInt8 *buffer,
   CFIndex maxBufLen
);
Parameters
url

The CFURL object whose native file system representation you want to obtain. 
resolveAgainstBase

Pass true to return an absolute path name. 
buffer

A pointer to a character buffer. On return, the buffer holds the native file system's representation of url. The buffer is null-terminated. This parameter must be at least maxBufLen in size for the file system in question to avoid failures for insufficiently large buffers.
maxBufLen

The maximum number of characters that can be written to buffer.
Return Value

true if successful, false if an error occurred. 
Discussion

No more than maxBufLen bytes are written to buffer. If url requires more than maxBufLen bytes to represent itself, including the terminating null byte, this function returns false. To avoid this possible failure, you should pass a buffer with size of at least the maximum path length for the file system in question.
Availability
Available in CarbonLib v1.1 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
Boolean CFURLGetFileSystemRepresentation (
   CFURLRef url,
   Boolean resolveAgainstBase,
   UInt8 *buffer,
   CFIndex maxBufLen
);

/*CFURLCreateWithString

Creates a CFURL object using a given CFString object.

CFURLRef CFURLCreateWithString (
   CFAllocatorRef allocator,
   CFStringRef URLString,
   CFURLRef baseURL
);
Parameters
allocator

The allocator to use to allocate memory for the new CFURL object. Pass NULL or kCFAllocatorDefault to use the current default allocator.
URLString

The CFString object containing the URL string. 
baseURL

The URL to which URLString is relative. Pass NULL if URLString contains an absolute URL or if you want to create a relative URL. If URLString contains an absolute URL, baseURL is ignored. 
Return Value

A new CFURL object. Ownership follows the Create Rule.
Discussion

Any escape sequences in URLString will be interpreted using UTF-8. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFURLRef CFURLCreateWithString (
   CFAllocatorRef allocator,
   CFStringRef URLString,
   CFURLRef baseURL
);

/*CFURLGetTypeID

Returns the type identifier for the CFURL opaque type.

CFTypeID CFURLGetTypeID ();
Return Value

The type identifier for the CFURL opaque type.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFTypeID CFURLGetTypeID ();

/*CFURLGetString

Returns the URL as a CFString object.

CFStringRef CFURLGetString (
   CFURLRef anURL
);
Parameters
anURL

The CFURL object to convert into a CFString object. 
Return Value

A string representation of anURL. Ownership follows the Get Rule.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFStringRef CFURLGetString (
   CFURLRef anURL
);

/*CFURLCreateWithFileSystemPath

Creates a CFURL object using a local file system path string.

CFURLRef CFURLCreateWithFileSystemPath (
   CFAllocatorRef allocator,
   CFStringRef filePath,
   CFURLPathStyle pathStyle,
   Boolean isDirectory
);
Parameters
allocator

The allocator to use to allocate memory for the new CFURL object. Pass NULL or kCFAllocatorDefault to use the current default allocator. 
filePath

The path string to convert to a CFURL object. 
pathStyle

The operating system path style used in filePath. See Path Style for a list of possible values. 
isDirectory

A Boolean value that specifies whether filePath is treated as a directory path when resolving against relative path components. Pass true if the pathname indicates a directory, false otherwise.
Return Value

A new CFURL object. Ownership follows the Create Rule.
Discussion

If filePath is not absolute, the resulting URL will be considered relative to the current working directory (evaluated when this function is being invoked). 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFURLRef CFURLCreateWithFileSystemPath (
   CFAllocatorRef allocator,
   CFStringRef filePath,
   CFURLPathStyle pathStyle,
   Boolean isDirectory
);

/*CFURLHasDirectoryPath

Determines if a given URL's path represents a directory.

Boolean CFURLHasDirectoryPath (
   CFURLRef anURL
);
Parameters
anURL

The CFURL object to examine. 
Return Value

true if anURL represents a directory, false otherwise. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
Boolean CFURLHasDirectoryPath (
   CFURLRef anURL
);

/*CFURLCopyLastPathComponent

Returns the last path component of a given URL.

CFStringRef CFURLCopyLastPathComponent (
   CFURLRef url
);
Parameters
url

The CFURL object to examine. 
Return Value

The last path component of url. Ownership follows the Create Rule. 
Discussion

Note that if there is no last path component, this function returns an empty string. In the code sample shown in Listing 1, lastPathComponent is an empty string. 

Listing 1  Code sample illustrating CFURLCopyLastPathComponentCFStringRef urlString = CFSTR("http://www.apple.com");
CFURLRef url = CFURLCreateWithString(NULL, urlString, NULL);
CFStringRef lastPathComponent = CFURLCopyLastPathComponent (url);


If urlString were created with CFSTR("http://www.apple.com/"), then lastPathComponent would be a CFString object containing the character “/“.

See also CFURLCopyPathExtension.
Availability
Available in CarbonLib v1.1 and later.
Available in Mac OS X v10.0 and later.
Declared In
CFURL.h*/
CFStringRef CFURLCopyLastPathComponent (
   CFURLRef url
);

#ifdef __cplusplus
}
#endif
#endif