#ifndef CFARRAY_H
#define CFARRAY_H

#include <CoreFoundation/CFBase.h>
#include <ETC/CoreFoundation/CFArray.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*CFArrayEqualCallBack

Prototype of a callback function used to determine if two values in an array are equal.

typedef Boolean (*CFArrayEqualCallBack) (
   const void *value1,
   const void *value2
);

If you name your function MyCallBack, you would declare it like this:

Boolean MyCallBack (
   const void *value1,
   const void *value2
);
Parameters
value1

A value in an array to be compared with value2 for equality.
value2

A value in an array to be compared with value1 for equality.
Return Value

true if value1 and value2 are equal, false otherwise.
Discussion

This callback is passed to CFArrayCreate in a CFArrayCallBacks structure.*/
typedef Boolean (*CFArrayEqualCallBack) (
   const void *value1,
   const void *value2
);

/*CFArrayCopyDescriptionCallBack

Prototype of a callback function used to get a description of a value in an array.

typedef CFStringRef (*CFArrayCopyDescriptionCallBack) (
   const void *value
);

If you name your function MyCallBack, you would declare it like this:

CFStringRef MyCallBack (
   const void *value
);
Parameters
value

The value to be described.
Return Value

A textual description of value. The caller is responsible for releasing this object.
Discussion

This callback is passed to CFArrayCreate in a CFArrayCallBacks structure. This callback is used by the CFCopyDescription function.*/
typedef CFStringRef (*CFArrayCopyDescriptionCallBack) (
   const void *value
);

/*CFArrayReleaseCallBack

Prototype of a callback function used to release a value before it’s removed from an array.

typedef void (*CFArrayReleaseCallBack) (
   CFAllocatorRef allocator,
   const void *value
);

If you name your function MyCallBack, you would declare it like this:

void MyCallBack (
   CFAllocatorRef allocator,
   const void *value
);
Parameters
allocator

The array’s allocator.
value

The value being removed from an array.
Discussion

This callback is passed to CFArrayCreate in a CFArrayCallBacks structure.*/
typedef void (*CFArrayReleaseCallBack) (
   CFAllocatorRef allocator,
   const void *value
);

/*CFArrayRetainCallBack

Prototype of a callback function used to retain a value being added to an array. 

typedef const void *(*CFArrayRetainCallBack) (
   CFAllocatorRef allocator,
   const void *value
);

If you name your function MyCallBack, you would declare it like this:

const void *MyCallBack (
   CFAllocatorRef allocator,
   const void *value
);
Parameters
allocator

The array’s allocator.
value

The value being added to an array.
Return Value

The value to store in an array, which is usually the value parameter passed to this callback, but may be a different value if a different value should be stored in an array.
Discussion

This callback is passed to CFArrayCreate in a CFArrayCallBacks structure.*/
typedef const void *(*CFArrayRetainCallBack) (
   CFAllocatorRef allocator,
   const void *value
);


/*CFArrayCallBacks

Structure containing the callbacks of a CFArray.

struct CFArrayCallBacks {
   CFIndex version;
   CFArrayRetainCallBack retain;
   CFArrayReleaseCallBack release;
   CFArrayCopyDescriptionCallBack copyDescription;
   CFArrayEqualCallBack equal;
};
typedef struct CFArrayCallBacks CFArrayCallBacks;
Fields
version

The version number of this structure. If not one of the defined version numbers for this opaque type, the behavior is undefined. The current version of this structure is 0.
retain

The callback used to retain each value as they are added to the collection. If NULL, values are not retained. See CFArrayRetainCallBack for a description of this callback.
release

The callback used to release values as they are removed from the collection. If NULL, values are not released. See CFArrayReleaseCallBack for a description of this callback.
copyDescription

The callback used to create a descriptive string representation of each value in the collection. If NULL, the collection will create a simple description of each value. See CFArrayCopyDescriptionCallBack for a description of this callback. 
equal

The callback used to compare values in the array for equality for some operations. If NULL, the collection will use pointer equality to compare values in the collection. See CFArrayEqualCallBack for a description of this callback.*/
struct CFArrayCallBacks {
   CFIndex version;
   CFArrayRetainCallBack retain;
   CFArrayReleaseCallBack release;
   CFArrayCopyDescriptionCallBack copyDescription;
   CFArrayEqualCallBack equal;
};
typedef struct CFArrayCallBacks CFArrayCallBacks;

/*Predefined Callback Structures

CFArray provides some predefined callbacks for your convenience.

const CFArrayCallBacks kCFTypeArrayCallBacks;
Constants
kCFTypeArrayCallBacks

Predefined CFArrayCallBacks structure containing a set of callbacks appropriate for use when the values in a CFArray are all CFType-derived objects. The retain callback is CFRetain, the release callback is CFRelease, the copy callback is CFCopyDescription, and the equal callback is CFEqual. Therefore, if you use this constant when creating the collection, items are automatically retained when added to the collection, and released when removed from the collection.

Available in Mac OS X v10.0 and later.*/
const CFArrayCallBacks kCFTypeArrayCallBacks;

/*CFArrayRef

A reference to an immutable array object.

typedef const struct __CFArray *CFArrayRef;
Availability
Available in Mac OS X v10.0 and later.*/


/*CFMutableArrayRef

A reference to a mutable array object.

typedef struct __CFArray *CFMutableArrayRef;
Availability
Available in Mac OS X v10.0 and later.*/
typedef struct __CFArray *CFMutableArrayRef;

/*CFArrayGetValueAtIndex

Retrieves a value at a given index.

const void *CFArrayGetValueAtIndex (
   CFArrayRef theArray,
   CFIndex idx
);
Parameters
theArray

The array to examine. 
idx

The index of the value to retrieve. If the index is outside the index space of theArray (0 to N-1 inclusive (where N is the count of theArray), the behavior is undefined. 
Return Value

The value at the idx index in theArray. If the return value is a Core Foundation Object, ownership follows the Get Rule.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
const void *CFArrayGetValueAtIndex (
   CFArrayRef theArray,
   CFIndex idx
);

/*CFArrayAppendValue

Adds a value to an array giving it the new largest index.

void CFArrayAppendValue (
   CFMutableArrayRef theArray,
   const void *value
);
Parameters
theArray

The array to which value is to be added. If theArray is a limited-capacity array and it is full before this operation, the behavior is undefined. 
value

A CFType object or a pointer value to add to theArray. 
Discussion

The value parameter is retained by theArray using the retain callback provided when theArray was created. If value is not of the type expected by the retain callback, the behavior is undefined. The value parameter is assigned to the index one larger than the previous largest index and the count of theArray is increased by one.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
void CFArrayAppendValue (
   CFMutableArrayRef theArray,
   const void *value
);

/*CFArrayGetCount

Returns the number of values currently in an array.

CFIndex CFArrayGetCount (
   CFArrayRef theArray
);
Parameters
theArray

The array to examine. 
Return Value

The number of values in theArray. 
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later*/
CFIndex CFArrayGetCount (
   CFArrayRef theArray
);

/*CFArrayCreateMutable

Creates a new empty mutable array.

CFMutableArrayRef CFArrayCreateMutable (
   CFAllocatorRef allocator,
   CFIndex capacity,
   const CFArrayCallBacks *callBacks
);
Parameters
allocator

The allocator to use to allocate memory for the new array and its storage for values. Pass NULL or kCFAllocatorDefault to use the current default allocator.
capacity

The maximum number of values that can be contained by the new array. The array starts empty and can grow to this number of values (and it can have less). If this parameter is 0, the array’s maximum capacity is not limited. The value must not be negative. 
callBacks

A pointer to a CFArrayCallBacks structure initialized with the callbacks for the array to use on each value in the array. A copy of the contents of the callbacks structure is made, so that a pointer to a structure on the stack can be passed in or can be reused for multiple array creations. This parameter may be NULL, which is treated as if a valid structure of version 0 with all fields NULL had been passed in. 

If any of the fields are not valid pointers to functions of the correct type, or this parameter is not a valid pointer to a CFArrayCallBacks structure, the behavior is undefined. If any value put into the array is not one understood by one of the callback functions, the behavior when that callback function is used is undefined. If the array contains CFType objects only, then pass kCFTypeArrayCallBacks as this parameter to use the default callback functions.
Return Value

A new mutable array, or NULL if there was a problem creating the object. Ownership follows the Create Rule.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
CFMutableArrayRef CFArrayCreateMutable (
   CFAllocatorRef allocator,
   CFIndex capacity,
   const CFArrayCallBacks *callBacks
);

/*CFArrayGetTypeID

Returns the type identifier for the CFArray opaque type.

CFTypeID CFArrayGetTypeID();
Return Value

The type identifier for the CFArray opaque type.
Discussion

CFMutableArray objects have the same type identifier as CFArray objects.
Availability
Available in CarbonLib v1.0 and later.
Available in Mac OS X v10.0 and later.*/
CFTypeID CFArrayGetTypeID();

#ifdef __cplusplus
}
#endif
#endif