{
   This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
}
unit Udialog;

interface

uses Messages, Windows, SysUtils, Classes, Controls, StdCtrls, Graphics,
  ExtCtrls, Buttons, Dialogs, GR32_Image, GR32, FreeBitmap, FreeImage;

type
  TSilentPaintPanel = class(TPanel)
  protected
    procedure WMPaint(var Msg: TWMPaint); message WM_PAINT;
  end;

  TmyOpenPictureDialog = class(TOpenDialog)
  private
    FPicturePanel: TPanel;
    FPictureLabel: TLabel;
    FPreviewButton: TSpeedButton;
    FPaintPanel: TPanel;
    FImageCtrl: TImage32;
    FSavedFilename: string;
    FExtraCheckbox : Tcheckbox;
    FBitmap: TFreeWinBitmap;

    function  IsFilterStored: Boolean;
    procedure PreviewKeyPress(Sender: TObject; var Key: Char);
    procedure FExtraCheckboxOnClick(sender:tobject);
  protected
    procedure PreviewClick(Sender: TObject); virtual;
    procedure DoClose; override;
    procedure DoSelectionChange; override;
    function loadfile(s:string):boolean;
    procedure DoShow; override;
    property ImageCtrl: TImage32 read FImageCtrl;
    property PictureLabel: TLabel read FPictureLabel;
  published
    property Filter stored IsFilterStored;
    property ExtraCheckbox: Tcheckbox read FExtraCheckbox write FExtraCheckbox;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean; override;
  end;

  TmySavePictureDialog = class(TmyOpenPictureDialog)
  public
    function Execute: Boolean; override;
  end;

implementation
uses Consts, math, CommDlg, forms;
{$R ExtDlgs.res}

procedure TSilentPaintPanel.WMPaint(var Msg: TWMPaint);
begin
  try
    inherited;
  except
    Caption := SInvalidImage;
  end;
end;

constructor TmyOpenPictureDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Filter := GraphicFilter(TGraphic);
  FPicturePanel := TPanel.Create(Self);
  with FPicturePanel do
  begin
    Name := 'PicturePanel';
    Caption := '';
    SetBounds(204, 5, 169, 200);
    BevelOuter := bvNone;
    BorderWidth := 6;
    TabOrder := 1;
    FPictureLabel := TLabel.Create(Self);
    with FPictureLabel do
    begin
      Name := 'PictureLabel';
      Caption := '';
      SetBounds(6, 6, 157, 23);
      Align := alTop;
      AutoSize := False;
      Parent := FPicturePanel;
    end;
    FPreviewButton := TSpeedButton.Create(Self);
    with FPreviewButton do
    begin
      Name := 'PreviewButton';
      SetBounds(77, 1, 23, 22);
      Enabled := False;
      Glyph.LoadFromResourceName(HInstance, 'PREVIEWGLYPH');
      Hint := SPreviewLabel;
      ParentShowHint := False;
      ShowHint := True;
      OnClick := PreviewClick;
      Parent := FPicturePanel;
    end;
    FPaintPanel := TSilentPaintPanel.Create(Self);
    with FPaintPanel do
    begin
      Name := 'PaintPanel';
      Caption := '';
      SetBounds(6, 29, 157, 145);
      Align := alClient;
      BevelInner := bvRaised;
      BevelOuter := bvLowered;
      TabOrder := 0;
      FImageCtrl := TImage32.Create(Self);
      Parent := FPicturePanel;
      with FImageCtrl do
      begin
        Name := 'PaintBox';
        Align := alClient;
        ScaleMode:=smResize;
        BitmapAlign:=baCenter;
        OnDblClick := PreviewClick;
        Parent := FPaintPanel;
      end;
    end;
  end;
  FExtraCheckbox := Tcheckbox.Create(Self);
  FExtraCheckbox.Caption:='̸ ';
  FExtraCheckbox.Align:=albottom;
  FExtraCheckbox.Parent:=FPaintPanel;
  FExtraCheckbox.OnClick:=FExtraCheckboxOnClick;

  FBitmap:=TFreeWinBitmap.Create();
end;

destructor TmyOpenPictureDialog.Destroy;
begin
  FBitmap.Free;
  inherited;
end;

procedure TmyOpenPictureDialog.FExtraCheckboxOnClick(sender:tobject);
begin
  if FExtraCheckbox.Checked then begin
    FSavedFilename:='';
    DoSelectionChange;
  end else begin
    FPictureLabel.Caption := SPictureLabel;
    FPreviewButton.Enabled := False;
    FImageCtrl.Bitmap.Delete;
    FImageCtrl.Enabled:=false;
    FPaintPanel.Caption := srNone;
  end;
end;

function TmyOpenPictureDialog.loadfile(s:string):boolean;
var
  w,h:integer;
  FreeMemoryIO1:TFreeMemoryIO;
  stream:tmemorystream;
  pdata:PByte;
  SizeInBytes: DWORD;
begin
  result:=false;
  if FBitmap.Load(s)=false then
    exit;
  w:=FBitmap.GetWidth;
  h:=FBitmap.GetHeight;
  FreeMemoryIO1:=TFreeMemoryIO.Create();
  try
    FBitmap.SaveToMemory(FIF_BMP,FreeMemoryIO1);
    FreeMemoryIO1.Acquire(pdata,SizeInBytes);
    stream:=tmemorystream.Create;
    try
      stream.Write(pdata^,SizeInBytes);
      stream.Position:=0;
      FImageCtrl.Bitmap.DrawMode:=dmblend;
      FImageCtrl.Bitmap.LoadFromStream(stream);
    finally
      stream.Free;
    end;
    result:=true;
  finally
    FreeMemoryIO1.Free;
  end;
end;

procedure TmyOpenPictureDialog.DoSelectionChange;
var
  FullName: string;
  ValidPicture: Boolean;

  function ValidFile(const FileName: string): Boolean;
  begin
    Result := GetFileAttributes(PChar(FileName)) <> $FFFFFFFF;
  end;

begin
  if FExtraCheckbox.Checked=false then exit;
  FullName := FileName;
  if FullName <> FSavedFilename then
  begin
    FSavedFilename := FullName;
    ValidPicture := FileExists(FullName) and ValidFile(FullName);
    if ValidPicture then
    try
      ValidPicture:=loadfile(FullName);
      if ValidPicture then begin
        FImageCtrl.Enabled:=true;
        FPictureLabel.Caption := Format(SPictureDesc,
          [FImageCtrl.Bitmap.Width, FImageCtrl.Bitmap.Height]);
        FPreviewButton.Enabled := True;
        FPaintPanel.Caption := '';
      end;
    except
      ValidPicture := False;
    end;
    if not ValidPicture then
    begin
      FPictureLabel.Caption := SPictureLabel;
      FPreviewButton.Enabled := False;
      FImageCtrl.Bitmap.Delete;
      FImageCtrl.Enabled:=false;
      FPaintPanel.Caption := srNone;
    end;
  end;
  inherited DoSelectionChange;
end;

procedure TmyOpenPictureDialog.DoClose;
begin
  inherited DoClose;
  { Hide any hint windows left behind }
  Application.HideHint;
end;

procedure TmyOpenPictureDialog.DoShow;
var
  PreviewRect, StaticRect: TRect;
begin
  { Set preview area to entire dialog }
  GetClientRect(Handle, PreviewRect);
  StaticRect := GetStaticRect;
  { Move preview area to right of static area }
  PreviewRect.Left := StaticRect.Left + (StaticRect.Right - StaticRect.Left);
  Inc(PreviewRect.Top, 4);
  FPicturePanel.BoundsRect := PreviewRect;
  FPreviewButton.Left := FPaintPanel.BoundsRect.Right - FPreviewButton.Width - 2;
  FImageCtrl.Bitmap.Delete;
  FSavedFilename := '';
  FPaintPanel.Caption := srNone;
  FPicturePanel.ParentWindow := Handle;
  inherited DoShow;
end;

function TmyOpenPictureDialog.Execute;
begin
  if NewStyleControls and not (ofOldStyleDialog in Options) then
    Template := 'DLGTEMPLATE' else
    Template := nil;
  Result := inherited Execute;
end;

procedure TmyOpenPictureDialog.PreviewClick(Sender: TObject);
var
  PreviewForm: TForm;
  Panel: TPanel;
  w,h:integer;
begin
  PreviewForm := TForm.Create(Self);
  with PreviewForm do
  try
    Name := 'PreviewForm';
    Visible := False;
    Caption := SPreviewLabel;
    BorderStyle := bsSizeToolWin;
    KeyPreview := True;
    Position := poScreenCenter;
    OnKeyPress := PreviewKeyPress;
    Panel := TPanel.Create(PreviewForm);
    with Panel do
    begin
      Name := 'Panel';
      Caption := '';
      Align := alClient;
      BevelOuter := bvNone;
      BorderStyle := bsSingle;
      BorderWidth := 3;
      Color := clWindow;
      Parent := PreviewForm;
      DoubleBuffered := True;
      with TImgView32.Create(PreviewForm) do
      begin
        Name := 'Image';
        Align := alClient;
        Bitmap.Assign(FImageCtrl.Bitmap);
        Parent := Panel;
      end;
    end;
    w:=FImageCtrl.Bitmap.Width+40;
    h:=FImageCtrl.Bitmap.Height+40;
    ClientWidth:=w;
    ClientHeight:=h;
    if Width>screen.Width then Width:=screen.Width;
    if Height>screen.Height then Height:=screen.Height;
    Left:=(screen.Width div 2)-(Width div 2);
    Top:=(screen.Height div 2)-(Height div 2);
    ShowModal;
  finally
    Free;
  end;
end;

procedure TmyOpenPictureDialog.PreviewKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #27 then TForm(Sender).Close;
end;

function TmyOpenPictureDialog.IsFilterStored: Boolean;
begin
  Result := not (Filter = GraphicFilter(TGraphic));
end;

function TmySavePictureDialog.Execute: Boolean;
begin
  if NewStyleControls and not (ofOldStyleDialog in Options) then
    Template := 'DLGTEMPLATE' else
    Template := nil;
  Result := DoExecute(@GetSaveFileName);
end;

end.
