/*
 Copyright (C) Information Equipment co.,LTD.
 All rights reserved.
 Code by JaeHyuk Cho <mailto:minzkn@infoeq.com>
*/

/*
 
  session buffer design
  ~~~~~~~~~~~~~~~~~~~~~

 IPv4 or IPv6                          IPv4 or IPv6
  +--------+      +--------------+      +--------+
  |        | <--- |   buffer[0]  | <--- |        |
  |redirect|      |              |      | listen |
  | socket |      | alloc_buffer |      | socket |
  |        |      |              |      |        |
  | connect| ---> |   buffer[1]  | ---> | accept |
  |        |      |              |      |        |
  |        |      |   decrpyt    |      |        |
  |        |      |   buffer[2]  |      |        |
  +--------+      +--------------+      +--------+

*/

/* optimize cache control macro */
#if defined(__GNUC__)
# if __GNUC__ < (3L) /* no expect macro */
#  define mzproxy_builtin_expect(m_expression,m_value)               m_expression
# else
#  define mzproxy_builtin_expect(m_expression,m_value)               __builtin_expect((long)(m_expression),(long)(m_value))
# endif
#else
# define mzproxy_builtin_expect(m_expression,m_value)                m_expression
#endif

#if !defined(__def_mzproxy_use_telnetd__)
# define __def_mzproxy_use_telnetd__ (1)
#endif

#if !defined(__def_mzproxy_use_ifaddrs__)
# define __def_mzproxy_use_ifaddrs__ (1)
#endif

#if !defined(__def_mzproxy_use_inet6__)
# define __def_mzproxy_use_inet6__ (1)
#endif

#if !defined(__def_mzproxy_use_inet_ntop__)
# define __def_mzproxy_use_inet_ntop__ (1)
#endif

#if !defined(__def_mzproxy_use_ief__)
# define __def_mzproxy_use_ief__ (1)
#endif

#if defined(WIN32) || defined(_WIN32)
# include <winsock2.h>
# include <ws2tcpip.h>
# include <windows.h>
# include <sys/types.h>
# include <io.h>
# include <direct.h> 
# include <stdio.h>
# include <stdlib.h>
# include <stdarg.h>
# include <memory.h>
# include <string.h>
# include <time.h>
# include <fcntl.h>
# include <errno.h>
# include <signal.h>
# define t_mzproxy_socket SOCKET
#else
# include <sys/types.h>
# include <sys/time.h>
# include <sys/ioctl.h>
# include <sys/socket.h>
# include <sys/sysinfo.h>
# if __def_mzproxy_use_telnetd__ != (0)
#  include <sys/wait.h>
# endif
# include <stdio.h>
# if __def_mzproxy_use_telnetd__ == (0)
#  include <stdlib.h>
# else
#  if !defined(__USE_GNU)
#   define __USE_GNU (1)
#  endif
#  if !defined(__USE_XOPEN)
#   define __USE_XOPEN (1)
#  endif
#  include <stdlib.h>
#  undef __USE_XOPEN
#  undef __USE_GNU
# endif
# include <string.h>
# include <fcntl.h>
# include <unistd.h>
# include <signal.h>
# include <errno.h>
# include <netdb.h>
# if __def_mzproxy_use_ifaddrs__ != (0)
#  include <ifaddrs.h>
# endif
# include <netinet/in.h>
# include <net/if.h>
# include <arpa/inet.h>
# if __def_mzproxy_use_telnetd__ != (0)
#  include <arpa/telnet.h>
# endif
# define t_mzproxy_socket int
#endif

#if __def_mzproxy_use_telnetd__ != (0)
# include <termios.h>
#endif

#if __def_mzproxy_use_ief__ != (0)
# include "mzseed.h"
#endif

#define __def_mzproxy_minimum_buffer_size__                          (256 << 10) /* recommanded 256bytes */
#define __def_mzproxy_default_buffer_size__                          __def_mzproxy_minimum_buffer_size__

/* debug options */
#define __def_mzproxy_rx_dump__                                      (0) /* 0=disable, 1=enable */
#define __def_mzproxy_tx_dump__                                      (1) /* 0=disable, 1=enable */

/* session flag */
#define def_mzproxy_session_flag_none                                (0)
#define def_mzproxy_session_flag_connecting                          (1 << 0)
#define def_mzproxy_session_flag_connected                           (1 << 1)
#define def_mzproxy_session_flag_cert_passed                         (1 << 2)
#define def_mzproxy_session_flag_cert_ack                            (1 << 3)
#define def_mzproxy_session_flag_nonblock                            (1 << 4)

#if __def_mzproxy_use_ief__ != (0)
# define def_mzproxy_session_flag_ief                                (1 << 16)
# define def_mzproxy_session_flag_iet                                (1 << 17)
#endif
     
/* ief options */
#if __def_mzproxy_use_ief__ != (0)
# define def_mzproxy_ief_magic_code                                  (0x00000000ul /* IEF-server */)
# define def_mzproxy_iet_magic_code                                  (0x00000000ul /* IEF-client */)
#endif

/* simple hexa dump function */
#if !defined(mzapi_dump_inline)
#include <stdio.h>
static void * (__mzapi_dump_inline_ex__)(size_t s_s, size_t s_d, void * s_da, size_t s_si)
{ size_t s_o, s_w, s_lo; unsigned char s_b[ 16 + 1 ];
 if(mzproxy_builtin_expect(s_da == NULL, 0))return(NULL); s_b[sizeof(s_b) - 1] = (unsigned char)'\0';
 for(s_o = (size_t)0;s_o < s_si;s_o += (size_t)16){
  s_w = ((s_si - s_o) <= ((size_t)16)) ? (s_si - s_o) : ((size_t)16);
  (void)fprintf(stdout, "%08lX", (unsigned long)(s_d + s_o));
  for(s_lo = (size_t)0;s_lo < s_w;s_lo++){
   if(s_lo == ((size_t)(16 >> 1)))(void)fputs(" | ", stdout);
   else (void)fputs(" ", stdout);
   s_b[s_lo] = *(((unsigned char *)s_da) + (s_s + s_o + s_lo));
   (void)fprintf(stdout, "%02X", (int)s_b[s_lo]);
   if((s_b[s_lo] & ((unsigned char)(1 << 7))) || (s_b[s_lo] < ((unsigned char)' ')) ||
      (s_b[s_lo] == ((unsigned char)0x7f)))s_b[s_lo] = (unsigned char)'.';}
  while(s_lo < ((size_t)16)){
   if(s_lo == ((size_t)(16 >> 1)))(void)fputs("     ", stdout);
   else (void)fputs("   ", stdout);
   s_b[s_lo] = (unsigned char)' '; s_lo++;}
  (void)fprintf(stdout, " [%s]\n", (char *)(&s_b[0]));}
 return(s_da);
}
static __inline void * (mzapi_dump_inline_ex)(size_t s_seek_offset, size_t s_display_offset, void * s_data, size_t s_size)
{ return(__mzapi_dump_inline_ex__(s_seek_offset, s_display_offset, s_data, s_size)); }
#define mzapi_dump_inline(m_data,m_size) mzapi_dump_inline_ex((size_t)0,(size_t)0,(void *)(m_data),(size_t)(m_size))
#define mzapi_dp() (void)fprintf(stdout, "DEBUG: %s:%d\n", __FUNCTION__, __LINE__)
#endif

/* proxy options */
struct ts_mzproxy_options {
    const char *application_name;
    int do_help;
    int verbose;
#if (!defined(_WIN32)) && (!defined(WIN32))
    int daemonize;
#endif
    int listen_port;
    long unsigned int connect_timeout;
    const char *connect_address;
    int connect_port;
    long unsigned int data_timeout;
#if __def_mzproxy_use_ifaddrs__ != (0)
    const char *interface;
#endif 
    size_t buffer_size;
    unsigned int limit_session_count;
#if __def_mzproxy_use_inet6__ != (0) 
    int use_inet6;
#endif 
    int use_xinetd;
    int use_pm;
#if __def_mzproxy_use_ief__ != (0)
    int use_ief;
    int ief_data_port;
#endif 
#if __def_mzproxy_use_telnetd__ != (0)
    int use_telnetd;
    const char *issue_file;
    const char *login_program;
#endif
    int dummy;
};

#if __def_mzproxy_use_ief__ != (0)
# define __def_mzproxy_buffer_count__ (3)
#else
# define __def_mzproxy_buffer_count__ (2)
#endif

/* proxy session */
struct ts_mzproxy_session {
    struct ts_mzproxy_session *prev, *next;
    long unsigned int connect_time_stamp, data_time_stamp;
    int flags; /* bit0=client_connect */
    t_mzproxy_socket accept_socket, client_socket;
    int connect_port;
    unsigned char *alloc_buffer;
    size_t size[ __def_mzproxy_buffer_count__ ];
    unsigned char *buffer[ __def_mzproxy_buffer_count__ ];
#if __def_mzproxy_use_ief__ != (0)
    /* ief info */
    __t_mzapi_byte__ padd_block[ __def_mzapi_seed_block_size__ ];
    __t_mzapi_dword__ round_key[ __def_mzapi_seed_round_keys__ ];
#endif 
#if __def_mzproxy_use_telnetd__ != (0)
    pid_t pid;
#endif
};

/* main control */
struct ts_mzproxy_control {
    int * volatile stop_service;
    struct ts_mzproxy_options *options;
#if __def_mzproxy_use_telnetd__ != (0)
    char *login_program_argv[2];
#endif
    t_mzproxy_socket socket;
    struct ts_mzproxy_session *session;
    unsigned int session_count;
    long unsigned int time_stamp;
    int max_fd;
    fd_set fd_set_rx, fd_set_tx;
    int events;
};

static void (mzproxy_signal)(int s_signal);
static void (mzproxy_closesocket)(t_mzproxy_socket s_socket);
static long unsigned int (mzproxy_time_stamp)(void);
static int (mzproxy_nonblock_socket)(t_mzproxy_socket s_socket, int s_enable);
static int (mzproxy_linger_on)(t_mzproxy_socket s_socket);
static struct ts_mzproxy_session * (mzproxy_free_session)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session, int s_nokill);
static struct ts_mzproxy_session * (mzproxy_new_session)(struct ts_mzproxy_control *s_control, t_mzproxy_socket s_accept_socket);
static void (mzproxy_del_session)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session);
static void (mzproxy_register_event)(struct ts_mzproxy_control *s_control);
static void (mzproxy_check_listen)(struct ts_mzproxy_control *s_control);
static int (mzproxy_to_server)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session);
static int (mzproxy_to_client)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session);
static int (mzproxy_from_client)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session);
static int (mzproxy_from_server)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session);
static int (mzproxy_do)(struct ts_mzproxy_control *s_control);
static int (mzproxy_main_process)(struct ts_mzproxy_options *s_options);

#if __def_mzproxy_use_telnetd__ != (0)
/* default IAC session start */
static const unsigned char g_mzproxy_default_iac[] = {
    IAC, DO, TELOPT_ECHO,
    IAC, DO, TELOPT_NAWS,
    IAC, DO, TELOPT_LFLOW, /* for winsize */
    IAC, WILL, TELOPT_ECHO,
    IAC, WILL, TELOPT_SGA
};
#endif

/* always 1 */
static const int gc_mzproxy_on = 1;

/* break signal */
static volatile int g_mzproxy_break = 0;

/* signal function */
static void (mzproxy_signal)(int s_signal)
{
    g_mzproxy_break = 1;

    /* reinstall signal */
    (void)signal(s_signal, mzproxy_signal);
}

/* close socket */
static void (mzproxy_closesocket)(t_mzproxy_socket s_socket)
{
#if defined(WIN32) || defined(_WIN32)
    closesocket(s_socket);
#else
# if defined(EINTR)
    while(mzproxy_builtin_expect((close(s_socket) == (-1)) && (errno == EINTR), 0));
# else
    (void)close(s_socket);
# endif
#endif
}

/* time stamp */
static long unsigned int (mzproxy_time_stamp)(void)
{
#if defined(WIN32) || defined(_WIN32)
    LARGE_INTEGER s_performance_frequency, s_performance_count;

    if((QueryPerformanceFrequency((LARGE_INTEGER *)(&s_performance_frequency)) == TRUE) &&
        (QueryPerformanceCounter((LARGE_INTEGER *)(&s_performance_count)) == TRUE)) {
        if(s_performance_frequency.QuadPart != ((LONGLONG)0)) {
            return((long unsigned int)(s_performance_count.QuadPart / s_performance_frequency.QuadPart));
        }
    }

    return((long unsigned int)(GetTickCount() / ((DWORD)1000)));
#else
    struct sysinfo s_sysinfo;

    if(mzproxy_builtin_expect(sysinfo((struct sysinfo *)(&s_sysinfo)) != 0, 0))return((long unsigned int)0);

    return((long unsigned int)s_sysinfo.uptime);
#endif 
}

/* non-block socket */
static int (mzproxy_nonblock_socket)(t_mzproxy_socket s_socket, int s_enable)
{
#if defined(WIN32) || defined(_WIN32)
    long unsigned int s_flags = (s_enable == 0) ? 0ul : 1ul;

    return((ioctlsocket(s_socket, FIONBIO, &s_flags) == 0) ? 1 : (-1));
#else
    int s_flags;

    s_flags = fcntl(s_socket, F_GETFL);
    if(mzproxy_builtin_expect(s_flags == (-1), 0))return(-1);

    return((fcntl((int)s_socket, F_SETFL, (s_enable == 0) ? (s_flags & (~(O_NONBLOCK))) : (s_flags | (O_NONBLOCK))) == 0) ? 1 : (-1));
#endif 
}
   
static int (mzproxy_linger_on)(t_mzproxy_socket s_socket)
{
    struct linger s_linger = {1, 0};

#if defined(_WIN32) || defined(WIN32)    
    if(mzproxy_builtin_expect(setsockopt((int)s_socket, SOL_SOCKET, SO_LINGER, (char *)(&s_linger), (int)sizeof(s_linger)) == (-1), 0))return((-1));
#else    
    if(mzproxy_builtin_expect(setsockopt((int)s_socket, SOL_SOCKET, SO_LINGER, (void *)(&s_linger), (socklen_t)sizeof(s_linger)) == (-1), 0))return((-1));
#endif

    return(1);
}

/* session management function */
static struct ts_mzproxy_session * (mzproxy_free_session)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session, int s_nokill)
{
    struct ts_mzproxy_session *s_prev_session;

    while(s_session != ((struct ts_mzproxy_session *)0)) {
        s_prev_session = s_session;
        s_session = s_session->next;

        if(s_prev_session->alloc_buffer != ((unsigned char *)0))free((void *)s_prev_session->alloc_buffer);

#if __def_mzproxy_use_telnetd__ != (0)  
        if((s_nokill == 0) && (s_prev_session->pid != ((pid_t)(-1))) && (s_prev_session->pid != ((pid_t)0))) {
            /* force kill */
            if(mzproxy_builtin_expect(kill(s_prev_session->pid, SIGKILL) == (-1), 0)) {
                if(s_control->options->verbose != 0)perror("kill");
            }

            /* wait for login program */
            if(mzproxy_builtin_expect(waitpid(s_prev_session->pid, (int *)0, 0) == ((pid_t)(-1)), 0)) {
                if(s_control->options->verbose != 0)perror("waitpid");
            }
        }
#endif  

        if(s_prev_session->accept_socket != ((t_mzproxy_socket)(-1)))mzproxy_closesocket(s_prev_session->accept_socket);

        if(s_prev_session->client_socket != ((t_mzproxy_socket)(-1))) {
#if __def_mzproxy_use_telnetd__ != (0)  
            if(s_control->options->use_telnetd != 0) {
# if defined(EINTR)                
                while(mzproxy_builtin_expect((close((int)s_prev_session->client_socket) == (-1)) && (errno == EINTR), 0));
# else
                (void)close((int)s_prev_session->client_socket);
# endif
            }
            else {
               mzproxy_closesocket(s_prev_session->client_socket);
            }
#else            
            mzproxy_closesocket(s_prev_session->client_socket);
#endif   
        }

        free((void *)s_prev_session);
    }

    return((struct ts_mzproxy_session *)0);
}

/* create session */
static struct ts_mzproxy_session * (mzproxy_new_session)(struct ts_mzproxy_control *s_control, t_mzproxy_socket s_accept_socket)
{
    struct ts_mzproxy_session *s_session;
    int s_index;

    s_session = (struct ts_mzproxy_session *)malloc((size_t)sizeof(struct ts_mzproxy_session));
    if(mzproxy_builtin_expect(s_session == ((struct ts_mzproxy_session *)0), 0)) {
        mzproxy_closesocket(s_accept_socket);

        return((struct ts_mzproxy_session *)0);
    }

#if defined(IP_TOS) && defined(IPTOS_LOWDELAY)
    do { /* adjust accept socket property: LOWDELAY */
       static const int s_tos = IPTOS_LOWDELAY;

       if(mzproxy_builtin_expect(setsockopt(s_accept_socket, IPPROTO_IP, IPTOS_LOWDELAY, (void *)(&s_tos), (socklen_t)sizeof(s_tos)) == (-1), 0)) {
          if(s_control->options->verbose != 0)perror("lowdelay tos socket");
       }
    }while(0);
#endif

#if defined(SO_OOBINLINE)
    /* adjust accept socket property: OOBINLINE */
    if(mzproxy_builtin_expect(setsockopt(s_accept_socket, SOL_SOCKET, SO_OOBINLINE, (void *)(&gc_mzproxy_on), (socklen_t)sizeof(gc_mzproxy_on)) == (-1), 0)) {
        if(s_control->options->verbose != 0)perror("oobinline socket");
    }
#endif   

    s_session->prev = (struct ts_mzproxy_session *)0;
    s_session->next = (struct ts_mzproxy_session *)0;
    s_session->connect_time_stamp = s_control->time_stamp;
    s_session->data_time_stamp = s_control->time_stamp;
    s_session->flags = def_mzproxy_session_flag_none;
    s_session->accept_socket = s_accept_socket;
    s_session->client_socket = (t_mzproxy_socket)(-1);
    s_session->connect_port = 0;

    if(s_control->options->use_pm == 0)s_session->connect_port = s_control->options->connect_port;

    /* allocate rx/tx buffer */
    s_session->alloc_buffer = (unsigned char *)malloc(s_control->options->buffer_size * __def_mzproxy_buffer_count__);
    if(mzproxy_builtin_expect(s_session->alloc_buffer == ((unsigned char *)0), 0)) {
        if(s_control->options->verbose != 0)(void)fputs("session buffer: not enough memory\n", stderr);

        return(mzproxy_free_session(s_control, s_session, 0));
    }

    for(s_index = 0;s_index < __def_mzproxy_buffer_count__;s_index++) {
        s_session->size[s_index] = (size_t)0;
        s_session->buffer[s_index] = (unsigned char *)(&s_session->alloc_buffer[s_control->options->buffer_size * s_index]);
    }

#if __def_mzproxy_use_telnetd__ != (0)
    if(s_control->options->use_telnetd != 0) { /* create pt handle */
        char *s_pts_name; 

        /* default iac code size to pt buffer */
        (void)memcpy((void *)(&s_session->buffer[1][0]), (void *)(&g_mzproxy_default_iac[0]), (size_t)sizeof(g_mzproxy_default_iac));
        
        s_session->size[1] += (size_t)sizeof(g_mzproxy_default_iac);
        
        /* open the pseudo-terminal master (PTM) */
        s_session->client_socket = (t_mzproxy_socket)getpt();
        if(mzproxy_builtin_expect(s_session->client_socket == ((t_mzproxy_socket)(-1)), 0)) {
            if(s_control->options->verbose != 0)perror("getpt");

            return(mzproxy_free_session(s_control, s_session, 0));
        }

        s_session->flags |= def_mzproxy_session_flag_connecting;

        /* grant access to the slave pseudo-terminal */
        if(mzproxy_builtin_expect(grantpt((int)s_session->client_socket) == (-1), 0)) {
         if(s_control->options->verbose != 0)perror("grantpt");

         return(mzproxy_free_session(s_control, s_session, 0));
        }

        /* unlock a pseudo-terminal master/slave pair */
        if(mzproxy_builtin_expect(unlockpt((int)s_session->client_socket) == (-1), 0)) {
            if(s_control->options->verbose != 0)perror("unlockpt");

            return(mzproxy_free_session(s_control, s_session, 0));
        }

        /* get pts name */
        s_pts_name = strdup(ptsname((int)s_session->client_socket));
        if(mzproxy_builtin_expect(s_pts_name == ((char *)0), 0)) {
            if(s_control->options->verbose != 0)(void)fputs("pts name: not enough memory", stderr);

            return(mzproxy_free_session(s_control, s_session, 0));
        }
        if(s_control->options->verbose != 0)(void)fprintf(stdout, "new pts name: \"%s\"\n", s_pts_name);

        s_session->flags |= def_mzproxy_session_flag_connected;
     
        s_session->pid = fork(); /* process fork */
        if(mzproxy_builtin_expect(s_session->pid == ((pid_t)(-1)), 0)) {
            if(s_control->options->verbose != 0)perror("fork");
            free((void *)s_pts_name);

            return(mzproxy_free_session(s_control, s_session, 0));
        }

        if(s_session->pid == ((pid_t)0)) {
            /* close all session (close handle) */
            s_control->session = mzproxy_free_session(s_control, s_control->session, 1);
            (void)mzproxy_free_session(s_control, s_session, 1); /* current session */
            (void)mzproxy_closesocket(s_control->socket); /* close listen socket */

            /* close stdin, stdout, stderr */
            (void)close(STDERR_FILENO); (void)close(STDOUT_FILENO); (void)close(STDIN_FILENO);

            if(setsid() != (-1)) {
                int s_tty_handle = open(s_pts_name, O_RDWR | O_NOCTTY); /* #0 open stdin */

                if(s_tty_handle != (-1)) {
                    int s_stdout_handle, s_stderr_handle;
                    struct termios s_termios;

                    s_stdout_handle = dup(s_tty_handle); /* #1 copy to stdout */
                    s_stderr_handle = dup(s_tty_handle); /* #2 copy to stderr */

                    (void)tcsetpgrp(0, getpid());
                    (void)ioctl(s_tty_handle, TIOCSCTTY, NULL);

                    if(tcgetattr(s_tty_handle, (struct termios *)(&s_termios)) == 0) { /* set pt property */
                        s_termios.c_iflag |= ICRNL;
                        s_termios.c_iflag &= ~(IXOFF);
                        s_termios.c_oflag |= XTABS;
                        s_termios.c_oflag |= ONLCR;
#if 0L /* TRY: enable implementation-defined output processing (SeeAlso: man termios) */
                        s_termios.c_oflag |= OPOST;
#endif      
                        s_termios.c_lflag |= ECHO;
                        (void)tcsetattr(s_tty_handle, TCSANOW, (struct termios *)(&s_termios));
                    }
                    if(s_control->options->issue_file != ((char *)0)) { /* issue file to pt */
                        int s_issue_handle = open(s_control->options->issue_file, O_RDONLY);
                        ssize_t s_read_bytes;
                        unsigned char s_byte, s_cr = (unsigned char)'\r';

                        if(s_issue_handle != (-1)) {
                            for(;;) {
                                s_read_bytes = read(s_issue_handle, (void *)(&s_byte), (size_t)sizeof(s_byte));
                                if(s_read_bytes != ((size_t)sizeof(s_byte)))break;

                                if(s_byte == '\n')(void)write(s_stdout_handle, (void *)(&s_cr), (size_t)sizeof(s_cr)); /* <LF> to <CR><LF> */
                                (void)write(s_stdout_handle, (void *)(&s_byte), (size_t)sizeof(s_cr));
                            }

                            (void)close(s_issue_handle);
                        }
                    }

                    if(execv(s_control->options->login_program, s_control->login_program_argv) == (-1))perror("execv"); /* execute login program */

                    (void)close(s_tty_handle);
                }
            }
            exit(EXIT_FAILURE);
        }

        free((void *)s_pts_name);
    }
    else
#endif
    {
#if __def_mzproxy_use_inet6__ != (0)   
        if(s_control->options->use_inet6 == 0)s_session->client_socket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
        else s_session->client_socket = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP);
#else
        s_session->client_socket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
#endif
        if(mzproxy_builtin_expect(s_session->client_socket == ((t_mzproxy_socket)(-1)), 0)) {
            if(s_control->options->verbose != 0)perror("client socket");

            return(mzproxy_free_session(s_control, s_session, 0));
        }

        (void)mzproxy_linger_on(s_session->client_socket);
    }
    
    if(s_control->session != ((struct ts_mzproxy_session *)0))s_control->session->prev = s_session;
    s_session->next = s_control->session;
    s_control->session = s_session;
    s_control->session_count++;

    return(s_session);
}

/* remove session entry */
static void (mzproxy_del_session)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session)
{
    if(s_session == ((struct ts_mzproxy_session *)0))return;

    if(s_session->next != ((struct ts_mzproxy_session *)0))s_session->next->prev = s_session->prev;
    if(s_session->prev != ((struct ts_mzproxy_session *)0))s_session->prev->next = s_session->next;
    else s_control->session = s_session->next;
    s_session->prev = (struct ts_mzproxy_session *)0;
    s_session->next = (struct ts_mzproxy_session *)0;

    (void)mzproxy_free_session(s_control, s_session, 0);

    s_control->session_count--;
}

/* register event */
static void (mzproxy_register_event)(struct ts_mzproxy_control *s_control)
{
    struct ts_mzproxy_session *s_session = s_control->session, *s_next_session;
 
    FD_ZERO(&s_control->fd_set_rx);
    FD_ZERO(&s_control->fd_set_tx);
 
    s_control->max_fd = 0;
 
    if(s_control->socket != ((t_mzproxy_socket)(-1))) {
        FD_SET(s_control->socket, &s_control->fd_set_rx);
        if(((int)s_control->socket) > s_control->max_fd)s_control->max_fd = (int)s_control->socket;
    }

    while(s_session != ((struct ts_mzproxy_session *)0)) {
        s_next_session = s_session->next;

        if((s_session->flags & def_mzproxy_session_flag_connected) == def_mzproxy_session_flag_none) { /* check connection timeout */
            if(mzproxy_builtin_expect((s_control->time_stamp - s_session->connect_time_stamp) >= s_control->options->connect_timeout, 0)) {
                s_next_session = s_session->next;
                if(s_control->options->verbose != 0)(void)fprintf(stderr, "connect timeout (id=%d, timeout=%lu/%lu)\n", (int)s_session->accept_socket, s_control->time_stamp - s_session->connect_time_stamp, s_control->options->connect_timeout);
                mzproxy_del_session(s_control, s_session);
                s_session = s_next_session;

                continue;
            }

            if(s_control->options->verbose != 0)(void)fprintf(stdout, "check connect timeout (id=%d, timeout=%lu/%lu)\n", (int)s_session->accept_socket, s_control->time_stamp - s_session->connect_time_stamp, s_control->options->connect_timeout);
        }

        if(s_control->options->data_timeout > 0ul) { /* check data timeout */
            if(mzproxy_builtin_expect((s_control->time_stamp - s_session->data_time_stamp) >= s_control->options->data_timeout, 0)) {
                s_next_session = s_session->next;
                if(s_control->options->verbose != 0)(void)fprintf(stderr, "data timeout (id=%d, timeout=%lu/%lu)\n", (int)s_session->accept_socket, s_control->time_stamp - s_session->data_time_stamp, s_control->options->data_timeout);
                mzproxy_del_session(s_control, s_session);
                s_session = s_next_session;

                continue;
            }

            if(s_control->options->verbose != 0)(void)fprintf(stdout, "check data timeout (id=%d, timeout=%lu/%lu)\n", (int)s_session->accept_socket, s_control->time_stamp - s_session->data_time_stamp, s_control->options->data_timeout);
        }

        if((s_session->client_socket != ((t_mzproxy_socket)(-1))) && (s_session->connect_port != 0) && ((s_session->flags & def_mzproxy_session_flag_connecting) == def_mzproxy_session_flag_none)) { /* connecting session */
            struct addrinfo s_hints;
            struct addrinfo *s_addrinfo_result;
            struct addrinfo *s_addrinfo;
            int s_addrinfo_check;
            t_mzproxy_socket s_try_socket;
            
            (void)memset((void *)(&s_hints), 0, (size_t)sizeof(s_hints));
            s_hints.ai_family = AF_UNSPEC; /* IPv4 or IPv6 */
            s_hints.ai_socktype = SOCK_STREAM; /* UDP or TCP */
            s_hints.ai_flags = AI_PASSIVE; /* wildcard IP address */
            s_hints.ai_protocol = IPPROTO_TCP; /* any protocol */
            s_hints.ai_canonname = NULL;
            s_hints.ai_addr = NULL;
            s_hints.ai_next = NULL;

            s_addrinfo_check = getaddrinfo(s_control->options->connect_address, (const char *)0, (const struct addrinfo *)(&s_hints), (struct addrinfo **)(&s_addrinfo_result));
            if(mzproxy_builtin_expect(s_addrinfo_check != 0, 0)) {
                if(s_control->options->verbose != 0)(void)fprintf(stderr, "getaddrinfo error: %s\n", (char *)gai_strerror(s_addrinfo_check));
                mzproxy_del_session(s_control, s_session);
                s_session = s_next_session;

                continue;
            }
            
            s_session->flags |= def_mzproxy_session_flag_connecting;
            s_try_socket = (t_mzproxy_socket)(-1);

            s_addrinfo = s_addrinfo_result;
            while(s_addrinfo != ((struct addrinfo *)0)) {
#if __def_mzproxy_use_inet6__ != (0)	    
                if(s_addrinfo->ai_family == AF_INET6) { /* try IPv6 connection */
                    struct sockaddr_in6 *s_sockaddr_in6;
                    
                    s_try_socket = (t_mzproxy_socket)socket(s_addrinfo->ai_family, s_addrinfo->ai_socktype, s_addrinfo->ai_protocol);
                    if(s_try_socket == ((t_mzproxy_socket)(-1))) {
                        continue; 
                    }

                    if(mzproxy_builtin_expect(mzproxy_nonblock_socket(s_try_socket, 1) == (-1), 0)) {
                        if(s_control->options->verbose != 0)perror("non-block socket");
                        mzproxy_closesocket(s_try_socket);
                        s_try_socket = (t_mzproxy_socket)(-1);
                    
                        continue;
                    }
                    
                    mzproxy_closesocket(s_session->client_socket);
                    s_session->client_socket = s_try_socket;
                    
                    s_sockaddr_in6 = (struct sockaddr_in6 *)s_addrinfo->ai_addr;
                    s_sockaddr_in6->sin6_port = htons(s_session->connect_port);

                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "connecting(IPv6)... (id=%d, \"%s:%d\")\n", (int)s_session->accept_socket, s_control->options->connect_address, s_session->connect_port);
                    }

                    if(connect(s_session->client_socket, (struct sockaddr *)s_sockaddr_in6, (socklen_t)sizeof(struct sockaddr_in6)) == 0) {
                        if(s_control->options->verbose != 0)(void)fprintf(stdout, "connected (id=%d,\"%s:%d\"[now])\n", (int)s_session->accept_socket, s_control->options->connect_address, s_session->connect_port);
                        (void)mzproxy_nonblock_socket(s_session->client_socket, 0);
                        s_session->flags &= ~(def_mzproxy_session_flag_nonblock);
                        s_session->data_time_stamp = s_control->time_stamp;
                        s_session->flags |= def_mzproxy_session_flag_connected;
                    }

                    break;
                }
#endif		

                if(s_addrinfo->ai_family == AF_INET) { /* try IPv4 connection */
                    struct sockaddr_in *s_sockaddr_in;
                
                    s_try_socket = (t_mzproxy_socket)socket(s_addrinfo->ai_family, s_addrinfo->ai_socktype, s_addrinfo->ai_protocol);
                    if(s_try_socket == ((t_mzproxy_socket)(-1))) {
                        continue; 
                    }

                    if(mzproxy_builtin_expect(mzproxy_nonblock_socket(s_try_socket, 1) == (-1), 0)) {
                        if(s_control->options->verbose != 0)perror("non-block socket");
                        mzproxy_closesocket(s_try_socket);
                        s_try_socket = (t_mzproxy_socket)(-1);
                    
                        continue;
                    }

                    
                    mzproxy_closesocket(s_session->client_socket);
                    s_session->client_socket = s_try_socket;
                    
                    s_sockaddr_in = (struct sockaddr_in *)s_addrinfo->ai_addr;
                    s_sockaddr_in->sin_port = htons(s_session->connect_port);

                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "connecting(IPv4)... (id=%d, \"%s:%d\")\n", (int)s_session->accept_socket, s_control->options->connect_address, s_session->connect_port);
                    }

                    if(connect(s_session->client_socket, (struct sockaddr *)s_sockaddr_in, (socklen_t)sizeof(struct sockaddr_in)) == 0) {
                        if(s_control->options->verbose != 0)(void)fprintf(stdout, "connected (id=%d,\"%s:%d\"[now])\n", (int)s_session->accept_socket, s_control->options->connect_address, s_session->connect_port);
                        (void)mzproxy_nonblock_socket(s_session->client_socket, 0);
                        s_session->flags &= ~(def_mzproxy_session_flag_nonblock);
                        s_session->data_time_stamp = s_control->time_stamp;
                        s_session->flags |= def_mzproxy_session_flag_connected;
                    }
                    
                    break;
                }
                
                s_addrinfo = s_addrinfo->ai_next;
            }

            freeaddrinfo(s_addrinfo_result);

            if(mzproxy_builtin_expect(s_try_socket == ((t_mzproxy_socket)(-1)), 0)) {
                if(s_control->options->verbose != 0)(void)fprintf(stderr, "oops! where to connection?\n");
                mzproxy_del_session(s_control, s_session);
                s_session = s_next_session;

                continue;
            }
        }

        if((s_session->size[1] > ((size_t)0))
#if __def_mzproxy_use_ief__ != (0)
            || (
                (s_control->options->use_ief != 0) &&
                ((s_session->flags & (def_mzproxy_session_flag_cert_passed | def_mzproxy_session_flag_cert_ack)) == def_mzproxy_session_flag_cert_passed)
            )
#endif      
        ) {
            FD_SET(s_session->accept_socket, &s_control->fd_set_tx); /* to client */
        }

        if(s_session->size[0] < s_control->options->buffer_size)FD_SET(s_session->accept_socket, &s_control->fd_set_rx); /* from client */
        
        if(((int)s_session->accept_socket) > s_control->max_fd)s_control->max_fd = (int)s_session->accept_socket;

        if((s_session->flags & def_mzproxy_session_flag_connecting) == def_mzproxy_session_flag_connecting) {
            if(((s_session->flags & def_mzproxy_session_flag_connected) == def_mzproxy_session_flag_none) ||
#if __def_mzproxy_use_ief__ != (0)
                ((s_control->options->use_ief == 0) && (s_session->size[0] > ((size_t)0))) /* normal buffer */ ||
                ((s_control->options->use_ief != 0) && (s_session->size[2] > ((size_t)0))) /* decrypted buffer */
#else      
                (s_session->size[0] > ((size_t)0))
#endif
            ) {
                FD_SET(s_session->client_socket, &s_control->fd_set_tx); /* to server */
            }

            if(s_session->size[1] < s_control->options->buffer_size)FD_SET(s_session->client_socket, &s_control->fd_set_rx); /* from server */

            if(((int)s_session->client_socket) > s_control->max_fd)s_control->max_fd = (int)s_session->client_socket;
        }

        s_session = s_next_session;
    }
}

static void (mzproxy_check_listen)(struct ts_mzproxy_control *s_control)
{
    t_mzproxy_socket s_accept_socket;
    socklen_t s_accept_socket_address_size;
    struct ts_mzproxy_session *s_new_session;
    char s_address_string[ 64 ];
    unsigned int s_address_port;

    /* check on connection (accept) */
    if(mzproxy_builtin_expect(s_control->socket == ((t_mzproxy_socket)(-1)), 0))return;
    if(!FD_ISSET(s_control->socket, &s_control->fd_set_rx))return;
    
    s_control->events--;
#if __def_mzproxy_use_inet6__ != (0)
    if(s_control->options->use_inet6 == 0) {
        struct sockaddr_in s_accept_socket_address;

        s_accept_socket_address_size = (socklen_t)sizeof(s_accept_socket_address);
        s_accept_socket = accept(s_control->socket, (struct sockaddr *)(&s_accept_socket_address), (socklen_t *)(&s_accept_socket_address_size));
        if(mzproxy_builtin_expect(s_accept_socket == (-1), 0)) {
            if(s_control->options->verbose != 0)perror("accept");

            return;
        }

        (void)mzproxy_linger_on(s_accept_socket);
#if __def_mzproxy_use_inet_ntop__ != (0)
        (void)inet_ntop(AF_INET, (void *)(&s_accept_socket_address.sin_addr), (char *)(&s_address_string[0]), sizeof(s_address_string));
#else
        (void)strcpy((char *)(&s_address_string[0]), inet_ntoa(s_accept_socket_address.sin_addr));
#endif
        s_address_port = ntohs(s_accept_socket_address.sin_port);
    }
    else {
        struct sockaddr_in6 s_accept_socket_address6;

        s_accept_socket_address_size = (socklen_t)sizeof(s_accept_socket_address6);
        s_accept_socket = accept(s_control->socket, (struct sockaddr *)(&s_accept_socket_address6), (socklen_t *)(&s_accept_socket_address_size));
        if(mzproxy_builtin_expect(s_accept_socket == (-1), 0)) {
            if(s_control->options->verbose != 0)perror("accept");

            return;
        }

        (void)mzproxy_linger_on(s_accept_socket);
        (void)inet_ntop(s_accept_socket_address6.sin6_family, (void *)(&s_accept_socket_address6.sin6_addr), (char *)(&s_address_string[0]), sizeof(s_address_string));
        s_address_port = ntohs(s_accept_socket_address6.sin6_port);
    }
#else
    do {
        struct sockaddr_in s_accept_socket_address;

        s_accept_socket_address_size = (socklen_t)sizeof(s_accept_socket_address);
        s_accept_socket = accept(s_control->socket, (struct sockaddr *)(&s_accept_socket_address), (socklen_t *)(&s_accept_socket_address_size));
        if(mzproxy_builtin_expect(s_accept_socket == (-1), 0)) {
            if(s_control->options->verbose != 0)perror("accept");

            return;
        }

        (void)mzproxy_linger_on(s_accept_socket);
#if __def_mzproxy_use_inet_ntop__ != (0)
        (void)inet_ntop(s_accept_socket_address.sin_family, (void *)(&s_accept_socket_address.sin_addr), (char *)(&s_address_string[0]), sizeof(s_address_string));
#else
        (void)strcpy((char *)(&s_address_string[0]), inet_ntoa(s_accept_socket_address.sin_addr));
#endif
        s_address_port = ntohs(s_accept_socket_address.sin_port);
    }while(0);
#endif

    if(mzproxy_builtin_expect((s_control->options->limit_session_count > 0u) && (s_control->options->limit_session_count <= s_control->session_count), 0)) { /* limited session count - refuse connection */
        if(s_control->options->verbose != 0)(void)fprintf(stderr, "limited session: refuse connection (%s:%u)\n", (char *)(&s_address_string[0]), s_address_port);
        mzproxy_closesocket(s_accept_socket);

        return;
    }

    /* create & add session entry */
    s_new_session = mzproxy_new_session(s_control, s_accept_socket);
    if(s_new_session != ((struct ts_mzproxy_session *)0)) {
        if(s_control->options->verbose != 0) {
            (void)fprintf(stdout, "connection from %s:%u (id=%d, sessions=%u/%u, redirect_port=%d)\n",
             (char *)(&s_address_string[0]), s_address_port,
             (int)s_new_session->accept_socket,
             s_control->session_count,
             s_control->options->limit_session_count,
             s_new_session->connect_port);
        }
    }
}

/* to server: buffer[0] -> server */
static int (mzproxy_to_server)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session)
{
    int s_send_bytes, s_select_buffer = 0;

    if(!FD_ISSET(s_session->client_socket, &s_control->fd_set_tx))return(0);

    s_control->events--;
    if((s_session->flags & def_mzproxy_session_flag_connecting) != def_mzproxy_session_flag_connecting)return(0);

#if __def_mzproxy_use_ief__ != (0)  
     if(s_control->options->use_ief != 0)s_select_buffer = 2;
#endif

    if((s_session->flags & def_mzproxy_session_flag_connected) == def_mzproxy_session_flag_none) { /* connection check */
        int s_socket_error;
#if defined(_WIN32) || defined(WIN32) 
        int s_socket_error_length;
#else
        socklen_t s_socket_error_length;
#endif

        s_socket_error_length = (socklen_t)sizeof(s_socket_error);
        if(mzproxy_builtin_expect(getsockopt(s_session->client_socket, SOL_SOCKET, SO_ERROR, (void *)(&s_socket_error), &s_socket_error_length) != 0, 0)) {
            if(s_control->options->verbose != 0)(void)fprintf(stderr, "can not connect (id=%d,getsockopt)\n", (int)s_session->accept_socket);

            return(-1);
        }

        if(mzproxy_builtin_expect(s_socket_error != 0, 0)) {
            if(s_control->options->verbose != 0)(void)fprintf(stderr, "can not connect (id=%d,error)\n", (int)s_session->accept_socket);

            return(-1);
        }

        /* connected */
        if(s_control->options->verbose != 0)(void)fprintf(stdout, "connected (id=%d,[tx])\n", (int)s_session->accept_socket);
        (void)mzproxy_nonblock_socket(s_session->client_socket, 0);
        s_session->flags &= ~(def_mzproxy_session_flag_nonblock);
        s_session->data_time_stamp = s_control->time_stamp;
        s_session->flags |= def_mzproxy_session_flag_connected;

        return(0);
    }

#if __def_mzproxy_use_ief__ != (0)
    if((s_control->options->use_ief != 0) && ((s_session->flags & def_mzproxy_session_flag_cert_ack) == def_mzproxy_session_flag_none)) {
        /* wait for cert ack */
        if(s_control->options->verbose != 0)(void)fprintf(stdout, "wait for cert ack (id=%d)\n", (int)s_session->accept_socket);

        return(0);
    }
#endif

    if(s_session->size[s_select_buffer] <= ((size_t)0))return(0);

    do {
        size_t s_from = (size_t)0, s_to = (size_t)0;

#if __def_mzproxy_use_telnetd__ != (0)  
        if(s_control->options->use_telnetd != 0) {
            while(s_from < s_session->size[s_select_buffer]) {
                if(s_session->buffer[s_select_buffer][s_from] != IAC) { /* pt data */
                    if(s_to != s_from)s_session->buffer[s_select_buffer][s_to] = s_session->buffer[s_select_buffer][s_from];
                    s_to++, s_from++;
                }
                else {
                    if(mzproxy_builtin_expect((s_session->size[s_select_buffer] - s_from) < ((size_t)3), 0))break;

                    switch(s_session->buffer[s_select_buffer][s_from + ((size_t)1)]) { 
                        case DO:
                        case DONT:
                        case WILL:
                        case WONT:
                             s_from += (size_t)3; /* ignore IAC */
                             break;
                        case SB:
                             if(s_session->buffer[s_select_buffer][s_from + ((size_t)2)] == TELOPT_NAWS) { /* setup pt window size */
                                 struct winsize s_winsize;

                                 if((s_session->size[s_select_buffer] - s_from) < ((size_t)9))break;
                                 s_winsize.ws_col = ((unsigned short int)s_session->buffer[s_select_buffer][s_from + ((size_t)3)] << 8) | ((unsigned short int)s_session->buffer[s_select_buffer][s_from + ((size_t)4)]);
                                 s_winsize.ws_row = ((unsigned short int)s_session->buffer[s_select_buffer][s_from + ((size_t)5)] << 8) | ((unsigned short int)s_session->buffer[s_select_buffer][s_from + ((size_t)6)]);
                                 (void)ioctl((int)s_session->client_socket, TIOCSWINSZ, (void *)(&s_winsize));
                                 s_from += (size_t)(3 + 4 + 2);

                                 if(s_control->options->verbose != 0)(void)fprintf(stdout, "winsize=%d,%d\n", (int)s_winsize.ws_col, (int)s_winsize.ws_row);
                             }
                             break;
                        default: s_from += (size_t)3; /* ignore IAC */ break;
                    }
                }
            }
        }
        else s_to = s_from = s_session->size[s_select_buffer];
#else        
        s_to = s_from = s_session->size[s_select_buffer];
#endif  

        s_session->size[s_select_buffer] = s_to + (s_session->size[s_select_buffer] - s_from);
        
        if(s_to <= ((size_t)0))return(0);

#if __def_mzproxy_use_telnetd__ != (0)
        if(s_control->options->use_telnetd != 0) {
            s_send_bytes = (int)write(s_session->client_socket, (void *)(&s_session->buffer[s_select_buffer][0]), s_to);
        }
        else
#endif
        { 
#if defined(_WIN32) || defined(WIN32)   
            s_send_bytes = send(s_session->client_socket, (void *)(&s_session->buffer[s_select_buffer][0]), (int)s_to, 0);
#else
            s_send_bytes = send(s_session->client_socket, (void *)(&s_session->buffer[s_select_buffer][0]), s_to, MSG_NOSIGNAL);
#endif
        }

        if((s_send_bytes == 0) || ((s_send_bytes == (-1)) && (errno != EINTR))) { /* disconnected by server */
            if(s_control->options->verbose != 0)(void)fprintf(stdout, "disconnected by server (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);

            return(-1);
        }

        if(s_control->options->verbose != 0) {
            (void)fprintf(stdout, "client -> server (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);
#if __def_mzproxy_tx_dump__ != (0)
            (void)mzapi_dump_inline(&s_session->buffer[s_select_buffer][0], s_send_bytes);
#endif    
        }

        if(((size_t)s_send_bytes) == s_session->size[s_select_buffer])s_session->size[s_select_buffer] = (size_t)0;
        else { /* strip left */
            (void)memmove((void *)(&s_session->buffer[s_select_buffer][0]), (void *)(&s_session->buffer[s_select_buffer][s_send_bytes]), s_session->size[s_select_buffer] - ((size_t)s_send_bytes));
            s_session->size[s_select_buffer] -= (size_t)s_send_bytes;
        }
    }while(0);
    
    return(0);
}

/* to client: buffer[1] -> client */
static int (mzproxy_to_client)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session)
{
    int s_send_bytes;

    if(!FD_ISSET(s_session->accept_socket, &s_control->fd_set_tx))return(0);

    s_control->events--;

#if __def_mzproxy_use_ief__ != (0)
    if(s_control->options->use_ief != 0) { /* ief mode */
        if((s_session->flags & def_mzproxy_session_flag_cert_ack) == def_mzproxy_session_flag_none) { /* cert ack (send decrypt key) */
            if((s_session->flags & def_mzproxy_session_flag_cert_passed) == def_mzproxy_session_flag_cert_passed) {
#if defined(_WIN32) || defined(WIN32)     
                s_send_bytes = send(s_session->accept_socket, (void *)(&s_session->padd_block[0]), (int)sizeof(s_session->padd_block), 0);
#else
                s_send_bytes = send(s_session->accept_socket, (void *)(&s_session->padd_block[0]), (size_t)sizeof(s_session->padd_block), MSG_NOSIGNAL);
#endif
                if(mzproxy_builtin_expect((s_send_bytes != ((int)sizeof(s_session->padd_block))) || (s_send_bytes == 0) || ((s_send_bytes == (-1)) && (errno != EINTR)), 0)) { /* can not cert ack */
                    if(s_control->options->verbose != 0)(void)fprintf(stderr, "can not cert ack (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);
                    return(-1);
                }

                s_session->flags |= def_mzproxy_session_flag_cert_ack;

                if(s_control->options->verbose != 0)(void)fprintf(stdout, "cert ack (id=%d)\n", (int)s_session->accept_socket);
            }
        }
        else if(s_session->size[1] > ((size_t)0)) { /* ief: encrypt packetize */
            unsigned char s_encrypt_buffer[ 4 << 10 ];
            size_t s_want_size, s_padd_size;

            (void)memset((void *)(&s_encrypt_buffer[0]), 0, (size_t)sizeof(s_encrypt_buffer));

            s_want_size = s_session->size[1];
            if(s_want_size > ((size_t)(sizeof(s_encrypt_buffer) - 32)))s_want_size = (size_t)(sizeof(s_encrypt_buffer) - 32);
            
            s_padd_size = ((size_t)2) + s_want_size + ((size_t)(sizeof(s_session->padd_block) - 1));
            s_padd_size -= s_padd_size % ((size_t)sizeof(s_session->padd_block));
            if((s_session->flags & def_mzproxy_session_flag_iet) == def_mzproxy_session_flag_iet) {
                size_t s_temp_size;

                s_temp_size = (s_padd_size / ((size_t)__def_mzapi_seed_block_size__)) - ((size_t)1);
                s_encrypt_buffer[0] = (unsigned char)(s_temp_size >>  0);
                s_encrypt_buffer[1] = (unsigned char)(s_temp_size >>  8);
            }
            else {
                s_encrypt_buffer[0] = (unsigned char)(s_padd_size >>  0);
                s_encrypt_buffer[1] = (unsigned char)(s_padd_size >>  8);
            }
            s_encrypt_buffer[2] = (unsigned char)(s_want_size >>  0);
            s_encrypt_buffer[3] = (unsigned char)(s_want_size >>  8);
            (void)memcpy((void *)(&s_encrypt_buffer[4]), (void *)(&s_session->buffer[1][0]), s_want_size);

            (void)mzapi_seed_encrypt((__t_mzapi_ptr__)(&s_encrypt_buffer[2]), (__t_mzapi_size__)s_padd_size, (__t_mzapi_ptr__)(&s_session->round_key[0]));

#if defined(_WIN32) || defined(WIN32)     
            s_send_bytes = send(s_session->accept_socket, (void *)(&s_encrypt_buffer[0]), (int)(((size_t)2) + s_padd_size), 0);
#else
            s_send_bytes = send(s_session->accept_socket, (void *)(&s_encrypt_buffer[0]), ((size_t)2) + s_padd_size, MSG_NOSIGNAL);
#endif

            if((s_send_bytes != (((size_t)2) + s_padd_size)) || (s_send_bytes == 0) || ((s_send_bytes == (-1)) && (errno != EINTR))) { /* disconnected by client */
                if(s_control->options->verbose != 0)(void)fprintf(stderr, "disconnected by client (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);

                return(-1);
            }

            if(s_control->options->verbose != 0) {
                (void)fprintf(stdout, "server -> client (id=%d, tx=%d, encrypt)\n", (int)s_session->accept_socket, (int)s_want_size);
#if  __def_mzproxy_tx_dump__ != (0)
                (void)mzapi_dump_inline(&s_session->buffer[1][0], s_want_size);
#endif   
            }

            if(s_want_size == s_session->size[1])s_session->size[1] = (size_t)0;
            else { /* strip left */
                (void)memmove((void *)(&s_session->buffer[1][0]), (void *)(&s_session->buffer[1][s_want_size]), s_session->size[1] - ((size_t)s_want_size));
                s_session->size[1] -= (size_t)s_want_size;
            }
        } 
    }
    else
#endif
    if(s_session->size[1] > ((size_t)0)) { /* normal send */
#if defined(_WIN32) || defined(WIN32)     
        s_send_bytes = send(s_session->accept_socket, (void *)(&s_session->buffer[1][0]), (int)s_session->size[1], 0);
#else
        s_send_bytes = send(s_session->accept_socket, (void *)(&s_session->buffer[1][0]), s_session->size[1], MSG_NOSIGNAL);
#endif
        if((s_send_bytes == 0) || ((s_send_bytes == (-1)) && (errno != EINTR))) { /* disconnected by client */
            if(s_control->options->verbose != 0)(void)fprintf(stdout, "disconnected by client (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);

            return(-1);
        }

        if(s_control->options->verbose != 0) {
            (void)fprintf(stdout, "server -> client (id=%d, tx=%d)\n", (int)s_session->accept_socket, (int)s_send_bytes);
#if __def_mzproxy_tx_dump__ != (0)
            (void)mzapi_dump_inline(&s_session->buffer[1][0], s_send_bytes);
#endif   
        }

        if(((size_t)s_send_bytes) == s_session->size[1])s_session->size[1] = (size_t)0;
        else { /* strip left */
            (void)memmove((void *)(&s_session->buffer[1][0]), (void *)(&s_session->buffer[1][s_send_bytes]), s_session->size[1] - ((size_t)s_send_bytes));
            s_session->size[1] -= (size_t)s_send_bytes;
        }
    }
    
    return(0);
}

/* from client: client -> buffer[0] */
static int (mzproxy_from_client)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session)
{
    int s_recv_bytes;

    if(!FD_ISSET(s_session->accept_socket, &s_control->fd_set_rx))return(0);

    s_control->events--;

    if(s_session->size[0] >= s_control->options->buffer_size)return(0);

#if defined(_WIN32) || defined(WIN32)     
    s_recv_bytes = recv(s_session->accept_socket, (void *)(&s_session->buffer[0][s_session->size[0]]), (int)(s_control->options->buffer_size - s_session->size[0]), 0);
#else
    s_recv_bytes = recv(s_session->accept_socket, (void *)(&s_session->buffer[0][s_session->size[0]]), s_control->options->buffer_size - s_session->size[0], MSG_NOSIGNAL);
#endif

    if((s_recv_bytes == 0) || ((s_recv_bytes == (-1)) && (errno != EINTR))) { /* disconnected by client */
        if(s_control->options->verbose != 0)(void)fprintf(stdout, "disconnected by client (id=%d, rx=%d)\n", (int)s_session->accept_socket, (int)s_recv_bytes);

        return(-1);
    }

    s_session->data_time_stamp = s_control->time_stamp;

    if(s_control->options->verbose != 0) {
        (void)fprintf(stdout, "from client (id=%d, rx=%d)\n", (int)s_session->accept_socket, (int)s_recv_bytes);
#if __def_mzproxy_rx_dump__ != (0)
        (void)mzapi_dump_inline(&s_session->buffer[0][s_session->size[0]], s_recv_bytes);
#endif   
    }

    s_session->size[0] += (size_t)s_recv_bytes;
    if((s_session->connect_port == 0) && (s_control->options->use_pm != 0)) { /* redirect port map recv */
        if(s_session->size[0] >= ((size_t)2)) {
            s_session->connect_port = (((int)s_session->buffer[0][1]) << 8) | ((int)s_session->buffer[0][0]);
            if(s_control->options->verbose != 0) {
                (void)fprintf(stdout, "recv info: redirect_port=%d(%d,%d)\n", s_session->connect_port, (int)s_session->buffer[0][0], (int)s_session->buffer[0][1]);
            }

            (void)memmove((void *)(&s_session->buffer[0][0]), (void *)(&s_session->buffer[0][2]), s_session->size[0] - ((size_t)2));
            
            s_session->size[0] -= (size_t)2;
        }
        else { /* need more data from client */
            if(s_control->options->verbose != 0) {
                (void)fprintf(stdout, "wait for port map recv (id=%d,size=%lu)\n", (int)s_session->accept_socket, (unsigned long)s_session->size[0]);
            }
        }
    }

#if __def_mzproxy_use_ief__ != (0)
    if(s_control->options->use_ief != 0) {
        if((s_session->flags & def_mzproxy_session_flag_cert_passed) == def_mzproxy_session_flag_none) { /* magic code & random 1 recv */
            if(s_session->size[0] >= ((size_t)4)) {
                long unsigned int s_magic_code, s_random_1, s_random_2;
                __t_mzapi_dword__ s_round_key[ __def_mzapi_seed_round_keys__ ];

                s_magic_code = (((long unsigned int)s_session->buffer[0][3]) << 24) | (((long unsigned int)s_session->buffer[0][2]) << 16) |
                               (((long unsigned int)s_session->buffer[0][1]) <<  8) | (((long unsigned int)s_session->buffer[0][0]) <<  0);
                if(s_magic_code == def_mzproxy_ief_magic_code) { /* ief protocol */
                    s_session->flags |= def_mzproxy_session_flag_ief;
                }
                else if(s_magic_code == def_mzproxy_iet_magic_code) { /* iet protocol */
                    s_session->flags |= def_mzproxy_session_flag_iet;
                }
                else {
                    if(s_control->options->verbose != 0)(void)fprintf(stderr, "bad magic code (id=%d)\n", (int)s_session->accept_socket);

                    return(-1);
                }

                if((s_session->flags & def_mzproxy_session_flag_iet) == def_mzproxy_session_flag_iet) {
                    s_random_1   = (long unsigned int)s_control->options->ief_data_port;
                    (void)memset((void *)(&s_session->padd_block[0]), 0, sizeof(s_session->padd_block));
                    s_session->padd_block[0] = (unsigned char)(s_random_1 >>  0);
                    s_session->padd_block[1] = (unsigned char)(s_random_1 >>  8);
                    s_session->padd_block[2] = (unsigned char)(s_random_1 >> 16);
                    s_session->padd_block[3] = (unsigned char)(s_random_1 >> 24);
                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "iet info: magic_code=%08lXH, random_1=%08lXH\n", s_magic_code, s_random_1);
                    }
                    (void)memmove((void *)(&s_session->buffer[0][0]), (void *)(&s_session->buffer[0][8]), s_session->size[0] - ((size_t)4));
                    s_session->size[0] -= (size_t)4;
                    s_random_2 = s_random_1 + ((long unsigned int)rand());
                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "accepted magic code (id=%d, random_2=%08lXH)\n", (int)s_session->accept_socket, s_random_2);
                    }
                    (void)mzapi_seed_make_round_key((__t_mzapi_ptr__)(&s_round_key[0]), (__t_mzapi_ptr__)(&s_session->padd_block[0]));
                    s_session->padd_block[0] = (__t_mzapi_byte__)(s_random_2 >>  0);
                    s_session->padd_block[1] = (__t_mzapi_byte__)(s_random_2 >>  8);
                    s_session->padd_block[2] = (__t_mzapi_byte__)(s_random_2 >> 16);
                    s_session->padd_block[3] = (__t_mzapi_byte__)(s_random_2 >> 24);
                    (void)mzapi_seed_make_round_key((__t_mzapi_ptr__)(&s_session->round_key[0]), (__t_mzapi_ptr__)(&s_session->padd_block[0]));
                    (void)mzapi_seed_encrypt((__t_mzapi_ptr__)(&s_session->padd_block[0]), (__t_mzapi_size__)sizeof(s_session->padd_block), (__t_mzapi_ptr__)(&s_round_key));
                    s_session->flags |= def_mzproxy_session_flag_cert_passed; 
                }
                else if(s_session->size[0] >= ((size_t)(4 + 4))) {
                    s_random_1   = (((long unsigned int)s_session->buffer[0][7]) << 24) | (((long unsigned int)s_session->buffer[0][6]) << 16) |
                                   (((long unsigned int)s_session->buffer[0][5]) <<  8) | (((long unsigned int)s_session->buffer[0][4]) <<  0);
                    (void)memset((void *)(&s_session->padd_block[0]), 0, sizeof(s_session->padd_block));
                    s_session->padd_block[0] = s_session->buffer[0][4];
                    s_session->padd_block[1] = s_session->buffer[0][5];
                    s_session->padd_block[2] = s_session->buffer[0][6];
                    s_session->padd_block[3] = s_session->buffer[0][7];
                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "ief info: magic_code=%08lXH, random_1=%08lXH\n", s_magic_code, s_random_1);
                    }
                    (void)memmove((void *)(&s_session->buffer[0][0]), (void *)(&s_session->buffer[0][8]), s_session->size[0] - ((size_t)8));
                    s_session->size[0] -= (size_t)8;
                    s_random_2 = s_random_1 + ((long unsigned int)rand());
                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "accepted magic code (id=%d, random_2=%08lXH)\n", (int)s_session->accept_socket, s_random_2);
                    }
                    (void)mzapi_seed_make_round_key((__t_mzapi_ptr__)(&s_round_key[0]), (__t_mzapi_ptr__)(&s_session->padd_block[0]));
                    s_session->padd_block[0] = (__t_mzapi_byte__)(s_random_2 >>  0);
                    s_session->padd_block[1] = (__t_mzapi_byte__)(s_random_2 >>  8);
                    s_session->padd_block[2] = (__t_mzapi_byte__)(s_random_2 >> 16);
                    s_session->padd_block[3] = (__t_mzapi_byte__)(s_random_2 >> 24);
                    (void)mzapi_seed_make_round_key((__t_mzapi_ptr__)(&s_session->round_key[0]), (__t_mzapi_ptr__)(&s_session->padd_block[0]));
                    s_session->padd_block[4] = (__t_mzapi_byte__)(s_control->options->ief_data_port >>  0);
                    s_session->padd_block[5] = (__t_mzapi_byte__)(s_control->options->ief_data_port >>  8);
                    (void)mzapi_seed_encrypt((__t_mzapi_ptr__)(&s_session->padd_block[0]), (__t_mzapi_size__)sizeof(s_session->padd_block), (__t_mzapi_ptr__)(&s_round_key));
                    s_session->flags |= def_mzproxy_session_flag_cert_passed; 
                }
                else
                {
                    if(s_control->options->verbose != 0) {
                        (void)fprintf(stdout, "wait for random 1 recv (id=%d,size=%lu)\n", (int)s_session->accept_socket, (unsigned long)s_session->size[0]);
                    }
                }
            }
            else
            { /* need more data from client */
                if(s_control->options->verbose != 0) {
                    (void)fprintf(stdout, "wait for magic code recv (id=%d,size=%lu)\n", (int)s_session->accept_socket, (unsigned long)s_session->size[0]);
                }
            }
        }
        else if(s_session->size[2] <= ((size_t)0))
        { /* decrypt ief packet */
            size_t s_padd_size, s_want_size;
            int s_invalid_packet = 0;

            while(s_session->size[0] >= ((size_t)2)) {
                s_padd_size = (((size_t)s_session->buffer[0][1]) << 8) | ((size_t)s_session->buffer[0][0]);
                if((s_session->flags & def_mzproxy_session_flag_iet) == def_mzproxy_session_flag_iet)s_padd_size = (s_padd_size + ((size_t)1)) * ((size_t)__def_mzapi_seed_block_size__);
                
                if(s_session->size[0] < (((size_t)2) + s_padd_size))break; /* more */
                
                (void)mzapi_seed_decrypt((__t_mzapi_ptr__)(&s_session->buffer[0][2]), (__t_mzapi_size__)s_padd_size, (__t_mzapi_ptr__)(&s_session->round_key[0]));
                
                s_want_size = (((size_t)s_session->buffer[0][3]) << 8) | ((size_t)s_session->buffer[0][2]);
                if(mzproxy_builtin_expect(s_padd_size < (((size_t)2) + s_want_size), 0)) {
                    if(s_control->options->verbose != 0)(void)fprintf(stderr, "invalid ief packetize (id=%d, padd_size=%lu, data_size=%lu)\n", (int)s_session->accept_socket, (unsigned long)s_padd_size, (unsigned long)s_want_size);
                    s_invalid_packet = 1;

                    break;
                }

                (void)memcpy((void *)(&s_session->buffer[2][s_session->size[2]]), (void *)(&s_session->buffer[0][4]), s_want_size);
                s_session->size[2] += s_want_size;
                (void)memmove((void *)(&s_session->buffer[0][0]), (void *)(&s_session->buffer[0][((size_t)2) + s_padd_size]), s_session->size[0] - (((size_t)2) + s_padd_size));
                s_session->size[0] -= ((size_t)2) + s_padd_size;
                if(s_control->options->verbose != 0) {
                    (void)fprintf(stdout, "from client (id=%d, padd_size=%lu, data_size=%lu, decrypt)\n", (int)s_session->accept_socket, (unsigned long)s_padd_size, (unsigned long)s_want_size);
                }
            }

            if(s_invalid_packet != 0)return(-1);
        }
    }
#endif   

    return(0);
}

/* from server: server->buffer[1] */
static int (mzproxy_from_server)(struct ts_mzproxy_control *s_control, struct ts_mzproxy_session *s_session)
{
    int s_recv_bytes;
    
    if(!FD_ISSET(s_session->client_socket, &s_control->fd_set_rx))return(0);
    
    s_control->events--;
    
    if((s_session->flags & def_mzproxy_session_flag_connecting) != def_mzproxy_session_flag_connecting)return(0);

    if((s_session->flags & def_mzproxy_session_flag_connected) == def_mzproxy_session_flag_none) { /* connection check */
        int s_socket_error;
#if defined(_WIN32) || defined(WIN32)
        int s_socket_error_length;
#else
        socklen_t s_socket_error_length;
#endif

        s_socket_error_length = (socklen_t)sizeof(s_socket_error);
        if(mzproxy_builtin_expect(getsockopt(s_session->client_socket, SOL_SOCKET, SO_ERROR, (void *)(&s_socket_error), &s_socket_error_length) != 0, 0)) {
            if(s_control->options->verbose != 0)(void)fprintf(stderr, "can not connect (id=%d,getsockopt)\n", (int)s_session->accept_socket);

            return(-1);
        }

        if(mzproxy_builtin_expect(s_socket_error != 0, 0)) {
            if(s_control->options->verbose != 0)(void)fprintf(stderr, "can not connect (id=%d,error)\n", (int)s_session->accept_socket);

            return(-1);
        }

        /* connected */
        if(s_control->options->verbose != 0)(void)fprintf(stdout, "connected (id=%d,[rx])\n", (int)s_session->accept_socket);
        (void)mzproxy_nonblock_socket(s_session->client_socket, 0);
        s_session->flags &= ~(def_mzproxy_session_flag_nonblock);
        s_session->data_time_stamp = s_control->time_stamp;
        s_session->flags |= def_mzproxy_session_flag_connected;
    }
    else if(s_session->size[1] < s_control->options->buffer_size) {
#if __def_mzproxy_use_telnetd__ != (0)
        if(s_control->options->use_telnetd != 0) {
            s_recv_bytes = (int)read((int)s_session->client_socket, (void *)(&s_session->buffer[1][s_session->size[1]]), s_control->options->buffer_size - s_session->size[1]);
        }
        else
#endif
        { 
#if __def_mzproxy_use_telnetd__ != (0)
            if(s_control->options->use_telnetd != 0) {
                s_recv_bytes = (int)read((int)s_session->client_socket, (void *)(&s_session->buffer[1][s_session->size[1]]), s_control->options->buffer_size - s_session->size[1]);
            }
            else
#endif
            {
#if defined(_WIN32) || defined(WIN32)     
                s_recv_bytes = recv(s_session->client_socket, (void *)(&s_session->buffer[1][s_session->size[1]]), (int)(s_control->options->buffer_size - s_session->size[1]), 0);
#else
                s_recv_bytes = recv(s_session->client_socket, (void *)(&s_session->buffer[1][s_session->size[1]]), s_control->options->buffer_size - s_session->size[1], MSG_NOSIGNAL);
#endif
            }
        }

        if((s_recv_bytes == 0) || ((s_recv_bytes == (-1)) && (errno != EINTR))) { /* disconnected by server */
            if(s_control->options->verbose != 0)(void)fprintf(stdout, "disconnected by server (id=%d, rx=%d)\n", (int)s_session->accept_socket, (int)s_recv_bytes);

            return(-1);
        }

        s_session->data_time_stamp = s_control->time_stamp;
        if(s_control->options->verbose != 0) {
            (void)fprintf(stdout, "from server (id=%d, rx=%d)\n", (int)s_session->accept_socket, (int)s_recv_bytes);
#if __def_mzproxy_rx_dump__ != (0)
            (void)mzapi_dump_inline(&s_session->buffer[1][s_session->size[1]], s_recv_bytes);
#endif   
        }

        s_session->size[1] += (size_t)s_recv_bytes;
    }

    return(0);
}

/* service function */
static int (mzproxy_do)(struct ts_mzproxy_control *s_control)
{
    struct ts_mzproxy_session *s_session, *s_next_session;
    struct timeval s_timeval = {1l, 0l};
 
    /* get time stamp */
    s_control->time_stamp = mzproxy_time_stamp();

    /* register event */
    mzproxy_register_event(s_control);
 
    /* synchronous I/O multiplexing */
    s_control->events = select(s_control->max_fd + 1, (fd_set *)(&s_control->fd_set_rx), (fd_set *)(&s_control->fd_set_tx), (fd_set *)0, (struct timeval *)(&s_timeval));
    if(mzproxy_builtin_expect(s_control->events == (-1), 0)) {
        if(s_control->options->verbose != 0)perror("select");

        return(-1);
    }
    if(s_control->options->verbose != 0)(void)fprintf(stdout, "events=%d\n", s_control->events);
    if(s_control->events == 0)return(0); /* timeout */

    /* listen event check */
    mzproxy_check_listen(s_control);

    /* event check */
    s_session = s_control->session;
    while((s_control->events > 0) && (s_session != ((struct ts_mzproxy_session *)0))) {
        s_next_session = s_session->next;
        
        if(mzproxy_to_server(s_control, s_session) == (-1)) {
            mzproxy_del_session(s_control, s_session);
            s_session = s_next_session;

            continue;
        }
        
        if(mzproxy_to_client(s_control, s_session) == (-1)) {
            mzproxy_del_session(s_control, s_session);
            s_session = s_next_session;

            continue;
        }
        
        if(mzproxy_from_client(s_control, s_session) == (-1)) {
            mzproxy_del_session(s_control, s_session);
            s_session = s_next_session;

            continue;
        }
        
        if(mzproxy_from_server(s_control, s_session) == (-1)) {
            mzproxy_del_session(s_control, s_session);
            s_session = s_next_session;

            continue;
        }

        s_session = s_next_session;
    }

    /* check remain events */
    if(s_control->options->verbose != 0) {
        if(s_control->events > 0)(void)fprintf(stdout, "\x1b[1;31mremain events=%d\x1b[0m\n", s_control->events);
    }

    return(0);
}

/* main control function */
static int (mzproxy_main_process)(struct ts_mzproxy_options *s_options)
{
    int s_result = EXIT_FAILURE, s_check;
    struct ts_mzproxy_control s_local_control, *s_control;

    /* handle mapping */
    s_control = (struct ts_mzproxy_control *)(&s_local_control);
    s_control->stop_service = (int *)(&g_mzproxy_break); /* mapping stop signal */
    s_control->options = s_options;
#if __def_mzproxy_use_telnetd__ != (0) 
    s_control->login_program_argv[0] = (char *)s_control->options->login_program;
    s_control->login_program_argv[1] = (char *)0;
#endif
    s_control->time_stamp = mzproxy_time_stamp();

    /* open listen socket */
    if(s_control->options->use_xinetd == 0) {
#if __def_mzproxy_use_inet6__ != (0)         
        if(s_control->options->use_inet6 == 0)s_control->socket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
        else s_control->socket = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP);
#else
        s_control->socket = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
#endif  
    }
    else s_control->socket = (t_mzproxy_socket)(-1);

    if(mzproxy_builtin_expect((s_control->options->use_xinetd == 0) && (s_control->socket == ((t_mzproxy_socket)(-1))), 0)) {
        if(s_control->options->verbose != 0)perror("listen socket");
        return(EXIT_FAILURE);
    }

    if(s_control->options->use_xinetd == 0) { /* standalone mode */
        int s_sa_family;
        struct sockaddr_in s_listen_socket_address;
#if __def_mzproxy_use_inet6__ != (0)  
        struct sockaddr_in6 s_listen_socket_address6;
#endif  
        struct sockaddr *s_listen_socket_address_ptr;
        socklen_t s_listen_socket_address_size;

        /* set reuse address */
        if(mzproxy_builtin_expect(setsockopt(s_control->socket, SOL_SOCKET, SO_REUSEADDR, (void *)(&gc_mzproxy_on), (socklen_t)sizeof(gc_mzproxy_on)) == (-1), 0)) {
            if(s_control->options->verbose != 0)perror("reuse socket");
        }

        /* set listen socket address */
#if __def_mzproxy_use_inet6__ != (0)  
        if(s_control->options->use_inet6 == 0) {
            s_sa_family = AF_INET;
            s_listen_socket_address.sin_family = s_sa_family;
            s_listen_socket_address.sin_addr.s_addr = htonl(INADDR_ANY);
            s_listen_socket_address.sin_port = htons(s_control->options->listen_port);
            (void)memset((void *)(&s_listen_socket_address.sin_zero[0]), 0, (size_t)sizeof(s_listen_socket_address.sin_zero));
            s_listen_socket_address_ptr = (struct sockaddr *)(&s_listen_socket_address);
            s_listen_socket_address_size = (socklen_t)sizeof(s_listen_socket_address);
        }
        else {
            s_sa_family = AF_INET6;
            (void)memset((void *)(&s_listen_socket_address6), 0, (size_t)sizeof(s_listen_socket_address6));
            s_listen_socket_address6.sin6_family = s_sa_family;
            s_listen_socket_address6.sin6_flowinfo = 0;
            s_listen_socket_address6.sin6_addr = in6addr_any;
            s_listen_socket_address6.sin6_port = htons(s_control->options->listen_port);
            s_listen_socket_address_ptr = (struct sockaddr *)(&s_listen_socket_address6);
            s_listen_socket_address_size = (socklen_t)sizeof(s_listen_socket_address6);
        }
#else
        s_sa_family = AF_INET;
        s_listen_socket_address.sin_family = s_sa_family;
        s_listen_socket_address.sin_addr.s_addr = htonl(INADDR_ANY);
        s_listen_socket_address.sin_port = htons(s_control->options->listen_port);
        (void)memset((void *)(&s_listen_socket_address.sin_zero[0]), 0, (size_t)sizeof(s_listen_socket_address.sin_zero));
        s_listen_socket_address_ptr = (struct sockaddr *)(&s_listen_socket_address);
        s_listen_socket_address_size = (socklen_t)sizeof(s_listen_socket_address);
#endif  
        
        /* apply interface to listen socket address */
        s_check = 0;
#if __def_mzproxy_use_ifaddrs__ != (0)
        if(s_control->options->interface != ((char *)0)) { /* get address of interface */
            struct ifaddrs *s_ifaddrs, *s_trace;
            
            s_check = (-1);

            if(getifaddrs((struct ifaddrs **)(&s_ifaddrs)) == 0) {
                if(s_ifaddrs != ((struct ifaddrs *)0)) {
                    s_trace = s_ifaddrs;
                    while(s_trace != ((struct ifaddrs *)0)) {
                        if((strcmp(s_control->options->interface, s_trace->ifa_name) == 0) && (s_trace->ifa_addr->sa_family == s_sa_family)) {
                            char s_address_string[ 64 ];
                            unsigned int s_address_port;

#if __def_mzproxy_use_inet6__ != (0)       
                            if(s_control->options->use_inet6 == 0) {
                                (void)inet_ntop(AF_INET, memcpy((void *)(&s_listen_socket_address.sin_addr), (void *)(&(((struct sockaddr_in *)s_trace->ifa_addr)->sin_addr)), (size_t)sizeof(s_listen_socket_address.sin_addr)), (char *)(&s_address_string[0]), sizeof(s_address_string));
                                s_address_port = ntohs(s_listen_socket_address.sin_port);
                            }
                            else {
                                (void)inet_ntop(AF_INET6, memcpy((void *)(&s_listen_socket_address6.sin6_addr), (void *)(&(((struct sockaddr_in6 *)s_trace->ifa_addr)->sin6_addr)), (size_t)sizeof(s_listen_socket_address6.sin6_addr)), (char *)(&s_address_string[0]), sizeof(s_address_string));
                                s_address_port = ntohs(s_listen_socket_address6.sin6_port);
                            }
#else
                            (void)inet_ntop(AF_INET, memcpy((void *)(&s_listen_socket_address.sin_addr), (void *)(&(((struct sockaddr_in *)s_trace->ifa_addr)->sin_addr)), (size_t)sizeof(s_listen_socket_address.sin_addr)), (char *)(&s_address_string[0]), sizeof(s_address_string));
                            s_address_port = ntohs(s_listen_socket_address.sin_port);
#endif       

                            if(s_control->options->verbose != 0)(void)fprintf(stdout, "interface %s binding (%s:%u)\n", s_control->options->interface, (char *)(&s_address_string[0]), s_address_port);

                            s_check = 0;

                            break;
                        }

                        s_trace = s_trace->ifa_next;
                    }

                    freeifaddrs(s_ifaddrs);
                }
            }
        }
#endif  
        
        if(s_check == 0) {
            if(s_control->options->use_xinetd == 0) {
                s_check = bind(s_control->socket, s_listen_socket_address_ptr, s_listen_socket_address_size);
                if(mzproxy_builtin_expect(s_check == (-1), 0)) {
                    if(s_control->options->verbose != 0)perror("bind");
                }
                else {
                    s_check = listen(s_control->socket, SOMAXCONN);
                    if(mzproxy_builtin_expect(s_check == (-1), 0)) {
                        if(s_control->options->verbose != 0)perror("listen");
                    }
                }
            }
        }
    }
    else s_check = 0;

    if(mzproxy_builtin_expect(s_check != 0, 0)) {
        if(s_control->options->verbose != 0)(void)fprintf(stderr, "not found interface\n");
    }
    else {
#if (!defined(_WIN32)) && (!defined(WIN32))
        if(s_control->options->daemonize != 0)s_check = daemon(0, 1); /* daemonize */
        else s_check = 0;
#endif  

        if(mzproxy_builtin_expect(s_check != 0, 0)) {
            if(s_control->options->verbose != 0)perror("daemon");
        }
        else {
            int s_bye = 0;

            /* install signal */
#if defined(SIGPIPE)   
            (void)signal(SIGPIPE, mzproxy_signal);
#endif
#if defined(SIGINT)
            (void)signal(SIGINT, mzproxy_signal);
#endif
#if defined(SIGQUIT)
            (void)signal(SIGQUIT, mzproxy_signal);
#endif
#if defined(SIGTERM)
            (void)signal(SIGTERM, mzproxy_signal);
#endif
#if defined(SIGHUP)
            (void)signal(SIGHUP, mzproxy_signal);
#endif   

            /* init session */
            s_control->session = (struct ts_mzproxy_session *)0;
            s_control->session_count = 0u;
            
            s_result = EXIT_SUCCESS;

            if(s_control->options->use_xinetd == 0) { /* standalone mode */
                do { /* service loop */
                    if(mzproxy_do(s_control) == (-1))s_bye = 1; /* do service */
                    else if(s_control->stop_service != ((int *)0))s_bye = *s_control->stop_service;
                }while(s_bye == 0);
            }
            else { /* xinetd mode */
                (void)mzproxy_new_session(s_control, 0); /* xinetd socket */
                do {
                    if(s_control->session == ((struct ts_mzproxy_session *)0))break;
                    if(mzproxy_do(s_control) == (-1))s_bye = 1; /* do service */
                    else if(s_control->stop_service != ((int *)0))s_bye = *s_control->stop_service;
                }while(s_bye == 0);
            }

            s_control->session = mzproxy_free_session(s_control, s_control->session, 0);
        }
    }
    
    mzproxy_closesocket(s_control->socket);

    return(s_result);
}

static void (mzproxy_get_options)(int s_argc, char **s_argv, struct ts_mzproxy_options *s_options)
{
    int s_index = 1, s_sub_options, s_few_argument = 0;

    if(mzproxy_builtin_expect(s_argc <= 1, 0)) {
        s_options->do_help = 1;
        return;
    }

    while(s_index < s_argc) {
        s_sub_options = s_argc - (s_index + 1);

        if((strcmp(s_argv[s_index], "--help") == 0) || (strcmp(s_argv[s_index], "-h") == 0))s_options->do_help = 1;
        else if((strcmp(s_argv[s_index], "--vervose") == 0) || (strcmp(s_argv[s_index], "-v") == 0))s_options->verbose = 1;
#if (!defined(_WIN32)) && (!defined(WIN32))
        else if((strcmp(s_argv[s_index], "--daemon") == 0) || (strcmp(s_argv[s_index], "-d") == 0))s_options->daemonize = 1;
#endif  
        else if((strcmp(s_argv[s_index], "--cp") == 0) || (strcmp(s_argv[s_index], "-p") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->listen_port = (int)atoi(s_argv[++s_index]);
        }
        else if((strcmp(s_argv[s_index], "--timeout") == 0) || (strcmp(s_argv[s_index], "-t") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->connect_timeout = (int)atoi(s_argv[++s_index]);
        }
        else if(strcmp(s_argv[s_index], "--dt") == 0) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->data_timeout = (int)atoi(s_argv[++s_index]);
        }
        else if((strcmp(s_argv[s_index], "--sa") == 0) || (strcmp(s_argv[s_index], "-A") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->connect_address = s_argv[++s_index];
        }
        else if((strcmp(s_argv[s_index], "--sp") == 0) || (strcmp(s_argv[s_index], "-S") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->connect_port = (int)atoi(s_argv[++s_index]);
        }
#if __def_mzproxy_use_ifaddrs__ != (0)
        else if(strcmp(s_argv[s_index], "--interface") == 0) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->interface = s_argv[++s_index];
        }
#endif  
        else if((strcmp(s_argv[s_index], "--buffer") == 0) || (strcmp(s_argv[s_index], "-b") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->buffer_size = (int)atoi(s_argv[++s_index]);
        }
        else if((strcmp(s_argv[s_index], "--session") == 0) || (strcmp(s_argv[s_index], "-s") == 0)) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->limit_session_count = (int)atoi(s_argv[++s_index]);
        }
#if __def_mzproxy_use_inet6__ != (0)             
        else if((strcmp(s_argv[s_index], "-6") == 0) || (strcmp(s_argv[s_index], "--inet6") == 0))s_options->use_inet6 = 1;
#endif       
        else if((strcmp(s_argv[s_index], "--xinetd") == 0) || (strcmp(s_argv[s_index], "-x") == 0))s_options->use_xinetd = 1;
        else if((strcmp(s_argv[s_index], "--pm") == 0) || (strcmp(s_argv[s_index], "-P") == 0))s_options->use_pm = 1;
#if __def_mzproxy_use_ief__ != (0)
        else if(strcmp(s_argv[s_index], "--ief") == 0)s_options->use_ief = 1;
        else if(strcmp(s_argv[s_index], "--ief_data_port") == 0) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->ief_data_port = (int)atoi(s_argv[++s_index]);
        }
#endif   
#if __def_mzproxy_use_telnetd__ != (0)
        else if(strcmp(s_argv[s_index], "--telnetd") == 0)s_options->use_telnetd = 1;
        else if(strcmp(s_argv[s_index], "--issue") == 0) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->issue_file = s_argv[++s_index];
        }
        else if(strcmp(s_argv[s_index], "--login") == 0) {
            if(s_sub_options < 1){ s_few_argument = 1; break; }
            s_options->login_program = s_argv[++s_index];
        }
#endif
        else s_options->do_help = 1;
        
        s_index++;
    }
    
    if(mzproxy_builtin_expect(s_few_argument != 0, 0)) {
        s_options->do_help = 1;
        (void)fprintf(stderr, "few argument \"%s\"\n", s_argv[s_index]);
    }

    /* check options */
    if(mzproxy_builtin_expect(s_options->buffer_size < ((size_t)__def_mzproxy_minimum_buffer_size__), 0))s_options->buffer_size = (size_t)__def_mzproxy_minimum_buffer_size__;
#if __def_mzproxy_use_inet6__ != (0)             
    if(s_options->verbose != 0)(void)fprintf(stdout, "verbose mode (inet%d,%s)\n", (s_options->use_inet6 == 0) ? 4 : 6, (s_options->use_xinetd == 0) ? "standalong" : "xinetd");
#else
    if(s_options->verbose != 0)(void)fprintf(stdout, "verbose mode (%s)\n", (s_options->use_xinetd == 0) ? "standalong" : "xinetd");
#endif 
    if(mzproxy_builtin_expect((s_options->use_xinetd == 0) && (s_options->listen_port <= 0), 0)) {
        (void)fprintf(stderr, "need listen port option ! (\"--cp <listen port>\")\n");
        s_options->do_help = 1;
    }
#if __def_mzproxy_use_telnetd__ != (0)
    if(s_options->use_telnetd == 0)
#endif
    {
        if(mzproxy_builtin_expect((s_options->use_pm == 0) && (s_options->connect_port <= 0), 0)) {
            (void)fprintf(stderr, "need connect port option ! (\"--sp <connect port>\")\n");
            s_options->do_help = 1;
        }
    }
}

/* main function */
int (main)(int s_argc, char **s_argv)
{
    static struct ts_mzproxy_options s_local_options = {
        "mzproxy",                     /* application name */
        0,                             /* do help */
        0,                             /* verbose */
#if (!defined(_WIN32)) && (!defined(WIN32))
        0,                             /* daemonize */
#endif
        0,                             /* listen port */
        60ul,                          /* connect timeout */
        "localhost",                   /* connect address */
        0,                             /* connect port */
        0,                             /* data timeout */
#if __def_mzproxy_use_ifaddrs__ != (0)
        (char *)0,                     /* interface */
#endif  
        (size_t)__def_mzproxy_default_buffer_size__, /* buffer size */
        0u,                            /* limit session count */
#if __def_mzproxy_use_inet6__ != (0)  
        0,                             /* use inet6 */
#endif  
        0,                             /* use xinetd */
        0,                             /* use port map */
#if __def_mzproxy_use_ief__ != (0)
        0,                             /* use ief */
        0,                             /* ief data port */
#endif
#if __def_mzproxy_use_telnetd__ != (0)
        0,                             /* use telnetd */
        "/etc/issue.net",              /* issue file */
        "/bin/sh",                     /* login program */
#endif
        0
    }, *s_options;
    int s_result;

    /* get application name */
    s_options = (struct ts_mzproxy_options *)(&s_local_options);
    if((s_argc >= 1) && (s_argv[0] != ((char *)0))) {
        s_options->application_name = strrchr(s_argv[0], '/');
        if(s_options->application_name != ((char *)0))s_options->application_name = (char *)(&s_options->application_name[1]);
        else s_options->application_name = s_argv[0];
    }
 
    /* get options */
    mzproxy_get_options(s_argc, s_argv, s_options);
 
    /* print help */
    if(mzproxy_builtin_expect(s_options->do_help != 0, 0)) {
        (void)fprintf(stdout,
            "\nmzproxy v2.0.0 build 0 (%s %s)\n"
            "Copyrights (C) JaeHyuk Cho <minzkn@minzkn.com> - All rights reserved.\n\n"
            "usage: %s [<options>]\noptions:\n"
            "\t-v, --verbose               : verbose mode (packet dump)\n"
#if (!defined(_WIN32)) && (!defined(WIN32))
            "\t-d, --daemon                : daemonize process\n"
#endif
            "\t-p, --cp=<listen port>      : listen for connections on port\n"
            "\t-A, --sa=<connect address>  : redirect address (default: \"%s\")\n"
            "\t-S, --sp=<connect port>     : redirect port\n"
            "\t-t, --timeout=<connect timeout> : connect timeout (default: %lu sec)\n"
            "\t    --dt=<data timeout>     : data timeout (default: %lu sec)\n"
#if __def_mzproxy_use_ifaddrs__ != (0)
            "\t    --interface=<interface> : specify the network interface to listen on\n"
#endif   
            "\t-b, --buffer=<buffer size>  : rx/tx fixed buffer size (default: %lu bytes x 2)\n"
            "\t-s, --session=<limit count> : session limit count (default: %u)\n"
#if __def_mzproxy_use_inet6__ != (0)   
            "\t-6, --inet6                 : use IPv6 (default: %d)\n"
#endif   
            "\t-x, --xinetd                : use xinetd mode (default: %d)\n"
            "\t-P, --pm                    : use port map data mode (default: %d)\n"
#if __def_mzproxy_use_ief__ != (0)
            "\t    --ief                   : use ief mode (default: %d)\n"
            "\t    --ief_data_port=<port>  : use ief data port (default: %d)\n"
#endif
#if __def_mzproxy_use_telnetd__ != (0)
            "\t    --telnetd               : use telnetd mode (default: %d)\n"
            "\t    --issue                 : issue file (default: %s)\n"
            "\t    --login                 : login program (default: %s)\n"
#endif
            "%s",
            __DATE__, __TIME__,
            s_options->application_name,
            s_options->connect_address,
            s_options->connect_timeout,
            s_options->data_timeout,
            (unsigned long)__def_mzproxy_default_buffer_size__,
            s_options->limit_session_count,
#if __def_mzproxy_use_inet6__ != (0)   
            s_options->use_inet6,
#endif   
            s_options->use_xinetd,
            s_options->use_pm,
#if __def_mzproxy_use_ief__ != (0)
            s_options->use_ief,
            s_options->ief_data_port,
#endif
#if __def_mzproxy_use_telnetd__ != (0)
            s_options->use_telnetd,
            s_options->issue_file,
            s_options->login_program,
#endif
            "\n"
        );
        
        return(EXIT_FAILURE);
    }

#if defined(_WIN32) || defined(WIN32)
    do {
        WSADATA s_wsadata;
        if(mzproxy_builtin_expect(WSAStartup(MAKEWORD(2, 2), &s_wsadata) != 0, 0)) {
            (void)fprintf(stderr, "WSAStartup failed");

            return(EXIT_FAILURE);
        }
    }while(0);
#endif 

    /* do work */
    s_result = mzproxy_main_process(s_options);
 
#if defined(_WIN32) || defined(WIN32)
    (void)WSACleanup();
#endif

    return(s_result);
}

/* vim: set expandtab: */
/* End of source */
