/**

@file	MySQLib.cpp

@note
	There is a good library ( which is on STL base API) for the MySQL named 'MySQL++'.
	It is licesed as 'LGPL' and I am not sure what affects the 'LGPL' to the whole 
	software when it has been linked it to my app as statically.
	Check it, and if it is okay, use the library instead of making my own mysql library.

@note
	In some case, like when the MySQL DB server is running in another host,
	the connection lost error occurs between clint/server (well, sometimes).
	But guess, it's okay ...
	the mysql_real_connect() function sets reconnect flag to 1 (by defualt), which means,
		when connection is lost, it retries to connect before give up, so the query
		would be ( hopefully ) committed eventually, ...

@todo
	- mysql_num_rows()

@example

  --------------------- connection ------------------------
  MySQLib my, my2;
	char *q;
	my.Connect("mail.mywebmail.co.kr", "mark", "1111", "mark");
	my2.Connect("mail.mywebmail.co.kr", "mark", "1111", "mark");
	my.Close();
	my.Connect("mail.mywebmail.co.kr", "mark", "1111", "mark");
	
  --------------------- fetch rows -------------------------
	q = "select * from table1";
	my.Query(q);
	while (my.NextRow() != MYSQL_EOF) {
		printf("%s : %s\r\n", my.Field(0), my.Field(1));
	}

*/

#include "MySQLib.h"
#include <stdio.h>
#include <string.h>
#include <stdarg.h>



/**
	Initializer
*/
MySQLib::MySQLib()
{
	m_pConn			= NULL;
	m_bConnected	= false;
	m_pResult		= NULL;
	m_Row			= NULL;
	m_nNumFields	= 0;
	m_bDebug		= false;
	m_bCloned		= false;
	m_bQueried=false;
}

/*
	Destructor
*/
MySQLib::~MySQLib()
{
	// free result if any
	if (m_pResult) {
		FreeResult();
	}
	if (m_bConnected) Close();
}

void MySQLib::DebugPrint(bool flag)
{
	m_bDebug=flag;
}

/**

Clone() & Attach()

  >>>>>>>>>>>>>>>>>>>>>>>> seems to work fine <<<<<<<<<<<<<<<<<<<<<<<


  These two functions make a new object of the class
		with the same MySQL Link Identifier.
	It will be useful when you need to query many times preserving the old result set.

	Calling 'Close()' by cloned object, will return false.
	Closing the connection will only take effect for the original object.

  WARNING	:
  What if the each cloned object will query at the same time (in many processes)?
	The MySQL link will allow this?
	Not sure.
	So, do use cloned object in process until you get the answer.

  EX)
  MySQLib my, re;
  if (! my.Connect(db_host, db_user, db_password, db_name)) { error; }
  re.Attach(my.Clone());
  my.Query(q.GetString());
  tmp = re.Result(q.GetString());
  my.Close();

*/
MYSQL *MySQLib::Clone() {
	return m_pConn;
}
bool MySQLib::Attach(MYSQL *pConn)
{
	if (pConn == NULL) return false;
	m_pConn = pConn;
	m_bCloned=true;
	return true;
}



/**

	It initializes the mysql connection object and connects to the mysql server.

@param user_name
	the user name of the database
@param password
	the password of the user
@param dbname
	database file name of the database
@param port
	port
	
	
@return
	TRUE if the connection has been established.
	FALSE if not.

@example
	MySQLib my;
	my.Connect("mail.mywebmail.co.kr", "mark", "1111", "mark");
*/
bool MySQLib::Connect(const char *host, const char *user_name, const char*password, const char *dbname, unsigned int port)
{

	// close before re-connect.
	if (m_bConnected) { Error(ALREADY_CONNECTED); return false; }
	if (host == NULL || user_name == NULL || password == NULL || dbname == NULL) return false;

	dbg("calling mysql_init () ...");
	if (!m_pConn) m_pConn = mysql_init(NULL);
	dbg("mysql_init ok ...");
	if (!m_pConn) {
		Error(MYSQL_INIT);
		return false;
	}
	strcpy(m_cHost, host);
	strcpy(m_cUserName, user_name);
	strcpy(m_cPassword, password);
	strcpy(m_cDBName, dbname);
	m_nPort = port;


	// init
	// CLIENT_FOUND_ROW flag means, that it returns exact number of the affected rows
	// by mysql*_query() functions.
	dbg("calling mysql_real_connect (%s, %s, %s, %d)...", m_cHost, m_cUserName, m_cPassword, m_nPort);

	m_pConn = mysql_real_connect(m_pConn, m_cHost, m_cUserName, m_cPassword,
		m_cDBName, m_nPort, NULL, CLIENT_FOUND_ROWS);
	dbg("finished calling mysql_real_connect () ...");

	if (!m_pConn) { // connection failed
		dbg("mysql_real_connect failed...");
		Error(MYSQL_REAL_CONNECT);
		return false;
	}
	else {	// connect success
		dbg("mysql_real_connect ok ...");
		m_bConnected = true;
	}
	return true;
}

/*
	Check if the object has a proper MySQL link idendifier.
@RETURN
	true if the connection has been established.
	false if not.
*/
bool MySQLib::IsConnected()
{
	return m_bConnected;
}


/**

	void Error(int nErrType)

@param
	Error Number
@return
	It puts error message into member variable - m_cErrMsg.

@warning
	m_cErrMsg has 1024 bytes buffer for holding error message.
*/
void MySQLib::Error(int nErrType)
{

	strcpy(m_cErrMsg, "ERROR: ");
	switch (nErrType)
	{
	case MYSQL_INIT:
		strcat(m_cErrMsg, "cannot mysql_init()\r\n");
		break;
	case MYSQL_REAL_CONNECT:
		strcat(m_cErrMsg, "cannot mysql_real_connect()\r\n");
		break;
	case ALREADY_CONNECTED:
		strcat(m_cErrMsg, "already connected\r\n");
		return;
	case ALREADY_CLOSED:
		strcat(m_cErrMsg, "already closed\r\n");
		return;
	case MYSQL_QUERY:
		strcat(m_cErrMsg, "cannot mysql_query()\r\n");
		break;
	case MYSQL_CONNECTION:
		strcat(m_cErrMsg, "something wrong with the connection\r\n");
		break;
	default :
		if (m_pConn == NULL) {
			strcat(m_cErrMsg, "connection has not established\r\n");
			return;
		}
	}

	if (m_pConn && mysql_errno(m_pConn)) {

		char *err = mysql_error(m_pConn);
		// Error message's length check. if it is greater than 1024 - 256,
		// just print warning.
		if (strlen(err) > 1024 - 256) { err = "Error message is too long. cannot hold it"; }
		char pErrno[100];
		sprintf(pErrno, " errno=%d ", mysql_errno(m_pConn));
		strcat(m_cErrMsg, pErrno);
		strcat(m_cErrMsg, err);
	}
	dbg(m_cErrMsg);
}


/**

	Close the connection
	After call this function, the object could re-connect to other server.
	
@return
	true if the connection closed successfully.
	false, otherwise.
	
*/
bool MySQLib::Close()
{

	if (m_bConnected == false) {
		Error(ALREADY_CLOSED);
		return false;
	}
	m_bConnected = false;

	if (m_bCloned == false) {
		mysql_close(m_pConn);
	}
	m_pConn = NULL;
	FreeResult();
	m_cErrMsg[0] = '\0';
	return true;
}





/**
	Query

@param	const char *query
	a query string without ending ; or \g
@return
	same as LongQuery()

@example
	q = "insert into tab values ('a1a', 'thename)";
	if (my.Query(q)) {
		printf("Affected: %lu\r\n", my.GetAffectedRows());
	}
	else {
		printf(my.GetErrorMessage());
	}

@example 1
	q = "select * from tab";
	my.Query(q);
	while (my.NextRow() != MYSQL_EOF) {
		printf("%s : %s\r\n", my.Field(0), my.Field(1));
	}
@example 2
	bool retvar;
	char *account="ABc", *domain="deF";

	retvar = _my.Query("SELECT idx, quota_max, quota_curr FROM account WHERE account='%s' AND domain='%s'", account, domain);
	if ( ! retvar ) {
		_error_fatal( "db query error" );
	}

	
	if ( _my.NextRow() == MYSQL_EOF ) 
		// ڵ尡  
		return 1;

	ui->idx			= atoi( _my.Field( 0 ) );
	ui->quota_curr	= atoi( _my.Field( 1 ) );
	ui->quota_max	= atoi( _my.Field( 2 ) );

*/
bool MySQLib::Query( char *format, ... )
{
	
	char query[1024];
	int status;
	va_list arglist;

	va_start(arglist, format);
	status = vsprintf( query, format, arglist );
	if(strlen( query ) + 1 > sizeof( query ) || status == EOF)
		printf("The query is too long or invalid.");
	va_end(arglist);

	return LongQuery( query );
}
/**

  Query for a long statement
@note
	The input query string is not formatted char string like 'Query()',
	so it does not have the limitation of the input buffer size.

@return
	TRUE if the query has been executed successfully.
	FALSE otherwise.
*/

bool MySQLib::LongQuery( const char *query )
{
	
	// check the connection.
	if (m_pConn == NULL) {Error(MYSQL_CONNECTION); return false;}
	FreeResult();
	//dbg((char *)query);
	m_nQueryRes = mysql_query(m_pConn, query);
	if (m_nQueryRes) {
		Error(MYSQL_QUERY);
		printf("m_nQueryRes=%d ", m_nQueryRes);
		printf("QUERY=%s\r\n", query);
		return false;
	}
	m_bQueried=true;
	return true;
}


/**
bool Query(char *query, unsigned long len)

DESC	:
	The defference from Query(char *) is that this function uses the length of the
		query string to query.
		That is, the query string can contain NULL character while Query(char *) can not.
		So, use this function when you need to deal with binary data.
INPUT	:
	query	- a query string
	len		- the length of the query string.
RETURN	: same as Query(char *)
NOTE	: same as Query(char *)

*/
bool MySQLib::Query(const char *query, unsigned long len)
{
	// check the connection.
	if (m_pConn == NULL) {Error(MYSQL_CONNECTION); return false;}
	FreeResult();
	m_nQueryRes = mysql_real_query(m_pConn, query, len);
	if (m_nQueryRes) { Error(MYSQL_QUERY); return false; }
	m_bQueried=true;
	return true;
}

/*

	It queries to MySQL server, but does not take any effect for the current result set.
	So, when you have a result set, and need to queary again, but you want to keep the result as it is,
	then call this function.
	This function has nothing to do with the result set.

*/
bool MySQLib::Execute(const char *query) {
	if ( mysql_query(m_pConn, query) != 0 ) {
	//	printf("EXECUTE ERROR: \r\n");
		Error(MYSQL_QUERY);
		return false;
	}

	//	printf("EXECUTE SUECCESS: \r\n");
	return true;
}


/*

	Escapes the input query string.
@important
	This does NOT use mysql_escape_string() or mysql_real_escape_string(),
	because mysql(_real)_escape_string() seems to allocate memory too much(length*2+1).

	MySQL needs only back-slash(\) and single-quote(') to be escaped.

@important
@todo
	Change the sheme to use mysql_real_escape_string()
	
@important
	This is * NOT * binary compatible!

@return
	An 'new'ed pointer of the input query.
	The return pointer must be 'delete'ed.
	
@example
	char *pEsc = mySQL.Escape(m_spData);
	q.Format("INSERT INTO MY_RECV_QUEUE (fr_idx, sender, stamp, data) values (%d, '%s', %d, '%s')",
		nUserIdx, m_spFrom, t, pEsc);
	_delete(pEsc);
*/
char *MySQLib::Escape(const char *query) {


	if (query == NULL) return NULL;

	int nQueryLen=0, nNewLen=0;
	nQueryLen = strlen(query);
	if (nQueryLen < 1024) {
		nNewLen = nQueryLen + 254;
	}
	else {
		nNewLen = nQueryLen + (nQueryLen / 5);
	}
	char *ptr = new char [nNewLen];
	char *ret = ptr;
	while (*query) {
		if (*query == '\'') {
			*ptr = '\\';
			ptr ++;
		}
		*ptr = *query;
		query ++;
		ptr ++;
	}
	*ptr = '\0';
	return ret;
}


/**

	Returns the last insert id
@note
	This Returns the ID(index key) generated for an AUTO_INCREMENT column by the previous query.
	Use this function after you have performed an INSERT query into a table
		that contains an AUTO_INCREMENT field.
@note
	mysql_insert_id() returns 0 if the previous query does not generate 
		an AUTO_INCREMENT value.
		If you need to get the value for later use, be sure to call mysql_insert_id() 
		immediately after the query that generates the value.

*/
unsigned long MySQLib::GetInsertId() {
	return (unsigned long )mysql_insert_id(m_pConn);
}


/**
	returns how many rows have been affected by the previous query.
*/
unsigned long MySQLib::GetAffectedRows()
{
	return (unsigned long) mysql_affected_rows(m_pConn);
}



/**
	returns the error message generated by the object.
*/
char *MySQLib::GetErrorMessage()
{
	return m_cErrMsg;
}




/**

	
	After query using 'SELECT' statement, the object initializes the result set.
	NextRow() gets the next row of the previous 'SELECT' query.
	
@return
	MYSQL_BOF : this is the first row.
	MYSQL_EOF : there are no more rows left to search in the result set, 
		so quit the process of the query.
	MYSQL_FAILED if it fails.
	MYSQL_OK  if it success.
	
NOTE:
*/
int MySQLib::NextRow()
{
	bool bFirst=false;
	//if (m_Row == NULL) {
	if (m_bQueried) {
		m_bQueried=false;
		FreeResult();
		if (m_pConn == NULL) return MYSQL_FAILED;
		m_pResult = mysql_store_result (m_pConn);
		bFirst = true;
	}
	// check if mysql_use_result error
	if (! m_pResult) return MYSQL_FAILED;

	// get next row
	m_Row = mysql_fetch_row(m_pResult);

	// if no more rows left in the result set, or an error occurrs.
	if (m_Row == NULL) {
		return MYSQL_EOF;
	}

	// if just get a result set,
	if (bFirst == true) {
		// get the fields num. it'll be used in Field()
		m_nNumFields = mysql_num_fields(m_pResult);
		return MYSQL_BOF;
	}

	// otherwise,
	return MYSQL_OK;
}

/**

@param int n
	the number of the field
@note
	the return pointer must NOT be freed.
@return
	NULL, if there is no rows in the result set.
		check the query order; Query(), NextRow(), Field()
	NULL, if the number of the field is greater than or equal to the m_nNumFields
		which holds the real max flied number of the result set's row.
		
	if success, a data string pointer of the field.
*/
char *MySQLib::Field(int n) {
	return f(n);
}

char *MySQLib::f( int n ) {
	char *ptr=NULL;
	if (m_Row == NULL) return NULL;
	if (n >= m_nNumFields) return NULL;
	ptr = m_Row[n];
	return ptr;
}


/**
	
	This function returns the first field of the first record in the result set created by the query.
	
@param
	a query string
@return
	it returns the first field's data pointer of the first row in the result set
	of the query.
	NULL if the query is wrong.
@warning
	It does affect the previous result set.
	So, if you don't want to this, use Clone() & Attach()

@example
	char *res = mySQL.Result(q);
	if (res == NULL) m_nSpaceLeft=0;
	else m_nSpaceLeft = atoi(res);

	q = "select name from tab where idx=3";
	printf("re=%s\r\n", my.Result(q));
*/
char *MySQLib::Result( char *format, ... ) {

	
	char query[1024];
	int status;
	va_list arglist;

	va_start(arglist, format);
	status = vsprintf( query, format, arglist );
	if(strlen( query ) + 1 > sizeof( query ) || status == EOF)
		printf("The query is too long or invalid.");
	va_end(arglist);

	Query(query);
	NextRow();
	return Field(0);
}


/**
	it frees the result set created by previous mysql*_result
*/
void MySQLib::FreeResult() {
	if (m_pResult) {
		mysql_free_result(m_pResult);
		m_pResult		= NULL;
		m_Row			= NULL;
		m_nNumFields	= 0;
	}
	sprintf(m_cErrMsg,"%s", "No Error: Result set freed");
}

/**
	debug message print
*/
void MySQLib::dbg(char *format, ...) {

	char query[1024];
	int status;
	va_list arglist;

	va_start(arglist, format);
	status = vsprintf( query, format, arglist );
	if(strlen( query ) + 1 > sizeof( query ) || status == EOF)
		printf("The query is too long or invalid.");
	va_end(arglist);



	if (m_bDebug==true) {
		if (m_bCloned) {
			printf("In Cloned Link:: ");
		}
		printf("MySQL Class Debug Message: %s\r\n", query);
	}
}
