package net.kldp.j2ee.kupload;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/**
 * MappedDiskFile의 입력 스트림을 제공한다.
 * 
 */
final class UploadFileInputStream extends InputStream {

	private ByteBuffer buffer;
	private int position;
	private int size;
	private int mark = 0;

	/**
	 * ByteBuffer를 이용하여 스트림을 생성한다.
	 * 
	 * @param buffer
	 * @param position
	 * @param size
	 */
	public UploadFileInputStream(ByteBuffer buffer, int position, int size) {
		if (position + size > buffer.capacity())
			throw new IllegalArgumentException("Size is large than buffer capacity.");
		this.buffer = buffer;
		this.position = position;
		this.size = size;
	}

	@Override
	public int read() throws IOException {
		if (mark == size)
			return -1;
		mark(mark + 1);
		return buffer.getInt(position + mark - 1);
	}

	@Override
	public int available() throws IOException {
		return size - mark;
	}

	@Override
	public long skip(long n) throws IOException {
		int nn = (int) n;
		if (mark + nn > size) {
			int a = size - mark;
			mark(size);
			return a;
		} else {
			mark(mark + nn);
			return n;
		}
	}

	@Override
	public synchronized void mark(int mark) {
		this.mark = mark;
	}

	@Override
	public boolean markSupported() {
		return true;
	}

	@Override
	public int read(byte[] dst) throws IOException {
		int capacity = size - mark;
		if (capacity < 1) {
			return -1;
		} else if (capacity < dst.length) {
			buffer.position(position + mark);
			buffer.get(dst, 0, capacity);
			mark(mark + capacity);
			return capacity;
		} else {
			buffer.position(position + mark);
			buffer.get(dst, 0, dst.length);
			mark(mark + dst.length);
			return dst.length;
		}
	}

	@Override
	public int read(byte[] dst, int offset, int length) throws IOException {
		int capacity = size - mark;
		if (capacity < 1) {
			return -1;
		}
		if (capacity < length) {
			buffer.position(position + mark);
			buffer.get(dst, offset, capacity);
			mark(mark + capacity);
			return capacity;
		} else {
			buffer.position(position + mark);
			buffer.get(dst, offset, length);
			mark(mark + dst.length);
			return length;
		}
	}

	@Override
	public void close() throws IOException {
		super.close();
	}

	@Override
	public synchronized void reset() throws IOException {
		buffer.position(position);
		mark(0);
		super.reset();
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		buffer = null;
	}
}