package net.kldp.j2ee.kupload;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/**
 * 업로드된 스트림을 메모리 매핑 파일로 보관하는 클래스.
 * 
 */
class MappedDiskFile implements FormFile {
	private String contentType = "";
	private String filePath = "";
	private int start;
	private int size;
	private ByteBuffer buffer;
	private String fieldName = "";

	/**
	 * 바이트 매퍼에 매핑된 FormFile파일을 생성한다.
	 * 
	 * @param buffer
	 * @param start
	 * @param end
	 */
	protected MappedDiskFile(ByteBuffer buffer, int start, int end) {
		this.buffer = buffer;
		this.start = start;
		this.size = end - start + 1;
	}

	public void destroy() {
		buffer = null;
	}

	public String getContentType() {
		return contentType;
	}

	public synchronized byte[] getFileData() throws IOException {
		buffer.position(start);
		byte[] data = new byte[size];
		buffer.get(data, 0, size);
		return data;
	}

	/**
	 * FilePath를 기초로 파일 이름을 리턴한다.
	 */
	public String getFileName() {
		int pos = filePath.lastIndexOf('/');
		if (pos != -1)
			return filePath.substring(pos + 1, filePath.length());
		pos = filePath.lastIndexOf('\\');
		if (pos != -1)
			return filePath.substring(pos + 1, filePath.length());
		else
			return filePath;
	}

	/**
	 * 파일 사이즈를 리턴한다.
	 */
	public int getFileSize() {
		return size;
	}

	/**
	 * MappedDiskFile의 입력스트림을 리턴한다.
	 */
	public synchronized InputStream getInputStream() throws IOException {
		return new UploadFileInputStream(buffer, start, size);
	}

	/**
	 * 파일의 컨텐츠 타입을 설정한다.
	 * 
	 * @param contentType
	 */
	public void setContentType(String contentType) {
		this.contentType = contentType;
	}

	/**
	 * 파일 확장자를 리턴한다.
	 */
	public String getFileSuffix() {
		String fileName = getFileName();
		if (fileName == null)
			return null;
		String value = fileName.substring(fileName.lastIndexOf('.') + 1, fileName.length());
		if (fileName.lastIndexOf('.') > 0)
			return value;
		else
			return "";
	}

	/**
	 * 파일 경로를 리턴한다.
	 * 
	 * @param filePath
	 */
	public void setFilePath(String filePath) {
		this.filePath = filePath;
	}

	/**
	 * Form의 필드 이름을 리턴한다.
	 */
	public String getFieldName() {
		return fieldName;
	}

	/**
	 * Form의 필드 이름을 설정한다.
	 * 
	 * @param fieldName
	 */
	public void setFieldName(String fieldName) {
		this.fieldName = fieldName;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj == this)
			return true;
		if (obj instanceof MappedDiskFile && hashCode() == obj.hashCode()) {
			MappedDiskFile file = (MappedDiskFile) obj;
			if (fieldName.equals(file.fieldName) && filePath.equals(file.filePath))
				return true;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return 27 * start * size * fieldName.length() + 29;
	}

	@Override
	protected void finalize() throws Throwable {
		super.finalize();
		buffer = null;
	}
}