/**
 * Zip 파일에 대한 정보를 담고 있는 클래스
 */
package net.kldp.jzip;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.zip.ZipException;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;

/**
 * Zip 파일 대한 정보를 담고 있는 클래스
 * 
 * @author jeongseungwon
 * 
 */
public class Zip {
	/**
	 * 새로운(빈) Zip 파일을 생성하는 메소드
	 * 
	 * @param saveFile
	 *            새로 생성할 {@link File}
	 */
	public static void createNew(File saveFile) {
		try {
			ZipOutputStream zos = new ZipOutputStream(saveFile);

			try {
				// 인코딩 설정
				zos.setEncoding("MS949");
			} finally {
				zos.finish();
			}
		} catch (IOException e1) {
			e1.printStackTrace();
		}
	}

	/**
	 * 파일 크기를 문자열로 반환하는 메소드
	 * 
	 * @param size
	 *            파일 크기
	 * @return 파일 크기를 표현하는 문자열
	 */
	public static String getSizeString(long size) {
		if (size <= -1) {
			return "알 수 없음";
		} else {
			final int KB = 1024;
			final int MB = 1024 * KB;
			final int GB = 1024 * MB;

			double newSize = size;
			NumberFormat nf = NumberFormat.getInstance();
			nf.setMaximumFractionDigits(2);

			if (size >= GB) {
				newSize /= GB;
				return nf.format(newSize) + " GB";
			} else if (size >= MB) {
				newSize /= MB;
				return nf.format(newSize) + " MB";
			} else if (size >= KB) {
				newSize /= KB;
				return nf.format(newSize) + " KB";
			} else {
				return size + " 바이트";
			}
		}
	}

	/**
	 * 시간을 문자열로 반환하는 메소드
	 * 
	 * @param time
	 *            시간
	 * @return 시간을 표현하는 문자열
	 */
	public static String getTimeString(long time, int dateFormat) {
		if (time == -1) {
			return "알 수 없음";
		} else {
			Date date = new Date(time);
			DateFormat df = DateFormat.getDateTimeInstance(dateFormat,
					dateFormat);
			return df.format(date);
		}
	}

	private File file; // Zip 파일

	private ZipFile zipFile; // ZipFile 객체

	private ArrayList<ZipEntry> entryList; // ZipEntry의 ArrayList

	private String path; // Zip 파일 내의 경로

	private int originalSize; // Zip 파일 내 항목의 숫자

	private final int BUFFER_SIZE = 64 * 1024; // 버퍼 크기 : 64 KB

	/**
	 * {@link Zip} 클래스의 생성자
	 * 
	 * @param file
	 *            {@link File}
	 * @param zipFile
	 *            {@link ZipFile}
	 * @param dir
	 *            디렉토리로 보기 여부
	 */
	public Zip(File file, ZipFile zipFile, boolean dir) {
		this.file = file;
		this.zipFile = zipFile;

		if (dir) {
			// 디렉토리로 보기
			path = "";
		} else {
			// 모든 파일로 보기
			path = null;
		}

		loadEntries();
	}

	/**
	 * Zip 파일에 디렉토리를 더하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param directory
	 *            더할 디렉토리 {@link File}
	 * @param tempDir
	 *            임시 디렉토리
	 */
	public void addDir(final Shell shell, final File directory,
			final File tempDir) {
		// FIXME 같은 이름을 가진 디렉토리를 처리함

		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {

				final String encoding = zipFile.getEncoding(); // 인코딩

				if (!tempDir.exists()) {
					tempDir.mkdirs();
				}

				File tempFile = new File(tempDir, file.getName());

				try {
					FileInputStream fis = new FileInputStream(file);
					FileOutputStream fos = new FileOutputStream(tempFile);

					try {
						byte[] buffer = new byte[BUFFER_SIZE];
						int read = 0;
						read = fis.read(buffer);
						while (read != -1) {
							fos.write(buffer, 0, read);
							read = fis.read(buffer);
						}
					} finally {
						fis.close();
						fos.close();
					}
				} catch (FileNotFoundException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}

				// 임시 ZipFile 객체 생성
				ZipFile tempZipFile = null;
				try {
					tempZipFile = new ZipFile(tempFile, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				// 새로운 Zip 파일 생성
				ZipOutputStream zos = null;
				try {
					zos = new ZipOutputStream(file);

					try {
						zos.setEncoding(encoding);

						Enumeration entries = tempZipFile.getEntries();

						while (entries.hasMoreElements()) {
							ZipEntry originalEntry = (ZipEntry) entries
									.nextElement();

							ZipEntry entry = new ZipEntry(originalEntry
									.getName());

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = tempZipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}

						// 디렉토리 더하기
						ZipEntry entry = new ZipEntry(directory.getName() + "/");
						entry.setTime(directory.lastModified());

						zos.putNextEntry(entry);

						zos.closeEntry();

						addFileNDir(shell, zos, directory.listFiles(),
								directory.getName());

					} finally {
						zos.finish();
					}
				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					zipFile = new ZipFile(file, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				loadEntries();
			}

		});
	}

	/**
	 * Zip 파일에 파일을 더하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param filePaths
	 *            더할 파일들의 경로
	 * @param tempDir
	 *            임시 디렉토리 {@link File}
	 */
	public void addFile(final Shell shell, final String[] filePaths,
			final File tempDir) {
		// FIXME 같은 이름을 가진 파일을 처리함

		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {
				File[] files = new File[filePaths.length];

				for (int i = 0; i < files.length; i++) {
					files[i] = new File(filePaths[i]);
				}

				final String encoding = zipFile.getEncoding(); // 인코딩

				if (!tempDir.exists()) {
					tempDir.mkdirs();
				}

				File tempFile = new File(tempDir, file.getName());

				try {
					FileInputStream fis = new FileInputStream(file);
					FileOutputStream fos = new FileOutputStream(tempFile);

					try {
						byte[] buffer = new byte[BUFFER_SIZE];
						int read = 0;
						read = fis.read(buffer);
						while (read != -1) {
							fos.write(buffer, 0, read);
							read = fis.read(buffer);
						}
					} finally {
						fis.close();
						fos.close();
					}
				} catch (FileNotFoundException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}

				// 임시 ZipFile 객체 생성
				ZipFile tempZipFile = null;
				try {
					tempZipFile = new ZipFile(tempFile, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				// 새로운 Zip 파일 생성
				ZipOutputStream zos = null;
				try {
					zos = new ZipOutputStream(file);

					try {
						zos.setEncoding(encoding);

						Enumeration entries = tempZipFile.getEntries();

						while (entries.hasMoreElements()) {
							ZipEntry originalEntry = (ZipEntry) entries
									.nextElement();

							ZipEntry entry = new ZipEntry(originalEntry
									.getName());

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = tempZipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}

						addFileNDir(shell, zos, files, null);
					} finally {
						zos.finish();
					}
				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					zipFile = new ZipFile(file, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				loadEntries();
			}

		});

	}

	/**
	 * {@link ZipOutputStream}에 파일을 추가하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param zos
	 *            {@link ZipOutputStream}
	 * @param files
	 *            더할 파일들
	 * @param parent
	 *            부모 엔트리 이름
	 */
	private void addFileNDir(Shell shell, ZipOutputStream zos, File[] files,
			String parent) {
		for (File file : files) {
			if (!file.exists()) {
				MessageBox messageBox = new MessageBox(shell, SWT.OK
						| SWT.ICON_ERROR);
				messageBox.setText("파일 또는 디렉토리 더하기 실패!");
				messageBox.setMessage(file.getPath()
						+ " 파일 또는 디렉토리가 존재하지 않습니다.");
				messageBox.open();

				continue;
			}

			if (!file.canRead()) {
				MessageBox messageBox = new MessageBox(shell, SWT.OK
						| SWT.ICON_ERROR);
				messageBox.setText("파일 또는 디렉토리 더하기 실패!");
				messageBox.setMessage(file.getPath()
						+ " 파일 또는 디렉토리에 대한 읽기 권한이 없습니다.");
				messageBox.open();

				continue;
			}

			if (file.isDirectory()) {
				// 디렉토리인 경우
				ZipEntry entry = null;
				if (parent == null) {
					entry = new ZipEntry(file.getName() + "/");
				} else {
					entry = new ZipEntry(parent + "/" + file.getName() + "/");
				}

				entry.setTime(file.lastModified());

				try {
					zos.putNextEntry(entry);

					zos.closeEntry();
				} catch (IOException e) {
					e.printStackTrace();
				}

				String parentName = file.getName();
				if (parent != null) {
					parentName = parent + "/" + parentName;
				}

				addFileNDir(shell, zos, file.listFiles(), parentName);
			} else {
				// 파일인 경우
				ZipEntry entry = null;
				if (parent == null) {
					entry = new ZipEntry(file.getName());
				} else {
					entry = new ZipEntry(parent + "/" + file.getName());
				}
				entry.setTime(file.lastModified());

				try {
					zos.putNextEntry(entry);

					FileInputStream fis = new FileInputStream(file);

					byte[] buffer = new byte[BUFFER_SIZE];
					int read = 0;

					read = fis.read(buffer);
					while (read != -1) {
						zos.write(buffer, 0, read);
						read = fis.read(buffer);
					}

					fis.close();

					zos.closeEntry();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * Zip 파일에 대한 쓰기 권한이 있는지의 여부를 확인하는 메소드
	 * 
	 * @return Zip 파일에 대한 쓰기 권한 여부
	 */
	public boolean canWrite() {
		return file.canWrite();
	}

	/**
	 * 테이블의 인덱스를 {@link ZipEntry} {@link ArrayList}의 인덱스로 바꿔주는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @return {@link ZipEntry} {@link ArrayList}의 인덱스
	 */
	private int convertIndex(int index) {
		int newIndex = 0;

		for (ZipEntry entry : entryList) {
			if (getParentEntryName(entry).equals(path)) {
				if (index == 0) {
					break;
				} else {
					index--;
				}
			}

			newIndex++;
		}

		return newIndex;
	}

	/**
	 * {@link ZipEntry}를 삭제하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param indices
	 *            테이블의 인덱스
	 * @param tmpDir
	 *            임시 디렉토리
	 */
	public void delete(Shell shell, final int[] indices, final File tmpDir) {
		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {

				ArrayList<Integer> indexList = new ArrayList<Integer>();

				if (path != null) {
					for (int i = 0; i < indices.length; i++) {
						indices[i] = convertIndex(indices[i]);
					}

					for (int index : indices) {
						ZipEntry dirEntry = entryList.get(index);

						if (dirEntry.isDirectory()) {
							// 디렉토리인 경우
							// 하위 디렉토리의 엔트리 추가
							final String dirName = dirEntry.getName();

							for (ZipEntry entry : entryList) {
								String entryPath = getParentEntryName(entry);

								while (!entryPath.equals("")) {
									if (entryPath.equals(dirName)) {
										indexList.add(entryList.indexOf(entry));
										break;
									}

									entryPath = getParentEntryName(entryPath);
								}
							}
						}
					}
				}

				for (int i = 0; i < indices.length; i++) {
					indexList.add(indices[i]);
				}

				if (!tmpDir.exists()) {
					tmpDir.mkdirs();
				}

				File tempFile = new File(tmpDir, file.getName());

				final String encoding = zipFile.getEncoding(); // 인코딩

				try {
					ZipOutputStream zos = new ZipOutputStream(tempFile);

					try {
						zos.setEncoding(encoding);

						for (int i = 0; i < entryList.size(); i++) {
							if (indexList.contains(i)) {
								continue;
							}

							ZipEntry originalEntry = entryList.get(i);

							ZipEntry entry = new ZipEntry(originalEntry
									.getName());

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = zipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}
					} finally {
						zos.finish();
					}

				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					FileInputStream fis = new FileInputStream(tempFile);
					FileOutputStream fos = new FileOutputStream(file);

					try {
						byte[] buffer = new byte[BUFFER_SIZE];
						int read = 0;
						read = fis.read(buffer);
						while (read != -1) {
							fos.write(buffer, 0, read);
							read = fis.read(buffer);
						}
					} finally {
						fis.close();
						fos.close();
					}
				} catch (FileNotFoundException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					zipFile = new ZipFile(file, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				loadEntries();
			}

		});
	}

	/**
	 * 선택된 항목을 임시 디렉토리에 압축 해제하는 메소드
	 * 
	 * @param index
	 *            선택된 테이블의 인덱스
	 * @param tmpDir
	 *            임시 디렉토리 {@link File}
	 * @return 임시 디렉토리에 압축 해제된 파일 경로
	 */
	public String extract(int index, File tmpDir) {
		if (path != null) {
			index = convertIndex(index);
		}

		if (!tmpDir.exists()) {
			tmpDir.mkdirs();
		}

		ZipEntry entry = entryList.get(index);
		File file = new File(tmpDir, getEntryName(entry));

		if (entry.isDirectory()) {
			// 디렉토리인 경우
			if (file.isFile()) {
				// 같은 이름의 파일이 이미 존재하는 경우
				file.delete();
			}

			file.mkdirs();

			if (path != null) {
				// 디렉토리로 보기
				// 모든 하위 디렉토리와 파일까지 압축 해제
				for (ZipEntry zipEntry : entryList) {
					String parent = getParentEntryName(zipEntry);

					while (!parent.equals("")) {
						if (parent.equals(entry.getName())) {
							File entryFile = new File(file, zipEntry.getName()
									.substring(parent.length()));

							if (zipEntry.isDirectory()) {
								// 디렉토리인 경우
								if (entryFile.isFile()) {
									// 같은 이름의 파일이 이미 존재하는 경우
									entryFile.delete();
								}

								entryFile.mkdirs();
							} else {
								// 파일인 경우
								if (entryFile.isDirectory()) {
									// 같은 이름의 디렉토리가 이미 존재하는 경우
									JZip.deleteDir(entryFile);
								}

								extractFile(entryFile, zipEntry);
							}

							break;
						}

						parent = getParentEntryName(parent);
					}
				}
			}
		} else {
			// 파일인 경우
			if (file.isDirectory()) {
				// 같은 이름의 디렉토리가 이미 존재하는 경우
				// 모든 하위 디렉토리와 파일까지 삭제
				JZip.deleteDir(file);
			}

			extractFile(file, entry);
		}

		return file.getPath();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 압축을 푸는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param directory
	 *            압축을 풀 대상 디렉토리 {@link File}
	 * @param indices
	 *            테이블 인덱스
	 */
	public void extract(final Shell shell, final File directory,
			final int[] indices) {
		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {
				int overwrite = OverwriteDialog.NO; // 덮어쓰기 설정
				ArrayList<Integer> indexList = new ArrayList<Integer>();

				if (path != null) {
					// 디렉토리로 보기
					// 인덱스 변환
					for (int i = 0; i < indices.length; i++) {
						indices[i] = convertIndex(indices[i]);
					}

					for (int index : indices) {
						ZipEntry dirEntry = entryList.get(index);

						if (dirEntry.isDirectory()) {
							// 디렉토리인 경우
							// 하위 디렉토리의 엔트리 추가
							final String dirName = dirEntry.getName();

							for (ZipEntry entry : entryList) {
								String entryPath = getParentEntryName(entry);

								while (!entryPath.equals("")) {
									if (entryPath.equals(dirName)) {
										indexList.add(entryList.indexOf(entry));
										break;
									}

									entryPath = getParentEntryName(entryPath);
								}
							}
						}
					}
				}

				for (int index : indices) {
					indexList.add(index);
				}

				for (Integer index : indexList) {
					ZipEntry entry = entryList.get(index);

					if (entry.isDirectory()) {
						// 디렉토리인 경우
						File entryDir = new File(directory, entry.getName());

						if (entryDir.isFile()) {
							// 같은 이름의 파일이 이미 존재하는 경우
							if (overwrite == OverwriteDialog.YES
									|| overwrite == OverwriteDialog.NO) {
								OverwriteDialog overwriteDialog = new OverwriteDialog();
								overwrite = overwriteDialog.open(shell,
										entryDir.getPath());
							}

							switch (overwrite) {
							case OverwriteDialog.ALL_YES:
							case OverwriteDialog.YES:
								// 덮어씀
								if (!entryDir.canWrite()) {
									MessageBox messageBox = new MessageBox(
											shell, SWT.OK | SWT.ICON_ERROR);
									messageBox.setText("압축 풀기 실패!");
									messageBox.setMessage(entryDir.getPath()
											+ " 파일에 대한 쓰기 권한이 없습니다.");
									messageBox.open();

									continue;
								} else {
									entryDir.delete();
								}

								break;

							case OverwriteDialog.ALL_NO:
							case OverwriteDialog.NO:
								// 덮어쓰지 않음
								continue;

							case OverwriteDialog.CANCEL:
								// 압축 풀기 취소
								MessageBox messageBox = new MessageBox(shell,
										SWT.OK | SWT.ICON_INFORMATION);
								messageBox.setText("압축 풀기 취소!");
								messageBox.setMessage("압축 풀기가 취소되었습니다.");
								messageBox.open();

								return;
							}
						}

						entryDir.mkdirs();
					} else {
						// 파일인 경우
						File entryFile = new File(directory, entry.getName());

						File parentDir = file.getParentFile(); // 부모 디렉토리
						if (parentDir != null && !parentDir.exists()) {
							parentDir.mkdirs();
						}

						if (entryFile.exists()) {
							// 같은 이름의 파일이나 디렉토리가 이미 존재하는 경우
							if (overwrite == OverwriteDialog.YES
									|| overwrite == OverwriteDialog.NO) {
								OverwriteDialog overwriteDialog = new OverwriteDialog();
								overwrite = overwriteDialog.open(shell,
										entryFile.getPath());
							}

							switch (overwrite) {
							case OverwriteDialog.ALL_YES:
							case OverwriteDialog.YES:
								// 덮어씀
								if (!entryFile.canWrite()) {
									MessageBox messageBox = new MessageBox(
											shell, SWT.OK | SWT.ICON_ERROR);
									messageBox.setText("압축 풀기 실패!");
									messageBox.setMessage(entryFile.getPath()
											+ " 파일 또는 디렉토리에 대한 쓰기 권한이 없습니다.");
									messageBox.open();

									continue;
								}

								break;

							case OverwriteDialog.ALL_NO:
							case OverwriteDialog.NO:
								// 덮어쓰지 않음
								continue;

							case OverwriteDialog.CANCEL:
								// 압축 풀기 취소
								MessageBox messageBox = new MessageBox(shell,
										SWT.OK | SWT.ICON_INFORMATION);
								messageBox.setText("압축 풀기 취소!");
								messageBox.setMessage("압축 풀기가 취소되었습니다.");
								messageBox.open();

								return;
							}

							if (entryFile.isDirectory()) {
								JZip.deleteDir(entryFile);
							}
						}

						extractFile(entryFile, entry);
					}
				}

			}

		});

		MessageBox messageBox = new MessageBox(shell, SWT.YES | SWT.NO
				| SWT.ICON_INFORMATION);
		messageBox.setText("압축 풀기 완료!");
		messageBox.setMessage("압축 풀기가 완료되었습니다.\n\n압축 푼 항목들을 표시할까요?");

		if (messageBox.open() == SWT.YES) {
			Program.launch(directory.getPath());
		}

	}

	/**
	 * 압축 파일 내의 모든 파일과 디렉토리를 압축 해제하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param directory
	 *            압축을 풀 디렉토리 {@link File}
	 */
	public void extractAll(final Shell shell, final File directory) {
		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {

				int overwrite = OverwriteDialog.NO;

				for (ZipEntry entry : entryList) {
					if (entry.isDirectory()) {
						// 디렉토리인 경우
						File entryDir = new File(directory, entry.getName());

						if (entryDir.isFile()) {
							// 같은 이름의 파일이 이미 존재하는 경우
							if (overwrite == OverwriteDialog.YES
									|| overwrite == OverwriteDialog.NO) {
								OverwriteDialog overwriteDialog = new OverwriteDialog();
								overwrite = overwriteDialog.open(shell,
										entryDir.getPath());
							}

							switch (overwrite) {
							case OverwriteDialog.ALL_YES:
							case OverwriteDialog.YES:
								// 덮어 씀
								if (!entryDir.canWrite()) {
									MessageBox messageBox = new MessageBox(
											shell, SWT.OK | SWT.ICON_ERROR);
									messageBox.setText("압축 풀기 실패!");
									messageBox.setMessage(entryDir.getPath()
											+ " 파일에 대한 쓰기 권한이 없습니다.");
									messageBox.open();

									continue;
								} else {
									entryDir.delete();
								}

								break;

							case OverwriteDialog.ALL_NO:
							case OverwriteDialog.NO:
								// 덮어 쓰지 않음
								continue;

							case OverwriteDialog.CANCEL:
								// 압축 풀기 취소
								MessageBox messageBox = new MessageBox(shell,
										SWT.OK | SWT.ICON_INFORMATION);
								messageBox.setText("압축 풀기 취소!");
								messageBox.setMessage("압축 풀기가 취소되었습니다.");
								messageBox.open();

								return;
							}
						}

						entryDir.mkdirs();
					} else {
						// 파일인 경우
						File entryFile = new File(directory, entry.getName());

						File parentDir = file.getParentFile();
						if (parentDir != null && !parentDir.exists()) {
							parentDir.mkdirs();
						}

						if (entryFile.exists()) {
							// 같은 이름의 파일이나 디렉토리가 이미 존재하는 경우
							if (overwrite == OverwriteDialog.YES
									|| overwrite == OverwriteDialog.NO) {
								OverwriteDialog overwriteDialog = new OverwriteDialog();
								overwrite = overwriteDialog.open(shell,
										entryFile.getPath());
							}

							switch (overwrite) {
							case OverwriteDialog.ALL_YES:
							case OverwriteDialog.YES:
								// 덮어 씀
								if (!entryFile.canWrite()) {
									MessageBox messageBox = new MessageBox(
											shell, SWT.OK | SWT.ICON_ERROR);
									messageBox.setText("압축 풀기 실패!");
									messageBox.setMessage(entryFile.getPath()
											+ " 파일 또는 디렉토리에 대한 쓰기 권한이 없습니다.");
									messageBox.open();

									continue;
								}

								break;

							case OverwriteDialog.ALL_NO:
							case OverwriteDialog.NO:
								// 덮어 쓰지 않음
								continue;

							case OverwriteDialog.CANCEL:
								// 압축 풀기 취소
								MessageBox messageBox = new MessageBox(shell,
										SWT.OK | SWT.ICON_INFORMATION);
								messageBox.setText("압축 풀기 취소!");
								messageBox.setMessage("압축 풀기가 취소되었습니다.");
								messageBox.open();

								return;
							}

							if (entryFile.isDirectory()) {
								JZip.deleteDir(entryFile);
							}
						}

						extractFile(entryFile, entry);
					}
				}
			}

		});

		MessageBox messageBox = new MessageBox(shell, SWT.YES | SWT.NO
				| SWT.ICON_INFORMATION);
		messageBox.setText("압축 풀기 완료!");
		messageBox.setMessage("압축 풀기가 완료되었습니다.\n\n압축 푼 항목들을 표시할까요?");
		if (messageBox.open() == SWT.YES) {
			Program.launch(directory.getPath());
		}

	}

	/**
	 * {@link ZipEntry}의 압축을 푸는 메소드
	 * 
	 * @param entryFile
	 *            압축을 풀 대상 {@link File}
	 * @param entry
	 *            압축을 풀 엔트리
	 */
	private void extractFile(File entryFile, ZipEntry entry) {
		try {
			InputStream is = zipFile.getInputStream(entry);
			FileOutputStream fos = new FileOutputStream(entryFile);

			try {
				byte[] buffer = new byte[BUFFER_SIZE];
				int read = 0;

				read = is.read(buffer);
				while (read != -1) {
					fos.write(buffer, 0, read);
					read = is.read(buffer);
				}
			} finally {
				is.close();
				fos.close();
			}
		} catch (IOException e1) {
			e1.printStackTrace();
		}
	}

	/**
	 * 압축 파일 내 디렉토리의 크기를 반환하는 메소드
	 * 
	 * @param dirEntry
	 *            크기를 계산할 디렉토리 엔트리
	 * @return 디렉토리의 크기
	 */
	private long getDirSize(ZipEntry dirEntry) {
		long size = 0;

		final String dirName = dirEntry.getName();

		for (ZipEntry entry : entryList) {
			if (!entry.isDirectory()) {
				// 파일인 경우
				String entryPath = getParentEntryName(entry);

				while (!entryPath.equals("")) {
					if (entryPath.equals(dirName)) {
						size += entry.getSize();
						break;
					}

					entryPath = getParentEntryName(entryPath);
				}
			}
		}

		return size;
	}

	/**
	 * {@link ZipEntry}의 이름(경로 포함 안함)을 반환하는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @return {@link ZipEntry}의 이름(경로 포함 안함)
	 */
	public String getEntryName(int index) {
		if (path != null) {
			index = convertIndex(index);
		}

		return getEntryName(entryList.get(index));
	}

	/**
	 * {@link ZipEntry}의 이름(경로 포함 안함)을 반환하는 메소드
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return {@link ZipEntry}의 이름(경로 포함 안함)
	 */
	private String getEntryName(ZipEntry entry) {
		String name = entry.getName();

		if (entry.isDirectory()) {
			// 디렉토리인 경우
			name = name.substring(0, name.length() - 1);
		}

		if (name.indexOf('/') == -1) {
			// 최상위 디렉토리의 파일인 경우
			return name;
		} else {
			return name.substring(name.lastIndexOf('/') + 1);
		}
	}

	/**
	 * {@link ZipEntry}의 위치를 반환합니다.
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return {@link ZipEntry}의 위치
	 */
	private String getEntryPath(ZipEntry entry) {
		String name = entry.getName(); // 엔트리의 이름

		if (name.endsWith("/")) {
			// 디렉토리인 경우
			name = name.substring(0, name.length() - 1);
		}

		if (name.indexOf('/') == -1) {
			// 최상위 디렉토리의 파일인 경우
			return "/";
		} else {
			return "/" + name.substring(0, name.lastIndexOf('/'));
		}
	}

	/**
	 * {@link ZipEntry}의 크기를 반환하는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @return {@link ZipEntry}의 크기
	 */
	public long getEntrySize(int index) {
		if (path != null) {
			// 디렉토리로 보기
			index = convertIndex(index);

			ZipEntry zipEntry = entryList.get(index);
			if (zipEntry.isDirectory()) {
				// 디렉토리인 경우
				return getDirSize(zipEntry);
			} else {
				// 파일인 경우
				return zipEntry.getSize();
			}
		} else {
			// 모든 파일 보기
			return entryList.get(index).getSize();
		}

	}

	/**
	 * Zip 파일의 이름(경로 포함 안함)을 반환하는 메소드
	 * 
	 * @return Zip 파일의 이름(경로 포함 안함)
	 */
	public String getFileName() {
		return file.getName();
	}

	/**
	 * Zip 파일의 경로(파일 이름 포함 안함)를 반환하는 메소드
	 * 
	 * @return Zip 파일의 경로(파일 이름 포함 안함)
	 */
	public String getFileParentPath() {
		return file.getParent();
	}

	/**
	 * Zip 파일의 이름(경로 포함)을 반환하는 메소드
	 * 
	 * @return Zip 파일의 이름(경로 포함)
	 */
	public String getFilePath() {
		return file.getPath();
	}

	/**
	 * Zip 파일의 크기(압축 크기)를 반환하는 메소드
	 * 
	 * @return Zip 파일의 크기(압축 크기)
	 */
	public long getLength() {
		return file.length();
	}

	/**
	 * Zip 파일에 빠진 디렉토리 엔트리를 추가하는 메소드
	 */
	private void getMissingEntries() {
		HashSet<String> hashSet = new HashSet<String>();

		for (ZipEntry entry : entryList) {
			String parent = getParentEntryName(entry);

			while (!parent.equals("")) {
				if (zipFile.getEntry(parent) == null) {
					hashSet.add(parent);
					break;
				}

				parent = getParentEntryName(parent);
			}
		}

		for (String name : hashSet) {
			ZipEntry entry = new ZipEntry(name);
			entryList.add(entry);
		}
	}

	/**
	 * Zip 파일의 실제 크기를 반한하는 메소드
	 * 
	 * @return Zip 파일의 실제 크기
	 */
	public long getOriginalLength() {
		long size = 0;

		for (ZipEntry entry : entryList) {
			size += entry.getSize();
		}

		return size;
	}

	/**
	 * Zip 파일 내의 모든 항목의 숫자를 반환하는 메소드
	 * 
	 * @return Zip 파일 내의 모든 항목의 숫자
	 */
	public int getOriginalSize() {
		return originalSize;
	}

	/**
	 * {@link ZipEntry}의 경로(부모 엔트리 이름)을 반환하는 메소드
	 * 
	 * @param name
	 *            {@link ZipEntry}의 이름
	 * @return {@link ZipEntry}의 경로(부모 엔트리 이름)
	 */
	private String getParentEntryName(String name) {
		if (name.endsWith("/")) {
			// 디렉토리인 경우
			name = name.substring(0, name.length() - 1);
		}

		if (name.indexOf('/') == -1) {
			// 최상위 디렉토리의 파일인 경우
			return "";
		} else {
			return name.substring(0, name.lastIndexOf('/') + 1);
		}
	}

	/**
	 * {@link ZipEntry}의 경로(부모 엔트리 이름)을 반환하는 메소드
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return {@link ZipEntry}의 경로(부모 엔트리 이름)
	 */
	private String getParentEntryName(ZipEntry entry) {
		return getParentEntryName(entry.getName());
	}

	/**
	 * Zip 파일 내의 현재 경로를 반환하는 메소드
	 * 
	 * @return Zip 파일 내의 현재 경로
	 */
	public String getPath() {
		if (path == null) {
			return null;
		}

		if (path.equals("")) {
			return "/";
		} else {
			return "/" + path.substring(0, path.length() - 1);
		}
	}

	/**
	 * 현재 경로에 해당하는 항목의 숫자를 반환하는 메소드
	 * 
	 * @return 현재 경로에 해당하는 항목의 숫자
	 */
	public int getSize() {
		if (path != null) {
			// 디렉토리로 보기
			int count = 0;

			for (ZipEntry entry : entryList) {
				if (getParentEntryName(entry).equals(path)) {
					count++;
				}
			}

			return count;
		} else {
			// 모든 파일 보기
			return originalSize;
		}
	}

	/**
	 * 테이블 출력을 위한 {@link String} 배열을 반환하는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @param dateFormat
	 *            바뀐 시간 형식
	 * @return 테이블 출력을 위한 {@link String} 배열
	 */
	public String[] getStrings(int index, int dateFormat) {
		if (path != null) {
			// 디렉토리로 보기
			index = convertIndex(index);
			ZipEntry entry = entryList.get(index);

			String name = getEntryName(entry);
			String size;
			if (entry.isDirectory()) {
				// 디렉토리인 경우
				size = getSizeString(getDirSize(entry));
			} else {
				// 파일인 경우
				size = getSizeString(entry.getSize());
			}
			String type = getType(entry);
			String time = getTimeString(entry.getTime(), dateFormat);

			return new String[] { name, size, type, time };
		} else {
			// 모든 파일 보기
			ZipEntry entry = entryList.get(index);

			String name = getEntryName(entry);
			String size = getSizeString(entry.getSize());
			String type = getType(entry);
			String time = getTimeString(entry.getTime(), dateFormat);
			String path = getEntryPath(entry);

			return new String[] { name, size, type, time, path };
		}

	}

	/**
	 * {@link ZipEntry}의 형식을 문자열로 반환하는 메소드
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return {@link ZipEntry}의 형식을 나타내는 문자열
	 */
	private String getType(ZipEntry entry) {
		if (entry.isDirectory()) {
			return "디렉토리";
		} else {
			return "파일";
		}
	}

	/**
	 * 부모 디렉토리로 현재 경로를 변경하는 메소드
	 */
	public void goToParent() {
		if (!path.equals("")) {
			path = getParentEntryName(path);
		}
	}

	/**
	 * 최상위 디렉토리로 현재 경로를 변경하는 메소드
	 */
	public void goToTop() {
		if (!path.equals("")) {
			path = "";
		}
	}

	/**
	 * 인덱스에 해당하는 ZipEntry가 디렉토리이면 true, 파일이면 false를 반환하는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @return 디렉토리 여부
	 */
	public boolean isDirecotry(int index) {
		if (path != null) {
			// 디렉토리로 보기
			index = convertIndex(index);
		}

		return entryList.get(index).isDirectory();
	}

	/**
	 * Zip 파일의 바뀐 시간을 반환하는 메소드
	 * 
	 * @return Zip 파일의 바뀐 시간
	 */
	public long lastModified() {
		return file.lastModified();
	}

	/**
	 * {@link ZipEntry}를 읽어들이는 메소드
	 * 
	 * @param zipFile
	 *            {@link ZipFile}
	 */
	private void loadEntries() {
		entryList = new ArrayList<ZipEntry>();
		Enumeration entries = zipFile.getEntries();
		while (entries.hasMoreElements()) {
			ZipEntry entry = (ZipEntry) entries.nextElement();
			entryList.add(entry);
		}

		originalSize = entryList.size();

		if (path != null) {
			getMissingEntries();
		}
	}

	/**
	 * 선택된 항목으로 현재 경로를 변경하는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 */
	public void openDir(int index) {
		index = convertIndex(index);

		path = entryList.get(index).getName();
	}

	/**
	 * 압축 파일 내의 파일을 여는 메소드
	 * 
	 * @param index
	 *            테이블의 인덱스
	 * @param tmpDir
	 *            임시 디렉토리 {@link File}
	 */
	public void openFile(int index, File tmpDir) {
		if (path != null) {
			index = convertIndex(index);
		}

		if (!tmpDir.exists()) {
			tmpDir.mkdirs();
		}

		ZipEntry zipEntry = entryList.get(index);
		File tempFile = new File(tmpDir, getEntryName(zipEntry));

		if (tempFile.isDirectory()) {
			JZip.deleteDir(tempFile);
		}

		try {
			InputStream is = zipFile.getInputStream(zipEntry);
			FileOutputStream fos = new FileOutputStream(tempFile);

			try {
				byte[] buffer = new byte[BUFFER_SIZE];
				int read = 0;

				read = is.read(buffer);
				while (read != -1) {
					fos.write(buffer, 0, read);
					read = is.read(buffer);
				}
			} finally {
				is.close();
				fos.close();
			}
		} catch (ZipException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		Program.launch(tempFile.getPath());
	}

	/**
	 * Zip 파일 내의 파일을 열 프로그램을 선택하여 여는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param index
	 *            테이블의 인덱스
	 * @param tmpDir
	 *            임시 디렉토리 {@link File}
	 */
	public void openWith(Shell shell, int index, File tmpDir) {
		if (path != null) {
			index = convertIndex(index);
		}

		ZipEntry zipEntry = entryList.get(index);

		ProgramSelectDialog select = new ProgramSelectDialog();
		final String command = select.open(shell, getEntryName(zipEntry));

		if (command != null && !command.equals("")) {
			if (!tmpDir.exists()) {
				tmpDir.mkdirs();
			}

			File tempFile = new File(tmpDir, getEntryName(zipEntry));

			if (tempFile.isDirectory()) {
				JZip.deleteDir(tempFile);
			}

			try {
				InputStream is = zipFile.getInputStream(zipEntry);
				FileOutputStream fos = new FileOutputStream(tempFile);

				try {
					byte[] buffer = new byte[BUFFER_SIZE];
					int read = 0;

					read = is.read(buffer);
					while (read != -1) {
						fos.write(buffer, 0, read);
						read = is.read(buffer);
					}
				} finally {
					is.close();
					fos.close();
				}
			} catch (ZipException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			}

			try {
				Runtime.getRuntime().exec(
						new String[] { command, tempFile.getPath() });
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * {@link ZipEntry}의 이름을 변경하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param index
	 *            테이블의 인덱스
	 * @param newName
	 *            새로운 이름
	 * @param tmpDir
	 *            임시 디렉토리 {@link File}
	 */
	public void rename(Shell shell, int index, String name, final File tmpDir) {
		if (path != null) {
			index = convertIndex(index);
		}

		final ZipEntry zipEntry = entryList.get(index);

		final String parentName = getParentEntryName(zipEntry);
		if (!parentName.equals("")) {
			name = parentName + name;
		}

		if (zipEntry.isDirectory() && !name.endsWith("/")) {
			name += "/";
		}

		// 같은 이름을 가진 파일이나 디렉토리가 있는지 확인함
		for (ZipEntry entry : entryList) {
			if (entry.getName().equals(name)) {
				MessageBox messageBox = new MessageBox(shell, SWT.OK
						| SWT.ICON_ERROR);
				messageBox.setText("이름 바꾸기 실패!");
				messageBox.setMessage("같은 이름을 가진 파일 또는 디렉토리가 이미 존재합니다.");
				messageBox.open();

				return;
			}
		}

		final String newName = name;

		final int newIndex = index;

		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {
				if (!tmpDir.exists()) {
					tmpDir.mkdirs();
				}

				File tempFile = new File(tmpDir, file.getName());

				final String encoding = zipFile.getEncoding(); // 인코딩

				try {
					ZipOutputStream zos = new ZipOutputStream(tempFile);

					try {
						zos.setEncoding(encoding);

						for (int i = 0; i < entryList.size(); i++) {
							ZipEntry originalEntry = entryList.get(i);

							final String name = originalEntry.getName();

							ZipEntry entry = null;

							if (i == newIndex) {
								entry = new ZipEntry(newName);
							} else if (path != null) {
								// 디렉토리로 보기인 경우
								String parent = getParentEntryName(name);

								while (!parent.equals("")) {
									if (parent.equals(zipEntry.getName())) {
										entry = new ZipEntry(newName
												+ name.substring(parent
														.length()));

										break;
									}

									parent = getParentEntryName(parent);
								}
							}

							if (entry == null) {
								entry = new ZipEntry(name);
							}

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = zipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}
					} finally {
						zos.finish();
					}

				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					FileInputStream fis = new FileInputStream(tempFile);
					FileOutputStream fos = new FileOutputStream(file);

					try {
						byte[] buffer = new byte[BUFFER_SIZE];
						int read = 0;
						read = fis.read(buffer);
						while (read != -1) {
							fos.write(buffer, 0, read);
							read = fis.read(buffer);
						}
					} finally {
						fis.close();
						fos.close();
					}
				} catch (FileNotFoundException e) {
					e.printStackTrace();
				} catch (IOException e) {
					e.printStackTrace();
				}

				try {
					zipFile = new ZipFile(file, encoding);
				} catch (IOException e) {
					e.printStackTrace();
				}

				loadEntries();
			}

		});
	}

	/**
	 * Zip 파일을 다른 이름으로 저장하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param saveFile
	 *            저장할 {@link File}
	 * @param indices
	 *            테이블의 인덱스 배열
	 */
	public void save(Shell shell, final File saveFile, final int[] indices) {
		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {

				ArrayList<Integer> indexList = new ArrayList<Integer>();

				if (path != null) {
					// 디렉토리로 보기
					// 인덱스 변환
					for (int i = 0; i < indices.length; i++) {
						indices[i] = convertIndex(indices[i]);
					}

					for (int index : indices) {
						ZipEntry dirEntry = entryList.get(index);

						if (dirEntry.isDirectory()) {
							// 디렉토리인 경우
							// 하위 디렉토리의 엔트리 추가
							final String dirName = dirEntry.getName();

							for (ZipEntry entry : entryList) {
								String entryPath = getParentEntryName(entry);

								while (!entryPath.equals("")) {
									if (entryPath.equals(dirName)) {
										indexList.add(entryList.indexOf(entry));
										break;
									}

									entryPath = getParentEntryName(entryPath);
								}
							}
						}
					}
				}

				for (int index : indices) {
					indexList.add(index);
				}

				try {
					ZipOutputStream zos = new ZipOutputStream(file);

					try {
						// 인코딩 설정
						zos.setEncoding("MS949");

						for (int index : indexList) {
							ZipEntry originalEntry = entryList.get(index);
							ZipEntry entry = new ZipEntry(originalEntry
									.getName());

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = zipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}
					} finally {
						zos.finish();
					}
				} catch (IOException e1) {
					e1.printStackTrace();
				}
			}

		});
	}

	/**
	 * Zip 파일을 다른 이름으로 저장하는 메소드
	 * 
	 * @param shell
	 *            {@link Shell}
	 * @param saveFile
	 *            저장할 {@link File}
	 */
	public void saveAll(Shell shell, final File saveFile) {
		BusyIndicator.showWhile(shell.getDisplay(), new Runnable() {

			public void run() {

				try {
					ZipOutputStream zos = new ZipOutputStream(saveFile);

					try {
						// 인코딩 설정
						zos.setEncoding("MS949");

						for (ZipEntry originalEntry : entryList) {
							ZipEntry entry = new ZipEntry(originalEntry
									.getName());

							long time = originalEntry.getTime();
							if (time != -1) {
								entry.setTime(time);
							}

							zos.putNextEntry(entry);

							InputStream is = zipFile
									.getInputStream(originalEntry);

							if (is != null) {
								// 디렉토리가 아닌 경우
								byte[] buffer = new byte[BUFFER_SIZE];
								int read = 0;

								read = is.read(buffer);
								while (read != -1) {
									zos.write(buffer, 0, read);
									read = is.read(buffer);
								}

								is.close();
							}

							zos.closeEntry();
						}
					} finally {
						zos.finish();
					}
				} catch (IOException e1) {
					e1.printStackTrace();
				}

			}
		});
	}

	/**
	 * Zip 파일 내의 현재 경로를 설정하는 메소드
	 * 
	 * @param text
	 *            Zip 파일 내의 현재 경로
	 */
	public void setPath(String text) {
		if (text.equals("/") || text.equals("")) {
			path = "";
			return;
		}

		if (text.startsWith("/")) {
			text = text.substring(1);
		}

		if (!text.endsWith("/")) {
			text += "/";
		}

		for (ZipEntry entry : entryList) {
			if (entry.getName().equals(text)) {
				path = text;
				return;
			}
		}
	}

	/**
	 * {@link ZipEntry}를 이름으로 정렬하는 메소드
	 * 
	 * @param reverse
	 *            역순 여부
	 */
	public void sortByName(final boolean reverse) {
		Collections.sort(entryList, new Comparator<ZipEntry>() {

			public int compare(ZipEntry o1, ZipEntry o2) {
				final String name1 = getEntryName(o1);
				final String name2 = getEntryName(o2);

				if (reverse) {
					return name2.compareTo(name1);
				} else {
					return name1.compareTo(name2);
				}
			}

		});
	}

	/**
	 * {@link ZipEntry}를 위치로 정렬하는 메소드
	 * 
	 * @param reverse
	 *            역순 여부
	 */
	public void sortByPath(final boolean reverse) {
		if (path != null) {
			return;
		}

		Collections.sort(entryList, new Comparator<ZipEntry>() {

			public int compare(ZipEntry o1, ZipEntry o2) {
				final String path1 = getEntryPath(o1);
				final String path2 = getEntryPath(o2);

				if (reverse) {
					return path2.compareTo(path1);
				} else {
					return path1.compareTo(path2);
				}
			}

		});
	}

	/**
	 * {@link ZipEntry}를 크기로 정렬하는 메소드
	 * 
	 * @param reverse
	 *            역순 여부
	 */
	public void sortBySize(final boolean reverse) {
		Collections.sort(entryList, new Comparator<ZipEntry>() {

			public int compare(ZipEntry o1, ZipEntry o2) {
				long size1 = 0;
				long size2 = 0;

				if (path == null) {
					// 모든 파일 보기
					size1 = o1.getSize();
					size2 = o2.getSize();
				} else {
					// 디렉토리로 보기
					if (o1.isDirectory()) {
						// 디렉토리인 경우
						size1 = getDirSize(o1);
					} else {
						// 파일인 경우
						size1 = o1.getSize();
					}

					if (o2.isDirectory()) {
						// 디렉토리인 경우
						size2 = getDirSize(o2);
					} else {
						// 파일인 경우
						size2 = o2.getSize();
					}
				}

				if (reverse) {
					return (int) (size2 - size1);
				} else {
					return (int) (size1 - size2);
				}
			}

		});
	}

	/**
	 * {@link ZipEntry}를 바뀐 시간으로 정렬하는 메소드
	 * 
	 * @param reverse
	 *            역순 여부
	 */
	public void sortByTime(final boolean reverse) {
		Collections.sort(entryList, new Comparator<ZipEntry>() {

			public int compare(ZipEntry o1, ZipEntry o2) {
				final long time1 = o1.getTime();
				final long time2 = o2.getTime();

				if (reverse) {
					return (int) (time2 - time1);
				} else {
					return (int) (time1 - time2);
				}
			}

		});
	}

	/**
	 * {@link ZipEntry}를 형태로 정렬하는 메소드
	 * 
	 * @param reverse
	 *            역순 여부
	 */
	public void sortByType(final boolean reverse) {
		Collections.sort(entryList, new Comparator<ZipEntry>() {

			public int compare(ZipEntry o1, ZipEntry o2) {
				final String type1 = getType(o1);
				final String type2 = getType(o2);

				if (reverse) {
					return type2.compareTo(type1);
				} else {
					return type1.compareTo(type2);
				}
			}

		});
	}
}