/**
 * JZip 0.4
 * 이 프로그램은 Zip 파일의 압축을 풀어주는 GUI 프로그램입니다.
 *
 * 윈도우에서 사용하는 MS949 인코딩과 리눅스에서 사용하는 UTF-8 인코딩을 모두 지원하기 때문에,
 * 윈도우나 리눅스에서 압축된 Zip 파일을 한글 파일명이 깨지는 문제 없이 풀 수 있습니다.
 *
 * 또한 이 프로그램은 Zip 파일 외에도 Zip 파일과 같은 포맷을 사용하는 다른 파일도 처리할 수 있습니다.
 * Zip 파일과 같은 포맷을 사용하는 사용하는 파일에는 JAR(Java Archive)와
 * OpenDocument, Office Open XML 등이 있습니다.
 *
 * 이 프로그램은 자바로 만들어졌기 때문에, 플랫폼에 관계 없이 사용하실 수 있습니다.
 *
 * 이 프로그램은 SWT(Standard Widget Toolkit)와 Apache Ant를 사용합니다.
 * 
 * 저작권 : GNU General Public License
 * 홈페이지 : http://jzip.kldp.net
 * 개발자 : 정승원 jeongseungwon@hanmail.net
 */
package net.kldp.jzip;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;

import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.ArmListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.program.Program;

/**
 * JZip 프로그램의 메인 클래스
 * 
 * @author jeongseungwon
 * 
 */
public class JZip {
	/**
	 * {@link ArmListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class DeleteArmListener implements
			org.eclipse.swt.events.ArmListener {
		public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
			statusLine.setText("선택된 항목을 삭제합니다.");
		}
	}

	/**
	 * {@link SelectionListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class DeleteSelectionListener implements
			org.eclipse.swt.events.SelectionListener {
		public void widgetDefaultSelected(
				org.eclipse.swt.events.SelectionEvent e) {
		}

		public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
			delete();
		}
	}

	/**
	 * {@link DragSourceListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class DragListener implements DragSourceListener {
		private DropListener listener;

		private DragListener(DropListener listener) {
			this.listener = listener;
		}

		public void dragFinished(DragSourceEvent event) {
			target.addDropListener(listener);
		}

		public void dragSetData(DragSourceEvent event) {
			final int selectionCount = table.getSelectionCount();

			if (selectionCount >= 1) {
				// 선택된 항목이 있는 경우
				statusLine.setText("압축 해제 중입니다.");
				statusLine.update();

				String[] fileNames = new String[selectionCount];
				int[] indices = table.getSelectionIndices();

				if (!tempDir.exists()) {
					// 임시 디렉토리가 존재하지 않는 경우
					tempDir.mkdirs();
				}

				for (int i = 0; i < indices.length; i++) {
					File file = new File(tempDir, zip.getEntryName(indices[i]));

					if (zip.isDirectory(indices[i])) {
						// 디렉토리인 경우
						file.mkdirs();
					} else {
						// 파일인 경우
						// 압축 풀기
						extract(indices[i], file);
					}

					fileNames[i] = file.getAbsolutePath(); // 파일 명
				}

				event.data = fileNames;

				setStatusLine();
			} else {
				// 선택된 항목이 없는 경우
				event.doit = false;
				event.data = null;
				event.detail = DND.DROP_NONE;

				return;
			}

		}

		public void dragStart(DragSourceEvent event) {
			target.removeDropListener(listener);
		}
	}

	/**
	 * {@link DropTargetListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class DropListener implements DropTargetListener {
		public void dragEnter(DropTargetEvent event) {
		}

		public void dragLeave(DropTargetEvent event) {
		}

		public void dragOperationChanged(DropTargetEvent event) {
		}

		public void dragOver(DropTargetEvent event) {
		}

		public void drop(DropTargetEvent event) {
			if (event.data == null) {
				event.detail = DND.DROP_NONE;

				return;
			}

			String[] fileNames = (String[]) event.data;

			if (zip == null) {
				// 현재 열려있는 압축 파일이 없는 경우
				if (fileNames.length == 1) {
					// 파일이 하나인 경우
					if (fileNames[0] != null) {
						// 압축 파일 열기
						open(fileNames[0]);
					}
				} else {
					event.detail = DND.DROP_NONE;
				}
			} else {
				// 열려 있는 압축 파일이 있는 경우
				if (fileNames.length != 1) {
					// 파일이 여러 개인 경우
					addFile(fileNames);
				} else {
					// 파일이 하나인 경우
					if (canOpen(fileNames[0])) {
						// 열 수 있는 파일인 경우
						// 동작 선택 대화상자
						ActionSelectDialog selectDialog = new ActionSelectDialog();
						final int select = selectDialog.open(sShell,
								fileNames[0]);

						switch (select) {
						case ActionSelectDialog.CANCEL:
							event.detail = DND.DROP_NONE;
							break;

						case ActionSelectDialog.ADD:
							addFile(fileNames);
							break;

						case ActionSelectDialog.OPEN:
							open(fileNames[0]);
							break;
						}
					} else {
						statusLine.setText("파일을 더하는 중입니다.");
						statusLine.update();

						addFile(fileNames);
					}
				}
			}
		}

		public void dropAccept(DropTargetEvent event) {
		}
	}

	/**
	 * ArmListener를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class ExtractArmListener implements
			org.eclipse.swt.events.ArmListener {
		public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
			statusLine.setText("선택된 항목을 압축 해제합니다.");
		}
	}

	/**
	 * {@link SelectionListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class ExtractSelectionListener implements
			org.eclipse.swt.events.SelectionListener {
		public void widgetDefaultSelected(
				org.eclipse.swt.events.SelectionEvent e) {
		}

		public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
			extract();
		}
	}

	/**
	 * {@link ArmListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class RenameArmListener implements
			org.eclipse.swt.events.ArmListener {
		public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
			statusLine.setText("선택된 항목의 이름을 변경합니다.");
		}
	}

	/**
	 * {@link SelectionListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private class RenameSelectionListener implements
			org.eclipse.swt.events.SelectionListener {
		public void widgetDefaultSelected(
				org.eclipse.swt.events.SelectionEvent e) {
		}

		public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
			rename();
		}
	}

	/**
	 * JZip 프로그램의 메인 클래스
	 * 
	 * @param args
	 *            불러올 Zip 파일의 이름
	 */
	public static void main(String[] args) {
		Display display = Display.getDefault();
		JZip thisClass = new JZip();
		thisClass.createSShell();

		// 임시 디렉토리 설정
		thisClass.tempDir = new File(System.getProperty("java.io.tmpdir"),
				"JZip");

		// 기본 경로 설정
		thisClass.defaultPath = System.getProperty("user.home");

		// 레이아웃 설정
		thisClass.showStatusLine();

		// DnD 설정
		thisClass.setDnD();

		// 문맥 메뉴 설정
		thisClass.setContextMenu();

		if (args.length == 1 && args[0] != null) {
			// 인자로 주어진 압축 파일을 불러옴
			thisClass.open(args[0]);
		}

		thisClass.sShell.open();

		while (!thisClass.sShell.isDisposed()) {
			if (!display.readAndDispatch()) {
				display.sleep();
			}
		}
		display.dispose();
	}

	/**
	 * 버퍼 크기 (8 KB)
	 */
	private final int BUFFER_SIZE = 8 * 1024;

	private MenuItem checkReverse = null;

	private MenuItem checkStatusLine = null;

	private String defaultPath = null;

	/**
	 * 프로그램 이름
	 */
	private final String jzip = "JZip 0.4";

	private Menu menuBar = null;

	private MenuItem pushAddFile = null;

	private MenuItem pushClose = null;

	private MenuItem pushDeselectAll = null;

	private MenuItem pushExtract = null;

	private MenuItem pushProperty = null;

	private MenuItem pushRefresh = null;

	private MenuItem pushSaveAs = null;

	private MenuItem pushSelectAll = null;

	private MenuItem radioMs949 = null;

	private MenuItem radioName = null;

	private MenuItem radioPath = null;

	private MenuItem radioSize = null;

	private MenuItem radioTime = null;

	private MenuItem radioType = null;

	private MenuItem radioUtf8 = null;

	private DragSource source = null;

	private Shell sShell = null; // @jve:decl-index=0:visual-constraint="9,10"

	private Label statusLine = null;

	private Label statusSeparator = null;

	private Menu submenuAlignment = null;

	private Menu submenuArchive = null;

	private Menu submenuEdit = null;

	private Menu submenuHelp = null;

	private MenuItem submenuItemAlignment = null;

	private Menu submenuView = null;

	private Table table = null;

	private TableColumn tableColumnName = null;

	private TableColumn tableColumnPath = null;

	private TableColumn tableColumnSize = null;

	private TableColumn tableColumnTime = null;

	private TableColumn tableColumnType = null;

	private DropTarget target = null;

	/**
	 * 프로그램의 임시 디렉토리
	 */
	private File tempDir = null;

	/**
	 * 현재 열려있는 {@link Zip} 클래스 객체
	 */
	private Zip zip = null; // @jve:decl-index=0:

	private MenuItem pushRename = null;

	private MenuItem pushDelete = null;

	private Menu contextMenu = null;

	private MenuItem itemRename = null;

	private MenuItem pushAddDirectory = null;

	private MenuItem itemView = null;

	private MenuItem itemOpen = null;

	private Menu submenuFormat = null;

	private MenuItem submenuItemFormat = null;

	private MenuItem radioLong = null;

	private MenuItem radioShort = null;

	private MenuItem radioMedium = null;

	/**
	 * 프로그램 정보를 보여주는 메소드
	 */
	private void about() {
		StringBuffer message = new StringBuffer(70);
		message.append(jzip + "\n");
		message.append("\n이 프로그램은 Zip 파일 포맷을 지원하는 압축 프로그램입니다.\n");
		message.append("\n윈도우나 리눅스에서 압축된 Zip 파일을 한글 파일명이 깨지는 문제 없이 풀어줍니다.\n");
		message.append("\n저작권 : GNU General Public License");
		message.append("\n홈페이지 : http://jzip.kldp.net");
		message.append("\n개발자 : 정승원 (jeongseungwon@hanmail.net)");

		MessageBox messageBox = new MessageBox(sShell, SWT.OK
				| SWT.ICON_INFORMATION);
		messageBox.setText("JZip 정보");
		messageBox.setMessage(message.toString());
		messageBox.open();
	}

	/**
	 * 현재 열려있는 압축 파일에 디렉토리를 추가하는 메소드
	 */
	private void addDir() {
		if (!zip.canWrite()) {
			// 쓰기 권한이 없는 경우
			final String text = "디렉토리 더하기 실패!";
			final String message = zip.getName() + " 파일에 대한 쓰기 권한이 없습니다.";
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		// 디렉토리 선택 대화상자
		DirectoryDialog dialog = new DirectoryDialog(sShell, SWT.OPEN);
		dialog.setText("압축 파일에 더할 디렉토리를 선택하세요.");
		dialog.setMessage("압축 파일에 더할 디렉토리를 선택하세요.\n" +
				"모든 하위 디렉토리와 디렉토리에 포함된 파일들까지 한꺼번에 더합니다.");
		dialog.setFilterPath(defaultPath);
		final String directoryName = dialog.open();

		if (directoryName != null) {
			statusLine.setText("디렉토리를 더하는 중입니다.");
			statusLine.update();

			addDir(directoryName);
		}
	}

	/**
	 * 현재 열려있는 압축 파일에 디렉토리를 추가하는 메소드
	 * 
	 * @param directoryName
	 *            추가할 디렉토리명
	 */
	private void addDir(String directoryName) {
		File directory = new File(directoryName);

		if (!directory.exists()) {
			// 디렉토리가 존재하지 않는 경우
			final String message = directory.getAbsolutePath()
					+ " 디렉토리가 존재하지 않습니다.";
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("디렉토리 더하기 실패!");
			messageBox.setMessage(message);
			messageBox.open();

			setStatusLine();

			return;
		}

		final String encoding = zip.getEncoding(); // 인코딩

		// 임시 Zip 파일 생성
		if (!tempDir.exists()) {
			tempDir.mkdirs();
		}

		File tempFile = new File(tempDir, zip.getFileName());
		copyFile(zip.getFile(), tempFile);

		// 임시 Zip 객체 생성
		ZipFile tempZipFile = null;
		try {
			tempZipFile = new ZipFile(tempFile, encoding);
		} catch (IOException e) {
			e.printStackTrace();
		}
		Zip tempZip = new Zip(tempFile, tempZipFile); // 임시 Zip 객체 생성

		// 새로운 Zip 파일 생성
		ZipOutputStream zos = zip.getZipOutputStream();
		zos.setEncoding(encoding);

		for (int i = 0; i < tempZip.getSize(); i++) {
			ZipEntry originalEntry = tempZip.getEntry(i);
			ZipEntry entry = new ZipEntry(originalEntry.getName());
			entry.setTime(originalEntry.getTime());

			// 엔트리 압축
			saveEntry(zos, entry, tempZip.getInputStream(i));
		}

		// 새로운 디렉토리 엔트리 생성
		ZipEntry entry = new ZipEntry(directory.getName() + "/");
		entry.setTime(directory.lastModified());

		// 디렉토리 더하기
		saveEntry(zos, entry, null);

		// 디렉토리내 파일 및 하위 디렉토리 더하기
		addFileNDir(zos, directory.listFiles(), directory.getName());

		try {
			zos.finish();
		} catch (IOException e) {
		}

		// 새로 고침
		refresh();
	}

	/**
	 * 파일을 더하는 메소드
	 */
	private void addFile() {
		if (!zip.canWrite()) {
			// 쓰기 권한이 없는 경우
			final String message = zip.getName() + " 파일에 대한 쓰기 권한이 없습니다.";
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText("파일 더하기 실패");
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		// 파일 선택 대화상자
		FileDialog dialog = new FileDialog(sShell, SWT.OPEN | SWT.MULTI);
		dialog.setText("압축 파일에 더할 파일을 선택하세요.");
		dialog.setFilterPath(defaultPath);

		if (dialog.open() != null) {
			final String parentName = dialog.getFilterPath();
			final String[] fileNames = dialog.getFileNames();

			statusLine.setText("파일을 더하는 중입니다.");
			statusLine.update();

			addFile(parentName, fileNames);
		}
	}

	/**
	 * 현재 압축 파일에 파일을 더하는 메소드
	 * 
	 * @param parent
	 *            부모 디렉토리
	 * @param fileNames
	 *            더할 파일의 이름
	 */
	private void addFile(String parent, String[] fileNames) {
		String absoluteFileNames[] = new String[fileNames.length];

		for (int i = 0; i < absoluteFileNames.length; i++) {
			File file = new File(parent, fileNames[i]);
			absoluteFileNames[i] = file.getAbsolutePath();
		}

		addFile(absoluteFileNames);
	}

	/**
	 * 현재 열려있는 압축 파일에 파일을 더하는 메소드
	 * 
	 * @param fileNames
	 *            더할 파일명
	 */
	private void addFile(String[] fileNames) {
		File files[] = new File[fileNames.length]; // 더할 파일들

		for (int i = 0; i < fileNames.length; i++) {
			files[i] = new File(fileNames[i]);
		}

		final String encoding = zip.getEncoding(); // 인코딩

		// 임시 Zip 파일 생성
		if (!tempDir.exists()) {
			tempDir.mkdirs();
		}

		File tempFile = new File(tempDir, zip.getFileName());
		copyFile(zip.getFile(), tempFile);

		// 임시 Zip 객체 생성
		ZipFile tempZipFile = null;
		try {
			tempZipFile = new ZipFile(tempFile, encoding);
		} catch (IOException e) {
			e.printStackTrace();
		}
		Zip tempZip = new Zip(tempFile, tempZipFile); // 임시 Zip 객체 생성

		// 새로운 Zip 파일 생성
		ZipOutputStream zos = zip.getZipOutputStream();
		zos.setEncoding(encoding);

		for (int i = 0; i < tempZip.getSize(); i++) {
			ZipEntry originalEntry = tempZip.getEntry(i);
			ZipEntry entry = new ZipEntry(originalEntry.getName());
			entry.setTime(originalEntry.getTime());

			// 엔트리 압축
			saveEntry(zos, entry, tempZip.getInputStream(i));
		}

		// 파일 추가
		addFileNDir(zos, files, null);

		try {
			zos.finish();
		} catch (IOException e) {
		}

		// 새로 고침
		refresh();
	}

	/**
	 * {@link ZipOutputStream}에 파일이나 디렉토리를 더하는 메소드
	 * 
	 * @param zos
	 *            {@link ZipOutputStream}
	 * @param files
	 *            추가할 {@link File}의 배열
	 * @param parent
	 *            압축 파일내 경로
	 */
	private void addFileNDir(ZipOutputStream zos, File[] files, String parent) {
		MessageBox messageBox = null;
		final String text = "압축 파일 더하기 실패";
		String message = null;

		for (File file : files) {
			if (file.isDirectory()) {
				// 디렉토리인 경우
				if (parent == null) {
					// 상위 디렉토리가 없는 경우
					// 새로운 디렉토리 엔트리 생성
					ZipEntry entry = new ZipEntry(file.getName() + "/");
					entry.setTime(file.lastModified());

					// 디렉토리 더하기
					saveEntry(zos, entry, null);

					// 디렉토리내 파일 및 하위 디렉토리 더하기
					addFileNDir(zos, file.listFiles(), file.getName());
				} else {
					// 상위 디렉토리가 있는 경우
					// 새로운 디렉토리 엔트리 생성
					ZipEntry entry = new ZipEntry(parent + "/" + file.getName()
							+ "/");
					entry.setTime(file.lastModified());

					// 디렉토리 더하기
					saveEntry(zos, entry, null);

					// 디렉토리내 파일 및 하위 디렉토리 더하기
					addFileNDir(zos, file.listFiles(), parent + "/"
							+ file.getName());
				}
			} else {
				// 파일인 경우
				if (!file.exists()) {
					// 파일이 존재하지 않는 경우
					message = file.getAbsolutePath() + " 파일이 존재하지 않습니다.";
					messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
					messageBox.setText(text);
					messageBox.setMessage(message);
					messageBox.open();

					return;
				}
				if (!file.canRead()) {
					// 파일에 대한 읽기 권한이 없는 경우
					message = file.getAbsolutePath() + " 파일에 대한 읽기 권한이 없습니다.";
					messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
					messageBox.setText(text);
					messageBox.setMessage(message);
					messageBox.open();

					return;
				}

				// 새로운 파일 엔트리 생성
				ZipEntry entry = null;
				if (parent == null) {
					// 상위 디렉토리가 없는 경우
					entry = new ZipEntry(file.getName());
				} else {
					// 상위 디렉토리가 있는 경우
					entry = new ZipEntry(parent + "/" + file.getName());
				}
				entry.setTime(file.lastModified());

				InputStream is = null;
				try {
					is = new FileInputStream(file);
				} catch (IOException e1) {
					e1.printStackTrace();
				}

				// 파일 더하기
				saveEntry(zos, entry, is);
			}
		}
	}

	/**
	 * 열 수 있는 파일인지 확인하는 메소드
	 * 
	 * @param fileName
	 *            확인할 파일의 이름
	 * @return 열 수 있으면 true, 없으면 false
	 */
	public boolean canOpen(String fileName) {
		File file = new File(fileName);

		if (!file.exists()) {
			// 파일이 존재하지 않는 경우
			return false;
		}

		if (file.isDirectory()) {
			// 디렉토리인 경우
			return false;
		}

		if (!file.canRead()) {
			// 읽기 권한이 없는 경우
			return false;
		}

		return true;
	}

	/**
	 * 압축 파일을 닫는 메소드
	 */
	private void close() {
		zip = null;

		table.removeAll();
		table.setVisible(false);

		// 메뉴 비활성화
		disableMenus();

		setStatusLine();

		sShell.setText(jzip);
	}

	/**
	 * {@link File}을 복사하는 메소드
	 * 
	 * @param file1
	 *            복사할 {@link File}
	 * @param file2
	 *            복사될 {@link File}
	 */
	private void copyFile(File file1, File file2) {
		try {
			FileInputStream fis = new FileInputStream(file1);
			FileOutputStream fos = new FileOutputStream(file2);

			byte[] buffer = new byte[BUFFER_SIZE];
			int read = 0;

			read = fis.read(buffer);
			while (read != -1) {
				fos.write(buffer, 0, read);
				read = fis.read(buffer);
			}

			fis.close();
			fos.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 새로운 압축 파일을 생성하는 메소드
	 */
	private void createNew() {
		// 새로운 압축 파일 선택 대화상자
		FileDialog dialog = new FileDialog(sShell, SWT.SAVE);
		dialog.setText("새로 만들 압축 파일을 선택하세요.");
		dialog.setFilterPath(defaultPath);
		dialog.setFilterExtensions(new String[] { "*.zip", "*.jar", "*.*" });
		dialog.setFilterNames(new String[] { "Zip 파일 (*.zip)",
				"Jar 파일 (*.jar)", "모든 파일 (*.*)" });
		String file = dialog.open();

		if (file != null) {
			if (saveAs(file)) {
				radioMs949.setSelection(true);
				radioUtf8.setSelection(false);

				open(file);
			}
		}
	}

	/**
	 * This method initializes sShell
	 */
	private void createSShell() {
		sShell = new Shell();
		sShell.setText(jzip);
		sShell.setLayout(new FormLayout());
		sShell.setSize(new Point(720, 400));
		menuBar = new Menu(sShell, SWT.BAR);
		MenuItem submenuItemArchive = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemArchive.setText("압축 파일(&A)");
		MenuItem submenuItemEdit = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemEdit.setText("편집(&E)");
		submenuEdit = new Menu(submenuItemEdit);
		pushAddFile = new MenuItem(submenuEdit, SWT.PUSH);
		pushAddFile.setText("파일 더하기(&F)...");
		pushAddFile.setEnabled(false);
		pushAddDirectory = new MenuItem(submenuEdit, SWT.PUSH);
		pushAddDirectory.setText("디렉토리 더하기(&D)...");
		pushAddDirectory.setEnabled(false);
		pushAddDirectory
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("압축 파일에 디렉토리를 더합니다.");
					}
				});
		pushAddDirectory
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addDir();
					}
				});
		pushAddFile.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일에 파일을 더합니다.");
			}
		});
		pushAddFile
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						addFile();
					}
				});
		@SuppressWarnings("unused")
		MenuItem menuSeparator2 = new MenuItem(submenuEdit, SWT.SEPARATOR);
		pushRename = new MenuItem(submenuEdit, SWT.PUSH);
		pushRename.setText("이름 바꾸기(&R)...\tF2");
		pushRename.setAccelerator(SWT.F2);
		pushRename.setEnabled(false);
		pushRename.addArmListener(new RenameArmListener());
		pushRename.addSelectionListener(new RenameSelectionListener());
		pushDelete = new MenuItem(submenuEdit, SWT.PUSH);
		pushDelete.setText("지우기(&D)...\tDel");
		pushDelete.setAccelerator(SWT.DEL);
		pushDelete.setEnabled(false);
		pushDelete.addArmListener(new DeleteArmListener());
		pushDelete.addSelectionListener(new DeleteSelectionListener());
		@SuppressWarnings("unused")
		MenuItem menuSeparator4 = new MenuItem(submenuEdit, SWT.SEPARATOR);
		pushSelectAll = new MenuItem(submenuEdit, SWT.PUSH);
		pushDeselectAll = new MenuItem(submenuEdit, SWT.PUSH);
		pushDeselectAll.setEnabled(false);
		pushDeselectAll.setText("모두 선택 해제(&U)");
		pushDeselectAll
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("모든 항목을 선택 해제합니다.");
					}
				});
		pushDeselectAll
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						deselectAll();
					}
				});
		submenuEdit.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		pushSelectAll.setText("모두 선택(&A)\tCtrl+A");
		pushSelectAll.setAccelerator(SWT.CTRL | 'A');
		pushSelectAll.setEnabled(false);
		pushSelectAll
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						selectAll();
					}
				});
		pushSelectAll.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("모든 항목을 선택합니다.");
			}
		});
		submenuItemEdit.setMenu(submenuEdit);
		MenuItem submenuItemView = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemView.setText("보기(&V)");
		MenuItem submenuItemHelp = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemHelp.setText("도움말(&H)");
		submenuView = new Menu(submenuItemView);
		pushRefresh = new MenuItem(submenuView, SWT.PUSH);
		pushRefresh.setText("새로 고침(&R)\tF5");
		pushRefresh.setEnabled(false);
		pushRefresh.setAccelerator(SWT.F5);
		pushRefresh
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						refresh();
					}
				});
		pushRefresh.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 다시 불러옵니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator6 = new MenuItem(submenuView, SWT.SEPARATOR);
		radioMs949 = new MenuItem(submenuView, SWT.RADIO);
		submenuView.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		radioUtf8 = new MenuItem(submenuView, SWT.RADIO);
		radioUtf8.setText("&UTF-8 (리눅스)");
		radioUtf8.setEnabled(false);
		radioUtf8
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && !radioUtf8.getSelection()) {
							refresh();
						}
					}
				});
		radioUtf8.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("인코딩을 UTF-8(리눅스)로 변경합니다.");
			}
		});
		radioMs949.setText("&MS949 (윈도우)");
		radioMs949.setEnabled(false);
		radioMs949.setSelection(true);
		radioMs949
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (zip != null && !radioMs949.getSelection()) {
							refresh();
						}
					}
				});
		radioMs949.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("인코딩을 MS949(윈도우)로 변경합니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator1 = new MenuItem(submenuView, SWT.SEPARATOR);
		checkStatusLine = new MenuItem(submenuView, SWT.CHECK);
		checkStatusLine.setText("상태 표시줄(&S)");
		checkStatusLine.setSelection(true);
		@SuppressWarnings("unused")
		MenuItem menuSeparator7 = new MenuItem(submenuView, SWT.SEPARATOR);
		submenuItemAlignment = new MenuItem(submenuView, SWT.CASCADE);
		submenuItemAlignment.setText("항목 정렬(&A)");
		submenuItemAlignment.setEnabled(false);
		submenuItemFormat = new MenuItem(submenuView, SWT.CASCADE);
		submenuItemFormat.setEnabled(false);
		submenuItemFormat.setText("바뀐 시간 출력 형식(&F)");
		submenuFormat = new Menu(submenuItemFormat);
		radioShort = new MenuItem(submenuFormat, SWT.RADIO);
		radioShort.setText("간략하게(&S)");
		radioShort.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간을 간략하게 출력합니다."); 
			}
		});
		radioShort.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
			public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {
			}
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				if (!radioShort.getSelection()) {
					refresh();
				}
			}
		});
		radioMedium = new MenuItem(submenuFormat, SWT.RADIO);
		radioMedium.setSelection(true);
		radioMedium.setText("보통(&M)");
		radioMedium.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간을 기본 형식으로 출력합니다.");
			}
		});
		radioMedium
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}
					public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
						if (!radioMedium.getSelection()) {
							refresh();
						}
					}
				});
		radioLong = new MenuItem(submenuFormat, SWT.RADIO);
		radioLong.setText("자세하게(&L)");
		radioLong.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간을 자세하게 출력합니다.");
			}
		});
		radioLong.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
			public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {
			}
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {
				if (!radioLong.getSelection()) {
					refresh();
				}
			}
		});
		submenuItemFormat.setMenu(submenuFormat);
		submenuItemFormat.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간 출력 형식을 변경합니다.");
			}
		});
		submenuAlignment = new Menu(submenuItemAlignment);
		radioName = new MenuItem(submenuAlignment, SWT.RADIO);
		radioName.setText("이름(&N)");
		radioName.setSelection(true);
		radioName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortByName();
					}
				});
		radioName.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("이름에 따라 정렬합니다.");
			}
		});
		radioSize = new MenuItem(submenuAlignment, SWT.RADIO);
		radioSize.setText("크기(&S)");
		radioSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortBySize();
					}
				});
		radioSize.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("크기에 따라 정렬합니다.");
			}
		});
		radioType = new MenuItem(submenuAlignment, SWT.RADIO);
		radioType.setText("형식(&T)");
		radioType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortByType();
					}
				});
		radioType.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("형식에 따라 정렬합니다.");
			}
		});
		radioTime = new MenuItem(submenuAlignment, SWT.RADIO);
		radioTime.setText("바뀐 시간(&D)");
		radioTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortByTime();
					}
				});
		radioTime.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간에 따라 정렬합니다.");
			}
		});
		radioPath = new MenuItem(submenuAlignment, SWT.RADIO);
		radioPath.setText("위치(&P)");
		radioPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortByPath();
					}
				});
		radioPath.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("위치에 따라 정렬합니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator3 = new MenuItem(submenuAlignment, SWT.SEPARATOR);
		checkReverse = new MenuItem(submenuAlignment, SWT.CHECK);
		checkReverse.setText("역순(&R)");
		checkReverse
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						sortReverse();
					}
				});
		checkReverse.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("거꾸로 정렬합니다.");
			}
		});
		submenuItemAlignment.setMenu(submenuAlignment);
		submenuItemAlignment
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("항목을 정렬합니다.");
					}
				});
		checkStatusLine
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						toggleStatusLine();
					}
				});
		checkStatusLine
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("상태 표시줄을 보이거나 감춥니다.");
					}
				});
		submenuItemView.setMenu(submenuView);
		submenuHelp = new Menu(submenuItemHelp);
		submenuHelp.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		MenuItem pushAbout = new MenuItem(submenuHelp, SWT.PUSH);
		pushAbout.setText("JZip 정보(&A)...");
		pushAbout
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						about();
					}
				});
		pushAbout.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("프로그램 정보를 보여줍니다.");
			}
		});
		submenuItemHelp.setMenu(submenuHelp);
		submenuArchive = new Menu(submenuItemArchive);
		MenuItem pushNew = new MenuItem(submenuArchive, SWT.PUSH);
		pushNew.setText("새로 만들기(&N)...\tCtrl+N");
		pushNew.setAccelerator(SWT.CTRL | 'N');
		pushNew.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("새로운 압축 파일을 생성합니다.");
			}
		});
		pushNew
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						createNew();
					}
				});
		submenuArchive
				.addMenuListener(new org.eclipse.swt.events.MenuListener() {
					public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
						setStatusLine();
					}

					public void menuShown(org.eclipse.swt.events.MenuEvent e) {
					}
				});
		MenuItem pushOpen = new MenuItem(submenuArchive, SWT.PUSH);
		pushOpen.setText("열기(&O)...\tCtrl+O");
		pushOpen.setAccelerator(SWT.CONTROL | 'O');
		pushSaveAs = new MenuItem(submenuArchive, SWT.PUSH);
		pushSaveAs.setText("다른 이름으로 저장(&S)...\tCtrl+S");
		pushSaveAs.setEnabled(false);
		pushSaveAs.setAccelerator(SWT.CTRL | 'S');
		pushSaveAs.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("선택된 항목을 다른 이름을 가진 압축 파일로 저장합니다.");
			}
		});
		pushSaveAs
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						saveAs();
					}
				});
		pushOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						open();
					}
				});
		pushOpen.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 엽니다.");
			}
		});
		pushExtract = new MenuItem(submenuArchive, SWT.PUSH);
		pushExtract.setText("압축 풀기(&E)...\tCtrl+E");
		pushExtract.setAccelerator(SWT.CONTROL | 'E');
		pushExtract.setEnabled(false);
		pushProperty = new MenuItem(submenuArchive, SWT.PUSH);
		pushProperty.setText("속성(&P)...");
		pushProperty.setEnabled(false);
		pushProperty.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일의 속성을 보여줍니다.");
			}
		});
		pushProperty
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						showProperty();
					}
				});
		pushExtract.addSelectionListener(new ExtractSelectionListener());
		pushExtract.addArmListener(new ExtractArmListener());
		pushClose = new MenuItem(submenuArchive, SWT.PUSH);
		pushClose.setText("닫기(&C)");
		pushClose.setEnabled(false);
		pushClose
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 압축 파일 닫기
						close();
					}
				});
		pushClose.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 닫습니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator = new MenuItem(submenuArchive, SWT.SEPARATOR);
		MenuItem pushQuit = new MenuItem(submenuArchive, SWT.PUSH);
		pushQuit.setText("프로그램 종료(&Q)\tCtrl+Q");
		pushQuit.setAccelerator(SWT.CONTROL | 'Q');
		pushQuit
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 임시 파일 삭제
						deleteAllFiles(tempDir);

						// 프로그램 종료
						System.exit(0);
					}
				});
		pushQuit.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("프로그램을 종료합니다.");
			}
		});
		submenuItemArchive.setMenu(submenuArchive);
		sShell.setMenuBar(menuBar);
		sShell.addShellListener(new org.eclipse.swt.events.ShellAdapter() {
			@Override
			public void shellClosed(org.eclipse.swt.events.ShellEvent e) {
				// 임시 파일 삭제
				deleteAllFiles(tempDir);
			}
		});

		table = new Table(sShell, SWT.MULTI | SWT.VIRTUAL);
		table.addListener(SWT.SetData, new Listener() {
			public void handleEvent(Event event) {
				TableItem item = (TableItem) event.item;
				int index = table.indexOf(item);
				
				int dateFormat = DateFormat.MEDIUM;
				
				if (radioLong.getSelection()) {
					dateFormat = DateFormat.LONG;
				} else if (radioShort.getSelection()){
					dateFormat = DateFormat.SHORT;
				}
				
				item.setText(zip.getStrings(index, dateFormat));
			}
		});

		statusSeparator = new Label(sShell, SWT.SEPARATOR | SWT.HORIZONTAL);

		statusLine = new Label(sShell, SWT.NONE);
		statusLine.setText("");

		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		table.setVisible(false);
		table.addMouseListener(new org.eclipse.swt.events.MouseAdapter() {

			public void mouseUp(org.eclipse.swt.events.MouseEvent e) {
				if (e.button == 3) {
					contextMenu.setVisible(true);

					table.setMenu(contextMenu);
				}
			}
		});
		table
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						final int index = table.indexOf((TableItem) e.item);
						
						if (!zip.isDirectory(index)) {
							// 파일인 경우
							view(index);
						}
					}

					@Override
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						int selectionCount = table.getSelectionCount();

						if (selectionCount == 0) {
							// 선택된 항목이 없는 경우
							pushDeselectAll.setEnabled(false);
							pushDelete.setEnabled(false);
						} else {
							// 선택된 항목이 있는 경우
							pushDeselectAll.setEnabled(true);
							pushDelete.setEnabled(true);
						}

						if (selectionCount == 1) {
							// 하나의 항목만 선택된 경우
							pushRename.setEnabled(true);
							itemRename.setEnabled(true);
							if (zip.isDirectory(table.getSelectionIndex())) {
								// 디렉토리인 경우
								itemView.setEnabled(false);
								itemOpen.setEnabled(false);
							} else {
								// 파일인 경우
								itemView.setEnabled(true);
								itemOpen.setEnabled(true);
							}
						} else {
							pushRename.setEnabled(false);
							itemRename.setEnabled(false);
							itemView.setEnabled(false);
							itemOpen.setEnabled(false);
						}

						if (selectionCount == table.getItemCount()) {
							// 모든 항목이 선택된 경우
							pushSelectAll.setEnabled(false);
						} else {
							pushSelectAll.setEnabled(true);
						}

						setStatusLine();
					}
				});
		tableColumnName = new TableColumn(table, SWT.LEFT);
		tableColumnName.setText("이름");
		tableColumnName.setWidth(230);
		tableColumnName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 이름순으로 정렬
						if (table.getSortColumn() == tableColumnName) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnName, true);

								table.setSortColumn(tableColumnName);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioName, true);
							}
						} else {
							sortTable(tableColumnName, false);

							table.setSortColumn(tableColumnName);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioName, false);
						}
					}
				});
		tableColumnSize = new TableColumn(table, SWT.RIGHT);
		tableColumnSize.setText("크기");
		tableColumnSize.setWidth(90);
		tableColumnSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 크기순으로 정렬
						if (table.getSortColumn() == tableColumnSize) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnSize, true);

								table.setSortColumn(tableColumnSize);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioSize, true);
							}
						} else {
							sortTable(tableColumnSize, false);

							table.setSortColumn(tableColumnSize);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioSize, false);
						}
					}
				});
		tableColumnType = new TableColumn(table, SWT.RIGHT);
		tableColumnType.setText("형식");
		tableColumnType.setWidth(80);
		tableColumnType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 형식순으로 정렬
						if (table.getSortColumn() == tableColumnType) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnType, true);

								table.setSortColumn(tableColumnType);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioType, true);
							}
						} else {
							sortTable(tableColumnType, false);

							table.setSortColumn(tableColumnType);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioSize, false);
						}
					}
				});
		tableColumnTime = new TableColumn(table, SWT.LEFT);
		tableColumnTime.setText("바뀐 시간");
		tableColumnTime.setWidth(200);
		tableColumnTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 바뀐 시간순으로 정렬
						if (table.getSortColumn() == tableColumnTime) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnTime, true);

								table.setSortColumn(tableColumnTime);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioTime, true);
							}
						} else {
							sortTable(tableColumnTime, false);

							table.setSortColumn(tableColumnTime);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioTime, false);
						}
					}
				});
		tableColumnPath = new TableColumn(table, SWT.LEFT);
		tableColumnPath.setWidth(160);
		tableColumnPath.setText("위치");
		tableColumnPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 위치순으로 정렬
						if (table.getSortColumn() == tableColumnPath) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnPath, true);

								table.setSortColumn(tableColumnPath);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioPath, true);
							}
						} else {
							sortTable(tableColumnPath, false);

							table.setSortColumn(tableColumnPath);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioPath, false);
						}
					}
				});
	}

	/**
	 * 선택된 항목을 삭제하는 메소드
	 */
	private void delete() {
		// 선택된 항목이 있는지 확인
		if (table.getSelectionCount() == 0) {
			return;
		}

		MessageBox messageBox = null;

		// 삭제 확인 대화상자
		messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
				| SWT.ICON_QUESTION);
		messageBox.setText("정말로 삭제할까요?");
		String message = "선택된 항목을 삭제합니다.\n한 번 삭제되면 다시 되돌릴 수 없습니다.\n\n계속 할까요?";
		messageBox.setMessage(message);

		if (messageBox.open() != SWT.YES) {
			return;
		}

		if (!zip.canWrite()) {
			// 쓰기 권한이 없는 경우
			final String text = "지우기 실패!";
			message = zip.getName() + " 파일에 대한 쓰기 권한이 없습니다.";
			messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		statusLine.setText("선택된 항목을 삭제하는 중입니다.");
		statusLine.update();

		int[] indices = table.getSelectionIndices(); // 선택된 항목의 인덱스들

		// 지우기
		delete(indices);

		// 새로 고침
		refresh();
	}

	/**
	 * 선택된 항목을 삭제하는 메소드
	 * 
	 * @param indices
	 *            삭제할 항목에 대한 인덱스의 배열
	 */
	private void delete(int[] indices) {
		if (!tempDir.exists()) {
			// 임시 디렉토리가 존재하지 않는 경우
			tempDir.mkdirs();
		}

		// 임시 Zip 파일 생성
		File tempFile = new File(tempDir, zip.getFileName());

		ZipOutputStream zos = null;
		try {
			zos = new ZipOutputStream(tempFile);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		zos.setEncoding(zip.getEncoding());

		for (int i = 0; i < zip.getSize(); i++) {
			// 삭제할 파일은 제외
			boolean delete = false;
			for (int j : indices) {
				if (i == j) {
					delete = true;
					break;
				}
			}

			if (!delete) {
				ZipEntry originalEntry = zip.getEntry(i);
				ZipEntry entry = new ZipEntry(originalEntry.getName());
				entry.setTime(originalEntry.getTime());

				// 엔트리 압축
				saveEntry(zos, entry, zip.getInputStream(i));
			}
		}

		try {
			zos.finish();
		} catch (IOException e) {
		}

		// 새로운 Zip 파일 생성
		copyFile(tempFile, zip.getFile());
	}

	/**
	 * 하위 디렉토리의 모든 파일까지 다 삭제하는 메소드
	 * 
	 * @param file
	 *            삭제할 파일이나 디렉토리
	 */
	private void deleteAllFiles(File file) {
		if (file.isDirectory()) {
			// 디렉토리인 경우
			File[] files = file.listFiles();

			if (files != null) {
				for (File f : files) {
					deleteAllFiles(f);
				}
			}
		}

		file.delete();
	}

	/**
	 * 모든 항목을 선택 해제하는 메소드
	 */
	private void deselectAll() {
		table.deselectAll();

		pushSelectAll.setEnabled(true);
		pushDeselectAll.setEnabled(false);

		pushRename.setEnabled(false);
		pushDelete.setEnabled(false);

		setStatusLine();
	}

	/**
	 * 메뉴를 비활성화하는 메소드
	 */
	private void disableMenus() {
		pushExtract.setEnabled(false);
		pushClose.setEnabled(false);
		pushRefresh.setEnabled(false);
		pushSelectAll.setEnabled(false);
		pushDeselectAll.setEnabled(false);
		pushProperty.setEnabled(false);
		pushSaveAs.setEnabled(false);
		pushAddFile.setEnabled(false);
		pushAddDirectory.setEnabled(false);
		pushRename.setEnabled(false);
		pushDelete.setEnabled(false);
		radioUtf8.setEnabled(false);
		radioMs949.setEnabled(false);
		submenuItemAlignment.setEnabled(false);
		submenuItemFormat.setEnabled(false);
	}

	/**
	 * 메뉴를 활성화하는 메소드
	 */
	private void enablemenus() {
		pushExtract.setEnabled(true);
		pushClose.setEnabled(true);
		pushRefresh.setEnabled(true);
		pushSelectAll.setEnabled(true);
		pushProperty.setEnabled(true);
		pushSaveAs.setEnabled(true);
		pushAddFile.setEnabled(true);
		pushAddDirectory.setEnabled(true);
		radioUtf8.setEnabled(true);
		radioMs949.setEnabled(true);
		submenuItemAlignment.setEnabled(true);
		submenuItemFormat.setEnabled(true);
	}

	/**
	 * 압축 파일을 푸는 메소드
	 */
	private void extract() {
		// 압축 해제 디렉토리 선택 대화상자
		DirectoryDialog dialog = new DirectoryDialog(sShell);
		dialog.setText("압축을 풀 디렉토리를 선택하세요.");
		dialog.setMessage("압축을 풀 디렉토리를 선택하세요.");
		dialog.setFilterPath(defaultPath);
		String directoryName = dialog.open();

		if (directoryName != null) {
			extract(directoryName);
		}
	}

	/**
	 * 인자로 주어진 디렉토리에 압축을 푸는 메소드
	 * 
	 * @param directory
	 *            디렉토리명
	 * @return 압축 해제 성공 여부
	 */
	private boolean extract(File directory) {
		int overwrite = OverwriteDialog.NO;

		if (table.getSelectionCount() >= 1) {
			// 선택된 항목이 있는 경우
			int[] selectionIndices = table.getSelectionIndices();
			for (int index : selectionIndices) {
				overwrite = extract(index, directory, overwrite);

				if (overwrite == OverwriteDialog.CANCEL) {
					return false;
				}
			}
		} else {
			// 선택된 항목이 없는 경우
			for (int i = 0; i < zip.getSize(); i++) {
				overwrite = extract(i, directory, overwrite);

				if (overwrite == OverwriteDialog.CANCEL) {
					return false;
				}
			}
		}

		// 압축 해제 성공
		return true;
	}

	/**
	 * {@link File}에 인덱스에 해당하는 {@link ZipEntry}의 압축을 푸는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @param entryFile
	 *            저장할 {@link File}
	 */
	private void extract(int index, File entryFile) {
		try {
			InputStream is = zip.getInputStream(index);
			FileOutputStream fos = new FileOutputStream(entryFile);

			byte[] buffer = new byte[BUFFER_SIZE];
			int read = 0;

			read = is.read(buffer);
			while (read != -1) {
				fos.write(buffer, 0, read);
				read = is.read(buffer);
			}

			is.close();
			fos.close();
		} catch (IOException e1) {
			e1.printStackTrace();
		}
	}

	/**
	 * 하나의 Zip 엔트리를 압축 해제하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @param directory
	 *            압축을 해제할 디렉토리
	 * @param overwrite
	 *            덮어쓰기 설정
	 * @return 덮어쓰기 설정
	 */
	private int extract(int index, File directory, int overwrite) {
		File entryFile = new File(directory, zip.getAbsoluteEntryName(index));

		MessageBox messageBox = null;
		final String text = "압축 해제 실패";
		String message = null;

		if (zip.isDirectory(index)) {
			// 디렉토리인 경우
			entryFile.mkdirs();

			return overwrite;
		} else {
			File parentDir = entryFile.getParentFile(); // 부모 디렉토리

			if (parentDir != null && !parentDir.exists()) {
				// 부모 디렉토리가 존재하지 않는 경우
				// 디렉토리 생성
				parentDir.mkdirs();

				return overwrite;
			}

			if (entryFile.exists()) {
				// 같은 이름의 파일이 존재하는 경우
				if (overwrite == OverwriteDialog.ALL_NO) {
					// 모두 아니오인 경우
					return overwrite;
				} else if (overwrite == OverwriteDialog.ALL_YES) {
					// 모두 예인 경우
					if (!entryFile.canWrite()) {
						// 쓰기 권한이 없는 경우
						message = entryFile.getAbsolutePath()
								+ " 파일에 대한 쓰기 권한이 없습니다.";
						messageBox = new MessageBox(sShell, SWT.OK
								| SWT.ICON_ERROR);
						messageBox.setText(text);
						messageBox.setMessage(message);
						messageBox.open();

						return overwrite;
					}
				} else {
					// 예거나 아니오인 경우
					// 파일 덮어쓰기 확인 대화상자
					OverwriteDialog overwriteDialog = new OverwriteDialog();
					overwrite = overwriteDialog.open(sShell, entryFile
							.getAbsolutePath());

					if (overwrite == OverwriteDialog.CANCEL) {
						// 취소인 경우
						message = "압축 해제가 취소되었습니다.";
						messageBox = new MessageBox(sShell, SWT.OK
								| SWT.ICON_INFORMATION);
						messageBox.setText("압축 해제 취소!");
						messageBox.setMessage(message);
						messageBox.open();

						return overwrite;
					} else if (overwrite == OverwriteDialog.YES
							|| overwrite == OverwriteDialog.ALL_YES) {
						// 예거나 모두 예인 경우
						if (!entryFile.canWrite()) {
							// 쓰기 권한이 없는 경우
							message = entryFile.getAbsolutePath()
									+ " 파일에 대한 쓰기 권한이 없습니다.";
							messageBox = new MessageBox(sShell, SWT.OK
									| SWT.ICON_ERROR);
							messageBox.setText(text);
							messageBox.setMessage(message);
							messageBox.open();

							return overwrite;
						}
					} else {
						// 아니오나 모두 아니오인 경우
						return overwrite;
					}
				}
			}

			// 압축 해제
			extract(index, entryFile);

			return overwrite;
		}
	}

	/**
	 * 인자로 주어진 디렉토리에 압축을 해제하는 메소드
	 * 
	 * @param directoryName
	 *            압축을 해제할 디렉토리
	 */
	private void extract(String directoryName) {
		File directory = new File(directoryName); // 압축을 풀 디렉토리

		MessageBox messageBox = null;
		String text = "압축 해제 실패!";
		String message = null;

		if (!directory.exists()) {
			// 해당 디렉토리가 존재하지 않는 경우
			// 디렉토리 생성
			if (!directory.mkdirs()) {
				// 디렉토리 생성 실패 대화상자
				message = directoryName + " 디렉토리 생성에 실패했습니다.";
				messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
				messageBox.setText(text);
				messageBox.setMessage(message);
				messageBox.open();

				return;
			}
		}

		if (!directory.canWrite()) {
			// 해당 디렉토리에 대한 쓰기 권한이 없는 경우
			message = directoryName + " 디렉토리에 대한 쓰기 권한이 없습니다.";
			messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		statusLine.setText("압축 해제 중입니다.");
		statusLine.update();

		if (extract(directory)) {
			// 압축 해제 성공
			// 압축 해제 완료 대화상자
			message = "압축 해제가 완료되었습니다.\n\n해제된 항목을 표시할까요?";
			messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
					| SWT.ICON_INFORMATION);
			messageBox.setText("압축 해제 성공!");
			messageBox.setMessage(message);
			if (messageBox.open() == SWT.YES) {
				Program.launch(directoryName);
			}
		}

		setStatusLine();
	}

	/**
	 * 상태 표시줄을 숨기는 메소드
	 */
	private void hideStatusLine() {
		FormData formData;

		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(100);
		table.setLayoutData(formData);

		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(0);
		formData.bottom = new FormAttachment(0);
		statusSeparator.setLayoutData(formData);

		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(0);
		formData.bottom = new FormAttachment(0);
		statusLine.setLayoutData(formData);

		sShell.layout();
	}

	/**
	 * 압축 파일을 여는 메소드
	 */
	private void open() {
		// 불러올 압축 파일 선택 대화상자
		FileDialog dialog = new FileDialog(sShell, SWT.OPEN);
		dialog.setText("불러올 압축 파일을 선택하세요.");
		dialog.setFilterPath(defaultPath);
		dialog.setFilterExtensions(new String[] { "*.zip", "*.jar", "*.*" });
		dialog.setFilterNames(new String[] { "Zip 파일 (*.zip)",
				"Jar 파일 (*.jar)", "모든 파일 (*.*)" });
		String fileName = dialog.open();

		if (fileName != null) {
			// 선택된 압축 파일 열기
			open(fileName);
		}
	}

	/**
	 * 인자로 주어진 압축 {@link File}을 여는 메소드
	 * 
	 * @param file
	 *            압축 {@link File}
	 */
	private void open(File file) {
		// 인코딩
		String encoding = null;
		if (radioMs949.getSelection()) {
			encoding = "MS949";
		} else {
			encoding = "UTF8";
		}

		ZipFile zipFile = null;
		try {
			zipFile = new ZipFile(file, encoding);
		} catch (IOException e1) {
			// 압축 파일 열기 실패 대화상자
			final String text = "압축 파일 열기 실패!";
			final String message = file.getAbsolutePath()
					+ " 파일을 열 수 없습니다.\n정상적인 Zip 파일이 아닙니다..";
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		zip = new Zip(file, zipFile);

		table.removeAll();
		table.setItemCount(zip.getSize());

		TableColumn sortColumn = null;
		if (radioSize.getSelection()) {
			sortColumn = tableColumnSize;
		} else if (radioType.getSelection()) {
			sortColumn = tableColumnType;
		} else if (radioTime.getSelection()) {
			sortColumn = tableColumnTime;
		} else if (radioPath.getSelection()) {
			sortColumn = tableColumnPath;
		} else {
			sortColumn = tableColumnName;
		}
		sortTable(sortColumn, checkReverse.getSelection());

		table.setVisible(true);

		// 메뉴 활성화
		enablemenus();

		// 상태 표시줄 설정
		setStatusLine();

		// 제목 표시줄 설정
		sShell.setText(zip.getFileName() + " - " + jzip);
	}

	/**
	 * 인자로 주어진 압축 파일을 여는 메소드
	 * 
	 * @param fileName
	 *            압축 파일의 이름
	 */
	private void open(String fileName) {
		// 불러올 압축 파일
		File file = new File(fileName);

		MessageBox messageBox = null;
		String text = "압축  파일 열기 실패!";
		String message = null;

		if (!file.exists()) {
			// 파일이 존재하지 않는 경우
			// 압축 파일 열기 실패 대화상자
			message = fileName + " 파일이 존재하지 않습니다.";
			messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		if (file.isDirectory()) {
			// 디렉토리인 경우 존재하지 않는 경우
			// 압축 파일 열기 실패 대화상자
			message = fileName + " 디렉토리는 파일이 아니라 디렉토리입니다.";
			messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		if (!file.canRead()) {
			// 파일에 대한 읽기 권한이 없는 경우
			// 압축 파일 열기 실패 대화상자
			message = fileName + " 파일에 대한 읽기 권한이 없습니다.";
			messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		// 압축 파일 열기
		open(file);
	}

	/**
	 * 현재의 압축 파일을 다시 읽어들이는 메소드
	 */
	private void refresh() {
		open(zip.getName());
	}

	/**
	 * 선택된 항목의 이름을 변경하는 메소드
	 */
	private void rename() {
		if (table.getSelectionCount() != 1) {
			// 선택된 항목이 하나가 아닌 경우
			return;
		}

		if (!zip.canWrite()) {
			// 쓰기 권한이 없는 경우
			String text = "이름 바꾸기 실패!";
			String message = zip.getName() + " 파일에 대한 쓰기 권한이 없습니다.";
			MessageBox messageBox = new MessageBox(sShell, SWT.OK
					| SWT.ICON_ERROR);
			messageBox.setText(text);
			messageBox.setMessage(message);
			messageBox.open();

			return;
		}

		int index = table.getSelectionIndex(); // 선택된 항목의 인덱스
		String originalName = zip.getEntryName(index); // 기존 이름

		// 사용자로부터 새로운 이름을 입력 받음
		RenameDialog renameDialog = new RenameDialog(originalName);
		String newName = renameDialog.open(sShell); // 새로운 이름

		if (newName == null || newName.equals(originalName)) {
			return;
		}

		statusLine.setText("선택된 항목의 이름을 바꾸는 중입니다.");
		statusLine.update();

		// 이름 변경
		rename(index, newName);

		// 새로 고침
		refresh();
	}

	/**
	 * 선택된 항목의 이름을 바꾸는 메소드
	 * 
	 * @param selectionIndex
	 *            이름을 바꿀 항목의 인덱스
	 * @param newName
	 *            새로운 이름
	 */
	private void rename(int selectionIndex, String newName) {
		if (!tempDir.exists()) {
			tempDir.mkdirs();
		}

		File tempFile = new File(tempDir, zip.getFileName());

		ZipOutputStream zos = null;
		try {
			zos = new ZipOutputStream(tempFile);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		zos.setEncoding(zip.getEncoding());

		for (int i = 0; i < zip.getSize(); i++) {
			ZipEntry originalEntry = zip.getEntry(i); // 원본 엔트리
			ZipEntry entry = null; // 새로운 엔트리

			if (i == selectionIndex) {
				// 이름 바꾸기
				entry = new ZipEntry(Zip.getNewName(originalEntry, newName));
				entry.setTime(originalEntry.getTime());
			} else {
				entry = new ZipEntry(originalEntry.getName());
				entry.setTime(originalEntry.getTime());
			}

			// 엔트리 압축
			saveEntry(zos, entry, zip.getInputStream(i));
		}

		try {
			zos.finish();
		} catch (IOException e) {
		}

		// 새로운 Zip 파일 생성
		copyFile(tempFile, zip.getFile());
	}

	/**
	 * 현재 열린 압축 파일을 다른 이름으로 저장하는 메소드
	 */
	private void saveAs() {
		// 압축 파일 선택 대화상자
		FileDialog dialog = new FileDialog(sShell, SWT.SAVE);
		dialog.setText("저장할 압축 파일을 선택하세요.");
		dialog.setFilterPath(defaultPath);
		dialog.setFilterExtensions(new String[] { "*.zip", "*.jar", "*.*" });
		dialog.setFilterNames(new String[] { "Zip 파일 (*.zip)",
				"Jar 파일 (*.jar)", "모든 파일 (*.*)" });
		String fileName = dialog.open();

		if (fileName != null) {
			if (saveAs(fileName)) {
				// 압축 파일 저장 완료 대화상자
				final String message = fileName
						+ " 파일 생성이 완료되었습니다.\n\n새로 생성된 파일을 불러올까요?";
				MessageBox messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
						| SWT.ICON_INFORMATION);
				messageBox.setText("압축 완료!");
				messageBox.setMessage(message);

				if (messageBox.open() == SWT.YES) {
					// 인코딩 설정 변경
					radioUtf8.setSelection(false);
					radioMs949.setSelection(true);
					
					// 압축 파일 열기
					open(fileName);
				}
			}
		}
	}

	/**
	 * 인자로 주어진 {@link File}에 압축 파일을 저장하는 메소드
	 * 
	 * @param file
	 *            {@link File}
	 * @return 압축 파일 저장 성공 여부
	 */
	private boolean saveAs(File file) {
		ZipOutputStream zos = null;
		try {
			zos = new ZipOutputStream(file);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		// 인코딩 설정
		zos.setEncoding("MS949");

		if (zip != null) {
			// 현재 열려있는 Zip 파일이 있는 경우
			if (table.getSelectionCount() >= 1) {
				// 선택된 항목이 있는 경우
				int[] selectionIndices = table.getSelectionIndices();

				for (int index : selectionIndices) {
					ZipEntry originalEntry = zip.getEntry(index);
					ZipEntry entry = new ZipEntry(originalEntry.getName());
					entry.setTime(originalEntry.getTime());

					// 엔트리 압축
					saveEntry(zos, entry, zip.getInputStream(index));
				}
			} else {
				// 선택된 항목이 없는 경우
				for (int i = 0; i < zip.getSize(); i++) {
					ZipEntry originalEntry = zip.getEntry(i);
					ZipEntry entry = new ZipEntry(originalEntry.getName());
					entry.setTime(originalEntry.getTime());

					// 엔트리 압축
					saveEntry(zos, entry, zip.getInputStream(i));
				}
			}
		}

		try {
			zos.finish();
		} catch (IOException e) {
		}

		// 압축 파일 저장 성공
		return true;
	}

	/**
	 * 인자로 주어진 파일명으로 압축 파일을 저장하는 메소드
	 * 
	 * @param fileName
	 *            저장할 압축 파일명
	 * @return 압축 파일 저장 성공 여부
	 */
	private boolean saveAs(String fileName) {
		File file = new File(fileName); // 저장할 압축 파일
		File parent = file.getParentFile(); // 저장할 압축파일에 대한 부모 디렉토리

		MessageBox messageBox = null;
		String text = "압축 파일 저장 실패!";
		String message = null;

		if (parent.exists()) {
			// 부모 디렉토리가 존재하는 경우
			if (!parent.canWrite()) {
				// 해당 디렉토리에 대한 쓰기 권한이 없는 경우
				message = parent.getAbsolutePath() + " 디렉토리에 대한 쓰기 권한이 없습니다.";
				messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
				messageBox.setText(text);
				messageBox.setMessage(message);
				messageBox.open();

				return false;
			}
		} else {
			// 부모 디렉토리가 존재하지 않는 경우
			if (!parent.mkdirs()) {
				// 디렉토리 생성에 실패한 경우
				message = parent.getAbsolutePath() + " 디렉토리 생성에 실패했습니다.";
				messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
				messageBox.setText(text);
				messageBox.setMessage(message);
				messageBox.open();

				return false;
			}
		}

		if (file.exists()) {
			// 같은 이름을 가진 파일이 이미 존재하는 경우
			messageBox = new MessageBox(sShell, SWT.YES | SWT.NO
					| SWT.ICON_QUESTION);
			messageBox.setText("파일을 덮어쓸까요?");
			messageBox.setMessage(fileName + " 파일이 이미 존재합니다.\n\n이 파일을 덮어쓸까요?");

			if (messageBox.open() == SWT.YES) {
				if (!file.canWrite()) {
					// 해당 파일에 대한 쓰기 권한이 없는 경우
					message = fileName + " 파일에 대한 쓰기 권한이 없습니다.";
					messageBox = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
					messageBox.setText(text);
					messageBox.setMessage(message);
					messageBox.open();

					return false;
				}
			} else {
				message = "압축 파일 저장이 취소되었습니다.";
				messageBox = new MessageBox(sShell, SWT.OK
						| SWT.ICON_INFORMATION);
				messageBox.setText("압축 파일 저장 취소!");
				messageBox.setMessage(message);
				messageBox.open();

				return false;
			}
		}

		statusLine.setText("압축하는 중입니다.");
		statusLine.update();

		if (saveAs(file)) {
			// 압축 파일 저장 성공
			setStatusLine();

			return true;
		} else {
			// 압축 파일 저장 실패
			setStatusLine();

			return false;
		}
	}

	/**
	 * 하나의 {@link ZipEntry}를 압축하는 메소드
	 * 
	 * @param zos
	 *            {@link ZipOutputStream}
	 * @param entry
	 *            {@link ZipEntry}
	 * @param is
	 *            {@link InputStream}
	 */
	private void saveEntry(ZipOutputStream zos, ZipEntry entry, InputStream is) {
		try {
			zos.putNextEntry(entry);

			if (is != null) {
				// 디렉토리가 아닌 경우
				byte[] buffer = new byte[BUFFER_SIZE];
				int read = 0;

				read = is.read(buffer);
				while (read != -1) {
					zos.write(buffer, 0, read);
					read = is.read(buffer);
				}

				is.close();
			}

			zos.closeEntry();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * 모두 항목을 선택하는 메소드
	 */
	private void selectAll() {
		table.selectAll();

		pushSelectAll.setEnabled(false);
		pushDeselectAll.setEnabled(true);

		pushRename.setEnabled(false);
		pushDelete.setEnabled(true);

		setStatusLine();
	}

	/**
	 * 정렬 메뉴를 선택하는 메소드
	 * 
	 * @param menu
	 *            선택할 정렬 메뉴
	 * @param reverse
	 *            역순이면 true, 아니면 false
	 */
	private void setAlignmentMenu(MenuItem menu, boolean reverse) {
		// 메뉴 선택
		menu.setSelection(true);

		// 다른 메뉴는 선택 해제
		if (menu != radioName) {
			radioName.setSelection(false);
		}
		if (menu != radioSize) {
			radioSize.setSelection(false);
		}
		if (menu != radioType) {
			radioType.setSelection(false);
		}
		if (menu != radioTime) {
			radioTime.setSelection(false);
		}
		if (menu != radioPath) {
			radioPath.setSelection(false);
		}

		// 역순 설정
		checkReverse.setSelection(reverse);
	}

	/**
	 * 문맥 메뉴를 설정하는 메소드
	 */
	private void setContextMenu() {
		contextMenu = new Menu(sShell, SWT.POP_UP);
		contextMenu.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});

		itemView = new MenuItem(contextMenu, SWT.PUSH);
		itemView.setText("파일 보기(&V)");
		itemOpen = new MenuItem(contextMenu, SWT.PUSH);
		itemOpen.setText("파일을 열 프로그램(&O)...");
		itemOpen.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("파일을 열 프로그램을 선택합니다.");
			}
		});
		itemOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						final int index = table.getSelectionIndex();
						final String fileName = zip.getEntryName(index);

						ProgramSelectDialog select = new ProgramSelectDialog();
						final String command = select.open(sShell, fileName);

						if (command != null) {
							if (!tempDir.exists()) {
								tempDir.mkdirs();
							}

							File file = new File(tempDir, fileName);
							extract(index, file);

							String[] cmd = { command, file.getAbsolutePath() };
							try {
								Runtime.getRuntime().exec(cmd);
							} catch (IOException e1) {
								e1.printStackTrace();
							}
						}
					}
				});
		itemView.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("선택된 파일을 엽니다.");
			}
		});
		itemView
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						final int index = table.getSelectionIndex();
						view(index);
					}
				});
		@SuppressWarnings("unused")
		MenuItem itemSeparator2 = new MenuItem(contextMenu, SWT.SEPARATOR);

		MenuItem itemExtract = new MenuItem(contextMenu, SWT.PUSH);
		itemExtract.setText("압축 풀기(&E)...");
		itemExtract.addArmListener(new ExtractArmListener());
		itemExtract.addSelectionListener(new ExtractSelectionListener());

		@SuppressWarnings("unused")
		MenuItem itemSeparator = new MenuItem(contextMenu, SWT.SEPARATOR);

		itemRename = new MenuItem(contextMenu, SWT.PUSH);
		itemRename.setText("이름 바꾸기(&R)...");
		itemRename.addArmListener(new RenameArmListener());
		itemRename.addSelectionListener(new RenameSelectionListener());

		MenuItem itemDelete = new MenuItem(contextMenu, SWT.PUSH);
		itemDelete.setText("지우기(&D)...");
		itemDelete.addArmListener(new DeleteArmListener());
		itemDelete.addSelectionListener(new DeleteSelectionListener());
	}

	/**
	 * DnD 기능을 설정하는 메소드
	 */
	private void setDnD() {
		// Drop
		int operations = DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK;
		Transfer[] types = new Transfer[] { FileTransfer.getInstance() };

		target = new DropTarget(sShell, operations);
		target.setTransfer(types);
		DropListener dropListener = new DropListener();
		target.addDropListener(dropListener);

		// Drag
		source = new DragSource(table, operations);
		source.setTransfer(types);
		source.addDragListener(new DragListener(dropListener));
	}

	/**
	 * 상태 표시줄의 텍스트를 설정하는 메소드
	 */
	private void setStatusLine() {
		String text = null;

		if (zip != null) {
			// Zip 파일이 열려 있는 경우
			text = "전체 " + zip.getSize() + " 항목 ("
					+ Zip.getSizeString(zip.getOriginalSize()) + ")";

			if (table.getSelectionCount() >= 1) {
				// 선택된 항목이 있는 경우
				long size = 0;
				int[] selectionIndices = table.getSelectionIndices();
				for (int index : selectionIndices) {
					size += zip.getEntrySize(index);
				}

				text += ", " + selectionIndices.length + " 항목 선택됨 ("
						+ Zip.getSizeString(size) + ")";
			}
		} else {
			// Zip 파일이 열려있지 않은 경우
			text = "";
		}

		statusLine.setText(text);
	}

	/**
	 * 압축 파일의 속성을 보여주는 메소드
	 */
	private void showProperty() {
		// 압축 정도 계산
		double ratio = (double) zip.getOriginalSize() / zip.getFileSize();

		NumberFormat nf = NumberFormat.getInstance();
		nf.setMaximumFractionDigits(2);
		StringBuffer message = new StringBuffer(70);
		int dateFormat = DateFormat.MEDIUM;
		if (radioLong.getSelection()) {
			dateFormat = DateFormat.LONG;
		} else if (radioShort.getSelection()){
			dateFormat = DateFormat.SHORT;
		}

		message.append("파일 이름 : " + zip.getFileName() + "\n");
		message.append("파일 경로 : " + zip.getFilePath() + "\n");
		message.append("바뀐 시간 : " + Zip.getTimeString(zip.getLastModified(), dateFormat)
				+ "\n");
		message
				.append("압축 크기 : " + Zip.getSizeString(zip.getFileSize())
						+ "\n");
		message.append("실제 크기 : " + Zip.getSizeString(zip.getOriginalSize())
				+ "\n");
		message.append("압축 정도 : " + nf.format(ratio) + "\n");
		message.append("항목 개수 : " + zip.getSize());

		String text = zip.getFileName() + " 속성";
		MessageBox messageBox = new MessageBox(sShell, SWT.OK
				| SWT.ICON_INFORMATION);
		messageBox.setText(text);
		messageBox.setMessage(message.toString());
		messageBox.open();
	}

	/**
	 * 상태 표시줄을 보이는 메소드
	 */
	private void showStatusLine() {
		FormData formData;

		formData = new FormData();
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(100);
		statusLine.setLayoutData(formData);

		formData = new FormData();
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(statusLine);
		statusSeparator.setLayoutData(formData);

		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(statusSeparator);
		table.setLayoutData(formData);

		sShell.layout();
	}

	/**
	 * 이름으로 정렬하는 메소드
	 */
	private void sortByName() {
		if (table.getSortColumn() == tableColumnName) {
			// 이미 정렬되어 있는 경우
			return;
		}

		boolean reverse = checkReverse.getSelection();

		sortTable(tableColumnName, reverse);

		table.setSortColumn(tableColumnName);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 위치로 정렬하는 메소드
	 */
	private void sortByPath() {
		if (table.getSortColumn() == tableColumnPath) {
			// 이미 정렬되어 있는 경우
			return;
		}

		boolean reverse = checkReverse.getSelection();

		sortTable(tableColumnPath, reverse);

		table.setSortColumn(tableColumnPath);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 크기로 정렬하는 메소드
	 */
	private void sortBySize() {
		if (table.getSortColumn() == tableColumnSize) {
			// 이미 정렬되어 있는 경우
			return;
		}

		boolean reverse = checkReverse.getSelection();

		sortTable(tableColumnSize, reverse);

		table.setSortColumn(tableColumnSize);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 바뀐 시간으로 정렬하는 메소드
	 */
	private void sortByTime() {
		if (table.getSortColumn() == tableColumnTime) {
			// 이미 정렬되어 있는 경우
			return;
		}

		boolean reverse = checkReverse.getSelection();

		sortTable(tableColumnTime, reverse);

		table.setSortColumn(tableColumnTime);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 형식으로 정렬하는 메소드
	 */
	private void sortByType() {
		if (table.getSortColumn() == tableColumnType) {
			return;
		}

		boolean reverse = checkReverse.getSelection();

		sortTable(tableColumnType, reverse);

		table.setSortColumn(tableColumnType);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 역순으로 정렬하는 메소드
	 */
	private void sortReverse() {
		boolean reverse = checkReverse.getSelection();

		sortTable(table.getSortColumn(), reverse);

		if (reverse) {
			table.setSortDirection(SWT.UP);
		} else {
			table.setSortDirection(SWT.DOWN);
		}
	}

	/**
	 * 항목을 정렬하는 메소드
	 * 
	 * @param 정렬
	 *            기준이 되는 테이블 컬럼
	 * @param 역순이면
	 *            true, 아니면 false
	 */
	private void sortTable(TableColumn col, boolean reverse) {
		zip.sort(col.getText(), reverse);

		table.clearAll();
	}

	/**
	 * 상태 표시줄을 보이거나 숨기는 메소드
	 */
	private void toggleStatusLine() {
		if (checkStatusLine.getSelection()) {
			// 상태 표시줄 보이기
			showStatusLine();
		} else {
			// 상태 표시줄 숨기기
			hideStatusLine();
		}
	}

	/**
	 * 인덱스에 해당하는 파일을 여는 메소드
	 * 
	 * @param index
	 *            선택된 파일의 인덱스
	 */
	private void view(int index) {
		if (!tempDir.exists()) {
			tempDir.mkdirs();
		}

		File file = new File(tempDir, zip.getEntryName(index));
		extract(index, file);

		Program.launch(file.getAbsolutePath());
	}
}
// @jve:decl-index=0:
