/**
 * Zip : Zip 파일에 대한 정보를 담고 있는 클래스
 */
package net.kldp.jzip;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.zip.ZipException;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipOutputStream;

/**
 * Zip 파일에 대한 정보를 담고 있는 클래스
 * 
 * @author jeongseungwon
 * 
 */
public class Zip {
	/**
	 * {@link ZipEntry}의 배열을 이름으로 비교하는 {@link Comparator}
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class NameComparator implements Comparator<ZipEntry> {
		private boolean reverse;

		private NameComparator(boolean reverse) {
			this.reverse = reverse;
		}

		public int compare(ZipEntry o1, ZipEntry o2) {
			String name1 = getEntryName(o1);
			String name2 = getEntryName(o2);

			if (!reverse) {
				return name1.compareTo(name2);
			} else {
				return name2.compareTo(name1);
			}
		}
	}

	/**
	 * {@link ZipEntry}의 배열을 위치로 비교하는 {@link Comparator}
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class PathComparator implements Comparator<ZipEntry> {
		private boolean reverse;

		private PathComparator(boolean reverse) {
			this.reverse = reverse;
		}

		public int compare(ZipEntry o1, ZipEntry o2) {
			String path1 = getEntryPath(o1);
			String path2 = getEntryPath(o2);

			if (!reverse) {
				return path1.compareTo(path2);
			} else {
				return path2.compareTo(path1);
			}
		}
	}

	/**
	 * {@link ZipEntry}의 배열을 크기로 비교하는 {@link Comparator}
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class SizeComparator implements Comparator<ZipEntry> {
		private boolean reverse;

		private SizeComparator(boolean reverse) {
			this.reverse = reverse;
		}

		public int compare(ZipEntry o1, ZipEntry o2) {
			long size1 = o1.getSize();
			long size2 = o2.getSize();

			if (size1 == size2) {
				return 0;
			}

			if (!reverse) {
				return size1 > size2 ? 1 : -1;
			}

			return size1 < size2 ? 1 : -1;
		}
	}

	/**
	 * {@link ZipEntry}의 배열을 바뀐 시간으로 비교하는 {@link Comparator}
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class TimeComparator implements Comparator<ZipEntry> {
		private boolean reverse;

		private TimeComparator(boolean reverse) {
			this.reverse = reverse;
		}

		public int compare(ZipEntry o1, ZipEntry o2) {
			long time1 = o1.getTime();
			long time2 = o2.getTime();

			if (time1 == time2) {
				return 0;
			}

			if (!reverse) {
				return time1 > time2 ? 1 : -1;
			} else {
				return time1 < time2 ? 1 : -1;
			}
		}
	}

	/**
	 * {@link ZipEntry}의 배열을 형식으로 비교하는 {@link Comparator}
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class TypeComparator implements Comparator<ZipEntry> {
		private boolean reverse;

		private TypeComparator(boolean reverse) {
			this.reverse = reverse;
		}

		public int compare(ZipEntry o1, ZipEntry o2) {
			String type1 = null;
			if (o1.isDirectory()) {
				type1 = "디렉토리";
			} else {
				type1 = "파일";
			}

			String type2 = null;
			if (o2.isDirectory()) {
				type2 = "디렉토리";
			} else {
				type2 = "파일";
			}

			if (!reverse) {
				return type1.compareTo(type2);
			} else {
				return type2.compareTo(type1);
			}
		}
	}

	/**
	 * 기존 {@link ZipEntry}의 이름을 새로운 {@link ZipEntry} 이름으로 바꿔서 반환하는 메소드
	 * 
	 * @param originalEntry
	 *            기존 {@link ZipEntry}
	 * @param newName
	 *            새로운 이름
	 * @return 새로운 {@link ZipEntry} 이름
	 */
	public static String getNewName(ZipEntry originalEntry, String newName) {
		File file = new File(originalEntry.getName());
		String parent = file.getParent();

		if (parent == null) {
			return newName;
		} else {
			File newFile = new File(parent, newName);

			return newFile.getPath();
		}
	}

	/**
	 * 파일 크기를 문자열로 반환하는 메소드
	 * 
	 * @param size
	 *            파일 크기
	 * @return 파일 크기를 표현하는 문자열
	 */
	public static String getSizeString(long size) {
		final int KB = 1024;
		final int MB = 1024 * KB;
		final int GB = 1024 * MB;

		double newSize = size;
		String sizeString = "";
		NumberFormat nf = NumberFormat.getInstance();
		nf.setMaximumFractionDigits(2);

		if (size >= GB) {
			newSize /= GB;
			sizeString = nf.format(newSize) + " GB";
		} else if (size >= MB) {
			newSize /= MB;
			sizeString = nf.format(newSize) + " MB";
		} else if (size >= KB) {
			newSize /= KB;
			sizeString = nf.format(newSize) + " KB";
		} else {
			sizeString = size + " 바이트";
		}

		return sizeString;
	}

	/**
	 * 시간을 문자열로 반환하는 메소드
	 * 
	 * @param time
	 *            시간
	 * @return 시간을 표현하는 문자열
	 */
	public static String getTimeString(long time, int dateFormat) {
		String timeString = null;

		if (time != -1) {
			Date date = new Date(time);
			DateFormat df = DateFormat.getDateTimeInstance(dateFormat, dateFormat);
			timeString = df.format(date);
		} else {
			timeString = "알 수 없음";
		}

		return timeString;
	}

	/**
	 * {@link ZipFile}에 대한 {@link File} 객체
	 */
	private File file;

	/**
	 * {@link ZipEntry}의 배열
	 */
	private ZipEntry[] zipEntries;

	/**
	 * {@link ZipFile}
	 */
	private ZipFile zipFile;
	
	/**
	 * Zip 클래스의 생성자
	 * 
	 * @param file
	 *            Zip {@link File}
	 * @param zipFile
	 *            {@link ZipFile}
	 */
	public Zip(File file, ZipFile zipFile) {
		this.file = file; // File
		this.zipFile = zipFile; // ZipFile

		Enumeration entries = zipFile.getEntries();

		int i = 0;

		for (i = 0; entries.hasMoreElements(); i++) {
			entries.nextElement();
		}

		zipEntries = new ZipEntry[i]; // ZipEntry의 배열

		entries = zipFile.getEntries();

		for (i = 0; entries.hasMoreElements(); i++) {
			ZipEntry entry = (ZipEntry) entries.nextElement();

			zipEntries[i] = entry;
		}
	}

	/**
	 * Zip {@link File}에 쓰기 권한이 있는지 확인하는 메소드
	 * 
	 * @return Zip {@link File}에 쓰기 권한이 있으면 true, 없으면 false
	 */
	public boolean canWrite() {
		return file.canWrite();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 이름(경로 포함)을 반환하는 메소드
	 * 
	 * @param index	{@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 이름(경로 포함)
	 */
	public String getAbsoluteEntryName(int index) {
		return getEntry(index).getName();
	}

	/**
	 * {@link ZipFile}의 인코딩을 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 인코딩
	 */
	public String getEncoding() {
		return zipFile.getEncoding();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}를 반환하는 메소드
	 * 
	 * @param i
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return {@link ZipEntry}
	 */
	public ZipEntry getEntry(int i) {
		return zipEntries[i];
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 이름(경로 포함 안함)을 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 이름(경로 포함 안함)
	 */
	public String getEntryName(int index) {
		File file = new File(getEntry(index).getName());

		return file.getName();
	}

	/**
	 * 해당 {@link ZipEntry}에 대한 이름(경로를 포함하지 않음)을 반환하는 메소드
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return 해당 {@link ZipEntry}에 대한 이름(경로를 포함하지 않음)
	 */
	private String getEntryName(ZipEntry entry) {
		File file = new File(entry.getName());

		return file.getName();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 경로를 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 경로
	 */
	private String getEntryPath(int index) {
		File file = new File(getEntry(index).getName());
		String parent = file.getParent();

		if (parent != null) {
			return "/" + parent;
		} else {
			return "/";
		}
	}

	/**
	 * 해당 {@link ZipEntry}에 대한 경로를 반환하는 메소드
	 * 
	 * @param entry
	 *            {@link ZipEntry}
	 * @return 해당 {@link ZipEntry}에 대한 경로
	 */
	private String getEntryPath(ZipEntry entry) {
		File file = new File(entry.getName());
		String parent = file.getParent();

		if (parent != null) {
			return "/" + parent;
		} else {
			return "/";
		}
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 크기를 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 크기
	 */
	public long getEntrySize(int index) {
		return getEntry(index).getSize();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 바뀐 시간을 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 바뀐 시간
	 */
	private long getEntryTime(int index) {
		return getEntry(index).getTime();
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}의 형태를 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 인덱스에 해당하는 {@link ZipEntry}의 형태
	 */
	private String getEntryType(int index) {
		String type = null;

		if (getEntry(index).isDirectory()) {
			type = "디렉토리";
		} else {
			type = "파일";
		}

		return type;
	}

	/**
	 * Zip {@link File}을 반환하는 메소드
	 * 
	 * @return Zip {@link File}
	 */
	public File getFile() {
		return file;
	}

	/**
	 * {@link ZipFile}의 파일명을 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 파일명
	 */
	public String getFileName() {
		return file.getName();
	}

	/**
	 * {@link ZipFile}의 경로를 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 경로
	 */
	public String getFilePath() {
		return file.getParent();
	}

	/**
	 * {@link ZipFile}의 압축 크기를 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 압축 크기
	 */
	public long getFileSize() {
		return file.length();
	}

	/**
	 * {@link ZipEntry}에 해당하는 {@link InputStream}을 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry}
	 * @return {@link ZipEntry}에 해당하는 {@link InputStream}
	 */
	public InputStream getInputStream(int index) {
		InputStream is = null;

		try {
			is = zipFile.getInputStream(getEntry(index));
		} catch (ZipException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}

		return is;
	}

	/**
	 * {@link ZipFile}의 바뀐 시간을 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 바뀐 시간
	 */
	public long getLastModified() {
		return file.lastModified();
	}

	/**
	 * {@link ZipFile}의 이름을 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 이름
	 */
	public String getName() {
		return file.getAbsolutePath();
	}

	/**
	 * {@link ZipFile}의 실제 크기를 반환하는 메소드
	 * 
	 * @return {@link ZipFile}의 실제 크기
	 */
	public long getOriginalSize() {
		long zipFileSize = 0;

		for (int i = 0; i < zipEntries.length; i++) {
			zipFileSize += getEntrySize(i);
		}

		return zipFileSize;
	}

	/**
	 * {@link ZipEntry}의 숫자를 반환하는 메소드
	 * 
	 * @return {@link ZipEntry}의 숫자
	 */
	public int getSize() {
		return zipEntries.length;
	}

	/**
	 * 테이블에 표시하기 위해, 인덱스에 해당하는 {@link ZipEntry}의 정보를 텍스트로 반환하는 메소드
	 * 
	 * @param index
	 *            {@link ZipEntry} 배열의 인덱스
	 * @return 텍스트로 표현된 {@link ZipEntry}의 정보
	 */
	public String[] getStrings(int index, int dateFormat) {
		String name = getEntryName(index);	// 이름
		String size = getSizeString(getEntrySize(index));	// 크기
		String type = getEntryType(index); // 형식
		String time = getTimeString(getEntryTime(index), dateFormat);	// 바뀐 시간
		String path = getEntryPath(index);	// 위치

		return new String[] { name, size, type, time, path };
	}

	/**
	 * {@link ZipOutputStream}을 반환하는 메소드
	 * 
	 * @return {@link ZipOutputStream}
	 */
	public ZipOutputStream getZipOutputStream() {
		ZipOutputStream zos = null;

		try {
			zos = new ZipOutputStream(file);
		} catch (IOException e) {
			e.printStackTrace();
		}

		return zos;
	}

	/**
	 * 인덱스에 해당하는 {@link ZipEntry}가 디렉토리이면 true, 아니면 false를 반환하는 메소드
	 * 
	 * @param index {@link ZipEntry} 배열의 인덱스
	 * @return 디렉토리 여부
	 */
	public boolean isDirectory(int index) {
		return getEntry(index).isDirectory();
	}

	/**
	 * {@link ZipEntry}의 배열을 정렬하는 메소드
	 * 
	 * @param string
	 *            정렬 기준
	 * @param reverse
	 *            정렬 방향
	 */
	public void sort(String string, boolean reverse) {
		if (string.equals("이름")) {
			// 이름으로 정렬
			Arrays.sort(zipEntries, new NameComparator(reverse));
		} else if (string.equals("크기")) {
			// 크기로 정렬
			Arrays.sort(zipEntries, new SizeComparator(reverse));
		} else if (string.equals("형식")) {
			// 형식으로 정렬
			Arrays.sort(zipEntries, new TypeComparator(reverse));
		} else if (string.equals("바뀐 시간")) {
			// 바뀐 시간으로 정렬
			Arrays.sort(zipEntries, new TimeComparator(reverse));
		} else if (string.equals("위치")) {
			// 위치로 정렬
			Arrays.sort(zipEntries, new PathComparator(reverse));
		}
	}
}
