/**
 * ZipEntryData : Zip 엔트리 정보 클래스
 */
package com.tistory.jeongsw.jzip;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.util.Date;

import net.sf.jazzlib.ZipEntry;

/**
 * @author jeongseungwon
 * 
 */
public class ZipEntryData {
	/**
	 * Zip 엔트리의 크기를 문자열로 반환함
	 * 
	 * @param size
	 *            Zip 엔트리의 크기
	 * @return 크기를 표현하는 문자열
	 */
	public static String getSizeString(long size) {
		final int KB = 1024;
		final int MB = 1024 * KB;
		final int GB = 1024 * MB;

		double newSize = size;
		String sizeString = "";
		NumberFormat nf = NumberFormat.getInstance();
		nf.setMaximumFractionDigits(2);

		if (size >= GB) {
			newSize /= GB;
			sizeString = nf.format(newSize) + " GB";
		} else if (size >= MB) {
			newSize /= MB;
			sizeString = nf.format(newSize) + " MB";
		} else if (size >= KB) {
			newSize /= KB;
			sizeString = nf.format(newSize) + " KB";
		} else {
			sizeString = size + " 바이트";
		}

		return sizeString;
	}

	private ZipEntry zipEntry; // Zip 엔트리

	/**
	 * ZipEntryData의 생성자
	 * 
	 * @param entry
	 *            Zip 엔트리
	 */
	public ZipEntryData(ZipEntry entry) {
		setZipEntry(entry);

	}

	/**
	 * 문자열을 해독하는 메소드
	 * 
	 * @param input
	 *            해독할 문자열
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @return 해독된 문자열
	 */
	private String decodeString(String input, boolean unicode) {
		String output = null;

		try {
			byte[] bytes = input.getBytes("ISO-8859-1");
			if (unicode) {
				output = new String(bytes, "UTF-8");
			} else {
				output = new String(bytes, "MS949");
			}
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}

		return output;
	}

	/**
	 * Zip 엔트리의 파일명을 반환하는 메소드
	 * 
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @return Zip 엔트리의 파일명
	 */
	public String getEntryName(boolean unicode) {
		return decodeString(zipEntry.getName(), unicode);
	}

	/**
	 * Zip 엔트리의 이름을 반환하는 메소드
	 * 
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @return Zip 엔트리의 이름
	 */
	public String getName(boolean unicode) {
		File file = new File(zipEntry.getName());

		return decodeString(file.getName(), unicode);
	}

	/**
	 * Zip 엔트리의 경로를 반환하는 메소드
	 * 
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @return Zip 엔트리의 경로
	 */
	public String getPath(boolean unicode) {
		File file = new File(zipEntry.getName());
		String parent = file.getParent();

		if (parent != null) {
			return "/" + decodeString(parent, unicode);
		} else {
			return "/";
		}
	}

	/**
	 * Zip 엔트리의 크기를 반환하는 메소드
	 * 
	 * @return Zip 엔트리의 크기
	 */
	public long getSize() {
		return zipEntry.getSize();
	}

	/**
	 * 테이블에 표시하기 위해, Zip 엔트리의 정보를 텍스트로 반환하는 메소드
	 * 
	 * @param unicode
	 * @return Zip 엔트리의 정보
	 */
	public String[] getStrings(boolean unicode) {
		// 이름
		String name = getName(unicode);

		// 크기
		String size = getSizeString(getSize());

		// 형식
		String type = getType();

		// 바뀐 시간
		String time;
		long lTime = getTime();
		if (lTime != -1) {
			Date date = new Date(lTime);
			DateFormat df = DateFormat.getDateTimeInstance(DateFormat.LONG,
					DateFormat.LONG);
			time = df.format(date);
		} else {
			time = "알 수 없음";
		}

		// 위치
		String path = getPath(unicode);

		return new String[] { name, size, type, time, path };
	}

	public long getTime() {
		return zipEntry.getTime();
	}

	/**
	 * Zip 엔트리의 형식을 문자열로 반환하는 메소드
	 * 
	 * @return 디렉토리이면 "디렉토리", 파일이면 "파일"
	 */
	public String getType() {
		String type = null;

		if (zipEntry.isDirectory()) {
			type = "디렉토리";
		} else {
			type = "파일";
		}

		return type;
	}

	/**
	 * Zip 엔트리를 반환하는 메소드
	 * 
	 * @return Zip 엔트리
	 */
	public ZipEntry getZipEntry() {
		return zipEntry;
	}

	/**
	 * Zip 엔트리를 설정하는 메소드
	 * 
	 * @param zipEntry
	 *            Zip 엔트리
	 */
	public void setZipEntry(ZipEntry zipEntry) {
		this.zipEntry = zipEntry;
	}
}
