/**
 * JZip 0.1
 * Zip 파일의 압축을 해제하는 프로그램입니다.
 * 정승원 jeongseungwon@hanmail.net
 */
package com.tistory.jeongsw.jzip;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Enumeration;

import net.sf.jazzlib.ZipEntry;
import net.sf.jazzlib.ZipFile;

import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSource;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTarget;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.FileTransfer;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.layout.FormLayout;

/**
 * JZip 프로그램의 메인 클래스
 * 
 * @author jeongseungwon
 * 
 */
public class JZip {
	/**
	 * {@link DropTargetListener}를 구현하는 클래스
	 * 
	 * @author jeongseungwon
	 * 
	 */
	private final class DropListener implements DropTargetListener {
		public void dragEnter(DropTargetEvent event) {
		}

		public void dragLeave(DropTargetEvent event) {
		}

		public void dragOperationChanged(DropTargetEvent event) {
		}

		public void dragOver(DropTargetEvent event) {
		}

		public void drop(DropTargetEvent event) {
			if (event.data == null) {
				event.detail = DND.DROP_NONE;
				return;
			}

			String[] files = (String[]) event.data;

			if (files.length != 1) {
				event.detail = DND.DROP_NONE;
				return;
			} else {
				// 압축 파일 열기
				openArchive(files[0]);
			}
		}

		public void dropAccept(DropTargetEvent event) {
		}
	}

	/**
	 * JZip 프로그램의 메인 클래스
	 * 
	 * @param args
	 *            불러올 Zip 파일의 이름
	 */
	public static void main(String[] args) {
		Display display = Display.getDefault();
		final JZip thisClass = new JZip();
		thisClass.createSShell();

		// 임시 디렉토리 설정
		thisClass.tempDirectory = new File(
				System.getProperty("java.io.tmpdir"), "JZip");

		// 기본 인코딩 설정
		if (System.getProperty("os.name") == "Windows") {
			thisClass.radioMs949.setSelection(true);
			thisClass.radioUtf8.setSelection(false);
		} else {
			thisClass.radioMs949.setSelection(false);
			thisClass.radioUtf8.setSelection(true);
		}

		// 레이아웃 설정
		FormData formData;

		formData = new FormData();
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(100);
		thisClass.statusLine.setLayoutData(formData);

		formData = new FormData();
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(thisClass.statusLine);
		thisClass.statusSeparator.setLayoutData(formData);

		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(0);
		formData.right = new FormAttachment(100);
		formData.bottom = new FormAttachment(thisClass.statusSeparator);
		thisClass.table.setLayoutData(formData);

		// DND 설정
		thisClass.setDND();

		if (args.length == 1) {
			// 인자로 주어진 압축 파일을 불러옴
			thisClass.openArchive(args[0]);
		}

		thisClass.sShell.open();

		while (!thisClass.sShell.isDisposed()) {
			if (!display.readAndDispatch())
				display.sleep();
		}
		display.dispose();
	}

	private final int BUFFER_SIZE = 1024 * 128; // 버퍼 크기(128 KB)

	private Shell sShell = null; // @jve:decl-index=0:visual-constraint="9,10"

	private Menu menuBar = null;

	private Menu submenuArchive = null;

	private Menu submenuHelp = null;

	private MenuItem pushExtract = null;

	private Label statusLine = null;

	private Label statusSeparator = null;

	private Table table = null;

	private ZipFile zipFile = null; // @jve:decl-index=0:

	private Menu submenuView = null;

	private MenuItem radioUtf8 = null;

	private MenuItem radioMs949 = null;

	private MenuItem checkStatusLine = null;

	private TableColumn tableColumnName = null;

	private TableColumn tableColumnSize = null;

	private TableColumn tableColumnType = null;

	private TableColumn tableColumnTime = null;

	private TableColumn tableColumnPath = null;

	private Menu submenuAlignment = null;

	private MenuItem radioName = null;

	private MenuItem checkReverse = null;

	private MenuItem radioPath = null;

	private MenuItem radioTime = null;

	private MenuItem radioType = null;

	private MenuItem radioSize = null;

	private MenuItem submenuItemAlignment = null;

	private MenuItem pushClose = null;

	private MenuItem pushRefresh = null;

	private ZipEntryData[] zipEntries = null; // Zip 엔트리 정보 배열

	private Menu submenuEdit = null;

	private MenuItem pushSelectAll = null;

	private File tempDirectory = null;

	private DragSource source;

	private DropTarget target;

	/**
	 * 압축 파일을 닫는 메소드
	 */
	private void closeArchive() {
		if (zipFile != null) {
			try {
				zipFile.close();
			} catch (IOException e1) {
				e1.printStackTrace();
			}
		}

		zipFile = null;
		zipEntries = null;

		table.removeAll();
		table.setVisible(false);

		setStatusLine();

		// 메뉴 비활성화
		pushExtract.setEnabled(false);
		pushClose.setEnabled(false);
		pushRefresh.setEnabled(false);
		pushSelectAll.setEnabled(false);
		submenuItemAlignment.setEnabled(false);
		radioUtf8.setEnabled(false);
		radioMs949.setEnabled(false);
	}

	/**
	 * This method initializes sShell
	 */
	private void createSShell() {
		sShell = new Shell();
		sShell.setText("JZip 0.1");
		sShell.setLayout(new FormLayout());
		sShell.setSize(new Point(720, 400));
		menuBar = new Menu(sShell, SWT.BAR);
		MenuItem submenuItemArchive = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemArchive.setText("압축 파일(&A)");
		MenuItem submenuItemEdit = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemEdit.setText("편집(&E)");
		submenuEdit = new Menu(submenuItemEdit);
		submenuEdit.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		pushSelectAll = new MenuItem(submenuEdit, SWT.PUSH);
		pushSelectAll.setText("모두 선택(&A)\tCtrl+A");
		pushSelectAll.setEnabled(false);
		pushSelectAll.setAccelerator(SWT.CTRL | 'A');
		pushSelectAll
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSelectionCount() == table.getItemCount()) {
							// 모두 선택 해제
							table.deselectAll();
						} else {
							// 모두 선택
							table.selectAll();
						}

						setStatusLine();
					}
				});
		pushSelectAll.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("모든 항목을 선택하거나, 선택 해제합니다.");
			}
		});
		submenuItemEdit.setMenu(submenuEdit);
		MenuItem submenuItemView = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemView.setText("보기(&V)");
		MenuItem submenuItemHelp = new MenuItem(menuBar, SWT.CASCADE);
		submenuItemHelp.setText("도움말(&H)");
		submenuView = new Menu(submenuItemView);
		pushRefresh = new MenuItem(submenuView, SWT.PUSH);
		pushRefresh.setText("새로 고침(&R)\tF5");
		pushRefresh.setEnabled(false);
		pushRefresh.setAccelerator(SWT.F5);
		pushRefresh
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						refresh();
					}
				});
		pushRefresh.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 다시 불러옵니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem separator1 = new MenuItem(submenuView, SWT.SEPARATOR);
		submenuView.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		radioUtf8 = new MenuItem(submenuView, SWT.RADIO);
		radioUtf8.setText("UTF-8 (리눅스)(&U)");
		radioUtf8.setEnabled(false);
		radioUtf8
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (!radioUtf8.getSelection()) {
							refresh();
						}
					}
				});
		radioUtf8.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("인코딩을 UTF-8(리눅스)로 변경합니다.");
			}
		});
		radioMs949 = new MenuItem(submenuView, SWT.RADIO);
		radioMs949.setText("MS949 (윈도우)(&M)");
		radioMs949.setEnabled(false);
		radioMs949
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (!radioMs949.getSelection()) {
							refresh();
						}
					}
				});
		radioMs949.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("인코딩을 MS949(윈도우)로 변경합니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator1 = new MenuItem(submenuView, SWT.SEPARATOR);
		checkStatusLine = new MenuItem(submenuView, SWT.CHECK);
		checkStatusLine.setText("상태 표시줄(&S)");
		checkStatusLine.setSelection(true);
		@SuppressWarnings("unused")
		MenuItem separator = new MenuItem(submenuView, SWT.SEPARATOR);
		submenuItemAlignment = new MenuItem(submenuView, SWT.CASCADE);
		submenuItemAlignment.setText("항목 정렬(&A)");
		submenuItemAlignment.setEnabled(false);
		submenuAlignment = new Menu(submenuItemAlignment);
		radioName = new MenuItem(submenuAlignment, SWT.RADIO);
		radioName.setText("이름(&N)");
		radioName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() != tableColumnName) {
							// 이름순으로 정렬
							boolean reverse = checkReverse.getSelection();

							sortTable(tableColumnName, reverse);

							table.setSortColumn(tableColumnName);

							if (reverse) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioName.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("이름에 따라 정렬합니다.");
			}
		});
		radioSize = new MenuItem(submenuAlignment, SWT.RADIO);
		radioSize.setText("크기(&S)");
		radioSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() != tableColumnSize) {
							// 크기순으로 정렬
							boolean reverse = checkReverse.getSelection();

							sortTable(tableColumnSize, reverse);

							table.setSortColumn(tableColumnSize);

							if (reverse) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioSize.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("크기에 따라 정렬합니다.");
			}
		});
		radioType = new MenuItem(submenuAlignment, SWT.RADIO);
		radioType.setText("형식(&T)");
		radioType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() != tableColumnType) {
							// 형식순으로 정렬
							boolean reverse = checkReverse.getSelection();

							sortTable(tableColumnType, reverse);

							table.setSortColumn(tableColumnType);

							if (reverse) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioType.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("형식에 따라 정렬합니다.");
			}
		});
		radioTime = new MenuItem(submenuAlignment, SWT.RADIO);
		radioTime.setText("바뀐 시간(&D)");
		radioTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() != tableColumnTime) {
							// 바뀐 시간순으로 정렬
							boolean reverse = checkReverse.getSelection();

							sortTable(tableColumnTime, reverse);

							table.setSortColumn(tableColumnTime);

							if (reverse) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioTime.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("바뀐 시간에 따라 정렬합니다.");
			}
		});
		radioPath = new MenuItem(submenuAlignment, SWT.RADIO);
		radioPath.setText("위치(&P)");
		radioPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						if (table.getSortColumn() != tableColumnPath) {
							// 위치순으로 정렬
							boolean reverse = checkReverse.getSelection();

							sortTable(tableColumnPath, reverse);

							table.setSortColumn(tableColumnPath);

							if (reverse) {
								table.setSortDirection(SWT.UP);
							} else {
								table.setSortDirection(SWT.DOWN);
							}
						}
					}
				});
		radioPath.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("위치에 따라 정렬합니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator3 = new MenuItem(submenuAlignment, SWT.SEPARATOR);
		checkReverse = new MenuItem(submenuAlignment, SWT.CHECK);
		checkReverse.setText("역순(&R)");
		checkReverse
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						TableColumn col = table.getSortColumn();
						boolean reverse = checkReverse.getSelection();

						if (col == null) {
							col = tableColumnName;
						}

						sortTable(col, reverse);

						table.setSortColumn(col);

						if (reverse) {
							table.setSortDirection(SWT.UP);
						} else {
							table.setSortDirection(SWT.DOWN);
						}

					}
				});
		checkReverse.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("거꾸로 정렬합니다.");
			}
		});
		submenuItemAlignment.setMenu(submenuAlignment);
		submenuItemAlignment
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("항목을 정렬합니다.");
					}
				});
		checkStatusLine
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						FormData formData;

						if (checkStatusLine.getSelection()) {
							formData = new FormData();
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(100);
							formData.bottom = new FormAttachment(100);
							statusLine.setLayoutData(formData);

							formData = new FormData();
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(100);
							formData.bottom = new FormAttachment(statusLine);
							statusSeparator.setLayoutData(formData);

							formData = new FormData();
							formData.top = new FormAttachment(0);
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(100);
							formData.bottom = new FormAttachment(
									statusSeparator);
							table.setLayoutData(formData);
						} else {
							formData = new FormData();
							formData.top = new FormAttachment(0);
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(100);
							formData.bottom = new FormAttachment(100);
							table.setLayoutData(formData);

							formData = new FormData();
							formData.top = new FormAttachment(0);
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(0);
							formData.bottom = new FormAttachment(0);
							statusSeparator.setLayoutData(formData);

							formData = new FormData();
							formData.top = new FormAttachment(0);
							formData.left = new FormAttachment(0);
							formData.right = new FormAttachment(0);
							formData.bottom = new FormAttachment(0);
							statusLine.setLayoutData(formData);
						}

						sShell.layout();
					}
				});
		checkStatusLine
				.addArmListener(new org.eclipse.swt.events.ArmListener() {
					public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
						statusLine.setText("상태 표시줄을 보이거나 감춥니다.");
					}
				});
		submenuItemView.setMenu(submenuView);
		submenuHelp = new Menu(submenuItemHelp);
		submenuHelp.addMenuListener(new org.eclipse.swt.events.MenuListener() {
			public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
				setStatusLine();
			}

			public void menuShown(org.eclipse.swt.events.MenuEvent e) {
			}
		});
		MenuItem pushAbout = new MenuItem(submenuHelp, SWT.PUSH);
		pushAbout.setText("JZip 정보...(&A)");
		pushAbout
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// JZip 정보
						MessageBox dialog = new MessageBox(sShell, SWT.OK
								| SWT.ICON_INFORMATION);
						String message = "JZip 0.1\n"
								+ "Zip 파일의 압축을 해제하는 프로그램입니다.\n"
								+ "정승원 (jeongseungwon@hanmail.net)";
						dialog.setMessage(message);
						dialog.setText("JZip 정보");
						dialog.open();
					}
				});
		pushAbout.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("프로그램 정보를 보여줍니다.");
			}
		});
		submenuItemHelp.setMenu(submenuHelp);
		submenuArchive = new Menu(submenuItemArchive);
		submenuArchive
				.addMenuListener(new org.eclipse.swt.events.MenuListener() {
					public void menuHidden(org.eclipse.swt.events.MenuEvent e) {
						setStatusLine();
					}

					public void menuShown(org.eclipse.swt.events.MenuEvent e) {
					}
				});
		MenuItem pushOpen = new MenuItem(submenuArchive, SWT.PUSH);
		pushOpen.setText("압축 파일 열기...(&O)\tCtrl+O");
		pushOpen.setAccelerator(SWT.CONTROL | 'O');
		pushOpen
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 압축 파일 선택 대화상자
						FileDialog dialog = new FileDialog(sShell, SWT.OPEN);
						dialog.setText("불러올 압축 파일을 선택하세요.");
						dialog.setFilterExtensions(new String[] { "*.zip",
								"*.jar", "*.*" });
						String file = dialog.open();

						if (file != null) {
							// 선택된 압축 파일 열기
							openArchive(file);
						}

					}
				});
		pushOpen.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 엽니다.");
			}
		});
		pushExtract = new MenuItem(submenuArchive, SWT.PUSH);
		pushExtract.setText("압축 풀기...(&E)\tCtrl+E");
		pushExtract.setAccelerator(SWT.CONTROL | 'E');
		pushExtract.setEnabled(false);
		pushExtract
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						extractArchive();
					}
				});
		pushExtract.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 풉니다.");
			}
		});
		pushClose = new MenuItem(submenuArchive, SWT.PUSH);
		pushClose.setText("압축 파일 닫기(&C)");
		pushClose.setEnabled(false);
		pushClose
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						closeArchive();
					}
				});
		pushClose.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("압축 파일을 닫습니다.");
			}
		});
		@SuppressWarnings("unused")
		MenuItem menuSeparator = new MenuItem(submenuArchive, SWT.SEPARATOR);
		MenuItem pushQuit = new MenuItem(submenuArchive, SWT.PUSH);
		pushQuit.setText("프로그램 종료(&Q)\tCtrl+Q");
		pushQuit.setAccelerator(SWT.CONTROL | 'Q');
		pushQuit
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 임시 파일 삭제
						deleteAllFiles(tempDirectory);

						// 프로그램 종료
						System.exit(0);
					}
				});
		pushQuit.addArmListener(new org.eclipse.swt.events.ArmListener() {
			public void widgetArmed(org.eclipse.swt.events.ArmEvent e) {
				statusLine.setText("프로그램을 종료합니다.");
			}
		});
		submenuItemArchive.setMenu(submenuArchive);
		sShell.setMenuBar(menuBar);
		sShell.addShellListener(new org.eclipse.swt.events.ShellAdapter() {
			@Override
			public void shellClosed(org.eclipse.swt.events.ShellEvent e) {
				deleteAllFiles(tempDirectory);
			}
		});
		table = new Table(sShell, SWT.MULTI | SWT.VIRTUAL);
		table.addListener(SWT.SetData, new Listener() {
			public void handleEvent(Event event) {
				TableItem item = (TableItem) event.item;
				int index = table.indexOf(item);
				item.setText(zipEntries[index].getStrings(radioUtf8
						.getSelection()));
			}
		});

		statusSeparator = new Label(sShell, SWT.SEPARATOR | SWT.HORIZONTAL);

		statusLine = new Label(sShell, SWT.NONE);
		statusLine.setText("");

		table.setLinesVisible(true);
		table.setHeaderVisible(true);
		table.setVisible(false);

		table
				.addSelectionListener(new org.eclipse.swt.events.SelectionAdapter() {
					@Override
					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						setStatusLine();
					}
				});
		tableColumnName = new TableColumn(table, SWT.LEFT);
		tableColumnName.setText("이름");
		tableColumnName.setWidth(170);
		tableColumnName
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 이름순으로 정렬
						if (table.getSortColumn() == tableColumnName) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnName, true);

								table.setSortColumn(tableColumnName);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioName, true);
							}
						} else {
							sortTable(tableColumnName, false);

							table.setSortColumn(tableColumnName);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioName, false);
						}
					}
				});
		tableColumnSize = new TableColumn(table, SWT.RIGHT);
		tableColumnSize.setText("크기");
		tableColumnSize.setWidth(100);
		tableColumnSize
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 크기순으로 정렬
						if (table.getSortColumn() == tableColumnSize) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnSize, true);

								table.setSortColumn(tableColumnSize);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioSize, true);
							}
						} else {
							sortTable(tableColumnSize, false);

							table.setSortColumn(tableColumnSize);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioSize, false);
						}
					}
				});
		tableColumnType = new TableColumn(table, SWT.RIGHT);
		tableColumnType.setText("형식");
		tableColumnType.setWidth(80);
		tableColumnType
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 형식순으로 정렬
						if (table.getSortColumn() == tableColumnType) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnType, true);

								table.setSortColumn(tableColumnType);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioType, true);
							}
						} else {
							sortTable(tableColumnType, false);

							table.setSortColumn(tableColumnType);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioSize, false);
						}
					}
				});
		tableColumnTime = new TableColumn(table, SWT.LEFT);
		tableColumnTime.setText("바뀐 시간");
		tableColumnTime.setWidth(270);
		tableColumnTime
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 바뀐 시간순으로 정렬
						if (table.getSortColumn() == tableColumnTime) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnTime, true);

								table.setSortColumn(tableColumnTime);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioTime, true);
							}
						} else {
							sortTable(tableColumnTime, false);

							table.setSortColumn(tableColumnTime);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioTime, false);
						}
					}
				});
		tableColumnPath = new TableColumn(table, SWT.LEFT);
		tableColumnPath.setWidth(150);
		tableColumnPath.setText("위치");
		tableColumnPath
				.addSelectionListener(new org.eclipse.swt.events.SelectionListener() {
					public void widgetDefaultSelected(
							org.eclipse.swt.events.SelectionEvent e) {
					}

					public void widgetSelected(
							org.eclipse.swt.events.SelectionEvent e) {
						// 위치순으로 정렬
						if (table.getSortColumn() == tableColumnPath) {
							if (table.getSortDirection() == SWT.UP) {
								table.setSortColumn(null);
							} else {
								sortTable(tableColumnPath, true);

								table.setSortColumn(tableColumnPath);
								table.setSortDirection(SWT.UP);

								setAlignmentMenu(radioPath, true);
							}
						} else {
							sortTable(tableColumnPath, false);

							table.setSortColumn(tableColumnPath);
							table.setSortDirection(SWT.DOWN);

							setAlignmentMenu(radioPath, false);
						}
					}
				});
	}

	/**
	 * 하위 디렉토리의 모든 파일까지 다 삭제하는 메소드
	 * 
	 * @param file
	 *            삭제할 파일이나 디렉토리
	 */
	private void deleteAllFiles(File file) {
		if (file.isDirectory()) {
			// 디렉토리인 경우
			File[] files = file.listFiles();

			if (files != null) {
				for (int i = 0; i < files.length; i++) {
					deleteAllFiles(files[i]);
				}
			}
		}

		file.delete();
	}

	/**
	 * 압축을 해제하는 메소드
	 */
	private void extractArchive() {
		// 압축을 풀 디렉토리 선택 대화상자
		DirectoryDialog dialog = new DirectoryDialog(sShell);
		dialog.setText("압축을 풀 디렉토리를 선택하세요.");
		dialog.setMessage("압축을 풀 디렉토리를 선택하세요.");
		String directory = dialog.open();

		if (directory != null) {
			if (zipFile == null || zipEntries == null) {
				// 열려있는 압축 파일이 존재하지 않음
				MessageBox message = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				message.setMessage("열려있는 압축 파일이 없습니다.");
				message.setText("열려있는 압축 파일이 없음!");
				message.open();

				return;
			}

			final boolean unicode = radioUtf8.getSelection();

			int overwrite = OverwriteDialog.NO;

			statusLine.setText("압축 해제 중입니다.");

			if (table.getSelectionCount() >= 1) {
				// 선택된 항목이 있는 경우
				int[] selectionIndices = table.getSelectionIndices();
				for (int i = 0; i < selectionIndices.length; i++) {
					overwrite = extractEntry(zipEntries[selectionIndices[i]],
							unicode, directory, overwrite);
				}
			} else {
				// 선택된 항목이 없는 경우
				for (int i = 0; i < zipEntries.length; i++) {
					overwrite = extractEntry(zipEntries[i], unicode, directory,
							overwrite);
				}
			}

			// 압축 해제 완료 메시지
			MessageBox message = new MessageBox(sShell, SWT.OK
					| SWT.ICON_INFORMATION);
			message.setMessage("압축 해제가 완료되었습니다.");
			message.setText("압축 해제 완료!");
			message.open();

			setStatusLine();
		}
	}

	/**
	 * 하나의 Zip 엔트리를 압축해제하는 메소드
	 * 
	 * @param entry
	 *            Zip 엔트리
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @param directory
	 *            압축을 해제할 디렉토리
	 * @param overwrite
	 *            덮어쓰기 설정
	 * @return 덮어쓰기 설정
	 */
	private int extractEntry(ZipEntryData entry, boolean unicode,
			String directory, int overwrite) {
		String entryName = entry.getEntryName(unicode);

		File entryFile = new File(directory, entryName);

		if (entryFile.isDirectory()) {
			// 디렉토리일 경우
			entryFile.mkdirs();
		} else {
			// 파일일 경우
			File parentDir = entryFile.getParentFile();

			// 새로운 파일(빈 파일) 생성
			if (!parentDir.exists()) {
				parentDir.mkdirs();
			}

			try {
				if (entryFile.createNewFile() == false) {
					// 같은 이름의 파일이 이미 존재하는 경우
					if (overwrite != OverwriteDialog.ALL_YES
							&& overwrite != OverwriteDialog.ALL_NO) {
						// 파일 덮어쓰기 확인 대화상자
						OverwriteDialog overwriteDialog = new OverwriteDialog();
						overwrite = overwriteDialog.open(sShell, entryFile
								.getAbsolutePath());
					}

					if (overwrite == OverwriteDialog.CANCEL) {
						// 압축 해제 취소 메시지
						MessageBox message = new MessageBox(sShell, SWT.OK
								| SWT.ICON_INFORMATION);
						message.setMessage("압축 해제가 취소되었습니다.");
						message.setText("압축 해제 취소!");
						message.open();

						setStatusLine();

						return overwrite;
					} else if (overwrite == OverwriteDialog.YES
							|| overwrite == OverwriteDialog.ALL_YES) {
						// 파일 덮어쓰기
						entryFile.delete();
						entryFile.createNewFile();
					} else {
						// 파일을 덮어쓰지 않고, 다음으로 넘어감
						return overwrite;
					}
				}
			} catch (IOException e1) {
				// 압축 파일 풀기 실패 메시지
				MessageBox message = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				message.setMessage("압축 파일을 풀지 못했습니다.");
				message.setText("압축 해제 실패!");
				message.open();

				setStatusLine();

				return overwrite;
			}

			// 파일 내용 저장
			try {
				InputStream is = zipFile.getInputStream(entry.getZipEntry());
				FileOutputStream fos = new FileOutputStream(entryFile);

				byte[] buffer = new byte[BUFFER_SIZE];
				int read = 0;

				read = is.read(buffer);
				while (read >= 0) {
					fos.write(buffer, 0, read);
					read = is.read(buffer);
				}

				is.close();
				fos.close();
			} catch (IOException e1) {
				// 압축 파일 풀기 실패 메시지
				MessageBox message = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				message.setMessage("압축 파일을 풀지 못했습니다.");
				message.setText("압축 해제 실패!");
				message.open();

				setStatusLine();

				return overwrite;
			}
		}

		return overwrite;
	}

	/**
	 * Zip 엔트리의 압축을 해제하는 메소드 같은 이름을 지닌 파일이 있으면 무조건 덮어씀
	 * 
	 * @param entry
	 *            압축을 풀 Zip 엔트리 정보
	 * @param directory
	 *            압축을 풀 디렉토리
	 * @param unicode
	 *            유니코드이면 true, 아니면 false
	 * @return 압축이 해제된 파일명
	 */
	private String extractTempEntry(ZipEntryData entry, File directory,
			boolean unicode) {
		String entryName = entry.getEntryName(unicode);

		File entryFile = new File(directory, entryName);

		if (entryFile.isDirectory()) {
			// 디렉토리일 경우
			entryFile.mkdirs();
		} else {
			// 파일일 경우
			File parentDir = entryFile.getParentFile();

			// 새로운 파일(빈 파일) 생성
			if (!parentDir.exists()) {
				parentDir.mkdirs();
			}

			try {
				if (entryFile.createNewFile() == false) {
					// 파일 덮어쓰기
					entryFile.delete();
					entryFile.createNewFile();
				}
			} catch (IOException e1) {
				// 압축 파일 풀기 실패 메시지
				MessageBox message = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				message.setMessage("압축 파일을 풀지 못했습니다.");
				message.setText("압축 해제 실패!");
				message.open();

				return null;
			}

			// 파일 내용 저장
			try {
				InputStream is = zipFile.getInputStream(entry.getZipEntry());
				FileOutputStream fos = new FileOutputStream(entryFile);

				byte[] buffer = new byte[BUFFER_SIZE];
				int read = 0;

				read = is.read(buffer);
				while (read >= 0) {
					fos.write(buffer, 0, read);
					read = is.read(buffer);
				}

				is.close();
				fos.close();
			} catch (IOException e1) {
				// 압축 파일 풀기 실패 메시지
				MessageBox message = new MessageBox(sShell, SWT.OK
						| SWT.ICON_ERROR);
				message.setMessage("압축 파일을 풀지 못했습니다.");
				message.setText("압축 해제 실패!");
				message.open();

				return null;
			}
		}

		return entryFile.getAbsolutePath();
	}

	/**
	 * 압축 파일을 여는 메소드
	 * 
	 * @param fileName
	 *            압축 파일의 이름
	 */
	private void openArchive(String fileName) {
		if (fileName == null) {
			return;
		}

		File file = new File(fileName);

		if (!file.exists()) {
			// 압축 파일 열기 실패 대화상자
			MessageBox message = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			message.setMessage(fileName + " 파일이 존재하지 않습니다.");
			message.setText("압축 파일 열기 실패!");
			message.open();

			zipFile = null;

			return;
		}

		Enumeration entries = null;

		try {
			zipFile = new ZipFile(file);
			entries = zipFile.entries();
		} catch (IOException e1) {
			// 압축 파일 열기 실패 대화상자
			MessageBox message = new MessageBox(sShell, SWT.OK | SWT.ICON_ERROR);
			message.setMessage(fileName + " 파일을 열지 못했습니다.");
			message.setText("압축 파일 열기 실패!");
			message.open();

			zipFile = null;

			return;
		}

		int zipEntrySize = zipFile.size();

		// 압축 파일 내용 표시
		table.removeAll();

		long totalSize = 0; // 전체 크기
		zipEntries = new ZipEntryData[zipEntrySize];

		int i = 0;
		while (entries.hasMoreElements()) {
			ZipEntry entry = (ZipEntry) entries.nextElement();
			zipEntries[i] = new ZipEntryData(entry);
			totalSize += entry.getSize();
			i++;
		}

		table.setItemCount(zipEntrySize);

		// 항목 정렬
		sortTable(tableColumnName, false);
		table.setVisible(true);

		// 메뉴 활성화
		pushExtract.setEnabled(true);
		pushClose.setEnabled(true);
		pushRefresh.setEnabled(true);
		pushSelectAll.setEnabled(true);
		submenuItemAlignment.setEnabled(true);
		radioUtf8.setEnabled(true);
		radioMs949.setEnabled(true);

		// 정렬 메뉴 설정
		setAlignmentMenu(radioName, false);

		setStatusLine();
	}

	/**
	 * 현재의 압축 파일을 다시 읽어들이는 메소드
	 */
	private void refresh() {
		if (zipFile != null) {
			openArchive(zipFile.getName());
		}
	}

	/**
	 * 정렬 메뉴를 선택하는 메소드
	 * 
	 * @param menu
	 *            선택할 정렬 메뉴
	 * @param reverse
	 *            역순이면 true, 아니면 false
	 */
	private void setAlignmentMenu(MenuItem menu, boolean reverse) {
		// 메뉴 선택
		menu.setSelection(true);

		// 다른 메뉴는 선택 해제
		if (menu != radioName) {
			radioName.setSelection(false);
		}
		if (menu != radioSize) {
			radioSize.setSelection(false);
		}
		if (menu != radioType) {
			radioType.setSelection(false);
		}
		if (menu != radioTime) {
			radioTime.setSelection(false);
		}
		if (menu != radioPath) {
			radioPath.setSelection(false);
		}

		// 역순 설정
		checkReverse.setSelection(reverse);
	}

	/**
	 * DND 기능을 설정하는 메소드
	 */
	private void setDND() {
		int operations = DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK;
		Transfer[] types = new Transfer[] { FileTransfer.getInstance() };

		target = new DropTarget(sShell, operations);
		target.setTransfer(types);
		final DropListener dropListener = new DropListener();
		target.addDropListener(dropListener);

		operations = DND.DROP_MOVE | DND.DROP_COPY;

		source = new DragSource(table, operations);
		source.setTransfer(types);
		source.addDragListener(new DragSourceListener() {

			public void dragFinished(DragSourceEvent event) {
				target.addDropListener(dropListener);
			}

			public void dragSetData(DragSourceEvent event) {
				String[] files = null;

				final int selectionCount = table.getSelectionCount();

				if (selectionCount >= 1) {
					// 선택된 항목이 있는 경우
					final boolean unicode = radioUtf8.getSelection();

					files = new String[selectionCount];
					int[] selectionIndices = table.getSelectionIndices();

					for (int i = 0; i < selectionIndices.length; i++) {
						files[i] = extractTempEntry(
								zipEntries[selectionIndices[i]], tempDirectory,
								unicode);

						if (files[i] == null) {
							event.doit = false;
							event.data = null;
							event.detail = DND.DROP_NONE;
							return;
						}
					}

					event.data = files;
				} else {
					// 선택된 항목이 없는 경우
					event.doit = false;
					event.data = null;
					event.detail = DND.DROP_NONE;
					return;
				}

			}

			public void dragStart(DragSourceEvent event) {
				target.removeDropListener(dropListener);
			}

		});
	}

	/**
	 * 상태 표시줄의 텍스트를 설정하는 메소드
	 */
	private void setStatusLine() {
		String text = "";

		if (zipEntries != null) {
			long totalSize = 0;
			for (int i = 0; i < zipEntries.length; i++) {
				totalSize += zipEntries[i].getSize();
			}

			text = zipEntries.length + " 항목 ("
					+ ZipEntryData.getSizeString(totalSize) + ")";

			if (table.getSelectionCount() >= 1) {
				// 선택된 항목이 있는 경우
				long size = 0;
				int[] selectionIndices = table.getSelectionIndices();
				for (int i = 0; i < selectionIndices.length; i++) {
					size += zipEntries[selectionIndices[i]].getSize();
				}

				text += ", " + selectionIndices.length + " 항목 선택됨 ("
						+ ZipEntryData.getSizeString(size) + ")";
			}
		}

		statusLine.setText(text);
	}

	/**
	 * 항목을 정렬하는 메소드
	 * 
	 * @param 정렬
	 *            기준이 되는 테이블 컬럼
	 * @param 역순이면
	 *            true, 아니면 false
	 */
	private void sortTable(TableColumn col, boolean reverse) {
		int dir;
		if (reverse) {
			dir = SWT.UP;
		} else {
			dir = SWT.DOWN;
		}

		final TableColumn sortColumn = col;
		final int direction = dir;

		Arrays.sort(zipEntries, new Comparator<ZipEntryData>() {
			public int compare(ZipEntryData entryData1, ZipEntryData entryData2) {
				if (sortColumn == tableColumnSize) {
					// 크기로 비교
					long size1 = entryData1.getSize();
					long size2 = entryData2.getSize();

					if (size1 == size2) {
						return 0;
					}

					if (direction == SWT.DOWN) {
						return size1 > size2 ? 1 : -1;
					}

					return size1 < size2 ? 1 : -1;
				} else if (sortColumn == tableColumnType) {
					// 형식으로 비교
					String type1 = entryData1.getType();
					String type2 = entryData2.getType();

					if (direction == SWT.DOWN) {
						return type1.compareTo(type2);
					} else {
						return type2.compareTo(type1);
					}
				} else if (sortColumn == tableColumnTime) {
					// 바뀐 시간으로 비교
					long time1 = entryData1.getTime();
					long time2 = entryData2.getTime();

					if (time1 == time2) {
						return 0;
					}

					if (direction == SWT.DOWN) {
						return time1 > time2 ? 1 : -1;
					} else {
						return time1 < time2 ? 1 : -1;
					}
				} else if (sortColumn == tableColumnPath) {
					// 위치로 비교
					boolean unicode = radioUtf8.getSelection();

					String path1 = entryData1.getPath(unicode);
					String path2 = entryData2.getPath(unicode);

					if (direction == SWT.DOWN) {
						return path1.compareTo(path2);
					} else {
						return path2.compareTo(path1);
					}
				} else {
					// 이름으로 비교
					boolean unicode = radioUtf8.getSelection();

					String name1 = entryData1.getName(unicode);
					String name2 = entryData2.getName(unicode);

					if (direction == SWT.DOWN) {
						return name1.compareTo(name2);
					} else {
						return name2.compareTo(name1);
					}
				}
			}
		});

		table.clearAll();
	}
}
// @jve:decl-index=0:
