/*
    sayclubserver.cpp - Kopete Sayclub Protocol

    Copyright (c) 2006      by Park J. K.		 <nemesis@planetmono.org>
    Kopete    (c) 2002-2003 by the Kopete developers <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This library is free software; you can redistribute it and/or         *
    * modify it under the terms of the GNU General Public                   *
    * License as published by the Free Software Foundation; either          *
    * version 2 of the License, or (at your option) any later version.      *
    *                                                                       *
    *************************************************************************
*/

#include "sayclubserver.h"

#include <iostream> // STUB

#include <stdlib.h>
#include <time.h>

#include <qtimer.h>
#include <qtextcodec.h>
#include <qsignal.h>
#include <qvariant.h>

#include <kbufferedsocket.h>
#include <kmessagebox.h>
#include <kdebug.h>

#include <kopetegroup.h>
#include <kopetecontactlist.h>

#include "sayclubaccount.h"
#include "sayclubprotocol.h"
#include "sayclubutilities.h"
#include "gaea_phpobject.h"

using namespace KNetwork;
using namespace std;

SayclubServer::SayclubServer( SayclubAccount *account, QObject * )
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	m_peerswitched = false;
	m_valid = true;
	m_account = account;
	
	m_requestQueue.setAutoDelete( true );
	m_signalList.setAutoDelete( true );
	m_objectList.setAutoDelete( true );
	m_messageList.setAutoDelete( true );
	
	m_newstatus = SayclubProtocol::protocol()->sayclubOffline;
	
	QObject::connect( this, SIGNAL( changeAllContacts( const Kopete::OnlineStatus & ) ), m_account, SLOT( slotChangeAllStatus( const Kopete::OnlineStatus & ) ) );
	QObject::connect( this, SIGNAL( changeContactLoginStatus( PHPMapObject & ) ), m_account, SLOT( slotChangeContactLoginStatus( PHPMapObject & ) ) );
	QObject::connect( this, SIGNAL( changeContactStatus( PHPMapObject & ) ), m_account, SLOT( slotChangeContactStatus( PHPMapObject & ) ) );
	QObject::connect( this, SIGNAL( incomingMemo( PHPMapObject & ) ), m_account, SLOT( slotIncomingMemo( PHPMapObject & ) ) );
	QObject::connect( this, SIGNAL( accountClosed( Kopete::Account::DisconnectReason ) ), m_account, SLOT( disconnected( Kopete::Account::DisconnectReason ) ) );
	QObject::connect( this, SIGNAL( socketClosed() ), this, SLOT( slotSocketClosed() ) );
	QObject::connect( this, SIGNAL( closeSocket() ), this, SLOT( slotSocketClose() ) );
}

SayclubServer::~SayclubServer()
{
	QObject::disconnect( this, 0, this, 0 );
	QObject::disconnect( this, 0, m_account, 0 );
	m_account->setDisconnected();
	
	setOnlineStatus( Disconnected );
	if( m_socket ) {
		m_socket->close();
		slotSocketClosed();
	}
}

// Connect to the server
void SayclubServer::connect( const QString &server, uint port )
{
	if( m_onlineStatus == Connected || m_onlineStatus == Connecting )
		return;
	
	if( m_onlineStatus == Disconnecting )
		delete m_socket;
	
	setOnlineStatus( Connecting );
	
	m_waitBlockSize = 0;
	m_buffer = Buffer( 0 );
	
	m_server = server;
	m_port = port;
	
	m_socket = new KBufferedSocket( server, QString::number(port) );
	
	m_socket->enableRead( true );
	m_socket->enableWrite( false );
	
	QObject::connect( m_socket, SIGNAL( readyRead() ), this, SLOT( slotDataReceived() ) );
	QObject::connect( m_socket, SIGNAL( readyWrite() ), this, SLOT( slotReadyWrite() ) );
	QObject::connect( m_socket, SIGNAL( hostFound() ), this, SLOT( slotHostFound() ) );
	QObject::connect( m_socket, SIGNAL( connected( const KResolverEntry & ) ), this, SLOT( slotConnectionSuccess() ) );
	QObject::connect( m_socket, SIGNAL( gotError( int ) ), this, SLOT( slotSocketError( int ) ) );
	QObject::connect( m_socket, SIGNAL( closed() ), this, SLOT( slotSocketClosed() ) );
	
	m_socket->connect();
}

// Disconnect
void SayclubServer::disconnect()
{
	slotSocketClosed();
}

void SayclubServer::setOnlineStatus( SayclubServer::OnlineStatus status )
{
	if( m_onlineStatus == status )
		return;
	
	m_onlineStatus = status;
}

void SayclubServer::slotSocketError( int error )
{
	if( !KSocketBase::isFatalError(error) )
		return;
	
	QString errormsg = QString::fromUtf8( "세이클럽 서버에 접속하는 동안 다음 오류가 발생하였습니다:\n" );
	if( error == KSocketBase::LookupFailure )
		errormsg += QString::fromUtf8( "다음 주소를 룩업하는데 실패하였습니다: %1" ).arg( m_socket->peerResolver().nodeName() );
	else
		errormsg += m_socket->errorString();
	
	m_socket->deleteLater();
	m_socket = 0L;
	
	emit connectionFailed();
	emit accountClosed( Kopete::Account::InvalidHost );
	emit socketClosed();
	emit errorMessage( ErrorNormal, errormsg );
}

void SayclubServer::slotDataReceived()
{
	int avail = m_socket->bytesAvailable();
	if( avail < 0 )
		return;
	
	char *buffer = new char[ avail+1 ];
	int ret = m_socket->readBlock( buffer, avail );
	
	m_buffer.add( buffer, ret );
	slotReadLine();
	
	delete[] buffer;
}

void SayclubServer::slotReadLine()
{
	if( !pollReadBlock() )
	{
		if( m_buffer.size() >= 2 && ( m_buffer.data()[0] == '\0' || m_buffer.data()[0] == '\1' ) ) {
			bytesReceived( m_buffer.take( 2 ) );
			QTimer::singleShot( 0, this, SLOT( slotReadLine() ) );
			return;
		}
		
		int index = -1;
		for( uint x = 0; m_buffer.size() > x; ++x) {
			if( m_buffer[x] == '\n' ) {
				index = x;
				break;
			}
		}
		
		if( index != -1 ) {
			QString pline = m_account->getCodec()->toUnicode( m_buffer.take( index+1 ), index );
			pline.replace( "\n", "" );
			QTimer::singleShot( 0, this, SLOT( slotReadLine() ) );
			
			if( pline == "IRCR_HELLO" )
				emit receivedHello();
			else
				parseLine( pline );
		}
	}
}

void SayclubServer::readBlock( uint len )
{
	if( m_waitBlockSize )
		return;
	
	m_waitBlockSize = len;
	pollReadBlock();
}

// Send command to the server
int SayclubServer::sendCmd( QCString cmdname, PHPObject &array, QObject *receiver, const char *member )
{
	int reqid = 10000000+(rand()%89999999);
	
	while( m_requestQueue.find( reqid ) != NULL )
		reqid = 10000000+(rand()%89999999);
	
	QSignal *sig = new QSignal;
	if( receiver != NULL && member != NULL )
		sig->connect( receiver, member );
	else
		sig->connect( m_account, SLOT( slotDisposeRequest( int ) ) );
	QVariant param( reqid );
	sig->setValue( param );
	m_signalList.insert( reqid, sig );
	
	QCString query;
	QCString serialized = std2qcstr( array.serialize() );
	query = "APPSVR_CALL_SERIAL method LOGINE:"+QCString( QString::number( reqid ).latin1() )+" &method_name="+cmdname+"&arrparam="+SayclubUtilities::urlQuote( serialized );
	write( query );
	
	return reqid;
}

void SayclubServer::write( QCString str )
{
	m_sendQueue.append( str+'\n' );
	m_socket->enableWrite( true );
}

bool SayclubServer::pollReadBlock()
{
	if( !m_waitBlockSize )
		return false;
	else if( m_buffer.size() < m_waitBlockSize )
		return true;
	
	QByteArray block = m_buffer.take( m_waitBlockSize );
	m_waitBlockSize = 0;
	emit blockRead( block );
	
	return false;
}

// Parses line.
void SayclubServer::parseLine( const QString &str )
{
	QString tmp = str;
	
	// To remove useless conditional flow, divide range by before-peerswitched and after-peerswitched
	if( !m_peerswitched ) {
		if( str.startsWith( "PEER_SWITCH FAILED" ) ) {
			emit errorMessage( ErrorNormal, QString::fromUtf8( "아이디가 틀립니다." ) );
			emit accountClosed( Kopete::Account::BadUserName );
			emit closeSocket();
		}
		else if( str.startsWith( "ERROR :Closing Link:" ) ) {
			emit errorMessage( ErrorNormal, QString::fromUtf8( "서버에서 접속을 끊었습니다.\n%1" ).arg( str.section( ' ', 6 ) ) );
			emit accountClosed( Kopete::Account::ConnectionReset );
			emit closeSocket();
		}
		else if( str.startsWith( "MYIP" ) )
			m_p_myip = tmp.remove(0, 6);
		else if( str.startsWith( "MYPORT" ) )
			m_p_myport = tmp.remove(0, 8).toUInt();
		else if( str.startsWith( "UDPPORT" ) )
			m_p_udpport = tmp.remove(0, 9).toUInt();
		else if( str.startsWith( "SVRTIME" ) )
			m_p_svrtime = tmp.remove(0, 9).toUInt();
		else if( str.startsWith( "PEER_SWITCH SUCCESS" ) ) {
			m_peerswitched = true;
			emit endNegotiation();
		}
	}
	else {
		if( str.startsWith( "PING" ) )
			write( "PONG :%1"+tmp.remove( 0, 6 ).utf8() );
		else if( str.startsWith( "APPSVR_RESULT BEGIN" ) )
			m_requestQueue.replace( str.section( ' ', 2, 2 ).right(8).toLong(), new QCString );
		else if( str.startsWith( "APPSVR_RESULT DATA" ) )
			*m_requestQueue[str.section( ' ', 2, 2 ).right(8).toLong()] += str.section(' ', 3, 3).latin1();
		else if( str.startsWith( "APPSVR_RESULT END" ) ) {
			long reqid = str.section( ' ', 2, 2 ).right(8).toLong();
			QCString decoded;
			if( !SayclubUtilities::base64Decode( decoded, *m_requestQueue[reqid] )) {
				m_requestQueue.remove( reqid );
				m_signalList.remove( reqid );
				emit errorMessage( ErrorInternal, QString::fromUtf8( "내부 문자열을 디코딩하는데 오류가 발생하였습니다." ) );
				return;
			}
			m_requestQueue.remove( reqid );
			PHPObject *o = new PHPObject;
			*o = PHPObject::unserialize( qcstr2std( decoded ) );
			m_objectList.insert( reqid, o );
			m_signalList[reqid]->activate();
		}
		else if( str.startsWith( "NOTIFY" ) ) {
			QCString decoded = SayclubUtilities::urlUnquote( str.section( ' ', 1, 1 ).latin1() );
			PHPObject so = PHPObject::unserialize( qcstr2std ( decoded ) );
			parseNotification( so );
		}
		else
			cout << str << endl;
	}
}

void SayclubServer::parseNotification( const PHPObject &o )
{
	PHPMapObject &m = o.asMap();
	
	if( !HAS_KEY( m, "notify_type" ) || !m["notify_type"].isString() )
		return;
	
	if( m["notify_type"].asString() == "FRIENDINOUT" )
		emit changeContactLoginStatus( m );
	else if( m["notify_type"].asString() == "INFOCHG" )
		emit changeContactStatus( m );
	else if( m["notify_type"].asString() == "MEMO" )
		emit incomingMemo( m );
}

void SayclubServer::bytesReceived( const QByteArray & )
{
}

void SayclubServer::slotConnectionSuccess()
{
	kdDebug( 14210 ) << k_funcinfo << endl;
}

void SayclubServer::slotReadyWrite()
{
	if( !m_sendQueue.isEmpty() ) {
		QValueList<QByteArray>::Iterator it = m_sendQueue.begin();
		m_socket->writeBlock( *it, (*it).size() );
		m_sendQueue.remove( it );
		if( m_sendQueue.isEmpty() )
			m_socket->enableWrite( false );
	}
	else
		m_socket->enableWrite( false );
}

void SayclubServer::slotHostFound()
{
}

void SayclubServer::slotSocketClosed()
{
	kdDebug( 14210 ) << k_funcinfo << endl;
	
	if( !m_socket || m_onlineStatus == Disconnected )
		return;
	
	emit changeAllContacts( SayclubProtocol::protocol()->sayclubOffline );
	
	setOnlineStatus( Disconnected );
	invalidate();
	
	m_account->setDisconnected();
	m_buffer = Buffer( 0 );
	m_socket->closeNow();
	m_socket->deleteLater();
	m_socket = NULL;
	
	emit socketClosed();
}

void SayclubServer::slotSocketClose()
{
	slotSocketClosed();
}

void SayclubServer::incomingMessage( QString message )
{
	emit messageReceived( message );
}

SayclubServer::Buffer::Buffer( unsigned int sz )
	: QByteArray( sz )
{
}

SayclubServer::Buffer::~Buffer()
{
}

void SayclubServer::Buffer::add( char *str, unsigned int sz )
{
	char *b = new char[ size() + sz ];
	for ( uint f = 0; f < size(); f++ )
		b[ f ] = data()[ f ];
	for ( uint f = 0; f < sz; f++ )
		b[ size() + f ] = str[ f ];

	duplicate( b, size() + sz );
	delete[] b;
}

QByteArray SayclubServer::Buffer::take( unsigned blockSize )
{
	if ( size() < blockSize )
	{
		kdWarning( 14140 ) << k_funcinfo << "Buffer size " << size() << " < asked size " << blockSize << "!" << endl;
		return QByteArray();
	}

	QByteArray rep( blockSize );
	for( uint i = 0; i < blockSize; i++ )
		rep[ i ] = data()[ i ];

	char *str = new char[ size() - blockSize ];
	for ( uint i = 0; i < size() - blockSize; i++ )
		str[ i ] = data()[ blockSize + i ];
	duplicate( str, size() - blockSize );
	delete[] str;

	return rep;
}

#include "sayclubserver.moc"
