/*
 * TokigunStudio Gaea Library: PHPObject
 * Copyright (c) 2006, Kang Seonghoon (Tokigun).
 * This library can be distributed under the terms of the GNU LGPL.
 */

#ifndef _GAEA_PHPOBJECT_H_
#define _GAEA_PHPOBJECT_H_

#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <utility>
#include <cstring>
#include <cstdlib>
#include "gaea_union.h"

using namespace std;

// forward declaration
class PHPObject;

////////////////////////////////////////////////////////////////////////////////
// declaration

#define PARENT Union<bool, int, double, string>
class PHPAtomObject: public PARENT
{
	public:
		enum type_t { NullType = 0, BoolType = 1, IntType = 2, FloatType = 3, StringType = 4 };

		PHPAtomObject(): PARENT() {}
		PHPAtomObject(bool other): PARENT(other) {}
		PHPAtomObject(int other): PARENT(other) {}
		PHPAtomObject(double other): PARENT(other) {}
		PHPAtomObject(const char *other): PARENT(string(other)) {}
		PHPAtomObject(const string& other): PARENT(other) {}
		PHPAtomObject(const PHPAtomObject& other): PARENT(other) {}
	
		type_t getType() const { return (type_t)PARENT::getType(); }

		bool isNull() const { return getType() == NullType; }
		bool isBool() const { return getType() == BoolType; }
		bool isInt() const { return getType() == IntType; }
		bool isFloat() const { return getType() == FloatType; }
		bool isString() const { return getType() == StringType; }

		bool asBool() const { return operator bool(); }
		int asInt() const { return operator int(); }
		double asFloat() const { return operator double(); }
		string& asString() const { if (!isString()) throw bad_cast(); return *(string*)getPointer(); }

#define DECLARE_EQ_OP(arg, targ) const PHPAtomObject operator=(arg) { PARENT::operator=(targ); return *this; }
	DECLARE_EQ_OP(const bool rhs, rhs)
	DECLARE_EQ_OP(const int rhs, rhs)
	DECLARE_EQ_OP(const double rhs, rhs)
	DECLARE_EQ_OP(const char *rhs, string(rhs))
	DECLARE_EQ_OP(const string& rhs, rhs)
	DECLARE_EQ_OP(const PHPAtomObject& rhs, rhs)
#undef DECLARE_EQ_OP

	void serialize(ostream& os) const;
	string serialize(void) const { ostringstream oss; serialize(oss); return oss.str(); }
	static int unserialize(istream& is, PHPObject& obj);

	friend ostream& operator<<(ostream& os, const PHPAtomObject& obj);
};
#undef PARENT

class PHPArrayObject: public vector<PHPObject>
{
	public:
		PHPArrayObject();
		PHPArrayObject(size_type n);
		PHPArrayObject(size_type n, const PHPObject& t);
		PHPArrayObject(const PHPArrayObject& other);
	
		void serialize(ostream& os) const;
		string serialize(void) const { ostringstream oss; serialize(oss); return oss.str(); }
		static inline int unserialize(istream& is, PHPObject& obj);

		friend ostream& operator<<(ostream& os, const PHPArrayObject& obj);
};

namespace _phpobject_detail {
	class PHPAtomObjectCompare
	{
		private:
			pair<bool,int> toNumber(const PHPAtomObject& obj)
			{
				switch (obj.getType()) {
					case PHPAtomObject::NullType: return make_pair(true, 0);
					case PHPAtomObject::BoolType: return make_pair(true, obj.asBool() ? 1 : 0);
					case PHPAtomObject::IntType: return make_pair(true, obj.asInt());
					case PHPAtomObject::FloatType: return make_pair(true, int(obj.asFloat()));
					case PHPAtomObject::StringType: {
						string& str = obj.asString();
						int len = str.length();
						if (len > 0 && !isspace(str[0])) {
							const char *ptr = str.c_str(); char *endptr;
							int value = strtol(ptr, &endptr, 10);
							if (endptr == ptr + len) return make_pair(true, value);
						}
				// fall through
					}
					default: return make_pair(false, 0);
				}
			}

			string toString(const PHPAtomObject& obj)
			{
				if (obj.getType() == PHPAtomObject::StringType) {
					return obj.asString();
				} else {
					ostringstream oss;
					oss << obj;
					return oss.str();
				}
			}

		public:
			bool operator()(const PHPAtomObject& lhs, const PHPAtomObject& rhs)
			{
				pair<bool,int> lhsv = toNumber(lhs);
				if (lhsv.first) {
					pair<bool,int> rhsv = toNumber(rhs);
					if (rhsv.first) return lhsv.second < rhsv.second;
				}
				return toString(lhs) < toString(rhs);
			}
	};
}

class PHPMapObject: public map<PHPAtomObject, PHPObject, _phpobject_detail::PHPAtomObjectCompare>
{
	public:
		PHPMapObject();
		PHPMapObject(const key_compare& comp);
		template <class T> PHPMapObject(T f, T l);
		template <class T> PHPMapObject(T f, T l, const key_compare& comp);
		PHPMapObject(const PHPMapObject& other);
	
		void serialize(ostream& os) const;
		string serialize(void) const { ostringstream oss; serialize(oss); return oss.str(); }
		static inline int unserialize(istream& is, PHPObject& obj);

		friend ostream& operator<<(ostream& os, const PHPMapObject& obj);
};

#define PARENT Union<PHPAtomObject, PHPArrayObject, PHPMapObject>
class PHPObject: public PARENT
{
	public:
		enum type_t { NullType = 0, AtomType = 1, ArrayType = 2, MapType = 3 };

		PHPObject(): PARENT() {}
		PHPObject(bool other): PARENT(PHPAtomObject(other)) {}
		PHPObject(int other): PARENT(PHPAtomObject(other)) {}
		PHPObject(double other): PARENT(PHPAtomObject(other)) {}
		PHPObject(const char *other): PARENT(PHPAtomObject(string(other))) {}
		PHPObject(const string& other): PARENT(PHPAtomObject(other)) {}
		PHPObject(const PHPAtomObject& other): PARENT(other) {}
		PHPObject(const PHPArrayObject& other): PARENT(other) {}
		PHPObject(const PHPMapObject& other): PARENT(other) {}
		PHPObject(const PHPObject& other): PARENT(other) {}
	
		type_t getType() const { return (type_t)PARENT::getType(); }

		bool isNull() const { return getType() == NullType || (getType() == AtomType && asAtom().isNull()); }
		bool isAtom() const { return getType() == AtomType; }
		bool isBool() const { return getType() == AtomType && asAtom().isBool(); }
		bool isInt() const { return getType() == AtomType && asAtom().isInt(); }
		bool isFloat() const { return getType() == AtomType && asAtom().isFloat(); }
		bool isString() const { return getType() == AtomType && asAtom().isString(); }
		bool isArray() const { return getType() == ArrayType; }
		bool isMap() const { return getType() == MapType; }

		bool asBool() const { return asAtom().asBool(); }
		int asInt() const { return asAtom().asInt(); }
		double asFloat() const { return asAtom().asFloat(); }
		string& asString() const { return asAtom().asString(); }
		PHPAtomObject &asAtom() const
		{
			if (!isAtom()) throw bad_cast();
			return *(PHPAtomObject*)getPointer();
		}
		PHPArrayObject &asArray() const
		{
			if (!isArray()) throw bad_cast();
			return *(PHPArrayObject*)getPointer();
		}
		PHPMapObject &asMap() const
		{
			if (!isMap()) throw bad_cast();
			return *(PHPMapObject*)getPointer();
		}

		operator bool() const { return asAtom().asBool(); }
		operator int() const { return asAtom().asInt(); }
		operator double() const { return asAtom().asFloat(); }
		operator string() const { return asAtom().asString(); }

#define DECLARE_EQ_OP(arg, targ) const PHPObject operator=(arg) { PARENT::operator=(targ); return *this; }
	DECLARE_EQ_OP(const bool rhs, PHPAtomObject(rhs))
	DECLARE_EQ_OP(const int rhs, PHPAtomObject(rhs))
	DECLARE_EQ_OP(const double rhs, PHPAtomObject(rhs))
	DECLARE_EQ_OP(const char *rhs, PHPAtomObject(string(rhs)))
	DECLARE_EQ_OP(const string& rhs, PHPAtomObject(rhs))
	DECLARE_EQ_OP(const PHPAtomObject& rhs, rhs)
	DECLARE_EQ_OP(const PHPArrayObject& rhs, rhs)
	DECLARE_EQ_OP(const PHPMapObject& rhs, rhs)
	DECLARE_EQ_OP(const PHPObject& rhs, rhs)
#undef DECLARE_EQ_OP

	void serialize(ostream& os) const
{
	switch (getType()) {
		case NullType: os << 'N' << ';'; break;
		case AtomType: asAtom().serialize(os); break;
		case ArrayType: asArray().serialize(os); break;
		case MapType: asMap().serialize(os); break;
	}
}
	string serialize(void) const { ostringstream oss; serialize(oss); return oss.str(); }

	static PHPObject unserialize(string str)
	{
		istringstream iss(str);
		PHPObject obj;
		unserialize(iss, obj);
		return obj;
	}
	static int unserialize(istream& is, PHPObject& obj);

	friend ostream& operator<<(ostream& os, const PHPObject& obj)
	{
		switch (obj.getType()) {
			case NullType: os << "NULL"; break;
			case AtomType: os << obj.asAtom(); break;
			case ArrayType: os << obj.asArray(); break;
			case MapType: os << obj.asMap(); break;
		}
		return os;
	}
};
#undef PARENT

////////////////////////////////////////////////////////////////////////////////
// basic implementation (placed here due to recursive type)

#define PARENT vector<PHPObject>
inline PHPArrayObject::PHPArrayObject(): PARENT() {}
inline PHPArrayObject::PHPArrayObject(size_type n): PARENT(n) {}
inline PHPArrayObject::PHPArrayObject(size_type n, const PHPObject& t): PARENT(n, t) {}
inline PHPArrayObject::PHPArrayObject(const PHPArrayObject& other): PARENT(other) {}
#undef PARENT

#define PARENT map<PHPAtomObject, PHPObject, _phpobject_detail::PHPAtomObjectCompare>
inline PHPMapObject::PHPMapObject(): PARENT() {}
inline PHPMapObject::PHPMapObject(const key_compare& comp): PARENT(comp) {}
template <class T> inline PHPMapObject::PHPMapObject(T f, T l): PARENT(f, l) {}
template <class T> inline PHPMapObject::PHPMapObject(T f, T l, const key_compare& comp): PARENT(f, l, comp) {}
inline PHPMapObject::PHPMapObject(const PHPMapObject& other): PARENT(other) {}
#undef PARENT

#define std2qcstr(s) QCString(s.c_str())
#define qcstr2std(s) std::string(s)
#define std2qstr(s) QString::fromUtf8(s.c_str())
#define std2qstrenc(s, e) e->toUnicode(s.c_str())
#define qstr2std(s) std::string(s.utf8())
#define qstr2stdenc(s, e) std::string(e->fromUnicode(s))

#endif /* !_GAEA_PHPOBJECT_H_ */
