/*
 * TokigunStudio Gaea Library: PHPObject
 * Copyright (c) 2006, Kang Seonghoon (Tokigun).
 * This library can be distributed under the terms of the GNU LGPL.
 */

#include "gaea_phpobject.h"

using namespace std;

////////////////////////////////////////////////////////////////////////////////

#define TESTCHAR(is,ch) do { \
		int c = is.get(); \
		if (c != ch) return (c == EOF ? 2 : 1); \
} while(0)

#define READUNTIL(is,ch,value) while (true) { \
		int c = is.get(); \
		if (c == ch) break; else if (c == EOF) return 2; \
		value += c; \
}

// mode = 0: make array or map
// mode = 1: make array
// mode = 2: make map
static int unserialize_array(istream& is, PHPObject& obj, int mode)
{
	switch (is.get()) {
		case 'a': {
			TESTCHAR(is, ':');
			string value;
			READUNTIL(is, ':', value)
					int len = atoi(value.c_str());
			TESTCHAR(is, '{');

			vector<PHPObject> items;
			bool parity = false;
			while (is.peek() != '}') {
				PHPObject iobj;
				int result;
				if (parity = !parity) { // key
					result = PHPAtomObject::unserialize(is, iobj);
				} else { // value
					result = PHPObject::unserialize(is, iobj);
				}
				if (result != 0) return result;
				items.push_back(iobj);
			}
			TESTCHAR(is, '}');
			if (int(items.size()) != len * 2) return 1;

		// determine whether object is array or map
			bool isarray = true;
			for (int i = 0; i < len; ++i) {
				PHPObject& iobj = items[2*i];
			// check array: 0 v 1 v 2 v ... (len-1) v
				if (!iobj.isInt() || iobj.asInt() != i) {
					isarray = false;
					break;
				}
			}

			if (isarray) {
				if (mode == 2) return -1;
				PHPArrayObject array;
				for (int i = 1; i < 2*len; i += 2)
					array.push_back(items[i]);
				obj = array;
			} else {
				if (mode == 1) return -1;
				PHPMapObject array;
				for (int i = 0; i < 2*len; i += 2)
					array[items[i]] = items[i+1];
				obj = array;
			}
			return 0;
		}

		case EOF: return 2;
		default: return -1;
	}
}

////////////////////////////////////////////////////////////////////////////////
// PHPAtomObject implementation

void PHPAtomObject::serialize(ostream& os) const
{
	switch (getType()) {
		case NullType: os << 'N' << ';'; break;
		case BoolType: os << 'b' << ':' << (asBool() ? '1' : '0') << ';'; break;
		case IntType: os << 'i' << ':' << asInt() << ';'; break;
		case FloatType: os << 'd' << ':' << asFloat() << ';'; break;
		case StringType: {
			string& value = asString();
			os << 's' << ':' << value.length() << ':' << '"' << value << '"' << ';';
		} break;
	}
}

// -1: this class cannot unserialize stream (but someone can)
// 0: stream is successfully unserialized -- obj is modified
// 1: cannot unserialize this data but may sync to next data
// 2: cannot unserialize this data and cannot sync
// 3: some fatal error (e.g. operator new failed)
int PHPAtomObject::unserialize(istream& is, PHPObject& obj)
{
	char delim;

	switch (char mode = is.get()) {
		case 'N':
			TESTCHAR(is, ';');
			obj.clear();
			return 0;

		case 'b':
		case 'i':
		case 'd':
			delim = ';';
			if (false) { // ugly hack
				case 's':
					delim = ':';
			}
			{
				TESTCHAR(is, ':');
				string value;
				READUNTIL(is, delim, value)
						switch (mode) {
					case 'b': obj = (atoi(value.c_str()) != 0); break;
					case 'i': obj = atoi(value.c_str()); break;
					case 'd': obj = atof(value.c_str()); break;
					case 's': {
						int len = atoi(value.c_str());
						TESTCHAR(is, '"');
						char *buf = new char[len];
						if (!buf) return 3;
						if (is.readsome(buf, len) < len) {
							delete [] buf;
							return 2;
						}
						string result(buf, len);
						delete [] buf;
						TESTCHAR(is, '"');
						TESTCHAR(is, ';');
						obj = result;
					} break;
						}
						return 0;
			}

			case EOF: return 2;
			default: return -1;
	}
}

ostream& operator<<(ostream& os, const PHPAtomObject& obj)
{
	switch (obj.getType()) {
		case PHPAtomObject::NullType: os << "NULL"; break;
		case PHPAtomObject::BoolType: os << (obj.asBool() ? "true" : "false"); break;
		case PHPAtomObject::IntType: os << obj.asInt(); break;
		case PHPAtomObject::FloatType: os << obj.asFloat(); break;
		case PHPAtomObject::StringType: {
			string& str = obj.asString();
			os << '"';
			for (string::iterator it = str.begin(); it != str.end(); ++it) {
				if (*it == '\\' | *it == '"') os << '\\';
				os << *it;
			}
			os << '"';
		} break;
	}
	return os;
}
////////////////////////////////////////////////////////////////////////////////
// PHPArrayObject implementation

void PHPArrayObject::serialize(ostream& os) const
{
	int index = 0;

	os << 'a' << ':' << size() << ':' << '{';
	for (const_iterator it = begin(); it != end(); ++it, ++index) {
		os << 'i' << ':' << index << ';';
		(*it).serialize(os);
	}
	os << '}';
}

inline int PHPArrayObject::unserialize(istream& is, PHPObject& obj)
{
	return unserialize_array(is, obj, 1);
}

ostream& operator<<(ostream& os, const PHPArrayObject& obj)
{
	os << "array(";
	bool first = true;
	for (PHPArrayObject::const_iterator it = obj.begin(); it != obj.end(); ++it) {
		if (first) first = false; else os << ',' << ' ';
		os << *it;
	}
	os << ')';
	return os;
}

////////////////////////////////////////////////////////////////////////////////
// PHPMapObject implementation

void PHPMapObject::serialize(ostream& os) const
{
	os << 'a' << ':' << size() << ':' << '{';
	for (const_iterator it = begin(); it != end(); ++it) {
		(*it).first.serialize(os);
		(*it).second.serialize(os);
	}
	os << '}';
}

inline int PHPMapObject::unserialize(istream& is, PHPObject& obj)
{
	return unserialize_array(is, obj, 2);
}

ostream& operator<<(ostream& os, const PHPMapObject& obj)
{
	os << "array(";
	bool first = true;
	for (PHPMapObject::const_iterator it = obj.begin(); it != obj.end(); ++it) {
		if (first) first = false; else os << ',' << ' ';
		os << (*it).first << " => " << (*it).second;
	}
	os << ')';
	return os;
}

////////////////////////////////////////////////////////////////////////////////
// PHPObject implementation

inline int PHPObject::unserialize(istream& is, PHPObject& obj)
{
	int result;
	streampos pos_save = is.tellg();

	result = PHPAtomObject::unserialize(is, obj);
	if (result == -1) {
		is.seekg(pos_save);
		result = unserialize_array(is, obj, 0);
		if (result == -1) is.seekg(pos_save);
	}
	return result;
}

