<?php
/**
 * 시스템 기본 라이브러리
 *
 * 본 시스템(프로그램 전체)에서 사용되는 가장 기본 적인 필수적인 함수들의 묶음이다.
 *
 * @return boolean 시스템이 올바로 설치가 안되어 있으면 false 를 리턴한다.
 * @package library
 *
 * @important @note @since 2007/01/15 if $DONOTAUTH is set 1, then this library does not check any kind of user authentication.
 * This might be happen when the script calling this library would authenticate the user's information itself.
 *
 * @note 상수 dirRoot 와 workDir 은 같은 값을 가지는 상수로서 시스템 디렉토리를 가르킨다.
 * @note 상수 urlHomepage 는 홈페이지 주소를 가르킨다.
 */
 


/**
 * 시스템의 현제 작업 디렉토리 경로를 가지는 전역 변수이다.
 *
 * 모든 경로는 이 변수가 가지는 경로를 기준으로 상대적으로 기록(연결 경로)되어야한다.
 *
 * @global string $_dirRoot
 * @note 본 스크립트가 실행되기 전에, $rootDir, $dirRoot 또는 $_dirRoot 이 셋중에 하나의 변수에 값이 기록되어서 들어와야한다. 이 변수들은 결과적으로 dirRoot 문자열 상수에게 현재 경로를 지정하는 것이 된다.
 */
if ( isset( $rootDir ))
	$_dirRoot = &$rootDir;
else if ( isset( $dirRoot ))
	$_dirRoot = &$dirRoot;

/** 상수 정의 */
if ( isset( $_dirRoot ))
	define ('dirRoot', $_dirRoot, true);
else
	die ("No root dir is specified in any of rootDir, dirRoot, _dirRoot");


/**
 * 홈페이지 주소
 *
 * 현제 작업 디렉토리의 홈페이지 주소(URL)을 가지는 상수
 *
 * @since 2007/02/01 맨 뒤에 / 를 붙인다. 이것은 http://jangnans.com 과 같을 경우, http://jangnans.com?abc 와 같은 상황을 막기 위한 것이다. http://jangnans.com/index.html? 과같이 되게하기 위해서는 뒤에 / 를 붙여야한다.
 */
if ( isset($_SERVER['HTTP_HOST']) )
{
	$pp = pathinfo($_SERVER['PHP_SELF']);
	$url = "http://".$_SERVER['HTTP_HOST'].$pp['dirname'].'/';
	/**
	 * urlHomepage 가 이미 정의 되어 있다면, 재정의 하지 않는다.
	 * @note soaps 에서 이 값을 정의한다. soaps 에 의해서 이 스크립트가 불려질때에는 이 상수가 필요없다.
	 */
	if ( ! defined('urlHomepage') ) define('urlHomepage', $url, true);
}

// ============================================================================
//
//                  [ 기본 라이브러리 로드 ]
//
// ============================================================================
// 가장 기본 적인 라이브러리를 로드한다.
// 따라서 여기에 기록되는 함수는 다른 곳에서 로드할 필요가 없다.
/**
 * 파일 라이브러리 로드
 *
 * 시스템 설정을 읽기 위해서 필수적으로 로딩되는 함수이다. 따라서 파일 작업과 관련된 라이브러리는 다시 로딩될 필요없다.
 */
lib('file');




// ============================================================================
//
//                  [ 기본 값 (상수, 변수, 기타) 지정 ]
//
// ============================================================================
// 중복되지 않게 조심해야한다.
// 이곳에 설정되는 변수는 기본 변수로서 어디서든지 사용이 가능하다.
// @todo 내용이 많으면 include('default.var.php'); 로 모듈을 분기한다.
//
//
define ("workDir", dirRoot, true);
define ("br", "<br>", true);
define ("rn", "\r\n", true);					// what about ln



$faultExCode=$faultExString=NULL;






// ============= 웹브라우저로 부터 전달된 사용자 입력. =====================
//		주의: 이 $ui 값을 사용할 때에는 GET, POST 메소드가 섞여 있으며 중복될 경우,
//		POST 값이 유효하다.
//		필요할 경우 직접 웹브라우저 입력 변수를 가공해서 사용해야한다.
//		함수내에서는 $GLOBALS['ui'] 와 같이 사용을 할 수 있다.
//		@example echo $GLOBALS['ui']['cate'];

$ui = array_merge($_GET, $_POST);

/**
 * ?module=abc 를 ?skin=abc 와 같게 처리를 한다.
 *
 * /index.html?cate=bbs 나 /index.html?module=bbs 나 같은 것이다.
 *
 * cate=bbs 와 같이 입력 변수 cate 를 우선 순위로 두고 내부적으로 이용을 한다.
 *
 */
if ( empty($ui['cate']) && !empty($ui['module']) ) $ui['cate'] = $ui['module'];

// aliasing
$userInput = &$ui;




/**#@+
 * 소프트웨어 정보
 *
 * @note 이 변수들을 수정해서는 안된다.
 * @note $software[version] 은 아래와 같이 설치와 관련되어 있다. 따라서 임의로 수정을 하면안된다.
 * @code $file_sql = dirRoot."/etc/install/sql/$software[version].sql";
 */
$software['name']				= 'hometools';
$software['version']		= '2.2';
$software['patch']			= 'p0';//p2, p4, p6, p8, p10, p12 등등 짝수로 진행
$software['author']			= 'ky (thruthesky)';
$software['email']			= 'thruthesky@yahoo.co.kr';
$software['homepage']		= 'http://jangnans.com/';
$software['title']			= "$software[name] version $software[version]$software[patch]";
$software['copyright']	= "Powered by $software[name] version $software[version]\n\n$software[email]\n$software[homepage]";
/**#@-*/



/**#@+
 * 등급 코드 상수
 *
 * 사용자의 등급(권한) 표시를 위한 상수이다.
 * @note 만약, 어떤 정보에 대해서 관리자도 못보게 정보 권한을 설정하려면 GRADE_ROOT + 1 로 하면된다.
 *
 */
define("GRADE_PUBLIC",					0,	true);// define("GRADE_PUBLIC_TEXT",			"공개", true);
define("GRADE_MEMBER",					1,	true);// define("GRADE_MEMBER_TEXT",			"회원",	true);
define("GRADE_USER",	GRADE_MEMBER,	true);
define("GRADE_ADMIN",				10000,	true);// define("GRADE_ADMIN_TEXT",				"관리자",	true);
define("GRADE_SUPER",				30000,	true);// define("GRADE_SUPER_TEXT",				"슈퍼관리자",	true);
define("GRADE_ROOT",				30000,	true);// define("GRADE_ROOT_TEXT",				"계정관리자",	true);
define("GRADE_PRIVATE",			30001,	true);// define("GRADE_PRIVATE_TEXT",			"비밀",	true);
/**#@-*/



/**#@+
 * 이 상수 코드는  lib/category.php 에서 가져왔다.
 * @since 2007/01/16 더 이상 문자열의 값을 가지지 않고, 정수형으로 카테고리를 나타낸다.
 * 이로 인해서 카테고리를 나타내는 관련된 모든 것이 변경이되고, 글 영역, 파일 영역에 관련된 정보가 세팅된다.
 *
 * 카테고리별 구분 상수.
 * 카테고리 생성 등의 작업을 할 때 사용된다.
 * 타입 번호는 0 부터 255 까지이다. 0부터 99까지는 시스템에서 사용한다.
 * 카테고리 타입 99 는 기타(정해지지 않은) 타입이다.
 * 카테고리 타입 0 는 ok 의 값이며 카테고리에서는 에러의 값이다. 존재하지 않는 타입이다.
 * post.idx_category_type 과 연관되어 사용되는 값이다.
 */
define('CATE_NONE',					0);					// 없음. 카테고리 정보 자체가 존재하지 않는 경우.
define('CATE_USER',					1);
define('CATE_BBS',					2);					// 게시판, 토론실, 질문과 답변, 뉴스, 새소식, 신문, 기사, 강좌, 문서 등
	// 만약 새로운 모듈을 작성한다고 하더라도, 이런 류의 글을 다루면 CATE_BBS 타입이어야한다.
	// 그래야 검색이라든지 다른 기능에서 올바로 데이터가 이용이된다.
define('CATE_MEMO',					3);					// 게인 메모
define('CATE_BLOG',					4);					// 블로그, 수필, (일기장 제외)
define('CATE_MALL',					5);					// 쇼핑몰 상품 정보
define('CATE_MALL_ORDER',		6);					// 쇼핑몰 주문 정보
define('CATE_USER_INFO',		7);					// 사용자 정보에 대한 코멘트 관련 글

define('CATE_CATEGORY',			8);					// 카테고리 자체에 대한 카테고리 정보. 빌드가이드 [*] 카테고리 참고
																				// 메뉴가 이 타입을 가진다.

define('CATE_MESSAGE',			10);				// 송/수신 메세지. 쪽지,내부 메일 송/수신 관련. (웹메일, email은 제외)
define('CATE_TRACKBACK',		21);				// 트랙백,엮인글 글 영역
//define('CATE_DIARY',			40);				// 일기장, 일지, 기록표
define('CATE_ETC',					99);				// 기타. 카테고리 정보는 존재하는데, 표현을 하지 않는 경우.
/**#@-*/

/**
 * returns the type of category based on the given input string.
 * @since 2001/01/17 function name has changed from cateCode to cateType.
 */
function cateType($type)
{
	switch ($type)
	{
		case 'etc'				: $type = CATE_ETC;		break;
		case 'bbs'				: $type = CATE_BBS;		break;
		case 'memo'				: $type = CATE_MEMO;	break;
		case 'blog'				: $type = CATE_BLOG;	break;
		case 'user'				: $type = CATE_USER;	break;
		case 'mall'				: $type = CATE_MALL;	break;
		case 'category'		: $type = CATE_CATEGORY;	break;
		default : $type = CATE_ERROR;
	}
	return $type;
}






/**#@+
 * 오류 코드에 대한 상수 정의
 *
 * 오류 코드(값)들을 faultCode 라 하며 모두 음수의 값을 가진다.
 * 함수에서 값을 리턴할 때에 오류가 있으면 faultCode 를 리턴한다.
 * 각 함수에서 리턴하는 값들이다. 어떤 함수가 faultCode 를 리턴한다는 것은 그 함수가 올바로 수행이 되었을 경우 _ok 를 리턴하며 이것은 '거짓'의 값임을 의미한다. 그렇지 않으면(함수 수행에 문제가 있으면) 에러 코드가 리턴된다는 것을 의미하며 이것은 '참' 값을 가진다. 리턴되는 에러 코드와 그 설명은 아래와 같다.
 * 함수가 faultCode 를 리턴할 때에는 이 상수 값을 직접 리턴해서는 안되며, faultEx 함수를 통해서 에러 메세지 내용과 같이 리턴해야한다.
 * 그리고 faultCode 를 리턴하는 함수의 결과를 체크할 때에는 반드시 {@link faultCode()}를 통해야한다.
 *
 * @todo 여기서 기록되는 모든 상수와 상수 관련 문자열은 언어팩으로 이동해야한다.
 * @see etc/language/en.php
 * @since 2007/01/10 _ok, _error, _fault have been obsoleted.
 * @since 2007/01/28 여기에 정의되는 상수는 언어팩에서 lang 변수의 키값으로 적용되어 스크립트에서 사용이된다. 이때, lang 변수에서 아래의 오류 코드 상수를 사용할 때에는 $faultString 변수를 통해서 작업을 하도록 한다.
 */

define("ok",									0,			true);
define("fault",								-1, 		true);
define("failed",							fault, 	true);
	
define("wrongID",							-10,		true);
define("wrongPassword",				-11,		true);
define("wrongAddress",				-20,		true);
//
define("wrongIndex",					-101,	true);
define("accessDenied",				-111,	true);
define("databaseQuery",				-121,	true);
define("databaseInfo",				-122, true);

define("wrongInput",					-131,	true);
define("wrongParam",					wrongInput,	true);
define('duplicated',					-141,	true);
define('doneBefore',	duplicated,		true);
define("diskAccess",					-151, true);
define("deprecated",					-161, true);
define("wrongAction",					-400,	true);
//
define("notAvailable",				-500, true);

define("emptyName",						-1100);	// should be deprecatedd
define("emptyTitle",					-1101); // should be deprecatedd

/**#@-*/

	
/**
 * 입력 된 값이 오류 코드이면 true 를 리턴한다.
 *
 * 특정 함수로 부터 리턴되는 값이 일반적인 숫자값인지 오류코드인지 알아보기 위해서 사용되는 함수이다.
 * @param faultCode faultCode 오류 값
 * @return 오류이면 true 아니면 false
 *
 * @note 입력 값이 참 거짓에 상관없이, 에러가 있을 경우 참을 리턴한다. 따라서 아래와 같이 사용할 수 있다.
 * <code>
 * $res = getPosts($ui['idx'], 100);
 * if ( faultCode($res) ) goBack("bbs::list->getPosts at " . __FILE__ . " line: " . __LINE__);
 * faultCode($res) and goBack("bbs::list->getPosts at " . __FILE__ . " line: " . __LINE__);
 * </code>
 *
 * db.php 에 있는 faultDb 함수는 faultCode 를 리턴한다. 쿼리의 결과값을 리턴해야하는 경우, faultDb 를 사용하면 안된다.
 * faultCode (상수)를 리턴하는 경우, 즉, faultCode (함수)를 사용하거나 fualtDb 를 통해서 값을 리턴하는 경우에는 곧바로 and 를 이용해서 에러 문장을 만들 수 있다. 에러가 있으면 참을 리턴하며 and 를 이용해서 에러 문장을 출력하는 것이다.
 */
function faultCode($n)
{
	if ( ! is_int($n) ) return false;
	if ( $n >= 0 ) return false;
	return true;
}
/**
 * 단순히 faultCode 의 연결이다.
 *
 * 이 함수가 의미의 전달이 쉽다. 이 함수를 사용하도록 한다.
 * @see faultCode
 */
function error($n) { return faultCode($n); }














// ======================= [ 환경 설정 파일 로드 ] ============================
//
// 환경 설정 파일을 로드한 다는 것은 설치가 선행되어 있다는 뜻이다.
// 만약, 환경 설정 파일을 로드하지 못하면 설치가 안되었다는 것으로 간주하고
// 설치 과정을 진행해야한다.
//
// 시스템 기본 설정 값을 읽어서 기본 변수에 지정한다.
// 데이터베이스 정보, 스킨 정보 등을 설정한다.
//
// -------------------------- system 설정의 기본 값 지정
/**#@+
 * 시스템 환경 설정의 기본 값을 지정한다.
 *
 * 관리자 모드의 초기값을 지정한다. 시스템 설정은 include 로 로드하므로 미리 초기값을 주어도 상관이 없다.
 */
/**
 * 기본 스킨
 */
$system['skin'] = 'default';


/**
 * 기본 리스트 갯수
 */
$system['numberOfRecords'] = 10;

/**
 * 페이징 블럭에서 기본 페이지 갯수
 */
$system['pagesInBlock'] = 10;
/**
 * 새 정보 표시 시간 범위
 */
$system['newDate'] = 24;
/**
 * 레이아웃 테이블 보더
 */
$system['layout_border'] = 0;
/**
 * 파일 저장 공간.
 *
 * 파일 저장 공간은 상대 경로로 지정할 때, ./file 과 같이 기록이 되면 안되고, dirRoot.'/file' 과 같이 기록이되어야한다. 그래야 다른 스크립트에서 올바로 연결이 가능하다.
 */
$system['repository'] = dirRoot.'/file';
/**#@-*/


/**
 * 시스템 정보 파일
 *
 * 시스템 정보 파일이다. 이 파일은 중요한 정보를 가지므로 패스를 변경해서 웹에서 액세스가 불가능 한 곳에 놓을 수 있다.
 * 이곳은 시스템 정보 파일을 로드하기 전에 기본값을 입력하는 곳이다.
 * 따라서, 위의 $system['repository'] 의 값을 변경하면, system.conf.php 의 위치를 /etc 와 같은 곳으로 이동하게 할 수도 있다.
 */
$pathSystem					= $system['repository']."/system.conf.php";


/**
 * 시스템 설정 정보(파일) 로드
 *
 * 설정 파일이 없는데, install 모듈이 아니면, 설치 화면으로 간다.
 * 설정 파일 로드에 문제가 있으면 에러를 리턴한다. 즉, 설치를 하는 과정으로 이동이된다.
 * $system 변수는 다른 곳에서 수정될 수 없다. 읽기 전용이다.
 * @return boolean 에러가 있으면 false
 * @note 설치 과정에서는 리턴을 하지 않는다.
 */
if ( (! @include($pathSystem)) && ( $ui['cate'] != "install" ) )
{
	logout();
	return false;
}

/**
 * 시스템 설정 파일의 값이 $system 변수에 세팅되었다.
 * 에러 처리를 한다.
 */
if ( isset($system['skin']) && empty($system['skin']) ) $system['skin'] = "default";

/**
 *
 */
define("numberOfRecords", 					$system['numberOfRecords'], true);
define("newDate",										$system['newDate'], true);
define("pagesInBlock",							$system['pagesInBlock'], true);
define("numberOfPages",							pagesInBlock);



//
//
// ========================== [ 환경 설정에 따라 달라지는 값 ] ================
// @todo 파일 저장 공간은 설정에서 따로 할 수 있도록 한다.
//
//



/**
 * 환경 설정에 의해서 변경이 될 수 있다. 따라서, 환경 설정을 로드한 다음에 정의한다.
 *
 * 만약, $system[repository] 가 . 으로 시작한다면, $system['repository'] = dirRoot.'/'.$system['repository']; 와 같이 변경을 해야한다.
 * 이것은 디렉토리 위치가 다른 스크립트 간의 연결을 맞추기 위한 것이다. 예를 들어, ./index.html 에서 파일 저장 공간을 액세스하는 것과 ./siteapi/index.php 에서 저장 공간을 액세스하는 것을 동일하게 맞추기 위한 것이다.
 */
if ( $system['repository'][0] == '.' && $system['repository'][1] == '/' )
{
	$system['repository'] = substr($system['repository'],2);
	$system['repository'] = dirRoot.'/'.$system['repository'];
}

define ("tmp_repository",					$system['repository'].'/var', true);
define ("data_repository",				$system['repository'].'/data', true);
define ("log_repository",					$system['repository'].'/log', true);
define ("module_repository",			$system['repository'].'/module',true);


// 맨 처음, 맨 끝에 실행될 스크립트를 지정
define ("file_scripts",						$system['repository'].'/scripts.php', true);


/**
 * Language pack.
 *
 *
 * This could be changed in admin page.
 * @todo 웹브라우저의 지역별 설정을 보고 수정을 할 수 있도록 한다.
 */
if ( ! isset($system['language']) || empty($system['language']) ) $system['language'] = 'ko';
include( dirRoot."/etc/language/$system[language].php");


/**
 * 오류 코드에 대한 오류 문자열은 언어팩에 지정되어 있다.
 * $faultString 변수를 이용하는 이유는 오류 코드를 사용한다는 보다 정확한 표시를 위해서이다.
 */
$faultString = &$lang;





















// ============================================================================
//
//                  [ 주소 변환 ]
//
// ============================================================================
// @since 2007/01/19 블로그 옵션에 따라서, 주소 변환이 환경 설정을 로드 한 다음에 실행되도록 했다.
// 본 시스템은 기본적으로 cate 와 mode 에 의해서 어떤 모듈(스크립틐)을 실행할지 결정을 한다.
// 그러나 cate 와 mode 만으로 표현이 어렵거나 예외를 두어야하는 경우, 이곳에서 처리를 한다.
//
//		cate 와 mode 의 값이 empty 일 경우, 짧게 입력하는 URI 값을 바탕으로 게시판과 연결이 되도록 주소 변환을 시도한다.
//		URI 가 '=' 기호 표시가 없이, /?free 와 같은 경우 게시판 리스트를 하게한다.
//		아래와 같이 미리 정해진 URI 의 경우 특정 수행을 한다. 이것은 URL 작성을 편리하게 한다.
//		미리 정해진 URI: /?logout, /?user_read etc,...



/**
 * 주소 변환
 *
 * @since 2007/02/12 "/?1234" 와 같이 query string 이 is_numeric 조건을 만족할 때, 게시물 읽기 주소로 변환을 한다.
 *	예를 들면, /?1234 와 같이 입력된 경우, $ui[cate]=bbs, $ui[mode]=read, $ui[idx]=1234 가 된다.
 *
 */
if ( !isset($_SERVER['REQUEST_METHOD']) )
{
	// 커맨드 라인에서 실행이 될 때,
}
else
{
	/**
	 * thruthesky.jangnans.com 과 같이 접속을 했을 때, thruthesky 사용자의 블로그로 연결 시킨다.
	 *
	 */
	if ( isset($system['subdomain_as_userid']) && $system['subdomain_as_userid'] == 'checked' && isset($system['domain']) && !empty($system['domain']))
	{
		$host = $_SERVER['HTTP_HOST'];
		if ( strpos($host, $system['domain']) )
		{
			$subdomain = str_replace('.'.$system['domain'], '', $host);
	
			if ( !empty($subdomain) && $subdomain != 'www' )
			{
				$_SERVER['QUERY_STRING'] = "/$subdomain";
			}
		}
	}
	
	//
	// cate, mode 가 empty 이면 변환을 시도한다.
	//
	if ( empty($ui['cate']) && empty($ui['mode']) )
	{
		//
		// !! cate,mode 가 empty 이고 id,password 값이 들어온 경우 사용자 인증을 한다.
		//
		if ( isset($ui['id']) && isset($ui['password']) )
		{
			$ui['cate'] = "user";
			$ui['mode'] = "login_submit";
		}
		//
		// 쿼리에 = 이 들어가면 주소 변환을 하지 않는다.
		//
		else if ( $_SERVER['QUERY_STRING'] && strstr($_SERVER['QUERY_STRING'], '=') === FALSE )
		{
			//
			// 쿼리가 /?/userid 와 같을 때에는 블로그로 연결을 한다.
			//
			if ( $_SERVER['QUERY_STRING'][0] == '/' )
			{
				$ui['cate']		= 'blog';
				$ui['mode']		= 'index';
				$ui['id']			= substr($_SERVER['QUERY_STRING'], 1);
			}
			//
			// 쿼리가 /?readme 또는 /?1234 와 같을 때,
			//
			else
			{
				// 쿼리가 숫자로 들어오면 게시물 읽기
				if ( is_numeric($_SERVER['QUERY_STRING']) )
				{
					$ui['cate'] = 'bbs';
					$ui['mode'] = 'read';
					$ui['idx']	= $_SERVER['QUERY_STRING'];
				}
				// 문자로 들어오면 게시판 리스트
				else
				{
					switch ( $_SERVER['QUERY_STRING'] )
					{//
						case ""							: break;
						case "admin"				: $ui['cate'] = "admin"; break;
						case "help"					: $ui['cate'] = 'help'; break;
						case "memo"					:	$ui['cate'] = 'memo'; break;
						case "install"			:	$ui['cate'] = 'install'; break;
						case "signup"				: $ui['cate'] = "user";			$ui['mode'] = "signup"; break;
						case "logout"				: $ui['cate'] = "user";			$ui['mode'] = "logout"; break;
						case "logout_submit":	$ui['cate'] = "user";			$ui['mode'] = "logout_submit"; break;
						case "userinfo"			: $ui['cate'] = "user";			$ui['mode'] = "read"; break;
						case "user_modify"	: $ui['cate'] = "user";			$ui['mode'] = "modify"; break;
						case "leave"				: $ui['cate'] = "user";			$ui['mode'] = "leave"; break;
						case 'password'			: $ui['cate'] = 'user';			$ui['mode'] = 'password';break;
						default								:
								$ui['cate'] = 'bbs';
								$ui['mode'] = "list";
								$ui['id'] = $_SERVER['QUERY_STRING'];
							break;
					} // eo switch
				} // eo if
			}
		}
	}
	
	if ( empty($ui['cate']) )
		$ui['cate'] = "index";
	if ( empty($ui['mode']) )
		$ui['mode'] = "index";
} // EO 주소 변환












// 
// 아래와 같이 자주 사용되는 변수들은 타이핑 횟수를 줄이기 위해서 참조 변수를 만든다.
// @note 위치가 여기여야한다. 여기서 $ui['cate'] 변수가 존재한다는 보장이 된다.
// 이 변수의 사용을 조심해야한다.
$cate = &$ui['cate'];
$mode = &$ui['mode'];
// 이전 페이지
if ( isset($_SERVER['HTTP_REFERER']) ) $url_prevpage = $_SERVER['HTTP_REFERER'];
else $url_prevpage = NULL;








/**
 * 스킨 (직접) 지정
 *
 * 스킨은 $system[skin] 에 지정된다. 이 값이 변경되면, 스킨의 참조가 엉망이된다.
 * 스킨 파일이 존재하는 디렉토리를 변경하고자 할 경우, changeSkin() 함수를 사용해야한다.
 *
 * 웹브라우저의 주소 입력창을 통해 HTTP URL 입력 변수 skin 에 값을 지정하면,
 * 이 값에 따라 시스템에 지정된 스킨을 변경하는 것이다.
 * 예를 들어 관리자가 스킨 설정을 잘 못해서, 관리자 화면으로 들어가지 못할 때  http://jangnans.com/?cate=system&mode=modify&skin=default 와 같이 해서 스킨을 변경할 수 있다.
 * 또는 실시간으로 스킨을 변경하고자 할 때 사용이 가능하다.
 * 주로 테스트를 위한 목적으로 사용을 한다.
 * @since 2007/01/10 스킨의 값은 쿠키로 저장을 해서 한번 선택을 하면 계속 사용 가능하게 한다.
 * @since 2007/01/16 스킨 직접 지정은 GET 메소드로만 되게 한다.
 */
if ( ( isset($_GET['skin']) && ! empty($_GET['skin']) ) )
{
	
	/**@note 한번 skin 을 지정하면, 세션값은 매번 지정한다. */
	session('skin', $_GET['skin']);
	$system['skin'] = $_GET['skin'];
}
else if ( session('skin') ) $system['skin'] = session('skin');



$dirSkin						= dirRoot."/skin/$system[skin]";


/**
 * reference of $dirSkin
 */
$sd									= &$dirSkin;




/**
 * 실시간으로 스킨 파일 디렉토리를 변경한다.
 *
 * 스크립트 실행 중에 스킨을 변경해야할 경우, 반드시 이 함수를 통해서 변경을 해야한다.
 */
function changeSkin($name)
{
	if ( empty($name) ) return;
	global $dirSkin, $system;
	$system['skin'] = $name;
	$dirSkin				= dirRoot."/skin/$system[skin]";
}





/**
 * @deprecated @since 2006/12/13 데이터베이스 정보를 위해서 전역 변수를 사용하지 않는다.
 */
/*
$_db_user						= &$system['db_user'];
$_db_password				= &$system['db_password'];
$_db_database				= &$system['db_database'];
$_db_host						= &$system['db_host'];
*/










///////////////////////////////////////////////////////////////////////////////
//
//
// 사용자 정보 체크
//
//
///////////////////////////////////////////////////////////////////////////////

// 5.2 ====================== 로그인을 한 경우, 사용자 정보 설정
//
//	로그인이 되어 있는 상태이면, DB 에 접속해서 사용자 정보를 설정한다.
// 
//	@note 이름이 석자이면, 예를 들어서 '홍길동' 이면, 성을 제외한 이름 '길동'만 따로 쓰일 수 있도록 한다.
/**
 * 사용자 정보
 *
 * @note $DONOTAUTH 변수가 1로 지정되어 있으면, 사용자 정보 관련 체크는 하지 말아야한다.
 * @since 2007/02/02 로그인 설치 과정에서 DB 미리 접속을 하지 않아야한다.
 * @todo 설치 과정과 default.php 에서 DB 접속 관계 등이 엇갈려있다. 정리가 한번 필요하다.
 */
if ( login() && $ui['cate'] != 'install' && (!isset($DONOTAUTH) || empty($DONOTAUTH)) )
{
	lib('db');
	lib('user');

	/**@note 기본 변수*/
	$user = userData(session('id'));
	
	// ======================== 로그인 세션이 있는 상태에서 회원 정보를 찾을 수 없는 경우, 예를 들면 회원 정보 수정에서 아이디 변경을 한 경우, 이 때에는 로그아웃을 시키고 메인 화면으로 이동시킨다.
	if (! $user )
	{
		logout();
		go("?");
	}
	
	
	// 사용자 로그인이 되었다면, 스킨을 변경해 준다.
	changeSkin($user['skin']);
	


	$member = &$user;		// $user 변수의 참조자
	$user['firstname'] = firstName($user['name']);
	// @todo 닉네임 처리
	$user['nick'] = nickname();



	/** 성별 표기를 $user[sex] 변수에 기록한다. 이 값이 없으면, 주민번호의 값을 보고 지정한다. */	
	// 성별. sexkr 변수에 기록
	// 남자인지 여자인지 구분하는 변수. $user['sex'] 는 영문 한글자로 되어 있다.
	// 그리고 이 값은 필수가 아니므로 empty 일 수 있다.
	if ( empty($user['sex']) && empty($user['jumin2']) ) ;
	else if ( $user['sex'] )
	{
		$user['sexkr'] = sexKr($user['sex']);
	}
	else if ( $user['jumin2'] )
	{
		$user['jumin2'][0] == '1' ? $user['sex']='M' : $user['sex']='F';
		$user['sexkr']	= sexKr($user['jumin2']);
	}
}

/**
 * 성별을 텍스트로 표시한다.
 *
 *
 * @param string $s F, M, 1903923 과 같은 값을 가질 수 있다.
 * @return string '남자', '여자', '모름' 중 하나를 리턴한다.
 */
function sexKr($s)
{
	if ( empty($s) ) return '모름';
	else if ( $s == 'M' ) return '남자';
	else if ( $s == 'F' ) return '여자';
	else if ( $s[0] == '1' ) return '남자';
	else if ( $s[0] == '2' ) return '여자';
	return '모름';
}


/**
 * sexKr 과는 달리 사용자 정보 전체를 입력받아서 주민번호, 생일 등의 값을 기준으로 나이를 판단한다.
 *
 * @param ref-associative-array 사용자 데이터베이스 레코드
 * @return int 나이 or NULL
 *
 */
function age(&$user)
{
	if ( $user['jumin1'] )
	{
		$year = substr($user['jumin1'], 0,2);
		if ( $user['jumin2'][0] == '1' || $user['jumin2'][0] == '2' )
		{
			$year = 1900 + $year;
		}
		else $year = 2000 + $year;
		return date("Y") - $year + 1;
	}
	else if ( $user['birth_year'] )
	{
		return date( "Y" ) - $user['birth_year'] + 1;
	}
	else return NULL;
}







// 6. ======================= 권한 체크 =======================================
// 기본적으로 checkAuthority 함수를 사용한다. 필요한 경우 각 카테고리의 begin.php 에서 호출한다.

//checkAuthority();


















//
///////////////////////////////////////////////////////////////////////////////
//
//
// 7. 기본 변수. 스킨에서 사용할 기본 값.
//
//
///////////////////////////////////////////////////////////////////////////////
//
// 만약, 개별적으로 모듈을 만들 경우, 시스템 코드를 만지기가 힘들다.
// 기본 변수를 지정해야할 경우가 있으면, 모듈에서 시작스크립트에 등록해서
// 모든 스킨에서 사용이 가능하게 할 수 있다.

// rss 모듈로 넘길 값. 주소 변한 작업후에 지정.
isset($ui['id']) ? $__id = $ui['id'] : $__id=NULL;
isset($ui['idx']) ? $__idx = $ui['idx'] : $__idx=NULL;
$url_rss = "?cate=rss&id=$__id&idx=$__idx";
unset($__id); unset($__idx);










// ////////////////////////////////////////////////////////////////////////////
//
// 8. 세션 변수 기록
//
// @doc 이 변수를 활용해야한다.
//
// ////////////////////////////////////////////////////////////////////////////


/**
 * 클라이언트별 세션 아이디
 *
 * 모든 클라이언트는 접속(세션)마다 고유한 세션 키를 가진다.
 */
$sessionid = session("sessionid");
if ( $sessionid === NULL )
{
	$uniqid = md5(uniqid(rand(), true));
	session("sessionid", $uniqid);
}





























//					시스템 함수

/**
 * 시스템 설정 값을 읽는다.
 *
 * @param mixed $k associative-array 에서 사용되는 키
 * @return mixed 값이 없으면, 세팅 되지 않았으면 NULL 리턴. 아니면 값을 리턴
 * @code sv('levelPost')
 */
function sv($k)
{
	if ( isset($GLOBALS['system'][$k]) ) return $GLOBALS['system'][$k];
	else return NULL;
}



















/**
 * 파일 명칭만 입력받아서 해당 파일을 단 한번만 인클루드되게 한다.
 *
 * @note 입력 형식 예: lib/system
 * @param string $file 상대 경로를 포함한 파일 명칭. 예, lib/system 은 lib 디렉토리의 system.php 파일을 가르킨다.
 * @code includeOnce("lib/system"); Once("lib/bbs");
 * @since 12/12/2006 함수 내에서 include 를 할 때에 전역변수를 사용하기 위해서 루프 처리를 했는데, 더 이상 유효하지 않다.
 */
function includeOnce($file)
{
	// @deprecated
	// foreach ( $GLOBALS as $k=>$v ) { $$k = $v; }
	include_once(syspath($file));
}
function Once($file) { includeOnce($file); }

// 인클루드 경로를 만드는 함수이다. 입력되는 파일 명칭의 형식은 "module/module-name/mode" 이다.
function syspath($file) { return sysp($file . ".php"); }
function sysp($file) {return dirRoot."/".$file;}
/**
 *
 * HTML("menu/cate");
 * LIB("system"); 문장은 includeOnce("lib/sytem"); 과 같으며 Once("lib/system"); 과 같다.
 */
function LIB($file){Once("lib/$file");}
function HTML($file){include("html/$file");}






/**
 * 스킨 파일 경로를 리턴한다.
 *
 * 모든 스킨 파일의 경로는 이 함수를 통해서 이루어져야한다. 스킨 경로와 관련된 작업을 할 때에는 이 함수 하나만 수정하면 전체가 다 같이 수정이된다.
 *
 * @param string $file 스킨 파일을 로드할 파일 이름. 이 값이 생략되면 $cate, $mode 변수에 따라서 스킨 파일을 로드한다.
 * @doc 메인 페이지를 나타내는 경우 cate=index, mode=index 이다. 스킨이 index.index.html  로 되니 이것의 이름이 이상하다. 따라서 특별히 index.html 로 되게한다.
 * @return string 스킨 파일 경로
 * @since 2007/01/10 URL 변수 include 에 값이 들어올 경우, 그 값을 스킨 HTML 파일로 인식을 하고 인클루드한다. 즉, 임시 HTML 파일을 보여줄 때 사용한다.
 * @since 2007/02/11 입력변수 $file 에 ".html" 이 들어오면 자동으로 처리를 한다. 즉, "index.header" 나 "index.header.html" 이나 같은 것으로 처리를 한다. 따라서 "index.html.html" 과 같이 파일 명칭에 ".html" 이 여러번 들어가면 안된다.
 */
function skin($file='')
{
	global $ui, $dirSkin;
	
	//
	if ( $file ) $file = str_replace(".html", "", $file);

	/**@important @note $ui[include] 에 값이 있으면 그 파일을 로드한다. 본문에만 넣어주면 된다.*/
	
	if ( empty($file) && isset($ui['include']) && !empty($ui['include']) )
	{
		$inc = $ui['include'];
		if ( strpos($inc, "file/") === FALSE)		// 첨부 파일 영역의 정보는 인클루드할 수 없도록 한다.
		{
			return $inc;
		}
		return skin("wrongpage");	// 첨부 파일 영역을 보려한다면, 현제 스킨의 wrongpage 를 리턴한다.
	}
	

	// $file 의 값이 empty 일 경우, 지정된 cate, mode 에 따라 파일을 로드
	if ( empty($file) && $ui['cate'] == 'index' && $ui['mode'] == 'index' ) $file = "index";
	else if ( empty($file) ) $file = "$ui[cate].$ui[mode]";
	$skinA = "$dirSkin/$file.html";
	
	// 스킨 디렉토리에 스킨 파일이 존재하는가?
	if ( file_exists( $skinA) )
	{
		return $skinA;
	}
	// 그렇지 않다면 기본 스킨을 사용
	$skin = dirRoot."/skin/default/$file.html";
	if ( file_exists( $skin) )
	{
		
		//g($skin . " default exists");
		return $skin;
	}
	
	// 스킨 디렉토리에도 스킨 파일이 없고, 기본 스킨 디렉토리에도 스킨 파일이 없다.
	// 이 경우, 스킨 디렉토리의 경로를 리턴한다.
	//g($skin . " does not exists");
	return $skinA;
}

function header_skin() { return skin("header"); }
function category_header_skin() { return skin($GLOBALS['ui']['cate'].".header"); }
function category_footer_skin() { return skin($GLOBALS['ui']['cate'].".footer"); }
function footer_skin() { return skin("footer"); }




/**
 * 현제 모듈에 있는 스크립트 파일 인클루드 경로 리턴
 *
 * 현제 모듈에 있는 스크립트 파일을 인클루드 할 수 있는 경로를 리턴한다.
 * 
 * @param string 스크립트 파일 이름(확장자 .php 는 제외)
 * <code>
 * include( script('info') );
 * </code>
 *
 */
function script($filename)
{
	return syspath("module/".$GLOBALS['ui']['cate']."/".$filename);
}

/**
 * 템플릿의 경로를 리턴한다.
 *
 */
function template($filename)
{
	return dirRoot."/skin/template/$filename.html";
}


/**
 * 입력값이 있을 경우에는 입력 값을 바탕으로 없으면, 'URL 값'을 바탕으로 해당 모듈 파일의 경로를 리턴한다.
 *
 *
 * @see 개발자노트#'URL 값'
 * @return string 인클루드될 모듈의 파일 경로
 *
 */
function module()
{
	global $ui;
	return syspath("module/$ui[cate]/$ui[mode]");
}
/**
 * module_begin, module_end 함수는 해당 모듈(정보 관련 영역의 스크립트)의 소스 프로그램(스크립트)이 실행되기 이전과 이후에 인클루드되어 실행될 스크립트 파일의 경로를 리턴한다.
 *
 * @return string 스크립트 파일 경로
 */
function module_begin()
{
	global $ui;
	return syspath("module/$ui[cate]/begin");
}
// @see module_begin
function module_end()
{
	global $ui;
	return syspath("module/$ui[cate]/end");
}





//??
/*
function CODE($file, $id='')
{
	list($cate, $bbs) = split('/', $file);
	includeModule($cate,$bbs);
}
*/


/**
 *
 * debug_backtrace() >= 4.3
 * @todo 이 함수는 사용되지 않는다. 6월 이후에 모두 삭제한다.
 */
function deprecated()
{
	assert(0);
	$ar = debug_backtrace();
	$b1 = $ar[1];
	g("called $b1[file] at $b1[line]");
	javascriptAlert("You have called deprecated function");
	return false;
}







































	

// ================ 시간 관련 함수 ============================================
// 따로 모듈을 만들지 않고 기본 함수에 포함한다.
/**
 * 기본 날짜 시간 함수.
 *
 * 시간 형식: 년도4자리,월,일,시,분,초 각각 2자리
 * @param int $timet 이 값이 참이면 전달된 값은 Unix Time Stamp 값으로 간주하고
 * 해당 값을 바탕으로 기본 시간 날짜 형식 포멧을 만들어 리턴한다.
 * @return string '기본 시간 날짜 형식'
 * @code $limit['fromNumber']					= dateTime(mktime(0,0,0,date("m"), date("d")-7, date("Y")));
 */
function dateTime($timet='') {

	if ( $timet && is_numeric($timet))
	{
		return date("YmdHis",$timet);
	}
	else
		return date("YmdHis");
}
/**
 * @deprecated
 *
 * Unix TIME-STAMP 값을 입력받아서 Iso8601.dateTime 형식으로 변환 후 리턴한다.
 *
 * @return string Iso8601.dateTime
 */
function toIso8601($timet) { return iso8601($timet); }

/**
 * iso8601.dateTime 형식을 리턴
 *
 * Unix TIME-STAMP 값을 입력받아서 Iso8601.dateTime 형식으로 변환 후 리턴한다.
 *
 * @return string Iso8601.dateTime
 */
function iso8601($timet=NULL)
{
	if ($timet === NULL) $timet = time();
	return strftime("%Y%m%dT%H:%M:%S", $timet);
}
/**
 * 기본 날짜 시간 형식을 입력 받아 Unix Time Stamp 값으로 변환해서 리턴한다.
 *
 * @param string 기본 날짜 시간 형식을 가지는 문자열
 * @return int Unix Time Stamp
 */
function toStamp($dateTime)
{
	$dt = decodeDateTime2($dateTime);
	return mktime( $dt['hour'], $dt['minute'], $dt['second'], $dt['month'], $dt['day'], $dt['year'] );
}

// 년월일을 리턴
function today()
{
	return date("Y년m월d일");
}
// 년 월 일, 시 분, 요일 형식의 문자열을 리턴
// 입력값은 기본 날짜 시간 형식이다.
function day($dateTime)
{
	return date("y/m/d h:ia", toStamp($dateTime));
	return strftime("%y/%m/%d %a %p %I시%M분", toStamp($dateTime));
}
/**
 * 간단한 날짜-시간 형식의 표현
 *
 * 오늘 날짜가 들어오면 시/분 의 형태를 티런한다.
 * 올해의 날짜가 들어오면 월/일 형태를 리턴한다.
 * 아니면 년/월/일 형태를 리턴한다.
 * @param string dateTime 형
 * @return string
 */
function dt($dateTime)
{
	if (empty($dateTime)) return "";
	$time = toStamp($dateTime);
	// 오늘 날짜
	if ( date("Ymd",$time) == date("Ymd") )
		return date("h:i a", $time);
	if ( date("Y", $time) == date("Y") )
		return date("m/d", $time);
	else
		return date("m/d/y", $time);
}
/**
 * 일의 초 값을 리턴한다.
 *
 * @param int $day 하루 이틀, 일의 수를 나타낸다.
 * @return int second 값. 하루는 60 초 * 60 분 * 24 시간이다. 이 값에 일을 곱해서 리턴.
 * @code $limit['fromDate'] = dateTime(time()-stampday(7));
 */
function stampday($day)
{
	return 60 * 60 * 24 * $day;
}





/**
 * 새 정보인지 체크를 한다.
 *
 * 환경 설정 파일의 $system['newDate'] 의 값을 기준으로 새로운 정보인지 체크를 한다.
 * @see 개발자노트
 * @param dateTime $dateTime 정보의 시간
 * @return 새 정보이면 true, 아니면 false
 */
function isNew($dateTime)
{
	$old = toStamp($dateTime);
	$now = time();

	if ( ($old + newDate * 60 * 60) > $now ) return true;
	else return false;
}




/**
 * 기본 날짜 시간 형식을 입력 받아 연관 배열로 분석해서 리턴한다.
 *
 * @param string $dateTime 기본 날짜 시간 형식 시간을 표현하는 스트링 문자열
 * @return 연관배열로 날짜 정보를 리턴한다.
 * year		년도 4자리
 * month	월
 * day		일
 * hour		시간
 * minute	분
 * second	초
 *
 */
function decodeDateTime2($dateTime)
{
	$ar = array();
	if ( empty($dateTime) ) return $ar;
	if(preg_match('/([0-9]{4})([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{2})([0-9]{2})/', $dateTime, $regs))
	{
		$ar['year']		= $regs[1];
		$ar['month']	= $regs[2];
		$ar['day']		= $regs[3];
		$ar['hour']		= $regs[4];
		$ar['minute']	= $regs[5];
		$ar['second']	= $regs[6];
	}
	return $ar;
}



/**
 * 검색 날짜 값을 파싱한다.
 *
 * search-date
 *
 * @see buildguide#search-date
 * @return associative-array of associative-array
 *	$ar['date']	날짜값 하루만 들어온 경우,
 *	범위 값이 들어온 경우에는 ar['begin'] ar['end'] 에 값이 기록된다. 둘 중 하나는 empty 가 될 수 있다.
 *
 */
function sdate($date)
{
	$ar = array();
	
	$found = strpos($date, '-');
	
	if ( $found === false ) $ar['date'] = pdate($date);
	else
	{
		list($b,$e) = explode('-', $date);

		$ar['begin'] = pdate($b);
		$ar['end'] = pdate($e);
	}
	return $ar;
}

/**
 * 문자열로된 날짜를 해쉬 배열로 리턴한다.
 *
 * 2007/03/04 07/3/4 와 같이 값이 들어올 수 있다.
 * 리턴되는 값은 년도 4자리, 월, 일 로서 모두 숫자의 값이다.
 */
function pdate($date)
{
	
	@list($y, $m, $d ) = explode("/", $date);
	if ( !$y || !$m || !$d ) return array();
	
	$as['y'] = $y + 0;
	$as['m'] = $m + 0;
	$as['d'] = $d + 0;
	
	if ( $as['y'] < 100 ) $as['y'] += 2000;
	return $as;
}







/**
 * 디버깅(실행) 시간 체크
 *
 * 디버깅 시간 기록 함수.
 * @since 2007/04/01 jangnan.php 에서 이곳으로 옮겨왔다.
 */
function microtime_float()
{
	list($usec, $sec) = explode(" ", microtime());
	return ((float)$usec + (float)$sec);
}





























// ========================== 사용자 로그인 및 권한, 정보 관련 함수 =================
//	사용자 로그인 관련 함수는 따로 lib/user.php 라이브러리에 두지 않고, 기본 함수로 포함시킨다.
//	
//
//
// subadmin, admin, super, root 세개의 함수가 있다.
// subadmin 은 해당 카테고리(정보 영역) 한 개만 관리할 수 있다.
// admin 은 모듈을 관리할 수 있다.
// super 는 전체 카테고리와 정보를 관리할 수 있다.
// root 는 시스템 설정을 할 수 있다.


/**
 * 해당 정보 영역(1개의 카테고리)에 대한 권한이 있는지 검사를 한다.
 *
 * category 테이블의 manager 항목에는 해당 카테고리를 관리할 수 있는 사용자의 아이디가 기록이되어있다.
 * 입력된 category.manager 에 해당 사용자 사용자가 기록되어있으면 참을 리턴하낟.
 *
 * 해당 정보 영역의 관리자일 경우에는 그냥 true 를 리턴한다.
 * 이 함수는 가장 낮은 권한의 관리자를 체크한다. 게시판과 같은 곳에서 이 함수를 사용해야한다.
 *
 * @param string $managers category.manager
 * @param string $module 현제 모듈 이름. 생략시 $cate 변수를 사용한다.
 * @return boolean 권한이 올바르면 true, 아니면 false 를 리턴한다.
 * @example ../module/bbs/begin.php
 */
function subadmin($managers, $module=NULL)
{
	if ( ! login() ) return false;
	if ( $module == NULL ) $module = $GLOBALS['ui']['cate'];
	if ( admin($module) ) return true;
	
	$ar = explode(",", $managers);
	return in_array($GLOBALS['user']['id'], $ar);
}


/**
 * check if the user logged-in is category admin.
 *
 * {@link isAdmin}과 같은 역활을 한다. 차이점은 결과에 대하여 isAdmin 은 faultCode 를 리턴하고 admin 은 boolean 을 리턴한다.
 *
 * @param string $category 카테고리(정확하게는 모듈) 이름. 생략되면, 관리자면 무조건 true 리턴.
 *	중요: 관리자면 무조건 true 리턴이라는 것은 user.manager 에 관리할 수 있는 특정 모듈이 기록되어있으면 
 *	무조건 true 를 리턴하는 것이다.
 *	즉, 어떤 모듈이든지 상관없이 관리자면 무조건 true 를 리턴한다.
 *
 * @return boolean 관리자 일 경우 참, 아니면 거짓
 *
 * @note 주의: 로그인을 하고 있는 상태에서 이 함수는 실패할 것이다.
 *	예를 들면, user/login_submit.php 에서 로그인을 성공했다고 하나, 세션이 적용되려면,
 *	페이지가 변경이되어야한다. 따라서 user/login_submit.php 에서는 이 함수가 실패할 것이다.
 */
function admin($category=NULL)
{
	
	if ( ! login() ) return false;
	if ( super() ) return true;
	if(empty($GLOBALS['user']['manager'])) return false;

	if ( $category === NULL ) return true;
	
	$ar = explode(",", $GLOBALS['user']['manager']);

	return in_array($category, $ar);
}

/**
 *
 * @return boolean true if user has super grade.
 */
function super()
{
	if ( !login() ) return false;
	if ( root() ) return true;
	if ( $GLOBALS['user']['grade'] >= GRADE_SUPER ) return true;
	return false;
}
/**
 * 루트 유저이면 true 를 리턴한다.
 *
 * @since 02007/02/13 루트 유저는 등급에 상관없이 시스템 설정 파일에 아이디가 기록되어 있으면 루트 유저로 인정을 한다.
 * @return boolean true if root
 * @see admin, super
 */
function root()
{
	if ( !login() ) return false;
	global $user;
	if ( $GLOBALS['system']['admin_id'] == $user['id'] ) return true;
	else return false;
}

/**
 * 관리자 인지 체크한다.
 *
 * 관리자를 체크하는데 사용되는 기본 함수이다.
 * 입력된 모듈(카테고리)을 관리할 수 있는 권한이 있는지 체크를 한다.
 *
 * @see 관리자 권한
 * @return faultCode 권한이 있으면 ok, 없으면 fault
 * @code user 모듈을 관리할 수 있는 권한이 있으면 ok 를 리턴한다. 그렇지 않으면 사용자 정보를 수정하는 경우 인지 검사를 한다.
 *	if ( isAdmin('user') != ok ) checkUserAuthority();
 * @note checkAuthority() 함수는 사라졌다. 대신 isAdmin 이나 admin 을 사용한다.
 */
function isAdmin($category='')
{
	return deprecated();
	if ( isSuperAdmin() == ok ) return ok;
	
	if ( empty($category) )
	{
		//관리자는 모두 ok 를 리턴
	}
	return fault;
}


/*
function subAdmin() { }
*/
/**
 * 슈퍼 관리자인 경우 ok 를 리턴한다.
 *
 * @return faultCode
 */
function isSuperAdmin()
{
	return deprecated();
	if ( ! login() )
		return fault;

	global $user, $system;
	
	if ( $system['admin_id'] == $user['id'] )
		return ok;

	return fault;
}


/**
 * 슈퍼 관리자인지 체크한다.
 *
 * @todo isSuperAmdin() 함수를 제거할 것. 함수내에 alert('이 함수를 사용하지 마십시오.')와 같은 메세지를 입력할 것.
 * @return boolean 슈퍼 관리자의 권한이 있을 경우 true 가 리턴된다.
 */
function superAdmin()
{
	return deprecated();
	return !isSuperAdmin();
}
/**
 * @see superAmdin
 */
/* deprecated
function super()
{
	return superAdmin();
}
*/



/**
 * 세션관련 함수. 모든 세션 정보는 이 함수를 통해야한다.
 *
 *
 * @param string $key 키
 * @param string $value 값 이 값이 empty 이면 $key 에 해당하는 값을 리턴한다. 이 값이 참이면 세션에 저장을 한다.
 * @param int $last 초단위의 값. 웹 브라우자가 재 시작되어도 세션을 유지할 시간을 초로 기록한다.
 * @return string 입력된 $key 에 해당하는 값이 없을 경우 NULL 이 리턴된다.
 * @note 주의할 점은 세션을 지우기 위해서 session(키,' ') 를 해서는 안된다. deleteSession 를 사용해야한다.
 * @since 2007/01/20 session function now could be used to delete session. so, the default value of $value is NULL.
 * @code session("uniqid", $uniqid);
 * @code $uniqid = session("uniqid");
 * @code session("co_name", $user['name'], time() + $system['login_session_last'] * 60 * 60 * 24);
 */
function session($key, $value=NULL, $last=0, $path='/', $domain=NULL)
{
	
	if ( $value === NULL )
	{
		if ( isset($_COOKIE[$key]) )
			return $_COOKIE[$key];
		else
			return NULL;
	}
	else
	{
		if ( $domain ) setcookie($key, $value, $last, $path, $domain);
		else setcookie($key, $value, $last, $path);
	}
}
/**
 * you must set the same value as session to delete session.
 *
 */
function deleteSession($key, $path='/', $domain=NULL) {
	session($key, "", time() - 3600, $path, $domain);
}

/**
 * 회원 로그인 여부를 체크한다.
 *
 * @return faultCode
 */
function hasLogin()
{
	if ( session('id') )
		return ok;
	return fault;
}
/**
 * 회원 인증 여부를 체크한다.
 *
 * {@link hasLogin}과 틀린점은 로그인 결과의 여부를 faultCode 를 리턴하는 것과 boolean 을 리턴하는 것이다.
 * @changed faultCode 를 리턴하는 것에서 boolean 을 리턴하도록 변경했다.
 * @return boolean 로그인을 했으면 참, 아니면 거짓을 리턴한다.
 */
function login() { return !hasLogin(); }

/**
 * 로그인 페이지 URL
 *
 * 로그인 페이지로 이동할 수 있는 URL 을 리턴한다. 이때, 로그인 후 돌아올 주소를 자동으로 현제 페이지로 한다.
 * 즉, 인증에 실패하면 이 함수를 이용하여 로그인을 시킨 후, 다시 인증 시도를 할 수 있다.
 *
 * 리턴되는 URL 의 인코딩 값으로는 cate, mode 뿐만 아니라, 출력할 메세지, 그리고 돌아올 주소까지 인코딩된다.
 *
 * @param string 자바스크립트로 출력할 $messageCode 메세지. user/login.php 에서 출력된다. 없으면 생략
 * @return string 이동할 URL 주소
 * @code go(urlLoginPage('먼저 로그인을 하십시오.'));
 * @code login() or go(urlLoginPage($lang["loginfirst"]));
 * @since 2007/03/01 $messageCode 를 인코딩한다.
 */
function urlLoginPage($messageCode = '')
{
	$messageCode = urlencode($messageCode);
	$enc = urlencode("?".$_SERVER['QUERY_STRING']);
	return "?cate=user&mode=login&messageCode=$messageCode&nextpage=$enc";
}


/**
 * 로그 아웃을 시킨다.
 *
 * login 은 인증 여부를 체크한다. 이와는 달리 로그 아웃 상태를 만든다.
 */
function logout() { deleteSession('id'); }

/** crypt function */
function check_crypt_std_des($password, $en_password)
{
	@define('CRYPT_STD_DES', 1);
	$enc = crypt($password,substr($en_password, 0, 2));
	return $enc == $en_password;
}
/**
 * 데이터 컨버젼이 일어났을 경우, 비밀번호 유지를 하기 위해서 기존 암호화된 종류를 저장해 놓고 비교를 한다.
 *
 * @param string $input_password user password which was just submitted thru the HTTP
 * @param string $old_password user password in database.
 * @param string $old_password_type the enryption type of $old_password
 *
 * @return boolean true if password matches. otherwise false.
 * @todo 범용적이지 못하다. MYSQL_OLD_PASSWORD, MYSQL_PASSWORD, MD5, CRYPT_DESC 등으로 표기해서 사용할 수 있도록 한다.
 */
function checkPassword($input_password, $old_password, $old_password_type)
{
	if ( empty($old_password_type) ) return md5($input_password) == $old_password;


	global $db;
	switch( $old_password_type )
	{
		case 'rgboard'			:
			$re = $db->result("SELECT OLD_PASSWORD('$input_password')");
			return $old_password == $re;
		case 'zeroboard4'		:
			$re = $db->result("SELECT password('$input_password')");
			return $old_password == $re;
		case 'gnuboard4'		:
			$re = $db->result("SELECT password('$input_password')");
			return $old_password == $re;
		case 'technote'			:
			return check_crypt_std_des($input_password, $old_password);
		default : return false;
	}
}




/**
 * 등급을 체크한다.
 *
 * 비교할 등급이 0 이면, 즉 public 이면, 비교할 것도 없이 그냥 true 를 리턴한다.
 *
 * 입력 변수 $my 가 $grade 보다 크거나 같으면 true 를 리턴한다. 아니면 false 리턴
 * @param int $grade 등급 GRADE_ 상수 참고
 * @param int $my 비교할 등급. 생략되면 자신의 것이 사용된다.
 * @return boolean 등급이 올바르면 ($my 가 $grade 보다 크거나 같으면) true, 아니면 false
 */
function grade($grade, $my=NULL)
{
	if ( super() ) return true;
	if ( $grade == GRADE_PUBLIC ) return true;

	
	if ( $my )
	{
		if ( $my >= $grade ) return true;
		else return false;
	}
	if ( login() )
	{
		global $user;
		if ( empty($user) ) return false;
		
		$my = $user['grade'];
		if ( $my >= $grade ) return true;
	}
	return false;
}


/**
 * 한글 이름에서 성을 제외한 이름을 리턴한다.
 *
 * 한글 이름 중 성을 제외한 이름, 6 바이트 이하이면 첫번째 2바이트 제외. 7 바이트 이상이면 첫번째 4바이트 제외
 */
function firstName($name)
{
	if ( strlen($name) <= 6 ) return substr($name, 2);
	else return substr($name, 4);
}
/**
 *
 * @return string user nicke if the user has one. or user name.
 */
function nickname() {
	if ( empty($GLOBALS['user']['nick']) ) return $GLOBALS['user']['name'];
	return $GLOBALS['user']['nick'];
}

/**
 * see if the user's name is okay. if not, it returns the default name in language pack.
 */
function name($name)
{
	$name = trim($name);
	if ( empty($name) ) return $GLOBALS['lang']['emptyName'];
	return $name;
}



/**
 * 사용자 아이디가 올바른지 체크를 한다.
 *
 * 아이디(회원 아이디, 게시판 아이디 등)는 2 글자 이상 ~ 255 글자이하 이어야한다.
 * 아이디의 문자 조합은 키보드를 통해서 직접 표현할 수 있는 모든 것이 가능하다.
 * 한글, 일본어, 중국어 등 입력가능한 모든 문자가 가능하며 첫글자가 숫자, 기호라도 상관이 없다.
 * 모든 아이디는 변경이 가능하다는 것이 원칙이다.
 * 프로그램 내부적으로는 모두 데이터 레코드 번호 idx 를 사용한다. 이 값이 해당 정보에 대한 불변의 고유한 값이다. 아이디는 고유한 값이지만,  변하는 값이다.
 * 아이디를 생성하는 부분에서 반드시 이 함수를 호출해야한다.
 * 각(사용자 아이디 포함) 아이디 변경에 대해서는 관련 함수를 참고한다.
 *
 * 아이디 조합의 규칙:
 * - 255 글자 이하이어야한다.
 * - 숫자만으로 구성될 수 없다. (숫자로 시작할 수 있다.)
 * - 화면에 보이지 않는 white-space (공백, 탭키, 라인 피드 등) 문자는 아이디 조합에 사용할 수 없다.
 * - 특수문자 < > . , - + | \ ) 등과 ESC, 백스페이스, 엔터, CTRL+Z 키 등은 사용이 불가능하다.
 *
 *
 * @param string $id 아이디
 * @param int $level 아이디 조합에서 가능한 문자셋의 표현 범위이다.
 *	1, 2, 3 단계까지 가능하다.
 *	1 이면 2,3 단계 전부 다 된다.
 *	2 이면 특수 기호는 안된다. 따옴표, 슬래쉬, 물음표, < > & ; 기타 등 등 안됨.
 *		단, 소숫점, 언더바 두개는 제외
 *	3 이면 첫 글자가 영문자이어야 하며, 영숫자만 된다.
 *
 * <code>
 * $id = "A1htmla한글안bcdefg1234679090909";
 * $id = "0Aa2df1324";
 * $id = "A1htmlabcdefg1234679090909";
 * !validID($id) and javascriptAlert("wrong id: $id");
 * !validID($id,2) and javascriptAlert("wrong id on level 2: $id");
 * !validID($id,3) and javascriptAlert("wrong id on level 3: $id");
 * </code>
 */
function validID($id, $level=2)
{
	if ( strlen($id) > 255 ) return false;
	if ( is_numeric($id) ) return false;
	if ( preg_match("/\s/", $id) ) return false;
	
	// 특수 문자 제거
	// @note 다른 문자셋에서의 처리
	if ( $level >= 2 )
	{
		// . _ 두개를 제외 했다.
		$etc = "`~!@#$%^&*()-=+\|[]{};':\",/<>?";
		for( $i = 0; $i < strlen($etc); $i++ )
		{
			$ar[] = $etc[$i];
		}
		$str = str_replace($ar, "", $id);
		if ( $str != $id ) return false;
	}
	
	if ( $level >= 3 )
	{
		if ( ! preg_match("/^[a-zA-Z][a-zA-Z0-9]+$/", $id, $m) )
			return false;
	}
	
	return true;
}

































///////////////////////////////////////////////////////////////////////////////
//
// System functions (시스템 관련 함수)
//
// 만약, 디렉토리 분리자 / 와 \ 를 구분하기 위해서라면 pathinfo, dirname 과 같은 함수를 사용해야한다.
//
///////////////////////////////////////////////////////////////////////////////
/**
 * @return boolean true if server os is window. otherwise false.
 */
function isWindows()
{
	if ( !isset($_ENV['OS']) ) return false;
	$os = strtolower($_ENV['OS']);
	if ( strstr($os, "window") ) return true;
	return false;
}

function systeminfo()
{
	global $software;
	return "$software[name] version $software[version], site: $software[homepage]";
}











/**@#+
 * 스킨 관련 함수
 */
/**
 * 스킨을 포함하지 않도록 한다.
 *
 * 즉, layout.html 을 실행하지 않도록 하는 것이다.
 */
function noskin()
{
	return array('skin'=>'noskin');
}
/**
 * 메인 스킨만 출력.
 *
 * 관리자 화면에서 헤더, 푸터 와 같은 파일 필요없이 메인 스킨만 사용할 때 적당하다.
 * 이 함수를 사용하면 layout.html 을 포함하지 않고, 스킨 중 {cate}.{mode}.html 만 포함을 한다.
 * @code return mainskin();
 */
function mainskin()
{
	return array('skin'=>'mainskin');
}
/**
 * 페이지를 이동시킨다.
 *
 * 이 함수는 go 함수와 비슷한 역활을 한다.
 * go 함수는 스크립트를 종료시키지만, 이 함수는 끝까지 수행을 시킨다.
 * @param string $url 이동할 위치
 * @param string $message 자바스크립트로 화면에 출력할 메세지
 */
function move($url, $message=NULL)
{
	if ( $message ) javascriptAlert($message);
	if ( $url ) javascriptGo($url);
	return noskin();
}


/**
 * @since version 2.2 스킨의 로딩 방식이 변경되었다. 이에 따라 이 함수는 사용되지 않는다.
 * @todo version 3.0 부터는 이 함수를 삭제한다. 더 이상 지원을 하지 않는다.
 */
function checkLayout()
{
	return array(1,1,1,1,1,1,1,1);
} // eo




/**@#-*/






// ========================== 에러 & 경고 메세지와 이동 관련 함수 =============


/**
 * 에러 처리 루틴
 *
 * 모든 함수는 에러가 있을 경우 faultCode 를 리턴한다. 이 함수를 통해서 전역변수 $faultExCode와 $faultExString에 에러 문장을 추가하고 에러 코드 값을 리턴하도록한다. 이렇게 해서 차후에 다른 함수들이 이 변수들을 이용할 수 있도록 한다.
 *
 * @param int $code faultCode
 * @param string $string 기본 에러 코드 상수에 대한 에러 문장에 정보를 추가할 내용이다.
 * @return faultCode 입력받은 $code 를 그냥 리턴한다.
 */
function faultEx($code, $string)
{
	global $faultExCode, $faultExString;
	
	$faultExCode = $code;
	if ( !empty($faultExString) ) $faultExString .= "\n\t";
	$faultExString .= $string;
	return $code;
}






function getLastError() {	return array($GLOBALS['faultExCode'], $GLOBALS['faultExString']); }
// 마지막 에러 내용의 문장을 리턴한다.
// 에러 내용은 $faultExCode, $faultExString, $faultString 변수에 기반을 한다.
function getLastErrorMessage()
{
	global $faultString, $software;
	list($code,$string) = getLastError();
	if ( !empty($code) && !empty($string) )
	{
		$msg = "code\t: $code\nstring\t: ".$faultString[$code]."\nextra\t: $string";
		return $msg;
	}
}


/**
 * 에러 내용을 출력하고, 이전 페이지로 이동하는 자바스크립트 문장을 출력한 뒤, 함수 내에서 종료한다.
 *
 *
 * @note 동작 방식
 *	메세지만 입력하면 기존 에러 메세지에다가 해당 메세지를 출력하고 이전 페이지로 이동.
 *	메세지와 에러 코드를 입력하면 faultEx 에 값을 넣고 에러 메세지를 출력하고 이전 페이지로 이동
 *	메세지와 에러코드 모두 empty 이면 faultEx 에 의해서 기록된 에러 내용을 출력하고 이전 페이지로 이동
 *
 * @param string $msg 에러 내용. 생략되면 마지막 에러 코드와 에러 문장을 사용한다.
 * @param string $ex 이 변수에 값이 들어면 $msg 는 에러 코드, $ex 는 에러 문장이된다. 이 것은 faultEx() 함수 호출 뒤 goBack() 함수를 호출하는 효과가 있다.
 * @note 입력 변수 $ex 형식
 *	위치(라인, 또는 함수) 에러메세지
 *	단, __FILE__ 이나 __LINE__ 메세지를 직접 쓰지 않는다.
 *
 * @note goBack, go 의 문자열 입력 변수에는 따옴표, 쌍따옵표를 이용할 수 없다.
 * <code>
 *	lib('category');
 *	$cidx = getCategoryIndex(category_id) or goBack(wrongIndex, "카테고리 아이디가 잘못 지정되어 있습니다.");
 * </code>
 * @code hasLogin() and goBack(accessDenied, "메모장을 이용하기 위해서는 회원 로그인을 해야합니다.");
 * @code $limit = getPosts(category_idx); if ( $limit == ok ) goBack("memo::index->getPosts at " . __FILE__ . " line: " . __LINE__);
 */
function goBack($msg='', $ex='')
{
	// 값이 둘다 들어온 경우
	if ( !empty($ex) )
	{
		faultEx($msg, $ex);
		$msg = getLastErrorMessage();
	}
	// $msg 만 들어온 경우,
	else if ( !empty($msg) )
	{
		$msg = getLastErrorMessage() . "\n" . $msg;
	}
	// 둘자 empty 인 경우
	else if ( empty($msg) && empty($ex) )
	{
		$msg = getLastErrorMessage();
	}
	javascriptAlert($msg);
	javascriptGo(-1);
	exit;
}
/**
 * 자바 스크립트 알림
 *
 * faultCode 방식을 따른다.
 *
 * {@link goBack}의 소스를 복사해서 작성하였으며,
 * {@link goBack}와 차이점은 종료를 하지 않고, 다른 페이지로 이동을 하지 않는다는 것이다.
 *
 * @param same as {@link goBack}
 * @return javascript 화면으로 직접 출력을 한다.
 * @code alert(faultEx(accessDenied, $lang['accessDenied']));
 */
function alert($msg='', $ex='')
{
	// 값이 둘다 들어온 경우
	if ( !empty($ex) )
	{
		faultEx($msg, $ex);
		$msg = getLastErrorMessage();
	}
	// $msg 만 들어온 경우,
	else if ( !empty($msg) )
	{
		$msg = getLastErrorMessage() . "\n" . $msg;
	}
	// 둘자 empty 인 경우
	else if ( empty($msg) && empty($ex) )
	{
		$msg = getLastErrorMessage();
	}
	javascriptAlert($msg);
}

/**
 * 입력된 주소로 웹브라우저를 이동시킨다.
 *
 * 함수내에서 종료한다.
 *
 * @note goBack 은 입력값이 없을 경우, 기본적으로 마지막 에러 값을 사용하며 무조건 이전 페이지로만 이동을 한다.
 * go 는 입력값을 바탕으로 지정된 페이지로 이동을 한다.
 *
 * @changed 2006/12/5 $fcode 가 empty 일 경우, $fstring 을 출력한다.
 * @note 예제 go('?','메인 페이지로 이동합니다.');
 */
function GO($url, $fstring='', $fcode='')
{
	
	if ( ! empty($fcode) )
	{
		faultEx($fcode, $fstring);
		$fstring = getLastErrorMessage();
		javascriptAlert( $fstring );
	}
	else if ( !empty($fstring) )
	{
		javascriptAlert( $fstring );
	}
	javascriptGo($url);
	exit;
}








// ========================== 디버그 & 문자열 출력 관련 (메세지) 함수 ================
//
// debug.php 와 string.php 라이브러리가 따로 준비되어있다. 그럼에도 불구하고 여기서
// 관련 함수를 기록하는 이유는 활용의 빈도에 차이가 있다.
// 자주 쓰는 함수는 그냥 기본 라이브러리로 로드되게 하기 위해서 작성한다.
//
/**
 * 디버깅 모드이면 참을 리턴
 *
 * @returns string if the debug mode on, then it returns not empty. otherwise empty is returned.
 */
function debug()
{
	if ( !isset($GLOBALS['system']['debug']) ) return NULL;
	return $GLOBALS['system']['debug'];
}


/**
 * This function acts just like the one in STL.
 * assertion is internal function. so,
 * (PHP 4 >= 4.3.0, PHP 5)
 * @todo backtrace 는 4.3.0 이상의 버젼이 요구된다. 당장 필요 없는 기능이므로,
 * @todo 1.4 p8 부터 PHP 4.3 이상이 사용된다. 따라서 backtrace 과련 함수를 사용하면된다.
 * assert 함수와 옵션 처리로 한다.
 */
function _assert($b)
{
	return deprecated();
	if ( $b ) return;
	echo "assertion failed<hr>";
	var_dump(debug_backtrace());
	exit;
}

/**
 * 조용한 모드
 *
 * 에러 리포팅(메세지) 출력을 최대한 자제시킨다.
 * @see 개발자노트
 * @code 처음 설치를 할 때, DB 접속 관련 에러를 숨기기 위해서 module/install/submit.php 에서 quiet() 를 호출한다.
 */
function quiet() { $GLOBALS["QUIET"] = true; }
/**
 *
 * @return boolean 조용한 상태이면 true, 아니면 false 를 리턴
 */
function isQuiet()
{
	if ( isset($GLOBALS['QUIET']) ) return $GLOBALS['QUIET'];
	return false;
}




/**
 * 변수의 값을 화면으로 출력하는 디버깅 함수
 *
 * @note This function does not need in release. well, if you do not call this function, lib('debug') will not be commited.
 */
function g($var)
{
	lib('debug');
	dbgMix($var);
}


/**
 * 디버깅 정보를 화면에 출력한다.
 *
 * 디버깅 모드가 아니면, 간단한 정보를 소스내에 주석으로 출력한다.
 */
$db_queries = array();
$db_count_queries = 0;
function debugPrint($time=NULL)
{
	global $system, $ui;
	if ($system['debug'])
	{
		$skinfile = skin();
		echo br;
		echo "<br>스킨 파일: $skinfile";
		if ( strstr($skinfile, "/$system[skin]/") === FALSE )
			echo " [<b>선택된 스킨: <font color=darkred>$system[skin]</font></b>] 스킨 파일이 존재하지 않아, 기본 스킨 파일을 사용합니다.";
			
		echo br;
		echo "소스 파일: ".module().br;
		echo "cate=$ui[cate] , mode=$ui[mode]".br;
		echo "run-time: $time".br;
		echo "[*] 본 메세지는 시스템 설정에서 디버그 옵션이 선택되었기 때문에 나타납니다.";
		echo br;
	}
	
	echo "\r\n<!-- run-time: $time\r\n";
	echo $GLOBALS['db_count_queries'] . " quries\r\n";
		
	if ( isset($GLOBALS['db_queries']) )
	{
		$qs = $GLOBALS['db_queries'];
		foreach($qs as $q)
		{
			if ( !$q ) continue;
			echo "$q\r\n";
		}
	}
	echo "//-->";
		
}

// 자바 스크립트의 alert 에서 출력이 가능한 메세지를 만든다.
// 라인피드, 쌍따옴표 등을 ESCAPE 한다.
function messageJavascript($msg)
{

	$msg = str_replace("\\", "\\\\", $msg);
	$msg = str_replace("\n", "\\n", $msg);
	$msg = str_replace("\r", "\\r", $msg);
	$msg = str_replace("\"", "'", $msg);

	return $msg;
}

function messageSystem($message)
{
	global $software, $system;
	if ( isset($system['name']) && $system['name'] ) $title = $system['name'];
	else $title = $software['title'];
	$msg = "[ $title ]\n\n";
	$msg .= $message;
	$msg .= "\n\n$software[copyright]";
	
	return messageJavascript($msg);
}


/**
 * 현제 윈도우를 닫는 자바스크립트 코드를 출력한다.
 *
 * 직접 자바스크립트 코드를 출력한다.
 * <code>
 *	alert($lang['writeOkay']);
 *	javascript_selfclose();
 * </code>
 */
function javascript_selfclose()
{
	echo "<script>self.close();</script>";
}

/**
 * 자바 스크립트의 alert 를 구현
 *
 * @todo alias to javascript_alert
 */
function javascriptAlert($msg)
{
	$msg = messageSystem($msg);
	/**
	 * @todo 문자셋 표현이 여기에 오면안된다. 따로 함수화해야한다. 문자셋이 옵션일 경우 여기서는 처리가 곤란하다.
	 */
	echo "<meta http-equiv=Content-Type content='text/html;charset=utf-8'>";
	echo "<script>alert(\"$msg\")</script>";
}
function javascriptGo($url)
{
	if ( $url == - 1 )
	{
		echo "<script>history.go(-1);</script>";
	}
	else
	{	
		echo "<script>location.href=\"$url\";</script>";
	}
}



/**
 * 입력된 문자열을 강조해서 리턴한다.
 *
 * <font> ... </font> 구문으로 감싸서 출력을 한다.
 * 입력된 값이 empty 이면 그냥 empty 를 리턴한다.
 * @param mixed $message 강조할 문자열 또는 숫자 등등
 * @param string $pre 문자열 앞에 추가할 구문
 * @param int $type 표시 형태
 * @param string $style 스타일 시트 문자열
 * @return string
 * <code>
 *	<?=em($post[children])?>
 *	<?=em($post[files],'file:',2)?>
 *	<?=em($post[children],"답변")?>
 * </code>
 */
function em($message, $pre='', $type=1, $style='color:darkred; font-size:8pt;')
{
	if ( empty($message) ) return null;
	if ( $pre ) $pre = "$pre:";
	if ( $type == 1 ) $message = "($pre$message)";
	else if ( $type == 2 ) $message = "[$pre$message]";
	else if ( $type == 3 ) $message = "&lt;$pre$message>";
	else if ( $type == 4 ) $message = "\{$pre$message}";
	$newstring = "<font style='$style'>$message</font>";
	
	return $newstring;
}



/**
 * 입력값을 HTML FORM 의 text, textarea 등에서 사용할 수 있도록 처리를 한다.
 *
 * @param string $text 문자열
 * @return string escape 된 문자열
 */
function form_value($text)
{
	$text = stripslashes($text);
	$text = htmlspecialchars($text);
	return $text;
}

	




	























// ========================== 숫자 관련 함수
// 숫자 관련 함수의 량이 많을 경우 이를 따로 묶는다.
//
//


/**
 * 입력값이 양수인지 체크한다.
 *
 * @return boolean 양수의 값이 입력되었으면 true 를 리턴한다.
 * @code if ( ! positive($post['category_idx']) ) return faultEx(wrongParam, '글을 저장할 카테고리(정보 영역)이 입력되지 않았습니다.');
 * <code>
 *	if ( positive($ui['idx']) )
 *	{
 *		$post = getPost($ui['idx']);
 *		foreach( $post as $k=>$v ) { $$k = $v; }
 *	}
 * </code>
 */
function positive($i) {
	if ( empty($i) ) return false;
	if ( !is_numeric($i) ) return false;
	if ( $i <= 0 ) return false;
	return true;
}

/**
 * 파일의 용량을 KB/MB 단위로 표현
 *
 * @since 2006/12/18 이 함수는 filesystem.php 에서 default.php 로 옮겨짐
 */
function kbsize($n)
{
	$k = (int) ($n / 1000) + 1;
	if ( $k < 1000 ) return $k . "Kb";
	else
	{
		$m = round($k / 1000, 1);
		return $m . "Mb";
	}
}






// ========================== 배열 관련 함수
function array_remove($arr,$value) { return array_values(array_diff($arr,array($value))); }





// ========================== 네트워킹, 인터넷 주소, URL, HTTP 관련 함수


/**
 * c 언어의 inet_adrr 를 구현한다.
 *
 * PHP version 5.0 이상 부터는 inet_ntop, inet_pton 함수가 준비되어있다.
 *
 * @param string $ip 아이피 주소를 나타내는 문자열
 * @return int 4 바이트 unsigned int 형의 값을 리턴한다.
 * @note inet_ntoa 에 해당하는 함수는 준비하지 않는다. 각자 알아서 해결을 한다.
 * 예를 들면, MySQL 에서는 inet_ntoa 처리를 SELECT INET_NTOA( 2078941741 ) 와 같이 해결 할 수 있다.
 */
function inet_addr( $ip )
{
	$b = explode('.',$ip);
	if (count($b) == 4)
	{
		$n = ($b[0]*256*256*256) + ($b[1]*256*256) + ($b[2]*256) + ($b[3]*1);
		return $n;
	}
	else
		return 0;
}


/**
 * 헤더의 키/값을 전역변수에 보관하고 전송한다.
 *
 * headers_list() 가 PHP 버젼 5 부터 사용이 가능하다. 어떤 헤더를 전송했는지 체크가 필요한데, PHP < 5.0 의 내장 함수로서는 해결이 안된다.
 * 따라서, 본 함수를 이용한다.
 */
$__HEADERS = array();
function sendHeader($key, $value)
{
	global $__HEADERS;
	$__HEADERS[$key] = $value;
	header("$key".": $value");
}
/**
 * 해더 전송. 반드시 이 함수를 이용할 것.
 *
 * 중요: 모든 header() 대신에 headers() 를 사용해야한다.
 * @param string $string 헤더 문자열. header() 와 동일
 * @return mixed $string 이 empty 일 경우, 출력된 헤더들을 연관 배열로 리턴한다.
 * @code headers("Content-Transfer-Encoding: binary");
 */
function headers($string='')
{
	if ( empty($string) ) return $GLOBALS['__HEADERS'];
	list ($key, $value) = explode(":", $string);
	sendHeader($key, $value);
}








// ========================== 파일 관련
//
// filesystem.php 파일에 보관되어야하는 함수들이다. 그러나 사용빈도는 높은데, filesystem.php 라이브러리를 로드하기가 부담이 된다.
// 테스트 후에 결저을 한다.
// 
//
/**
 * @deprecated use log class
 * 로그 파일 작성
 *
 * 함수명이 __ 로 시작한다. 기존의 함수명과 충돌한다.
 * @param string $file 파일 경로
 * @param string $message 메세지
 * @return boolean 작성 성공이면 true 아니면 false
 */
function __log($file, $message)
{
	$filename = log_repository . "/$file";
	$dt = date("Y/m/d H:i:s");
	$message = "$message \"$dt\"\n";
	
	if ( ! @attachString($filename, $message) )
	{
		if ( is_dir( $file ) ) return false;
		mkdirEx(log_repository);
		return attachString($filename, $message);
	}
	return true;
}


/**
 * 시스템 로그를 한다.
 *
 * 기본적인 시스템 로그를 기록하는 함수이다. 자주 사용이된다면, 클래스 자체를 default.php 로 끌어들일 수 있다.
 * @see log.php
 * @param string $file 로그 파일 이름 (파일 명칭만 있으면 된다. 자동으로 로그 디렉토리에 기록을 한다.)
 * @param string $message log 클래스로 전달하는 값. 자세한 내용은 log 클래스 참조
 * @return boolean 성공이면, true. 아니면 false
 */
/*
function systemlog($file, $message = '') {
	global $ui, $user;
	if ( $ui['cate'] == 'download' ) return true;
	
	lib('log');
	$info['id'] = $user['id'];
	$info['name'] = $user['name'];
	$info['message'] = $message;
	return log::access(log_repository.'/'.$file, $info);
}
*/
/**
 * 파일 기록
 *
 */
/*
function accesslog()
{
	return systemlog('access');
}
*/
/*
function installlog($modulename, $version, $type)
{
	$message = "$modulename $version ".time()." $type\n";
  return systemlog('install', $message);
}
*/

/**
 * 데이터 저장소가 있는지 체크
 *
 * @since version 0.8 data 디렉토리는 기본적으로 같이 배포가된다. 따라서 이 함수는 사용되지 않는다.
 * @return boolean is_dir 과 동일
 */
function checkRepository() { return is_dir(repository); }
/**
 * 데이터 저장소를 생성
 *
 * @return boolean mkdir, mkdirEx 와 동일 (성공이면 true)
 */
function makeRepository() { return mkdirEx(repository); }
/**
 * 데이터 저장소(디렉토리)가 없으면 생성을 한다.
 * @return boolean 저장소를 생성한 경우 true, 아니면 false. 이미 만들어져있거나 만들 수 없는 경우 등에서 false 를 리턴한다.
 * <code>
 * 설치 과정(또는 시스템 설정 변경 과정)에서 디렉토리 준비를 미리 할 것. 매번 파일 과련 작업을 할 때 당연히 에러 체크를 해야한다. 하지만 좀 더 간단한 방법을 찾을 것.
 * faultEx(diskAccess, "bbs::addfile_submit 업로드된 파일 ($name) 복사에 실패하였습니다.");
 * if ( checkmakeRepository() )
 * {
 *		alert();
 *		faultEx(diskAccess, "(파일) 데이터 저장소가 준비되어있지 않아서 생성을 했습니다. 파일 업로드를 다시 시도하십시오.");
 * }
 * alert();
 * </code>
 */
function checkmakeRepository()
{
	if ( checkRepository() ) return false;
	return makeRepository();
}

/**
 *
 */
function checkmakeModuleRepository()
{
	if ( is_dir(module_repository) ) return false;
	return mkdirEx(module_repository);
}

/**
 * 임시 파일 경로를 리턴한다.
 *
 * @code $file = sessionfile("mall");
 * @code <?=sessionfile('mall');?>
 */
function sessionfile($ext=NULL)
{
	$filename = tmp_repository . "/session/" . $GLOBALS['sessionid'];
	if ( $ext ) $filename = $filename . ".$ext";
	return $filename;
}


/**#@+
 * 설정 파일을 액세스한다.
 *
 * 모듈의 설정에 대해서 설정 정보를 액세스한다. 모듈 관련 스크립트 내에서만 사용이 가능하다.
 * 일반 스크립트에서 모듈 설정을 읽어야할 때에는 그냥 readAs, writAs 를 통해서 읽는다.
 * 이 함수를 이용하면, 모듈 정보 저장 디렉토리가 존재하지 않을 경우 자동으로 생성을 해 준다.
 *
 * @doc 이 함수의 활용도를 높여야한다.
 * @return same as readAs, writeAs
 */
function readConf()
{
	$file = module_repository."/$GLOBALS[cate].conf";
	$as = @readAs($file);
	if ( is_bool($as) ) checkmakeModuleRepository();
	return $as;
}
function writeConf($as)
{
	$file = module_repository."/$GLOBALS[cate].conf";
	return writeAs($file, $as);
}
/**#@-*/



















///////////////////////////////////////////////////////////////////////////////
//
//                            [ 추가 스크립트 ]
//
///////////////////////////////////////////////////////////////////////////////
//
// 전역 스크립트
/**
 * 필요한 전처리를 한다.
 *
 */
$__arScripts = readParse(file_scripts, "\n", " ", 1);
$ar = &$__arScripts;
if ( ! isset($ar['begin']) ) $ar['begin'] = array();
if ( ! isset($ar['end']) ) $ar['end'] = array();
$__arBeginScripts = array_unique($ar['begin']);
$__arEndScripts = array_unique($ar['end']);
/**#@+
 * 추가 스크립트
 * @todo include 시에 에러가 나면 (인클루드 파일이 존재하지 않거나 등등) scripts 기록 파일을 올바르게 수정한다.
 */
/**
 * 시작 스크립트 실행
 */
function beginScript()
{
	global $__arBeginScripts;
	foreach($__arBeginScripts as $file)
	{
		if ( debug() )
			include(dirRoot.'/'.$file);
		else
			@include(dirRoot.'/'.$file);
	}
}
/**
 * 종료 스크립트 실행
 */
function endScript()
{
	global $__arEndScripts;
	foreach($__arEndScripts as $file)
	{
		if ( debug() )
			include(dirRoot.'/'.$file);
		else
			@include(dirRoot.'/'.$file);
	}
}
/**
 * BUILDGUIDE 참고
 *
 * @todo 플러그인 기능을 사용하면 더 이상 이 함수의 사용이 필요가 없다.
 *	따라서 6월 이후 부터는 이 함수와 관련된 모든 메세지, 문서 등을 삭제한다.
 */
function display($func, $arg=array())
{
	$name = "display_$func";
	if ( ! function_exists($name) ) return NULL;
	return $name($arg);
}
/**#@-*/

?>
