<?php
/**
 * 파일(디렉토리) 관련 라이브러리
 *
 * 파일 라이브러리는 두가지로 나뉜다.
 * 첫째, 시스템 독립적인 파일 함수. 둘째 시스템 파일 함수.
 * 예를 들면, mkdirEx 의 경우 언제든지 아무렇게나 사용이 될 수 있다.
 * 그러나 insertFile 의 경우, 시스템의 규격에 맞추어서 호출이되어야한다.
 * 시스템 파일 함수는 filesystem.php 따로 라이브러리르 만들었다.
 *
 * @package library
 */

/**
 * 설정 파일을 읽어서 리턴한다.
 *
 * @return mixed 파일에 이상이 있을 경우 false 가 리턴된다.
 *	이상이 없으면 리턴되는 변수의 값은 파일의 데이터이다.
 *	즉, false 를 리턴하는 경우에는 설정 파일이 잘못되었음을 의미하며 설정 파일에 false 값만 기록을 해서는 안된다.
 * <code>
 *	config("cfg.test.php", $plugins);
 *	$var = config("cfg.test.php");
 *	g($var);
 * </code>
 */
function config($file, $var=NULL)
{
	if ( $var === NULL )
	{
		$re = readfileEx($file);
		if ( $re === false ) return false;
		return unserialize(base64_decode($re));
	}
	else
	{
		return writeStringEx($file, base64_encode(serialize($var)));
	}
}

/**
 * 파일의 내용물이 숫자라고 가정을 하고, 입력된 값을 더한다.
 *
 */
function sumFile($file, $num)
{
	$str = readfileEx($file);
	if ( is_bool($str) && $str === FALSE ) ;
	$res = $str + $num;

	return writeString($file, $res);
}
/**
 * 파일의 내용물이 숫자라고 가정을 하고, 값을 1씩 증가시킨다.
 */
function increasefile($file) { return sumFile($file, 1); }


/**
 * 디렉토리 생성
 *
 * PHP version 5.0 의 mkdir 함수의 recursive 를 구현한다.
 * @param string $dir 디렉토리 경로. mkdir 와 동일하다.
 * @param int $mode 생성될 디렉토리의 퍼미션 mkdir 과 동일하다.
 * @param boolean $recursive true 이면 중간에 포함된 디렉토리를 생성한다. 
 * @return boolean 성공이면 true, 아니면 false
 *
 * <code>
 * 	$rc = writeString($file, $data);
 * 	if ( $rc ) return $rc;
 * 	$rc = mkdirEx(tmp_repository."/cache");
 * 	if ( ! $rc ) return $rc;
 * 	return writeString($file, $data);
 * </code>
 */
function mkdirEx($dir, $mode=0777, $recursive=true)
{
	if ( $recursive )
	{
		$ar = array();
		$dirs = explode('/', $dir);
		foreach( $dirs as $p )
		{
			$ar[] = $p;
			$path = implode('/', $ar);
			if ( ! is_dir($path) )
			{
				if ( ! mkdir($path, $mode) ) return false;
			}
		}
	}
	else
	{
		if ( ! mkdir($dir, $mode) ) return false;
	}
	return true;
}


/**
 * 파일에 기록된 연관 배열 내용을 읽어들인다.
 *
 * 한 라인이 '키=값' 으로 구성된 설정 파일에서 키/값을 읽어서 연관 배열로 리턴한다.
 *
 * 환경 설정 파일의 예는 다음과 같다.
 * <code>
 * a = b ...
 * c = this is configuration file ...
 * </code>
 * @param string $filename 환경 설정 파일
 * <code>
 * lib('file');
 * $readme = readAs($dirSkin."/README", 20);
 * </code>
 * @return mixed 에러시 boolean 값 FALSE 리턴. 에러가 아닐 시, 배열을 리턴.
 */
function readAs($filename, $limit=0)
{
	$content = @readfileEx($filename);
	if ( is_bool($content) && $content === FALSE ) return FALSE;
	$ar = explode("\n", $content);
	$cnt = 0;
	foreach( $ar as $line )
	{
		$line = trim($line);
		if ( empty($line) ) continue;
		@list ($k, $v) = @explode('=', $line, 2);
		$k = trim($k);
		$v = trim($v);
		$as[$k] = $v;
		if ( $limit )
		{
			$cnt ++;
			if ( $cnt > $limit )
				break;
		}
	}
	return $as;
}
/**
 * 연관 배열을 파일에 저장
 */
function writeAs($filename, $as)
{
	$str = null;
	foreach( $as as $k => $v )
	{
		$str .= "$k = $v\n";
	}
	return writeString($filename, $str);
}


/**
 * 파일을 읽어서 파싱한 후, 배열로 리턴한다.
 *
 * 이 함수는 readAs 의 확장판이다. readAs 는 연관배열로만 리턴하지만, 이 함수는 연관 배열 뿐만아니라 2차원 배열로도 리턴을 한다.
 * 키/값에 대해서 trim 을 한다.
 * @param string $filename 파일 경로
 * @param string $sep1 구분자. 큰 단락
 * @param string $sep2 구분자. 키/값
 * @param int $arType 1 일 경우, 중복되는 값이 있을 것이라 예상을 미리하고, 2차원 배열로 리턴한다.
 * @return mixed false on error. otherwise array.
 *
 * <code>
 *	lib('file');
 *	$dirs = getDirs(dirRoot."/skin");
 *	foreach ($dirs as $dir)
 *	{
 *		$file = dirRoot."/skin/$dir/README";
 *		if ( file_exists( $file ) )
 *		{
 *			$ar = readParse($file);
 *			$skins[$dir] = "$ar[title] ($dir) by $ar[author] v$ar[version]";
 *		}
 *	}
 * </code>
 * <code>
 * $file = "module/$GLOBALS[cate]/$mode.php";
 * $ar = readParse(file_scripts, "\n", " ", 1);
 * g($ar['begin']);
 * g($ar['end']);
 * </code>
 * 동일한 키에 중복된 값이 있을 경우, 아래와 같이 array_unique 를 통해서 값의 중복을 없앨 수 있다.
 * <code>
 * $ar = readParse(file_scripts, "\n", " ", 1);
 * g(array_unique($ar['begin']));
 * </code>
 */
function readParse($filename, $sep1="\n", $sep2="=", $arType=0)
{
	$content = @readfileEx($filename);
	if ( is_bool($content) && $content === FALSE ) return FALSE;
	$ar = explode($sep1, $content);

	foreach( $ar as $line )
	{
		if ( empty($line) ) continue;
		@list ($k, $v) = @explode($sep2, $line, 2);
		$k = trim($k);
		$v = trim($v);
		if ( $arType ) $as[$k][] = $v;
		else $as[$k] = $v;
	}
	return $as;
}


/**
 * 연관 배열 (associative-array 형) 변수를 입력받아서 PHP 코드 형식으로 파일에 저자을한다.
 *
 * @param string $filename 파일 경로
 * @param associative-array $as 변수/값의 쌍을 가지는 연관 배열
 * @return boolean {@link writeString}과 동일 한 값을 리턴한다.
 * @example ../module/system/modify_submit.php
 */
function writeAsInPHPCode($filename, $as)
{
	$code = "<?php\r\n";
	foreach( $as as $k => $v )
	{
		$code .= '$' . "system[$k]='$v';".rn;
	}
	$code .= "?>";
	
	lib('file');
	return @writeString($filename, $code);
}



/**
 * 파일에 문자열을 기록한다.
 *
 * 기본적으로 writeString 과 동일하다.
 * 내용을 저장할 (생성될) 파일 경로가 존재하지 않는다면, 즉, 디렉토리가 존재하지 않는다면, 생성을 한다.
 *
 * 생성에 실패를 하면 false 를 리턴한다.
 *
 * @param string $file same as writeString
 * @param string $data same as writeString
 * @return boolean same as writeString
 * @code return writeStringEx($GLOBALS['skin_cache_file'], $data);
 */
function writeStringEx($file, $data)
{	
	$rc = @writeString($file, $data);
	if ( $rc ) return $rc;
	
	/** @since 2007/03/20 잘못된 패스 수정 */
	$pp = pathinfo($file);
	$rc = mkdirEx($pp['dirname']);
	
	if ( ! $rc ) return $rc;

	return writeString($file, $data);
}
/**
 * 파일에 문자열을 기록한다.
 *
 * @param string $filename 생성할 파일 이름
 * @return boolean 성공이면 true, 아니면 false 를 리턴
 */
function writeString($filename, $content)
{
	if (!$handle = fopen($filename, 'w')) {
		return false;
	}
	
	if (fwrite($handle, $content) === FALSE) {
		return false;
	}
	fclose($handle);
	return true;
}
/**
 * 파일의 맨 밑에 문자열을 추가한다.
 *
 * 필요한 라인 피드는 직접 넣어야한다. 라인 피드를 넣을 때, 앞에 넣을 것인지 뒤에 넣을 것인지 결정을 잘 해야한다.
 * @return boolean 성공시 true, 실패시 false
 */
function attachString($filename, $content)
{
	if (!$handle = fopen($filename, 'a')) {
		return false;
	}
	
	if (fwrite($handle, $content) === FALSE) {
		return false;
	}
	fclose($handle);
	return true;
}
/**
 * 파일 내용을 리턴한다.
 *
 * 바이너리 데이터를 읽을 수 있다.
 * 이 함수는 PHP 내장 함수 file_get_contents 와 비슷하다.
 * @return mixed FALSE on error. otherwise file content
 *	- 파일을 열지 못했을 때, false 리턴.
 *	- 내용을 읽지 못했을 때 false 리턴.
 * <code>
 * readfileEx("abc.jpg", "rb");
 * </code>
 */
function readfileEx($filename, $mode='r')
{
	$handle = @fopen($filename, $mode);
	if ( $handle === FALSE ) return FALSE;
	$contents = @fread($handle, filesize($filename));
	if ( $contents === FALSE )
	{
		fclose($handle);
		return FALSE;
	}
	fclose($handle);
	return $contents;
}


/**
 * 파일 삭제
 *
 * 디렉토리는 삭제하지 않는다.
 * 입력된 디렉토리내의 파일들을 삭제한다. 하위 디렉토리가 존재하는 경우, 파일들을 삭제한다.
 */
function unlinkDir($dir)
{
	$ar = getfiles($dir);
	foreach( $ar as $e )
	{
		@unlink($e);
	}
}

/**
 * 지정된 디렉토리 내에 존재하는 디렉토리를 리턴한다.
 *
 *
 */
function getDirs($directory) {

	// Create an array for all files found
	$tmp = Array();

	// Try to open the directory
	if($dir = opendir($directory)) {
		
		// read the files
		while($file = readdir($dir)) {
			// Make sure the file exists
			if($file != "." && $file != ".." && $file[0] != '.') {
				// If it's a directiry, 
				if(is_dir($directory . "/" . $file))
				{
					$tmp[] = $file;
				}
			}
		}
		
		// Finish off the function
		closedir($dir);
		return $tmp;
	}
}


/**
 * 재귀 호출을 통해서 파일 리스트를 얻어서 리턴한다.
 *
 *
 */
function getFiles($directory) {

	// Try to open the directory
	if($dir = opendir($directory)) {
		
	// Create an array for all files found
	$tmp = Array();
	
	// Add the files
	while($file = readdir($dir)) {
		// Make sure the file exists
		if($file != "." && $file != ".." && $file[0] != '.') {
			// If it's a directiry, list all files within it
			if(is_dir($directory . "/" . $file))
			{
				array_push($tmp, $directory . "/" . $file);
				$tmp2 = getFiles($directory . "/" . $file);
				if(is_array($tmp2))
				{
					$tmp = array_merge($tmp, $tmp2);
				}
			}
			else
			{
				array_push($tmp, $directory . "/" . $file);
			}
		}
	}
	
	// Finish off the function
	closedir($dir);
	return $tmp;
	}
}

?>
