/* 
 * FFS File System Driver for Windows
 *
 * ffsdrv.h
 *
 * 2004.5.6 ~
 *
 * Lee Jae-Hong, http://www.pyrasis.com
 *
 */

#ifndef _FFS_HEADER_
#define _FFS_HEADER_

/* Include files */
#include <ntdddisk.h>
#include "fs.h"
#include "dinode.h"
#include "dir.h"

/* Debug */
#undef  DBG
#define DBG 1

/* Structs and Consts */
#define FFS_RO 1

#pragma pack(1)

#define SECTOR_SIZE (512)


typedef struct fs FFS_SUPER_BLOCK, *PFFS_SUPER_BLOCK;

typedef struct ufs1_dinode FFS_INODE, *PFFS_INODE;

typedef struct direct FFS_DIR_ENTRY, *PFFS_DIR_ENTRY;


#define DRIVER_NAME     "FFS"
#define DEVICE_NAME     L"\\ffs"
#if DBG
#define DOS_DEVICE_NAME L"\\DosDevices\\ffs"
#endif


/*
 * FFS_IDENTIFIER_TYPE
 *
 * Identifiers used to mark the structures
 */
typedef enum _FFS_IDENTIFIER_TYPE {
	FGD = ':DGF',
	VCB = ':BCV',
	FCB = ':BCF',
	CCB = ':BCC',
	ICX = ':XCI',
	FSD = ':DSF'
} FFS_IDENTIFIER_TYPE;


/*
 * FFS_IDENTIFIER
 *
 * Header used to mark the structures
 */
typedef struct _FFS_IDENTIFIER {
	FFS_IDENTIFIER_TYPE     Type;
	ULONG                   Size;
} FFS_IDENTIFIER, *PFFS_IDENTIFIER;


/*
 * FFS_GLOBAL_DATA
 *
 * Data that is not specific to a mounted volume
 */
typedef struct _FFS_GLOBAL {
	
	/* Identifier for this structure */
	FFS_IDENTIFIER              Identifier;
	
	/* Syncronization primitive for this structure */
	ERESOURCE                   Resource;
	
	/* Table of pointers to the fast I/O entry points */
	FAST_IO_DISPATCH            FastIoDispatch;
	
	/* Table of pointers to the Cache Manager callbacks */
	CACHE_MANAGER_CALLBACKS     CacheManagerCallbacks;
	
	/* Pointer to the driver object */
	PDRIVER_OBJECT              DriverObject;
	
	/* Pointer to the main device object */
	PDEVICE_OBJECT              DeviceObject;
	
	/* List of mounted volumes */
	LIST_ENTRY                  VcbList;
	
	/* Global flags for the driver */
	ULONG                       Flags;
	
	LARGE_INTEGER				TimeZone;
	
} FFS_GLOBAL, *PFFS_GLOBAL;


/*
 * Flags for FFS_GLOBAL_DATA
 */
#define FFS_UNLOAD_PENDING 0x00000001


/*
 * Driver Extension define
 */
typedef struct {
	FFS_GLOBAL	FFSGlobal;
} FFS_EXT, *PFFS_EXT;


/*
 * FFS_VCB Volume Control Block
 *
 * Data that represents a mounted logical volume
 * It is allocated as the device extension of the volume device object
 */
typedef struct _FFS_VCB {
	
	/* 
	 * FCB header required by NT
	 * The VCB is also used as an FCB for file objects
	 * that represents the volume itself
	 */
	FSRTL_COMMON_FCB_HEADER     CommonFCBHeader;
	SECTION_OBJECT_POINTERS     SectionObject;
	ERESOURCE                   MainResource;
	ERESOURCE                   PagingIoResource;
	/* end FCB header required by NT */
	
	/* Identifier for this structure */
	FFS_IDENTIFIER              Identifier;
	
	LIST_ENTRY                  Next;
	
	/* 
	 * Incremented on IRP_MJ_CREATE, decremented on IRP_MJ_CLEANUP
	 * for files on this volume.
	 */
	ULONG                       OpenFileHandleCount;
	
	/*
	 * Incremented on IRP_MJ_CREATE, decremented on IRP_MJ_CLOSE
	 * for both files on this volume and open instances of the
	 * volume itself.
	 */
	ULONG                       ReferenceCount;
	
	/* Pointer to the VPB in the target device object */
	PVPB                        Vpb;
	
#ifndef FFS_RO
	
	/* If the volume is mounted read only */
	BOOLEAN                     ReadOnly;
	
#endif
	
	/* List of FCBs for open files on this volume */
	LIST_ENTRY                  FcbList;
	
	/* This volumes device object */
	PDEVICE_OBJECT              DeviceObject;
	
	/* The physical device object (the disk) */
	PDEVICE_OBJECT              TargetDeviceObject;
	
	/* Information about the physical device object */
	DISK_GEOMETRY               DiskGeometry;
	PARTITION_INFORMATION       PartitionInformation;
	
	/* Pointer to the file system superblock */
	PFFS_SUPER_BLOCK		ffs_super_block;
	
	/* Number of Group Decsciptions */
	ULONG			ffs_groups;
	
	/* Block and fragment size */
	ULONG			ffs_block;
	ULONG			ffs_frag;
	
	/* Flags for the volume */
	ULONG                       Flags;
	
} FFS_VCB, *PFFS_VCB;


/*
 * Flags for FFS_VCB
 */
#define VCB_VOLUME_LOCKED       0x00000001
#define VCB_DISMOUNT_PENDING    0x00000002


/*
 * FFS_FCB File Control Block
 *
 * Data that represents an open file
 * There is a single instance of the FCB for every open file
 */
typedef struct _FFS_FCB {
	
	/* FCB header required by NT */
	FSRTL_COMMON_FCB_HEADER         CommonFCBHeader;
	SECTION_OBJECT_POINTERS         SectionObject;
	ERESOURCE                       MainResource;
	ERESOURCE                       PagingIoResource;
	/* end FCB header required by NT */
	
	/* Identifier for this structure */
	FFS_IDENTIFIER                  Identifier;
	
	/* List of FCBs for this volume */
	LIST_ENTRY                      Next;
	
#ifndef FFS_RO
	
	/* Share Access for the file object */
	SHARE_ACCESS                    ShareAccess;
	
#endif
	/* List of byte-range locks for this file */
	FILE_LOCK_ANCHOR                FileLockAnchor;

	/* Incremented on IRP_MJ_CREATE, decremented on IRP_MJ_CLEANUP */
	ULONG                           OpenHandleCount;
	
	/* Incremented on IRP_MJ_CREATE, decremented on IRP_MJ_CLOSE */
	ULONG                           ReferenceCount;
	
#ifndef FFS_RO
	
	/* If the file is a page file */
	BOOLEAN                         IsPageFile;
	
	/* If the file is to be deleted */
	BOOLEAN                         DeletePending;
	
#endif
	
	/* The filename */
	UNICODE_STRING                  FileName;
	
#if DBG
	
	/* Non unicode filename for use in debug prints */
	PUCHAR                          DbgFileName;
	
#endif
	
	/* The file attributes */
	ULONG                           FileAttributes;
	
	/* The offset in it's directory */
	LARGE_INTEGER                   IndexNumber;
	
	/* Pointer to the inode */
	struct ufs1_dinode*              dinode;
	
	/* Inode number and it's directory's inode number */
	ULONG			    inode;
	ULONG			    dir_inode;
	
} FFS_FCB, *PFFS_FCB;


/*
 * FFS_CCB Context Control Block
 *
 * Data that represents one instance of an open file
 * There is one instance of the CCB for every instance of an open file
 */
typedef struct _FFS_CCB {
	
	// Identifier for this structure
	FFS_IDENTIFIER  Identifier;
	
	// State that may need to be maintained
	ULONG           CurrentByteOffset;
	UNICODE_STRING  DirectorySearchPattern;
	
} FFS_CCB, *PFFS_CCB;


/*
 * FFS_IRP_CONTEXT
 *
 * Used to pass information about a request between the drivers functions
 */
typedef struct _FFS_IRP_CONTEXT {
	
	/* Identifier for this structure */
	FFS_IDENTIFIER     Identifier;
	
	/* Pointer to the IRP this request describes */
	PIRP                Irp;
	
	/* The major and minor function code for the request */
	UCHAR               MajorFunction;
	UCHAR               MinorFunction;
	
	/* The device object */
	PDEVICE_OBJECT      DeviceObject;
	
	/* The file object */
	PFILE_OBJECT        FileObject;
	
	/* If the request is synchronous (we are allowed to block) */
	BOOLEAN             IsSynchronous;
	
	/* If the request is top level */
	BOOLEAN             IsTopLevel;
	
	/* Used if the request needs to be queued for later processing */
	WORK_QUEUE_ITEM     WorkQueueItem;
	
	/* If an exception is currently in progress */
	BOOLEAN             ExceptionInProgress;
	
	/* The exception code when an exception is in progress */
	NTSTATUS            ExceptionCode;
	
} FFS_IRP_CONTEXT, *PFFS_IRP_CONTEXT;

/*
 * FFS_ALLOC_HEADER
 *
 * In the checked version of the driver this header is put in the beginning of
 * every memory allocation
 */
typedef struct _FFS_ALLOC_HEADER {
	FFS_IDENTIFIER Identifier;
} FFS_ALLOC_HEADER, *PFFS_ALLOC_HEADER;

typedef struct _FCB_LIST_ENTRY {
	PFFS_FCB     Fcb;
	LIST_ENTRY   Next;
} FCB_LIST_ENTRY, *PFCB_LIST_ENTRY;


VOID
FFSUnload(
	IN PDRIVER_OBJECT DriverObject);

LARGE_INTEGER
FFSSysTime(
	IN ULONG i_time);

NTSTATUS
FFSQueryVolumeInformation(
	IN PFFS_IRP_CONTEXT IrpContext);

VOID
FFSCharToWchar(
	IN OUT PWCHAR   Destination,
	IN PCHAR        Source,
	IN ULONG        Length);

NTSTATUS
FFSWcharToChar(
	IN OUT PCHAR    Destination,
	IN PWCHAR       Source,
	IN ULONG        Length);

VOID
FFSFreeFcb(
	IN PFFS_FCB Fcb);

PFFS_CCB
FFSAllocateCcb(VOID);

PFFS_FCB
FFSAllocateFcb(
	IN PFFS_VCB             Vcb,
	IN PUNICODE_STRING      FileName,
	IN ULONG                IndexNumber,
	IN ULONG                inode,
	IN ULONG                dir_inode,
	IN struct ufs1_dinode*  dinode);

VOID
FFSFreeCcb(
	IN PFFS_CCB Ccb);

VOID
FFSFreeVcb(
	IN PFFS_VCB Vcb);

VOID
FFSSetVpbFlag(
	IN PVPB     Vpb,
	IN USHORT   Flag);

VOID
FFSClearVpbFlag(
	IN PVPB     Vpb,
	IN USHORT   Flag);

BOOLEAN
FFSGetInodeLba(
	IN PFFS_VCB vcb,
    IN ULONG    inode,
	OUT PULONG  lba,
	OUT PULONG  offset);

ULONG
FFSBlockMap(
	IN PFFS_VCB            vcb,
	IN struct ufs1_dinode* dinode,
	IN ULONG               Index);

ULONG
FFSReadInode(
	IN PFFS_VCB            vcb,
	IN struct ufs1_dinode* dinode,
	IN ULONG               offset,
	IN PVOID               Buffer,
	IN ULONG               Size,
	IN BOOLEAN             ReadFileData);

BOOLEAN
FFSLoadInode(
	IN PFFS_VCB           vcb,
	IN ULONG              inode,
	IN struct ufs1_dinode *dinode);

NTSTATUS
FFSLookupFileName(
	IN PFFS_VCB         Vcb,
	IN PUNICODE_STRING  FullFileName,
	IN OUT PULONG       Offset,
	IN OUT PULONG       Inode,
	IN OUT PULONG       DirInode,
	IN OUT PFFS_INODE   dinode);

NTSTATUS
FFSScanDir(
	IN PFFS_VCB            Vcb,
	IN ULONG               inode,
	IN PUNICODE_STRING     FileName,
	IN OUT PULONG          Index,
	IN PFFS_INODE          dinode,
	IN OUT PFFS_DIR_ENTRY  ffs_dir);

NTSTATUS
FFSOpenFile(
	PFFS_VCB Vcb, PIRP Irp);

NTSTATUS
FFSCreate(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS 
FFSDiskIoControl(
	IN PDEVICE_OBJECT   pDeviceObject,
	IN ULONG            IoctlCode,
	IN PVOID            InputBuffer,
	IN ULONG            InputBufferSize,
	IN OUT PVOID        OutputBuffer,
	IN OUT PULONG       OutputBufferSize);

BOOLEAN
FFSAcquireForLazyWrite(
	IN PVOID    Context,
	IN BOOLEAN  Wait);

VOID
FFSReleaseFromLazyWrite(
	IN PVOID Context);

BOOLEAN
FFSAcquireForReadAhead(
	IN PVOID    Context,
	IN BOOLEAN  Wait);

VOID
FFSReleaseFromReadAhead(
	IN PVOID Context);

NTSTATUS
FFSClose(
	IN PFFS_IRP_CONTEXT IrpContext);

VOID
FFSQueueCloseRequest(
	IN PFFS_IRP_CONTEXT IrpContext);

VOID
FFSDeQueueCloseRequest(
	IN PVOID Context);

BOOLEAN
FFSFastIoUnlockAll(
	IN PFILE_OBJECT         FileObject,
	IN PEPROCESS            ProcessId,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

BOOLEAN
FFSFastIoUnlockAllByKey(
	IN PFILE_OBJECT         FileObject,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

BOOLEAN
FFSFastIoUnlockSingle(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN PLARGE_INTEGER       Length,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

BOOLEAN
FFSFastIoLock(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN PLARGE_INTEGER       Length,
	IN PEPROCESS            ProcessId,
	IN ULONG                Key,
	IN BOOLEAN              FailImmediately,
	IN BOOLEAN              ExclusiveLock,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

BOOLEAN
FFSFastIoQueryStandardInfo(
	IN PFILE_OBJECT                 FileObject,
	IN BOOLEAN                      Wait,
	OUT PFILE_STANDARD_INFORMATION  Buffer,
	OUT PIO_STATUS_BLOCK            IoStatus,
	IN PDEVICE_OBJECT               DeviceObject);

BOOLEAN
FFSFastIoQueryBasicInfo(
	IN PFILE_OBJECT             FileObject,
	IN BOOLEAN                  Wait,
	OUT PFILE_BASIC_INFORMATION Buffer,
	OUT PIO_STATUS_BLOCK        IoStatus,
	IN PDEVICE_OBJECT           DeviceObject);

BOOLEAN
FFSFastIoQueryNetworkOpenInfo(
	IN PFILE_OBJECT                     FileObject,
	IN BOOLEAN                          Wait,
	OUT PFILE_NETWORK_OPEN_INFORMATION  Buffer,
	OUT PIO_STATUS_BLOCK                IoStatus,
	IN PDEVICE_OBJECT                   DeviceObject);

BOOLEAN
FFSFastIoRead(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN ULONG                Length,
	IN BOOLEAN              Wait,
	IN ULONG                LockKey,
	OUT PVOID               Buffer,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

BOOLEAN
FFSFastIoCheckIfPossible(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN ULONG                Length,
	IN BOOLEAN              Wait,
	IN ULONG                LockKey,
	IN BOOLEAN              CheckForReadOperation,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

VOID
FFSFreeIrpContext(
	IN PFFS_IRP_CONTEXT IrpContext);

PFFS_IRP_CONTEXT
FFSAllocateIrpContext(
	IN PDEVICE_OBJECT DeviceObject,
	IN PIRP Irp);

NTSTATUS
FFSReadSectors(
	IN PDEVICE_OBJECT pDeviceObject,
	IN ULONG          DiskSector,
	IN ULONG          SectorCount,
	IN PVOID          Buffer);

NTSTATUS
FFSReadDisk(
	IN PDEVICE_OBJECT pDeviceObject,
	IN ULONG          lba,
	IN ULONG          offset,
	IN ULONG          Size,
	IN PVOID          Buffer);

struct fs *
FFSLoadSuper(
	IN PDEVICE_OBJECT pDeviceObject);

NTSTATUS
FFSVerifyVolume(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSIsVolumeMounted(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSDismountVolume(
	IN PFFS_IRP_CONTEXT IrpContext);

VOID
FFSPurgeVolume(
	IN PFFS_VCB Vcb,
	IN BOOLEAN  FlushBeforePurge);

VOID
FFSPurgeFile(
	IN PFFS_FCB Fcb,
	IN BOOLEAN  FlushBeforePurge);

NTSTATUS
FFSLockVolume(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSUnlockVolume(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSUserFsRequest(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSMountVolume(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSFileSystemControl(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSReadFileData(
	IN PFFS_VCB             Vcb,
	IN ULONG                Index,
	IN struct ufs1_dinode*  dinode,
	IN PLARGE_INTEGER       Offset,
	IN ULONG                Length,
	IN OUT PVOID            Buffer);

NTSTATUS
FFSRead(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSReadNormal(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSReadComplete(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSQueryInformation(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSSetInformation(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSLockUserBuffer(
	IN PIRP             Irp,
	IN ULONG            Length,
	IN LOCK_OPERATION   Operation);

PVOID
FFSGetUserBuffer(
	IN PIRP Irp);

ULONG
FFSGetInfoLength(
	IN FILE_INFORMATION_CLASS  FileInformationClass);

ULONG
FFSProcessDirEntry(
	IN PFFS_VCB                Vcb,
	IN FILE_INFORMATION_CLASS  FileInformationClass,
	IN ULONG                   in,
	IN PVOID                   Buffer,
	IN ULONG                   UsedLength,
	IN ULONG                   Length,
	IN ULONG                   FileIndex,
	IN PUNICODE_STRING         pName,
	IN BOOLEAN                 Single);

NTSTATUS
FFSQueryDirectory(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSDirectoryControl(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSQueueRequest(
	IN PFFS_IRP_CONTEXT IrpContext);

VOID
FFSDeQueueRequest(
	IN PVOID Context);

NTSTATUS
FFSCleanup(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSDispatchRequest(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSExceptionFilter(
	IN PFFS_IRP_CONTEXT IrpContext,
	IN NTSTATUS ExceptionCode);

NTSTATUS
FFSExceptionHandler(
	IN PFFS_IRP_CONTEXT IrpContext);

NTSTATUS
FFSBuildRequest(
	IN PDEVICE_OBJECT DeviceObject,
	IN PIRP Irp);

PUCHAR
FFSNtStatusToString(
	IN NTSTATUS Status);

#if DBG

BOOLEAN
FFSFastIoRead(
	IN PFILE_OBJECT         FileObject,
	IN PLARGE_INTEGER       FileOffset,
	IN ULONG                Length,
	IN BOOLEAN              Wait,
	IN ULONG                LockKey,
	OUT PVOID               Buffer,
	OUT PIO_STATUS_BLOCK    IoStatus,
	IN PDEVICE_OBJECT       DeviceObject);

VOID
FFSDbgPrintCall(
	IN PDEVICE_OBJECT   DeviceObject,
	IN PIRP             Irp);

VOID
FFSDbgPrintComplete(
	IN PIRP Irp);

#define FFSCompleteRequest(Irp, PriorityBoost) \
        FFSDbgPrintComplete(Irp); \
        IoCompleteRequest(Irp, PriorityBoost)

#else

#define FFSDbgPrintCall(DeviceObject, Irp)

#define FFSCompleteRequest(Irp, PriorityBoost) \
        IoCompleteRequest(Irp, PriorityBoost)

#endif /* DBG */


#endif /* _FFS_HEADER_ */
