/*
 * @(#) ApplicationTrace.java 0.1 
 * $Rev: 48 $ 
 * $Date: 2009-05-26 23:38:02 +0900 (26 5 2009) $
 * $Author: $      
 */
/*
 * Copyright 2009 the original author or authors.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.archi.tools.excatj;

import java.util.HashMap;
import java.util.Map;

import com.sun.jdi.ThreadReference;
import com.sun.jdi.event.BreakpointEvent;
import com.sun.jdi.event.ClassPrepareEvent;
import com.sun.jdi.event.ClassUnloadEvent;
import com.sun.jdi.event.Event;
import com.sun.jdi.event.EventSet;
import com.sun.jdi.event.ExceptionEvent;
import com.sun.jdi.event.LocatableEvent;
import com.sun.jdi.event.MethodEntryEvent;
import com.sun.jdi.event.MethodExitEvent;
import com.sun.jdi.event.StepEvent;
import com.sun.jdi.event.ThreadDeathEvent;
import com.sun.jdi.event.ThreadStartEvent;
import com.sun.jdi.event.VMDeathEvent;
import com.sun.jdi.event.VMDisconnectEvent;
import com.sun.jdi.event.VMStartEvent;
import com.sun.jdi.event.WatchpointEvent;

public class ApplicationTrace implements EventNotifier {
	static String nextBaseIndent = ""; // Starting indent for next thread
	private final TraceOutput appTraceLog;

	// Maps ThreadReference to ThreadTrace instances
	private Map traceMap = new HashMap();

	public ApplicationTrace() {
		appTraceLog = new TraceOutput(this);
		appTraceLog.println("ExCatJLogo", 
				new Object [] { ExCatJ.progname, ExCatJ.majorVersion, ExCatJ.minorVersion, ExCatJ.author});
		
		appTraceLog.printDirectln(Env.connection().description());
	}
	
	public void applicationResume() {
		if (ThreadInfo.getCurrentThreadInfo() == null) {
			MessageOutput.printDirectln("Nothing suspended.");
			return;
		}
		ThreadInfo.invalidateAll();
		Env.vm().resume();
	}

	public void breakpointEvent(BreakpointEvent event) {
        Thread.yield();  // fetch output
		threadTrace(event.thread()).breakpointEvent(event);
	}

	public void classPrepareEvent(ClassPrepareEvent event) {
		// 
	}

	public void classUnloadEvent(ClassUnloadEvent event) {
		appTraceLog.printDirectln("-- Class Unload Event --");
	}

	private ThreadReference eventThread(Event event) {
		if (event instanceof ClassPrepareEvent) {
			return ((ClassPrepareEvent) event).thread();
		} else if (event instanceof LocatableEvent) {
			return ((LocatableEvent) event).thread();
		} else if (event instanceof ThreadStartEvent) {
			return ((ThreadStartEvent) event).thread();
		} else if (event instanceof ThreadDeathEvent) {
			return ((ThreadDeathEvent) event).thread();
		} else if (event instanceof VMStartEvent) {
			return ((VMStartEvent) event).thread();
		} else {
			return null;
		}
	}

	public void exceptionEvent(ExceptionEvent event) {
		Thread.yield(); // fetch output

		ThreadTrace trace = threadTrace(event.thread());
		// ThreadTrace trace = (ThreadTrace)traceMap.get(event.thread());
		if (trace != null) { // only want threads we care about
			trace.exceptionEvent(event); // Forward event
		}
		// applicationResume();				
	}

	public void fieldWatchEvent(WatchpointEvent event) {
		// threadTrace(event.thread()).fieldWatchEvent(event);
	}

	public void methodEntryEvent(MethodEntryEvent event) {
        Thread.yield();  // fetch output		
		threadTrace(event.thread()).methodEntryEvent(event);
	}

	public void methodExitEvent(MethodExitEvent event) {
        Thread.yield();  // fetch output		
		threadTrace(event.thread()).methodExitEvent(event);
	}

	public void receivedEvent(Event event) {
		setCurrentThread(eventThread(event));
	}

	private void setCurrentThread(EventSet set) {
		ThreadReference thread;
		if (set.size() > 0) {
			/*
			 * If any event in the set has a thread associated with it, they all
			 * will, so just grab the first one.
			 */
			Event event = (Event) set.iterator().next(); // Is there a better
			// way?
			thread = eventThread(event);
		} else {
			thread = null;
		}
		setCurrentThread(thread);
	}

	private void setCurrentThread(ThreadReference thread) {
		ThreadInfo.invalidateAll();
		ThreadInfo.setCurrentThread(thread);
	}

	public void stepEvent(StepEvent event) {
		//
	}

	public void threadDeathEvent(ThreadDeathEvent event) {
        ThreadInfo.removeThread(event.thread());
		ThreadTrace trace = (ThreadTrace) traceMap.get(event.thread());
		if (trace != null) { // only want threads we care about
			trace.threadDeathEvent(event); // Forward event
			traceMap.remove(event.thread());
		}
	}

	public void threadStartEvent(ThreadStartEvent event) {
        ThreadInfo.addThread(event.thread());		
		threadTrace(event.thread());
	}

	/**
	 * Returns the ThreadTrace instance for the specified thread, creating one
	 * if needed.
	 */
	ThreadTrace threadTrace(ThreadReference thread) {
		ThreadTrace trace = (ThreadTrace) traceMap.get(thread);
		if (trace == null) {
			trace = new ThreadTrace(thread, nextBaseIndent, appTraceLog);
			ThreadInfo.addThread(thread);
			traceMap.put(thread, trace);
		}
		return trace;
	} // ThreadTrace Class

	public void vmDeathEvent(VMDeathEvent event) {
		appTraceLog.printDirectln("-- The application exited --");
	}

	public void vmDisconnectEvent(VMDisconnectEvent event) {
		appTraceLog.printDirectln("-- The application has been disconnected --");
	}

	public void vmInterrupted(EventSet eventSet) {
		setCurrentThread(eventSet);
	}

	public void vmStartEvent(VMStartEvent event) {
        Thread.yield();  // fetch output
		appTraceLog.printDirectln("-- VM Started --");
	}
}
