/* rovm.c - ROVM Server  main .
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdlib.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "ssl_conf.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "connection.h"
#include "request.h"
#include "proc_rc.h"

#include "log.h"
#include "common.h"
#include "ticket.h"

#include "gc.h"
#include "utils.h"

const char *
rv_get_server_version (void)
{
  return VERSION;
}

/**
   ROVM  ߽ Լ.   Լ MPM  Ѵ.

   @param r	ROVM ü.
*/
int
rovm_main (r)
     struct rovm *r;
{
  gc_run (r);

  mpm_pre_config (r);
  mpm_run (r);

  return 0;
}

/**
   Process     ü Ѵ.

   @param argc	Argument Count
   @param argv	Argument Value 
   @return	Ӱ  struct process_rec ü ȯѴ.
 */
static struct process_rec *
create_process_rec (argc, argv)
     int argc;
     const char * const *argv;
{
  struct process_rec *process;
  rc_pool_t *cntx;
  rc_status_t stat;
  
  stat = mp_create (&cntx, NULL);
  if (stat != RC_SUCCESS)
    {
      /* XXX From the time that we took away the NULL pool->malloc mapping
	     we have been unable to log here without segfaulting.  */
      rovm_log (NULL, ROVMLOG_ERR, ROVMLOG_MARK,
		"rc_pool_create() failed to create initial context");
      rc_terminate ();
      exit (1);
    }

  mp_tag (cntx, "process");
  
  process = mp_alloc (cntx, sizeof (struct process_rec));
  process->pool = cntx;
  
  mp_create (&process->pconf, process->pool);
  mp_tag (process->pconf, "pconf");

  mp_create (&process->ptemp, process->pool);
  mp_tag (process->ptemp, "ptemp");

  process->argc = argc;
  process->argv = argv;
  process->short_name = rc_filepath_name_get (argv[0]);

  return process;
}

/**
   ROVM Server   ¸ ʱȭմϴ.

   @param r	ROVM ü
   @param p	޸ pool
 */
int
init_rovm_status (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
  struct rovm_status *rs;

  rs = (struct rovm_status *) mp_alloc (p, sizeof (struct rovm_status));
  if (!rs)
    return -1;

  STATUS_SHUTDOWN_PENDING (rs) = 0;
  STATUS_RESTART_PENDING (rs) = 0;

  ROVM_STATUS (r) = rs;

  return 0;
}

/**
   Multi-Process  ۵ , ȿ ڽ μ ٷ
   ؼ Ʒ  scoreboard   մϴ.  ̸ ʱȭϴ
   ԼԴϴ.

   @param r     ROVM ü
   @param p      ޸ pool.
   @return      н -1,  0
 */
int
init_rovm_sb (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
  struct rovm_scoreboard *sb;

  sb = (struct rovm_scoreboard *) mp_calloc (p, sizeof (struct rovm_scoreboard));
  SB_PARENT (sb) = mp_calloc (p, CONF_DAEMONS_TO_START (ROVM_CONF (r)) * sizeof (struct process_score));
  
  ROVM_SCOREBOARD (r) =  sb;

  return 0;
}

static const char *
conf_section_server (cmd_parms *cmd, void *dummy, const char *arg)
{
  rovm_log (NULL, ROVMLOG_DEBUG, ROVMLOG_MARK, "TODO");

  return NULL;
}

static const char *
conf_section_option (cmd_parms *cmd, void *dummy, const char *arg)
{
  rovm_log (NULL, ROVMLOG_DEBUG, ROVMLOG_MARK, "TODO");

  return NULL;
}

static const char *
conf_todo (cmd_parms *cmd, void *dummy, const char *arg)
{
  rovm_log (NULL, ROVMLOG_DEBUG, ROVMLOG_MARK, "TODO = %s", cmd->cmd->name);

  return NULL;
}

static const char *
conf_nothing (cmd_parms *cmd, void *dummy, const char *arg)
{
  return NULL;
}

static const char *
conf_set_classroot (cmd_parms *cmd, void *dummy, const char *arg)
{
  struct rovm_config *c = ROVM_CONF (CMD_ROVM (cmd));

  if (!arg || (arg && strlen (arg) <= 0))
    return "Invalid ClassRoot";

  CONF_CLASSROOT (c) = mp_strdup (cmd->pool, arg);

  return NULL;
}

static const char *
conf_set_PermitAnonymous (cmd_parms *cmd, void *dummy, const char *arg)
{
  struct rovm_config *c = ROVM_CONF (CMD_ROVM (cmd));

  if (!arg || (arg && strlen (arg) <= 0))
    return "Invalid PermitAnonymous";

  if (!strcasecmp (arg, "yes"))
    CONF_PERMIT_ANONYMOUS (c) = 1;

  return NULL;
}

/*
  `Listen' ɼ óմϴ.
 */
static const char *
conf_set_listener (cmd_parms *cmd, void *dummy, int argc, char *const argv[])
{
  char *host, *scope_id, *proto;
  rc_port_t port;
  rc_status_t rv;

  if (argc < 1 || argc > 2)
    return "Listen requires 1 or 2 arguments.";

  rv = rc_parse_addr_port (&host, &scope_id, &port, argv[0], cmd->pool);
  if (rv != RC_SUCCESS)
    return "Invalid address or port";

  if (host && !strcmp (host, "*"))
    host = NULL;

  if (scope_id)
    /* XXX scope id support is useful with link-local IPv6 addresses */
    return "Scope id is not supported";
  
  if (!port)
    return "Port must be specified";

  if (argc != 2)
    proto = "envlang";
  else 
    {
      proto = mp_strdup (cmd->pool, argv[1]);
      rc_str_tolower (proto);
    }

  return alloc_listener (CMD_ROVM (cmd), host, port, proto);
}

extern rc_array_header_t *rc_server_pre_read_config;
extern rc_array_header_t *rc_server_post_read_config;
extern rc_directive_t *rc_conftree;

static const command_rec core_cmds[] = 
  {
    RC_INIT_RAW_ARGS ("<server", conf_section_server, NULL, RSRC_CONF, "Server Container."),
    RC_INIT_RAW_ARGS ("<option", conf_section_option, NULL, RSRC_CONF, "Option Container."),
    RC_INIT_TAKE_ARGV ("Listen", conf_set_listener, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("ClassRoot", conf_set_classroot, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("ServerName", conf_nothing, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("ServerAdmin", conf_nothing, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("ErrorLog", conf_nothing, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("TransferLog", conf_nothing, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("PermitAnonymous", conf_set_PermitAnonymous, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLPassPhraseDialog", ssl_cmd_SSLPassPhraseDialog, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLSessionCache", ssl_cmd_SSLSessionCache, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLSessionCacheTimeout", ssl_cmd_SSLSessionCacheTimeout, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLMutex", ssl_cmd_SSLMutex, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLEngine", ssl_cmd_SSLEngine, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLCipherSuite", ssl_cmd_SSLCipherSuite, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLCertificateFile", ssl_cmd_SSLCertificateFile, NULL, RSRC_CONF,  "No description"),
    RC_INIT_TAKE1 ("SSLCertificateKeyFile", ssl_cmd_SSLCertificateKeyFile, NULL, RSRC_CONF,  "No description"),
    { NULL }
  };

extern int rc_process_config_tree (struct rovm *, rc_directive_t *, rc_pool_t *, rc_pool_t *);

/**
   ROVM  õ   ʱȭѴ.

   @param r	ROVM ü
   @param p	޸ pool
 */
int
init_rovm_config (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
#define DEFAULT_LISTENBACKLOG 511
  struct rovm_config *c;
  rc_pool_t *pcommands;

  c = (struct rovm_config *) mp_calloc (p, sizeof (struct rovm_config));
  if (!c)
    return -1;

  CONF_OPTION (c) |= OPTION_LOG_DEBUG_MESSAGES;

  CONF_PERMIT_ANONYMOUS (c) = 0;

  /*  ROVM  , debugging  Ϲ    ֽϴ.  
     Ʒ  0    Ϲ  ۵ϰ Ǹ, 1   
     debugging  ۵ϰ ˴ϴ.  
     
     Ϲ  , daemon  ϴ  , foreground  
     ϴ  մϴ.  */
  CONF_ONE_PROCESS (c) = 1;
  /*  daemon  ϱ Ѵٸ Ʒ  0  ϰ, foreground
      ϱ Ѵٸ 1  Ͻñ ٶϴ.  */
  CONF_NO_DETACH (c) = 0;

  CONF_PASSWORD_PATH (c) = "conf/passwd";

  CONF_HOSTNAME (c) = "localhost";
  CONF_PORT (c) = DEFAULT_ROVM_LISTENING_PORT;

  /* Multi-Process  ̿ 񽺸 ǽ , Ʒ ڸ ø 
      ̴.    ROVM    ݵ Ʒ 
      1 ̿߸ Ѵ.  */
  CONF_DAEMONS_TO_START (c) = 1;
  CONF_DAEMONS_LIMIT (c) = 1;

  /* ϳ Process   thread  ִ  ϴ κԴϴ.  
      5  ⺻ Ǿ , 1  listener  5 
     worker 尡 ǰ ˴ϴ.  */
  CONF_THREADS_PER_CHILD (c) = 5;

  /* Listening    */
  CONF_SEND_BUFFER_SIZE (c) = 0;        /* ý default  ϱ ؼ 0  ؾ Ѵ.  */
  CONF_RECEIVE_BUFFER_SIZE (c) = 0;     /* ý default  ϱ ؼ 0  ؾ Ѵ.  */
  CONF_LISTENBACKLOG (c) = DEFAULT_LISTENBACKLOG;

  /* ERROR ߻   descriptor    ֽϴ.  ν
      debugging  stderr  µǵ Ǿ ֽϴ.  */
  CONF_ERROR_LOG (c) = stderr;
  CONF_ERROR_FNAME (c) = "logs/error_log";
  CONF_LOGLEVEL (c) = DEFAULT_LOGLEVEL;

  /* Ticket ü ⺻  stack slot   ϴ 
     κԴϴ.  ⺻ DEFAULT_STACK_SLOT   Ǿ ֽϴ. */
  CONF_DEFAULT_STACK_SLOT (c) = DEFAULT_STACK_SLOT;

  /* Class ϵ root θ ϰ ˴ϴ.   ο 
     e://<hostname>[:<port>]<path> URL  <path>  Ͽ ϳ ý
     θ Ű ˴ϴ.

     (20050205)  ROVM Server  ,    ʱ 
     ڰ `NEW' Opcode  ׽Ʈϱ ؼ  κ  ϵ
     ȳ  ҰԴϴ.  */
  CONF_CLASSROOT (c) = "/usr/local/rovm/classroot";

  ROVM_CMDS (r) = (command_rec *) core_cmds;
  ROVM_CONF (r) = c;

  mp_create (&pcommands, ROVM_POOL (r));
  mp_tag (pcommands, "pcommands");
  rc_server_pre_read_config  = rc_array_make (pcommands, 1, sizeof(char *));
  rc_server_post_read_config = rc_array_make (pcommands, 1, sizeof(char *));

  if (rc_read_config (r, ROVM_PTEMP (r), "conf/rovm.conf", &rc_conftree))
    return -1;
  if (rc_process_config_tree (r, rc_conftree, ROVM_PCONF (r), ROVM_PTEMP (r)))
    return -1;

  return 0;
}

/**
   Config   б    κ óմϴ.
*/
int
init_rovm_pre_config (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
  /* Ticket TREE  ʱȭѴ.  */
  if (init_rovm_tktree (r))
    return -1;
  /* Class TREE  ʱȭմϴ.  */
  if (init_rovm_clstree (r))
    return -1;
  if (init_rovm_ssl_pre_config (r))
    return -1;  

  return 0;
}

/**
   Config    Ŀ   κ óմϴ.
*/
int
init_rovm_post_config (r, p)
     struct rovm *r;
     rc_pool_t *p;
{
  /* ROVM Server    ʱȭմϴ.  */
  if (init_rovm_status (r, p))
    return -1;
  if (init_rovm_sb (r, p))
    return -1;
  /* Listening Socket   ʱȭ ǽմϴ.  */
  if (init_rovm_listen (r, p))
    return -1;
  if (init_rovm_ssl (r))
    return -1;
  /* Garbage Collector   ʱȭ ǽմϴ.  */
  if (init_rovm_gc (r, p))
    return -1;

  return 0;
}

/**
   ROVM  ʿ ʱȭ մϴ.

   @return	ʱȭ struct rovm *  ȯմϴ.
*/
struct rovm *
init_rovm (argc, argv)
     int argc;
     char *argv[];
{
  rc_status_t rv;
  struct rovm *r;
  struct process_rec *p;

  rc_initialize ();

  p = create_process_rec (argc, argv);

  r = (struct rovm *) mp_calloc (PROCREC_POOL (p), sizeof (struct rovm));
  if (!r)
    return NULL;

  ROVM_PROCESS (r) = p;

  rv = rc_thread_mutex_create (&ROVM_MUTEX (r), RC_THREAD_MUTEX_DEFAULT, PROCREC_POOL (p));
  if (rv != RC_SUCCESS)
    return NULL;
  
  if (init_rovm_pre_config (r, PROCREC_POOL (p)))
    return NULL;
  /* ROVM Configuration  κ Ѵ.  */
  if (init_rovm_config (r, PROCREC_POOL (p)))
    return NULL;
  if (init_rovm_post_config (r, PROCREC_POOL (p)))
    return NULL;

  return r;
}

/**
   ROVM  մϴ. 

   @param r	ROVM     ü.
 */
int
finish_rovm (r)
     struct rovm *r;
{
  rc_terminate ();

  return 0;
}

/**
   ROVM  main Լ.
 */
int
main (argc, argv)
     int argc;
     char *argv[];
{
  struct rovm *r;

  r = init_rovm (argc, argv);
  if (!r)
    return -1;

  rovm_main (r);

  finish_rovm (r);

  return 0;
}
