#include <stdarg.h>
#include <assert.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "ssl_conf.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "connection.h"
#include "request.h"
#include "proc_rc.h"

#include "ticket.h"
#include "file.h"

#include "el.h"
#include "gc.h"

#ifdef WANT_SIGFPE_HANDLER
/* Define locally if they are not defined in Python.  This gives only
 * the limited control to induce a core dump in case of an exception.
 */
#include <setjmp.h>
jmp_buf RvFPE_jbuf;
int RvFPE_counter = 0;
#endif

/**
   FDS  ǵǾ ִ field  Ŭ CLS   field  Ѵ.

   @param cls    Class   Ŭ  
   @param fds   Field  Ǿ ִ ü .
   @return         ߻ , -1 , ׷   0  ȯϰ
                ȴ.
 */
int
RvDefineFields (RvClass *cls, RvFieldDef *fds)
{
  RvFieldDef *fd = fds;

  while (fd->name)
    {
      RvField *f;

      if (rc_find_efield (cls, (u1 *) fd->name, (u1 *) fd->type))
        return 0;

      f = init_efield (cls, (u1 *) fd->name, (u1 *) fd->type);
      if (!f)
        return -1;

      FIELD_NEXT (f) = CLASS_FIELD (cls);
      CLASS_FIELD (cls) = f;

      fd++;
    }

  return 0;
}

/**
   CLS  MDS  о ش Method  ǵ CLS ο Ѵ.

   @param cls    Class   Ŭ  
   @param mds   Method  Ǿ ִ .
   @return         ߻ , -1 , ׷   0  ȯϰ
                ȴ.
 */
int
RvDefineMethods (RvClass *cls, RvMethodDef *mds)
{
  RvMethodDef *md = mds;

  while (md->name)
    {
      RvMethod * m;

      if (rc_find_emethod (cls, (u1 *) md->name, (u1 *) md->type))
        return 0;

      /* Native Function  ȣ  ֵ method  մϴ.  */
      m = init_emethod ((u1 *) md->name, (u1 *) md->type);
      if (!m)
        return -1;

      METHOD_FLAG (m) |= ACCESS_NATIVE;
      METHOD_NATIVE (m) = md->func;
      METHOD_NEXT (m) = CLASS_METHOD (cls);
      
      CLASS_METHOD (cls) = m;

      md++;
    }

  return 0;
}

/**
   ObjectRef  ڰ Ÿ   ˴ϴ.

   @param self   ObjectRef
   @param idx     index ȣ (ݵ OBJECT_MAX_USERDATA 
                 ۾ƾ մϴ.
   @param value  .
*/
int
RvSetUserData (RvObject *self, int idx, void *value)
{
  if (idx >= OBJECT_MAX_USERDATA)
    return -1;

  OBJECT_USERDATA_IDX (self, idx) = (unsigned long) value;

  return 0;
}

/**
   ObjectRef  ڰ  Ÿ  , Ѵ.

   (ܺ  Լ)

   @param self   ObjectRef
   @param idx    Ÿ index ȣ
   @return      ڰ   ȯմϴ.
*/
void *
RvGetUserData (RvObject *self, int idx)
{
  if (!self || idx >= OBJECT_MAX_USERDATA || idx < 0)
    return NULL;

  return (void *) OBJECT_USERDATA_IDX (self, idx);
}

/**
   ARGS  FMT  Ͽ ش  ùٸ   ִ Ȱ ϴ
   ԼԴϴ.

   (ܺ  Լ)

   @param args  RvValue üԴϴ.  ̰ Stack Slot   ͸
                 ˴ϴ.
   @param fmt   ؼ Դϴ.
   @return        0 , ׷   -1  ȯѴ.
*/
int
RvParseArg (RvObject *self, RvValue *args, const char *fmt, ...)
{
  va_list ap;
  int idx;
  char *p = (char *) fmt;

  va_start (ap, fmt);

  idx = 0;
  while (*p)
    {
      RvValue *col = args + idx;

      switch (*p)
        {
        case 'x':
          {
            RvValue **t = va_arg (ap, RvValue **);

            *t = col;
            break;
          }
        case '[':
          {
            RvObject **t = va_arg (ap, RvObject **);

            if (STACK_TYPE (col) != STACK_TYPE_ARRAYREF)
              goto error;

            *t = GET_STACK_VALUE (ARRAYREF, col);
            break;
          }
        case 't':
          {
            RvObject **t = va_arg (ap, RvObject **);

            if (STACK_TYPE (col) != STACK_TYPE_OBJREF)
              goto error;

            *t = GET_STACK_VALUE (OBJREF, col);
            break;
          }
        case 'S':
          {
            RvObject **t = va_arg (ap, RvObject **);

            if (STACK_TYPE (col) != STACK_TYPE_STRINGREF)
              goto error;

            *t = GET_STACK_VALUE (STRINGREF, col);
            break;
          }
        case 's':
          {
            char **s = va_arg (ap, char **);
            RvValue *value;
            RvObject *strobj;

            if (STACK_TYPE (col) != STACK_TYPE_STRINGREF)
              goto error;

            strobj = GET_STACK_V_STRINGREF (col);
            value = RvGetFieldValue (strobj, "value", "[");
            if (!value)
              goto error;
            *s = RvValue2Str (strobj, value);

            if (p[1] == '#')
              {
                int *len = va_arg (ap, int *);

                p++;
                if (RvGetField (strobj, "len", "I", len))
                  goto error;
              }
            break;
          }
        case 'b':
          {
            unsigned char *c = va_arg (ap, unsigned char *);

            if (STACK_TYPE (col) != STACK_TYPE_BOOLEAN)
              goto error;
           
            *c = GET_STACK_VALUE (BOOLEAN, col);
            break;
          }
        case 'c':
          {
            char *c = va_arg (ap, char *);

            if (STACK_TYPE (col) != STACK_TYPE_CHAR)
              goto error;
           
            *c = GET_STACK_VALUE (CHAR, col);
            break;
          }
        case 'h':
          {
            short *h = va_arg (ap, short *);

            if (STACK_TYPE (col) != STACK_TYPE_SHORT)
              goto error;
           
            *h = GET_STACK_VALUE (SHORT, col);
            break;
          }
        case 'i':
          {
            int *i = va_arg (ap, int *);
           
            if (STACK_TYPE (col) != STACK_TYPE_INT)
              goto error;

            *i = GET_STACK_VALUE (INT, col);
            break;
          }
        case 'f':
          {
            float *f = va_arg (ap, float *);
           
            if (STACK_TYPE (col) != STACK_TYPE_FLOAT)
              goto error;

            *f = GET_STACK_VALUE (FLOAT, col);
            break;
          }
        case 'd':
          {
            double *d = va_arg (ap, double *);
           
            if (STACK_TYPE (col) != STACK_TYPE_DOUBLE)
              goto error;

            *d = GET_STACK_VALUE (DOUBLE, col);
            break;
          }
        default:
          goto error;
          break;
        }

      idx++;
      p++;
    }

  va_end (ap);

  return 0;

 error:
  va_end (ap);

  return -1;
}

/**
   Field  ϴ ԼԴϴ.

   @param self  SELF ObjectRef
   @param name  Field ̸
   @param type  Field Ÿ
 */
int
RvSetField (RvObject *self, const char *name, const char *type, ...)
{
  va_list ap;
  RvValue *val;
  void *ptr;

  va_start (ap, type);

  val = rc_getfieldvalue (self, (u1 *) name, (u1 *) type);
  if (!val)
    goto error;

  switch (type[0])
    {
    case 'B':
      SET_STACK_VALUE (val, BOOLEAN, (unsigned char) va_arg (ap, int));
      break;
    case 'C':
      SET_STACK_VALUE (val, CHAR, (char) va_arg (ap, int));
      break;
    case 'H':
      SET_STACK_VALUE (val, SHORT, (short) va_arg (ap, int));
      break;
    case 'I':
      SET_STACK_VALUE (val, INT, (int) va_arg (ap, int));
      break;
    case 'F':
      SET_STACK_VALUE (val, FLOAT, (float) va_arg (ap, double));
      break;
    case 'D':
      SET_STACK_VALUE (val, DOUBLE, (double) va_arg (ap, double));
      break;
    case 'T':
      ptr = (void *) va_arg (ap, RvObject *);
      SET_STACK_VALUE (val, OBJREF, ptr);
      break;
    case 'S':
      ptr = (void *) va_arg (ap, RvObject *);
      SET_STACK_VALUE (val, STRINGREF, ptr);
      break;
    case '[':
      ptr = (void *) va_arg (ap, RvObject *);
      SET_STACK_VALUE (val, ARRAYREF, ptr);
      break;
    default:
      goto error;
    }

  va_end (ap);
  return 0;

 error:
  va_end (ap);
  return -1;
}

char *
RvArray2Str (RvObject *self, RvObject *array)
{
  RvValue *val;

  if (!self || !array)
    return "(error)";

  if (!OBJECT_IS_ARRAY (array))
    return "(error)";

  val = rc_getfieldvalue (array, "value", "[");
  if (!val)
    return "(error)";

  if (STACK_IS_STRINGVALUE (val))
    {
      char *p = (char *) GET_STACK_VALUE (ADDR, val);
      RvValue *v2;

      v2 = RvGetFieldValue (array, "len", "I");
      if (!v2)
        return "(broken_string)";
      
      return gc_strndup (OBJECTL_ROVM (self), OBJECT_TICKET (self), p, GET_STACK_VALUE (INT, v2));
    }
  else
    return "(can't process)";

  return "(error)";
}

/**
   RvValue  ڿ ȯϿ ȯմϴ.
 */
char *
RvValue2Str (RvObject *self, RvValue *val)
{
  if (!self || !val)
    return NULL;

  switch (STACK_TYPE (val))
    {
    case STACK_TYPE_VOID:
      return "(void)";
    case STACK_TYPE_BOOLEAN:
      return "(boolean)";
    case STACK_TYPE_CHAR:
      return "(char)";
    case STACK_TYPE_SHORT:
      return "(short)";
    case STACK_TYPE_INT:
      return "(int)";
    case STACK_TYPE_FLOAT:
      return "(float)";
    case STACK_TYPE_DOUBLE:
      return "(double)";
    case STACK_TYPE_OBJREF:
      return "(objref)";
    case STACK_TYPE_STRINGREF:
      return "(stringref)";
    case STACK_TYPE_ARRAYREF:
      {
        if (STACK_IS_STRINGVALUE (val))
          {
            char *p = (char *) GET_STACK_VALUE (ADDR, val);
            RvValue *v2;

            if (!OBJECT_IS_STRING (self))
              return "(invalid StringRef)";

            v2 = RvGetFieldValue (self, "len", "I");
            if (!v2)
              return "(broken_string)";
            
            return gc_strndup (OBJECTL_ROVM (self), OBJECT_TICKET (self), p, GET_STACK_VALUE (INT, v2));
          }
        else
          return "(can't process)";
        break;
      }
    default:
      break;
    }

  return "(unknown)";
}

/**
   RvValue   ȯϿ ȯմϴ.
 */
static int
RvValue2Int (RvObject *self, RvValue *val)
{
  if (!self || !val)
    return 0;

  switch (STACK_TYPE (val))
    {
    case STACK_TYPE_BOOLEAN:
    case STACK_TYPE_SHORT:
    case STACK_TYPE_CHAR:
    case STACK_TYPE_INT:
      return GET_STACK_VALUE (INT, val);
    default:
      break;
    }

  return 0;
}

/**
   Ÿ FTYPE   ʵ FNAME  SELF  ã ش  Ǵ
    RvValue ü ȯϰ Ǵϴ.

   @param self  SELF ü.
   @param fname ʵ ̸
   @param ftype ʵ Ÿ
   @return      SELF  ϴ ʵ带   ȯմϴ.
 */
RvValue *
RvGetFieldValue (RvObject *self, const char *fname, const char *ftype)
{
  RvValue *val;

  if (!self || !fname || !ftype)
    return NULL_VALUE;

  val = rc_getfieldvalue (self, fname, ftype);
  if (!val)
    return NULL_VALUE;

  return val;
}

/**
   Field   ԼԴϴ.  Type   ...  Ǵ 
   Ư¡Դϴ.

   @param self  SELF ObjectRef
   @param name  Field ̸
   @param type  Field Ÿ
 */
int
RvGetField (RvObject *self, const char *fname, const char *ftype, ...)
{
  va_list ap;
  RvValue * val;

  if (!self || !fname || !ftype)
    return -1;

  val = rc_getfieldvalue (self, fname, ftype);
  if (!val)
    return -1;

  va_start (ap, ftype);

  switch (ftype[0])
    {
    case '[':
      {
        RvObject **o = va_arg (ap, RvObject **);

        *o = GET_STACK_VALUE (ARRAYREF, val);
        break;
      }
    case 'S':
      {
        RvObject **o = va_arg (ap, RvObject **);

        *o = GET_STACK_VALUE (STRINGREF, val);
        break;
      }
    case 'I':
      {
        int *i = va_arg (ap, int *);

        *i = RvValue2Int (self, val);
        break;
      }
    default:
      va_end (ap);
      return -1;
    }

  va_end (ap);
  return 0;
}

/**
   TYPE  迭 Ͽ ش ͸ ȯմϴ.
 */
RvObject *
RvArrayNew (RvObject *self, int len)
{
  if (!self)
    return NULL_OBJECT;

  return NewRvArray (OBJECTL_ROVM (self), OBJECT_TICKET (self), len);
}

/**
   迭 ü ARRAY    Ǵ  ʵ VALUE  ּ ͸
   ȯϰ ˴ϴ.

   迭 ü ڿ ü ʵ VALUE  Ѵ  Ưϰ ϴ
   κ մϴ.

   @param array 迭 ü
   @return      ʵ VALUE  ּ 
 */
RvValue *
RvArrayGetEntry (RvObject *array)
{
  RvValue *val;

  if (!array || (array && !OBJECT_IS_ARRAY (array)))
    return NULL_VALUE;

  val = rc_getfieldvalue (array, "value", "[");
  if (!val)
    return NULL_VALUE;

  return (RvValue *) GET_STACK_VALUE (ADDR, val);
}

/**
    ENTRY  RvValue ü ̷ 迭̶  ϰ, ش IDX 
   ´ Ͱ ȯϰ ˴ϴ.

   @param entry RvValue ü 
   @param idx   迭 index ȣ
   @return      ش  
 */
RvValue *
RvValueEntryItem (RvValue *entry, int idx)
{
  return entry + idx;
}

/**
   迭 ü ARRAY   TOTAL  ũ⸦ մϴ.  ̿  
   迭 ü ʵ LEN  ǰ ˴ϴ.

   @param array 迭 ü
   @param total 迭  
   @return       ߻ -1,  0  ȯմϴ.
 */
int
RvArrayResize (RvObject *array, int total)
{
  int diff, len = RvArrayLength (array);

  diff = total - len;
  if (diff <= 0)
    return 0;

  if (rc_expandarray (array, diff))
    return -1;

  if (RvSetField (array, "len", "I", total))
    return -1;

  return 0;
}

/**
   迭 ARRAY  Ÿ TYPE  ׸ Ӱ ߰  ˴ϴ.

   @param array ArraryRef 
   @param type  Ӱ ߰Ǵ ׸ Ÿ ڿ
   @return       0, н -1  ȯմϴ.
 */
int
RvArrayPush (RvObject *array, const char *type, ...)
{
  int arrlen;
  RvValue *entry;
  va_list ap;

  if (!array || !type)
    return -1;

  va_start (ap, type);

  if (rc_expandarray (array, 1))
    {
      va_end (ap);
      return -1;
    }

  if (rc_getinfo_array (array, &entry, &arrlen))
    {
      va_end (ap);
      return -1;
    }

  entry = entry + arrlen;

  switch (type[0])
    {
    case 'B':
      SET_STACK_VALUE (entry, BOOLEAN, (unsigned char) va_arg (ap, int));
      break;
    case 'C':
      SET_STACK_VALUE (entry, CHAR, (char) va_arg (ap, int));
      break;
    case 'H':
      SET_STACK_VALUE (entry, SHORT, (short) va_arg (ap, int));
      break;
    case 'I':
      SET_STACK_VALUE (entry, INT, (int) va_arg (ap, int));
      break;
    case 'F':
      SET_STACK_VALUE (entry, FLOAT, (float) va_arg (ap, double));
      break;
    case 'D':
      SET_STACK_VALUE (entry, DOUBLE, (double) va_arg (ap, double));
      break;
    case 'T':
      SET_STACK_VALUE (entry, OBJREF, (void *) va_arg (ap, RvObject *));
      break;
    case 'S':
      SET_STACK_VALUE (entry, STRINGREF, (void *) va_arg (ap, RvObject *));
      break;
    case '[':
      SET_STACK_VALUE (entry, ARRAYREF, (void *) va_arg (ap, RvObject *));
      break;
 
    default:
      va_end (ap);
      return -1;
    }

  if (RvSetField (array, "len", "I", arrlen + 1))
    return -1;

  return 0;
}

/**
   迭 ARRAY    ִ entry  մϴ.

   @param array         ArrayRef 
   @param poped_value    Լ  poped Ǵ RvValue ü  
                         ݴϴ.
   @return       0, н -1  ȯմϴ.
 */
int
RvArrayPop (RvObject *array, RvValue **poped_value)
{
  int len;

  len = RvArrayLength (array);
  if (len < 0)
    return -1;

  if (poped_value)
    *poped_value = RvValueCopy (array, RvArrayGetItem (array, len - 1));

  if (RvSetField (array, "len", "I", len - 1))
    return -1;  

  return 0;
}

static RvObject *
list_slice (RvObject *a, int ilow, int ihigh)
{
  RvValue *entry, *src, *dest;
  RvObject *np;
  int i, len, arrlen;

  if (rc_getinfo_array (a, &entry, &arrlen))
    return NULL_OBJECT;

  if (ilow < 0)
    ilow = 0;
  else if (ilow > arrlen)
    ilow = arrlen;

  if (ihigh < ilow)
    ihigh = ilow;
  else if (ihigh > arrlen)
    ihigh = arrlen;

  len = ihigh - ilow;
  np = (RvObject *) RvArrayNew (a, len);
  if (np == NULL)
    return NULL_OBJECT;
  
  src = entry + ilow;
  dest = RvArrayGetEntry (np);
  for (i = 0; i < len; i++)
    {
      RvValue v = src[i];
      dest[i] = v;
    }

  return (RvObject *) np;
}

/* a[ilow:ihigh] = v if v != NULL.
 * del a[ilow:ihigh] if v == NULL.
 *
 * Special speed gimmick:  when v is NULL and ihigh - ilow <= 8, it's
 * guaranteed the call cannot fail.
 */
static int
list_ass_slice (RvObject *a, int ilow, int ihigh, RvObject *v)
{
  int n; /* # of elements in replacement list */
  int norig; /* # of elements in list getting replaced */
  int d; /* Change in size */
  int result = -1;	/* guilty until proved innocent */
  int k;
  RvValue *item, *vitem = NULL_VALUE;

#define b       ((RvObject *) v)

  if (v == NULL_OBJECT)
    n = 0;
  else
    {
      if (a == b)
        {
          /* Special case "a[i:j] = a" -- copy b first */
          v = list_slice (b, 0, RvArrayLength (b));
          if (v == NULL_OBJECT)
            return result;

          return list_ass_slice (a, ilow, ihigh, v);
        }

      n = RvArrayLength (v);
      vitem = RvArrayGetEntry (v);
    }

  if (ilow < 0)
    ilow = 0;
  else if (ilow > RvArrayLength (a))
    ilow = RvArrayLength (a);

  if (ihigh < ilow)
    ihigh = ilow;
  else if (ihigh > RvArrayLength (a))
    ihigh = RvArrayLength (a);

  norig = ihigh - ilow;
  d = n - norig;
  if (RvArrayLength (a) + d == 0)
    {
      if (RvSetField (a, "len", "I", 0))
        return -1;
      
      return 0;
    }

  item = RvArrayGetEntry (a);

  if (d < 0) 
    { /* Delete -d items */
      memmove (&item[ihigh+d], &item[ihigh], (RvArrayLength (a) - ihigh) * sizeof(RvValue));
      
      if (RvArrayResize (a, RvArrayLength (a) + d))
        return -1;

      item = RvArrayGetEntry (a);
    }
  else if (d > 0)
    { /* Insert d items */
      k = RvArrayLength (a);
      if (RvArrayResize (a, k + d))
        return -1;

      item = RvArrayGetEntry (a);
      memmove (&item[ihigh+d], &item[ihigh], (k - ihigh) * sizeof(RvValue));
    }

  if (vitem)
    {
      for (k = 0; k < n; k++, ilow++)
        {
          RvValue w = vitem[k];
          item[ilow] = w;
        }
    }
  
#undef b  
  return 0;
}

/**
   迭 ü ARRAY  ε I   POP ϰ  POP   POPED_VALUE
    ϰ ˴ϴ.   迭 ߰ κ POP  , ڵ  
   پ ˴ϴ.

   @param array 迭 ü
   @param i     POP  ׸ ε
   @param poped_value POP  ׸ ִ 
   @return       ߻ -1,  0  ȯմϴ.
 */
int
RvArrayPopWithIndex (RvObject *array, int i, RvValue **poped_value)
{
  int arrlen;
  RvValue *entry;

  if (rc_getinfo_array (array, &entry, &arrlen))
    return -1;

  if (i < 0)
    i += arrlen;
  if (i < 0 || i >= arrlen) 
    {
      RvLogError (array, ERRLOG_ERR, ERRLOG_MARK, "pop index out of range");
      return -1;
    }

  if (poped_value)
    *poped_value = RvValueCopy (array, entry + i);

  if (i == arrlen - 1)
    {
      if (RvArrayResize (array, arrlen - 1))
        return -1;

      return 0;
    }

  return list_ass_slice (array, i, i+1, NULL_OBJECT);
}

/**
   ڿ STR ̰ ̰ LEN  StringRef  ȯѴ.
 */
RvObject *
RvStrNew (RvObject *self, char *str, size_t len)
{
  return NewRvStringObject (OBJECTL_ROVM (self), OBJECT_TICKET (self),
                            str, len);
}

/**
   ڿ ߶ ׿  StringRef  ȯѴ.
 */
RvObject *
RvStrSubstr (RvObject *self, char *str, int offset, size_t len)
{
  return NewRvStringObject (OBJECTL_ROVM (self), OBJECT_TICKET (self),
                            str + offset, len);
}

/**
   Return  TYPE   մϴ.
 */
int
RvSetReturn (RvObject *self, RvValue *ret, const char *type, ...)
{
  va_list ap;
  void *ptr;

  if (!self || !ret || !type)
    return -1;

  va_start (ap, type);

  switch (type[0])
    {
    case 'V':
      SET_STACK_TYPE (ret, STACK_TYPE_VOID);
      break;
    case 'B':
      SET_STACK_VALUE (ret, BOOLEAN, (unsigned char) va_arg (ap, int));
      break;
    case 'C':
      SET_STACK_VALUE (ret, CHAR, (char) va_arg (ap, int));
      break;
    case 'H':
      SET_STACK_VALUE (ret, SHORT, (short) va_arg (ap, int));
      break;
    case 'I':
      SET_STACK_VALUE (ret, INT, (int) va_arg (ap, int));
      break;
    case 'F':
      SET_STACK_VALUE (ret, FLOAT, (float) va_arg (ap, double));
      break;
    case 'D':
      SET_STACK_VALUE (ret, DOUBLE, (double) va_arg (ap, double));
      break;
    case 'T':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (ret, OBJREF, ptr);
      break;
    case 'S':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (ret, STRINGREF, ptr);
      break;
    case '[':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (ret, ARRAYREF, ptr);
      break;
    default:
      break;
    }

  va_end (ap);
  return 0;

 error:
  va_end (ap);
  return -1;
}

/**
   ڿ ü STR  ̸ Ͽ ȯմϴ.

   @param str   ڿ ü
   @return      ڿ 
 */
int
RvStrSize (RvObject *str)
{
  int len;

  if (RvGetField (str, "len", "I", &len))
    return -1;
  
  return len;
}

/**
   StringRef  ڿ ̸  PTR  LEN   ȯմϴ.

   @param str   StringRef 
   @param ptr   ڿ  ڿ 
   @param len   ڿ ̸  int 
   @return       0, н -1  ȯմϴ.
 */
int
RvStrGetInfo (RvObject *str, char **ptr, int *len)
{
  RvValue *value;

  /* StringRef  ArrayRef  value  Ưϰ   Ѵ.  */
  value = RvGetFieldValue (str, "value", "[");
  if (!value)
    return -1;
  *ptr = RvValue2Str (str, value);

  if (RvGetField (str, "len", "I", len))
    return -1;

  return 0;
}

/**
   迭 ARRAY   ̸ Ͽ ȯմϴ.

   @param array ArrayRef 
   @return       迭 ũ
*/
int
RvArrayLength (RvObject *array)
{
  int len;

  if (RvGetField (array, "len", "I", &len))
    return -1;
  
  return len;
}

/**
   迭 ü ARRAY  index IDX  ϴ  ü ȯմϴ.

   @param array 迭 ü
   @param idx   ã index ȣ
   @return      ã  ü ȯ
 */
RvValue *
RvArrayGetItem (RvObject *array, size_t idx)
{
  int len;
  RvValue *value, *entry;

  len = RvArrayLength (array);
  if (idx >= len)
    return NULL_VALUE;

  /* StringRef  ArrayRef  value  Ưϰ   Ѵ.  */
  value = RvGetFieldValue (array, "value", "[");
  if (!value)
    return NULL_VALUE;

  if (!STACK_IS_ARRAYVALUE (value))
    return NULL_VALUE;

  entry = (RvValue *) GET_STACK_VALUE (ADDR, value);

  return &entry[idx];
}

/**
   迭 ü ARRAY    ϴ ׸   ȯմϴ.

   @param array 迭 ü
   @return        ϴ  
 */
RvValue *
RvArrayLastValue (RvObject *array)
{
  int len;

  len = RvArrayLength (array);
  if (len <= 0)
    return NULL_VALUE;

  return RvArrayGetItem (array, len - 1);
}

/**
   迭 ü ARRAY  index IDX   VAL  ϴ ԼԴϴ.

   @param array 迭 ü
   @param idx     index ȣ
   @param val    
   @return       ߻ -1,  0
 */
int
RvArraySetItem (RvObject *array, size_t idx, RvValue *val)
{
  RvValue *entry;

  entry = RvArrayGetItem (array, idx);
  if (!entry)
    return -1;

  *entry = *val;

  return 0;
}

/* Reverse a slice of a list in place, from lo up to (exclusive) hi. */
static void
reverse_slice (RvValue *lo, RvValue *hi)
{
  assert (lo && hi);

  --hi;
  while (lo < hi)
    {
      RvValue t = *lo;
      *lo = *hi;
      *hi = t;
      ++lo;
      --hi;
    }
}

/**
   迭 ü ARRAY   ݴ ϴ.

   @param array 迭 ü
   @return       ߻ -1,  0  ȯ
 */
int
RvArrayReverse (RvObject *array)
{
  int arrlen;
  RvValue *entry;

  if (rc_getinfo_array (array, &entry, &arrlen))
    return -1;

  reverse_slice (entry, entry + arrlen);
  
  return 0;
}

/**
   迭 ü ARRAY  index WHERE   FMT  ְ ˴ϴ.  FMT  ٷ
   ִ  ƴ϶ ...  ° ؼ  ˸°  ˴ϴ.

   @param array 迭 ü
   @param where   ΰ ϴ index ȣ
   @param fmt   ڿ 
   @return       ߰  ߰ߵǾ  -1, ׷  
                0  ȯϰ ˴ϴ.
 */
int
RvArrayInsert (RvObject *array, int where, const char *fmt, ...)
{
  int i, n, unused;
  void *ptr;
  va_list ap;
  RvValue *entry, *e;

  n = RvArrayLength (array);
  if (n == INT_MAX)
    {
      RvLogError (array, ERRLOG_ERR, ERRLOG_MARK, "cannot add more objects to list");
      return -1;
    }

  if (RvArrayResize (array, n + 1))
    return -1;

  /* RvArrayResize () Լ Ǹ  value  ּҰ ޸Ƿ
     ؾ Ѵ.  */
  if (rc_getinfo_array (array, &entry, &unused))
    return -1;

  if (where < 0)
    {
      where += n;
      if (where < 0)
        where = 0;
    }
  if (where > n)
    where = n;
  
  for (i = n; --i >= where; )
    entry[i+1] = entry[i];

  va_start (ap, fmt);

  e = &(entry[where]);
  switch (*fmt)
    {
    case 'V':
      SET_STACK_TYPE (e, STACK_TYPE_VOID);
      break;
    case 'B':
      SET_STACK_VALUE (e, BOOLEAN, (unsigned char) va_arg (ap, int));
      break;
    case 'C':
      SET_STACK_VALUE (e, CHAR, (char) va_arg (ap, int));
      break;
    case 'H':
      SET_STACK_VALUE (e, SHORT, (short) va_arg (ap, int));
      break;
    case 'I':
      SET_STACK_VALUE (e, INT, (int) va_arg (ap, int));
      break;
    case 'F':
      SET_STACK_VALUE (e, FLOAT, (float) va_arg (ap, double));
      break;
    case 'D':
      SET_STACK_VALUE (e, DOUBLE, (double) va_arg (ap, double));
      break;
    case 'T':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (e, OBJREF, ptr);
      break;
    case 'S':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (e, STRINGREF, ptr);
      break;
    case '[':
      ptr = (void *) va_arg (ap, RvObject *);
      if (!ptr)
        goto error;
      SET_STACK_VALUE (e, ARRAYREF, ptr);
      break;
    default:
      goto error;
    }

  va_end (ap);
  return 0;
 error:
  va_end (ap);
  return -1;
}

/**
    VALUE  StringRef  ȯմϴ.

   @param value   ü
   @return       true   1 , ׷ ʴٸ 0  ȯմϴ.
 */
int
RvValueIsStringRef (RvValue *value)
{
  if (STACK_TYPE (value) == STACK_TYPE_STRINGREF)
    return TRUE;

  return FALSE;
}

/**
    VALUE  ArrayRef   TRUE  ȯϰ ׷   FALSE 
   ȯմϴ.
 */
int
RvValueIsArrayRef (RvValue *value)
{
  if (STACK_TYPE (value) == STACK_TYPE_ARRAYREF)
    return TRUE;

  return FALSE;
}

/**
    VALUE  StringRef  Ͽ ȯմϴ.

   @return       VALUE  StringRef  , ׿ شϴ ͸ ȯϰ
                ׷ ʴٸ NULL_OBJECT  ȯմϴ.
 */
RvObject *
RvValueStringRef (RvValue *value)
{
  if (!RvValueIsStringRef (value))
    return NULL_OBJECT;

  return GET_STACK_V_STRINGREF (value);
}

/**
    VALUE  ArrayRef  ,  Ǿ ִ ArrayRef ͸ 
   ȯմϴ.

   @param value    RvValue ü
   @return       ArrayRef  , ش ArrayRef ͸, ׷
                  NULL_OBJECT  ȯ˴ϴ.
 */
RvObject *
RvValueArrayRef (RvValue *value)
{
  if (!RvValueIsArrayRef (value))
    return NULL_OBJECT;

  return GET_STACK_V_ARRAYREF (value);
}

/**
   ڿ ü STR    Ǿ ִ  value   
    ȯմϴ.

   @param str   ڿ ü
   @return       char * .
 */
char *
RvStrGetPointer (RvObject *str)
{
  RvValue *value;

  /* StringRef  ArrayRef  value  Ưϰ   Ѵ.  */
  value = RvGetFieldValue (str, "value", "[");
  if (!value)
    return NULL;
  
  if (!STACK_IS_STRINGVALUE (value))
    return NULL;

  return (char *) GET_STACK_VALUE (ADDR, value);
}

/**
   GC ޸  Ҵ  Ǵ ԼԴϴ.  Ʒ ҴǴ ޸𸮴
   ڵ Delayed Stack Slot  ϵǰ ˴ϴ.

   @param self  ROVM ü TICKET ü ͸   Ǵ
                ü
   @param size  Ҵ ޸ ũ
   @return      Ҵ ޸ 
 */
void *
RvMemMalloc (RvObject *self, size_t size)
{
  return gc_alloc (OBJECTL_ROVM (self), OBJECT_TICKET (self), size);
}

/**
    VALUE  纻    ȯϰ ˴ϴ.   Լ ϴ 
   Ȯ  ۼ GC ޸ ͸ û ϰ    GC  
    , ش ü    ϱ ؼ մϴ.

   Ʒ RvMemMalloc ()  ҴǴ ޸𸮴 Delayed Stack Slot  ڵ
   ϵǰ ˴ϴ.

   @param self     Ǵ Object ü
   @param value  
   @return       
 */
RvValue *
RvValueCopy (RvObject *self, RvValue *value)
{
  RvValue *new;

  new = (RvValue *) RvMemMalloc (self, sizeof (RvValue));
  if (!new)
    return NULL_VALUE;

  *new = *value;

  return new;
}

extern const char *rovm_stack_name[MAX_STACK_TYPE];

/**
   VAL  Ÿ ̸ ȯմϴ.

   @param val   ˰ ϴ 
   @return      ش Ÿ ̸
 */
const char *
RvGetTypeName (RvValue *val)
{
  return rovm_stack_name[STACK_TYPE (val)];
}

/**
      RET   VALUE  ϴµ Ǵ Լν ȯ
   RvValue ״ ϰ   ˴ϴ.

   @param ret   Ȯ  ޽ ȣ 3 ° ڰ Ǵ Դϴ.
   @param value  
   @return       ߻ -1, ׷   0  ȯ
 */
int
RvSetReturnValue (RvValue *ret, RvValue *value)
{
  if (!ret || !value)
    return -1;

  *ret = *value;

  return 0;
}

/**
      O1  O2  Ͽ  ٸ 1  ȯϰ ׷
   ʴٸ 0  ȯմϴ.   ߰  ߻Ͽٸ -1  ȯϰ
   ˴ϴ.   迭  ,  ׸  ȣν 񱳸 ϰ
   Ǹ, ڿ ü , ڿ    ϰ ˴ϴ. 
    ObjectRef   ּҰ  񱳸 ϰ ˴ϴ.   
     ش  ν ̷ ˴ϴ.

   @param o1      1 
   @param o2      2 
   @return       ٸ 1  ȯϰ ׷ ʴٸ 0  ȯմϴ.
                 ߰  ߻Ͽٸ -1  ȯ
 */
int
RvValueRichCompare (RvValue *o1, RvValue *o2)
{
  if (!o1 || !o2)
    return -1;

  if (STACK_TYPE (o1) != STACK_TYPE (o2))
    return FALSE;

#define RICHCOMPARECASE(TYPE)                 \
  case STACK_TYPE_##TYPE:                       \
    {                                           \
      if (GET_STACK_VALUE (TYPE, o1) != GET_STACK_VALUE (TYPE, o2))     \
        return FALSE;                                                   \
      break;                                                            \
    }
  
  switch (STACK_TYPE (o1))
    {
      RICHCOMPARECASE (BOOLEAN);
      RICHCOMPARECASE (CHAR);
      RICHCOMPARECASE (SHORT);
      RICHCOMPARECASE (INT);
      RICHCOMPARECASE (FLOAT);
      RICHCOMPARECASE (DOUBLE);
      RICHCOMPARECASE (OBJREF);
    case STACK_TYPE_STRINGREF:
      {
        int str1len, str2len;
        char *str1, *str2;
        RvObject *s1, *s2;

        s1 = GET_STACK_VALUE (STRINGREF, o1);
        s2 = GET_STACK_VALUE (STRINGREF, o2);
        if (!s1 || !s2)
          return -1;
        if (RvStrGetInfo (s1, &str1, &str1len))
          return -1;
        if (RvStrGetInfo (s2, &str2, &str2len))
          return -1;

        if (str1len != str2len)
          return FALSE;
        if (memcmp (str1, str2, str1len))
          return FALSE;
        break;
      }
    case STACK_TYPE_ARRAYREF:
      {
        int i, e1len, e2len;
        RvValue *e1, *e2;

        e1len = RvArrayLength (GET_STACK_VALUE (ARRAYREF, o1));
        e2len = RvArrayLength (GET_STACK_VALUE (ARRAYREF, o2));
        if (e1len != e2len)
          return FALSE;

        e1 = RvArrayGetEntry (GET_STACK_VALUE (ARRAYREF, o1));
        e2 = RvArrayGetEntry (GET_STACK_VALUE (ARRAYREF, o2));
        if (!e1 || !e2)
          return -1;

        for (i=0; i<e1len; i++)
          {
            int ret = RvValueRichCompare (e1 + i, e2 + i);
            if (ret == FALSE)
              return FALSE;
            else if (ret == -1)
              return -1;
          }
        break;
      }
    default:
      return -1;
    }

  return TRUE;
}

/**
   迭 ü ARRAY  LO  HI  ó  ˴ϴ.   V  NULL
     del ARRAY[LO:HI]   ǹ̰ Ǹ, ׷   
   ARRAY[LO:HI] = v  ǹ̰ ˴ϴ.

   ̽ ڵ尡 ſ  Ǿϴ. ^^   ٽ Ӹ ؼ
     󿡼  Ⱦմϴ.

   @param array 迭 ü
   @param lo    迭 lowest index
   @param hi    迭 highest index
   @param v      , ׿ شϴ 
   @return          ߻߾ٸ -1  ׷ ʴٸ 0  
                ȯմϴ.
 */
int
RvArraySliceAdvanced (RvObject *array, int lo, int hi, RvObject *v)
{
  return list_ass_slice (array, lo, hi, v);
}

/**
   Array  Լν SELF  B   ArrayRef üԴϴ.   Լ
   B   ִ ׸  SELF  ߰ϴ  մϴ.

   @param self  迭 ü 1 
   @param b     迭 ü 2 
   @return      迭 ü 2  迭 ü 1   Ե 
                µ,   ߻Ͽ  -1  ׷
                  0  ȯմϴ.
 */
int
RvArrayExtend (self, b)
     RvObject *self, *b;
{
  int i, m, n, mn;
  RvValue *src, *dest;

  if (self == b)
    {
      n = RvArrayLength (b);
      if (n == 0)
        return 0;
      m = RvArrayLength (self);

      if (RvArrayResize (self, m + n))
        return -1;

      src = RvArrayGetEntry (b);
      dest = RvArrayGetEntry (self) + m;
      for (i = 0; i < n; i++) 
        {
          RvValue o = src[i];
          dest[i] = o;
        }
      return 0;
    }

  src = RvArrayGetEntry (b);
  dest = RvArrayGetEntry (self);

  n = RvArrayLength (b);
  if (n < 0)
    return -1;

  m = RvArrayLength (self);
  mn = m + n;
  if (mn >= m)
    {
      /* Make room. */
      if (RvArrayResize (self, mn))
        return -1;
    }

  for (i = 0; i < n; i++)
    {
      if (RvArraySetItem (self, m + i, src + i))
        return -1;
    }

  return 0;
}

/**
   ROVM Request  ߻ϴ  ޼ ϱ ؼ ϴ logger Լ.

   @param self	SELF ü
   @param level  α 赵 
   @param file	޼ ߻  ̸
   @param line	޼ ߻   ȣ
   @param fmt	vprintf  
 */
int
RvLogError (RvObject *self,
            int level,
            const char *file, 
            int line, 
            const char *fmt, ...)
{
  va_list args;

  va_start (args, fmt);
  if (request_add_error_core (OBJECT_REQUEST (self), level, file, line, fmt, args))
    {
      va_end (args);
      return -1;
    }

  va_end (args);
  return 0;
}

RvObject *
RvObjectNew (RvObject *self, 
             int iptype,
             char *hostname, 
             unsigned short port,
             char *path)
{
  RvClass *cls;
  RvObject *obj;

  cls = rc_getclassbyhost (OBJECT_REQUEST (self), iptype, hostname, port, path);
  if (cls == NULL_CLASS)
    {
      request_add_error (OBJECT_REQUEST (self), ERRLOG_ERR, ERRLOG_MARK, "Can't find class.");
      return NULL_OBJECT;
    }
  else if (cls == REMOTE_CLASS)
    {
      obj = NewRemoteRvObject (OBJECT_REQUEST (self), iptype, hostname, port, path);
      if (!obj)
        {
          request_add_error (OBJECT_REQUEST (self), ERRLOG_ERR, ERRLOG_MARK,
                             "NewRemoteRvObject is under construction.");
          return NULL_OBJECT;
        }
    }
  else
    obj = NewRvObject (OBJECTL_ROVM (self), OBJECT_TICKET (self), cls);
  
  return obj;
}
