/* ticket.c - Ticket ü óϴ Դϴ.
   Copyright (C) 2006 Weongyo Jeong (weongyo@gmail.com)

This file is part of ROVM.

ROVM is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

ROVM is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>

#include "types.h"
#include "mpool.h"
#include "tktree.h"
#include "clstree.h"
#include "sha1.h"
#include "listen.h"
#include "rovm.h"

#include "thread.h"
#include "thread_mutex.h"
#include "thread_cond.h"

#include "mpm_worker_fdqueue.h"
#include "mpm_worker_pod.h"
#include "mpm_worker.h"

#include "utils.h"
#include "connection.h"
#include "request.h"

#include "ticket.h"
#include "log.h"

void *
safe_tktree_lookup (r, sp, key)
     struct rovm *r;
     tktree_t *sp;
     char *key;
{
  void *ret;
  rc_thread_mutex_lock (ROVM_TKMUTEX (r));
  ret = tktree_lookup (sp, key);
  rc_thread_mutex_unlock (ROVM_TKMUTEX (r));
  return ret;
}

void
safe_tktree_remove (r, sp, key)
     struct rovm *r;
     tktree_t *sp;
     char *key;
{
  rc_thread_mutex_lock (ROVM_TKMUTEX (r));
  tktree_remove (sp, key);
  rc_thread_mutex_unlock (ROVM_TKMUTEX (r));
}

int
safe_tktree_insert (r, sp, key, value)
     struct rovm *r;
     tktree_t *sp;
     char *key;
     void *value;
{
  int ret;
  rc_thread_mutex_lock (ROVM_TKMUTEX (r));
  ret = tktree_insert (sp, key, value);
  rc_thread_mutex_unlock (ROVM_TKMUTEX (r));
  return ret;
}

int
safe_tktree_walking (r, sp, fn, data)
     struct rovm *r;
     tktree_t *sp;
     tktree_foreach_fn fn;
     void *data;
{
  int ret;
  rc_thread_mutex_lock (ROVM_TKMUTEX (r));
  ret = tktree_walking (sp, fn, data);
  rc_thread_mutex_unlock (ROVM_TKMUTEX (r));
  return ret;
}

/**
   Ticket Delay Slot   ʱȭմϴ.

   @param tk    Ticket 
 */
void
ticket_clear_delay_slot (tk)
     struct ticket *tk;
{
  if (!tk)
    return;

  TICKET_DELAYED_SLOT_NELM (tk) = 0;
}

/**
   struct ticket ü  TK   ϱ   ʿ伺
   ִ ׸ ؼ   ֵ Ѵ.

   @param tk     TK 
   @return       Ϸ 0  ׷   -1  ȯѴ.
 */
int
ticket_pre_config (tk)
     struct ticket *tk;
{
  /*  Ticket   Ÿϴ.  ° `ϴ'  , ֱ 
     ϰ  GC  ʽϴ.  */
  TICKET_STATUS (tk) = TICKET_IS_WORKING;

  /*  ֱ  Ticket  ð Ʈմϴ.    Ticket  GC 
     ̷µ,   Ĩϴ.  */
  TICKET_TIME_UPDATE (tk) = rc_time_now ();

  /*
    struct ticket ü   RET  ʱȭѴ.  ̷ ʱȭ 
     ׻ request `OPCODE'  óϰ  , ׿  return  û
    client   ֱ  return   ʿ䰡  ̴.
  */
  STACK_TYPE (TICKET_RET (tk)) = STACK_TYPE_VOID;

  return 0;
}

/**
   Ticket ü    ȣǴ ԼԴϴ.
 */
int
ticket_post_config (tk)
     struct ticket *tk;
{
  /*  Ticket  ޽ Ÿϴ.  ° `޽'  ,  ֱ 
     ð Ҿ  Ticket  GC Ǿ    ǹմϴ.  */
  TICKET_STATUS (tk) = TICKET_IS_RELAX;

  /*  ֱ  Ticket  ð Ʈմϴ.    Ticket  GC 
     ̷µ,   Ĩϴ.

       ѹ  update  ݴϴ.  Լ  ſ  ɸ ,
     ticket_pre_config ()    ð 귶 ɼ , ̷ 
     Ticket ü GC Ǿ  һ縦  ؼ Դϴ.
  */
  TICKET_TIME_UPDATE (tk) = rc_time_now ();

  return 0;
}

/**
   Ŭ method  Ǳ , ѹ ȣǴ Լν, struct ticket ü
    Ҹ  Լ ȣ⿡ °  ִ Ȱ մϴ.   Ǿ
   δ Ʒ   ֽϴ.

     - Local 

   @param tk    Ticket ü
   @param m     ȣ Ǵ ޽ 
   @param argc  ޽ argument count.
   @param argv  ޽ argument value.
   @return          ߻Ͽ  -1 , ׷ ʴٸ
                0  ȯѴ.
 */
int
ticket_beforecall (tk, m, argc, argv)
     struct ticket *tk;
     RvMethod *m;
     u2 argc, *argv;
{
  if (ticket_pre_config (tk))
    return -1;

  TICKET_LOCAL (tk) = TICKET_SP (tk) - ((int) (argc - 1));
  TICKET_MAXLOCAL (tk) = TICKET_SP (tk) + 1;

  return 0;
}

/**
   ü ticket  Ѵ.

   @return Ӱ  ticket  ȯѴ.
 */
struct ticket *
create_ticket (r)
     struct rovm *r;
{
  struct ticket *t;

  t = (struct ticket *) rc_calloc (1, sizeof (struct ticket));
  if (!t)
    return NULL;

  TICKET_TIME_INIT (t) = rc_time_now ();

  /*   ʱȭմϴ.  */
  TICKET_MAX_STACK_SLOT (t) = CONF_DEFAULT_STACK_SLOT (ROVM_CONF (r));
  TICKET_STACK (t) = (rovm_stack_t *) rc_calloc (1, sizeof (rovm_stack_t) * TICKET_MAX_STACK_SLOT (t));
  if (!TICKET_STACK (t))
    return NULL;

  /* Stack   Local  ͸ ʱȭմϴ.  */
  TICKET_SP (t) = &TICKET_STACK_IDX (t, -1);
  TICKET_LOCAL (t) = TICKET_SP (t);
  TICKET_MAXLOCAL (t) = TICKET_SP (t);
  TICKET_MAX_SP (t) = &TICKET_STACK_IDX (t, TICKET_MAX_STACK_SLOT (t));
  TICKET_MIN_SP (t) = &TICKET_STACK_IDX (t, -1);

  /* Delayed Slot  ʱȭѴ.  */
  TICKET_DELAYED_SLOT (t) = (void **) rc_calloc (DEFAULT_DELAYED_SLOT, sizeof (void *));
  TICKET_DELAYED_SLOT_MAX (t) = DEFAULT_DELAYED_SLOT;
  TICKET_DELAYED_SLOT_NELM (t) = 0;

  if (ticket_pre_config (t))
    return NULL;

  /* ticket_pre_config () Լ ticket  ° working  ϱ
     , ̸ relax   ݴϴ.  */
  TICKET_STATUS (t) = TICKET_IS_RELAX;

  return t;
}

/**
   ü ticket  ϴ  ڿ free Ѵ.

   @param tk    ıϰ ϴ tk ü 
 */
void
destroy_ticket (tk)
     struct ticket *tk;
{
  if (!tk)
    return;

  rc_free (TICKET_DELAYED_SLOT (tk));
  rc_free (TICKET_STACK (tk));
  rc_free (tk);
}

/**
   Ticket ID  Ticket   Ѵ.

   @param r     ROVM ü
   @param tid    Ticket ID.  Ticket ID   Ticket  Բ
                ŵȴ.
   @note        destroy_ticket () Լ ܼ `s'   ̹Ƿ 
                 ǰ ʿϴ.
 */
int
destroy_tickets (r, tid)
     struct rovm *r;
     char *tid;
{
  safe_tktree_remove (r, ROVM_TKTREE (r), tid);

  return 0;
}

/**
   TICKETID  Ӱ  , װ ȯѴ.

   @param r     Request ü
   @return      Ӱ  ticketid  ȯѴ.
 */
char *
create_ticketid (r)
     request_rec *r;
{
  char *tbuf;

  tbuf = (char *) rc_malloc (TICKETID_SIZE);
  if (!tbuf)
    return NULL;

  if (gen_unique_ticketid (tbuf))
    {
      rc_free (tbuf);
      return NULL;
    }

  return tbuf;
}

/**
   Unique Ticket ID  Ͽ Ѵ.  ̸ ϱ  ϴ δ
   PID   ð ̿ϸ, SHA1  ̿Ͽ unique key  Ѵ.

   @param r	ä ü.
   @param tbuf	߱޵ ticket ۰   ϴ .    ũ
                ݵ TICKET_SIZE ũ⿩ Ѵ.   
 */
int
gen_unique_ticketid (tbuf)
     char *tbuf;
{
  pid_t pid;
  unsigned short counter;
#define MAX_TICKETSTR	256
  char tmpbuf[MAX_TICKETSTR];

  /*  Unique Ticket   ؼ PID  COUNTER  ϰ ִ.
         ϱ ؼ Ʒ  ׸ ߰   ̴.

       - Channel  open  ð.  (Request   ð)
       - Client IP ּҰ  */
  pid = getpid ();
  counter = (unsigned short) (rc_time_usec (rc_time_now ()) / 10);

  /* PID  COUNTER   ڿ ̰ MAX_TICKETSTR ũ⸦  ʴ´ٰ 
      Ѵ.  */
  snprintf (tmpbuf, MAX_TICKETSTR, "%d%d", pid, counter);

  rovm_sha1 ((const char *) tmpbuf, strlen (tmpbuf), tbuf);

  return 0;
}

/**
   ڰ û Ticket ID  ̿Ͽ ׿  ticket ü ´.

   @param r     Request ü
   @param tbuf  Ticket ID  ִ 
   @return      TKTREE  ã ticket ü.    ʴ´ٸ 
                NULL  ȯѴ.
 */
struct ticket *
lookup_ticket (r, tbuf)
     request_rec *r;
     char *tbuf;
{
  struct rovm *rovm = REQUEST_ROVM (r);

  return (struct ticket *) safe_tktree_lookup (rovm, ROVM_TKTREE (rovm), tbuf);
}

/**
    TBUF  ü ticket tree ü Ѵ.  ̷ ϵ TBUF  ߿ 
   ̿ شϴ request   , ش  ϰ ȴ.

   ,   Session   Ȱ Ѵٰ    ̴.

   @param r	ä ü.
   @param tbuf	߱޵ ticket ۰  ִ .   ticket  
                gen_unique_ticket () Լ   Ǿ.

   @see gen_unique_ticketid
 */
int
register_ticketid (r, tbuf)
     request_rec *r;
     char *tbuf;
{
  int ret = 0;
  struct rovm *rovm = REQUEST_ROVM (r);

  if (!safe_tktree_lookup (rovm, ROVM_TKTREE (rovm), tbuf))
    {
      struct ticket *tk = create_ticket (rovm);

      if (!tk)
        {
          request_add_error (r, ERRLOG_ERR, ERRLOG_MARK, "Failed to create ticket.");
          return -1;
        }

      /* Insert  Remove  Locking ī ʿմϴ. */
      if (safe_tktree_insert (rovm, ROVM_TKTREE (rovm), tbuf, (void *) tk))
        return -1;
    }

  return ret;
}

#define DEFAULT_TICKET_COLLECTOR_SLOT   1024
struct ticket_collector
{
  struct rovm *r;

  int cur_slot;
  int max_slot;
  tktree_node_t *slot;
};

/**
   Ticket Tree  walking   , ȣǴ callback ԼԴϴ.
    κп   Ticket Tree    ۾ ̷ ˴ϴ.

    ȯ -1  , walking  ߰    ϸ鼭 ڵ
   ؾ Ѵ.

   @param node  ش node    .
 */
int
rovm_ticket_walking_cb (tktree_node_t node, void *arg)
{
  struct ticket_collector *tc = (struct ticket_collector *) arg;
  struct ticket *tk = (struct ticket *) node->value;
  rc_time_t timec = rc_time_now ();

  if (TICKET_STATUS (tk) == TICKET_IS_WORKING)
    return 0;

  if (rc_time_sec (timec - TICKET_TIME_UPDATE (tk)) < DEFAULT_TICKET_EXPIRE_TIME)
    return 0;

  /* ⿡ ߴٴ  expire Ǿ ϴ Ticket  Ѵٴ ̾߱ ̴.  */

  /* Ƽ  ̻ ȿ  մϴ.  */
  TICKET_STATUS (tk) = TICKET_IS_ON_GCSLOT;

  if (tc->cur_slot >= tc->max_slot)
    {
      tc->max_slot += DEFAULT_TICKET_COLLECTOR_SLOT;
      tc->slot = (tktree_node_t *) rc_realloc (tc->slot, sizeof (tktree_node_t) * tc->max_slot);
    }

  /* ٷ Ŵ ϴϱ, tc slot  ׾Ƶд.  ٷ  , walking ߿ tree 
       ֽϴ.  */
  tc->slot[tc->cur_slot++] = node;

  return 0;
}

/**
   Ticket ü  Garbage Collection  ̷  ȣǴ Լν
    tree node  walking ϸ鼭  ð  Ticket ü
   մϴ.

   @param r     ROVM ü
   @return       0, н -1  ȯմϴ.
 */
int
ticket_collect (r)
     struct rovm *r;
{
  int i;
  struct ticket_collector tc;

  tc.r = r;
  tc.cur_slot = 0;
  tc.max_slot = DEFAULT_TICKET_COLLECTOR_SLOT;
  tc.slot = (tktree_node_t *) rc_malloc (sizeof (tktree_node_t) * tc.max_slot);

  /* üũ */
  safe_tktree_walking (r, ROVM_TKTREE (r), rovm_ticket_walking_cb, (void *) &tc);

  /*  */
  for (i=0; i<tc.cur_slot; i++)
    destroy_tickets (r, (char *) (tc.slot[i]->key));

  rc_free (tc.slot);

  return 0;
}
