#include "types.h"
#include "rovm.h"

#define COMMAND_TICKET          0x04
#define COMMAND_OK              0x05
#define COMMAND_ERROR           0x06
#define COMMAND_RETURN          0x07
#define COMMAND_GETSTACK        0x08

typedef struct
{
  char  *t_name;
  int   t_val;
} TRANS;

static const TRANS priorities[] =
  {
    {"emerg",   ROVMLOG_EMERG},
    {"alert",   ROVMLOG_ALERT},
    {"crit",    ROVMLOG_CRIT},
    {"error",   ROVMLOG_ERR},
    {"warn",    ROVMLOG_WARNING},
    {"notice",  ROVMLOG_NOTICE},
    {"info",    ROVMLOG_INFO},
    {"debug",   ROVMLOG_DEBUG},
    {NULL,      -1},
  };

static void
rovm_printreturn (rovm_t *r, rovmret_t *ret)
{
  if (!ret)
    {
      fprintf (stderr, "(nil)\n");
      return;
    }

  switch (ret->type)
    {
    case RETTYPE_VOID:
      fprintf (stderr, "(void)");
      break;
    case RETTYPE_ARRAYREF:
      {
        int i;
        fprintf (stderr, "(arrayref [\n");
        
        for (i=0; i<ret->arraylen; i++)
          {
            struct rovmarray *array = ret->v._array + i;
            switch (array->type)
              {
              case RETTYPE_STRINGREF:
                fprintf (stderr, "    (string \"%.*s\")\n", array->v._string->len, array->v._string->ptr);
                break;
              default:
                fprintf (stderr, "    (unknown)");
                break;
              }
          }
        }
      fprintf (stderr, "])");
      break;
    case RETTYPE_INT:
      fprintf (stderr, "(int %d)", ret->v._int);
      break;
    default:
      fprintf (stderr, "printreturn: Can't handle this return type\n");
      break;
    }

  fputc ('\n', stderr);
}

static void
rovm_printerror (rovm_t *r)
{
  struct rovmerror *error = r->errmsg;

  while (error)
    {
      fprintf (stderr, "[%s] %s\n", priorities[error->level].t_name, error->msg);

      error = error->next;
    }
}

static void
rovm_add_error (rovm_t *r, int level, const char *fmt, ...)
{
  int len;
  struct rovmerror *error;
  char errstr[8*1024];
  va_list args;

  va_start (args, fmt);
  len = vsnprintf (errstr, sizeof (errstr), fmt, args);
  va_end (args);

  error = (struct rovmerror *) apr_pcalloc (r->p, sizeof (struct rovmerror));
  error->level = level;
  error->msg = apr_pstrndup (r->p, errstr, len);
  error->next = r->errmsg;

  r->errmsg = error;
}

static rovmres_t *
rovm_process_response (rovm_t *r, rovmret_t *ret)
{
  int recv;
  rovmres_t *rs;
  apr_status_t rv;

  rs = (rovmres_t *) apr_pcalloc (r->p, sizeof (rovmres_t));

  /* ٷ Ʒ ڵ带 İƼ ڵ θ.  */
  recv = 1;
  rv = apr_socket_recv (r->sock, &rs->type, &recv);
  if (recv != 1)
    return NULL;

  switch ((int) rs->type)
    {
    case COMMAND_TICKET:
      {
        recv = TICKET_SIZE;
        rv = apr_socket_recv (r->sock, r->ticket, &recv);
        if (recv != TICKET_SIZE)
          return NULL;
        break;
      }
    case COMMAND_OK:
      break;
    case COMMAND_ERROR:
      {
        int i;
        uint16_t errcount;

        recv = sizeof (uint16_t);
        rv = apr_socket_recv (r->sock, (char *) &errcount, &recv);
        if (recv != sizeof (uint16_t))
          return NULL;
        errcount = ntohs (errcount);

        for (i=0; i<errcount; i++)
          {
            char *msg, level;
            unsigned short len;

            recv = sizeof (char);
            rv = apr_socket_recv (r->sock, (char *) &level, &recv);
            if (recv != sizeof (char))
              return NULL;

            recv = sizeof (unsigned short);
            rv = apr_socket_recv (r->sock, (char *) &len, &recv);
            if (recv != sizeof (unsigned short))
              return NULL;
            len = ntohs (len);

            msg = (char *) apr_pcalloc (r->p, len + 1);
            recv = len;
            rv = apr_socket_recv (r->sock, msg, &recv);
            if (recv != len)
              return NULL;
            msg[len] = '\0';

            rovm_add_error (r, level, msg);
          }
        return NULL;
      }
    case COMMAND_RETURN:
      {
        recv = sizeof (char);
        rv = apr_socket_recv (r->sock, (char *) &ret->type, &recv);
        if (recv != sizeof (char))
          return NULL;

        switch (ret->type)
          {
          case RETTYPE_VOID:
            break;
          case RETTYPE_ARRAYREF:
            {
              int i;

              ret->is_array = 1;

              recv = sizeof (size_t);
              rv = apr_socket_recv (r->sock, (char *) &ret->arraylen, &recv);
              if (recv != sizeof (size_t))
                return NULL;
              ret->arraylen = (size_t) ntohl (ret->arraylen);

              ret->v._array = (struct rovmarray *) apr_pcalloc (r->p, sizeof (struct rovmarray) * ret->arraylen);

              for (i=0; i<ret->arraylen; i++)
                {
                  struct rovmarray *array = ret->v._array + i;

                  recv = sizeof (char);
                  rv = apr_socket_recv (r->sock, (char *) &array->type, &recv);
                  if (recv != sizeof (char))
                    return NULL;

                  switch (array->type)
                    {
                    case RETTYPE_STRINGREF:
                      {
                        array->v._string = (struct rovmstr *) apr_pcalloc (r->p, sizeof (struct rovmstr));

                        recv = sizeof (size_t);
                        rv = apr_socket_recv (r->sock, (char *) &(array->v._string->len), &recv);
                        if (recv != sizeof (size_t))
                          return NULL;
                        array->v._string->len = (size_t) ntohl (array->v._string->len);
                        
                        array->v._string->ptr = (char *) apr_pcalloc (r->p, array->v._string->len);
                        recv = array->v._string->len;
                        rv = apr_socket_recv (r->sock, array->v._string->ptr, &recv);
                        if (recv != array->v._string->len)
                          return NULL;
                        break;
                      }
                    default:
                      fprintf (stderr, "Can't handle this arrayref type.\n");
                      break;
                    }
                }
              break;
            }
          case RETTYPE_INT:
            {
              recv = sizeof (int);
              rv = apr_socket_recv (r->sock, (char *) &(ret->v._int), &recv);
              if (recv != sizeof (int))
                return NULL;
              break;
            }
          default:
            fprintf (stderr, "Can't handle this return type\n");
            break;
          }
        break;
      }
    default:
      fprintf (stderr, "Can't handle this type\n");
    }

  return rs;
}

static int
rovm_connect (rovm_t *r)
{
  apr_status_t rv;
  apr_sockaddr_t *connect_addr = NULL;

  if (r->sock)
    {
      apr_socket_close (r->sock);
      r->sock = NULL;
    }

  /* do a DNS lookup for the destination host */
  rv = apr_sockaddr_info_get (&connect_addr, r->host, APR_UNSPEC, r->port, 0, r->p);
  if ((rv = apr_socket_create (&r->sock, connect_addr->family, SOCK_STREAM, 0, r->p)) != APR_SUCCESS)
    {
      fprintf (stderr, "error creating socket");
      return -1;
    }

  rv = apr_socket_connect (r->sock, connect_addr);
  if (rv != APR_SUCCESS)
    {
      fprintf (stderr, "attempt to connect to %pI failed - Firewall/NAT?\n", r->sock);
      return -1;
    }

  return 0;
}

static int
rovm_send_req (rovm_t *r)
{
  rovmres_t *rs;
  apr_status_t rv;
  int len = 1;

  rv = apr_socket_send (r->sock, "\x01", &len);
  if (len != 1)
    return -1;

  rs = rovm_process_response (r, NULL);
  if (rs->type != COMMAND_TICKET)
    return -1;

  return 0;
}

static int
rovm_send_reqend (rovm_t *r)
{
  rovmres_t *rs;
  apr_status_t rv;
  int len = 1;

  rv = apr_socket_send (r->sock, "\x02", &len);
  if (len != 1)
    return -1;
  len = TICKET_SIZE;
  rv = apr_socket_send (r->sock, r->ticket, &len);
  if (len != TICKET_SIZE)
    return -1;
    
  rs = rovm_process_response (r, NULL);
  if (rs->type != COMMAND_OK)
    return -1;

  return 0;
}

static rovmret_t *
rovm_send (rovm_t *r)
{
  rovmret_t *ret;
  rovmres_t *rs;
  apr_status_t rv;
  int len = 1;
  uint32_t val;

  if (rovm_connect (r))
    return NULL;

  rv = apr_socket_send (r->sock, "\x03", &len);
  if (len != 1)
    return NULL;
  len = TICKET_SIZE;
  rv = apr_socket_send (r->sock, r->ticket, &len);
  if (len != TICKET_SIZE)
    return NULL;
  len = 4;
  val = htonl (r->oplen);
  rv = apr_socket_send (r->sock, (char *) &val, &len);
  if (len != 4)
    return NULL;
  len = r->oplen;
  rv = apr_socket_send (r->sock, r->op, &len);
  if (len != r->oplen)
    return NULL;

  ret = (rovmret_t *) apr_pcalloc (r->p, sizeof (rovmret_t));

  rs = rovm_process_response (r, ret);
  if (!rs || (rs && rs->type != COMMAND_RETURN))
    return NULL;

  apr_socket_close (r->sock);
  r->sock = NULL;

  return ret;
}

static void
rovm_printstack (rovm_t *r)
{
  apr_status_t rv;
  int len = 1;

  if (rovm_connect (r))
    return;

  rv = apr_socket_send (r->sock, "\x08", &len);
  if (len != 1)
    return;
  len = TICKET_SIZE;
  rv = apr_socket_send (r->sock, r->ticket, &len);
  if (len != TICKET_SIZE)
    return;

  rovm_process_response (r, NULL);

  rovm_printerror (r);

  apr_socket_close (r->sock);
  r->sock = NULL;  
}

extern struct rovmops rovm_opcodes;

rovm_t *
rovm_req (const char *host, apr_port_t port)
{
  rovm_t *r;
  apr_pool_t *p;

  apr_initialize ();

  apr_pool_create (&p, NULL);
  apr_pool_tag (p, "rovm");

  r = (rovm_t *) apr_pcalloc (p, sizeof (rovm_t));
  r->p = p;
  r->host = apr_pstrdup (r->p, host);
  r->port = port;

  r->oplen = 0;
  r->max_oplen = DEFAULT_OPCODE_LEN;
  r->op = (unsigned char *) apr_pcalloc (p, r->max_oplen);
  r->ops = rovm_opcodes;

  r->send = rovm_send;
  r->printerror = rovm_printerror;
  r->printstack = rovm_printstack;
  r->printreturn = rovm_printreturn;

  if (rovm_connect (r))
    return NULL;

  if (rovm_send_req (r))
    return NULL;

  apr_socket_close (r->sock);
  r->sock = NULL;

  return r;
}

void
rovm_reqend (rovm_t *r)
{
  if (rovm_connect (r))
    return;

  if (rovm_send_reqend (r))
    return;

  apr_socket_close (r->sock);
  apr_pool_destroy (r->p);
  apr_terminate ();
}
