#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys, getopt, struct, time

OPCODE_NOP              = '\x00' # dec : 0
OPCODE_CPUSH            = "\x10" # dec : 16
OPCODE_ILOAD            = "\x15" # dec : 21
OPCODE_IADD             = "\x60" # dec : 96
OPCODE_IRETURN          = "\xac" # dec : 172

ATTRIBUTE_CLASSNAME     = 0xFF
ATTRIBUTE_METHOD        = 0xFE
ATTRIBUTE_METHOD_CODE   = 0xFD

STATUS_IN_NOTHING       = 0
STATUS_IN_CLASS         = 1
STATUS_IN_METHOD        = 2

outfilename = ""
infilename = ""
progname = sys.argv[0]
package = ""
status = STATUS_IN_NOTHING
current_class = None
current_attribute = None
last_methodattr = None

class ROVM_Attribute:
    def __init__ (self, atype):
        self.atype = atype
        self.attribute_length = 0
        self.attributes = []

        if atype == ATTRIBUTE_CLASSNAME:
            self.c = {}
        elif atype == ATTRIBUTE_METHOD:
            self.m = {}
        elif atype == ATTRIBUTE_METHOD_CODE:
            self.mcode = {}
            self.mcode['Opcode'] = []
        else:
            print "ROVM_Attribute: TODO"

    def add_attribute (self, aclass):
        self.attributes.append (aclass);

    def calc_attribute_length (self):
        self.attribute_length = 0
        
        if self.atype == ATTRIBUTE_CLASSNAME:
            self.attribute_length += 1      # Len 을 위하여
            self.attribute_length += len (self.c['Name'])
        elif self.atype == ATTRIBUTE_METHOD:
            self.attribute_length += 4      # 예약된 공간
            self.attribute_length += 1      # NameLen
            self.attribute_length += self.m['NameLen'] # Name
            self.attribute_length += 1      # TypeLen
            self.attribute_length += self.m['TypeLen']
            self.attribute_length += 2      # Attribute Count
            for a in self.attributes:
                self.attribute_length += 1  # Attribute Type
                self.attribute_length += 4  # Attribute Length
                self.attribute_length += a.calc_attribute_length ()
        elif self.atype == ATTRIBUTE_METHOD_CODE:
            self.attribute_length += 4  # Code Length

            # Code
            for o in self.mcode['Opcode']:
                self.attribute_length += len (o)

            self.attribute_length += 2  # 예약된 공간
            self.attribute_length += 2      # Attribute Count
            for a in self.attributes:
                self.attribute_length += 1  # Attribute Type
                self.attribute_length += 4  # Attribute Length
                self.attribute_length += a.calc_attribute_length ()
        else:
            print "calc_attribute_length: ToDO"
        
        return self.attribute_length

    def set_classname (self, name):
        name = name.encode ('UTF-8')
        
        self.c = {}
        self.c['Len'] = len (name)
        self.c['Name'] = name

    def set_method_NameAndType (self, name, mtype):
        """
        """
        name = name.encode ('UTF-8')
        mtype = mtype.encode ('UTF-8')
        
        self.m = {}
        self.m['Name'] = name
        self.m['NameLen'] = len (name)
        self.m['Type'] = mtype
        self.m['TypeLen'] = len (mtype)


    def add_method_code (self, opcode, arg1=None, arg2=None):
        """
        Opcode 를 추가한다.
        """
        total = ""
        total += opcode
        
        if arg1:
            total += arg1
            
        if arg1 and arg2:
            total += arg2

        self.mcode['Opcode'].append (total)

        self.mcode['OpcodeLen'] = 0
        for o in self.mcode['Opcode']:
            self.mcode['OpcodeLen'] += len (o)

    def printinfo (self):
        if self.atype == ATTRIBUTE_METHOD:
            print "Method Type"
            print self.m
        elif self.atype == ATTRIBUTE_CLASSNAME:
            print "Class Name"
            print self.c
        elif self.atype == ATTRIBUTE_METHOD_CODE:
            print "Method Code"
            print self.mcode
        else:
            print "ROVM_Attribute: TODO2"

        for a in self.attributes:
            a.printinfo ()

    def writefile (self, w):
        w.write (struct.pack ("B", self.atype))
        w.write (struct.pack ("I", self.calc_attribute_length ()))

        if self.atype == ATTRIBUTE_CLASSNAME:
            w.write (struct.pack ("B", self.c['Len']))
            w.write (self.c['Name'])
        elif self.atype == ATTRIBUTE_METHOD:
            w.write (struct.pack ("I", 0x00000000))     # 예약된 공간
            w.write (struct.pack ("B", self.m['NameLen']))
            w.write (self.m['Name'])
            w.write (struct.pack ("H", self.m['TypeLen']))
            w.write (self.m['Type'])
            w.write (struct.pack ("H", len (self.attributes)))
            for a in self.attributes:
                a.writefile (w)
        elif self.atype == ATTRIBUTE_METHOD_CODE:
            w.write (struct.pack ("I", self.mcode['OpcodeLen']))
            for o in self.mcode['Opcode']:
                w.write (o)
            w.write (struct.pack ("H", 0x0000)) # 예약된 공간
            w.write (struct.pack ("H", len (self.attributes)))
            for a in self.attributes:
                a.writefile (w)
        else:
            print "writefile: TODO"

"""
ROVM 에서 하나의 클래스를 다루는데, 사용하는 클래스.  아래에서 컴파일한
클래스에 대한 모든 정보들을 가지고 있습니다.
"""
class ROVM_Class:
    def __init__ (self, package, name):
        self.package = package
        self.name = name
        self.magic = 0x0e0a0209
        self.fileformat_version = struct.pack ("H", 0x0001)
        self.made_time =struct.pack ("I", time.time ())
        self.attributes = []

        a = ROVM_Attribute (ATTRIBUTE_CLASSNAME)
        if len (package) > 0:
            a.set_classname (package + "." + name)
        else:
            a.set_classname (name)
        self.add_attribute (a)

    def add_attribute (self, aclass):
        self.attributes.append (aclass);

    def writefile (self):
        w = open (self.name + ".e", "wb")

        # Magic
        w.write (struct.pack ("I", self.magic))
        # File Format Version
        w.write (self.fileformat_version)
        # Made Time
        w.write (self.made_time)
        # Attribute Count
        w.write (struct.pack ("H", len (self.attributes)))
        # Attributes
        for a in self.attributes:
            a.writefile (w)
            
def opcode_cpush (c):
    """
    OPCODE `cpush' 를 처리하여 ROVM_Class 클래스에 추가한다.
    """
    global current_attribute

    if current_attribute == None:
        raise ValueError, "TICKET == None"
    
    c = int (c[5:].rstrip ())

    if (c < -128 or c > 127):
        print "CPUSH: Invalid number."
        return

    current_attribute.add_opcode (OPCODE_CPUSH, struct.pack ("!b", c))
    current_attribute.push ()

def opcode_iadd (c):
    """
    OPCODE `iadd' 를 처리하여 ROVM_Class 클래스에 추가합니다.
    """
    global current_attribute

    if current_attribute == None:
        raise ValueError, "current_attribute == None"
    
    current_attribute.add_method_code (OPCODE_IADD)

def opcode_nop (c):
    """
    OPCODE `nop' 를 처리하여 ROVM_Class 클래스에 추가합니다.
    """
    global current_attribute

    if current_attribute == None:
        raise ValueError, "current_attribute == None"
    
    current_attribute.add_method_code (OPCODE_NOP)

def opcode_ireturn (c):
    """
    OPCODE `ireturn' 를 처리하여 ROVM_Class 클래스에 추가합니다.
    """
    global current_attribute

    if current_attribute == None:
        raise ValueError, "current_attribute == None"
    
    current_attribute.add_method_code (OPCODE_IRETURN)

def opcode_iload (c):
    """
    OPCODE `iload' 를 처리하여 ROVM_Class 클래스에 추가한다.
    """
    global current_attribute

    if current_attribute == None:
        raise ValueError, "TICKET == None"
    
    c = int (c[5:].rstrip ())

    if (c < 0 or c > 255):
        print "ILOAD: Invalid number."
        return

    current_attribute.add_method_code (OPCODE_ILOAD, struct.pack ("!B", c))

def compile_opcode (c):
    '''
    Opcode 를 컴파일 한다.
    '''
    if len (c) == 0:
        return

    cl = c.lower ()

    if cl[0:5] == "cpush":
        opcode_cpush (c)
    elif cl[0:5] == "iload":
        opcode_iload (c)
    elif cl == "ireturn":
        opcode_ireturn (c)
    elif cl == "iadd":
        opcode_iadd (c)
    elif cl == "nop":
        opcode_nop (c)
    else:
        print "TODO", c

def compile_and_make_output ():
    '''
    Input 파일을 읽어서 컴파일하고 그에 대한 output 파일을 생성합니다.
    현재 아래의 코드는 깨끗하게 완성된 멋진 코드가 *전혀* 아닙니다.
    '''
    global infilename, package, current_class, status, current_attribute
    global last_methodattr
    
    infile = open (infilename, "r")

    for line in infile.readlines ():
        line = line.rstrip ()
        line = line.lstrip ()
        if line[0:8] == ".package":
            package = line[8:]
            package = package.lstrip ()
        elif line[0:6] == ".class" and line != ".classend":
            status = STATUS_IN_CLASS

            classname = line[6:]
            classname = classname.lstrip ()

            current_class = ROVM_Class (package, classname)
        elif line == ".classend":
            current_class.writefile ()
        elif line[0:4] == ".def" and line != ".defend":
            status = STATUS_IN_METHOD

            splited = line.split (' ')

            current_attribute = ROVM_Attribute (ATTRIBUTE_METHOD)
            current_attribute.set_method_NameAndType (splited[1], splited[2])
        elif line == ".defend":
            current_class.add_attribute (last_methodattr)
        elif line == "":
            pass
        else:
            if current_attribute.atype == ATTRIBUTE_METHOD:
                newattr = ROVM_Attribute (ATTRIBUTE_METHOD_CODE)
                current_attribute.add_attribute (newattr)
                last_methodattr = current_attribute
                current_attribute = newattr
            # 여기 도달했다는 것은 Directive 가 아니고 opcode 임을 의미한다.
            compile_opcode (line)
        
    infile.close ()

def usage ():
    '''
    ROVM Compiler 를 사용하는 방법에 대해서 출력합니다.
    '''
    print '''Usage: %(progname)s [options] file...
Options:
  --help                Display this information
''' % globals ()

def main ():
    global outfilename, infilename
    
    try:
        opts, args = getopt.getopt (sys.argv[1:], "o:", ["help"])
    except getopt.GetoptError:
        usage ()
        sys.exit ()

    for o, a in opts:
        if o == "--help":
            usage ()
            sys.exit ()
        if o == "-o":
            outfilename = a        
    
    if len (args) == 0:
        print "%(progname)s: no input files." % globals ()
        return
        
    infilename = args[0]

    compile_and_make_output ()

if __name__ == "__main__":
    main ()
