/* ================================================================
 * EnhancedPreferences API - Alternative preferences package
 * ================================================================
 * 
 * Project Info:  http://development.poopu.com/ehprefs
 * Project Lead:  Justin Kim(jst.kim@gmail.com)
 * Create Date : 2004. 11. 3
 *
 * (C) Copyright 2004, by Justin Kim
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
package com.poopu.util.prefs;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.prefs.AbstractPreferences;
import java.util.prefs.BackingStoreException;

import net.sf.hibernate.Criteria;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Query;
import net.sf.hibernate.Session;
import net.sf.hibernate.Transaction;
import net.sf.hibernate.expression.Expression;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.picocontainer.MutablePicoContainer;
import org.picocontainer.defaults.DefaultPicoContainer;

import com.poopu.util.prefs.hibernate.HibernateManager;
import com.poopu.util.prefs.impl.PreferencesProviderImpl;
import com.poopu.util.prefs.om.Node;
import com.poopu.util.prefs.om.Property;

/**
 * @author jskim
 *
 */
public class HibernatePreferences extends EnhancedPreferences
{
  /* log instance */
  private final static Log log = LogFactory.getLog(HibernatePreferences.class);

  /* hibernate manager */
  private HibernateManager hibernateManager;

  /* preferences node object  */
  private Node node;

  /**
   * 
   * @param nodeType
   */
  public HibernatePreferences(Integer nodeType)
  {
    this(null, "", nodeType);
  }

  /**
   * 
   * @param parent
   * @param name
   * @param nodeType
   */
  public HibernatePreferences(AbstractPreferences parent, String name,
      Integer nodeType)
  {
    super(parent, name, nodeType);

    initHibernateManager();

    createNode();
  }

  protected Node getNode()
  {
    return this.node;
  }

  private void initHibernateManager()
  {

    // The instance of PreferencesProvider will be created.
    // but its attribute is not created again. because the implementation 
    // of provider has static fields.
    MutablePicoContainer pico = new DefaultPicoContainer();
    pico.registerComponentImplementation(PreferencesProvider.class,
        PreferencesProviderImpl.class);
    PreferencesProvider prefsProvider = (PreferencesProvider) pico
        .getComponentInstance(PreferencesProvider.class);

    Properties props = prefsProvider.getProperties();
    try
    {
      Class hmClazz = Class.forName(props
          .getProperty("ehprefs.hibernate.manager.impl"));
      pico.registerComponentImplementation(HibernateManager.class, hmClazz);
      hibernateManager = (HibernateManager) pico
          .getComponentInstance(HibernateManager.class);
    }
    catch (Exception e)
    {
      throw new InternalError("Can't get HibernateManager. "
          + "Check ehprefs.properties file, please");
    }
  }

  private void createNode()
  {
    // If node doesn't exist in database, it will be created.
    Session session = null;

    try
    {
      session = hibernateManager.openSession();
      Criteria crit = session.createCriteria(Node.class);

      crit.add(Expression.eq("absolutePath", absolutePath()));
      crit.add(Expression.eq("name", name()));
      crit.add(Expression.eq("type", getNodeType()));

      List nodes = crit.list();

      if (nodes.size() == 0)
      {
        // create node
        log.trace("Node does not exist. It'll be created.");

        Transaction tx = session.beginTransaction();

        this.node = new Node();
        node.setAbsolutePath(absolutePath());
        node.setName(name());
        node.setType(getNodeType());

        Calendar calendar = Calendar.getInstance();
        node.setCreationDate(calendar.getTime());
        node.setModifiedDate(calendar.getTime());

        if (this.parent() != null)
        {
          HibernatePreferences hprefs = (HibernatePreferences) parent();
          node.setParentId(hprefs.getNode().getId());
        }
        session.save(node);

        tx.commit();
      }
      else
      {
        if (nodes.size() > 1)
          log.warn("Detected too many nodes.");

        this.node = (Node) nodes.get(0);
      }
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }

  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#putSpi(java.lang.String, java.lang.String)
   */
  protected void putSpi(String key, String value)
  {
    Session session = null;
    try
    {
      session = hibernateManager.openSession();
      Transaction tx = session.beginTransaction();

      Criteria crit = session.createCriteria(Property.class);
      crit.add(Expression.eq("nodeId", getNode().getId()));
      crit.add(Expression.eq("key", key));

      List result = crit.list();

      Property property = new Property();
      Date modifiedDate = Calendar.getInstance().getTime();

      if (result.size() == 0)
      {
        property.setNodeId(getNode().getId());
        property.setKey(key);

        property.setCreationDate(modifiedDate);

      }
      else
      {
        property = (Property) result.get(0);
      }

      property.setValue(value);
      property.setModifiedDate(modifiedDate);
      session.save(property);
      tx.commit();
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#getSpi(java.lang.String)
   */
  protected String getSpi(String key)
  {
    String value = null;
    Session session = null;

    try
    {
      session = hibernateManager.openSession();
      Query query = session.createQuery("select property.value from "
          + "  com.poopu.util.prefs.om.Property property "
          + "where property.nodeId = :nodeId and property.key = :propertyKey");

      query.setLong("nodeId", getNode().getId().longValue());
      query.setString("propertyKey", key);

      List result = query.list();

      if (result.size() != 0)
      {
        value = (String) result.get(0);
      }
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }

    return value;
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#removeSpi(java.lang.String)
   */
  protected void removeSpi(String key)
  {
    Session session = null;

    try
    {
      session = hibernateManager.openSession();

      Transaction tx = session.beginTransaction();

      Criteria crit = session.createCriteria(Property.class);

      crit.add(Expression.eq("key", key));
      crit.add(Expression.eq("nodeId", getNode().getId()));

      List result = crit.list();

      if (result.size() == 1)
      {
        Property property = (Property) result.get(0);
        session.delete(property);
      }

      tx.commit();
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#removeNodeSpi()
   */
  protected void removeNodeSpi() throws BackingStoreException
  {
    Session session = null;

    try
    {
      session = hibernateManager.openSession();
      Transaction tx = session.beginTransaction();
      session.delete(getNode());
      tx.commit();
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#keysSpi()
   */
  protected String[] keysSpi() throws BackingStoreException
  {
    ArrayList list = new ArrayList();
    Session session = null;

    try
    {
      session = hibernateManager.openSession();
      Query query = session.createQuery("select property.key from "
          + "  com.poopu.util.prefs.om.Property property "
          + "where property.nodeId = :nodeId");

      query.setLong("nodeId", getNode().getId().longValue());

      list.addAll(query.list());
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }

    String[] keys = new String[list.size()];
    list.toArray(keys);

    return keys;
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#childrenNamesSpi()
   */
  protected String[] childrenNamesSpi() throws BackingStoreException
  {
    ArrayList list = new ArrayList();
    Session session = null;

    try
    {
      session = hibernateManager.openSession();
      Query query = session.createQuery("select node.name from "
          + "  com.poopu.util.prefs.om.Node node "
          + "where node.parentId = :parentId and node.type = :nodeType");

      query.setLong("parentId", getNode().getId().longValue());
      query.setInteger("nodeType", getNodeType().intValue());

      list.addAll(query.list());
    }
    catch (HibernateException e)
    {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    finally
    {
      try
      {
        if (session != null)
          session.close();
      }
      catch (HibernateException e1)
      {
        log.fatal("Can't close Hibernate Session. "
            + "This exception mustn't be occurred. "
            + "Please, check database connection", e1);
      }
    }

    String[] names = new String[list.size()];
    list.toArray(names);

    return names;
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#childSpi(java.lang.String)
   */
  protected AbstractPreferences childSpi(String name)
  {
    return new HibernatePreferences(this, name, getNodeType());
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#syncSpi()
   */
  protected void syncSpi() throws BackingStoreException
  {
    flushSpi();
  }

  /* (non-Javadoc)
   * @see java.util.prefs.AbstractPreferences#flushSpi()
   */
  protected void flushSpi() throws BackingStoreException
  {
    // TODO I don't have a idea. What shoud I do here?

  }

}