//===================================================================
//
// teno.c (@haekim)
// Date : 2008.01
//
//===================================================================           
// Copyright 2004-2010, ETRI 
//===================================================================
#include "teno.h"

//#define TENO_DEBUG
#ifdef UART_M
#include "uart.h"
#include "platform.h"
#endif


#ifdef TENO_M
#include <string.h>
#include <stdlib.h>
#include "arch.h"
#include "mac.h"
#include "critical_section.h"
#include "user_timer.h"

TENO_NIB teno_nib;
UINT8	join_timer_id;	// remember timer ID created to send JOIN msg

// We don't need to support re-entrance for RX. So we use global variables to reduce overhead for each thread stack memory
void (*teno_rx_callback)(void);
NMAC_RX_INFO teno_mac_rx_info;
UINT8 teno_mac_payload[NMAC_MAX_PAYLOAD_SIZE];
TENO_JOIN_PACKET teno_join_packet;
TENO_REPLY_PACKET teno_reply_packet;
TENO_RELEASE_PACKET teno_release_packet;
TENO_DATA_PACKET teno_rx_data_packet;

// Concetric Circles routing has 1-item long data queue.
struct _teno_val_data
{
	UINT16 src_id;
	UINT16 dest_id;
	UINT16 parent_id;
	UINT16 rx_seq;
	UINT8 port, data_length;
	void* data_ptr;
} teno_val_data;

extern BOOL nos_taskq_reg(void (*func)(void));


// Initializing  [ Channel:0x0B~0x1A (11~26), PAN ID : 0x0000~0xfffE(0~65534), Node ID(MAC short address) : 0~65534 ]
void teno_init(UINT8 channel, UINT16 pan_id, UINT16 node_id)
{
	NOS_ENTER_CRITICAL_SECTION();
	// NWK layer initialization
	teno_set_rx_cb(NULL);
	teno_mac_rx_info.payload_ptr = teno_mac_payload;
	teno_nib.my_id = node_id;
	srand(node_id);
	teno_nib.tx_seq = (UINT16)rand();
	teno_nib.parent_id = TENO_GENERAL_SINK_ID;
	teno_nib.parent_candidate = TENO_GENERAL_SINK_ID;
	join_timer_id = TIMER_CREATE_ERROR;	//no join timer created
	teno_val_data.rx_seq = 0xFFFF;	//dummy sequence
	
	// MAC layer initialization
	nmac_init(channel, pan_id, node_id); // We should add (void *) for net_payload. refer to [void reno_recv_handler ( UINT8 prev_node_id )]
	nos_mac_set_rx_cb( teno_cb_from_mac );

	// to join a network
	teno_start_alone_state();
	NOS_EXIT_CRITICAL_SECTION();
}


//Callback function for MAC RX interrupt.
void teno_cb_from_mac(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rMAC_RX");
#endif
	nos_taskq_reg( teno_recv_from_mac );
}


//Sets callback function for NWK RX.
void teno_set_rx_cb(void (*func)(void))
{
	teno_rx_callback = func;
}


// Make a node to sink
void teno_role_as_sink(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrole_as_sink");
#endif
	NOS_ENTER_CRITICAL_SECTION();
	teno_nib.state = TENO_SINK_STATE;
	teno_nib.my_depth = 0;
	teno_nib.parent_id = TENO_GENERAL_SINK_ID;
	teno_nib.parent_candidate = TENO_GENERAL_SINK_ID;
	teno_nib.candidate = FALSE;
	NOS_EXIT_CRITICAL_SECTION();
	teno_send_reply_to_mac(TENO_BROADCAST_ID);
}

// Make a sink to a normal node
void teno_role_as_node(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrole_as_node");
#endif
	NOS_ENTER_CRITICAL_SECTION();
	if(teno_nib.state == TENO_SINK_STATE)
	{
		teno_start_alone_state();
	}
	NOS_EXIT_CRITICAL_SECTION();
}


void teno_start_alone_state(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rstart_alone_state");
#endif
	NOS_ENTER_CRITICAL_SECTION();
	if (teno_nib.state != TENO_ALONE_STATE)
	{
		teno_nib.my_depth = 0xFF;
		teno_nib.state = TENO_ALONE_STATE;
		teno_nib.candidate = FALSE;
		
		// to join a network
		teno_send_join_to_mac();
		if(join_timer_id >= MAX_NUM_TOTAL_TIMER)
		{
			join_timer_id = nos_timer_create_sec(teno_join_timer, 5, TIMER_PERIODIC);
			if (join_timer_id == TIMER_CREATE_ERROR)
			{
				// we can't control this case.
				NOS_DISABLE_GLOBAL_INTERRUPT();
				#ifdef UART_M
				nos_uart_puts(STDOUT, "\n\rError! : TENO routing failed to create join timer. System has halted.");
				#endif
				while (1);
			}
		}
	}
	NOS_EXIT_CRITICAL_SECTION();
}


// Broadcast JOIN msg. 
void teno_join_timer(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rjoin_timer");
#endif
	if (teno_nib.state == TENO_ALONE_STATE)
	{
		teno_send_join_to_mac();
	}
	else
	{
		NOS_ENTER_CRITICAL_SECTION();
		nos_timer_destroy(join_timer_id);
		join_timer_id = TIMER_CREATE_ERROR;	// means no join timer created
		NOS_EXIT_CRITICAL_SECTION();
	}
}


// Broadcast JOIN msg.
void teno_send_join_to_mac(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend join");
#endif
	NMAC_TX_INFO mac_tx_info;
	if (teno_nib.state != TENO_SINK_STATE)
	{
		teno_join_packet.msg_type = TENO_JOIN;
		teno_join_packet.depth = teno_nib.my_depth;
		mac_tx_info.dest_addr = 0xffff;
		mac_tx_info.routing_header_length 	= TENO_JOIN_PACKET_SIZE;
		mac_tx_info.routing_header_ptr 	= &teno_join_packet;
		mac_tx_info.payload_length = mac_tx_info.routing_header_length + 0;
		mac_tx_info.payload_ptr = NULL;
		nmac_tx_noack(&mac_tx_info);
	}
}


// Send REPLY msg.
BOOL teno_send_reply_to_mac(UINT16 dest_id)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend reply");
#endif
	NMAC_TX_INFO mac_tx_info;
	if(teno_nib.state != TENO_ALONE_STATE)
	{
		teno_reply_packet.msg_type = TENO_REPLY;
		teno_reply_packet.depth = teno_nib.my_depth;
		mac_tx_info.dest_addr = dest_id;
		mac_tx_info.routing_header_length = TENO_REPLY_PACKET_SIZE;
		mac_tx_info.routing_header_ptr = &teno_reply_packet;
		mac_tx_info.payload_length = mac_tx_info.routing_header_length + 0;
		mac_tx_info.payload_ptr = NULL;
		return nmac_tx(&mac_tx_info);
	}
	return FALSE;
}


// Send RELEASE msg.
BOOL teno_send_release_to_mac(UINT16 dest_id)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend release");
#endif
	NMAC_TX_INFO mac_tx_info;
	if (teno_nib.state != TENO_SINK_STATE)
	{
		teno_release_packet.msg_type = TENO_RELEASE;
		mac_tx_info.dest_addr = dest_id;
		mac_tx_info.routing_header_length 	= TENO_RELEASE_PACKET_SIZE;
		mac_tx_info.routing_header_ptr 	= &teno_release_packet;
		mac_tx_info.payload_length = mac_tx_info.routing_header_length + 0;
		mac_tx_info.payload_ptr = NULL;
		return nmac_tx(&mac_tx_info);
	}
	return FALSE;
}


// Send DATA msg.
BOOL teno_send_data_to_mac(TENO_DATA_PACKET* tx_packet_ptr, UINT16 dest_id)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend data");
#endif
	NMAC_TX_INFO mac_tx_info;
	++tx_packet_ptr->msg_hop_count;
	tx_packet_ptr->depth = teno_nib.my_depth;
	mac_tx_info.dest_addr = dest_id;
	mac_tx_info.routing_header_length 	= TENO_DATA_PACKET_HEADER_SIZE;
	mac_tx_info.routing_header_ptr 	= tx_packet_ptr;
	mac_tx_info.payload_length 	= mac_tx_info.routing_header_length + tx_packet_ptr->payload_size;
	mac_tx_info.payload_ptr    	= tx_packet_ptr->payload_ptr;
	if(dest_id == TENO_BROADCAST_ID)
	{
		nmac_tx_noack(&mac_tx_info);
		return TRUE;
	}
	else
	{
		return nmac_tx(&mac_tx_info);
	}
}


BOOL teno_broadcast_data_downward(TENO_DATA_PACKET* tx_packet_ptr)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend_data_downward");
#endif
	if (teno_nib.state == TENO_SINK_STATE)
	{
		tx_packet_ptr->msg_type = TENO_DOWNDATA;
		tx_packet_ptr->msg_hop_count = 0;
		teno_send_data_to_mac(tx_packet_ptr, TENO_BROADCAST_ID);
		return TRUE;
	}
	return FALSE;
}


BOOL teno_send_data_upward(TENO_DATA_PACKET* tx_packet_ptr)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend_data_upward");
#endif

	while (teno_nib.state == TENO_JOIN_STATE)
	{
		if (teno_send_data_to_mac(tx_packet_ptr, teno_nib.parent_id))
		{
			return TRUE;
		}
		else
		{
			if (teno_del_curr_parent())
			{
				if (tx_packet_ptr->src_id == teno_nib.my_id)
				{
					tx_packet_ptr->parent_id = teno_nib.parent_id;
				}
			}
		}
	}
	return FALSE;
}


// return node's state. JOIN(True), ALONE or SINK(False).
BOOL teno_del_curr_parent(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rdel_curr_parent");
#endif
	if (teno_nib.state == TENO_JOIN_STATE)
	{
		if (teno_nib.candidate == TRUE)
		{
			NOS_ENTER_CRITICAL_SECTION();
			teno_nib.candidate = FALSE;
			teno_nib.parent_id = teno_nib.parent_candidate;
			NOS_EXIT_CRITICAL_SECTION();
#ifdef MONITOR
			teno_send_to_sink(TENO_MONITOR_PORT, 0, NULL);	// send dummy data to maintain tree topology
#endif
			// to fresh the link, broadcast JOIN message 1 times.
			teno_send_join_to_mac();
			return TRUE;
		}
		else
		{
			// this will broadcast JOIN message until the node is joined to a network.
			teno_start_alone_state();
			return FALSE;
		}
	}
	else
	{
		return FALSE;
	}
}


// Do not support re-entrance. Should be called by taskq_reg only.
// 1. check teno_nib.state
// 2. check depth
// 3. check rx_seq (for downward packet)
// 4. check previous node
// 5. check destination
void teno_recv_from_mac(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrecv_from_mac");
#endif

	UINT8 msg_type, depth;
	while ( nmac_rx(&teno_mac_rx_info) )
	{
		msg_type = ((UINT8*)teno_mac_rx_info.payload_ptr)[0];

		//================== JOIN ===================//
		if(msg_type == TENO_JOIN)
		{
#ifdef TENO_DEBUG
			nos_uart_puts(STDOUT, "-join");
#endif		
			if (teno_nib.state == TENO_JOIN_STATE)
			{
				if (teno_nib.my_depth < TENO_MAX_TREE_DEPTH)
				{
					depth = ((UINT8*)teno_mac_rx_info.payload_ptr)[1];
					if(depth > teno_nib.my_depth)
					{
						if(teno_mac_rx_info.src_addr != teno_nib.parent_id)
						{
							nos_delay_us(teno_nib.my_depth*100);
							teno_send_reply_to_mac(teno_mac_rx_info.src_addr);
						}
					}
				}

			}
			else if (teno_nib.state == TENO_SINK_STATE)
			{
				teno_send_reply_to_mac(teno_mac_rx_info.src_addr);
			}
		}
		
		//================== REPLY ===================//
		else if (msg_type == TENO_REPLY)
		{
#ifdef TENO_DEBUG
			nos_uart_puts(STDOUT, "-reply");
#endif		
			depth = ((UINT8*)teno_mac_rx_info.payload_ptr)[1];
			if (teno_nib.state == TENO_ALONE_STATE)
			{
				teno_nib.my_depth = depth+1;
				teno_nib.state = TENO_JOIN_STATE;
				teno_nib.parent_id = teno_mac_rx_info.src_addr;
#ifdef MONITOR
				teno_send_to_sink(TENO_MONITOR_PORT, 0, NULL);	// send dummy data to maintain tree topology
#endif
			}
			else if (teno_nib.state == TENO_JOIN_STATE)
			{
				// shorter depth to sink
				if (depth+1 < teno_nib.my_depth)
				{
					teno_nib.candidate = FALSE;
					teno_nib.my_depth = depth+1;
					teno_nib.parent_id = teno_mac_rx_info.src_addr;
#ifdef MONITOR
					teno_send_to_sink(TENO_MONITOR_PORT, 0, NULL);	// send dummy data to maintain tree topology
#endif
				}
				// the same depth to sink
				else if (depth+1 == teno_nib.my_depth)
				{
					if(teno_mac_rx_info.src_addr != teno_nib.parent_id)
					{
						teno_nib.candidate = TRUE;
						teno_nib.parent_candidate = teno_mac_rx_info.src_addr;
					}
				}
			}
		}

		//================== RELEASE  ===================//
		else if (msg_type == TENO_RELEASE)
		{
#ifdef TENO_DEBUG
			nos_uart_puts(STDOUT, "-release");
#endif		
			if(teno_nib.state == TENO_JOIN_STATE)
			{
				if (teno_mac_rx_info.src_addr == teno_nib.parent_id)
				{
					teno_del_curr_parent();
				}
				else if (teno_mac_rx_info.src_addr == teno_nib.parent_candidate)
				{
					teno_nib.candidate = FALSE;
				}
			}
		}

		//================== DOWNDATA ===================//
		else if (msg_type == TENO_DOWNDATA)
		{
#ifdef TENO_DEBUG
			nos_uart_puts(STDOUT, "-downdata");
#endif		
			if (teno_nib.state != TENO_SINK_STATE)
			{
				teno_recv_downdata_from_mac();
			}
		}
		
		//================== UPDATA ===================//
		else if (msg_type == TENO_UPDATA)
		{
#ifdef TENO_DEBUG
			nos_uart_puts(STDOUT, "-updata");
#endif		
			teno_recv_updata_from_mac();
		}
	}
}


void teno_recv_downdata_from_mac(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrecv_downdata_from_mac");
#endif
	teno_rx_data_packet.msg_type 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[0];
	teno_rx_data_packet.msg_hop_limit 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[1];
	teno_rx_data_packet.msg_hop_count 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[2];
	if (teno_rx_data_packet.msg_hop_count > teno_rx_data_packet.msg_hop_limit)
	{
		return;
	}
	teno_rx_data_packet.depth	= ((UINT8*)teno_mac_rx_info.payload_ptr)[3];
	teno_rx_data_packet.seq = *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[4]));
	teno_rx_data_packet.dest_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[6]));
	teno_rx_data_packet.src_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[8]));
	teno_rx_data_packet.parent_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[10]));
	teno_rx_data_packet.port		= ((UINT8*)teno_mac_rx_info.payload_ptr)[12];
	teno_rx_data_packet.payload_size = ((UINT8*)teno_mac_rx_info.payload_ptr)[13];
	teno_rx_data_packet.payload_ptr = (void*)( &(((UINT8*)teno_mac_rx_info.payload_ptr)[14]) );

	if(teno_nib.state == TENO_ALONE_STATE)
	{
		if (teno_rx_data_packet.seq != teno_val_data.rx_seq)
		{
			teno_val_data.rx_seq = teno_rx_data_packet.seq;
			// I am the packet's destination.
			if (teno_rx_data_packet.dest_id == teno_nib.my_id || teno_rx_data_packet.dest_id == TENO_GENERAL_SINK_ID)
			{
				teno_store_data(&teno_rx_data_packet);
			}
		}
	}
	else if(teno_nib.state == TENO_JOIN_STATE)
	{
		if (teno_rx_data_packet.depth < teno_nib.my_depth)
		{
			// there is a shorter link
			if (teno_rx_data_packet.depth+1 < teno_nib.my_depth)
			{
				// this can be changed to unicast to the source. 
				teno_send_join_to_mac();
			}
			if (teno_rx_data_packet.seq != teno_val_data.rx_seq)
			{
				teno_val_data.rx_seq = teno_rx_data_packet.seq;
				// I am the packet's destination.
				if (teno_rx_data_packet.dest_id == teno_nib.my_id)
				{
					teno_store_data(&teno_rx_data_packet);
				}
				else if (teno_rx_data_packet.dest_id == TENO_BROADCAST_ID)
				{
					teno_send_data_to_mac(&teno_rx_data_packet, TENO_BROADCAST_ID);
					teno_store_data(&teno_rx_data_packet);
				}
				else
				{
					teno_send_data_to_mac(&teno_rx_data_packet, TENO_BROADCAST_ID);
				}
			}
		}
	}
	
}


void teno_recv_updata_from_mac(void)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrecv_updata_from_mac");
#endif
	teno_rx_data_packet.msg_type 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[0];
	teno_rx_data_packet.msg_hop_limit 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[1];
	teno_rx_data_packet.msg_hop_count 	= ((UINT8*)teno_mac_rx_info.payload_ptr)[2];
	if (teno_rx_data_packet.msg_hop_count > teno_rx_data_packet.msg_hop_limit)
	{
		return;
	}
	teno_rx_data_packet.depth	= ((UINT8*)teno_mac_rx_info.payload_ptr)[3];
	teno_rx_data_packet.seq = *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[4]));
	teno_rx_data_packet.dest_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[6]));
	teno_rx_data_packet.src_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[8]));
	teno_rx_data_packet.parent_id		= *((UINT16*)&(((UINT8*)teno_mac_rx_info.payload_ptr)[10]));
	teno_rx_data_packet.port		= ((UINT8*)teno_mac_rx_info.payload_ptr)[12];
	teno_rx_data_packet.payload_size = ((UINT8*)teno_mac_rx_info.payload_ptr)[13];
	teno_rx_data_packet.payload_ptr = (void*)( &(((UINT8*)teno_mac_rx_info.payload_ptr)[14]) );

	if (teno_nib.state == TENO_ALONE_STATE)
	{
		// previous node is in ALONE or SINK state
		if(teno_rx_data_packet.depth == 0xFF || teno_rx_data_packet.depth == 0)
		{
			// I am the packet's destination.
			if (teno_rx_data_packet.dest_id == teno_nib.my_id)
			{
				teno_store_data(&teno_rx_data_packet);
			}
		}
		else
		{
			// I am the packet's destination.
			if (teno_rx_data_packet.dest_id == teno_nib.my_id)
			{
				teno_send_release_to_mac(teno_mac_rx_info.src_addr);
				teno_store_data(&teno_rx_data_packet);
			}
			else
			{
				// send back data to prev_hop_id. downdata from parent will make child delete curruent link. So don't need to send RELEASE msg.
				teno_send_data_to_mac(&teno_rx_data_packet, teno_mac_rx_info.src_addr);
			}
		}
	}
	else if (teno_nib.state == TENO_SINK_STATE)
	{
		if (teno_rx_data_packet.depth != 1)
		{
			teno_send_reply_to_mac(teno_mac_rx_info.src_addr);
		}
		// I am the packet's destination.
		if (teno_rx_data_packet.dest_id == teno_nib.my_id || teno_rx_data_packet.dest_id == TENO_GENERAL_SINK_ID)
		{
			teno_store_data(&teno_rx_data_packet);
		}
		else
		{
#ifdef MONITOR
			teno_store_data(&teno_rx_data_packet);
#else
			teno_broadcast_data_downward(&teno_rx_data_packet);
#endif
		}
	}
	else if (teno_nib.state == TENO_JOIN_STATE)
	{
		// past parent is in ALONE state now. delete current link.
		if (teno_mac_rx_info.src_addr == teno_nib.parent_id)
		{
			if(teno_del_curr_parent())
			{
				// to reduce packet loss, send previous data packet to other parent
				teno_send_data_upward(&teno_rx_data_packet);
			}
		}
		else
		{
			if (teno_rx_data_packet.depth <= teno_nib.my_depth)
			{
				teno_send_release_to_mac(teno_mac_rx_info.src_addr);
			}
			else if (teno_rx_data_packet.depth > teno_nib.my_depth+1)
			{
				teno_send_reply_to_mac(teno_mac_rx_info.src_addr);
			}
			// I am the packet's destination.
			if (teno_rx_data_packet.dest_id == teno_nib.my_id)
			{
				teno_store_data(&teno_rx_data_packet);
			}
			else
			{
				teno_send_data_upward(&teno_rx_data_packet);
			}
		}
	}
}


void teno_store_data(TENO_DATA_PACKET* rx_packet_ptr)
{
	teno_val_data.src_id = rx_packet_ptr->src_id;
	teno_val_data.dest_id = rx_packet_ptr->dest_id;
	teno_val_data.parent_id = rx_packet_ptr->parent_id;
	teno_val_data.rx_seq = rx_packet_ptr->seq;
	teno_val_data.port = rx_packet_ptr->port;
	teno_val_data.data_length = rx_packet_ptr->payload_size;
	teno_val_data.data_ptr = rx_packet_ptr->payload_ptr;
	if (teno_rx_callback)
	{
		// [teno_recv_from_nwk(...)] must be called from callback.
		teno_rx_callback();
	}
}

// This must be called in teno_rx_callback()
void teno_recv_from_nwk(UINT16* src_id, UINT8* port, UINT8* data_length, void* data, UINT16* parent_id, UINT16* dest_id)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rrecv_from_nwk");
#endif
	*src_id = teno_val_data.src_id;
	*dest_id = teno_val_data.dest_id;
	*parent_id = teno_val_data.parent_id;
	*port = teno_val_data.port;
	*data_length = teno_val_data.data_length;
	memcpy(data, teno_val_data.data_ptr, teno_val_data.data_length);
}


BOOL teno_send_to_sink(UINT8 port, UINT8 data_length, void* data)
{
	return teno_send_to_node(TENO_GENERAL_SINK_ID, port, data_length, data);
}


BOOL teno_send_to_node(UINT16 dest_id, UINT8 port, UINT8 data_length, void* data)
{
#ifdef TENO_DEBUG
	nos_uart_puts(STDOUT, "\n\rsend to node");
#endif
	TENO_DATA_PACKET teno_tx_data_packet;
	teno_tx_data_packet.msg_hop_limit = TENO_MAX_TREE_DEPTH;
	teno_tx_data_packet.msg_hop_count = 0;
	teno_tx_data_packet.seq = teno_nib.tx_seq++;
	teno_tx_data_packet.dest_id = dest_id;
	teno_tx_data_packet.src_id = teno_nib.my_id;
	teno_tx_data_packet.parent_id = teno_nib.parent_id;
	teno_tx_data_packet.port = port;
	teno_tx_data_packet.payload_size = data_length;
	teno_tx_data_packet.payload_ptr = data;
	
	if(dest_id == teno_nib.my_id)
	{
		teno_store_data(&teno_tx_data_packet);
		return TRUE;
	}
	else
	{
		if (teno_nib.state == TENO_ALONE_STATE)
		{
			return FALSE;
		}
		else if (teno_nib.state == TENO_JOIN_STATE)
		{
			teno_tx_data_packet.msg_type = TENO_UPDATA;
			return teno_send_data_upward(&teno_tx_data_packet);
		}
		else if (teno_nib.state == TENO_SINK_STATE)
		{
			return teno_broadcast_data_downward(&teno_tx_data_packet);
		}
		return FALSE;
	}
}



#ifdef MONITOR
BOOL teno_send_to_node_by_src_routing(UINT16 dest_id, UINT8 port, UINT8 data_length, void* data, UINT8 route_len, UINT16* route_ptr)
{
	// Source routing to destination.
	//teno_send_data_by_src_routing(&teno_tx_data_packet);
}
#endif


#ifdef DEMO_MAC_M
void teno_set_rx_range(UINT16 min_permit_id, UINT16 max_permit_id)
{
	nos_mac_set_rx_range(min_permit_id, max_permit_id);
}
#endif

#endif // TENO_M
