//===================================================================
//
// reno.c (@haekim)
// Date : 06.09.01
//
//===================================================================           
// Copyright 2004-2010, ETRI 
//===================================================================
#include "reno.h"

#ifdef RENO_M
#include <stdlib.h>
#include <string.h>
#include "mac.h"
#include "platform.h"
#include "arch.h"
#include "critical_section.h"






//#include "sched.h"
#include "thread.h"	// nos_thread_sleep_ms
#include "intr.h"
extern volatile INTR_STATUS intr_status;


// Basic configuration
#define RENO_RREQ_WAIT_TIME 	(1000) // [ms] the time that a node waits until RREP is received after RREQ is sent
#define RENO_RREQ_TRIES 	(3)   // The maximum number of tries to find a path by sending RREQ
#define MAX_FAIL_ENDURANCE (1)	// send RREP after some TX trial failure
#define ROUTE_TABLE_LOOKUP_FAILURE (255) // This value should be K, where MAX_ROUTE_ENTRY < K <= 255 (max limit of UINT8)

#define UNICAST		(0)
#define BROADCAST	(1)

static void (*reno_rx_callback)(void);

void reno_handler(RENO_SIB *sib_ptr);
BOOL reno_link_state_update(UINT8 next_hop_id, BOOL success);
void reno_route_lookup_fail(RENO_SIB* sib_ptr);
void reno_route_lookup_success(RENO_SIB* sib_ptr);
UINT8 reno_send_rreq(UINT8 dest_id);
BOOL reno_send_rrep(UINT8 dest_id);
BOOL reno_send_rerr(UINT8 dest_id, UINT8 prev_node_id, UINT8 err_node);
UINT8 reno_scan_route_table(UINT8 dest_id);
BOOL reno_insert_route_entry(RENO_SIB* sib_ptr);
void reno_delete_dest_route_entry(UINT8 dest_id);
void reno_delete_next_route_entry(UINT8 next_hop_id);


// Initializing  [ Channel:0x0B~0x1A (11~26), PAN ID : 0x0000~0xfffE(0~65534), Node ID(MAC short address) : 0~255 ]
void reno_init(UINT8 channel, UINT16 pan_id, UINT8 node_id, UINT8 min_permit_id, UINT8 max_permit_id)
{
	UINT8 i;

	// MAC layer operations
 	nmac_init(channel, pan_id, (UINT16) node_id); // We should add (void *) for net_payload. refer to [void reno_recv_handler ( UINT8 prev_node_id )]

	// Each node has one NIB (RENO_NIB)
   	RENO_NIB.id 	 = node_id;
	RENO_NIB.seq_num = rand()%60000;
	RENO_NIB.route_table_index = 0;
	for (i = 0; i < MAX_ROUTE_ENTRY; i++)
	{
		RENO_NIB.route_table[i].is_valid = 0;
		RENO_NIB.route_table[i].link_fail_count = 0;
	}

	// initialize for the reno_queue
	RENO_QUEUE.front 	= 0;
	RENO_QUEUE.rear 	= 0;
	RENO_QUEUE.nitem	= 0;
	memset( &RENO_QUEUE, 0, sizeof(struct reno_queue));
	
	// For simulation
	nos_mac_set_rx_range(min_permit_id, max_permit_id);

	// register callback functions
	nos_mac_set_rx_cb( reno_cb_from_mac );
	reno_set_rx_cb(NULL);
}


void reno_set_rx_cb(void (*func)(void))
{
	reno_rx_callback = func;
}


// This function is called from application layer (when sending) or MAC layer (when receiving).
// MAC receiving from RF function should be modified to store RENO_SIB information and then call this function.
void reno_handler(RENO_SIB *sib_ptr)
{
	UINT8 routing_table_entry;
	// See the destination of the arrived packet.
	// If the packet is mine, process this packet locally
	// Else, consult the route table to deal with this packet.

	// If this packet is for this node, process it locally
	if (sib_ptr->packet.dest_id == RENO_NIB.id)
	{
		switch (sib_ptr->packet.msg_type)
		{
			case RENO_DATA : 
				// Save data into reno_queue if queue is not full
				NOS_ENTER_CRITICAL_SECTION();
				//if (RENO_QUEUE.front != (RENO_QUEUE.rear+1)%RENO_QUEUE_LENGTH)
				if (RENO_QUEUE.nitem != RENO_QUEUE_LENGTH)
				{
					++RENO_QUEUE.nitem;
					RENO_QUEUE.item[RENO_QUEUE.rear].src_id = sib_ptr->packet.src_id;
					RENO_QUEUE.item[RENO_QUEUE.rear].data_length = sib_ptr->packet.payload_size;
					memcpy(RENO_QUEUE.item[RENO_QUEUE.rear].data, sib_ptr->packet.payload_ptr, sib_ptr->packet.payload_size);
					RENO_QUEUE.rear = (RENO_QUEUE.rear+1)%RENO_QUEUE_LENGTH;
					if (reno_rx_callback)
					{
						reno_rx_callback();
					}
				}
				// data have lost because queue was full.
				// even though the reno queue was full, 
				// we must read a frame from MAC because the frame may not be a data frame (RREQ,RREP).
				/*
				else
				{
					nos_uart_puts(STDOUT, "\n\rpacket droped from nwk!");
				}
				*/
				NOS_EXIT_CRITICAL_SECTION();
				break;

			case RENO_RREQ :
//				nos_uart_puts(STDOUT, "\n\rRREQ! : ");
//				nos_uart_puti(STDOUT, sib_ptr->packet.src_id);
				if (reno_insert_route_entry(sib_ptr))
				{
//					nos_uart_puts(STDOUT, " success!!");
					// send a RREP for the source node that sent this RREQ packet
					reno_send_rrep(sib_ptr->packet.src_id);
#ifdef MONITOR
					call_monitor(sib_ptr, 1);
#endif
				}
				break;

			case RENO_RREP :	
#ifdef MONITOR
					call_monitor(sib_ptr, 3);
#endif
				reno_insert_route_entry(sib_ptr);				
				break;

			case RENO_RERR :	
				// Deletes the broken link to the destination that is RERR payload (destination of DATA) from the routing table.
				reno_delete_dest_route_entry(*(UINT8*)sib_ptr->packet.payload_ptr);
				break;

			default :
				// do nothing
				break;
		}
	}
	// this packet is not mine, consult the lookup table to know how to deal with this packet
	else
	{
		// look up the routing table
		routing_table_entry = reno_scan_route_table(sib_ptr->packet.dest_id);

		if (routing_table_entry == ROUTE_TABLE_LOOKUP_FAILURE) // lookup failure
		{
			reno_route_lookup_fail(sib_ptr); // process when lookup is failed
		}
		else // lookup success; next_hop_id (to send this packet) is available
		{
			sib_ptr->next_hop_id = RENO_NIB.route_table[routing_table_entry].next_hop_id;		
			reno_route_lookup_success(sib_ptr); // process when lookup is succeed
		}
	}
}


BOOL reno_link_state_update(UINT8 next_hop_id, BOOL success)
{
	UINT8 i;
	if (success)
	{
		for ( i=0; i<MAX_ROUTE_ENTRY; i++)
		{
			if ( (RENO_NIB.route_table[i].next_hop_id == next_hop_id) && (RENO_NIB.route_table[i].link_fail_count > 0) )
			{
				--RENO_NIB.route_table[i].link_fail_count;
			}
		}
	}
	
	//if no ack is received from MAC layer
	else
	{
		for ( i=0; i<MAX_ROUTE_ENTRY; i++)
		{
			if (RENO_NIB.route_table[i].next_hop_id == next_hop_id)
			{
				if (++RENO_NIB.route_table[i].link_fail_count > MAX_FAIL_ENDURANCE)
				{
					// the next hop is no longer vaild. thus, delete the path in the routing table
					RENO_NIB.route_table[i].is_valid = FALSE;
				}
			}
		}
	}
	return success;
}


// Note that the variable RENO_SIB.next_hop_id is not defined yet.
// Same operations for DATA forwarding is failed (MAC layer auto ack has not received).
void reno_route_lookup_fail(RENO_SIB *sib_ptr)
{
	UINT8 route_table_entry;

	switch (sib_ptr->packet.msg_type)
	{
		case RENO_DATA: 
			if (sib_ptr->prev_hop_id == RENO_NIB.id) // This node is the source of DATA
			{
				// sends RREQ
				route_table_entry = reno_send_rreq(sib_ptr->packet.dest_id);

				if (route_table_entry != ROUTE_TABLE_LOOKUP_FAILURE) // If it receives RREP
				{
					// unicast message, and update link state in routing table
					sib_ptr->next_hop_id = RENO_NIB.route_table[route_table_entry].next_hop_id;		
					reno_link_state_update(sib_ptr->next_hop_id, reno_send_to_mac(sib_ptr, UNICAST));
					break;
				}
				else // If it does not receive RREP
				{
#ifdef MONITOR
					call_monitor(sib_ptr, 2);
#endif
				}

			}
			else // This node is NOT the source of DATA.				
			{
				reno_send_rerr(sib_ptr->packet.src_id, sib_ptr->prev_hop_id, sib_ptr->packet.dest_id);
				/*
				if (!reno_send_rerr(sib_ptr->packet.src_id, sib_ptr->prev_hop_id, sib_ptr->packet.dest_id)) // sends RERR to source backwards.
				{
					reno_delete_next_route_entry(sib_ptr->prev_hop_id); // Delete some routing table entries.
				} */
			}
			break;

		case RENO_RREQ:
			if (reno_insert_route_entry(sib_ptr)) // returns the sib_ptr is new or not (aging by ping-pong)
			{
#ifdef MONITOR
				rreq_for_monitor (sib_ptr);	// for monitoring			
#endif
				reno_send_to_mac(sib_ptr, BROADCAST); // broadcasting if it's new.
			}
			break;

		case RENO_RREP:
			reno_insert_route_entry(sib_ptr);
			// Do not try to send RREP.
			// It's OK. If the source of RREQ packet does not receive RREP, it sends RREQ again. 
			break;

		case RENO_RERR:	
			// Deletes the broken link to the destination that is RERR payload (destination of DATA) from routing table.
			reno_delete_dest_route_entry(*(UINT8*)sib_ptr->packet.payload_ptr);
			// do not try to send RERR.
			// little problem : The source of DATA packet may not notice DATA packet loss.
			break;

		default:
			break;
	}
}


void reno_route_lookup_success(RENO_SIB* sib_ptr)
{	
	switch (sib_ptr->packet.msg_type)
	{
		case RENO_DATA:
			// unicast message, and update link state in routing table
			reno_link_state_update(sib_ptr->next_hop_id, reno_send_to_mac(sib_ptr, UNICAST));
			break;

		case RENO_RREQ:	
			if (reno_insert_route_entry(sib_ptr)) // returns the sib_ptr is new or not (aging by ping-pong)
			{
#ifdef MONITOR
				rreq_for_monitor (sib_ptr);	// for monitoring
#endif
				reno_send_to_mac(sib_ptr, BROADCAST); // broadcasting if it's new.
			}
			break;

		case RENO_RREP:	
			reno_insert_route_entry(sib_ptr);
			// [reno_route_lookup_fail] do nothing. 
			// So you don't need to handle RENO_SIB again in [reno_route_lookup_fail] even though [reno_send_to mac] is failed 
			// because no auto ACK has been received.
#ifdef MONITOR
				rreq_for_monitor (sib_ptr);	// for monitoring
#endif
			reno_send_to_mac(sib_ptr, UNICAST);
			break;

		case RENO_RERR:
			// Deletes the broken link to the destination that is RERR payload (destination of DATA) from routing table.
			reno_delete_dest_route_entry(*(UINT8 *)sib_ptr->packet.payload_ptr);
			// [reno_route_lookup_fail] do nothing. 
			//So you don't need to handle RENO_SIB again in [reno_route_lookup_fail] even though [reno_send_to mac] is failed because no auto ACK has been received.
			reno_send_to_mac(sib_ptr, UNICAST);
			break;

		default:
			break;
	}
}

// Function : Sends an RREQ message to destination node.
// Called by : reno_route_lookup_fail(RENO_SIB*) when it makes and sends DATA.
// Return : The routing table entry number of destination ID. If failed, returns ROUTE_TABLE_LOOKUP_FAILURE.
UINT8 reno_send_rreq(UINT8 dest_id)
{
	UINT8 rreq_trials;
	UINT8 routing_table_entry;
	RENO_SIB rreq_sib;

	rreq_sib.packet.msg_type 	= RENO_RREQ;
	rreq_sib.packet.msg_hop_limit 	= NET_DIAMETER;
	rreq_sib.packet.msg_hop_count 	= 0; 
	rreq_sib.packet.reserved 	= 0; // nothing
	rreq_sib.packet.dest_id 	= dest_id;
	rreq_sib.packet.src_id 		= RENO_NIB.id;
	rreq_sib.packet.payload_size 	= 2; //sizeof( RENO_NIB.seq_num )
	rreq_sib.packet.payload_ptr 	= (void *) &RENO_NIB.seq_num;
	RENO_NIB.seq_num++;
	
	// Broadcast the message. Retransmittion up to RENO_RREQ_TRIES
	for (rreq_trials = 1; rreq_trials <= RENO_RREQ_TRIES; ++rreq_trials)
	{
		reno_send_to_mac(&rreq_sib, BROADCAST);
		// haekim@070312
		// cannot switch context (in Interrupt Service Routine)
		if (nested_intr_cnt || intr_status.cnt )
		{
			nos_delay_ms(RENO_RREQ_WAIT_TIME*rreq_trials*rreq_trials);
			reno_recv_from_mac();
		}
		else
		{
			nos_thread_sleep_ms(RENO_RREQ_WAIT_TIME*rreq_trials*rreq_trials);
		}

		routing_table_entry = reno_scan_route_table(dest_id);			
		if ( routing_table_entry != ROUTE_TABLE_LOOKUP_FAILURE)
		{
			return routing_table_entry; // has received RREP
		}
	}

	return ROUTE_TABLE_LOOKUP_FAILURE;
}

// Sends an RREP message to destination node.
// This function is called when the node receives RREQ and it is a destination of RREQ.
BOOL reno_send_rrep(UINT8 dest_id)
{
	RENO_SIB rrep_sib;
	rrep_sib.packet.msg_type 	= RENO_RREP; //11, 0x0B
	rrep_sib.packet.msg_hop_limit 	= NET_DIAMETER; // 10, 0x0A
	rrep_sib.packet.msg_hop_count 	= 0; 
	rrep_sib.packet.reserved 	= 0; // nothing
	rrep_sib.packet.dest_id 	= dest_id;
	rrep_sib.packet.src_id 		= RENO_NIB.id;
	rrep_sib.packet.payload_size 	= 2; //sizeof( RENO_NIB.seq_num )
	rrep_sib.packet.payload_ptr 	= (void*) &RENO_NIB.seq_num;
	RENO_NIB.seq_num++;
	
	// Unicast the message
	rrep_sib.next_hop_id = RENO_NIB.route_table[reno_scan_route_table(dest_id)].next_hop_id;

	return reno_send_to_mac(&rrep_sib, UNICAST);	
}

// Sends an RERR message to destination node.
// This function is called when the DATA transmission on any node is failed.
// We should know about a previous neighbor node from MAC layer source address
// because the node may not have the path to the source of DATA message.
BOOL reno_send_rerr(UINT8 dest_id, UINT8 next_hop_id, UINT8 err_node)
{
	RENO_SIB rerr_sib;

	rerr_sib.packet.msg_type 	= RENO_RERR;
	rerr_sib.packet.msg_hop_limit 	= NET_DIAMETER;
	rerr_sib.packet.msg_hop_count 	= 0; 
	rerr_sib.packet.reserved 	= 0; // nothing
	rerr_sib.packet.dest_id 	= dest_id;
	rerr_sib.packet.src_id 		= RENO_NIB.id;
	rerr_sib.packet.payload_size 	= 1;
	rerr_sib.packet.payload_ptr 	= (void *) &err_node;

	// Unicast the message
	rerr_sib.next_hop_id = next_hop_id;

	return reno_send_to_mac(&rerr_sib, UNICAST);
}

// Searches a destination node in the routing table and returns the position in the routing table for it.
// If not found, return ROUTE_TABLE_LOOKUP_FAILURE
UINT8 reno_scan_route_table(UINT8 dest_id)
{
	// scans dest_id field from the latest inserted position downward
	UINT8 position = RENO_NIB.route_table_index;

	do {
		if (RENO_NIB.route_table[position].is_valid == TRUE && 
			RENO_NIB.route_table[position].dest_id == dest_id)
		{
			return position; // Returns the position of (dest_id) in the routing table.
		}
		else
		{
			if (position == 0)
				position = MAX_ROUTE_ENTRY-1; // Search circularly
			else
				position--; // Downward
		}
	} 
	while (position != RENO_NIB.route_table_index);
	return ROUTE_TABLE_LOOKUP_FAILURE; // If there is no dest_id in routing table, returns search fail indication.
}

// Note that we don't currently take hop-count into account.
// This function also returns information the packet (RENO_SIB) is fresh or not (aging by ping-pong communication). 
// Returning TRUE means that the packet is fresh and a new route entry is updated or added.
BOOL reno_insert_route_entry(RENO_SIB *sib_ptr)
{
	UINT8 position;

	// prohibit ping-pong effect
	if (sib_ptr->packet.src_id == RENO_NIB.id)
	{
		return FALSE;
	}

	// Get a position in the route table for the destination node
	position = reno_scan_route_table(sib_ptr->packet.src_id);

	// If an entry for the same destination exists (SUCCESSFUL LOOKUP), what will we do?
	if (position == ROUTE_TABLE_LOOKUP_FAILURE) // It is new routing information
	{	
		// find an empty entry in the routing table.
		position = RENO_NIB.route_table_index;
		do 
		{
			if (!RENO_NIB.route_table[position].is_valid)
			{
				break;
			}
			position = (position+1) % MAX_ROUTE_ENTRY;
		}
		while (position != RENO_NIB.route_table_index);

		// position points the invalid entry. This will be used below. 
		// RENO_NIB.route_table_index simply points the next entry.
		RENO_NIB.route_table_index = (position+1) % MAX_ROUTE_ENTRY;
	}
	else // if there is already a path to destination in the routing table
	{
		// 1. check sequence number in the routing path.
		// For the same sequence number, it is simply dropped because it is considered as duplicated packet (at 2nd or 3rd RREQ tries).
		if (*(UINT16 *)sib_ptr->packet.payload_ptr == RENO_NIB.route_table[position].dest_seq_num)
		{
			if ((sib_ptr->packet.src_id == RENO_NIB.route_table[position].dest_id) && \
				(sib_ptr->prev_hop_id == RENO_NIB.route_table[position].next_hop_id))
			{
				// In this case, we don't need to insert the route entry because it is completely duplicated.
				// But we should send RREP again (because this would be requested by retransmission of RREQ packet)
				// In that sense, we must report that the route entry is added normally (even if it is not actually added).

				return TRUE;
			}
			else
			{
				// Droped because it's duplicated.
				// In this case, this packet is detoured and arrived lately.
				// If we don't drop this packet, the number of valid route entries are increased.
				// Thus, we will simply drop this packet to prevent a large amount of RREQ messages being kept in the network.
			return FALSE;
			}
		}

		else if ( (RENO_NIB.route_table[position].dest_seq_num - (*(UINT16*)sib_ptr->packet.payload_ptr) >= 0 )
			&& (RENO_NIB.route_table[position].dest_seq_num - (*(UINT16*)sib_ptr->packet.payload_ptr) <= 2 ) )
		{
			// Droped because it's old.
			// This packet has older routing information than one kept in the route table.
			// Thus, we don't have to add new routing information for this packet.
			return FALSE;
		}

		// elsewhere, this packet is a newly packet, so update route entry (replace the old one without searching an empty entry).
	}

	// update routing table
	RENO_NIB.route_table[position].dest_id 		= sib_ptr->packet.src_id;
	RENO_NIB.route_table[position].dest_seq_num 	= *(UINT16 *)sib_ptr->packet.payload_ptr;
	RENO_NIB.route_table[position].next_hop_id 	= sib_ptr->prev_hop_id;
	RENO_NIB.route_table[position].link_fail_count 	= 0;
	RENO_NIB.route_table[position].is_valid 	= TRUE;	

	return TRUE;
}

// Deletes an entry -> just changes VALID tag as INVALID.
void reno_delete_dest_route_entry(UINT8 dest_id)
{
	UINT8 i;

	for ( i=0; i<MAX_ROUTE_ENTRY; i++)
	{
		if (RENO_NIB.route_table[i].is_valid == TRUE 
			&& RENO_NIB.route_table[i].dest_id == dest_id)
		{
			RENO_NIB.route_table[i].is_valid = FALSE;
			return;
		}
	}
}

void reno_delete_next_route_entry(UINT8 next_hop_id)
{
	UINT8 i;

	for ( i=0; i<MAX_ROUTE_ENTRY; i++)
	{
		if (RENO_NIB.route_table[i].is_valid == TRUE && 
			RENO_NIB.route_table[i].next_hop_id == next_hop_id)
		{
			RENO_NIB.route_table[i].is_valid = FALSE;
		}
	}
}

#ifdef MONITOR
//==============================  for MONITORING  ==================================//
void  (*mon_func)(RENO_SIB* sib_ptr, UINT8 msgType) = NULL;
void set_mon_func(void (*func)(RENO_SIB* sib_ptr, UINT8 msgType))
{
	mon_func = func;
}
void rreq_for_monitor(RENO_SIB* sib_ptr)
{
	((UINT8*)sib_ptr->packet.payload_ptr)[sib_ptr->packet.payload_size++] = RENO_NIB.id;
}
void call_monitor(RENO_SIB* sib_ptr, UINT8 msgType)
{
	if (mon_func != NULL) 
		(*mon_func) (sib_ptr, msgType);
}
#endif


#ifdef UART_M
//==========================  for DEBUGING  ============================//
// returns whether this node receives packets from prev_node_id or not
// called by [void reno_handle(RENO_SIB* sib_ptr)]

void print_route_table(void)
{
	UINT8 i;
	nos_uart_puts(STDOUT, "\n\r======================= ROUTING TABLE =======================");
	nos_uart_puts(STDOUT, "\n\rDestination ID,  Next Hop ID,  Link Failure Count,  Sequence Number");
	for (i=0; i<MAX_ROUTE_ENTRY; i++)
	{
		if ( RENO_NIB.route_table[i].is_valid )
		{
			nos_uart_puts(STDOUT, "\n\r\t");			
			nos_uart_putu(STDOUT, RENO_NIB.route_table[i].dest_id);
			nos_uart_puts(STDOUT, "\t\t");
			nos_uart_putu(STDOUT, RENO_NIB.route_table[i].next_hop_id);
			nos_uart_puts(STDOUT, "\t\t");
			nos_uart_putu(STDOUT, RENO_NIB.route_table[i].link_fail_count);
			nos_uart_puts(STDOUT, "\t\t");
			nos_uart_putu(STDOUT, RENO_NIB.route_table[i].dest_seq_num);
			nos_uart_puts(STDOUT, "\n\r");
		}
	}
}

void print_sib(RENO_SIB* sib_ptr)
{
	nos_uart_puts(STDOUT, "\n\rPrevious ID : ");
	nos_uart_putu(STDOUT, sib_ptr->prev_hop_id);
	nos_uart_puts(STDOUT, "\n\rMessage type : ");
	nos_uart_putu(STDOUT, sib_ptr->packet.msg_type);
	nos_uart_puts(STDOUT, "\n\rHop limit : ");
	nos_uart_putu(STDOUT, sib_ptr->packet.msg_hop_limit);
	nos_uart_puts(STDOUT, "\n\rHop count: ");
	nos_uart_putu(STDOUT, sib_ptr->packet.msg_hop_count);
	nos_uart_puts(STDOUT, "\n\rReserved : ");
	nos_uart_putu(STDOUT, sib_ptr->packet.reserved);
	nos_uart_puts(STDOUT, "\n\rDestination ID: ");
	nos_uart_putu(STDOUT, sib_ptr->packet.dest_id);
	nos_uart_puts(STDOUT, "\n\rSource ID: ");
	nos_uart_putu(STDOUT, sib_ptr->packet.src_id);
	nos_uart_puts(STDOUT, "\n\rPayload size: ");
	nos_uart_putu(STDOUT, sib_ptr->packet.payload_size);
}

void print_nib(void)
{
	nos_uart_puts(STDOUT, "\n\rSequence Num : ");
	nos_uart_putu(STDOUT, RENO_NIB.seq_num);
}

// Usage : print_payload_int(0, sib.packet.packet_payload_size, (UINT8*)sib.packet.packet_payload);
void print_payload(INT8 payload_unit, UINT8 payload_size, void* payload)
{
	UINT8 i;
	nos_uart_puts(STDOUT, "\n\rPayload");		
	switch (payload_unit)
	{
		case 16:
			nos_uart_puts(STDOUT, "(UINT16) : ");
			payload_size /= 2;
			for (i =0; i<payload_size; i++)
			{
				nos_uart_putu(STDOUT, ((UINT16*)payload)[i]);
				nos_uart_putc(STDOUT, '\t');
			}
			break;
		case -16:	
			nos_uart_puts(STDOUT, "(INT16) : ");
			payload_size /= 2;
			for (i =0; i<payload_size; i++)
			{
				nos_uart_puti(STDOUT, ((INT16*)payload)[i]);
				nos_uart_putc(STDOUT, '\t');
			}
			break;
		case 8:
			nos_uart_puts(STDOUT, "(UINT8) : ");
			for (i =0; i<payload_size; i++)
			{
				nos_uart_putu(STDOUT, ((UINT8*)payload)[i]);
				nos_uart_putc(STDOUT, '\t');
			}
			break;
		case -8:
			nos_uart_puts(STDOUT, "(INT8) : ");
			for (i =0; i<payload_size; i++)
			{
				nos_uart_puti(STDOUT, ((INT8*)payload)[i]);
				nos_uart_putc(STDOUT, '\t');
			}
			break;
		default:
			nos_uart_puts(STDOUT, "(string) : ");
			nos_uart_puts(STDOUT, (INT8*) payload);
			break;
	}
}
#endif // UART_M

#endif // RENO_M

