//===================================================================
//
// nano_mac.c (@haekim, @jun361, @sheart)
// 
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#include "nano_mac.h"

#ifdef NANO_MAC_M
#include <stdlib.h>
#include <string.h>
#include "platform.h"
#include "arch.h"
#include "critical_section.h"
#include "spi.h"
#include "rf.h"

// the number of trial for CSMA-CA transmission
#define NMAC_MAX_ROUND	(250)

// frame type.
#define NORMAL_FRAME	(1)
#define ACK_FRAME		(2)
#define ERROR_FRAME		(3)

// Frame control field (16bit)  
// L byte (LSB first): Type(3) + Security Enabled(1) + Frmae Pending(1) + Ack request(1) + Intra PAN(1) + Reserved(1)
// H byte (LSB first): Reserved(2) + Destination addressing mode(2) + Reserved(2) + Source addressing mode (2)
// Typical FCF: 1000 1000 01x0 0001 - x: Auto ack on(1)/off(0)
// Ack request on 
#define NMAC_FCF_ACK_ON 0x8861	
#define NMAC_FCF_ACK_ON_L 0x61
#define NMAC_FCF_ACK_ON_H 0x88
// Ack request off
#define NMAC_FCF_ACK_OFF 0x8841	
#define NMAC_FCF_ACK_OFF_L 0x41	
#define NMAC_FCF_ACK_OFF_H 0x88

// Status of queue
#define NMAC_IS_RXQ_FULL() (NMAC_RX_QUEUE.nitem == NMAC_RXQ_LEN)
#define NMAC_IS_RXQ_EMPTY() (NMAC_RX_QUEUE.nitem == 0)
#define NMAC_RXQ_NITEM() (NMAC_RX_QUEUE.nitem)

// RX queue entity
typedef struct _nmac_rxq_entity
{
	UINT16 	src_addr; 		// source address
	INT8 	rssi;		//Received Signal Strength Indicator
	UINT8	corr; 		//Correlation value (unsigned 7bit) for LQI
	UINT8 	payload_length; // length of frame payload (frame:MAC, packet:Routing)
	UINT8 	payload[NMAC_MAX_PAYLOAD_SIZE]; // frame [(routing_header) + data]
} NMAC_RXQ_ENTITY;

// Rx queue
volatile static struct _nmac_rx_queue
{
	UINT8 	front, rear, nitem;
	NMAC_RXQ_ENTITY data[NMAC_RXQ_LEN];  // nmac_rx_info queue
} NMAC_RX_QUEUE;

// temporal RX frame 
volatile static struct _nmac_rx_frame
{
	UINT8	frame_length;
	UINT16	frame_ctrl_field;
	UINT8	rx_seq;
	NMAC_RXQ_ENTITY rx_qinfo;
} NMAC_RX_FRAME;

// Simple MAC environments
static struct _nmac_env
{
	UINT8	tx_seq;	// Tx : Packet sequence number.
	volatile UINT8	rx_seq;	// Rx : Packet sequence number. (distinguish duplicated packet)
	UINT16	panid;		// PAN address
	UINT16	address;	// my MAC address
	UINT8	rx_missing_cb;	// the number of missing RX callback 
	BOOL	rx_on_state;	// store rf recv state that user specified.
	UINT8	ovf_rx_len;		// total received packet length when overflow is idicated.
	BOOL	in_ovf_handle_state;	// am I now handling overflow??
#ifdef DEMO_MAC_M
	UINT16	min_permit_addr;
	UINT16	max_permit_addr;
#endif
} NMAC_ENV;


//-------------------------------for testing--------------------------
#ifdef DEMO_MAC_M
// This must be called after "nmac_init" is called.
void nmac_set_rx_range(UINT16 min_addr, UINT16 max_addr)
{
	NMAC_ENV.min_permit_addr = min_addr;
	NMAC_ENV.max_permit_addr = max_addr;
}

static BOOL nmac_rx_permit(UINT16 src_addr)
{
	if ( src_addr >= NMAC_ENV.min_permit_addr && src_addr <= NMAC_ENV.max_permit_addr )
	{
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}
#endif // DEMO_MAC_M
//---------------------------end of testing--------------------------

static void (*nmac_rx_callback)(void);
static void nmac_exe_missing_cb(void);
static void nmac_rx_intr_handler(void);
static void nmac_readout_rxbuf(void);
static UINT8 nmac_read_frame(void);
static void nmac_write_frame(NMAC_TX_INFO* nmac_tx_info_ptr, BOOL ack_req);
static BOOL nmac_csma_ca_tx_trial(void);


// Initialization. Set channel, PAN addr, addr and RX interrupt callback. Turn on RF and RX interrupt.
void nmac_init(UINT8 channel, UINT16 panid, UINT16 myaddr) 
{
NOS_ENTER_CRITICAL_SECTION();
	CC2420_SETREG(CC2420_MDMCTRL1, 0x0500); // Set the correlation threshold = 20 (must be. Do not change)
	CC2420_SETREG(CC2420_SECCTRL0, 0x01C4); // Turn off "RXFIFO_PROTECTION(bit9)" 
	CC2420_SETREG(CC2420_IOCFG0,   0x007F); // Set the FIFOP threshold to maximum
	CC2420_AUTOACK_REP_ON(); // Turn on automatic packet acknowledgment

	// Set environments
	cc2420_channel_init(channel); // Set the RF channel
	srand(myaddr);
	NMAC_ENV.tx_seq = rand()%256; // starts from random sequece number to avoid same ack sequence number
	NMAC_ENV.rx_seq = NMAC_ENV.tx_seq;
	NMAC_ENV.rx_missing_cb = 0;
	NMAC_ENV.panid = panid;
	NMAC_ENV.address = myaddr;
	NMAC_ENV.rx_on_state = TRUE;
	NMAC_ENV.in_ovf_handle_state = FALSE;

	// Set RX queue
	NMAC_RX_QUEUE.front = 0;
	NMAC_RX_QUEUE.rear = 0;
	NMAC_RX_QUEUE.nitem = 0;


	// Write the short address and the PAN ID to the CC2420 RAM (requires that the XOSC is on and stable)
	CC2420_WRITE_RAM( &myaddr, CC2420RAM_SHORTADDR, 2);
	CC2420_WRITE_RAM( &panid, CC2420RAM_PANID, 2);
	CC2420_SWITCH_ON();

	// Set RF interrupt callback function
	nos_set_rf_cb(nmac_rx_intr_handler);
	nmac_set_rx_cb(NULL);

	ENABLE_FIFOP_INTR();
	CLEAR_FIFOP_INTR();

NOS_EXIT_CRITICAL_SECTION();    
}

// Sets TX power. Default :31 (MAX)  Minimum:1
void nmac_set_tx_power(UINT8 level)
{
	CC2420_SET_TX_LEVEL(level);	
}

// Enables the CC2420 receiver (and the FIFOP interrupt).
void nmac_rx_on(void) 
{
NOS_ENTER_CRITICAL_SECTION();
	CC2420_SWITCH_ON();
	NMAC_ENV.rx_on_state = TRUE;	// RX on state.
NOS_EXIT_CRITICAL_SECTION();	
} 


// Disables the CC2420 receiver (and the FIFOP interrupt).
void nmac_rx_off(void) 
{
NOS_ENTER_CRITICAL_SECTION();
	CC2420_SWITCH_OFF();
	NMAC_ENV.rx_on_state = FALSE;	// RX off state.
NOS_EXIT_CRITICAL_SECTION();		
} 

void nmac_set_rx_cb(void (*func)(void))
{
NOS_ENTER_CRITICAL_SECTION();	
	// Set RX queue
	NMAC_RX_QUEUE.front = 0;
	NMAC_RX_QUEUE.rear = 0;
	NMAC_RX_QUEUE.nitem = 0;
	NMAC_ENV.rx_missing_cb = 0;
	nmac_rx_callback = func;
NOS_EXIT_CRITICAL_SECTION();
}


BOOL nmac_rx(NMAC_RX_INFO* nmac_rx_info_ptr)
{
	NOS_ENTER_CRITICAL_SECTION();
	ENABLE_FIFOP_INTR();	// CC2420_AUTOACK_REP_ON();
	if ( !NMAC_IS_RXQ_EMPTY() )
	{
		// remove an item from RX queue.
		nmac_rx_info_ptr->src_addr = NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.front].src_addr;
		nmac_rx_info_ptr->rssi = NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.front].rssi;
		nmac_rx_info_ptr->corr = NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.front].corr;
		nmac_rx_info_ptr->payload_length = NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.front].payload_length;
		memcpy( nmac_rx_info_ptr->payload_ptr, (void*)(NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.front].payload), nmac_rx_info_ptr->payload_length);
		NMAC_RX_QUEUE.front = (NMAC_RX_QUEUE.front+1)%NMAC_RXQ_LEN;
		--NMAC_RX_QUEUE.nitem;

		NOS_EXIT_CRITICAL_SECTION();
		return TRUE;
	}
	else
	{
		NOS_EXIT_CRITICAL_SECTION();
		return FALSE;
	}
}


// This is called from FIFOP interrupt handler only.
static void nmac_exe_missing_cb(void)
{
	while (NMAC_ENV.rx_missing_cb)
	{
		if (nmac_rx_callback)
		{
			--NMAC_ENV.rx_missing_cb;
			nmac_rx_callback();
		}
		else
		{
			NMAC_ENV.rx_missing_cb = 0;
			return;
		}
	}
}


// FIFOP interrupt handler. Callback function will be executed here until RX queue is empty.
static void nmac_rx_intr_handler(void)
{
	while(RF_FIFOP_IS_SET())
	{
		//========================= Overflow handling swamp =============================//
		if( !(RF_FIFO_IS_SET()) )
		{
			// to block infinite loop
			NMAC_ENV.ovf_rx_len = 0;
			NMAC_ENV.in_ovf_handle_state = TRUE;
			while( RF_FIFOP_IS_SET() && (NMAC_ENV.ovf_rx_len < NMAC_MAX_PAYLOAD_SIZE) )	// overflow may be handled in TX functino in callback function
			{
				if (NMAC_IS_RXQ_FULL())
				{
					if (nmac_rx_callback)
					{
						if (NMAC_ENV.rx_missing_cb)
						{
							--NMAC_ENV.rx_missing_cb;
							nmac_rx_callback();		// callback may flush RXFIFO (in TX function)
							continue;
						}
					}
					else
					{
						NMAC_ENV.rx_missing_cb=0;
					}
					NMAC_ENV.in_ovf_handle_state = FALSE;
					CC2420_STROBE(CC2420_SFLUSHRX);
					CC2420_STROBE(CC2420_SFLUSHRX);
					// if rx_queue is full
					if ( NMAC_IS_RXQ_FULL() )
					{
						DISABLE_FIFOP_INTR();	//CC2420_AUTOACK_REP_OFF();
					}
					return;
				}
				//RXQ is not full, and RXFIFO is not empty
				if ( nmac_read_frame() ==NORMAL_FRAME)
				{
					++NMAC_ENV.rx_missing_cb;
				}
			}
			NMAC_ENV.in_ovf_handle_state = FALSE;
			CC2420_STROBE(CC2420_SFLUSHRX);
			CC2420_STROBE(CC2420_SFLUSHRX);
			nmac_exe_missing_cb();
			// if rx_queue is full
			if ( NMAC_IS_RXQ_FULL() )
			{
				DISABLE_FIFOP_INTR();	//CC2420_AUTOACK_REP_OFF();
			}
			return;
		}
		//============================================ =============================//
		
		else
		{
			if (NMAC_IS_RXQ_FULL())
			{
				if (nmac_rx_callback)
				{
					if (NMAC_ENV.rx_missing_cb)
					{
						--NMAC_ENV.rx_missing_cb;
						nmac_rx_callback();
						continue;
					}
				}
				else
				{
					NMAC_ENV.rx_missing_cb=0;
				}
				CC2420_STROBE(CC2420_SFLUSHRX);
				CC2420_STROBE(CC2420_SFLUSHRX);
				// if rx_queue is full
				if ( NMAC_IS_RXQ_FULL() )
				{
					DISABLE_FIFOP_INTR();	//CC2420_AUTOACK_REP_OFF();
				}
				return;
			}
			//RXQ is not full, and RXFIFO is not empty
			if ( nmac_read_frame() ==NORMAL_FRAME)
			{
				++NMAC_ENV.rx_missing_cb;
			}
		}
	}

	// RXFIFO is empty now. 
	nmac_exe_missing_cb();
	// if rx_queue is full
	if ( NMAC_IS_RXQ_FULL() )
	{
		DISABLE_FIFOP_INTR();	//CC2420_AUTOACK_REP_OFF();
	}
}


// This is called in csma_tx_trial function. Fill RX queue frome RXFIFO and handle overflow.
// Callback function does not called here because it might be a recursion calling.
static void nmac_readout_rxbuf(void)
{
	while(RF_FIFOP_IS_SET())
	{
		// overflow is indicated
		if( !(RF_FIFO_IS_SET()) )
		{
			if (!NMAC_ENV.in_ovf_handle_state)
			{
				// to block infinite loop
				NMAC_ENV.ovf_rx_len = 0;
			}
			while( (!NMAC_IS_RXQ_FULL()) && (NMAC_ENV.ovf_rx_len < NMAC_MAX_PAYLOAD_SIZE) )
			{
				if ( nmac_read_frame() ==NORMAL_FRAME)
				{
					++NMAC_ENV.rx_missing_cb;	// we cannot register to execute callback function while TX... :(
				}
			}
			CC2420_STROBE(CC2420_SFLUSHRX);
			CC2420_STROBE(CC2420_SFLUSHRX);
			return;
		}
		else
		{
			if(!NMAC_IS_RXQ_FULL())
			{
				if ( nmac_read_frame() ==NORMAL_FRAME)
				{
					++NMAC_ENV.rx_missing_cb;	// we cannot register to execute callback function while TX... :(
				}
			}
			else
			{
				// there are frames in RXFIFO and no overflow is indicated. RX queue is full.
				return;
			}
		}
	}
}



// Call this function to check Ack frame. (it's okay to flush RXFIFO before checking received frame is Ack or not.)
static UINT8 nmac_read_frame(void)
{
	// Read MAC frame length in PHY header.
	CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.frame_length); //read 1byte from RXFIFO.
	CC2420_GET_FRAME_LENGTH(NMAC_RX_FRAME.frame_length); // Ignore MSB to get length.
	NMAC_ENV.ovf_rx_len += NMAC_RX_FRAME.frame_length+1;

	// Ignore the packet if the length is too short
	if (NMAC_RX_FRAME.frame_length < CC2420_ACK_PACKET_SIZE)
	{
		CC2420_READ_FIFO_GARBAGE(NMAC_RX_FRAME.frame_length);	// drop a frame
    	return ERROR_FRAME;
	}
	// Read the frame control field
	CC2420_READ_FIFO((UINT8*)&(NMAC_RX_FRAME.frame_ctrl_field), 2);
	// Read the data sequence number
	CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.rx_seq);

	//------------------------- Ack frame check. ------------------------//
	if ( (NMAC_RX_FRAME.frame_length == CC2420_ACK_PACKET_SIZE) && (NMAC_RX_FRAME.frame_ctrl_field == CC2420_ACK_FCF) )  
	{
		// Read RSSI
		CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.rx_qinfo.rssi);

		// Read the Correlation and CRC
		CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.rx_qinfo.corr);

		// Ack frame is mine? CRC check is OK? 
		if (NMAC_RX_FRAME.rx_qinfo.corr & CC2420_FCS_CRC_BM)
		{
			if (NMAC_RX_FRAME.rx_seq == NMAC_ENV.tx_seq)
			{
				return ACK_FRAME;	// Indicate the successful Ack reception
			}
			else
			{
				return FAIL;	// Ack not for me
			}
		}
		else
		{
			return ERROR_FRAME;	// Ack CRC check has been failed.
		}
	}
	//--------------------- End of Ack frame check. ----------------------//

	// Duplicated data packet
	if( NMAC_RX_FRAME.rx_seq == NMAC_ENV.rx_seq)
	{
		CC2420_READ_FIFO_GARBAGE(NMAC_RX_FRAME.frame_length - 3);	// drop a frame
		return FAIL;
	}
	
	/* 
	//Rx queue is full 
	else if (NMAC_IS_RXQ_FULL())
	{
		CC2420_READ_FIFO_GARBAGE(NMAC_RX_FRAME.frame_length - 3);	// drop a frame
		return FAIL;
	}
	*/
	
	//Invalid packet or unknown FCF
	else if ( (NMAC_RX_FRAME.frame_length < NMAC_FRAME_OVERHEAD_SIZE)
		|| ((NMAC_RX_FRAME.frame_ctrl_field & (~CC2420_FCF_ACK_BM)) != NMAC_FCF_ACK_OFF) )
	{
		CC2420_READ_FIFO_GARBAGE(NMAC_RX_FRAME.frame_length - 3);	// drop a frame
		return ERROR_FRAME;
	}
	
	// Skip the destination PAN id(2) and address(2) (that's taken care of by harware address recognition!)
	CC2420_READ_FIFO_GARBAGE(4);

	// Read the source address
	CC2420_READ_FIFO((UINT8*) &NMAC_RX_FRAME.rx_qinfo.src_addr, 2);

#ifdef DEMO_MAC_M
	if( !nmac_rx_permit(NMAC_RX_FRAME.rx_qinfo.src_addr) )
	{
		CC2420_READ_FIFO_GARBAGE(NMAC_RX_FRAME.frame_length - 9);	// drop a frame
		return FAIL;
	}
#endif


	// Calculate the length of payload.
	NMAC_RX_FRAME.rx_qinfo.payload_length = NMAC_RX_FRAME.frame_length -NMAC_FRAME_OVERHEAD_SIZE; 

	// Read the packet payload
	CC2420_READ_FIFO((UINT8*) NMAC_RX_FRAME.rx_qinfo.payload, NMAC_RX_FRAME.rx_qinfo.payload_length);

	// Read RSSI
	CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.rx_qinfo.rssi);

	// Read the Correlation and CRC
	CC2420_READ_FIFO_BYTE(NMAC_RX_FRAME.rx_qinfo.corr);

	// Insert MAC packet to Rx queue if the CRC is OK
	if ( NMAC_RX_FRAME.rx_qinfo.corr & CC2420_FCS_CRC_BM )
	{
		// Remember last received data packet sequence number.
		NMAC_ENV.rx_seq = NMAC_RX_FRAME.rx_seq;		
		// Get the Correlation value (7bit)
		NMAC_RX_FRAME.rx_qinfo.corr &= CC2420_FCS_CORR_BM;


		//=========== Insert a frame to the RX queue=================//
		// The rx_queue cannot be full. (already have been checked.)
		++NMAC_RX_QUEUE.nitem;
		memcpy( (NMAC_RXQ_ENTITY*)&(NMAC_RX_QUEUE.data[NMAC_RX_QUEUE.rear]), (NMAC_RXQ_ENTITY*)&(NMAC_RX_FRAME.rx_qinfo), sizeof(NMAC_RXQ_ENTITY) );
		NMAC_RX_QUEUE.rear = (NMAC_RX_QUEUE.rear+1)%NMAC_RXQ_LEN;
		//================================================//
		return NORMAL_FRAME;
	}
	else
	{
		return ERROR_FRAME;	// CRC fail
	}
}



// Write a frame to the TX FIFO (the FCS is sended automatically when AUTOCRC is enabled)
static void nmac_write_frame(NMAC_TX_INFO* nmac_tx_info_ptr, BOOL ack_req)
{
	UINT8 count;

   	SPI_ENABLE();
	CC2420_TX_ADDR(CC2420_SFLUSHTX);
	CC2420_TX_ADDR(CC2420_TXFIFO);
	SPI_TX(nmac_tx_info_ptr->payload_length + NMAC_FRAME_OVERHEAD_SIZE);
	if (ack_req)
	{
		SPI_TX(NMAC_FCF_ACK_ON_L);
		SPI_TX(NMAC_FCF_ACK_ON_H);
	}
	else
	{
		SPI_TX(NMAC_FCF_ACK_OFF_L);	// autoack request off
		SPI_TX(NMAC_FCF_ACK_OFF_H);
	}
	SPI_TX(NMAC_ENV.tx_seq);
	SPI_TX( ((UINT8*)(&(NMAC_ENV.panid)))[0] );
	SPI_TX( ((UINT8*)(&(NMAC_ENV.panid)))[1] );
	SPI_TX( ((UINT8*)(&(nmac_tx_info_ptr->dest_addr)))[0] );
	SPI_TX( ((UINT8*)(&(nmac_tx_info_ptr->dest_addr)))[1] );
	SPI_TX( ((UINT8*)(&(NMAC_ENV.address)))[0] );
	SPI_TX( ((UINT8*)(&(NMAC_ENV.address)))[1] );
#ifdef ROUTING_M
	for (count=0; count < nmac_tx_info_ptr->routing_header_length; ++count)
	{
		SPI_TX( ((UINT8*) nmac_tx_info_ptr->routing_header_ptr)[count] );
	}
	for (count=0; count < nmac_tx_info_ptr->payload_length - nmac_tx_info_ptr->routing_header_length; ++count)
	{
		SPI_TX( ((UINT8*) nmac_tx_info_ptr->payload_ptr)[count] );
	}
#else
	for (count=0; count < nmac_tx_info_ptr->payload_length; ++count)
	{
		SPI_TX( ((UINT8*) nmac_tx_info_ptr->payload_ptr)[count] );
	}
#endif		
	SPI_DISABLE();
}


// CSMA-CA TX trial. It handles RX frames and always flushes RXFIFO before returning.
static BOOL nmac_csma_ca_tx_trial(void)
{
   	UINT8 spi_status, i;

	// wait until RX is stabilized (8symbol period after RX has been turned on)
	CC2420_UPD_STATUS(spi_status);
	if (!(spi_status & (1 << CC2420_RSSI_VALID)))
		nos_delay_us(8*CC2420_SYMBOL_DURATION);
	
	//--------------------------------------- PHY TX(CSMA-CA) : start --------------------------------------------------//
	// (8symbol) --> CCA check  --> delay (12symbol) --> (8symbol) --> CCA check  --> TX_CALIBRATE state (12 (or 8)symbol) --> TX preamble(8symbol) --> sfd(2symbol), length, frame
	//                     |->check 'CCA' pin                                                       |-> STXONCCA                                                                                                          |->check 'TX_ACTIVE'
	//Both CCA pin and TX_ACTIVE (STXONCCA) is related to 'RSSI.CCA_THR'
	for (i =0; i<NMAC_MAX_ROUND; ++i) //while (TRUE)
	{
		//To confirm CCA, RX buf must not be overflowed.
		nmac_readout_rxbuf();
		
		// 1st CCA
		if (RF_CCA_IS_SET())
		{
			// wait 8(cca check)+12(transition) symbol period
			// 2007.11.29 haekim nos_delay_us(20*CC2420_SYMBOL_DURATION);
			nos_delay_us(8*CC2420_SYMBOL_DURATION);

			// 2nd CCA, TX begins after the CCA check has passed
			CC2420_STROBE(CC2420_STXONCCA);
			/* haekim, 2007.07.19 
			// wait 12 symbol periods for TX_CALIBRATE state and 8 symols for TX_PREAMBLE state
			nos_delay_us(20*CC2420_SYMBOL_DURATION);	// wait for TX_ACTIVE bit updated. (should be lower than 800us)
			*/
			CC2420_UPD_STATUS(spi_status);
			CC2420_UPD_STATUS(spi_status);	// for msp. We should call this twice. dunno why.
			if (spi_status & (1 << CC2420_TX_ACTIVE))	//is CC2420 in TX state?
			{
				while (!RF_SFD_IS_SET());	// waiting for TX beginning
				// ...> Transmitting ...>
				while (RF_SFD_IS_SET());		// Wait for TX completing

				/* 
				//check an underflow 
				CC2420_UPD_STATUS(spi_status);
				if (spi_status & (1 << CC2420_TX_UNDERFLOW))	
				{
					CC2420_STROBE(CC2420_SFLUSHTX);
					return FALSE;
				}
				*/
				return TRUE;
			}
		}
		// waiting 1 CCA time to avoid collision
		nos_delay_us( 20*CC2420_SYMBOL_DURATION );
	}
	return FALSE;
}


// TX without ack request. Always returns TRUE.
BOOL nmac_tx_noack (NMAC_TX_INFO* nmac_tx_info_ptr)
{
	BOOL tx_success;

	// Turn off interrupt.
	NOS_ENTER_CRITICAL_SECTION();

	//NMAC_ENV.is_tx_state = TRUE;

	// if RF module was turned off, turn it on.
	if (!NMAC_ENV.rx_on_state)
	{
		CC2420_SWITCH_ON();
	}	

	 // Initialize TX frame
	++NMAC_ENV.tx_seq;

	// Do not check underflow
	// write a frame to TX FIFO, no ack request (if TXFIFO underflow occurred)
	nmac_write_frame(nmac_tx_info_ptr, FALSE);	
	tx_success = nmac_csma_ca_tx_trial();

	/*
	// CSMA-CA TX trial until TXFIFO underflow does not occur.
	do
	{
		// write a frame to TX FIFO, no ack request (if TXFIFO underflow occurred)
		nmac_write_frame(nmac_tx_info_ptr, FALSE);
	}while( !nmac_csma_ca_tx_trial() );
	*/
	
    // Turn off RF module again if it was being turned off.
	if (!NMAC_ENV.rx_on_state) 
	{
		CC2420_SWITCH_OFF();
	}
	NOS_EXIT_CRITICAL_SECTION();

	return tx_success;		// Always returns true.
}


BOOL nmac_tx(NMAC_TX_INFO* nmac_tx_info_ptr) 
{
	UINT8 retrans_trial;
	BOOL tx_success;

	// Turn off interrupt.
	NOS_ENTER_CRITICAL_SECTION();

	//NMAC_ENV.is_tx_state = TRUE;
	tx_success = FAIL;
	
	// if RF module was turned off, turn it on.
	if (!NMAC_ENV.rx_on_state)
	{
		CC2420_SWITCH_ON();
	}

	 // Initialize TX frame
	++NMAC_ENV.tx_seq;

	// write a frame to TX FIFO, Ack request
	nmac_write_frame(nmac_tx_info_ptr, TRUE);
	
	for (retrans_trial = 0; retrans_trial < NMAC_MAX_TX_TRIALS; ++retrans_trial)
	{
		// Do not check underflow
		// write a frame to TX FIFO, no ack request (if TXFIFO underflow occurred)
		nmac_csma_ca_tx_trial();
		/*
		// CSMA-CA TX trial until TXFIFO underflow does not occur.
		while( !nmac_csma_ca_tx_trial() )
		{
			// write a frame to TX FIFO again if TXFIFO underflow has occurred. Ack request
			nmac_write_frame(nmac_tx_info_ptr, TRUE);
		}
		*/
		
		//RX fifo must be empty. 
		CC2420_STROBE(CC2420_SFLUSHRX);
		CC2420_STROBE(CC2420_SFLUSHRX);

		//---------------- Ack RX waiting : start -------------------//
		//ACK arrive time: 544us = 12symbol (Non-beacon mode Ack TX time after receiving packet==TX->RX coverting time) + 22symbol (preamble+SFD+ACK frame)
		nos_delay_us(12*CC2420_SYMBOL_DURATION+CC2420_ACK_DURATION);
		// small margin : 96us (in real : 12us(556us))
		nos_delay_us(6*CC2420_SYMBOL_DURATION);
		//---------------- Ack RX waiting : end -------------------//

		// ACK has been received
		if (RF_FIFOP_IS_SET())
		{
			if (nmac_read_frame() == ACK_FRAME )
			{
				tx_success = SUCCESS;
				break;
			}
		}

		// ACK has not received.
		// one of 5 slot is selected randomly for backoff.
		// 1slot : each CCA takes 20 symbol period.
		if (retrans_trial < NMAC_MAX_TX_TRIALS-1)
		{
			nos_delay_us((((UINT8)rand())%10+2) * (20*CC2420_SYMBOL_DURATION));
		}
	}
   	// Turn off RF module again if it was being turned off.
	if (!NMAC_ENV.rx_on_state) 
	{
		CC2420_SWITCH_OFF();
	}
	NOS_EXIT_CRITICAL_SECTION();

	return tx_success;
}


#endif //--- NANO_MAC_M---//

