//===================================================================
//
// thread_create_exit.c (@sheart, @jun361)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "thread.h"

#ifdef KERNEL_M
#include "arch.h"
#include "platform.h"
#include "critical_section.h"
#include "heap.h"
#include "sched.h"
#include "hal_sched.h"

#ifdef SWAPSTK_M
//extern UINT16	   _common_stack_data_size;
extern STACK_PTR  _common_stack_index;
extern STACK_PTR  _thread_stack_index;
#endif

extern THREADQ _rdy_q[PRIORITY_LEVEL_COUNT];
extern UINT8 	_ready_priority_bit;	// priority bit setting for threads, which is set for the priority of the thread when the thread becomes in ready state.

static void function_start(void)
{
	// pre-process if any

	// execute the corresponding function by the thread
	(tcb[_rtid]->func)(tcb[_rtid]->args_data);

	// post-process
	nos_thread_exit();
}

#ifdef THREAD_MANUAL_NUMBERING_M
UINT8 nos_thread_create(UINT8 tid, void (*func)(void *args), void *args_data, UINT16 stack_size, UINT8 priority)
{
	UINT8 align;

	NOS_ENTER_CRITICAL_SECTION();
	// Error handling when creating a thread with an existing tid 
	// duplicated id is used
	if (NOS_THREAD_IS_CREATED(tid))
	{
		NOS_EXIT_CRITICAL_SECTION();
		return THREAD_ID_DUPLICATE_ERROR;
	}
#else
INT8 nos_thread_create(void (*func)(void *args), void *args_data, UINT16 stack_size, UINT8 priority)
{
	UINT8 tid;
	UINT8 align;

	NOS_ENTER_CRITICAL_SECTION();
#ifndef THREAD_EXT_M
	if (_created_tid_bit == 0x3f) // full in slots
#else
	if (_created_tid_bit == 0xffff) // full in slots
#endif
	{
		NOS_EXIT_CRITICAL_SECTION();
		return THREAD_ID_EXHAUST_ERROR;
	}
    for (tid=1; tid<MAX_NUM_TOTAL_THREAD; ++tid)
   	{
		if (~_created_tid_bit & (1 << tid))
		{
			break; // found an empty slot
		}
   	}
#endif


	// Error handling when priority is given 0 or higher than defined (0 is reserved for the idle thread)
	if (priority == 0 || priority >= PRIORITY_LEVEL_COUNT)
	{
		NOS_EXIT_CRITICAL_SECTION();
		return THREAD_PRIORITY_OUT_OF_RANGE_ERROR;
	}

	// Record the tid of this thread in the _created_tid_bit
	_BIT_SET(_created_tid_bit, tid);
	
	// tcb (thread control block) settings when creating thread
	// A thread has id (*ptid), initialized stack memory, stack pointer and the number of sleeping ticks (sleep_tick) if sleeping.
#ifdef SWAPSTK_M
	stack_size = DEFAULT_STACK_SIZE;
#else
	stack_size += DEFAULT_STACK_SIZE;
#endif
	align = stack_size % (sizeof(STACK_ENTRY));
	if ( align )
	{
		stack_size = stack_size + sizeof(STACK_ENTRY) - align;
	}

	tcb[tid] = nos_malloc(sizeof(struct _tcb)); 
	tcb[tid]->stack_start_addr = nos_malloc(stack_size);
	tcb[tid]->sptr = nos_tcb_stack_init(function_start, tcb[tid]->stack_start_addr, stack_size);
	tcb[tid]->stack_size = stack_size;
	tcb[tid]->sleep_tick 	  = 0;
	tcb[tid]->func		  = func;
	tcb[tid]->args_data	  = args_data;
	tcb[tid]->id = tid;		//dummy data.
	THREADQ_INIT(tcb[tid]->waitq); // for nos_thread_join(..)

	// This thread becomes in ready state (inserted into ready_queue).
	// Three steps (in terms of the thread priority) :
	//   1) find a ready queue for the priority of this thread
	//   2) insert tid of this thread into the ready queue
	//   3) Set the ready priority bit for the priority of this thread

	THREADQ_ENQ(_rdy_q[priority], tid);
	tcb[tid]->state = READY_STATE;
	tcb[tid]->priority = priority;

	_BIT_SET(_ready_priority_bit, priority);

NOS_EXIT_CRITICAL_SECTION();
#ifdef THREAD_MANUAL_NUMBERING_M
	return THREAD_NO_ERROR;
#else
	return tid;
#endif
}

UINT8 nos_thread_exit()
{
	UINT8 i, k;

	// The idle thread cannot die or You cannot exit the thread in critical section.
	if (_rtid == 0 || !NOS_IS_CTX_SW_ALLOWABLE()) 
		return THREAD_EXIT_ERROR; // ignore it

	NOS_DISABLE_GLOBAL_INTERRUPT();
	// this thread will die
	_BIT_CLR(_created_tid_bit, _rtid);

	// the waiting threads to be joined that need to be awaken at this point
	while (!THREADQ_IS_EMPTY(tcb[_rtid]->waitq))
	{
		THREADQ_FETCH(tcb[_rtid]->waitq, &i); //i is a fetched thread id
		k = tcb[i]->priority;
		tcb[i]->state = READY_STATE;
		THREADQ_ENQ(_rdy_q[k], i);
		_BIT_SET(_ready_priority_bit, k);
	}
	// Because this thread is currently running, it is not in the ready queue 
	// So, we do not have to delete it from the ready_priority_bit and ready queue 
	// THREADQ_DEQ(_rdy_q[tcb[_rtid]->priority], _rtid);
	// if (THREADQ_IS_EMPTY(_rdy_q[tcb[_rtid]->priority]))
	//	_BIT_CLR(_ready_priority_bit, tcb[_rtid]->priority);

	nos_free(tcb[_rtid]->stack_start_addr); // dealloate the stack of this thread
	nos_free(tcb[_rtid]); // deallocate tcb memory for this thread

	// let this thread be in EXIT_STATE and then context switching.
	// this state will not make nos_ctx_sw() store the current context.
	tcb[_rtid]->state = EXIT_STATE;


        //===================Section for thread restoring  ===================//
        //Scheduling : Preemption RR scheduling
        
        // find a thread with highest priority
        for (_Gi=PRIORITY_LEVEL_COUNT-1; _Gi>0; _Gi--)
        {
                if (_ready_priority_bit & (1 << _Gi)) break; // found 
        }       
        
        // get the thread id of the thread with highest priority
        THREADQ_FETCH(_rdy_q[_Gi], &_rtid);
        if (THREADQ_IS_EMPTY(_rdy_q[_Gi])) 
                _BIT_CLR(_ready_priority_bit, _Gi); // because the thread queue for this priority is empty, the corresponding bit in _ready_priority_bit is cleared. 
                
        // the selected thread will run soon after restoring context
        tcb[_rtid]->state = RUNNING_STATE;
        
#ifdef SWAPSTK_M
        // Must not use memcpy. The return PC for memcpy will be overritten by memcpy itself.   //memcpy((STACK_PTR)(RAMEND - tcb[_rtid]->stack_size) + 1, tcb[_rtid]->stack_start_addr, tcb[_rtid]->stack_size);
        _common_stack_index = (STACK_PTR)(RAMEND - tcb[_rtid]->stack_size ) + 1;
        _thread_stack_index = tcb[_rtid]->stack_start_addr;
        for (int i=0; i<tcb[_rtid]->stack_size; ++i)
        {
                *_common_stack_index = *_thread_stack_index;
                ++_thread_stack_index;
                ++_common_stack_index;
        }
#endif

        //Load new thread state
        NOS_THREAD_LOAD_STATE(tcb[_rtid]);
        NOS_RETURN();   // __asm__ volatile ("ret\n\t")

	// never reach here.
	return THREAD_NO_ERROR;
}
#endif
