//===================================================================
//
// msgq.c (@sheart)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================

#include "msgq.h"


#ifdef KERNEL_M
#ifdef MSGQ_M
#include "heap.h"
#include "critical_section.h"
#include "thread.h"
#include "sched.h"

#ifndef THREAD_EXT_M
#include "threadq.h"
#else
#include "threadq_ext.h"
#endif

extern UINT8 	_rtid;

MSGQ nos_msgq_create(UINT8 type, UINT8 len)
{
	MSGQ mq;

NOS_ENTER_CRITICAL_SECTION();
	mq = (MSGQ) nos_malloc(sizeof(struct _msgq));

	if (mq == NULL)
		return NULL;

	mq->front  = 0;
	mq->rear  = 0;
	mq->nitem  = 0;
	mq->len  = len;
	mq->type = type;
	
	if (type == MSGQ_UINT8)
		mq->msg = nos_malloc(sizeof(UINT8)*len);
	else if (type == MSGQ_UINT16) 
		mq->msg = nos_malloc(sizeof(UINT16)*len);
	else if (type == MSGQ_UINT32) 
		mq->msg = nos_malloc(sizeof(UINT32)*len);
	else 
		mq->msg = NULL;

	if (mq->msg == NULL)
		return NULL;
NOS_EXIT_CRITICAL_SECTION();

	return mq;
}

void nos_msgq_destroy(MSGQ mq)
{
NOS_ENTER_CRITICAL_SECTION();
	nos_free(mq->msg);
	nos_free(mq);
NOS_EXIT_CRITICAL_SECTION();
}

UINT8 nos_msgq_isend(MSGQ mq, void *data)
{
NOS_ENTER_CRITICAL_SECTION();
	if (mq->nitem == mq->len)
	{
		 NOS_EXIT_CRITICAL_SECTION();
		 return MSGQ_FULL_ERROR;
	}
	else  // send a message into queue
	{
		// copy the data into kernel memory region
		if (mq->type == MSGQ_UINT8)
			*((UINT8 *)mq->msg + mq->rear) = *((UINT8 *)data);
		else if (mq->type == MSGQ_UINT16)
			*((UINT16 *)mq->msg + mq->rear) = *((UINT16 *)data);
		else if (mq->type == MSGQ_UINT32)
			*((UINT32 *)mq->msg + mq->rear) = *((UINT32 *)data);
		else 
		{
			 NOS_EXIT_CRITICAL_SECTION();
			 return MSGQ_TYPE_MATCH_ERROR;
		}

		++mq->nitem;
		mq->rear = (mq->rear+1)%mq->len;
	
		NOS_EXIT_CRITICAL_SECTION();
		nos_ctx_sw();
	
		return MSGQ_NO_ERROR;
	}
}

UINT8 nos_msgq_irecv(MSGQ mq, void *data)
{
NOS_ENTER_CRITICAL_SECTION();
	if (mq->nitem > 0)
	{
		// copy kernel data to user memory
		if (mq->type == MSGQ_UINT8)
			*((UINT8 *)data) = *((UINT8 *) mq->msg + mq->front);
		else if (mq->type == MSGQ_UINT16)
			*((UINT16 *)data) = *((UINT16 *) mq->msg + mq->front);
		else if (mq->type == MSGQ_UINT32)
			*((UINT32 *)data) = *((UINT32 *) mq->msg + mq->front);
		else 
		{
			NOS_EXIT_CRITICAL_SECTION();
			return MSGQ_TYPE_MATCH_ERROR;
		}

		mq->front = (mq->front+1)%mq->len;
		--mq->nitem;

		NOS_EXIT_CRITICAL_SECTION();
		return MSGQ_NO_ERROR;
	}
	else // in case of mq->nitem == 0; need to wait
	{
		NOS_EXIT_CRITICAL_SECTION();

		return MSGQ_EMPTY_ERROR;
	}

NOS_EXIT_CRITICAL_SECTION();
	return MSGQ_NO_ERROR;
}

UINT8 nos_msgq_send(MSGQ mq, void *data)
{
	while (nos_msgq_isend(mq, data) == MSGQ_FULL_ERROR)
		nos_ctx_sw();

	return MSGQ_NO_ERROR;
}

UINT8 nos_msgq_recv(MSGQ mq, void *data)
{
	while (nos_msgq_irecv(mq, data) == MSGQ_EMPTY_ERROR)
		nos_ctx_sw();

	return MSGQ_NO_ERROR;
}
#endif // MSGQ_M
#endif // KERNEL_M
