//===================================================================
//
// cc2420.h	(@haekim)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#ifndef CC2420_H
#define CC2420_H
#include "kconf.h"
#ifdef CC2420_M

#include "common.h"


// If RXFIFO is full (128byte), overflow is indicated.
#define CC2420_RXFIFO_SIZE (128)

// The time it takes for the acknowledgment packet to be received after the data packet has been transmitted
#define CC2420_SYMBOL_DURATION	(32 * 0.5)
#define CC2420_ACK_DURATION		( CC2420_SYMBOL_DURATION * 2 * ((4 + 1) + (1) + (2 + 1) + (2)))

// MAC frame Length byte in PHY : The MSB has no meaning.
// Note that the length byte itself is not included included in the packet length
#define CC2420_GET_FRAME_LENGTH(x) \
	do { \
		x &= (0x7F);\
	} while (0)

// ACK packet : (frame control field, sequence number, frame check sequence)
#define CC2420_ACK_PACKET_SIZE (2+1+2) // 5 byte



// Reset CC2420
#define CC2420_RESET() \
	do { \
		CC2420_SETREG(CC2420_MAIN, 0x0000); \
		CC2420_SETREG(CC2420_MAIN, 0xF800); \
	} while (0)

// To save more energy, use POWER DOWN mode for long RF off state.
//	1. CC2420_STROBE(CC2420_SXOSCOFF);	// crystal oscillator off
//	2. CC2420_STROBE(CC2420_SXOSCON); 	// crystal oscillator on
//	    wait_until_xosc_stable(); // wait for the crystal oscillator to become stable
//	    CC2420_STROBE(CC2420_SFLUSHRX);

// Turn on RF
#define CC2420_SWITCH_ON()	\
	do {		\
		SPI_ENABLE();	\
		CC2420_TX_ADDR(CC2420_SRXON);	\
		CC2420_TX_ADDR(CC2420_SFLUSHTX);	\
		CC2420_TX_ADDR(CC2420_SFLUSHRX);	\
		CC2420_TX_ADDR(CC2420_SFLUSHRX);	\
		SPI_DISABLE();	\
	} while(0)

// Turn off RF
#define CC2420_SWITCH_OFF()	\
	do{ CC2420_STROBE(CC2420_SRFOFF); } while(0)		

// Transmission Power Level (1~31)
#define CC2420_SET_TX_LEVEL(level) \
	do{ \
		if (level >= 31) \
			CC2420_SETREG (CC2420_TXCTRL, 0xA0FF); \
		else if (level <= 1) \
			CC2420_SETREG (CC2420_TXCTRL, 0xA0E0); \
		else \
			CC2420_SETREG (CC2420_TXCTRL, 0xA0E0+level); \
	}while(0)

// autoack reply(not request) on or off.
#define CC2420_AUTOACK_REP_ON()	do{ CC2420_SETREG(CC2420_MDMCTRL0, 0x0AF2); }while(0)
#define CC2420_AUTOACK_REP_OFF()	do{ CC2420_SETREG(CC2420_MDMCTRL0, 0x0AE2); }while(0)

// Frame control field (16bit)  
// L byte (LSB first): Type(3) + Security Enabled(1) + Frmae Pending(1) + Ack request(1) + Intra PAN(1) + Reserved(1)
// H byte (LSB first): Reserved(2) + Destination addressing mode(2) + Reserved(2) + Source addressing mode (2)
#define CC2420_FCF_ACK_BM 0x0020
// Auto-Ack FCF
#define CC2420_ACK_FCF	0x0002

// Frame check sequence (RSSI + Corr : 16 bit)
// L byte : RSSI (signed)
// H byte (LSB first) : Correlation value (unsigned, 7bit) + CRC OK(1)
#define CC2420_FCS_CRC_BM 0x80
#define CC2420_FCS_CORR_BM 0x78


// CC2420 register constants
// Command strobes
#define CC2420_SNOP             0x00
#define CC2420_SXOSCON          0x01
#define CC2420_STXCAL           0x02
#define CC2420_SRXON            0x03
#define CC2420_STXON            0x04
#define CC2420_STXONCCA         0x05
#define CC2420_SRFOFF           0x06
#define CC2420_SXOSCOFF         0x07
#define CC2420_SFLUSHRX         0x08
#define CC2420_SFLUSHTX         0x09
#define CC2420_SACK             0x0A
#define CC2420_SACKPEND         0x0B
#define CC2420_SRXDEC           0x0C
#define CC2420_STXENC           0x0D
#define CC2420_SAES             0x0E

// Registers
#define CC2420_MAIN             0x10
#define CC2420_MDMCTRL0         0x11
#define CC2420_MDMCTRL1         0x12
#define CC2420_RSSI             0x13
#define CC2420_SYNCWORD         0x14
#define CC2420_TXCTRL           0x15
#define CC2420_RXCTRL0          0x16
#define CC2420_RXCTRL1          0x17
#define CC2420_FSCTRL           0x18
#define CC2420_SECCTRL0         0x19
#define CC2420_SECCTRL1         0x1A
#define CC2420_BATTMON          0x1B
#define CC2420_IOCFG0           0x1C
#define CC2420_IOCFG1           0x1D
#define CC2420_MANFIDL          0x1E
#define CC2420_MANFIDH          0x1F
#define CC2420_FSMTC            0x20
#define CC2420_MANAND           0x21
#define CC2420_MANOR            0x22
#define CC2420_AGCCTRL          0x23
#define CC2420_AGCTST0          0x24
#define CC2420_AGCTST1          0x25
#define CC2420_AGCTST2          0x26
#define CC2420_FSTST0           0x27
#define CC2420_FSTST1           0x28
#define CC2420_FSTST2           0x29
#define CC2420_FSTST3           0x2A
#define CC2420_RXBPFTST         0x2B
#define CC2420_FSMSTATE         0x2C
#define CC2420_ADCTST           0x2D
#define CC2420_DACTST           0x2E
#define CC2420_TOPTST           0x2F
#define CC2420_RESERVED         0x30

// FIFOs
#define CC2420_TXFIFO           0x3E
#define CC2420_RXFIFO           0x3F

// Memory sizes
#define CC2420_RAM_SIZE         368
#define CC2420_FIFO_SIZE        128

// Addresses
#define CC2420RAM_TXFIFO        0x000
#define CC2420RAM_RXFIFO        0x080
#define CC2420RAM_KEY0          0x100
#define CC2420RAM_RXNONCE       0x110
#define CC2420RAM_SABUF         0x120
#define CC2420RAM_KEY1          0x130
#define CC2420RAM_TXNONCE       0x140
#define CC2420RAM_CBCSTATE      0x150
#define CC2420RAM_IEEEADDR      0x160
#define CC2420RAM_PANID         0x168
#define CC2420RAM_SHORTADDR     0x16A

// Status byte
#define CC2420_XOSC16M_STABLE    6
#define CC2420_TX_UNDERFLOW      5
#define CC2420_ENC_BUSY          4
#define CC2420_TX_ACTIVE         3
#define CC2420_LOCK              2
#define CC2420_RSSI_VALID        1

// Security
// SECCTRL0.SEC_MODE (security mode)
#define CC2420_SECCTRL0_NO_SECURITY         0x0000
#define CC2420_SECCTRL0_CBC_MAC             0x0001
#define CC2420_SECCTRL0_CTR                 0x0002
#define CC2420_SECCTRL0_CCM                 0x0003

// SECCTRL0.SEC_M (number of bytes in the authentication field)
#define CC2420_SECCTRL0_SEC_M_IDX           2

// SECCTRL0.RXKEYSEL/TXKEYSEL (key selection)
#define CC2420_SECCTRL0_RXKEYSEL0           0x0000
#define CC2420_SECCTRL0_RXKEYSEL1           0x0020
#define CC2420_SECCTRL0_TXKEYSEL0           0x0000
#define CC2420_SECCTRL0_TXKEYSEL1           0x0040

// Others
#define CC2420_SECCTRL0_SEC_CBC_HEAD        0x0100
#define CC2420_SECCTRL0_RXFIFO_PROTECTION   0x0200

// MAC Security flags definitions. Note that the bits are shifted compared to the actual security flags 
// defined by IEEE 802.15.4, please see the CC2420 datasheet for details.
#define MAC_CC2420_CTR_FLAGS                0x42
#define MAC_CC2420_CCM_FLAGS                0x09


/************* CC2420: SPI interface *************/
#define CC2420_TX_ADDR(a)	SPI_TX(a)
// 0x40 = 01000000 : 6th-bit (R/W) is set (READ data)
#define CC2420_RX_ADDR(a) \
	do { \
		SPI_TX((a) | 0x40); \
		SPI_RX_REG; \
	} while (0)

/*** Register access ***/
#define CC2420_STROBE(s) \
	do { \
		SPI_ENABLE(); \
		CC2420_TX_ADDR(s); \
		SPI_DISABLE(); \
	} while (0)

// Updates the CC2420 status byte
#define CC2420_UPD_STATUS(s) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(CC2420_SNOP); \
        s = SPI_RX_REG; \
        SPI_DISABLE(); \
    } while (0)

// write to register
#define CC2420_SETREG(a,v) \
    do { \
        SPI_ENABLE(); \
        CC2420_TX_ADDR(a); \
        SPI_TX((BYTE) ((v) >> 8)); \
        SPI_TX((BYTE) (v)); \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_GETREG(a,v) \
    do { \
        SPI_ENABLE(); \
        CC2420_RX_ADDR(a); \
        SPI_RX_WORD(v); \
        nos_delay_us(1); \
        SPI_DISABLE(); \
    } while (0)


/*** FIFO access ***/
//      p = pointer to the byte array to be read/written
//      c = the number of bytes to read/write
//      b = single data byte
#define CC2420_READ_FIFO_BYTE(b) \
    do { \
        SPI_ENABLE(); \
        CC2420_RX_ADDR(CC2420_RXFIFO); \
        SPI_RX(b); \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_READ_FIFO(p,c) \
    do { \
        SPI_ENABLE(); \
        CC2420_RX_ADDR(CC2420_RXFIFO); \
        for (UINT8 spiCnt=0; spiCnt < (c); ++spiCnt) { \
            SPI_RX(((BYTE*)(p))[spiCnt]); \
        } \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_READ_FIFO_GARBAGE(c) \
    do { \
        SPI_ENABLE(); \
        CC2420_RX_ADDR(CC2420_RXFIFO); \
        for (UINT8 spiCnt=0; spiCnt < (c); ++spiCnt) { \
            SPI_RX_GARBAGE(); \
        } \
        SPI_DISABLE(); \
    } while (0)


#define CC2420_WRITE_FIFO(p,c) \
    do { \
        SPI_ENABLE(); \
        CC2420_TX_ADDR(CC2420_TXFIFO); \
        for (UINT8 spiCnt=0; spiCnt < (c); spiCnt++) { \
            SPI_TX(((BYTE*)(p))[spiCnt]); \
        } \
        SPI_DISABLE(); \
    } while (0)


/*** RAM access ***/
//  write to RAM (little-endian)
//      p = pointer to the variable to be written
//      a = the CC2420 RAM address
//      c = the number of bytes to write
//	  Example of usage:
//     	UINT16 shortAddress = 0xBEEF;
//	       CC2420_WRITE_RAM(&shortAddress, CC2420RAM_SHORTADDR, 2);
#define CC2420_WRITE_RAM(p,a,c) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX((a >> 1) & 0xC0); \
        for (UINT8 ramCnt = 0; ramCnt < (c); ++ramCnt) { \
            SPI_TX(((BYTE*)(p))[ramCnt]); \
        } \
        SPI_DISABLE(); \
    } while (0)

/*
#define CC2420_READ_RAM(p,a,c) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX(((a >> 1) & 0xC0) | 0x20); \
        for (UINT8 ramCnt = 0; ramCnt < (c); ++ramCnt) { \
            SPI_RX(((BYTE*)(p))[ramCnt]); \
        } \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_WRITE_RAM_BE(p,a,c) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX((a >> 1) & 0xC0); \
        for (UINT8 ramCnt = (c); ramCnt >= 0; --ramCnt) { \
            SPI_TX(((BYTE*)(p))[ramCnt]); \
        } \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_READ_RAM_BE(p,a,c) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX(((a >> 1) & 0xC0) | 0x20); \
        n = c; \
        for (UINT8 ramCnt = (c); ramCnt >= 0; --ramCnt) { \
            SPI_RX(((BYTE*)(p))[ramCnt]); \
        } \        
        SPI_DISABLE(); \
    } while (0)
*/

//  SPI: CC2420 RAM access (big or little-endian order)
//      p = pointer to the variable to be written
//      a = the CC2420 RAM address ( & 0x7F : cast to unsigned type )
//      c = the number of bytes to write
//      n = counter variable which is used in for/while loops (UINT8)
//
//  Example of usage:
//      UINT8 n;
//      UINT16 shortAddress = 0xBEEF;
//      SPI_WRITE_RAM_LE(&shortAddress, CC2420RAM_SHORTADDR, 2);

// these functions are used in zigbee mac
#define CC2420_WRITE_RAM_LE(p,a,c,n) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX((a >> 1) & 0xC0); \
        for (n = 0; n < (c); n++) { \
            SPI_TX(((BYTE*)(p))[n]); \
        } \
        SPI_DISABLE(); \
    } while (0)

#define CC2420_READ_RAM_LE(p,a,c,n) \
    do { \
        SPI_ENABLE(); \
        SPI_TX(0x80 | (a & 0x7F)); \
        SPI_TX(((a >> 1) & 0xC0) | 0x20); \
        for (n = 0; n < (c); n++) { \
            SPI_RX(((BYTE*)(p))[n]); \
        } \
        SPI_DISABLE(); \
    } while (0)
// Functions and Variables
void cc2420_init(void);
void cc2420_channel_init(UINT8 channel);

#endif // CC2420_M
#endif // ~CC2420_H

