//===================================================================
//
// timer.c (@haekim)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#include <io.h>
#include <signal.h>

#include "timer.h"
#include "critical_section.h"

#ifdef UART_M
// TimerA interrupt period for DCO calibration. Here, overflow interrupt.
#define TA_CALIB_TICKS			65536
// Endurable the minmum of TimerB ticks while TimerA interrupt period
// TB_MIN_TICKS = TA_CALIB_TICKS * DCOCLK_MIN/DCOCLK * TBCLK/TACLK = 992
#define TB_MIN_TICKS			992
// Endurable the maximum of TimerB ticks while TimerA interrupt period
// TB_MAX_TICKS=TA_CALIB_TICKS * DCOCLK_MAX/DCOCLK * TBCLK/TACLK = 1056
#define TB_MAX_TICKS			1056
UINT16 prev_tbr;
#endif

#ifdef TIMECHK_M
#include "time_check.h"
extern void nos_local_clock_ticks(void);
#endif

#ifdef IEEE_802_15_4_MAC
// 802.15.4 MAC scheduling time = 320us
// 802.15.4 ticks = TACLK(2048)*1024(Hz) * 0.00032(sec)
#define MAC_15_4_TICKS	671
#endif


//Do not change anything.
void nos_timer_init()
{
	// TASSEL[9-8] - Timer A source select. 01:ACLK, 10:SMCLK
	// ID[7-6] - Input divider. 00:/1, 01:/2, 10:/4, 11/8
	// MC[5-4] - Mode control. 00:stop, 01:up, 10:continuous, 11:up/down
	// TACLR[2] - Timer A clear. Setting this bit resets TAR, the clock divider, the count direction (up direction sin up/down mode)
	// TAIE[1] - Timer A interrupt enable
	// TAIFG[0] - Timer A interrupt flag

	// Initialize TimerA.
	TACTL = TASSEL1| MC1;// SMCLK, continuous mode.  default : 0x0000
	// Initialize TimerB
	TBCTL = TBSSEL0 | MC1;	// ACLK, conitnuous mode.  default : 0x0000

#ifdef IEEE_802_15_4_MAC
	TACCR1 = MAC_15_4_TICKS;
	CLEAR_TIMERA_CCR1_vect();
    ENABLE_TIMERA_CCR1_vect();
#endif
#ifdef TIMECHK_M
	local_clock.sec =0;
	local_clock.min =0;
	local_clock.hour =0;
	CLEAR_TIMERB_OVF_vect();
	ENABLE_TIMERB_OVF_vect();
#endif
}


wakeup interrupt (TIMERA1_VECTOR) TAIV_ISR(void)
{
	NOS_ENTER_ISR();
#ifdef UART_M
	UINT16 now_tbr, delta;
#endif

	switch (TAIV)
	{
#ifdef IEEE_802_15_4_MAC
		//TACCR1 interrupt. 802.15.4 MAC scheduling time
		case 0x0002: 
			TACCR1 += MAC_15_4_TICKS;
			//Call 802.15.4MAC callback function here.
			break;
#endif
		/*
		//TACCR2 interrupt
		case 0x0004: 
			break;
		*/
#ifdef UART_M
		//Overflow interrupt. DCO calibration period
	 	case 0x000A: 
			now_tbr = TBR;
			delta = now_tbr - prev_tbr;
		   	prev_tbr = now_tbr;
			// If DCOosc is lower than 3968kHz, speeds it up.
		   	if (delta > TB_MAX_TICKS-6)
		   	{
	     		if (DCOCTL < 0xe0)
	     		{
	       			++DCOCTL;
	     		}
	     		else if ((BCSCTL1 & 0x07) < 0x07)
	     		{
	       			++BCSCTL1;
					DCOCTL = 0;
	     		}
		   	}
			// If DCOosc is higher than 4224kHz, slows it down.
		   	else if (delta < TB_MIN_TICKS+6)
		   	{
	     		if (DCOCTL > 0)
	     		{
	       			--DCOCTL;
	     		}
	     		else if ((BCSCTL1 & 7) > 0)
	     		{
	       			--BCSCTL1;
	       			DCOCTL = 0xe0;
	     		}
		   	}
			break;
#endif

		// Default.
		default:
			break;
	}
	NOS_EXIT_ISR();
}


wakeup interrupt (TIMERB1_VECTOR) TBIV_ISR(void)
{
	NOS_ENTER_ISR();
	switch (TBIV)
	{
		/*
		//Capture/compare1 interrupt
		case 0x0002:
			break;
		case 0x0004:
			break;
		case 0x0006:
			break;
		case 0x0008:
			break;
		case 0x000a:
			break;
		case 0x000c:
			break;
		*/

#ifdef TIMECHK_M
		//Timer overflow interrupt
		case 0x000e:
			nos_local_clock_ticks();
			break;
#endif
		default:
			break;
	}

	
	NOS_EXIT_ISR();
}


