//===================================================================
//
// arch.c (@haekim, @sheart)
//
//===================================================================
// Copyright 2004-2010, ETRI
//===================================================================
#include <io.h>

#include "arch.h"
#include "heap.h"
#include "critical_section.h"
#include "intr.h"
#include "clock.h"
#include "timer.h"

#ifdef ADC_M
#include "adc.h"
#endif

extern volatile INTR_STATUS intr_status;

void nos_arch_init()
{
	nested_intr_cnt = 0;
	intr_status.cnt = 0;

	NOS_HEAP_INIT();
	WDTCTL = WDTPW | WDTHOLD; // Stop WDT to conserve power.

	// Timer must be initialized after nos_clock_init() because nos_clock_init() uses timer conitnuous mode for calibration
	// or You should back up the registers to keep timer setting. see  "nos_adj_dco_clock()" at clock.c
	nos_clock_init();
	nos_timer_init();
	//If you use KERNEL_M nos_kernel_init() must be called after nos_timer_init() because kernel scheduler configures timerA again.
#ifdef ADC_M
	nos_adc_init();
#endif
}


// At least, this function takes about 4us
// Actual delay time = timeout_usec / 0.96
// It use dummy instructions for delay. We cannot use timer because the minimum time to configure timer takes over 1000us.
// ISR or context_switch does not affect the delay time.
void nos_delay_us(UINT16 timeout_usec)
{
	if (timeout_usec <= 4)
		return;
	timeout_usec -= 4;

	// each loop takes 4 instruction
   	do
	{
		NOS_NOP();
    }while (--timeout_usec);
}

void nos_delay_ms(UINT16 timeout_msec)
{
	if (timeout_msec == 0)
	{
		return;
	}
	do
	{
		nos_delay_us(1041);
		NOS_NOP();
		NOS_NOP();		
	}while (--timeout_msec);
}


// Precise delay_ms function. ISR or context_switch does not affect the delay time.
// It does not supprt re-entrance. use this function in critical section 
// This use TimerB overflow and CCR2.
void nos_count_ms(UINT16 timeout_msec)
{
	UINT8 ovf_time;
	if (timeout_msec >= 2000)
	{
		// remember TBR value at entrance.
		TBCCR2 = TBR;
		// check the number of overflow
		ovf_time = timeout_msec/2000; //65536/32.768 = 2000
		while(ovf_time--)
		{
			CLEAR_TIMERB_OVF_vect();
			while(!TIMERB_OVF_vect_IS_SET());
			CLEAR_TIMERB_CCR2_vect();
		}
		timeout_msec %= 2000;
		// Wait until TBR entrance value
		while(!TIMERB_CCR2_vect_IS_SET());
	}

	if (timeout_msec != 0)
	{
		// from here, timeout_msec means the number of TBR ticks. 29 is compensation for function call.
		timeout_msec = timeout_msec*32.768-29;
		CLEAR_TIMERB_CCR2_vect();
		TBCCR2 = TBR + timeout_msec;
		while(!TIMERB_CCR2_vect_IS_SET());
	}
}

