//===================================================================
//                                      
// nos_malloc.c (@jun361)         
//
// This code is modified from avr-libc 1.4.4
//===================================================================
                                
#include "heap.h"
#include "critical_section.h"

void *nos_malloc(UINT16 len)
{
        struct __freelist *fp1, *fp2;
        INT8 *cp;
        UINT16 s = 0, avail;


        NOS_ENTER_CRITICAL_SECTION();
        // The minimum chunk size is the size of a pointer (plus the size of the "sz" field, but we don't need to account for this), 
        // otherwise we could not possibly fit a freelist entry into the chunk later.

        if (len < sizeof(struct __freelist) - sizeof(UINT16))
                len = sizeof(struct __freelist) - sizeof(UINT16);

        // Step 1: Walk the free list and try finding a chunk that would match exactly.  
        // If we found one, we are done.  
        // While walking, note down the size of the largest chunk we found that would still fit the request -- we need it for step 2.

#ifdef DEBUG_HEAP	//070117 @haekim
	//nos_uart_printf("malloc : __flp = %u s= %u __brkval = %u\n", __flp, s, __brkval);
#endif

        for (s = 0, fp1 = __flp, fp2 = 0; fp1; fp2 = fp1, fp1 = fp1->nx)
        {
                // Exact match of the free list entries
                if (fp1->sz == len) // Found it. Disconnect the chunk from the freelist, and return it. 
                {
                        if (fp2)
                                fp2->nx = fp1->nx;
                        else
                                __flp = fp1->nx;

                        NOS_EXIT_CRITICAL_SECTION();

                        return &(fp1->nx);
                }

                // More larger free list entry than requested
                if (fp1->sz > len)
                {
                        if (s == 0 || fp1->sz < s)
                                s = fp1->sz;
                }
        }

        // Step 2: If we found a chunk on the freelist that would fit (but was too large), look it up again and use it, 
        // since it is our closest match now.  
        // Since the freelist entry needs to be split into two entries then, 
        // watch out that the difference between the requested size 
        // and the size of the chunk found is large enough for another freelist entry; 
        // if not, just enlarge the request size to what we have found, and use the entire chunk.

 	if (s)
        {
                if (s - len < sizeof(struct __freelist))
                        len = s;

                for (fp1 = __flp, fp2 = 0; fp1; fp2 = fp1, fp1 = fp1->nx)
                {
                        if (fp1->sz == s)
                        {
                                if (len == s) // Use entire chunk; same as above
                                {
                                        if (fp2)
                                                fp2->nx = fp1->nx;
                                        else
                                                __flp = fp1->nx;

                                        NOS_EXIT_CRITICAL_SECTION();

                                        return &(fp1->nx);
                                }

                                // Split them up.  Note that we leave the first part as the new (smaller) freelist entry, 
                                // and return the upper portion to the caller.  
                                // This saves us the work to fix up the freelist chain; 
                                // we just need to fixup the size of the current entry, 
                                // and note down the size of the new chunk before returning it to the caller.

                                cp = (INT8 *)fp1;
                                s -= len;
                                cp += s;
                                fp2 = (struct __freelist *)cp;
                                fp2->sz = len;
                                fp1->sz = s - sizeof(UINT16);

                                NOS_EXIT_CRITICAL_SECTION();

                                return &(fp2->nx);
                        }
                }
        }

        // Step 3: If the request could not be satisfied from a freelist entry, just prepare a new chunk.  
        // This means we need to obtain more memory first.
        // The largest address just not allocated so far is remembered in the brkval variable.
        // Under Unix, the "break value" was the end of the data segment as dynamically requested from the operating system.
        // Since we don't have an operating system, just make sure * that we don't collide with the stack.

        if (__brkval == 0)
                __brkval = (INT8 *) HEAP_START_ADDR;

        cp = (INT8 *) _heap_end_addr;

        avail = cp - __brkval; // currently available memory space for a new chunk

	// Both tests below are needed to catch the case len >= 0xfffe.
        if (avail >= len && avail >= len + sizeof(UINT16))
        {
                fp1 = (struct __freelist *)__brkval;
                __brkval += len + sizeof(UINT16);
                fp1->sz = len;

                NOS_EXIT_CRITICAL_SECTION();

                return &(fp1->nx);
        }

        // Step 4 : There's no help, just fail. :-/

        NOS_EXIT_CRITICAL_SECTION();

        return 0;
}
