package net.kldp.beat.action;

import java.lang.reflect.Method;

import net.kldp.beat.annotation.After;
import net.kldp.beat.annotation.Before;
import net.kldp.beat.annotation.BeforeResult;
import net.kldp.beat.exception.fatal.ActionExecuteException;

/**
 * 컨트롤러 클래스를 실행하는 클래스 입니다.
 *
 */
public class ActionInvoker {
	private Object action;
	private Method beforeMethod;
	private Method resultBeforeMethod;
	private Method afterMethod;

	public ActionInvoker(Object action) {
		this.action = action;
		classifyMethods(action.getClass().getMethods());
	}

	/**
	 * Action클래스안에서의 특별한 메서드들을 분류합니다. 이들 메서드는 Before, BeforResult, After를 선언한
	 * 메서드입니다.
	 * 
	 * @param methods
	 */
	private void classifyMethods(Method[] methods) {
		for (Method method : methods) {
			Before before = method.getAnnotation(Before.class);
			After after = method.getAnnotation(After.class);
			BeforeResult resultBefore = method.getAnnotation(BeforeResult.class);
			if (before != null && beforeMethod == null) {
				this.beforeMethod = method;
			} else if (resultBefore != null && resultBeforeMethod == null) {
				this.resultBeforeMethod = method;
			} else if (after != null && afterMethod == null) {
				this.afterMethod = method;
			}
		}
	}

	/**
	 * 컨트롤러 클래스의 execute 메서드를 실행합니다.
	 * @return
	 * @throws ActionExecuteException
	 */
	public String execute() throws ActionExecuteException {
		try {
			Method method = action.getClass().getMethod("execute");
			return (String) method.invoke(action);
		} catch (Exception e) {
			throw new ActionExecuteException(e);
		}
	}

	/**
	 * 컨트롤러 클래스에 @Before 어노테이션으로 설정되어 있는 메서드를 실행합니다.
	 * @throws ActionExecuteException
	 */
	public void executeBefore() throws ActionExecuteException {
		if (beforeMethod != null)
			invokeMethod(beforeMethod);
	}

	/**
	 * 컨트롤러 클래스에 @After 어노테이션으로 설정되어 있는 메서드를 실행합니다.
	 * @throws ActionExecuteException
	 */
	public void executeAfter() throws ActionExecuteException {
		if (afterMethod != null)
			invokeMethod(afterMethod);
	}

	/**
	 * 컨트롤러 클래스에 @BeforeResult 어노테이션으로 설정되어 있는 메서드를 실행합니다.
	 * @throws ActionExecuteException
	 */
	public void executeBeforeResult() throws ActionExecuteException {
		if (resultBeforeMethod != null)
			invokeMethod(resultBeforeMethod);
	}

	/**
	 * 리턴값이 void인 메서드를 실행합니다.
	 * @param method
	 * @throws ActionExecuteException
	 */
	private void invokeMethod(Method method) throws ActionExecuteException {
		try {
			method.invoke(action);
		} catch (Exception e) {
			throw new ActionExecuteException(e);
		}
	}
}